/*
 * Copyright (c) 1997, 1998
 *	Bill Paul <wpaul@ctr.columbia.edu>.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Bill Paul.
 * 4. Neither the name of the author nor the names of any co-contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY Bill Paul AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL Bill Paul OR THE VOICES IN HIS HEAD
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $FreeBSD: src/sys/dev/if_rl.c,v 1.38.2.7 2001/07/19 18:33:07 wpaul Exp $
 */

/*
 * RealTek 8129/8139 PCI NIC driver
 *
 * Written by Bill Paul <wpaul@ctr.columbia.edu>
 * Electrical Engineering Department
 * Columbia University, New York City
 */

#include <sys/cdefs.h>

#ifdef ENABLE_FIBER_SUPPORT
#define FIBER_SUFFIX "-FIBER"
#else
#define FIBER_SUFFIX ""
#endif
#define RE_VERSION "1.101.00" FIBER_SUFFIX

__FBSDID("$FreeBSD: src/sys/dev/re/if_re.c,v " RE_VERSION __DATE__ " " __TIME__ "  wpaul Exp $");

/*
* This driver also support Realtek RTL8110/RTL8169, RTL8111/RTL8168, RTL8125, RTL8126, and RTL8136/RTL810x.
*/

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/sockio.h>
#include <sys/mbuf.h>
#include <sys/malloc.h>
#include <sys/kernel.h>
#include <sys/socket.h>
#include <sys/sysctl.h>
#include <sys/taskqueue.h>

#include <net/if.h>
#include <net/if_var.h>
#include <net/if_arp.h>
#include <net/ethernet.h>
#include <net/if_dl.h>
#include <net/if_media.h>

#include <net/bpf.h>

#include <vm/vm.h>              /* for vtophys */
#include <vm/pmap.h>            /* for vtophys */
#include <machine/clock.h>      /* for DELAY */

#include <machine/bus.h>
#include <machine/resource.h>
#include <sys/bus.h>
#include <sys/rman.h>
#include <sys/endian.h>

#include <dev/mii/mii.h>
#include "if_rereg.h"
#ifdef ENABLE_FIBER_SUPPORT
#include <dev/re/if_fiber.h>
#endif //ENABLE_FIBER_SUPPORT

#if OS_VER < VERSION(5,3)
#include <pci/pcireg.h>
#include <pci/pcivar.h>
#include <machine/bus_pio.h>
#include <machine/bus_memio.h>
#else
#include <dev/pci/pcireg.h>
#include <dev/pci/pcivar.h>
#include <sys/module.h>
#endif

#if OS_VER > VERSION(5,9)
#include <sys/cdefs.h>
#include <sys/endian.h>
#include <net/if_types.h>
#include <net/if_vlan_var.h>
#endif

#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/ip6.h>

#include <machine/in_cksum.h>

#include "opt_inet.h"
#include "opt_inet6.h"

#define EE_SET(x)					\
	CSR_WRITE_1(sc, RE_EECMD,			\
		CSR_READ_1(sc, RE_EECMD) | x)

#define EE_CLR(x)					\
	CSR_WRITE_1(sc, RE_EECMD,			\
		CSR_READ_1(sc, RE_EECMD) & ~x)

struct bus_dma_tag {
        struct bus_dma_tag_common common;
        int                     map_count;
        int                     bounce_flags;
        bus_dma_segment_t       *segments;
        struct bounce_zone      *bounce_zone;
};

/*
 * Various supported device vendors/types and their names.
 */
static struct re_type re_devs[] = {
        {
                RT_VENDORID, RT_DEVICEID_8169,
                "Realtek PCI GbE Family Controller"
        },
        {
                RT_VENDORID, RT_DEVICEID_8169SC,
                "Realtek PCI GbE Family Controller"
        },
        {
                RT_VENDORID, RT_DEVICEID_8168,
                "Realtek PCIe GbE Family Controller"
        },
        {
                RT_VENDORID, RT_DEVICEID_8161,
                "Realtek PCIe GbE Family Controller"
        },
        {
                RT_VENDORID, RT_DEVICEID_8162,
                "Realtek PCIe GbE Family Controller"
        },
        {
                RT_VENDORID, RT_DEVICEID_8136,
                "Realtek PCIe FE Family Controller"
        },
        {
                DLINK_VENDORID, 0x4300,
                "Realtek PCI GbE Family Controller"
        },
        {
                RT_VENDORID, RT_DEVICEID_8125,
                "Realtek PCIe 2.5GbE Family Controller"
        },
        {
                RT_VENDORID, RT_DEVICEID_3000,
                "Killer PCIe 3x00 2.5GbE Family Controller"
        },
        {
                RT_VENDORID, RT_DEVICEID_8126,
                "Realtek PCIe 5GbE Family Controller"
        },
        {
                RT_VENDORID, RT_DEVICEID_8127,
                "Realtek PCIe 10GbE Family Controller"
        },
        { 0, 0, NULL }
};

static int	re_probe			__P((device_t));
static int	re_attach			__P((device_t));
static int	re_detach			__P((device_t));
static int	re_suspend 			__P((device_t));
static int	re_resume 			__P((device_t));
static int	re_shutdown			__P((device_t));

void re_mdio_write			__P((struct re_softc*, u_int8_t, u_int16_t));
u_int16_t re_mdio_read		__P((struct re_softc*, u_int8_t));
static void re_ephy_write			__P((struct re_softc*, u_int16_t, u_int16_t));
static void _re_ephy_write			__P((struct re_softc*, u_int16_t, u_int16_t));
static u_int16_t re_ephy_read		__P((struct re_softc*, u_int16_t));
static u_int8_t re_efuse_read			__P((struct re_softc*, u_int16_t));
static void re_real_ocp_phy_write       __P((struct re_softc*, u_int16_t, u_int16_t));
static u_int16_t re_real_ocp_phy_read   __P((struct re_softc*, u_int16_t));
static void re_ocp_phy_write       __P((struct re_softc*, u_int16_t, u_int8_t, u_int16_t));
static u_int16_t re_ocp_phy_read   __P((struct re_softc*, u_int16_t, u_int8_t));
void re_mac_ocp_write    __P((struct re_softc*, u_int16_t, u_int16_t));
u_int16_t re_mac_ocp_read  __P((struct re_softc*, u_int16_t));
static void re_csi_other_fun_write    __P((struct re_softc *, u_int8_t, u_int16_t, u_int32_t Regata));
static u_int32_t re_csi_other_fun_read   __P((struct re_softc *, u_int8_t, u_int16_t));
static void re_csi_write     __P((struct re_softc*, u_int16_t, u_int32_t));
static u_int32_t re_csi_read __P((struct re_softc*, u_int16_t));
//static u_int8_t re_csi_fun0_read_byte __P((struct re_softc*, u_int16_t));
static bool re_set_phy_mcu_patch_request  __P((struct re_softc *));
static bool re_clear_phy_mcu_patch_request  __P((struct re_softc *));

static int re_check_dash  __P((struct re_softc *));

static void re_driver_start             __P((struct re_softc*));
static void re_driver_stop         	__P((struct re_softc*));

static void re_hw_ephy_config		__P((struct re_softc *));
static void re_hw_phy_config		__P((struct re_softc *));
static void re_init			__P((void *));
static int  re_var_init			__P((struct re_softc *));
static void re_reset			__P((struct re_softc *));
static void re_stop			__P((struct re_softc *));
static void re_setwol			__P((struct re_softc *));
static void re_clrwol			__P((struct re_softc *));
static u_int8_t re_set_wol_linkspeed 	__P((struct re_softc *));

static void re_start				__P((struct ifnet *));
static void re_start_locked			__P((struct ifnet *, u_int32_t));
static int re_encap				__P((struct re_softc *, struct mbuf **));
static int re_8125_pad				__P((struct re_softc *, struct mbuf *));
static void WritePacket				__P((struct re_softc *, bus_dma_segment_t*, int, int, uint32_t, uint32_t, uint32_t, uint32_t));
static void re_start_tx				__P((struct re_softc *));
static uint32_t CountFreeTxDescNum			__P((struct re_descriptor *, u_int32_t));
//static int CountMbufNum				__P((struct mbuf *));
#ifdef RE_FIXUP_RX
static __inline void re_fixup_rx		__P((struct mbuf *));
#endif
static void re_txeof				__P((struct re_softc *, u_int32_t));

//static int re_rxeof				__P((struct re_softc *));

#if OS_VER < VERSION(7,0)
static void re_intr				__P((void *));
#else
static int re_intr				__P((void *));
#endif //OS_VER < VERSION(7,0)
#if OS_VER < VERSION(7,0)
static void re_intr_8125				__P((void *));
#else
static int re_intr_8125				__P((void *));
#endif //OS_VER < VERSION(7,0)
//static void re_set_multicast_reg	__P((struct re_softc *, u_int32_t, u_int32_t));
static void re_clear_all_rx_packet_filter	__P((struct re_softc *));
static void re_set_rx_packet_filter_in_sleep_state	__P((struct re_softc *));
static void re_set_rx_packet_filter	__P((struct re_softc *));
static void re_setmulti			__P((struct re_softc *));
static int  re_ioctl			__P((struct ifnet *, u_long, caddr_t));
static u_int8_t re_link_ok	__P((struct re_softc *));
static void re_link_on_patch	__P((struct re_softc *));
static void re_link_down_patch	__P((struct re_softc *));
static void re_init_timer	__P((struct re_softc *));
static void re_stop_timer	__P((struct re_softc *));
static void re_start_timer	__P((struct re_softc *));
static void re_tick				__P((void *));
#if OS_VER < VERSION(7,0)
static void re_watchdog				__P((struct ifnet *));
#endif

static int  re_ifmedia_upd			__P((struct ifnet *));
static void re_ifmedia_sts			__P((struct ifnet *, struct ifmediareq *));

static int  re_ifmedia_upd_8125			__P((struct ifnet *));
static void re_ifmedia_sts_8125			__P((struct ifnet *, struct ifmediareq *));

static void re_eeprom_ShiftOutBits		__P((struct re_softc *, int, int));
//static u_int16_t re_eeprom_ShiftInBits		__P((struct re_softc *));
//static void re_eeprom_EEpromCleanup		__P((struct re_softc *));
//static void re_eeprom_getword			__P((struct re_softc *, int, u_int16_t *));
static void re_read_eeprom			__P((struct re_softc *, caddr_t, int, int, int));
static void re_int_task_poll		(void *, int);
static void re_int_task				(void *, int);
static void re_int_task_8125_poll	(void *, int);
static void re_int_task_8125		(void *, int);

//static void re_phy_power_up(device_t dev);
//static void re_phy_power_down(device_t dev);
static int re_alloc_buf(struct re_softc *);
static void re_release_buf(struct re_softc *);
static void set_rxbufsize(struct re_softc*);
static void re_release_rx_buf(struct re_softc *);
static void re_release_tx_buf(struct re_softc *);
static u_int32_t re_eri_read(struct re_softc *, int, int, int);
static int re_eri_write(struct re_softc *, int, int, u_int32_t, int);
static void OOB_mutex_lock(struct re_softc *);
static void OOB_mutex_unlock(struct re_softc *);
static void re_hw_start_unlock(struct re_softc *sc);
static void re_hw_start_unlock_8125(struct re_softc *sc);

static void re_add_sysctls      (struct re_softc *);
static int re_sysctl_driver_variable      (SYSCTL_HANDLER_ARGS);
static int re_sysctl_stats      (SYSCTL_HANDLER_ARGS);
static int re_sysctl_registers  (SYSCTL_HANDLER_ARGS);
static int re_sysctl_registers2  (SYSCTL_HANDLER_ARGS);
static int re_sysctl_registers3  (SYSCTL_HANDLER_ARGS);
static int re_sysctl_registers4  (SYSCTL_HANDLER_ARGS);
static int re_sysctl_registers5  (SYSCTL_HANDLER_ARGS);
static int re_sysctl_eth_phy    (SYSCTL_HANDLER_ARGS);
static int re_sysctl_dump_rx_desc    (SYSCTL_HANDLER_ARGS);
static int re_sysctl_dump_tx_desc    (SYSCTL_HANDLER_ARGS);
static int re_sysctl_pcie_phy   (SYSCTL_HANDLER_ARGS);
static int re_sysctl_extended_registers   (SYSCTL_HANDLER_ARGS);
static int re_sysctl_pci_registers        (SYSCTL_HANDLER_ARGS);
static int re_sysctl_msix_tbl             (SYSCTL_HANDLER_ARGS);
static int re_sysctl_eee                  (SYSCTL_HANDLER_ARGS);

/* Tunables. */
SYSCTL_NODE(_hw, OID_AUTO, re, CTLFLAG_RW | CTLFLAG_MPSAFE, 0, "");
static int msi_disable = 1;
SYSCTL_INT(_hw_re, OID_AUTO, msi_disable, CTLFLAG_RDTUN, &msi_disable, 0,
    "");
static int msix_disable = 0;
SYSCTL_INT(_hw_re, OID_AUTO, msix_disable, CTLFLAG_RDTUN, &msix_disable, 0,
    "");
static int prefer_iomap = 0;
SYSCTL_INT(_hw_re, OID_AUTO, prefer_iomap, CTLFLAG_RDTUN, &prefer_iomap, 0,
    "");
static int re_lro_entry_count = 128;
SYSCTL_INT(_hw_re, OID_AUTO, lro_entry_count, CTLFLAG_RDTUN,
    &re_lro_entry_count, 0, "");
static int re_lro_mbufq_depth = RE_RX_BUF_NUM;
SYSCTL_INT(_hw_re, OID_AUTO, lro_mbufq_depth, CTLFLAG_RDTUN,
    &re_lro_mbufq_depth, 0, "");
#ifdef ENABLE_EEE
static int eee_enable = 1;
#else
static int eee_enable = 0;
#endif
SYSCTL_INT(_hw_re, OID_AUTO, eee_enable, CTLFLAG_RDTUN, &eee_enable, 0,
    "");
static int phy_power_saving = 1;
SYSCTL_INT(_hw_re, OID_AUTO, phy_power_saving, CTLFLAG_RDTUN,
    &phy_power_saving, 0, "");
static int phy_mdix_mode = RE_ETH_PHY_AUTO_MDI_MDIX;
SYSCTL_INT(_hw_re, OID_AUTO, phy_mdix_mode, CTLFLAG_RDTUN, &phy_mdix_mode,
    0, "");
#ifdef ENABLE_S5WOL
static int s5wol = 1;
#else
static int s5wol = 0;
SYSCTL_INT(_hw_re, OID_AUTO, s5wol, CTLFLAG_RDTUN, &s5wol, 0, "");
#endif
#ifdef ENABLE_S0_MAGIC_PACKET
static int s0_magic_packet = 1;
#else
static int s0_magic_packet = 0;
#endif
SYSCTL_INT(_hw_re, OID_AUTO, s0_magic_packet, CTLFLAG_RDTUN,
    &s0_magic_packet, 0, "");
#ifdef CONFIG_SOC_LAN
static int config_soc_lan = 1;
#else
static int config_soc_lan = 0;
#endif
SYSCTL_INT(_hw_re, OID_AUTO, config_soc_lan, CTLFLAG_RDTUN,
    &config_soc_lan, 0, "");
#ifdef ENABLE_INTERRUPT_MITIGATIN
static int interrupt_mitigation = 1;
#else
static int interrupt_mitigation = 0;
#endif
SYSCTL_INT(_hw_re, OID_AUTO, interrupt_mitigation, CTLFLAG_RDTUN,
    &interrupt_mitigation, 0, "");
static int max_rx_mbuf_sz = MJUM9BYTES;
SYSCTL_INT(_hw_re, OID_AUTO, max_rx_mbuf_sz, CTLFLAG_RDTUN,
    &max_rx_mbuf_sz, 0, "");

static int default_tx_qid = 0;

#define RE_CSUM_FEATURES_IPV4    (CSUM_IP | CSUM_TCP | CSUM_UDP)
#define RE_CSUM_FEATURES_IPV6    (CSUM_TCP_IPV6 | CSUM_UDP_IPV6)
#define RE_CSUM_FEATURES    (RE_CSUM_FEATURES_IPV4 | RE_CSUM_FEATURES_IPV6)

static device_method_t re_methods[] = {
        /* Device interface */
        DEVMETHOD(device_probe, re_probe),
        DEVMETHOD(device_attach, re_attach),
        DEVMETHOD(device_detach, re_detach),
        DEVMETHOD(device_suspend, re_suspend),
        DEVMETHOD(device_resume, re_resume),
        DEVMETHOD(device_shutdown, re_shutdown),
        { 0, 0 }
};

static driver_t re_driver = {
        "re",
        re_methods,
        sizeof(struct re_softc)
};

#if OS_VER>=VERSION(14,0)
DRIVER_MODULE(if_re, pci, re_driver, 0, 0);
#else
static devclass_t re_devclass;
DRIVER_MODULE(if_re, pci, re_driver, re_devclass, 0, 0);
#endif
MODULE_DEPEND(if_re, pci, 1, 1, 1);
MODULE_DEPEND(if_re, ether, 1, 1, 1);

static void
re_clear_set_eth_phy_bit(
        struct re_softc *sc,
        u_int8_t   addr,
        u_int16_t   clearmask,
        u_int16_t   setmask
)
{
        u_int16_t PhyRegValue;

        PhyRegValue = re_mdio_read(sc, addr);
        PhyRegValue &= ~clearmask;
        PhyRegValue |= setmask;
        re_mdio_write(sc, addr, PhyRegValue);
}

static void
re_clear_eth_phy_bit(
        struct re_softc *sc,
        u_int8_t   addr,
        u_int16_t   mask
)
{
        re_clear_set_eth_phy_bit(sc,
                                 addr,
                                 mask,
                                 0
                                );
}

static void
re_set_eth_phy_bit(
        struct re_softc *sc,
        u_int8_t   addr,
        u_int16_t   mask
)
{
        re_clear_set_eth_phy_bit(sc,
                                 addr,
                                 0,
                                 mask
                                );
}

static void
re_clear_set_eth_ocp_phy_bit(
        struct re_softc *sc,
        u_int16_t   addr,
        u_int16_t   clearmask,
        u_int16_t   setmask
)
{
        u_int16_t PhyRegValue;

        PhyRegValue = re_real_ocp_phy_read(sc, addr);
        PhyRegValue &= ~clearmask;
        PhyRegValue |= setmask;
        re_real_ocp_phy_write(sc, addr, PhyRegValue);
}

static void
re_clear_eth_ocp_phy_bit(
        struct re_softc *sc,
        u_int16_t   addr,
        u_int16_t   mask
)
{
        re_clear_set_eth_ocp_phy_bit(sc,
                                     addr,
                                     mask,
                                     0
                                    );
}

static void
re_set_eth_ocp_phy_bit(
        struct re_softc *sc,
        u_int16_t   addr,
        u_int16_t   mask
)
{
        re_clear_set_eth_ocp_phy_bit(sc,
                                     addr,
                                     0,
                                     mask
                                    );
}

static void
re_clear_set_mac_ocp_bit(
        struct re_softc *sc,
        u_int16_t   addr,
        u_int16_t   clearmask,
        u_int16_t   setmask
)
{
        u_int16_t PhyRegValue;

        PhyRegValue = re_mac_ocp_read(sc, addr);
        PhyRegValue &= ~clearmask;
        PhyRegValue |= setmask;
        re_mac_ocp_write(sc, addr, PhyRegValue);
}

static void
re_clear_mac_ocp_bit(
        struct re_softc *sc,
        u_int16_t   addr,
        u_int16_t   mask
)
{
        re_clear_set_mac_ocp_bit(sc,
                                 addr,
                                 mask,
                                 0
                                );
}

static void
re_set_mac_ocp_bit(
        struct re_softc *sc,
        u_int16_t   addr,
        u_int16_t   mask
)
{
        re_clear_set_mac_ocp_bit(sc,
                                 addr,
                                 0,
                                 mask
                                );
}

static void re_clear_phy_ups_reg(struct re_softc *sc)
{
        switch(sc->re_type) {
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                re_clear_eth_ocp_phy_bit(sc, 0xA466, BIT_0);
        /*	FALLTHROUGH */
        case MACFG_80:
        case MACFG_81:
                re_clear_eth_ocp_phy_bit(sc, 0xA468, BIT_3 | BIT_1);
                break;
        };
}

static int re_is_ups_resume(struct re_softc *sc)
{
        switch(sc->re_type) {
        case MACFG_80:
        case MACFG_81:
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                return (re_mac_ocp_read(sc, 0xD42C) & BIT_8);
        default:
                return (re_mac_ocp_read(sc, 0xD408) & BIT_0);
        }
}

static void re_clear_ups_resume_bit(struct re_softc *sc)
{
        switch(sc->re_type) {
        case MACFG_80:
        case MACFG_81:
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                re_mac_ocp_write(sc, 0xD42C, re_mac_ocp_read(sc, 0xD42C) & ~(BIT_8));
                break;
        default:
                re_mac_ocp_write(sc, 0xD408, re_mac_ocp_read(sc, 0xD408) & ~(BIT_0));
                break;
        }
}

static u_int8_t re_get_phy_state(struct re_softc *sc)
{
        switch(sc->re_type) {
        case MACFG_80:
        case MACFG_81:
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                return (re_real_ocp_phy_read(sc, 0xA420) & 0x7);
        case MACFG_68:
        case MACFG_69:
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
        case MACFG_74:
        case MACFG_75:
                return (re_ocp_phy_read(sc, 0x0A42, 0x10) & 0x7);
        default:
                return 0xff;
        };
}

static void re_wait_phy_ups_resume(struct re_softc *sc, u_int16_t PhyState)
{
        switch(sc->re_type) {
        case MACFG_68:
        case MACFG_69:
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
        case MACFG_74:
        case MACFG_75:
        case MACFG_80:
        case MACFG_81:
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                for (int i=0; i< 100; i++) {
                        if (re_get_phy_state(sc) == PhyState)
                                break;
                        else
                                DELAY(1000);
                }
                break;
        default:
                break;
        };
}

static void re_phy_power_up(device_t dev)
{
        struct re_softc		*sc;
        u_int8_t Data8;

        sc = device_get_softc(dev);

        if ((sc->re_if_flags & RL_FLAG_PHYWAKE_PM) != 0)
                CSR_WRITE_1(sc, RE_PMCH, CSR_READ_1(sc, RE_PMCH) | (BIT_6|BIT_7));

        re_mdio_write(sc, 0x1F, 0x0000);

        switch (sc->re_type) {
        case MACFG_4:
        case MACFG_5:
        case MACFG_6:
        case MACFG_21:
        case MACFG_22:
        case MACFG_23:
        case MACFG_24:
        case MACFG_25:
        case MACFG_26:
        case MACFG_27:
        case MACFG_28:
        case MACFG_31:
        case MACFG_32:
        case MACFG_33:
        case MACFG_63:
        case MACFG_64:
        case MACFG_65:
        case MACFG_66:
                re_mdio_write(sc, 0x0e, 0x0000);
                break;
        case MACFG_56:
        case MACFG_57:
        case MACFG_58:
        case MACFG_61:
                Data8 = re_eri_read(sc, 0x1AB, 1, ERIAR_ExGMAC);
                Data8 |= (BIT_2 | BIT_3 | BIT_4 | BIT_5 | BIT_6 | BIT_7);
                re_eri_write(sc, 0x1AB, 1, Data8, ERIAR_ExGMAC);
                break;
        default:
                break;
        };


        re_mdio_write(sc, MII_BMCR, BMCR_AUTOEN);

        //wait mdc/mdio ready
        switch(sc->re_type) {
        case MACFG_61:
        case MACFG_62:
        case MACFG_67:
                DELAY(10000);
                break;
        }

        //wait ups resume (phy state 3)
        switch(sc->re_type) {
        case MACFG_68:
        case MACFG_69:
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
        case MACFG_74:
        case MACFG_75:
        case MACFG_80:
        case MACFG_81:
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                re_wait_phy_ups_resume(sc, 3);
                break;
        };
}

static u_int16_t re_get_phy_lp_ability(struct re_softc *sc)
{
        u_int16_t anlpar;

        re_mdio_write(sc, 0x1F, 0x0000);
        anlpar = re_mdio_read(sc, MII_ANLPAR);

        return anlpar;
}

static void re_phy_power_down(device_t dev)
{
        struct re_softc		*sc;
        u_int8_t Data8;

        sc = device_get_softc(dev);

#ifdef ENABLE_FIBER_SUPPORT
        if (HW_FIBER_MODE_ENABLED(sc))
                return;
#endif //ENABLE_FIBER_SUPPORT

        if (sc->re_dash) {
                re_set_wol_linkspeed(sc);
                return;
        }

        re_mdio_write(sc, 0x1F, 0x0000);

        switch (sc->re_type) {
        case MACFG_21:
        case MACFG_22:
        case MACFG_23:
        case MACFG_24:
        case MACFG_25:
        case MACFG_26:
        case MACFG_27:
        case MACFG_28:
        case MACFG_31:
        case MACFG_32:
        case MACFG_33:
        case MACFG_63:
        case MACFG_64:
        case MACFG_65:
        case MACFG_66:
                re_mdio_write(sc, 0x0e, 0x0200);
                re_mdio_write(sc, MII_BMCR, (BMCR_AUTOEN|BMCR_PDOWN));
                break;
        case MACFG_56:
        case MACFG_57:
        case MACFG_58:
        case MACFG_61:
                Data8 = re_eri_read(sc, 0x1AB, 1, ERIAR_ExGMAC);
                Data8 &= ~(BIT_2 | BIT_3 | BIT_4 | BIT_5 | BIT_6 | BIT_7);
                re_eri_write(sc, 0x1AB, 1, Data8, ERIAR_ExGMAC);

                re_mdio_write(sc, MII_BMCR, (BMCR_AUTOEN|BMCR_PDOWN));
                break;
        default:
                re_mdio_write(sc, MII_BMCR, (BMCR_AUTOEN|BMCR_PDOWN));
                break;
        }

        switch (sc->re_type) {
        case MACFG_36:
        case MACFG_37:
        case MACFG_42:
        case MACFG_43:
        case MACFG_54:
        case MACFG_55:
                CSR_WRITE_1(sc, 0xD0, CSR_READ_1(sc, 0xD0) & ~BIT_6);
                break;
        case MACFG_38:
        case MACFG_39:
        case MACFG_50:
        case MACFG_51:
        case MACFG_52:
        case MACFG_56:
        case MACFG_57:
        case MACFG_58:
        case MACFG_59:
        case MACFG_60:
        case MACFG_61:
        case MACFG_62:
        case MACFG_67:
        case MACFG_68:
        case MACFG_69:
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
        case MACFG_74:
        case MACFG_75:
                CSR_WRITE_1(sc, 0xD0, CSR_READ_1(sc, 0xD0) & ~BIT_6);
                CSR_WRITE_1(sc, 0xF2, CSR_READ_1(sc, 0xF2) & ~BIT_6);
                break;
        }

        if (((sc->re_if_flags & RL_FLAG_PHYWAKE_PM) != 0) &&
            !(sc->re_type == MACFG_100 || sc->re_type == MACFG_101))
                CSR_WRITE_1(sc, RE_PMCH, CSR_READ_1(sc, RE_PMCH) & ~BIT_7);
}

/*
static void re_tx_dma_map_buf(void *arg, bus_dma_segment_t *segs, int nseg, int error)
{
        union TxDesc *txptr = arg;

        if (error) {
                txptr->ul[0] &= ~htole32(RL_TDESC_CMD_BUFLEN);
                txptr->so1.TxBuff = 0;
                return;
        }

        txptr->so1.TxBuff = htole64(segs->ds_addr);
}
*/

static void re_rx_dma_map_buf(void *arg, bus_dma_segment_t *segs, int nseg, int error)
{
        union RxDesc *rxptr = arg;

        if (error) {
                rxptr->ul[0] &= ~htole32(RL_RDESC_CMD_BUFLEN);
                /* make sure desc is releasing before change buffer address */
                wmb();
                rxptr->so0.RxBuff = 0;
                return;
        }

        rxptr->so0.RxBuff = htole64(segs->ds_addr);
}

static void re_dma_map_rxdesc(void *arg, bus_dma_segment_t *segs, int nseg, int error)
{
        struct re_dma_map_arg *re_dm_arg = arg;
        struct re_softc *sc = re_dm_arg->sc;
        u_int32_t qid = re_dm_arg->qid;

        if (error)
                return;

        if (sc->re_type == MACFG_100 || sc->re_type == MACFG_101) {
                u_int32_t addr;

                if (qid == 0)
                        addr = 0xe4;
                else if (qid < RL_RX_QUEUE_NUM)
                        addr = 0x4000 + ((qid - 1) * 8);
                else
                        return;

                CSR_WRITE_4(sc, addr, RL_ADDR_LO(segs->ds_addr));
                CSR_WRITE_4(sc, addr + 4, RL_ADDR_HI(segs->ds_addr));
        } else {
                CSR_WRITE_4(sc, 0xe4, RL_ADDR_LO(segs->ds_addr));
                CSR_WRITE_4(sc, 0xe8, RL_ADDR_HI(segs->ds_addr));
        }
}

static void re_dma_map_txdesc(void *arg, bus_dma_segment_t *segs, int nseg, int error)
{
        struct re_dma_map_arg *re_dm_arg = arg;
        struct re_softc *sc = re_dm_arg->sc;
        u_int32_t qid = re_dm_arg->qid;

        if (error)
                return;

        if (sc->re_type == MACFG_100 || sc->re_type == MACFG_101) {
                u_int32_t addr;

                if (qid == 0)
                        addr = 0x20;
                else if (qid < 8)
                        addr = 0x2100 + ((qid - 1) * 8);
                else if (qid < RL_TX_QUEUE_NUM)
                        addr = 0x2200 + ((qid - 8) * 8);
                else
                        return;

                CSR_WRITE_4(sc, addr, RL_ADDR_LO(segs->ds_addr));
                CSR_WRITE_4(sc, addr + 4, RL_ADDR_HI(segs->ds_addr));
        } else {
                CSR_WRITE_4(sc, 0x20, RL_ADDR_LO(segs->ds_addr));
                CSR_WRITE_4(sc, 0x24, RL_ADDR_HI(segs->ds_addr));
        }
}

/*
 * Probe for a RealTek 8129/8139 chip. Check the PCI vendor and device
 * IDs against our list and return a device name if we find a match.
 */
static int re_probe(device_t dev)	/* Search for Realtek NIC chip */
{
        struct re_type		*t;
        t = re_devs;
        while (t->re_name != NULL) {
                if ((pci_get_vendor(dev) == t->re_vid) &&
                    (pci_get_device(dev) == t->re_did)) {
                        device_set_desc(dev, t->re_name);
                        return(0);
                }
                t++;
        }

        return(ENXIO);
}


static u_int32_t re_eri_read_with_oob_base_address(struct re_softc *sc, int addr, int len, int type, const u_int32_t base_address)
{
        int i, val_shift, shift = 0;
        u_int32_t value1 = 0, value2 = 0, mask;
        const u_int32_t transformed_base_address = ((base_address & 0x00FFF000) << 6) | (base_address & 0x000FFF);

        if (len > 4 || len <= 0)
                return -1;

        while (len > 0) {
                val_shift = addr % ERIAR_Addr_Align;
                addr = addr & ~0x3;

                CSR_WRITE_4(sc,RE_ERIAR,
                            ERIAR_Read |
                            transformed_base_address |
                            type << ERIAR_Type_shift |
                            ERIAR_ByteEn << ERIAR_ByteEn_shift |
                            addr);

                for (i = 0; i < 10; i++) {
                        DELAY(100);

                        /* Check if the RTL8168 has completed ERI read */
                        if (CSR_READ_4(sc,RE_ERIAR) & ERIAR_Flag)
                                break;
                }

                if (len == 1)		mask = (0xFF << (val_shift * 8)) & 0xFFFFFFFF;
                else if (len == 2)	mask = (0xFFFF << (val_shift * 8)) & 0xFFFFFFFF;
                else if (len == 3)	mask = (0xFFFFFF << (val_shift * 8)) & 0xFFFFFFFF;
                else			mask = (0xFFFFFFFF << (val_shift * 8)) & 0xFFFFFFFF;

                value1 = CSR_READ_4(sc,RE_ERIDR) & mask;
                value2 |= (value1 >> val_shift * 8) << shift * 8;

                if (len <= 4 - val_shift)
                        len = 0;
                else {
                        len -= (4 - val_shift);
                        shift = 4 - val_shift;
                        addr += 4;
                }
        }

        return value2;
}

static u_int32_t re_eri_read(struct re_softc *sc, int addr, int len, int type)
{
        return re_eri_read_with_oob_base_address(sc, addr, len, type, 0);
}

static int re_eri_write_with_oob_base_address(struct re_softc *sc, int addr, int len, u_int32_t value, int type, const u_int32_t base_address)
{
        int i, val_shift, shift = 0;
        u_int32_t value1 = 0, mask;
        const u_int32_t transformed_base_address = ((base_address & 0x00FFF000) << 6) | (base_address & 0x000FFF);

        if (len > 4 || len <= 0)
                return -1;

        while (len > 0) {
                val_shift = addr % ERIAR_Addr_Align;
                addr = addr & ~0x3;

                if (len == 1)		mask = (0xFF << (val_shift * 8)) & 0xFFFFFFFF;
                else if (len == 2)	mask = (0xFFFF << (val_shift * 8)) & 0xFFFFFFFF;
                else if (len == 3)	mask = (0xFFFFFF << (val_shift * 8)) & 0xFFFFFFFF;
                else			mask = (0xFFFFFFFF << (val_shift * 8)) & 0xFFFFFFFF;

                value1 = re_eri_read_with_oob_base_address(sc, addr, 4, type, base_address) & ~mask;
                value1 |= ((value << val_shift * 8) >> shift * 8);

                CSR_WRITE_4(sc,RE_ERIDR, value1);
                CSR_WRITE_4(sc,RE_ERIAR,
                            ERIAR_Write |
                            transformed_base_address |
                            type << ERIAR_Type_shift |
                            ERIAR_ByteEn << ERIAR_ByteEn_shift |
                            addr);

                for (i = 0; i < 10; i++) {
                        DELAY(100);

                        /* Check if the RTL8168 has completed ERI write */
                        if (!(CSR_READ_4(sc,RE_ERIAR) & ERIAR_Flag))
                                break;
                }

                if (len <= 4 - val_shift)
                        len = 0;
                else {
                        len -= (4 - val_shift);
                        shift = 4 - val_shift;
                        addr += 4;
                }
        }

        return 0;
}

static int re_eri_write(struct re_softc *sc, int addr, int len, u_int32_t value, int type)
{
        return re_eri_write_with_oob_base_address(sc, addr, len, value, type, 0);
}

static void
re_dma_map_addr(void *arg, bus_dma_segment_t *segs, int nseg, int error)
{
        bus_addr_t              *addr;

        if (error)
                return;

        KASSERT(nseg == 1, ("too many DMA segments, %d should be 1", nseg));
        addr = arg;
        *addr = segs->ds_addr;
}

static int re_alloc_stats(device_t dev, struct re_softc *sc)
{
        int                     error;

        /* Create DMA map for statistics. */
        error = bus_dma_tag_create(sc->re_parent_tag, RE_DUMP_ALIGN, 0,
                                   BUS_SPACE_MAXADDR, BUS_SPACE_MAXADDR, NULL, NULL,
                                   sizeof(struct re_stats), 1, sizeof(struct re_stats), 0, NULL, NULL,
                                   &sc->re_tally.re_stag);
        if (error) {
                device_printf(dev, "could not create statistics DMA tag\n");
                return (error);
        }
        /* Allocate DMA'able memory for statistics. */
        error = bus_dmamem_alloc(sc->re_tally.re_stag,
                                 (void **)&sc->re_tally.re_stats,
                                 BUS_DMA_WAITOK | BUS_DMA_COHERENT | BUS_DMA_ZERO,
                                 &sc->re_tally.re_smap);
        if (error) {
                device_printf(dev,
                              "could not allocate statistics DMA memory\n");
                return (error);
        }
        /* Load the map for statistics. */
        sc->re_tally.re_stats_addr = 0;
        error = bus_dmamap_load(sc->re_tally.re_stag, sc->re_tally.re_smap,
                                sc->re_tally.re_stats, sizeof(struct re_stats), re_dma_map_addr,
                                &sc->re_tally.re_stats_addr, BUS_DMA_NOWAIT);
        if (error != 0 || sc->re_tally.re_stats_addr == 0) {
                device_printf(dev, "could not load statistics DMA memory\n");
                return (ENOMEM);
        }

        return (0);
}

static void re_release_rx_buf(struct re_softc *sc)
{
        int i;

        for (int j = 0; j < RL_RX_QUEUE_NUM; j++) {
                if (sc->re_desc.re_rx_mtag[j]) {
                        for (i = 0; i < RE_RX_BUF_NUM; i++) {
                                if (sc->re_desc.rx_buf[j][i]!=NULL) {
                                        bus_dmamap_sync(sc->re_desc.re_rx_mtag[j],
                                                        sc->re_desc.re_rx_dmamap[j][i],
                                                        BUS_DMASYNC_POSTREAD);
                                        bus_dmamap_unload(sc->re_desc.re_rx_mtag[j],
                                                          sc->re_desc.re_rx_dmamap[j][i]);
                                        bus_dmamap_destroy(sc->re_desc.re_rx_mtag[j],
                                                           sc->re_desc.re_rx_dmamap[j][i]);
                                        m_freem(sc->re_desc.rx_buf[j][i]);
                                        sc->re_desc.rx_buf[j][i] =NULL;
                                }
                        }
                        bus_dma_tag_destroy(sc->re_desc.re_rx_mtag[j]);
                        sc->re_desc.re_rx_mtag[j] =0;
                }
        }

}

static void re_release_tx_buf(struct re_softc *sc)
{
        int i;

        for (int j = 0; j < RL_TX_QUEUE_NUM; j++) {
                if (sc->re_desc.re_tx_mtag[j]) {
                        for (i = 0; i < RE_TX_BUF_NUM; i++) {
                                bus_dmamap_destroy(sc->re_desc.re_tx_mtag[j],
                                                   sc->re_desc.re_tx_dmamap[j][i]);
                                m_freem(sc->re_desc.tx_buf[j][i]);
                        }
                        bus_dma_tag_destroy(sc->re_desc.re_tx_mtag[j]);
                        sc->re_desc.re_tx_mtag[j] = 0;
                }
        }
}

static void
re_free_soft_lro(struct re_softc *sc)
{
#if defined(INET) || defined(INET6)
        if (sc->re_lro.ifp) {
                tcp_lro_free(&sc->re_lro);
                sc->re_lro.ifp = NULL;
        }
#endif
}

static int
re_config_soft_lro(struct re_softc *sc)
{
        struct lro_ctrl *lro;

        lro = &sc->re_lro;
        bzero(lro, sizeof(struct lro_ctrl));

#if defined(INET) || defined(INET6)
#if OS_VER >= VERSION(11,0)
        if (tcp_lro_init_args(lro, sc->re_ifp,
                              max(TCP_LRO_ENTRIES, re_lro_entry_count),
                              min(1024, re_lro_mbufq_depth)) != 0) {
                device_printf(sc->dev,
                              "%s: tcp_lro_init_args failed\n",
                              __func__);
                return (ENOMEM);
        }
#else
        if (tcp_lro_init(lro)) {
                device_printf(sc->dev,
                              "%s: tcp_lro_init failed\n",
                              __func__);
                return (-1);
        }
#endif //OS_VER >= VERSION(11,0)
#endif
        lro->ifp = sc->re_ifp;

        return (0);
}

static void re_release_buf(struct re_softc *sc)
{
        re_release_rx_buf(sc);
        re_release_tx_buf(sc);
}

static int re_alloc_buf(struct re_softc *sc)
{
        int error =0;
        int i,size;

        switch(sc->re_type) {
        case MACFG_3:
        case MACFG_4:
        case MACFG_5:
        case MACFG_6:
        case MACFG_11:
        case MACFG_12:
        case MACFG_13:
        case MACFG_21:
        case MACFG_22:
        case MACFG_23:
                size = RE_TX_MAXSIZE_32K;
                break;
        default:
                size = RE_TX_MAXSIZE_64K;
                break;
        }

        RE_UNLOCK(sc);
        for (int i = 0; i < RL_TX_QUEUE_NUM; i++) {
                error = bus_dma_tag_create(sc->re_parent_tag, 1, 0,
                                           BUS_SPACE_MAXADDR, BUS_SPACE_MAXADDR, NULL,
                                           NULL, size, RE_NTXSEGS, size, 0,
                                           NULL, NULL, &sc->re_desc.re_tx_mtag[i]);

                if (error) {
                        //device_printf(dev,"re_tx_mtag fail\n");
                        //goto fail;
                        RE_LOCK(sc);
                        return error;
                }
        }

        for (int i = 0; i < RL_RX_QUEUE_NUM; i++) {
                error = bus_dma_tag_create(
                                sc->re_parent_tag,
                                RE_RX_BUFFER_ALIGN, 0,		/* alignment, boundary */
                                BUS_SPACE_MAXADDR,		/* lowaddr */
                                BUS_SPACE_MAXADDR,		/* highaddr */
                                NULL, NULL,			/* filter, filterarg */
                                sc->re_rx_desc_buf_sz, 1,			/* maxsize,nsegments */
                                sc->re_rx_desc_buf_sz,			/* maxsegsize */
                                0,				/* flags */
                                NULL, NULL,			/* lockfunc, lockarg */
                                &sc->re_desc.re_rx_mtag[i]);
                if (error) {
                        //device_printf(dev,"re_rx_mtag fail\n");
                        //goto fail;
                        RE_LOCK(sc);
                        return error;
                }
        }

        RE_LOCK(sc);
        if (sc->re_rx_mbuf_sz <= MCLBYTES)
                size = MCLBYTES;
        else if (sc->re_rx_mbuf_sz <=  MJUMPAGESIZE)
                size = MJUMPAGESIZE;
        else
                size =MJUM9BYTES;

        for (int j = 0; j < RL_RX_QUEUE_NUM; j++) {
                for (i = 0; i < RE_RX_BUF_NUM; i++) {
                        sc->re_desc.rx_buf[j][i] = m_getjcl(M_DONTWAIT, MT_DATA, M_PKTHDR, size);
                        if (!sc->re_desc.rx_buf[j][i]) {
                                //device_printf(dev, "m_getcl fail!!!\n");
                                error = ENXIO;
                                //goto fail;
                                return error;
                        }

                        sc->re_desc.rx_buf[j][i]->m_len = sc->re_desc.rx_buf[j][i]->m_pkthdr.len = size;
#ifdef RE_FIXUP_RX
                        /*
                         * This is part of an evil trick to deal with non-x86 platforms.
                         * The RealTek chip requires RX buffers to be aligned on 64-bit
                         * boundaries, but that will hose non-x86 machines. To get around
                         * this, we leave some empty space at the start of each buffer
                         * and for non-x86 hosts, we copy the buffer back six bytes
                         * to achieve word alignment. This is slightly more efficient
                         * than allocating a new buffer, copying the contents, and
                         * discarding the old buffer.
                         */
                        m_adj(sc->re_desc.rx_buf[j][i], RE_ETHER_ALIGN);
#endif

                        error = bus_dmamap_create(sc->re_desc.re_rx_mtag[j], BUS_DMA_NOWAIT, &sc->re_desc.re_rx_dmamap[j][i]);
                        if (error) {
                                //device_printf(dev, "bus_dmamap_create fail!!!\n");
                                //goto fail;
                                return error;
                        }
                }
        }

        for (int j = 0; j < RL_TX_QUEUE_NUM; j++) {
                for (i = 0; i < RE_TX_BUF_NUM; i++) {
                        error = bus_dmamap_create(sc->re_desc.re_tx_mtag[j], BUS_DMA_NOWAIT, &sc->re_desc.re_tx_dmamap[j][i]);
                        if (error) {
                                //device_printf(dev, "bus_dmamap_create fail!!!\n");
                                //goto fail;
                                return error;
                        }
                }
        }

        return 0;
}

static void _set_rxbufsize(struct re_softc *sc,  u_int16_t rms)
{
        switch(sc->re_type) {
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                rms |= RE_RX_ACPT_VLAN_PF;
                break;
        default:
                rms &= ~RE_RX_ACPT_VLAN_PF;
                break;
        }

        CSR_WRITE_2(sc, RE_RxMaxSize, rms);
}

static void set_rxbufsize(struct re_softc *sc)
{
        struct ifnet		*ifp;
        ifp = RE_GET_IFNET(sc);
        sc->re_rx_desc_buf_sz = (ifp->if_mtu > ETHERMTU) ? ifp->if_mtu: ETHERMTU;
        sc->re_rx_desc_buf_sz += (ETHER_VLAN_ENCAP_LEN + ETHER_HDR_LEN + ETHER_CRC_LEN);
        if (!(sc->re_if_flags & RL_FLAG_8168G_PLUS) ||
            sc->re_type == MACFG_56 || sc->re_type == MACFG_57 ||
            sc->re_type == MACFG_58 || sc->re_type == MACFG_59 ||
            sc->re_type == MACFG_60)
                sc->re_rx_desc_buf_sz += 1;
        _set_rxbufsize(sc, sc->re_rx_desc_buf_sz);
}

static void re_enable_cfg9346_write(struct re_softc *sc)
{
        EE_SET(RE_EEMODE_WRITECFG);
}

static void re_disable_cfg9346_write(struct re_softc *sc)
{
        EE_CLR(RE_EEMODE_WRITECFG);
}

static void re_enable_force_clkreq(struct re_softc *sc, bool enable)
{
        if (enable)
                CSR_WRITE_1(sc, 0xF1, CSR_READ_1(sc, 0xF1) | BIT_7);
        else
                CSR_WRITE_1(sc, 0xF1, CSR_READ_1(sc, 0xF1) & ~BIT_7);
}

static void _re_enable_aspm_clkreq_lock(struct re_softc *sc, bool enable)
{
        switch(sc->re_type) {
        case MACFG_88:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                if (enable) {
                        CSR_WRITE_1(sc, RE_INT_CFG0_8125, CSR_READ_1(sc, RE_INT_CFG0_8125) | BIT_3);
                        CSR_WRITE_1(sc, RE_CFG5, CSR_READ_1(sc, RE_CFG5) | BIT_0);
                } else {
                        CSR_WRITE_1(sc, RE_INT_CFG0_8125, CSR_READ_1(sc, RE_INT_CFG0_8125) & ~BIT_3);
                        CSR_WRITE_1(sc, RE_CFG5, CSR_READ_1(sc, RE_CFG5) & ~BIT_0);
                }
                break;
        default:
                if (enable) {
                        CSR_WRITE_1(sc, RE_CFG5, CSR_READ_1(sc, RE_CFG5) | BIT_0);
                        CSR_WRITE_1(sc, RE_CFG2, CSR_READ_1(sc, RE_CFG2) | BIT_7);
                } else {
                        CSR_WRITE_1(sc, RE_CFG5, CSR_READ_1(sc, RE_CFG5) & ~BIT_0);
                        CSR_WRITE_1(sc, RE_CFG2, CSR_READ_1(sc, RE_CFG2) & ~BIT_7);
                }
                break;
        }
}

static void re_enable_aspm_clkreq_lock(struct re_softc *sc, bool enable)
{
        re_enable_cfg9346_write(sc);
        _re_enable_aspm_clkreq_lock(sc, enable);
        re_disable_cfg9346_write(sc);
}

static void re_disable_mcu_bps(struct re_softc *sc)
{
        u_int16_t regAddr;

        if (FALSE == HW_SUPPORT_MAC_MCU(sc))
                return;

        switch(sc->re_type) {
        case MACFG_56:
        case MACFG_57:
        case MACFG_58:
        case MACFG_59:
        case MACFG_60:
        case MACFG_61:
        case MACFG_62:
        case MACFG_67:
        case MACFG_68:
        case MACFG_69:
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
        case MACFG_74:
        case MACFG_75:
        case MACFG_76:
        case MACFG_80:
        case MACFG_81:
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                re_enable_aspm_clkreq_lock(sc, 0);
                break;
        }

        switch(sc->re_type) {
        case MACFG_68:
        case MACFG_69:
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
        case MACFG_74:
        case MACFG_75:
        case MACFG_76:
                re_mac_ocp_write(sc, 0xFC38, 0x0000);
                break;
        case MACFG_80:
        case MACFG_81:
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                re_mac_ocp_write(sc, 0xFC48, 0x0000);
                break;
        }

        switch(sc->re_type) {
        case MACFG_56:
        case MACFG_57:
        case MACFG_58:
        case MACFG_59:
        case MACFG_60:
        case MACFG_61:
        case MACFG_62:
        case MACFG_67:
        case MACFG_68:
        case MACFG_69:
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
        case MACFG_74:
        case MACFG_75:
        case MACFG_76:
                re_mac_ocp_write(sc, 0xFC28, 0x0000);
                re_mac_ocp_write(sc, 0xFC2A, 0x0000);
                re_mac_ocp_write(sc, 0xFC2C, 0x0000);
                re_mac_ocp_write(sc, 0xFC2E, 0x0000);
                re_mac_ocp_write(sc, 0xFC30, 0x0000);
                re_mac_ocp_write(sc, 0xFC32, 0x0000);
                re_mac_ocp_write(sc, 0xFC34, 0x0000);
                re_mac_ocp_write(sc, 0xFC36, 0x0000);

                DELAY(3000);

                re_mac_ocp_write(sc, 0xFC26, 0x0000);
                break;
        case MACFG_80:
        case MACFG_81:
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                for (regAddr = 0xFC28; regAddr < 0xFC48; regAddr += 2) {
                        re_mac_ocp_write(sc, regAddr, 0x0000);
                }

                DELAY(3000);

                re_mac_ocp_write(sc, 0xFC26, 0x0000);
                break;
        }
}

static void
re_switch_mac_mcu_ram_code_page(struct re_softc *sc, u_int16_t page)
{
        u_int16_t tmpUshort;

        page &= (BIT_1 | BIT_0);
        tmpUshort = re_mac_ocp_read(sc, 0xE446);
        tmpUshort &= ~(BIT_1 | BIT_0);
        tmpUshort |= page;
        re_mac_ocp_write(sc, 0xE446, tmpUshort);
}

static void
re_set_hw_mcu_patch_code_ver_common(struct re_softc *sc, u_int16_t reg, u_int64_t version)
{
        u_int16_t i;

        for (i = 0; i < 8; i += 2) {
                re_mac_ocp_write(sc, reg + 6 - i, (u_int16_t)version);
                version >>= 16;
        }
}

static void
re_set_hw_mcu_patch_code_ver_8168(struct re_softc *sc, u_int64_t version)
{
        re_set_hw_mcu_patch_code_ver_common(sc, 0xFBF8, version);
}

static void
re_set_hw_mcu_patch_code_ver_8125(struct re_softc *sc, u_int64_t version)
{
        /* Switch to page 2 */
        re_switch_mac_mcu_ram_code_page(sc, 2);

        re_set_hw_mcu_patch_code_ver_common(sc, 0xF9F8, version);

        /* Switch back to page 0 */
        re_switch_mac_mcu_ram_code_page(sc, 0);
}

static void
re_set_hw_mcu_patch_code_ver(struct re_softc *sc, u_int64_t version)
{
        switch (sc->HwSuppMacMcuVer) {
        case 1:
                re_set_hw_mcu_patch_code_ver_8168(sc, version);
                break;
        case 2:
                re_set_hw_mcu_patch_code_ver_8125(sc, version);
                break;
        default:
                return;
        }

        sc->HwMcuPatchCodeVer = version;
}

static u_int64_t
re_get_hw_mcu_patch_code_ver_common(struct re_softc *sc, u_int16_t reg)
{
        u_int64_t version;
        u_int16_t i;

        version = 0;
        for (i = 0; i < 8; i += 2) {
                version <<= 16;
                version |= re_mac_ocp_read(sc, reg + i);
        }

        return version;
}

static u_int64_t
re_get_hw_mcu_patch_code_ver_8168(struct re_softc *sc)
{
        u_int64_t version;

        version = re_get_hw_mcu_patch_code_ver_common(sc, 0xFBF8);

        return version;
}

static u_int64_t
re_get_hw_mcu_patch_code_ver_8125(struct re_softc *sc)
{
        u_int64_t version;

        /* Switch to page 2 */
        re_switch_mac_mcu_ram_code_page(sc, 2);

        version = re_get_hw_mcu_patch_code_ver_common(sc, 0xF9F8);

        /* Switch back to page 0 */
        re_switch_mac_mcu_ram_code_page(sc, 0);

        return version;
}

static u_int64_t
re_get_hw_mcu_patch_code_ver(struct re_softc *sc)
{
        switch (sc->HwSuppMacMcuVer) {
        case 1:
                return re_get_hw_mcu_patch_code_ver_8168(sc);
        case 2:
                return re_get_hw_mcu_patch_code_ver_8125(sc);
        default:
                return 0;
        }
}

static u_int64_t
re_get_bin_mcu_patch_code_ver(const u_int16_t *entry, u_int16_t entry_cnt)
{
        u_int64_t version;
        int i;

        if (entry == NULL || entry_cnt == 0 || entry_cnt < 4)
                return 0;

        version = 0;
        for (i = 0; i < 4; i++) {
                version <<= 16;
                version |= entry[entry_cnt - 4 + i];
        }

        return version;
}

static void
_re_write_mac_mcu_ram_code(struct re_softc *sc, const u_int16_t *entry, u_int16_t entry_cnt)
{
        u_int16_t i;

        for (i = 0; i < entry_cnt; i++) {
                re_mac_ocp_write(sc, 0xF800 + i * 2, entry[i]);
        }
}

static void
_re_write_mac_mcu_ram_code_with_page(struct re_softc *sc, const u_int16_t *entry, u_int16_t entry_cnt, u_int16_t page_size)
{
        u_int16_t i;
        u_int16_t offset;

        if (page_size == 0) return;

        for (i = 0; i < entry_cnt; i++) {
                offset = i % page_size;
                if (offset == 0) {
                        u_int16_t page = (i / page_size);
                        re_switch_mac_mcu_ram_code_page(sc, page);
                }
                re_mac_ocp_write(sc, 0xF800 + offset * 2, entry[i]);
        }
}

static void
re_write_mac_mcu_ram_code(struct re_softc *sc, const u_int16_t *entry, u_int16_t entry_cnt)
{
        if (FALSE == HW_SUPPORT_MAC_MCU(sc))
                return;

        if (entry == NULL || entry_cnt == 0)
                return;

        if (sc->MacMcuPageSize > 0)
                _re_write_mac_mcu_ram_code_with_page(sc, entry, entry_cnt, sc->MacMcuPageSize);
        else
                _re_write_mac_mcu_ram_code(sc, entry, entry_cnt);

        if (sc->BinMcuPatchCodeVer > 0)
                re_set_hw_mcu_patch_code_ver(sc, sc->BinMcuPatchCodeVer);
}

static void re_set_mac_mcu_8168g_1(struct re_softc *sc)
{
        static const uint16_t mcu_patch_code[] = {
                0xE008, 0xE01B, 0xE022, 0xE094, 0xE097, 0xE09A, 0xE0B3, 0xE0BA, 0x49D2,
                0xF10D, 0x766C, 0x49E2, 0xF00A, 0x1EC0, 0x8EE1, 0xC60A, 0x77C0, 0x4870,
                0x9FC0, 0x1EA0, 0xC707, 0x8EE1, 0x9D6C, 0xC603, 0xBE00, 0xB416, 0x0076,
                0xE86C, 0xC406, 0x7580, 0x4852, 0x8D80, 0xC403, 0xBC00, 0xD3E0, 0x02C8,
                0x8918, 0xE815, 0x1100, 0xF011, 0xE812, 0x4990, 0xF002, 0xE817, 0xE80E,
                0x4992, 0xF002, 0xE80E, 0xE80A, 0x4993, 0xF002, 0xE818, 0xE806, 0x4991,
                0xF002, 0xE838, 0xC25E, 0xBA00, 0xC056, 0x7100, 0xFF80, 0x7100, 0x4892,
                0x4813, 0x8900, 0xE00A, 0x7100, 0x4890, 0x4813, 0x8900, 0xC74B, 0x74F8,
                0x48C2, 0x4841, 0x8CF8, 0xC746, 0x74FC, 0x49C0, 0xF120, 0x49C1, 0xF11E,
                0x74F8, 0x49C0, 0xF01B, 0x49C6, 0xF119, 0x74F8, 0x49C4, 0xF013, 0xC536,
                0x74B0, 0x49C1, 0xF1FD, 0xC537, 0xC434, 0x9CA0, 0xC435, 0x1C13, 0x484F,
                0x9CA2, 0xC52B, 0x74B0, 0x49C1, 0xF1FD, 0x74F8, 0x48C4, 0x8CF8, 0x7100,
                0x4893, 0x8900, 0xFF80, 0xC520, 0x74B0, 0x49C1, 0xF11C, 0xC71E, 0x74FC,
                0x49C1, 0xF118, 0x49C0, 0xF116, 0x74F8, 0x49C0, 0xF013, 0x48C3, 0x8CF8,
                0xC516, 0x74A2, 0x49CE, 0xF1FE, 0xC411, 0x9CA0, 0xC411, 0x1C13, 0x484F,
                0x9CA2, 0x74A2, 0x49CF, 0xF1FE, 0x7100, 0x4891, 0x8900, 0xFF80, 0xE400,
                0xD3E0, 0xE000, 0x0481, 0x0C81, 0xDE20, 0x0000, 0x0992, 0x1B76, 0xC602,
                0xBE00, 0x059C, 0x1B76, 0xC602, 0xBE00, 0x065A, 0xB400, 0x18DE, 0x2008,
                0x4001, 0xF10F, 0x7342, 0x1880, 0x2008, 0x0009, 0x4018, 0xF109, 0x7340,
                0x25BC, 0x130F, 0xF105, 0xC00A, 0x7300, 0x4831, 0x9B00, 0xB000, 0x7340,
                0x8320, 0xC302, 0xBB00, 0x0C12, 0xE860, 0xC406, 0x7580, 0x4851, 0x8D80,
                0xC403, 0xBC00, 0xD3E0, 0x02C8, 0xC406, 0x7580, 0x4850, 0x8D80, 0xC403,
                0xBC00, 0xD3E0, 0x0298
        };

        re_mac_ocp_write(sc, 0xE43C, 0x0000);
        re_mac_ocp_write(sc, 0xE43E, 0x0000);

        re_mac_ocp_write(sc, 0xE434, 0x0004);
        re_mac_ocp_write(sc, 0xE43C, 0x0004);

        re_write_mac_mcu_ram_code(sc, mcu_patch_code, ARRAY_SIZE(mcu_patch_code));

        re_mac_ocp_write(sc, 0xDE30, 0x0080);

        re_mac_ocp_write(sc, 0xFC26, 0x8000);

        re_mac_ocp_write(sc, 0xFC28, 0x0075);
        re_mac_ocp_write(sc, 0xFC2A, 0x02B1);
        re_mac_ocp_write(sc, 0xFC2C, 0x0991);
        re_mac_ocp_write(sc, 0xFC2E, 0x059B);
        re_mac_ocp_write(sc, 0xFC30, 0x0659);
        re_mac_ocp_write(sc, 0xFC32, 0x0000);
        re_mac_ocp_write(sc, 0xFC34, 0x02C7);
        re_mac_ocp_write(sc, 0xFC36, 0x0279);
}

static void re_set_mac_mcu_8168gu_1(struct re_softc *sc)
{
        static const uint16_t mcu_patch_code[] = {
                0xE008, 0xE011, 0xE015, 0xE018, 0xE01B, 0xE027, 0xE043, 0xE065, 0x49E2,
                0xF005, 0x49EA, 0xF003, 0xC404, 0xBC00, 0xC403, 0xBC00, 0x0496, 0x051A,
                0x1D01, 0x8DE8, 0xC602, 0xBE00, 0x0206, 0x1B76, 0xC202, 0xBA00, 0x058A,
                0x1B76, 0xC602, 0xBE00, 0x0648, 0x74E6, 0x1B78, 0x46DC, 0x1300, 0xF005,
                0x74F8, 0x48C3, 0x48C4, 0x8CF8, 0x64E7, 0xC302, 0xBB00, 0x068E, 0x74E4,
                0x49C5, 0xF106, 0x49C6, 0xF107, 0x48C8, 0x48C9, 0xE011, 0x48C9, 0x4848,
                0xE00E, 0x4848, 0x49C7, 0xF00A, 0x48C9, 0xC60D, 0x1D1F, 0x8DC2, 0x1D00,
                0x8DC3, 0x1D11, 0x8DC0, 0xE002, 0x4849, 0x94E5, 0xC602, 0xBE00, 0x0238,
                0xE434, 0x49D9, 0xF01B, 0xC31E, 0x7464, 0x49C4, 0xF114, 0xC31B, 0x6460,
                0x14FA, 0xFA02, 0xE00F, 0xC317, 0x7460, 0x49C0, 0xF10B, 0xC311, 0x7462,
                0x48C1, 0x9C62, 0x4841, 0x9C62, 0xC30A, 0x1C04, 0x8C60, 0xE004, 0x1C15,
                0xC305, 0x8C60, 0xC602, 0xBE00, 0x0374, 0xE434, 0xE030, 0xE61C, 0xE906,
                0xC602, 0xBE00, 0x0000
        };

        re_write_mac_mcu_ram_code(sc, mcu_patch_code, ARRAY_SIZE(mcu_patch_code));

        re_mac_ocp_write(sc, 0xFC26, 0x8000);

        re_mac_ocp_write(sc, 0xFC28, 0x0493);
        re_mac_ocp_write(sc, 0xFC2A, 0x0205);
        re_mac_ocp_write(sc, 0xFC2C, 0x0589);
        re_mac_ocp_write(sc, 0xFC2E, 0x0647);
        re_mac_ocp_write(sc, 0xFC30, 0x0000);
        re_mac_ocp_write(sc, 0xFC32, 0x0215);
        re_mac_ocp_write(sc, 0xFC34, 0x0285);
}

static void re_set_mac_mcu_8168gu_2(struct re_softc *sc)
{
        static const uint16_t mcu_patch_code[] = {
                0xE008, 0xE00A, 0xE00D, 0xE02F, 0xE031, 0xE038, 0xE03A, 0xE051, 0xC202,
                0xBA00, 0x0DFC, 0x7444, 0xC502, 0xBD00, 0x0A30, 0x49D9, 0xF019, 0xC520,
                0x64A5, 0x1400, 0xF007, 0x0C01, 0x8CA5, 0x1C15, 0xC515, 0x9CA0, 0xE00F,
                0xC513, 0x74A0, 0x48C8, 0x48CA, 0x9CA0, 0xC510, 0x1B00, 0x9BA0, 0x1B1C,
                0x483F, 0x9BA2, 0x1B04, 0xC506, 0x9BA0, 0xC603, 0xBE00, 0x0298, 0x03DE,
                0xE434, 0xE096, 0xE860, 0xDE20, 0xD3C0, 0xC602, 0xBE00, 0x0A64, 0xC707,
                0x1D00, 0x8DE2, 0x48C1, 0xC502, 0xBD00, 0x00AA, 0xE0C0, 0xC502, 0xBD00,
                0x0132, 0xC50C, 0x74A2, 0x49CE, 0xF1FE, 0x1C00, 0x9EA0, 0x1C1C, 0x484F,
                0x9CA2, 0xC402, 0xBC00, 0x0AFA, 0xDE20, 0xE000, 0xE092, 0xE430, 0xDE20,
                0xE0C0, 0xE860, 0xE84C, 0xB400, 0xB430, 0xE410, 0xC0AE, 0xB407, 0xB406,
                0xB405, 0xB404, 0xB403, 0xB402, 0xB401, 0xC7EE, 0x76F4, 0xC2ED, 0xC3ED,
                0xC1EF, 0xC5F3, 0x74A0, 0x49CD, 0xF001, 0xC5EE, 0x74A0, 0x49C1, 0xF105,
                0xC5E4, 0x74A2, 0x49CE, 0xF00B, 0x7444, 0x484B, 0x9C44, 0x1C10, 0x9C62,
                0x1C11, 0x8C60, 0x1C00, 0x9CF6, 0xE0EC, 0x49E7, 0xF016, 0x1D80, 0x8DF4,
                0x74F8, 0x4843, 0x8CF8, 0x74F8, 0x74F8, 0x7444, 0x48C8, 0x48C9, 0x48CA,
                0x9C44, 0x74F8, 0x4844, 0x8CF8, 0x1E01, 0xE8DB, 0x7420, 0x48C1, 0x9C20,
                0xE0D5, 0x49E6, 0xF02A, 0x1D40, 0x8DF4, 0x74FC, 0x49C0, 0xF124, 0x49C1,
                0xF122, 0x74F8, 0x49C0, 0xF01F, 0xE8D3, 0x48C4, 0x8CF8, 0x1E00, 0xE8C6,
                0xC5B1, 0x74A0, 0x49C3, 0xF016, 0xC5AF, 0x74A4, 0x49C2, 0xF005, 0xC5AA,
                0x74B2, 0x49C9, 0xF10E, 0xC5A6, 0x74A8, 0x4845, 0x4846, 0x4847, 0x4848,
                0x9CA8, 0x74B2, 0x4849, 0x9CB2, 0x74A0, 0x484F, 0x9CA0, 0xE0AA, 0x49E4,
                0xF018, 0x1D10, 0x8DF4, 0x74F8, 0x74F8, 0x74F8, 0x4843, 0x8CF8, 0x74F8,
                0x74F8, 0x74F8, 0x4844, 0x4842, 0x4841, 0x8CF8, 0x1E01, 0xE89A, 0x7420,
                0x4841, 0x9C20, 0x7444, 0x4848, 0x9C44, 0xE091, 0x49E5, 0xF03E, 0x1D20,
                0x8DF4, 0x74F8, 0x48C2, 0x4841, 0x8CF8, 0x1E01, 0x7444, 0x49CA, 0xF103,
                0x49C2, 0xF00C, 0x49C1, 0xF004, 0x6447, 0x2244, 0xE002, 0x1C01, 0x9C62,
                0x1C11, 0x8C60, 0x1C00, 0x9CF6, 0x7444, 0x49C8, 0xF01D, 0x74FC, 0x49C0,
                0xF11A, 0x49C1, 0xF118, 0x74F8, 0x49C0, 0xF015, 0x49C6, 0xF113, 0xE875,
                0x48C4, 0x8CF8, 0x7420, 0x48C1, 0x9C20, 0xC50A, 0x74A2, 0x8CA5, 0x74A0,
                0xC505, 0x9CA2, 0x1C11, 0x9CA0, 0xE00A, 0xE434, 0xD3C0, 0xDC00, 0x7444,
                0x49CA, 0xF004, 0x48CA, 0x9C44, 0xE855, 0xE052, 0x49E8, 0xF024, 0x1D01,
                0x8DF5, 0x7440, 0x49C0, 0xF11E, 0x7444, 0x49C8, 0xF01B, 0x49CA, 0xF119,
                0xC5EC, 0x76A4, 0x49E3, 0xF015, 0x49C0, 0xF103, 0x49C1, 0xF011, 0x4849,
                0x9C44, 0x1C00, 0x9CF6, 0x7444, 0x49C1, 0xF004, 0x6446, 0x1E07, 0xE003,
                0x1C01, 0x1E03, 0x9C62, 0x1C11, 0x8C60, 0xE830, 0xE02D, 0x49E9, 0xF004,
                0x1D02, 0x8DF5, 0xE79C, 0x49E3, 0xF006, 0x1D08, 0x8DF4, 0x74F8, 0x74F8,
                0xE73A, 0x49E1, 0xF007, 0x1D02, 0x8DF4, 0x1E01, 0xE7A7, 0xDE20, 0xE410,
                0x49E0, 0xF017, 0x1D01, 0x8DF4, 0xC5FA, 0x1C00, 0x8CA0, 0x1C1B, 0x9CA2,
                0x74A2, 0x49CF, 0xF0FE, 0xC5F3, 0x74A0, 0x4849, 0x9CA0, 0x74F8, 0x49C0,
                0xF006, 0x48C3, 0x8CF8, 0xE820, 0x74F8, 0x74F8, 0xC432, 0xBC00, 0xC5E4,
                0x74A2, 0x49CE, 0xF1FE, 0x9EA0, 0x1C1C, 0x484F, 0x9CA2, 0xFF80, 0xB404,
                0xB405, 0xC5D9, 0x74A2, 0x49CE, 0xF1FE, 0xC41F, 0x9CA0, 0xC41C, 0x1C13,
                0x484F, 0x9CA2, 0x74A2, 0x49CF, 0xF1FE, 0xB005, 0xB004, 0xFF80, 0xB404,
                0xB405, 0xC5C7, 0x74A2, 0x49CE, 0xF1FE, 0xC40E, 0x9CA0, 0xC40A, 0x1C13,
                0x484F, 0x9CA2, 0x74A2, 0x49CF, 0xF1FE, 0xB005, 0xB004, 0xFF80, 0x0000,
                0x0481, 0x0C81, 0x0AE0
        };

        re_write_mac_mcu_ram_code(sc, mcu_patch_code, ARRAY_SIZE(mcu_patch_code));

        re_mac_ocp_write(sc, 0xFC26, 0x8000);

        re_mac_ocp_write(sc, 0xFC28, 0x0000);
        re_mac_ocp_write(sc, 0xFC2A, 0x0000);
        re_mac_ocp_write(sc, 0xFC2C, 0x0297);
        re_mac_ocp_write(sc, 0xFC2E, 0x0000);
        re_mac_ocp_write(sc, 0xFC30, 0x00A9);
        re_mac_ocp_write(sc, 0xFC32, 0x012D);
        re_mac_ocp_write(sc, 0xFC34, 0x0000);
        re_mac_ocp_write(sc, 0xFC36, 0x08DF);
}

static void re_set_mac_mcu_8168ep_1(struct re_softc *sc)
{
        static const uint16_t mcu_patch_code[] = {
                0xE008, 0xE0D3, 0xE0D6, 0xE0D9, 0xE0DB, 0xE0DD, 0xE0DF, 0xE0E1, 0xC251,
                0x7340, 0x49B1, 0xF010, 0x1D02, 0x8D40, 0xC202, 0xBA00, 0x2C3A, 0xC0F0,
                0xE8DE, 0x2000, 0x8000, 0xC0B6, 0x268C, 0x752C, 0x49D4, 0xF112, 0xE025,
                0xC2F6, 0x7146, 0xC2F5, 0x7340, 0x49BE, 0xF103, 0xC7F2, 0xE002, 0xC7F1,
                0x304F, 0x6226, 0x49A1, 0xF1F0, 0x7222, 0x49A0, 0xF1ED, 0x2525, 0x1F28,
                0x3097, 0x3091, 0x9A36, 0x752C, 0x21DC, 0x25BC, 0xC6E2, 0x77C0, 0x1304,
                0xF014, 0x1303, 0xF014, 0x1302, 0xF014, 0x1301, 0xF014, 0x49D4, 0xF103,
                0xC3D7, 0xBB00, 0xC618, 0x67C6, 0x752E, 0x22D7, 0x26DD, 0x1505, 0xF013,
                0xC60A, 0xBE00, 0xC309, 0xBB00, 0xC308, 0xBB00, 0xC307, 0xBB00, 0xC306,
                0xBB00, 0x25C8, 0x25A6, 0x25AC, 0x25B2, 0x25B8, 0xCD08, 0x0000, 0xC0BC,
                0xC2FF, 0x7340, 0x49B0, 0xF04E, 0x1F46, 0x308F, 0xC3F7, 0x1C04, 0xE84D,
                0x1401, 0xF147, 0x7226, 0x49A7, 0xF044, 0x7222, 0x2525, 0x1F30, 0x3097,
                0x3091, 0x7340, 0xC4EA, 0x401C, 0xF006, 0xC6E8, 0x75C0, 0x49D7, 0xF105,
                0xE036, 0x1D08, 0x8DC1, 0x0208, 0x6640, 0x2764, 0x1606, 0xF12F, 0x6346,
                0x133B, 0xF12C, 0x9B34, 0x1B18, 0x3093, 0xC32A, 0x1C10, 0xE82A, 0x1401,
                0xF124, 0x1A36, 0x308A, 0x7322, 0x25B5, 0x0B0E, 0x1C00, 0xE82C, 0xC71F,
                0x4027, 0xF11A, 0xE838, 0x1F42, 0x308F, 0x1B08, 0xE824, 0x7236, 0x7746,
                0x1700, 0xF00D, 0xC313, 0x401F, 0xF103, 0x1F00, 0x9F46, 0x7744, 0x449F,
                0x445F, 0xE817, 0xC70A, 0x4027, 0xF105, 0xC302, 0xBB00, 0x2E08, 0x2DC2,
                0xC7FF, 0xBF00, 0xCDB8, 0xFFFF, 0x0C02, 0xA554, 0xA5DC, 0x402F, 0xF105,
                0x1400, 0xF1FA, 0x1C01, 0xE002, 0x1C00, 0xFF80, 0x49B0, 0xF004, 0x0B01,
                0xA1D3, 0xE003, 0x0B02, 0xA5D3, 0x3127, 0x3720, 0x0B02, 0xA5D3, 0x3127,
                0x3720, 0x1300, 0xF1FB, 0xFF80, 0x7322, 0x25B5, 0x1E28, 0x30DE, 0x30D9,
                0x7264, 0x1E11, 0x2368, 0x3116, 0xFF80, 0x1B7E, 0xC602, 0xBE00, 0x06A6,
                0x1B7E, 0xC602, 0xBE00, 0x0764, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00,
                0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00,
                0x0000
        };

        re_write_mac_mcu_ram_code(sc, mcu_patch_code, ARRAY_SIZE(mcu_patch_code));

        re_mac_ocp_write(sc, 0xFC26, 0x8000);

        re_mac_ocp_write(sc, 0xFC28, 0x2549);
        re_mac_ocp_write(sc, 0xFC2A, 0x06A5);
        re_mac_ocp_write(sc, 0xFC2C, 0x0763);
}

static bool re_check_dash_other_fun_present(struct re_softc *sc)
{
        //check if func 2 exist
        if (re_csi_other_fun_read(sc, 2, 0xf000) != 0xffffffff)
                return true;

        return false;
}

static void re_set_mac_mcu_8168ep_2(struct re_softc *sc)
{
        static const uint16_t mcu_patch_code[] = {
                0xE008, 0xE017, 0xE052, 0xE056, 0xE058, 0xE05A, 0xE05C, 0xE05E, 0xC50F,
                0x76A4, 0x49E3, 0xF007, 0x49C0, 0xF103, 0xC607, 0xBE00, 0xC606, 0xBE00,
                0xC602, 0xBE00, 0x0BDA, 0x0BB6, 0x0BBA, 0xDC00, 0xB400, 0xB401, 0xB402,
                0xB403, 0xB404, 0xC02E, 0x7206, 0x49AE, 0xF1FE, 0xC12B, 0x9904, 0xC12A,
                0x9906, 0x7206, 0x49AE, 0xF1FE, 0x7200, 0x49A0, 0xF117, 0xC123, 0xC223,
                0xC323, 0xE808, 0xC322, 0xE806, 0xC321, 0xE804, 0xC320, 0xE802, 0xE00C,
                0x740E, 0x49CE, 0xF1FE, 0x9908, 0x990A, 0x9A0C, 0x9B0E, 0x740E, 0x49CE,
                0xF1FE, 0xFF80, 0xB004, 0xB003, 0xB002, 0xB001, 0xB000, 0xC604, 0xC002,
                0xB800, 0x1FC8, 0xE000, 0xE8E0, 0xF128, 0x0002, 0xFFFF, 0xF000, 0x8001,
                0x8002, 0x8003, 0x8004, 0x48C1, 0x48C2, 0xC502, 0xBD00, 0x0490, 0xC602,
                0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000, 0xC602,
                0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000
        };

        re_write_mac_mcu_ram_code(sc, mcu_patch_code, ARRAY_SIZE(mcu_patch_code));

        re_mac_ocp_write(sc, 0xFC26, 0x8000);

        re_mac_ocp_write(sc, 0xFC28, 0x0BB3);
        if (false == re_check_dash_other_fun_present(sc))
                re_mac_ocp_write(sc, 0xFC2A, 0x1FC7);
        //re_mac_ocp_write(sc, 0xFC2C, 0x0485);
}

static void re_set_mac_mcu_8411b_1(struct re_softc *sc)
{
        static const uint16_t mcu_patch_code[] = {
                0xE008, 0xE00A, 0xE00C, 0xE00E, 0xE027, 0xE04F, 0xE05E, 0xE065, 0xC602,
                0xBE00, 0x0000, 0xC502, 0xBD00, 0x074C, 0xC302, 0xBB00, 0x080A, 0x6420,
                0x48C2, 0x8C20, 0xC516, 0x64A4, 0x49C0, 0xF009, 0x74A2, 0x8CA5, 0x74A0,
                0xC50E, 0x9CA2, 0x1C11, 0x9CA0, 0xE006, 0x74F8, 0x48C4, 0x8CF8, 0xC404,
                0xBC00, 0xC403, 0xBC00, 0x0BF2, 0x0C0A, 0xE434, 0xD3C0, 0x49D9, 0xF01F,
                0xC526, 0x64A5, 0x1400, 0xF007, 0x0C01, 0x8CA5, 0x1C15, 0xC51B, 0x9CA0,
                0xE013, 0xC519, 0x74A0, 0x48C4, 0x8CA0, 0xC516, 0x74A4, 0x48C8, 0x48CA,
                0x9CA4, 0xC512, 0x1B00, 0x9BA0, 0x1B1C, 0x483F, 0x9BA2, 0x1B04, 0xC508,
                0x9BA0, 0xC505, 0xBD00, 0xC502, 0xBD00, 0x0300, 0x051E, 0xE434, 0xE018,
                0xE092, 0xDE20, 0xD3C0, 0xC50F, 0x76A4, 0x49E3, 0xF007, 0x49C0, 0xF103,
                0xC607, 0xBE00, 0xC606, 0xBE00, 0xC602, 0xBE00, 0x0C4C, 0x0C28, 0x0C2C,
                0xDC00, 0xC707, 0x1D00, 0x8DE2, 0x48C1, 0xC502, 0xBD00, 0x00AA, 0xE0C0,
                0xC502, 0xBD00, 0x0132
        };

        re_write_mac_mcu_ram_code(sc, mcu_patch_code, ARRAY_SIZE(mcu_patch_code));

        re_mac_ocp_write(sc, 0xFC26, 0x8000);

        re_mac_ocp_write(sc, 0xFC2A, 0x0743);
        re_mac_ocp_write(sc, 0xFC2C, 0x0801);
        re_mac_ocp_write(sc, 0xFC2E, 0x0BE9);
        re_mac_ocp_write(sc, 0xFC30, 0x02FD);
        re_mac_ocp_write(sc, 0xFC32, 0x0C25);
        re_mac_ocp_write(sc, 0xFC34, 0x00A9);
        re_mac_ocp_write(sc, 0xFC36, 0x012D);
}

static void re_set_mac_mcu_8168h_2(struct re_softc *sc)
{
        static const uint16_t mcu_patch_code[] = {
                0xE008, 0xE00F, 0xE011, 0xE047, 0xE049, 0xE073, 0xE075, 0xE07A, 0xC707,
                0x1D00, 0x8DE2, 0x48C1, 0xC502, 0xBD00, 0x00E4, 0xE0C0, 0xC502, 0xBD00,
                0x0216, 0xC634, 0x75C0, 0x49D3, 0xF027, 0xC631, 0x75C0, 0x49D3, 0xF123,
                0xC627, 0x75C0, 0xB405, 0xC525, 0x9DC0, 0xC621, 0x75C8, 0x49D5, 0xF00A,
                0x49D6, 0xF008, 0x49D7, 0xF006, 0x49D8, 0xF004, 0x75D2, 0x49D9, 0xF111,
                0xC517, 0x9DC8, 0xC516, 0x9DD2, 0xC618, 0x75C0, 0x49D4, 0xF003, 0x49D0,
                0xF104, 0xC60A, 0xC50E, 0x9DC0, 0xB005, 0xC607, 0x9DC0, 0xB007, 0xC602,
                0xBE00, 0x1A06, 0xB400, 0xE86C, 0xA000, 0x01E1, 0x0200, 0x9200, 0xE84C,
                0xE004, 0xE908, 0xC502, 0xBD00, 0x0B58, 0xB407, 0xB404, 0x2195, 0x25BD,
                0x9BE0, 0x1C1C, 0x484F, 0x9CE2, 0x72E2, 0x49AE, 0xF1FE, 0x0B00, 0xF116,
                0xC71C, 0xC419, 0x9CE0, 0x1C13, 0x484F, 0x9CE2, 0x74E2, 0x49CE, 0xF1FE,
                0xC412, 0x9CE0, 0x1C13, 0x484F, 0x9CE2, 0x74E2, 0x49CE, 0xF1FE, 0xC70C,
                0x74F8, 0x48C3, 0x8CF8, 0xB004, 0xB007, 0xC502, 0xBD00, 0x0F24, 0x0481,
                0x0C81, 0xDE24, 0xE000, 0xC602, 0xBE00, 0x0CA4, 0x48C1, 0x48C2, 0x9C46,
                0xC402, 0xBC00, 0x0578, 0xC602, 0xBE00, 0x0000
        };

        re_write_mac_mcu_ram_code(sc, mcu_patch_code, ARRAY_SIZE(mcu_patch_code));

        re_mac_ocp_write(sc, 0xFC26, 0x8000);

        re_mac_ocp_write(sc, 0xFC28, 0x00E2);
        re_mac_ocp_write(sc, 0xFC2A, 0x0210);
        re_mac_ocp_write(sc, 0xFC2C, 0x1A04);
        re_mac_ocp_write(sc, 0xFC2E, 0x0B26);
        re_mac_ocp_write(sc, 0xFC30, 0x0F02);
        re_mac_ocp_write(sc, 0xFC32, 0x0CA0);
        //re_mac_ocp_write(sc, 0xFC34, 0x056C);

        if (sc->re_device_id == RT_DEVICEID_8136)
                re_mac_ocp_write(sc, 0xFC38, 0x0033);
        else
                re_mac_ocp_write(sc, 0xFC38, 0x003F);
}

static void re_set_mac_mcu_8168h_3(struct re_softc *sc)
{
        static const uint16_t mcu_patch_code[] = {
                0xE008, 0xE00A, 0xE00C, 0xE00E, 0xE010, 0xE03E, 0xE040, 0xE069, 0xC602,
                0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000, 0xC002, 0xB800, 0x0000, 0xC602,
                0xBE00, 0x0000, 0xC72B, 0x76E2, 0x49EE, 0xF1FD, 0x1E00, 0x9EE0, 0x1E1C,
                0x9EE2, 0x76E2, 0x49EE, 0xF1FE, 0xC621, 0x9EE0, 0x1E1D, 0x486F, 0x9EE2,
                0x76E2, 0x49EE, 0xF1FE, 0xC71A, 0x76E0, 0x48E8, 0x48E9, 0x48EA, 0x48EB,
                0x48EC, 0x9EE0, 0xC70D, 0xC60D, 0x9EF4, 0xC60C, 0x9EF6, 0xC70E, 0x76E0,
                0x4863, 0x9EE0, 0xB007, 0xC602, 0xBE00, 0x0ACC, 0xE000, 0x03BF, 0x07FF,
                0xDE24, 0x3200, 0xE096, 0xD438, 0xC602, 0xBE00, 0x0000, 0x8EE6, 0xC726,
                0x76E2, 0x49EE, 0xF1FD, 0x1E00, 0x8EE0, 0x1E1C, 0x8EE2, 0x76E2, 0x49EE,
                0xF1FE, 0xC61C, 0x8EE0, 0x1E1D, 0x486F, 0x8EE2, 0x76E2, 0x49EE, 0xF1FE,
                0xC715, 0x76E0, 0x48E8, 0x48E9, 0x48EA, 0x48EB, 0x48EC, 0x9EE0, 0xC708,
                0xC608, 0x9EF4, 0xC607, 0x9EF6, 0xC602, 0xBE00, 0x0ABE, 0xE000, 0x03BF,
                0x07FF, 0xDE24, 0x3200, 0xE096, 0xC602, 0xBE00, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x6838, 0x0A17, 0x0613, 0x0D26
        };

        re_write_mac_mcu_ram_code(sc, mcu_patch_code, ARRAY_SIZE(mcu_patch_code));

        re_mac_ocp_write(sc, 0xFC26, 0x8000);

        re_mac_ocp_write(sc, 0xFC30, 0x0ACA);

        re_mac_ocp_write(sc, 0xFC38, 0x0010);
}

static void re_set_mac_mcu_8168fp_1(struct re_softc *sc)
{
        uint16_t breakPointEnabled = 0;

        if(sc->HwPkgDet == 0x00 || sc->HwPkgDet == 0x0F) {
                static const uint16_t mcu_patch_code[] = {
                        0xE00A, 0xE0C1, 0xE104, 0xE108, 0xE10D, 0xE112, 0xE11C, 0xE121, 0xE000,
                        0xE0C8, 0xB400, 0xC1FE, 0x49E2, 0xF04C, 0x49EA, 0xF04A, 0x74E6, 0xC246,
                        0x7542, 0x73EC, 0x1800, 0x49C0, 0xF10D, 0x49C1, 0xF10B, 0x49C2, 0xF109,
                        0x49B0, 0xF107, 0x49B1, 0xF105, 0x7220, 0x49A2, 0xF102, 0xE002, 0x4800,
                        0x49D0, 0xF10A, 0x49D1, 0xF108, 0x49D2, 0xF106, 0x49D3, 0xF104, 0x49DF,
                        0xF102, 0xE00C, 0x4801, 0x72E4, 0x49AD, 0xF108, 0xC225, 0x6741, 0x48F0,
                        0x8F41, 0x4870, 0x8F41, 0xC7CF, 0x49B5, 0xF01F, 0x49B2, 0xF00B, 0x4980,
                        0xF003, 0x484E, 0x94E7, 0x4981, 0xF004, 0x485E, 0xC212, 0x9543, 0xE071,
                        0x49B6, 0xF003, 0x49B3, 0xF10F, 0x4980, 0xF003, 0x484E, 0x94E7, 0x4981,
                        0xF004, 0x485E, 0xC204, 0x9543, 0xE005, 0xE000, 0xE0FC, 0xE0FA, 0xE065,
                        0x49B7, 0xF007, 0x4980, 0xF005, 0x1A38, 0x46D4, 0x1200, 0xF109, 0x4981,
                        0xF055, 0x49C3, 0xF105, 0x1A30, 0x46D5, 0x1200, 0xF04F, 0x7220, 0x49A2,
                        0xF130, 0x49C1, 0xF12E, 0x49B0, 0xF12C, 0xC2E6, 0x7240, 0x49A8, 0xF003,
                        0x49D0, 0xF126, 0x49A9, 0xF003, 0x49D1, 0xF122, 0x49AA, 0xF003, 0x49D2,
                        0xF11E, 0x49AB, 0xF003, 0x49DF, 0xF11A, 0x49AC, 0xF003, 0x49D3, 0xF116,
                        0x4980, 0xF003, 0x49C7, 0xF105, 0x4981, 0xF02C, 0x49D7, 0xF02A, 0x49C0,
                        0xF00C, 0xC721, 0x62F4, 0x49A0, 0xF008, 0x49A4, 0xF106, 0x4824, 0x8AF4,
                        0xC71A, 0x1A40, 0x9AE0, 0x49B6, 0xF017, 0x200E, 0xC7B8, 0x72E0, 0x4710,
                        0x92E1, 0xC70E, 0x77E0, 0x49F0, 0xF112, 0xC70B, 0x77E0, 0x27FE, 0x1AFA,
                        0x4317, 0xC705, 0x9AE2, 0x1A11, 0x8AE0, 0xE008, 0xE41C, 0xC0AE, 0xD23A,
                        0xC7A2, 0x74E6, 0x484F, 0x94E7, 0xC79E, 0x8CE6, 0x8BEC, 0xC29C, 0x8D42,
                        0x7220, 0xB000, 0xC502, 0xBD00, 0x0932, 0xB400, 0xC240, 0xC340, 0x7060,
                        0x498F, 0xF014, 0x488F, 0x9061, 0x744C, 0x49C3, 0xF004, 0x7562, 0x485E,
                        0x9563, 0x7446, 0x49C3, 0xF106, 0x7562, 0x1C30, 0x46E5, 0x1200, 0xF004,
                        0x7446, 0x484F, 0x9447, 0xC32A, 0x7466, 0x49C0, 0xF00F, 0x48C0, 0x9C66,
                        0x7446, 0x4840, 0x4841, 0x4842, 0x9C46, 0x744C, 0x4840, 0x9C4C, 0x744A,
                        0x484A, 0x9C4A, 0xE013, 0x498E, 0xF011, 0x488E, 0x9061, 0x744C, 0x49C3,
                        0xF004, 0x7446, 0x484E, 0x9447, 0x7446, 0x1D38, 0x46EC, 0x1500, 0xF004,
                        0x7446, 0x484F, 0x9447, 0xB000, 0xC502, 0xBD00, 0x074C, 0xE000, 0xE0FC,
                        0xE0C0, 0x4830, 0x4837, 0xC502, 0xBD00, 0x0978, 0x63E2, 0x4830, 0x4837,
                        0xC502, 0xBD00, 0x09FE, 0x73E2, 0x4830, 0x8BE2, 0xC302, 0xBB00, 0x0A12,
                        0x73E2, 0x48B0, 0x48B3, 0x48B4, 0x48B5, 0x48B6, 0x48B7, 0x8BE2, 0xC302,
                        0xBB00, 0x0A5A, 0x73E2, 0x4830, 0x8BE2, 0xC302, 0xBB00, 0x0A6C, 0x73E2,
                        0x4830, 0x4837, 0xC502, 0xBD00, 0x0A86
                };

                re_write_mac_mcu_ram_code(sc, mcu_patch_code, ARRAY_SIZE(mcu_patch_code));

                re_mac_ocp_write(sc, 0xFC26, 0x8000);

                re_mac_ocp_write(sc, 0xFC28, 0x0890);
                re_mac_ocp_write(sc, 0xFC2A, 0x0712);
                re_mac_ocp_write(sc, 0xFC2C, 0x0974);
                re_mac_ocp_write(sc, 0xFC2E, 0x09FC);
                re_mac_ocp_write(sc, 0xFC30, 0x0A0E);
                re_mac_ocp_write(sc, 0xFC32, 0x0A56);
                re_mac_ocp_write(sc, 0xFC34, 0x0A68);
                re_mac_ocp_write(sc, 0xFC36, 0x0A84);

        } else if (sc->HwPkgDet == 0x05 || sc->HwPkgDet == 0x06) {
                static const uint16_t mcu_patch_code[] = {
                        0xE008, 0xE00A, 0xE031, 0xE033, 0xE035, 0xE144, 0xE166, 0xE168, 0xC502,
                        0xBD00, 0x0000, 0xC725, 0x75E0, 0x48D0, 0x9DE0, 0xC722, 0x75E0, 0x1C78,
                        0x416C, 0x1530, 0xF111, 0xC71D, 0x75F6, 0x49D1, 0xF00D, 0x75E0, 0x1C1F,
                        0x416C, 0x1502, 0xF108, 0x75FA, 0x49D3, 0xF005, 0x75EC, 0x9DE4, 0x4853,
                        0x9DFA, 0xC70B, 0x75E0, 0x4852, 0x4850, 0x9DE0, 0xC602, 0xBE00, 0x04B8,
                        0xE420, 0xE000, 0xE0FC, 0xE43C, 0xDC00, 0xEB00, 0xC202, 0xBA00, 0x0000,
                        0xC002, 0xB800, 0x0000, 0xB401, 0xB402, 0xB403, 0xB404, 0xB405, 0xB406,
                        0xC44D, 0xC54D, 0x1867, 0xE8A2, 0x2318, 0x276E, 0x1601, 0xF106, 0x1A07,
                        0xE861, 0xE86B, 0xE873, 0xE037, 0x231E, 0x276E, 0x1602, 0xF10B, 0x1A07,
                        0xE858, 0xE862, 0xC247, 0xC344, 0xE8E3, 0xC73B, 0x66E0, 0xE8B5, 0xE029,
                        0x231A, 0x276C, 0xC733, 0x9EE0, 0x1866, 0xE885, 0x251C, 0x120F, 0xF011,
                        0x1209, 0xF011, 0x2014, 0x240E, 0x1000, 0xF007, 0x120C, 0xF00D, 0x1203,
                        0xF00D, 0x1200, 0xF00D, 0x120C, 0xF00D, 0x1203, 0xF00D, 0x1A03, 0xE00C,
                        0x1A07, 0xE00A, 0x1A00, 0xE008, 0x1A01, 0xE006, 0x1A02, 0xE004, 0x1A04,
                        0xE002, 0x1A05, 0xE829, 0xE833, 0xB006, 0xB005, 0xB004, 0xB003, 0xB002,
                        0xB001, 0x60C4, 0xC702, 0xBF00, 0x2786, 0xDD00, 0xD030, 0xE0C4, 0xE0F8,
                        0xDC42, 0xD3F0, 0x0000, 0x0004, 0x0007, 0x0014, 0x0090, 0x1000, 0x0F00,
                        0x1004, 0x1008, 0x3000, 0x3004, 0x3008, 0x4000, 0x7777, 0x8000, 0x8001,
                        0x8008, 0x8003, 0x8004, 0xC000, 0xC004, 0xF004, 0xFFFF, 0xB406, 0xB407,
                        0xC6E5, 0x77C0, 0x27F3, 0x23F3, 0x47FA, 0x9FC0, 0xB007, 0xB006, 0xFF80,
                        0xB405, 0xB407, 0xC7D8, 0x75E0, 0x48D0, 0x9DE0, 0xB007, 0xB005, 0xFF80,
                        0xB401, 0xC0EA, 0xC2DC, 0xC3D8, 0xE865, 0xC0D3, 0xC1E0, 0xC2E3, 0xE861,
                        0xE817, 0xC0CD, 0xC2CF, 0xE85D, 0xC0C9, 0xC1D6, 0xC2DB, 0xE859, 0xE80F,
                        0xC1C7, 0xC2CE, 0xE855, 0xC0C0, 0xC1D1, 0xC2D3, 0xE851, 0xE807, 0xC0BE,
                        0xC2C2, 0xE84D, 0xE803, 0xB001, 0xFF80, 0xB402, 0xC2C6, 0xE859, 0x499F,
                        0xF1FE, 0xB002, 0xFF80, 0xB402, 0xB403, 0xB407, 0xE821, 0x8882, 0x1980,
                        0x8983, 0xE81D, 0x7180, 0x218B, 0x25BB, 0x1310, 0xF014, 0x1310, 0xFB03,
                        0x1F20, 0x38FB, 0x3288, 0x434B, 0x2491, 0x430B, 0x1F0F, 0x38FB, 0x4313,
                        0x2121, 0x4353, 0x2521, 0x418A, 0x6282, 0x2527, 0x212F, 0x418A, 0xB007,
                        0xB003, 0xB002, 0xFF80, 0x6183, 0x2496, 0x1100, 0xF1FD, 0xFF80, 0x4800,
                        0x4801, 0xC213, 0xC313, 0xE815, 0x4860, 0x8EE0, 0xC210, 0xC310, 0xE822,
                        0x481E, 0xC20C, 0xC30C, 0xE80C, 0xC206, 0x7358, 0x483A, 0x9B58, 0xFF80,
                        0xE8E0, 0xE000, 0x1008, 0x0F00, 0x800C, 0x0F00, 0xB407, 0xB406, 0xB403,
                        0xC7F7, 0x98E0, 0x99E2, 0x9AE4, 0x21B2, 0x4831, 0x483F, 0x9BE6, 0x66E7,
                        0x49E6, 0xF1FE, 0xB003, 0xB006, 0xB007, 0xFF80, 0xB407, 0xB406, 0xB403,
                        0xC7E5, 0x9AE4, 0x21B2, 0x4831, 0x9BE6, 0x66E7, 0x49E6, 0xF1FE, 0x70E0,
                        0x71E2, 0xB003, 0xB006, 0xB007, 0xFF80, 0x4882, 0xB406, 0xB405, 0xC71E,
                        0x76E0, 0x1D78, 0x4175, 0x1630, 0xF10C, 0xC715, 0x76E0, 0x4861, 0x9EE0,
                        0xC713, 0x1EFF, 0x9EE2, 0x75E0, 0x4850, 0x9DE0, 0xE005, 0xC70B, 0x76E0,
                        0x4865, 0x9EE0, 0xB005, 0xB006, 0xC708, 0xC102, 0xB900, 0x279E, 0xEB16,
                        0xEB00, 0xE43C, 0xDC00, 0xD3EC, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00,
                        0x0000
                };

                re_write_mac_mcu_ram_code(sc, mcu_patch_code, ARRAY_SIZE(mcu_patch_code));

                re_mac_ocp_write(sc, 0xFC26, 0x8000);

                re_mac_ocp_write(sc, 0xFC28, 0x0000);
                re_mac_ocp_write(sc, 0xFC2A, 0x04b4);
                re_mac_ocp_write(sc, 0xFC2C, 0x0000);
                re_mac_ocp_write(sc, 0xFC2E, 0x0000);
                re_mac_ocp_write(sc, 0xFC30, 0x0000);
                re_mac_ocp_write(sc, 0xFC32, 0x279C);
                re_mac_ocp_write(sc, 0xFC34, 0x0000);
                re_mac_ocp_write(sc, 0xFC36, 0x0000);
        }

        if (sc->HwPkgDet == 0x00)
                breakPointEnabled = 0x00FC;
        else if (sc->HwPkgDet == 0x0F)
                breakPointEnabled = 0x00FF;
        else if (sc->HwPkgDet == 0x05 || sc->HwPkgDet == 0x06)
                breakPointEnabled = 0x0022;

        re_mac_ocp_write(sc, 0xFC38, breakPointEnabled);
}

static void re_set_mac_mcu_8168fp_8116as_2(struct re_softc *sc)
{
        static const uint16_t mcu_patch_code[] = {
                0xE008, 0xE00A, 0xE00F, 0xE014, 0xE016, 0xE018, 0xE01A, 0xE01C, 0xC602,
                0xBE00, 0x2AB2, 0x1BC0, 0x46EB, 0x1BFE, 0xC102, 0xB900, 0x0B1A, 0x1BC0,
                0x46EB, 0x1B7E, 0xC102, 0xB900, 0x0BEA, 0xC602, 0xBE00, 0x0000, 0xC602,
                0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000, 0xC602,
                0xBE00, 0x0000
        };

        re_write_mac_mcu_ram_code(sc, mcu_patch_code, ARRAY_SIZE(mcu_patch_code));

        re_mac_ocp_write(sc, 0xFC26, 0x8000);

        re_mac_ocp_write(sc, 0xFC28, 0x2AAC);
        re_mac_ocp_write(sc, 0xFC2A, 0x0B14);
        re_mac_ocp_write(sc, 0xFC2C, 0x0BE4);

        re_mac_ocp_write(sc, 0xFC38, 0x0007);
}

static void _re_set_mac_mcu_8168fp_2(struct re_softc *sc)
{
        static const uint16_t mcu_patch_code[] = {
                0xE008, 0xE00A, 0xE00F, 0xE014, 0xE05F, 0xE064, 0xE066, 0xE068, 0xC602,
                0xBE00, 0x0000, 0x1BC0, 0x46EB, 0x1BFE, 0xC102, 0xB900, 0x0B1A, 0x1BC0,
                0x46EB, 0x1B7E, 0xC102, 0xB900, 0x0BEA, 0xB400, 0xB401, 0xB402, 0xB403,
                0xB404, 0xB405, 0xC03A, 0x7206, 0x49AE, 0xF1FE, 0xC137, 0x9904, 0xC136,
                0x9906, 0x7206, 0x49AE, 0xF1FE, 0x7200, 0x49A0, 0xF10B, 0xC52F, 0xC12E,
                0xC232, 0xC332, 0xE812, 0xC331, 0xE810, 0xC330, 0xE80E, 0xE018, 0xC126,
                0xC229, 0xC525, 0xC328, 0xE808, 0xC523, 0xC326, 0xE805, 0xC521, 0xC324,
                0xE802, 0xE00C, 0x740E, 0x49CE, 0xF1FE, 0x9908, 0x9D0A, 0x9A0C, 0x9B0E,
                0x740E, 0x49CE, 0xF1FE, 0xFF80, 0xB005, 0xB004, 0xB003, 0xB002, 0xB001,
                0xB000, 0xC604, 0xC002, 0xB800, 0x2A5E, 0xE000, 0xE8E0, 0xF128, 0x3DC2,
                0xFFFF, 0x10EC, 0x816A, 0x816D, 0x816C, 0xF000, 0x8002, 0x8004, 0x8007,
                0x48C1, 0x48C2, 0x9C46, 0xC402, 0xBC00, 0x07BC, 0xC602, 0xBE00, 0x0000,
                0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000
        };

        re_write_mac_mcu_ram_code(sc, mcu_patch_code, ARRAY_SIZE(mcu_patch_code));

        re_mac_ocp_write(sc, 0xFC26, 0x8000);

        re_mac_ocp_write(sc, 0xFC28, 0x2AAC);
        re_mac_ocp_write(sc, 0xFC2A, 0x0B14);
        re_mac_ocp_write(sc, 0xFC2C, 0x0BE4);
        re_mac_ocp_write(sc, 0xFC2E, 0x2A5C);
        //re_mac_ocp_write(sc, 0xFC30, 0x07B0);

        if (true == re_check_dash_other_fun_present(sc))
                re_mac_ocp_write(sc, 0xFC38, 0x0006);
        else
                re_mac_ocp_write(sc, 0xFC38, 0x000E);
}

static void re_set_mac_mcu_8168fp_2(struct re_softc *sc)
{
        if (sc->hw_hw_supp_serdes_phy_ver == 1)
                re_set_mac_mcu_8168fp_8116as_2(sc);
        else
                _re_set_mac_mcu_8168fp_2(sc);
}

static void re_set_mac_mcu_8168fp_3(struct re_softc *sc)
{
        static const uint16_t mcu_patch_code[] = {
                0xE008, 0xE053, 0xE058, 0xE05A, 0xE05C, 0xE05E, 0xE060, 0xE062, 0xB400,
                0xB401, 0xB402, 0xB403, 0xB404, 0xB405, 0xC03A, 0x7206, 0x49AE, 0xF1FE,
                0xC137, 0x9904, 0xC136, 0x9906, 0x7206, 0x49AE, 0xF1FE, 0x7200, 0x49A0,
                0xF10B, 0xC52F, 0xC12E, 0xC232, 0xC332, 0xE812, 0xC331, 0xE810, 0xC330,
                0xE80E, 0xE018, 0xC126, 0xC229, 0xC525, 0xC328, 0xE808, 0xC523, 0xC326,
                0xE805, 0xC521, 0xC324, 0xE802, 0xE00C, 0x740E, 0x49CE, 0xF1FE, 0x9908,
                0x9D0A, 0x9A0C, 0x9B0E, 0x740E, 0x49CE, 0xF1FE, 0xFF80, 0xB005, 0xB004,
                0xB003, 0xB002, 0xB001, 0xB000, 0xC604, 0xC002, 0xB800, 0x2B16, 0xE000,
                0xE8E0, 0xF128, 0x3DC2, 0xFFFF, 0x10EC, 0x816A, 0x816D, 0x816C, 0xF000,
                0x8002, 0x8004, 0x8007, 0x48C1, 0x48C2, 0x9C46, 0xC402, 0xBC00, 0x07BC,
                0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000,
                0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000
        };

        re_write_mac_mcu_ram_code(sc, mcu_patch_code, ARRAY_SIZE(mcu_patch_code));

        re_mac_ocp_write(sc, 0xFC26, 0x8000);

        re_mac_ocp_write(sc, 0xFC28, 0x2B14);
        //re_mac_ocp_write(sc, 0xFC2A, 0x07B0);

        if (true == re_check_dash_other_fun_present(sc))
                re_mac_ocp_write(sc, 0xFC38, 0x0000);
        else
                re_mac_ocp_write(sc, 0xFC38, 0x0001);
}

static void re_set_mac_mcu_8125a_2(struct re_softc *sc)
{
        static const u_int16_t mcu_patch_code[] = {
                0xE010, 0xE012, 0xE022, 0xE024, 0xE029, 0xE02B, 0xE094, 0xE09D, 0xE09F,
                0xE0AA, 0xE0B5, 0xE0C6, 0xE0CC, 0xE0D1, 0xE0D6, 0xE0D8, 0xC602, 0xBE00,
                0x0000, 0xC60F, 0x73C4, 0x49B3, 0xF106, 0x73C2, 0xC608, 0xB406, 0xC609,
                0xFF80, 0xC605, 0xB406, 0xC605, 0xFF80, 0x0544, 0x0568, 0xE906, 0xCDE8,
                0xC602, 0xBE00, 0x0000, 0x48C1, 0x48C2, 0x9C46, 0xC402, 0xBC00, 0x0A12,
                0xC602, 0xBE00, 0x0EBA, 0x1501, 0xF02A, 0x1500, 0xF15D, 0xC661, 0x75C8,
                0x49D5, 0xF00A, 0x49D6, 0xF008, 0x49D7, 0xF006, 0x49D8, 0xF004, 0x75D2,
                0x49D9, 0xF150, 0xC553, 0x77A0, 0x75C8, 0x4855, 0x4856, 0x4857, 0x4858,
                0x48DA, 0x48DB, 0x49FE, 0xF002, 0x485A, 0x49FF, 0xF002, 0x485B, 0x9DC8,
                0x75D2, 0x4859, 0x9DD2, 0xC643, 0x75C0, 0x49D4, 0xF033, 0x49D0, 0xF137,
                0xE030, 0xC63A, 0x75C8, 0x49D5, 0xF00E, 0x49D6, 0xF00C, 0x49D7, 0xF00A,
                0x49D8, 0xF008, 0x75D2, 0x49D9, 0xF005, 0xC62E, 0x75C0, 0x49D7, 0xF125,
                0xC528, 0x77A0, 0xC627, 0x75C8, 0x4855, 0x4856, 0x4857, 0x4858, 0x48DA,
                0x48DB, 0x49FE, 0xF002, 0x485A, 0x49FF, 0xF002, 0x485B, 0x9DC8, 0x75D2,
                0x4859, 0x9DD2, 0xC616, 0x75C0, 0x4857, 0x9DC0, 0xC613, 0x75C0, 0x49DA,
                0xF003, 0x49D0, 0xF107, 0xC60B, 0xC50E, 0x48D9, 0x9DC0, 0x4859, 0x9DC0,
                0xC608, 0xC702, 0xBF00, 0x3AE0, 0xE860, 0xB400, 0xB5D4, 0xE908, 0xE86C,
                0x1200, 0xC409, 0x6780, 0x48F1, 0x8F80, 0xC404, 0xC602, 0xBE00, 0x10AA,
                0xC010, 0xEA7C, 0xC602, 0xBE00, 0x0000, 0x740A, 0x4846, 0x4847, 0x9C0A,
                0xC607, 0x74C0, 0x48C6, 0x9CC0, 0xC602, 0xBE00, 0x13FE, 0xE054, 0x72CA,
                0x4826, 0x4827, 0x9ACA, 0xC607, 0x72C0, 0x48A6, 0x9AC0, 0xC602, 0xBE00,
                0x07DC, 0xE054, 0xC60F, 0x74C4, 0x49CC, 0xF109, 0xC60C, 0x74CA, 0x48C7,
                0x9CCA, 0xC609, 0x74C0, 0x4846, 0x9CC0, 0xC602, 0xBE00, 0x2480, 0xE092,
                0xE0C0, 0xE054, 0x7420, 0x48C0, 0x9C20, 0x7444, 0xC602, 0xBE00, 0x12F8,
                0x1BFF, 0x46EB, 0x1BFF, 0xC102, 0xB900, 0x0D5A, 0x1BFF, 0x46EB, 0x1BFF,
                0xC102, 0xB900, 0x0E2A, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6486,
                0x0B15, 0x090E, 0x1139
        };

        re_write_mac_mcu_ram_code(sc, mcu_patch_code, ARRAY_SIZE(mcu_patch_code));

        re_mac_ocp_write(sc, 0xFC26, 0x8000);

        re_mac_ocp_write(sc, 0xFC2A, 0x0540);
        re_mac_ocp_write(sc, 0xFC2E, 0x0A06);
        re_mac_ocp_write(sc, 0xFC30, 0x0EB8);
        re_mac_ocp_write(sc, 0xFC32, 0x3A5C);
        re_mac_ocp_write(sc, 0xFC34, 0x10A8);
        re_mac_ocp_write(sc, 0xFC40, 0x0D54);
        re_mac_ocp_write(sc, 0xFC42, 0x0E24);

        re_mac_ocp_write(sc, 0xFC48, 0x307A);
}

static void re_set_mac_mcu_8125b_2(struct re_softc *sc)
{
        static const u_int16_t mcu_patch_code[] = {
                0xE010, 0xE01B, 0xE026, 0xE037, 0xE03D, 0xE057, 0xE05B, 0xE060, 0xE062,
                0xE064, 0xE066, 0xE068, 0xE06A, 0xE06C, 0xE06E, 0xE070, 0x740A, 0x4846,
                0x4847, 0x9C0A, 0xC607, 0x74C0, 0x48C6, 0x9CC0, 0xC602, 0xBE00, 0x13F0,
                0xE054, 0x72CA, 0x4826, 0x4827, 0x9ACA, 0xC607, 0x72C0, 0x48A6, 0x9AC0,
                0xC602, 0xBE00, 0x081C, 0xE054, 0xC60F, 0x74C4, 0x49CC, 0xF109, 0xC60C,
                0x74CA, 0x48C7, 0x9CCA, 0xC609, 0x74C0, 0x4846, 0x9CC0, 0xC602, 0xBE00,
                0x2494, 0xE092, 0xE0C0, 0xE054, 0x7420, 0x48C0, 0x9C20, 0x7444, 0xC602,
                0xBE00, 0x12DC, 0x733A, 0x21B5, 0x25BC, 0x1304, 0xF111, 0x1B12, 0x1D2A,
                0x3168, 0x3ADA, 0x31AB, 0x1A00, 0x9AC0, 0x1300, 0xF1FB, 0x7620, 0x236E,
                0x276F, 0x1A3C, 0x22A1, 0x41B5, 0x9EE2, 0x76E4, 0x486F, 0x9EE4, 0xC602,
                0xBE00, 0x4A26, 0x733A, 0x49BB, 0xC602, 0xBE00, 0x47A2, 0x48C1, 0x48C2,
                0x9C46, 0xC402, 0xBC00, 0x0A52, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00,
                0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00,
                0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00,
                0x0000, 0xC602, 0xBE00, 0x0000
        };

        re_write_mac_mcu_ram_code(sc, mcu_patch_code, ARRAY_SIZE(mcu_patch_code));

        re_mac_ocp_write(sc, 0xFC26, 0x8000);

        re_mac_ocp_write(sc, 0xFC28, 0x13E6);
        re_mac_ocp_write(sc, 0xFC2A, 0x0812);
        re_mac_ocp_write(sc, 0xFC2C, 0x248C);
        re_mac_ocp_write(sc, 0xFC2E, 0x12DA);
        re_mac_ocp_write(sc, 0xFC30, 0x4A20);
        re_mac_ocp_write(sc, 0xFC32, 0x47A0);
        //re_mac_ocp_write(sc, 0xFC34, 0x0A46);

        re_mac_ocp_write(sc, 0xFC48, 0x003F);
}

static void re_set_mac_mcu_8125bp_1(struct re_softc *sc)
{
        static const u_int16_t mcu_patch_code[] = {
                0xE010, 0xE014, 0xE027, 0xE04A, 0xE04D, 0xE050, 0xE052, 0xE054, 0xE056,
                0xE058, 0xE05A, 0xE05C, 0xE05E, 0xE060, 0xE062, 0xE064, 0x1BC8, 0x46EB,
                0xC302, 0xBB00, 0x0F14, 0xC211, 0x400A, 0xF00A, 0xC20F, 0x400A, 0xF007,
                0x73A4, 0xC20C, 0x400A, 0xF102, 0x48B0, 0x9B20, 0x1B00, 0x9BA0, 0xC602,
                0xBE00, 0x4364, 0xE6E0, 0xE6E2, 0xC01C, 0xB406, 0x1000, 0xF016, 0xC61F,
                0x400E, 0xF012, 0x218E, 0x25BE, 0x1300, 0xF007, 0x7340, 0xC618, 0x400E,
                0xF102, 0x48B0, 0x8320, 0xB400, 0x2402, 0x1000, 0xF003, 0x7342, 0x8322,
                0xB000, 0xE007, 0x7322, 0x9B42, 0x7320, 0x9B40, 0x0300, 0x0300, 0xB006,
                0xC302, 0xBB00, 0x413E, 0xE6E0, 0xC01C, 0x49D1, 0xC602, 0xBE00, 0x3F94,
                0x49D1, 0xC602, 0xBE00, 0x4030, 0xC602, 0xBE00, 0x3FDA, 0xC102, 0xB900,
                0x401A, 0xC102, 0xB900, 0x0000, 0xC002, 0xB800, 0x0000, 0xC602, 0xBE00,
                0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00,
                0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00,
                0x0000, 0x6936, 0x0A18, 0x0C02, 0x0D21
        };

        /* Get BIN mac mcu patch code version */
        sc->BinMcuPatchCodeVer = re_get_bin_mcu_patch_code_ver(mcu_patch_code, ARRAY_SIZE(mcu_patch_code));

        if (sc->HwMcuPatchCodeVer != sc->BinMcuPatchCodeVer)
                re_write_mac_mcu_ram_code(sc, mcu_patch_code, ARRAY_SIZE(mcu_patch_code));

        re_mac_ocp_write(sc, 0xFC26, 0x8000);

        re_mac_ocp_write(sc, 0xFC28, 0x0f10);
        re_mac_ocp_write(sc, 0xFC2A, 0x435c);
        re_mac_ocp_write(sc, 0xFC2C, 0x4112);
        re_mac_ocp_write(sc, 0xFC2E, 0x3F92);
        re_mac_ocp_write(sc, 0xFC30, 0x402E);
        re_mac_ocp_write(sc, 0xFC32, 0x3FD6);
        re_mac_ocp_write(sc, 0xFC34, 0x4018);

        re_mac_ocp_write(sc, 0xFC48, 0x007F);
}

static void re_set_mac_mcu_8125bp_2(struct re_softc *sc)
{
        static const u_int16_t mcu_patch_code[] = {
                0xE010, 0xE033, 0xE046, 0xE04A, 0xE04D, 0xE050, 0xE052, 0xE054, 0xE056,
                0xE058, 0xE05A, 0xE05C, 0xE05E, 0xE060, 0xE062, 0xE064, 0xB406, 0x1000,
                0xF016, 0xC61F, 0x400E, 0xF012, 0x218E, 0x25BE, 0x1300, 0xF007, 0x7340,
                0xC618, 0x400E, 0xF102, 0x48B0, 0x8320, 0xB400, 0x2402, 0x1000, 0xF003,
                0x7342, 0x8322, 0xB000, 0xE007, 0x7322, 0x9B42, 0x7320, 0x9B40, 0x0300,
                0x0300, 0xB006, 0xC302, 0xBB00, 0x4168, 0xE6E0, 0xC01C, 0xC211, 0x400A,
                0xF00A, 0xC20F, 0x400A, 0xF007, 0x73A4, 0xC20C, 0x400A, 0xF102, 0x48B0,
                0x9B20, 0x1B00, 0x9BA0, 0xC602, 0xBE00, 0x4392, 0xE6E0, 0xE6E2, 0xC01C,
                0x4166, 0x9CF6, 0xC002, 0xB800, 0x143C, 0x49D1, 0xC602, 0xBE00, 0x3FC4,
                0x49D1, 0xC602, 0xBE00, 0x405A, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00,
                0x0000, 0xC102, 0xB900, 0x0000, 0xC002, 0xB800, 0x0000, 0xC602, 0xBE00,
                0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00,
                0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00,
                0x0000, 0x6936, 0x0B18, 0x0C02, 0x0D22
        };

        /* Get BIN mac mcu patch code version */
        sc->BinMcuPatchCodeVer = re_get_bin_mcu_patch_code_ver(mcu_patch_code, ARRAY_SIZE(mcu_patch_code));

        if (sc->HwMcuPatchCodeVer != sc->BinMcuPatchCodeVer)
                re_write_mac_mcu_ram_code(sc, mcu_patch_code, ARRAY_SIZE(mcu_patch_code));

        re_mac_ocp_write(sc, 0xFC26, 0x8000);

        re_mac_ocp_write(sc, 0xFC28, 0x413C);
        re_mac_ocp_write(sc, 0xFC2A, 0x438A);
        re_mac_ocp_write(sc, 0xFC2C, 0x143A);
        re_mac_ocp_write(sc, 0xFC2E, 0x3FC2);
        re_mac_ocp_write(sc, 0xFC30, 0x4058);

        re_mac_ocp_write(sc, 0xFC48, 0x001F);
}

static void re_set_mac_mcu_8125d_1(struct re_softc *sc)
{
        static const u_int16_t mcu_patch_code[] = {
                0xE002, 0xE006, 0x4166, 0x9CF6, 0xC002, 0xB800, 0x14A4, 0xC102, 0xB900,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
                0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x6938,
                0x0A18, 0x0217, 0x0D2A
        };

        /* Get BIN mac mcu patch code version */
        sc->BinMcuPatchCodeVer = re_get_bin_mcu_patch_code_ver(mcu_patch_code, ARRAY_SIZE(mcu_patch_code));

        if (sc->HwMcuPatchCodeVer != sc->BinMcuPatchCodeVer)
                re_write_mac_mcu_ram_code(sc, mcu_patch_code, ARRAY_SIZE(mcu_patch_code));

        re_mac_ocp_write(sc, 0xFC26, 0x8000);

        re_mac_ocp_write(sc, 0xFC28, 0x14A2);

        re_mac_ocp_write(sc, 0xFC48, 0x0001);
}

static void re_set_mac_mcu_8126a_1(struct re_softc *sc)
{
        static const u_int16_t mcu_patch_code[] = {
                0xE010, 0xE019, 0xE01B, 0xE01D, 0xE01F, 0xE021, 0xE023, 0xE025, 0xE027,
                0xE029, 0xE02B, 0xE02D, 0xE02F, 0xE031, 0xE033, 0xE035, 0x48C0, 0x9C66,
                0x7446, 0x4840, 0x48C1, 0x48C2, 0x9C46, 0xC402, 0xBC00, 0x0AD6, 0xC602,
                0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000, 0xC602,
                0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000, 0xC602,
                0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000, 0xC602,
                0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000, 0xC602,
                0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000
        };

        re_write_mac_mcu_ram_code(sc, mcu_patch_code, ARRAY_SIZE(mcu_patch_code));

        re_mac_ocp_write(sc, 0xFC26, 0x8000);

        re_mac_ocp_write(sc, 0xFC28, 0x0AAA);

        re_mac_ocp_write(sc, 0xFC48, 0x0001);
}

static void re_set_mac_mcu_8126a_2(struct re_softc *sc)
{
        static const u_int16_t mcu_patch_code[] = {
                0xE010, 0xE02C, 0xE04E, 0xE0A4, 0xE0A8, 0xE0AB, 0xE0AE, 0xE0B1, 0xE0B3,
                0xE0B5, 0xE0B7, 0xE0B9, 0xE0BB, 0xE0BD, 0xE0BF, 0xE0C1, 0xC716, 0xC616,
                0x9EE0, 0xC616, 0x65C0, 0x1500, 0xF009, 0xC714, 0x66E0, 0x41B5, 0x8EE0,
                0xC611, 0x75C0, 0x4858, 0x9DC0, 0xC707, 0xC608, 0x9EE0, 0xC608, 0xC502,
                0xBD00, 0x0100, 0xE86C, 0xE000, 0xA000, 0xB404, 0xB430, 0xC070, 0xE926,
                0xC2FE, 0x400A, 0xF11A, 0x63A4, 0x1A00, 0x49B0, 0xF002, 0x4820, 0x49B1,
                0xF002, 0x4821, 0x49B2, 0xF002, 0x4822, 0x49B3, 0xF002, 0x4823, 0xC411,
                0x6380, 0x48B0, 0x8B80, 0x6320, 0x41DA, 0x8B20, 0x6380, 0x4830, 0x8B80,
                0xE003, 0x73A4, 0x9B20, 0xC302, 0xBB00, 0x4A18, 0xC070, 0xE022, 0xC054,
                0x7102, 0x4992, 0xF149, 0x4893, 0x9902, 0x1B1F, 0xC74E, 0x72E0, 0x2521,
                0x48A5, 0x0B01, 0x1C4F, 0x9C00, 0x2121, 0x1D01, 0x41AA, 0x2521, 0x9DE0,
                0x4856, 0x9DE0, 0x1CCF, 0xE839, 0x48D6, 0x9DE0, 0x7102, 0x4996, 0xF1FE,
                0x4814, 0x9902, 0x1CFF, 0x0C01, 0x1400, 0xF00C, 0x7102, 0x4996, 0xF0FB,
                0x7102, 0x4990, 0xF0FE, 0x1C1F, 0xE826, 0x7102, 0x4992, 0xF004, 0x4813,
                0x9902, 0xE01D, 0x1300, 0xF104, 0x4817, 0x9902, 0xE018, 0x4894, 0x9902,
                0x4995, 0xF00B, 0x121F, 0xF0F3, 0x131E, 0xF003, 0x4998, 0xF0EF, 0x0201,
                0x4818, 0x9902, 0xE7C9, 0x1200, 0xF0E9, 0x4998, 0xF002, 0x1B01, 0x0A01,
                0x4898, 0x9902, 0xE7C0, 0xC00A, 0xC606, 0xBE00, 0x0C01, 0x1400, 0xF1FE,
                0xFF80, 0x2362, 0xD456, 0xD404, 0xE400, 0x4166, 0x9CF6, 0xC002, 0xB800,
                0x14A6, 0x49D1, 0xC602, 0xBE00, 0x4160, 0x49D1, 0xC602, 0xBE00, 0x41E6,
                0x49D1, 0xC602, 0xBE00, 0x4282, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00,
                0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00,
                0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00,
                0x0000, 0xC602, 0xBE00, 0x0000, 0x6847, 0x0A18, 0x0C02, 0x0B30
        };

        /* Get BIN mac mcu patch code version */
        sc->BinMcuPatchCodeVer = re_get_bin_mcu_patch_code_ver(mcu_patch_code, ARRAY_SIZE(mcu_patch_code));

        if (sc->HwMcuPatchCodeVer != sc->BinMcuPatchCodeVer)
                re_write_mac_mcu_ram_code(sc, mcu_patch_code, ARRAY_SIZE(mcu_patch_code));

        re_mac_ocp_write(sc, 0xFC26, 0x8000);

        //re_mac_ocp_write(sc, 0xFC28, 0x00FE);
        //re_mac_ocp_write(sc, 0xFC2A, 0x4A14);
        re_mac_ocp_write(sc, 0xFC2C, 0x2360);
        re_mac_ocp_write(sc, 0xFC2E, 0x14A4);
        re_mac_ocp_write(sc, 0xFC30, 0x415E);
        re_mac_ocp_write(sc, 0xFC32, 0x41E4);
        re_mac_ocp_write(sc, 0xFC34, 0x4280);

        re_mac_ocp_write(sc, 0xFC48, 0x007C);
}

static void re_set_mac_mcu_8126a_3(struct re_softc *sc)
{
        static const u_int16_t mcu_patch_code[] = {
                0xE010, 0xE02C, 0xE04E, 0xE052, 0xE055, 0xE058, 0xE05B, 0xE05D, 0xE05F,
                0xE061, 0xE063, 0xE065, 0xE067, 0xE069, 0xE06B, 0xE06D, 0xC716, 0xC616,
                0x9EE0, 0xC616, 0x65C0, 0x1500, 0xF009, 0xC714, 0x66E0, 0x41B5, 0x8EE0,
                0xC611, 0x75C0, 0x4858, 0x9DC0, 0xC707, 0xC608, 0x9EE0, 0xC608, 0xC502,
                0xBD00, 0x0100, 0xE86C, 0xE000, 0xA000, 0xB404, 0xB430, 0xC070, 0xE926,
                0xC2FE, 0x400A, 0xF11A, 0x63A4, 0x1A00, 0x49B0, 0xF002, 0x4820, 0x49B1,
                0xF002, 0x4821, 0x49B2, 0xF002, 0x4822, 0x49B3, 0xF002, 0x4823, 0xC411,
                0x6380, 0x48B0, 0x8B80, 0x6320, 0x41DA, 0x8B20, 0x6380, 0x4830, 0x8B80,
                0xE003, 0x73A4, 0x9B20, 0xC302, 0xBB00, 0x55E2, 0xC070, 0xE022, 0x4166,
                0x9CF6, 0xC602, 0xBE00, 0x14A6, 0x49D1, 0xC602, 0xBE00, 0x4178, 0x49D1,
                0xC602, 0xBE00, 0x41FE, 0x49D1, 0xC602, 0xBE00, 0x429A, 0xC602, 0xBE00,
                0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00,
                0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00,
                0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00,
                0x0000, 0x6847, 0x0B18, 0x0C02, 0x0D10
        };

        /* Get BIN mac mcu patch code version */
        sc->BinMcuPatchCodeVer = re_get_bin_mcu_patch_code_ver(mcu_patch_code, ARRAY_SIZE(mcu_patch_code));

        if (sc->HwMcuPatchCodeVer != sc->BinMcuPatchCodeVer)
                re_write_mac_mcu_ram_code(sc, mcu_patch_code, ARRAY_SIZE(mcu_patch_code));

        re_mac_ocp_write(sc, 0xFC26, 0x8000);

        //re_mac_ocp_write(sc, 0xFC28, 0x00FE);
        //re_mac_ocp_write(sc, 0xFC2A, 0x55DE);
        re_mac_ocp_write(sc, 0xFC2C, 0x14A4);
        re_mac_ocp_write(sc, 0xFC2E, 0x4176);
        re_mac_ocp_write(sc, 0xFC30, 0x41FC);
        re_mac_ocp_write(sc, 0xFC32, 0x4298);

        re_mac_ocp_write(sc, 0xFC48, 0x003C);
}

static void re_set_mac_mcu_8127a_tc_1(struct re_softc *sc)
{
        static const u_int16_t mcu_patch_code[] = {
                0xE010, 0xE019, 0xE01B, 0xE01D, 0xE029, 0xE02C, 0xE0E1, 0xE192, 0xE194,
                0xE196, 0xE198, 0xE19A, 0xE19C, 0xE19E, 0xE1A0, 0xE1A2, 0xC008, 0x7100,
                0x4897, 0x9900, 0xC005, 0xC602, 0xBE00, 0x3D8E, 0xD428, 0xD400, 0xC602,
                0xBE00, 0x0000, 0xC502, 0xBD00, 0x0000, 0x48C3, 0x4847, 0x48C1, 0x8CF8,
                0x74F8, 0x74F8, 0x74F8, 0x4842, 0x8CF8, 0x1E10, 0xC502, 0xBD00, 0x14BA,
                0x1E10, 0xC502, 0xBD00, 0x14EE, 0xC643, 0x76C0, 0x49E1, 0xF13F, 0xC140,
                0x7720, 0x49E0, 0xF003, 0x1B00, 0xE00A, 0x49E2, 0xF003, 0x1B04, 0xE006,
                0x49E4, 0xF003, 0x1B08, 0xE002, 0x1B0C, 0x21B8, 0x1A0E, 0x44DA, 0xE893,
                0x481C, 0xE884, 0xE001, 0x49E0, 0xF003, 0x1B00, 0xE00A, 0x49E2, 0xF003,
                0x1B04, 0xE006, 0x49E4, 0xF003, 0x1B08, 0xE002, 0x1B0C, 0x21B8, 0x1A12,
                0x44DA, 0xE87F, 0x481F, 0xE870, 0xE001, 0x49E0, 0xF003, 0x1B00, 0xE00A,
                0x49E2, 0xF003, 0x1B04, 0xE006, 0x49E4, 0xF003, 0x1B08, 0xE002, 0x1B0C,
                0x21B8, 0x1A1C, 0x44DA, 0xE86B, 0x481F, 0xE85C, 0xE004, 0xE04F, 0xDD98,
                0xD450, 0x49E0, 0xF003, 0x1B00, 0xE00A, 0x49E2, 0xF003, 0x1B04, 0xE006,
                0x49E4, 0xF003, 0x1B08, 0xE002, 0x1B0C, 0x21B8, 0x1A0E, 0x44DA, 0xE854,
                0x489E, 0x481F, 0xE844, 0xE001, 0x1908, 0xE83E, 0x49E0, 0xF003, 0x1B00,
                0xE00A, 0x49E2, 0xF003, 0x1B04, 0xE006, 0x49E4, 0xF003, 0x1B08, 0xE002,
                0x1B0C, 0x21B8, 0x1A8A, 0x44DA, 0xE83D, 0x4813, 0xE82E, 0x49F9, 0xF106,
                0x4838, 0xE837, 0x4813, 0xE828, 0xE001, 0x49E0, 0xF003, 0x1B00, 0xE00A,
                0x49E2, 0xF003, 0x1B04, 0xE006, 0x49E4, 0xF003, 0x1B08, 0xE002, 0x1B0C,
                0x21B8, 0x1A84, 0x44DA, 0xE823, 0x4890, 0x4811, 0xE813, 0x49F9, 0xF106,
                0x4838, 0xE81C, 0x4890, 0x4811, 0xE80C, 0xC207, 0x7440, 0xC602, 0xBE00,
                0x14CC, 0x0FFE, 0xDE20, 0xE092, 0xC3FD, 0xE802, 0xFF80, 0xC0FB, 0x7202,
                0x49AE, 0xF1FE, 0x9900, 0x44D3, 0x4413, 0x482F, 0x9A02, 0x7202, 0x49AE,
                0xF1FE, 0xFF80, 0xC0EE, 0x7202, 0x49AE, 0xF1FE, 0x44D3, 0x4413, 0x48AF,
                0x9A02, 0x7202, 0x49AE, 0xF1FE, 0x7100, 0xFF80, 0xB401, 0xB402, 0xB404,
                0xB407, 0xC61F, 0x76C0, 0x49E1, 0xF164, 0xC11C, 0x7720, 0x1906, 0xE88A,
                0x1B0C, 0x21B8, 0x1A40, 0x44DA, 0xE895, 0x4810, 0xE886, 0x190C, 0xE881,
                0x1B08, 0x21B8, 0x1A26, 0x44DA, 0xE88C, 0x4890, 0x4891, 0xE87C, 0x49F9,
                0xF107, 0x4898, 0x4899, 0xE877, 0xE003, 0xDD98, 0xD450, 0x1908, 0xE86F,
                0x49E0, 0xF003, 0x1B00, 0xE00A, 0x49E2, 0xF003, 0x1B04, 0xE006, 0x49E2,
                0xF003, 0x1B08, 0xE002, 0x1B0C, 0x21B8, 0x1A5C, 0x44DA, 0xE86E, 0x4897,
                0x4898, 0x4819, 0x481A, 0xE85C, 0x49F9, 0xF109, 0x4838, 0xE865, 0x4897,
                0x4898, 0x4819, 0x481A, 0xE853, 0xE001, 0x190A, 0xE84D, 0x1B00, 0xE85B,
                0x44E1, 0x4838, 0xE858, 0x44E9, 0x1908, 0xE845, 0x49E0, 0xF003, 0x1B00,
                0xE00A, 0x49E2, 0xF003, 0x1B04, 0xE006, 0x49E4, 0xF003, 0x1B08, 0xE002,
                0x1B0C, 0x21B8, 0x1A86, 0x44DA, 0xE844, 0x44CC, 0xE835, 0x49F9, 0xF108,
                0x4838, 0xE83E, 0x44CD, 0xE82F, 0xE003, 0xE021, 0xFFC0, 0x190A, 0xE827,
                0x1B00, 0x4839, 0xE834, 0x249A, 0x1C00, 0x44E1, 0x1909, 0xE81F, 0x49E0,
                0xF003, 0x1B00, 0xE00A, 0x49E2, 0xF003, 0x1B04, 0xE006, 0x49E4, 0xF003,
                0x1B08, 0xE002, 0x1B0C, 0x21B8, 0x1A1A, 0x44DA, 0xE81E, 0xC5E4, 0x414D,
                0x418C, 0xE80D, 0xB007, 0xB004, 0xB002, 0xB001, 0xC602, 0xBE00, 0x14B2,
                0x0FFE, 0xDE20, 0xC3FE, 0xE802, 0xFF80, 0xC0FC, 0x7202, 0x49AE, 0xF1FE,
                0x9900, 0x44D3, 0x4413, 0x482F, 0x9A02, 0x7202, 0x49AE, 0xF1FE, 0xFF80,
                0xC0EF, 0x7202, 0x49AE, 0xF1FE, 0x44D3, 0x4413, 0x48AF, 0x9A02, 0x7202,
                0x49AE, 0xF1FE, 0x7100, 0xFF80, 0xC502, 0xBD00, 0x0000, 0xC502, 0xBD00,
                0x0000, 0xC502, 0xBD00, 0x0000, 0xC302, 0xBB00, 0x0000, 0xC602, 0xBE00,
                0x0000, 0xC102, 0xB900, 0x0000, 0xC102, 0xB900, 0x0000, 0xC602, 0xBE00,
                0x0000, 0xC602, 0xBE00, 0x0000, 0x1332, 0x0018, 0x0C05, 0x140D
        };

        /* Get BIN mac mcu patch code version */
        sc->BinMcuPatchCodeVer = re_get_bin_mcu_patch_code_ver(mcu_patch_code, ARRAY_SIZE(mcu_patch_code));

        if (sc->HwMcuPatchCodeVer != sc->BinMcuPatchCodeVer)
                re_write_mac_mcu_ram_code(sc, mcu_patch_code, ARRAY_SIZE(mcu_patch_code));

        re_mac_ocp_write(sc, 0xFC26, 0x8000);

        re_mac_ocp_write(sc, 0xFC2E, 0x14B8);
        re_mac_ocp_write(sc, 0xFC30, 0x14EC);

        re_mac_ocp_write(sc, 0xFC48, 0x0018);
}

static void _re_set_mac_mcu_8127a_1(struct re_softc *sc)
{
        static const u_int16_t mcu_patch_code[] = {
                0xE010, 0xE014, 0xE018, 0xE01C, 0xE020, 0xE033, 0xE035, 0xE037, 0xE039,
                0xE03B, 0xE03D, 0xE03F, 0xE041, 0xE043, 0xE045, 0xE047, 0x7020, 0x4809,
                0xC502, 0xBD00, 0x1522, 0x7760, 0x4879, 0xC002, 0xB800, 0x41E2, 0x7160,
                0x4819, 0xC302, 0xBB00, 0x508E, 0x7720, 0x4879, 0xC102, 0xB900, 0x50F8,
                0x9F86, 0xB400, 0xB401, 0xB402, 0xB403, 0xC00D, 0x7100, 0xC20C, 0x7340,
                0x418B, 0x9900, 0xB003, 0xB002, 0xB001, 0xB000, 0xC702, 0xBF00, 0x3550,
                0xFC48, 0xD482, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000, 0xC102,
                0xB900, 0x0000, 0xC302, 0xBB00, 0x0000, 0xC002, 0xB800, 0x0000, 0xC002,
                0xB800, 0x0000, 0xC502, 0xBD00, 0x0000, 0xC102, 0xB900, 0x0000, 0xC102,
                0xB900, 0x0000, 0xC602, 0xBE00, 0x0000, 0xC602, 0xBE00, 0x0000, 0x6961,
                0x0019, 0x0311, 0x1431
        };

        /* Get BIN mac mcu patch code version */
        sc->BinMcuPatchCodeVer = re_get_bin_mcu_patch_code_ver(mcu_patch_code, ARRAY_SIZE(mcu_patch_code));

        if (sc->HwMcuPatchCodeVer != sc->BinMcuPatchCodeVer)
                re_write_mac_mcu_ram_code(sc, mcu_patch_code, ARRAY_SIZE(mcu_patch_code));

        re_mac_ocp_write(sc, 0xFC26, 0x8000);

        re_mac_ocp_write(sc, 0xFC28, 0x1520);
        re_mac_ocp_write(sc, 0xFC2A, 0x41E0);
        re_mac_ocp_write(sc, 0xFC2C, 0x508C);
        re_mac_ocp_write(sc, 0xFC2E, 0x50F6);
        re_mac_ocp_write(sc, 0xFC30, 0x354E);

        re_mac_ocp_write(sc, 0xFC48, 0x001F);
}

static void re_set_mac_mcu_8127a_1(struct re_softc *sc)
{
        u_int8_t tmpUchar;

        tmpUchar = (u_int8_t)re_mac_ocp_read(sc, 0xD006);

        if (tmpUchar != 0x04)
                return;

        _re_set_mac_mcu_8127a_1(sc);
}

static void re_hw_mac_mcu_config(struct re_softc *sc)
{
        sc->HwMcuPatchCodeVer = re_get_hw_mcu_patch_code_ver(sc);

        re_disable_mcu_bps(sc);

        switch(sc->re_type) {
        case MACFG_56:
                re_set_mac_mcu_8168g_1(sc);
                break;
        case MACFG_58:
                re_set_mac_mcu_8168gu_1(sc);
                break;
        case MACFG_59:
                re_set_mac_mcu_8168gu_2(sc);
                break;
        case MACFG_60:
                re_set_mac_mcu_8411b_1(sc);
                break;
        case MACFG_62:
                re_set_mac_mcu_8168ep_1(sc);
                break;
        case MACFG_67:
                re_set_mac_mcu_8168ep_2(sc);
                break;
        case MACFG_69:
        case MACFG_76:
                re_set_mac_mcu_8168h_2(sc);
                break;
        case MACFG_70:
                re_set_mac_mcu_8168fp_1(sc);
                break;
        case MACFG_71:
                re_set_mac_mcu_8168fp_2(sc);
                break;
        case MACFG_72:
                re_set_mac_mcu_8168fp_3(sc);
                break;
        case MACFG_74:
                re_set_mac_mcu_8168h_3(sc);
                break;
        case MACFG_81:
                re_set_mac_mcu_8125a_2(sc);
                break;
        case MACFG_83:
                re_set_mac_mcu_8125b_2(sc);
                break;
        case MACFG_84:
                re_set_mac_mcu_8125bp_1(sc);
                break;
        case MACFG_85:
                re_set_mac_mcu_8125bp_2(sc);
                break;
        case MACFG_86:
                re_set_mac_mcu_8125d_1(sc);
                break;
        case MACFG_90:
                re_set_mac_mcu_8126a_1(sc);
                break;
        case MACFG_91:
                re_set_mac_mcu_8126a_2(sc);
                break;
        case MACFG_92:
                re_set_mac_mcu_8126a_3(sc);
                break;
        case MACFG_100:
                re_set_mac_mcu_8127a_tc_1(sc);
                break;
        case MACFG_101:
                re_set_mac_mcu_8127a_1(sc);
                break;
        case MACFG_68:
        case MACFG_73:
        case MACFG_75:
        case MACFG_80:
        case MACFG_82:
        case MACFG_87:
        case MACFG_88:
                /* no mac mcu patch code */
                break;
        default:
                break;
        }
}

#define ISRIMR_DASH_TYPE2_TX_DISABLE_IDLE BIT_5
static void Dash2DisableTx(struct re_softc *sc)
{
        u_int16_t WaitCnt;
        u_int8_t TmpUchar;

        if (!HW_DASH_SUPPORT_CMAC(sc))
                return;

        //Disable oob Tx
        RE_CMAC_WRITE_1(sc, RE_CMAC_IBCR2, RE_CMAC_READ_1(sc, RE_CMAC_IBCR2) & ~(BIT_0));
        WaitCnt = 0;

        //wait oob tx disable
        do {
                TmpUchar = RE_CMAC_READ_1(sc, RE_CMAC_IBISR0);

                if (TmpUchar & ISRIMR_DASH_TYPE2_TX_DISABLE_IDLE) {
                        break;
                }

                DELAY(50);
                WaitCnt++;
        } while(WaitCnt < 2000);

        //Clear ISRIMR_DASH_TYPE2_TX_DISABLE_IDLE
        RE_CMAC_WRITE_1(sc, RE_CMAC_IBISR0, RE_CMAC_READ_1(sc, RE_CMAC_IBISR0) | ISRIMR_DASH_TYPE2_TX_DISABLE_IDLE);
}

static void Dash2DisableRx(struct re_softc *sc)
{
        if (!HW_DASH_SUPPORT_CMAC(sc))
                return;

        RE_CMAC_WRITE_1(sc, RE_CMAC_IBCR0, RE_CMAC_READ_1(sc, RE_CMAC_IBCR0) & ~(BIT_0));
}

static void Dash2DisableTxRx(struct re_softc *sc)
{
        if (!HW_DASH_SUPPORT_CMAC(sc))
                return;

        Dash2DisableTx(sc);
        Dash2DisableRx(sc);
}

static inline bool
is_zero_ether_addr(const u_int8_t * addr)
{
        return ((addr[0] + addr[1] + addr[2] + addr[3] + addr[4] + addr[5]) == 0x00);
}

static inline bool
is_multicast_ether_addr(const u_int8_t * addr)
{
        return (0x01 & addr[0]);
}

/*
static inline bool
is_broadcast_ether_addr(const u_int8_t * addr)
{
        return ((addr[0] + addr[1] + addr[2] + addr[3] + addr[4] + addr[5]) == (6 * 0xff));
}
*/

static inline bool
is_valid_ether_addr(const u_int8_t * addr)
{
        return !is_multicast_ether_addr(addr) && !is_zero_ether_addr(addr);
}

static void re_disable_now_is_oob(struct re_softc *sc)
{
        if (sc->re_hw_supp_now_is_oob_ver == 1)
                CSR_WRITE_1(sc, RE_MCU_CMD, CSR_READ_1(sc, RE_MCU_CMD) & ~RE_NOW_IS_OOB);
}

static void re_switch_to_sgmii_mode(struct re_softc *sc)
{
        if (FALSE == HW_SUPP_SERDES_PHY(sc)) return;

        switch (sc->hw_hw_supp_serdes_phy_ver) {
        case 1:
                re_mac_ocp_write(sc, 0xEB00, 0x2);
                re_set_mac_ocp_bit(sc, 0xEB16, BIT_1);
                break;
        }
}

static void
re_enable_magic_packet(struct re_softc *sc)
{
        if (sc->re_if_flags & RL_FLAG_MAGIC_PACKET_V3)
                re_set_mac_ocp_bit(sc, 0xC0B6, BIT_0);
        else if (sc->re_if_flags & RL_FLAG_MAGIC_PACKET_V2)
                re_eri_write(sc, 0xDC, 4,
                             re_eri_read(sc, 0xDC, 4, ERIAR_ExGMAC) | BIT_16,
                             ERIAR_ExGMAC);
        else
                CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) | RL_CFG3_WOL_MAGIC);
}

static void
re_disable_magic_packet(struct re_softc *sc)
{
        if (sc->re_if_flags & RL_FLAG_MAGIC_PACKET_V3)
                re_clear_mac_ocp_bit(sc, 0xC0B6, BIT_0);
        else if (sc->re_if_flags & RL_FLAG_MAGIC_PACKET_V2)
                re_eri_write(sc, 0xDC, 4,
                             re_eri_read(sc, 0xDC, 4, ERIAR_ExGMAC) & ~BIT_16,
                             ERIAR_ExGMAC);
        else
                CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) & ~RL_CFG3_WOL_MAGIC);
}

static void
re_enable_mac_mcu(struct re_softc *sc, bool enable)
{
        u_int32_t data32;

        switch(sc->HwSuppMacMcuVer) {
        case 1:
                data32 = re_eri_read(sc, 0xdc, 4, ERIAR_ExGMAC);
                if (enable)
                        data32 |= BIT_0;
                else
                        data32 &= ~BIT_0;
                re_eri_write(sc, 0xdc, 4, data32, ERIAR_ExGMAC);
                break;
        case 2:
                if (enable)
                        re_set_mac_ocp_bit(sc, 0xC0B4, BIT_0);
                else
                        re_clear_mac_ocp_bit(sc, 0xC0B4, BIT_0);
                break;
        }
}

static void
re_reset_mac_mcu_packet_filter(struct re_softc *sc)
{
        re_enable_mac_mcu(sc, 0);
        re_enable_mac_mcu(sc, 1);
}

static void re_exit_oob(struct re_softc *sc)
{
        u_int16_t data16;
        int i;

        re_disable_cfg9346_write(sc);

        if (HW_SUPP_SERDES_PHY(sc)) {
                if (sc->hw_hw_supp_serdes_phy_ver == 1) {
                        re_switch_to_sgmii_mode(sc);
                }
        }

        switch(sc->re_type) {
        case MACFG_61:
        case MACFG_62:
        case MACFG_67:
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
        case MACFG_80:
        case MACFG_81:
                Dash2DisableTxRx(sc);
                break;
        }

        if (HW_DASH_SUPPORT_DASH(sc))
                re_driver_start(sc);

        switch(sc->re_type) {
        case MACFG_56:
        case MACFG_57:
        case MACFG_58:
        case MACFG_59:
        case MACFG_60:
        case MACFG_61:
        case MACFG_62:
        case MACFG_67:
        case MACFG_68:
        case MACFG_69:
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
        case MACFG_74:
        case MACFG_75:
        case MACFG_76:
                CSR_WRITE_1(sc, 0xF2, CSR_READ_1(sc, 0xF2) | BIT_3);
                DELAY(2000);

                for (i = 0; i < 3000; i++) {
                        DELAY(50);
                        if (CSR_READ_4(sc, RE_TXCFG) & BIT_11)
                                break;
                }

                if (CSR_READ_1(sc, RE_COMMAND) & (RE_CMD_TX_ENB | RE_CMD_RX_ENB)) {
                        DELAY(100);
                        CSR_WRITE_1(sc, RE_COMMAND, CSR_READ_1(sc, RE_COMMAND) & ~(RE_CMD_TX_ENB | RE_CMD_RX_ENB));
                }

                for (i = 0; i < 3000; i++) {
                        DELAY(50);
                        if ((CSR_READ_1(sc, RE_MCU_CMD) & (RE_TXFIFO_EMPTY | RE_RXFIFO_EMPTY)) == (RE_TXFIFO_EMPTY | RE_RXFIFO_EMPTY))
                                break;
                }
                break;
        case MACFG_80:
        case MACFG_81:
                CSR_WRITE_1(sc, 0xF2, CSR_READ_1(sc, 0xF2) | BIT_3);
                DELAY(2000);

                if (CSR_READ_1(sc, RE_COMMAND) & (RE_CMD_TX_ENB | RE_CMD_RX_ENB)) {
                        DELAY(100);
                        CSR_WRITE_1(sc, RE_COMMAND, CSR_READ_1(sc, RE_COMMAND) & ~(RE_CMD_TX_ENB | RE_CMD_RX_ENB));
                }

                for (i = 0; i < 3000; i++) {
                        DELAY(50);
                        if ((CSR_READ_1(sc, RE_MCU_CMD) & (RE_TXFIFO_EMPTY | RE_RXFIFO_EMPTY)) == (RE_TXFIFO_EMPTY | RE_RXFIFO_EMPTY))
                                break;
                }
                break;
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                CSR_WRITE_1(sc, 0xF2, CSR_READ_1(sc, 0xF2) | BIT_3);
                DELAY(2000);

                if (CSR_READ_1(sc, RE_COMMAND) & (RE_CMD_TX_ENB | RE_CMD_RX_ENB)) {
                        DELAY(100);
                        CSR_WRITE_1(sc, RE_COMMAND, CSR_READ_1(sc, RE_COMMAND) & ~(RE_CMD_TX_ENB | RE_CMD_RX_ENB));
                }

                CSR_WRITE_1(sc, RE_COMMAND, CSR_READ_1(sc, RE_COMMAND) | RE_CMD_STOP_REQ);

                for (i = 0; i < 3000; i++) {
                        DELAY(50);
                        if ((CSR_READ_1(sc, RE_MCU_CMD) & (RE_TXFIFO_EMPTY | RE_RXFIFO_EMPTY)) == (RE_TXFIFO_EMPTY | RE_RXFIFO_EMPTY))
                                break;
                }

                for (i = 0; i < 3000; i++) {
                        DELAY(50);
                        if ((CSR_READ_2(sc, RE_IntrMitigate) & (BIT_0 | BIT_1 | BIT_8)) == (BIT_0 | BIT_1 | BIT_8))
                                break;
                }
                break;
        }

        if (HW_SUPPORT_MAC_MCU(sc))
                re_reset_mac_mcu_packet_filter(sc);

        //Disable realwow function
        switch (sc->re_type) {
        case MACFG_50:
        case MACFG_51:
                CSR_WRITE_4(sc, RE_MCUACCESS, 0xE5A90000);
                CSR_WRITE_4(sc, RE_MCUACCESS, 0xF2100010);
                break;
        case MACFG_52:
                CSR_WRITE_4(sc, RE_MCUACCESS, 0xE5A90000);
                CSR_WRITE_4(sc, RE_MCUACCESS, 0xE4640000);
                CSR_WRITE_4(sc, RE_MCUACCESS, 0xF2100010);
                break;
        case MACFG_56:
        case MACFG_57:
                CSR_WRITE_4(sc, RE_MCUACCESS, 0x605E0000);
                CSR_WRITE_4(sc, RE_MCUACCESS, (0xE05E << 16) | (CSR_READ_4(sc, RE_MCUACCESS) & 0xFFFE));
                CSR_WRITE_4(sc, RE_MCUACCESS, 0xE9720000);
                CSR_WRITE_4(sc, RE_MCUACCESS, 0xF2140010);
                break;
        case MACFG_60:
                CSR_WRITE_4(sc, RE_MCUACCESS, 0xE05E00FF);
                CSR_WRITE_4(sc, RE_MCUACCESS, 0xE9720000);
                re_mac_ocp_write(sc, 0xE428, 0x0010);
                break;
        }

        if (sc->re_hw_supp_now_is_oob_ver >0)
                re_disable_now_is_oob(sc);

        switch(sc->re_type) {
        case MACFG_52:
                for (i = 0; i < 10; i++) {
                        DELAY(100);
                        if (CSR_READ_2(sc, 0xD2) & BIT_9)
                                break;
                }

                data16 = re_mac_ocp_read(sc, 0xD4DE) | BIT_15;
                re_mac_ocp_write(sc, 0xD4DE, data16);

                for (i = 0; i < 10; i++) {
                        DELAY(100);
                        if (CSR_READ_2(sc, 0xD2) & BIT_9)
                                break;
                }
                break;
        case MACFG_56:
        case MACFG_57:
        case MACFG_58:
        case MACFG_59:
        case MACFG_60:
        case MACFG_61:
        case MACFG_62:
        case MACFG_67:
        case MACFG_68:
        case MACFG_69:
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
        case MACFG_74:
        case MACFG_75:
        case MACFG_76:
                data16 = re_mac_ocp_read(sc, 0xE8DE) & ~BIT_14;
                re_mac_ocp_write(sc, 0xE8DE, data16);
                for (i = 0; i < 10; i++) {
                        DELAY(100);
                        if (CSR_READ_2(sc, 0xD2) & BIT_9)
                                break;
                }

                data16 = re_mac_ocp_read(sc, 0xE8DE) | BIT_15;
                re_mac_ocp_write(sc, 0xE8DE, data16);

                for (i = 0; i < 10; i++) {
                        DELAY(100);
                        if (CSR_READ_2(sc, 0xD2) & BIT_9)
                                break;
                }
                break;
        case MACFG_80:
        case MACFG_81:
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                data16 = re_mac_ocp_read(sc, 0xE8DE) & ~BIT_14;
                re_mac_ocp_write(sc, 0xE8DE, data16);
                for (i = 0; i < 10; i++) {
                        DELAY(100);
                        if (CSR_READ_2(sc, 0xD2) & BIT_9)
                                break;
                }

                re_mac_ocp_write(sc, 0xC0AA, 0x07D0);
                re_mac_ocp_write(sc, 0xC0A6, 0x01B5);
                re_mac_ocp_write(sc, 0xC01E, 0x5555);

                for (i = 0; i < 10; i++) {
                        DELAY(100);
                        if (CSR_READ_2(sc, 0xD2) & BIT_9)
                                break;
                }
                break;
        }

        //wait ups resume (phy state 2)
        switch(sc->re_type) {
        case MACFG_68:
        case MACFG_69:
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
        case MACFG_74:
        case MACFG_75:
        case MACFG_80:
        case MACFG_81:
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                if (re_is_ups_resume(sc)) {
                        re_wait_phy_ups_resume(sc, 2);
                        re_clear_ups_resume_bit(sc);
                        re_clear_phy_ups_reg(sc);
                }
                break;
        };

        /*
        * Config MAC MCU
        */
        re_hw_mac_mcu_config(sc);
}

static void re_hw_init(struct re_softc *sc)
{
        /*
        * disable EDT.
        */
        switch(sc->re_type) {
        case MACFG_16:
        case MACFG_17:
        case MACFG_18:
        case MACFG_19:
        case MACFG_41:
                CSR_WRITE_1(sc, 0xF4, CSR_READ_1(sc, 0xF4) & ~(BIT_0|BIT_1));
                break;
        case MACFG_36:
        case MACFG_37:
        case MACFG_38:
        case MACFG_39:
        case MACFG_42:
        case MACFG_43:
        case MACFG_50:
        case MACFG_51:
        case MACFG_54:
        case MACFG_55:
                CSR_WRITE_1(sc, 0xF2, CSR_READ_1(sc, 0xF2) & ~(BIT_0|BIT_1|BIT_2));
                break;
        }

        re_enable_cfg9346_write(sc);

        if (s0_magic_packet == 0)
                re_disable_magic_packet(sc);
        else
                re_enable_magic_packet(sc);

        re_disable_cfg9346_write(sc);

        switch(sc->re_type) {
        case MACFG_5:
                if (CSR_READ_1(sc, RE_CFG2) & 1) {
                        CSR_WRITE_4(sc, 0x7C, 0x000FFFFF);
                } else {
                        CSR_WRITE_4(sc, 0x7C, 0x000FFF00);
                }
                break;
        case MACFG_6:
                if (CSR_READ_1(sc, RE_CFG2) & 1) {
                        CSR_WRITE_4(sc, 0x7C, 0x003FFFFF);
                } else {
                        CSR_WRITE_4(sc, 0x7C, 0x003FFF00);
                }
                break;
        }

        switch(sc->re_type) {
        case MACFG_33:
        case MACFG_36:
        case MACFG_37:
                CSR_WRITE_1(sc, 0xF3, CSR_READ_1(sc, 0xF3) | BIT_2);
                break;
        }

        switch(sc->re_type) {
        case MACFG_100:
        case MACFG_101:
                re_clear_set_mac_ocp_bit(sc, 0xD40C, 0xE038, 0x8020);
                break;
        }

        switch(sc->re_type) {
        case MACFG_36:
        case MACFG_37:
        case MACFG_38:
        case MACFG_39:
        case MACFG_42:
        case MACFG_43:
        case MACFG_50:
        case MACFG_51:
        case MACFG_52:
        case MACFG_53:
        case MACFG_54:
        case MACFG_55:
        case MACFG_56:
        case MACFG_57:
        case MACFG_58:
        case MACFG_59:
        case MACFG_60:
        case MACFG_61:
        case MACFG_62:
        case MACFG_67:
        case MACFG_68:
        case MACFG_69:
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
        case MACFG_74:
        case MACFG_75:
        case MACFG_76:
        case MACFG_80:
        case MACFG_81:
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                re_enable_force_clkreq(sc, 0);
                re_enable_aspm_clkreq_lock(sc, 0);
                re_enable_cfg9346_write(sc);
                CSR_WRITE_1(sc, RE_CFG2, CSR_READ_1(sc, RE_CFG2) | BIT_5);
                re_disable_cfg9346_write(sc);
                break;
        }

        if (sc->re_if_flags & RL_FLAG_PCIE) {
                uint32_t Data32;
                //Set PCIE uncorrectable error status mask pcie 0x108
                Data32 = re_csi_read(sc, 0xF108);
                Data32 |= BIT_20;
                re_csi_write(sc, 0xF108, Data32);
        }
}

static void re_rar_set(struct re_softc *sc, u_int8_t *eaddr)
{
        re_enable_cfg9346_write(sc);

        CSR_WRITE_4(sc, RE_IDR0,
                    htole32(*(u_int32_t *)(&eaddr[0])));
        CSR_WRITE_2(sc, RE_IDR4,
                    htole16(*(u_int32_t *)(&eaddr[4])));

        switch (sc->re_type) {
        case MACFG_36:
        case MACFG_37:
        case MACFG_42:
        case MACFG_43:
        case MACFG_54:
        case MACFG_55:
                CSR_WRITE_4(sc, RE_SecMAC0,
                            htole32(*(u_int32_t *)(&eaddr[0])));
                CSR_WRITE_2(sc, RE_SecMAC4,
                            htole16(*(u_int16_t *)(&eaddr[4])));
                break;
        }

        switch (sc->re_type) {
        case MACFG_38:
        case MACFG_39:
                re_eri_write(sc, 0xF0, 4, *(u_int16_t *)(&eaddr[0])<<16, ERIAR_ExGMAC);
                re_eri_write(sc, 0xF4, 4, *(u_int32_t *)(&eaddr[2]), ERIAR_ExGMAC);
                break;
        }

        re_disable_cfg9346_write(sc);
}

static void re_get_hw_mac_address(struct re_softc *sc, u_int8_t *eaddr)
{
        device_t dev = sc->dev;
        u_int16_t re_eeid = 0;
        int i;

        for (i = 0; i < ETHER_ADDR_LEN; i++)
                eaddr[i] = CSR_READ_1(sc, RE_IDR0 + i);

        switch(sc->re_type) {
        case MACFG_50:
        case MACFG_51:
        case MACFG_52:
        case MACFG_53:
        case MACFG_56:
        case MACFG_57:
        case MACFG_58:
        case MACFG_59:
        case MACFG_60:
        case MACFG_61:
        case MACFG_62:
        case MACFG_67:
        case MACFG_68:
        case MACFG_69:
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
        case MACFG_74:
        case MACFG_75:
        case MACFG_76:
                *(u_int32_t *)&eaddr[0] = re_eri_read(sc, 0xE0, 4, ERIAR_ExGMAC);
                *(u_int16_t *)&eaddr[4] = (u_int16_t)re_eri_read(sc, 0xE4, 4, ERIAR_ExGMAC);
                break;
        case MACFG_80:
        case MACFG_81:
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                *(u_int32_t *)&eaddr[0] = CSR_READ_4(sc, RE_BACKUP_ADDR0_8125);
                *(u_int16_t *)&eaddr[4] = CSR_READ_2(sc, RE_BACKUP_ADDR4_8125);
                break;
        case MACFG_63:
        case MACFG_64:
        case MACFG_65:
        case MACFG_66:
                break;
        default:
                re_read_eeprom(sc, (caddr_t)&re_eeid, RE_EE_ID, 1, 0);
                if (re_eeid == 0x8129)
                        re_read_eeprom(sc, (caddr_t)&eaddr[0], RE_EE_EADDR, 3, 0);
                break;
        }

        if (!is_valid_ether_addr(eaddr)) {
                device_printf(dev,"Invalid ether addr: %6D\n", eaddr, ":");
                ether_gen_addr(sc->re_ifp, (struct ether_addr *)eaddr);
                device_printf(dev,"Random ether addr: %6D\n", eaddr, ":");
                sc->random_mac = 1;
        }

        re_rar_set(sc, eaddr);
}

static int re_check_mac_version(struct re_softc *sc)
{
        device_t dev = sc->dev;
        int error = 0;

        switch(CSR_READ_4(sc, RE_TXCFG) & 0xFCF00000) {
        case 0x00800000:
        case 0x04000000:
                sc->re_type = MACFG_3;
                sc->max_jumbo_frame_size = Jumbo_Frame_7k;
                CSR_WRITE_4(sc, RE_RXCFG, 0xFF00);
                break;
        case 0x10000000:
                sc->re_type = MACFG_4;
                sc->max_jumbo_frame_size = Jumbo_Frame_7k;
                CSR_WRITE_4(sc, RE_RXCFG, 0xFF00);
                break;
        case 0x18000000:
                sc->re_type = MACFG_5;
                sc->max_jumbo_frame_size = Jumbo_Frame_7k;
                CSR_WRITE_4(sc, RE_RXCFG, 0xFF00);
                break;
        case 0x98000000:
                sc->re_type = MACFG_6;
                sc->max_jumbo_frame_size = Jumbo_Frame_7k;
                CSR_WRITE_4(sc, RE_RXCFG, 0xFF00);
                break;
        case 0x34000000:
        case 0xB4000000:
                sc->re_type = MACFG_11;
                sc->max_jumbo_frame_size = ETHERMTU;
                CSR_WRITE_4(sc, RE_RXCFG, 0xE700);
                break;
        case 0x34200000:
        case 0xB4200000:
                sc->re_type = MACFG_12;
                sc->max_jumbo_frame_size = ETHERMTU;
                CSR_WRITE_4(sc, RE_RXCFG, 0xE700);
                break;
        case 0x34300000:
        case 0xB4300000:
                sc->re_type = MACFG_13;
                sc->max_jumbo_frame_size = ETHERMTU;
                CSR_WRITE_4(sc, RE_RXCFG, 0xE700);
                break;
        case 0x34900000:
        case 0x24900000:
                sc->re_type = MACFG_14;
                sc->max_jumbo_frame_size = ETHERMTU;
                sc->re_if_flags |= RL_FLAG_DESCV2;
                CSR_WRITE_4(sc, RE_RXCFG, 0xE700);
                break;
        case 0x34A00000:
        case 0x24A00000:
                sc->re_type = MACFG_15;
                sc->max_jumbo_frame_size = ETHERMTU;
                sc->re_if_flags |= RL_FLAG_DESCV2;
                CSR_WRITE_4(sc, RE_RXCFG, 0xE700);
                break;
        case 0x34B00000:
        case 0x24B00000:
                sc->re_type = MACFG_16;
                sc->max_jumbo_frame_size = ETHERMTU;
                sc->re_if_flags |= RL_FLAG_DESCV2;
                CSR_WRITE_4(sc, RE_RXCFG, 0xE700);
                break;
        case 0x34C00000:
        case 0x24C00000:
                sc->re_type = MACFG_17;
                sc->max_jumbo_frame_size = ETHERMTU;
                sc->re_if_flags |= RL_FLAG_DESCV2;
                CSR_WRITE_4(sc, RE_RXCFG, 0xE700);
                break;
        case 0x34D00000:
        case 0x24D00000:
                sc->re_type = MACFG_18;
                sc->max_jumbo_frame_size = ETHERMTU;
                sc->re_if_flags |= RL_FLAG_DESCV2;
                CSR_WRITE_4(sc, RE_RXCFG, 0xE700);
                break;
        case 0x34E00000:
        case 0x24E00000:
                sc->re_type = MACFG_19;
                sc->max_jumbo_frame_size = ETHERMTU;
                sc->re_if_flags |= RL_FLAG_DESCV2;
                CSR_WRITE_4(sc, RE_RXCFG, 0xE700);
                break;
        case 0x30000000:
                sc->re_type = MACFG_21;
                sc->max_jumbo_frame_size = Jumbo_Frame_4k;
                CSR_WRITE_4(sc, RE_RXCFG, 0xE700);
                break;
        case 0x38000000:
                sc->re_type = MACFG_22;
                sc->max_jumbo_frame_size = Jumbo_Frame_4k;
                CSR_WRITE_4(sc, RE_RXCFG, 0xE700);
                break;
        case 0x38500000:
        case 0xB8500000:
        case 0x38700000:
        case 0xB8700000:
                sc->re_type = MACFG_23;
                sc->max_jumbo_frame_size = Jumbo_Frame_4k;
                CSR_WRITE_4(sc, RE_RXCFG, 0xE700);
                break;
        case 0x3C000000:
                sc->re_type = MACFG_24;
                sc->max_jumbo_frame_size = Jumbo_Frame_6k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM;
                CSR_WRITE_4(sc, RE_RXCFG, 0xC700);
                break;
        case 0x3C200000:
                sc->re_type = MACFG_25;
                sc->max_jumbo_frame_size = Jumbo_Frame_6k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM;
                CSR_WRITE_4(sc, RE_RXCFG, 0xC700);
                break;
        case 0x3C400000:
                sc->re_type = MACFG_26;
                sc->max_jumbo_frame_size = Jumbo_Frame_6k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM;
                CSR_WRITE_4(sc, RE_RXCFG, 0xC700);
                break;
        case 0x3C900000:
                sc->re_type = MACFG_27;
                sc->max_jumbo_frame_size = Jumbo_Frame_6k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM;
                CSR_WRITE_4(sc, RE_RXCFG, 0xC700);
                break;
        case 0x3CB00000:
                sc->re_type = MACFG_28;
                sc->max_jumbo_frame_size = Jumbo_Frame_6k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM;
                CSR_WRITE_4(sc, RE_RXCFG, 0xC700);
                break;
        case 0x28100000:
                sc->re_type = MACFG_31;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM;
                CSR_WRITE_4(sc, RE_RXCFG, 0x8700);
                break;
        case 0x28200000:
                sc->re_type = MACFG_32;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM;
                CSR_WRITE_4(sc, RE_RXCFG, 0x8700);
                break;
        case 0x28300000:
                sc->re_type = MACFG_33;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM;
                CSR_WRITE_4(sc, RE_RXCFG, 0x8700);
                break;
        case 0x2C100000:
                sc->re_type = MACFG_36;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM;
                CSR_WRITE_4(sc, RE_RXCFG, 0x8700);
                break;
        case 0x2C200000:
                sc->re_type = MACFG_37;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM;
                CSR_WRITE_4(sc, RE_RXCFG, 0x8700);
                break;
        case 0x2C800000:
                sc->re_type = MACFG_38;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM | RL_FLAG_MAGIC_PACKET_V2;
                CSR_WRITE_4(sc, RE_RXCFG, 0xBF00);
                break;
        case 0x2C900000:
                sc->re_type = MACFG_39;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM | RL_FLAG_MAGIC_PACKET_V2;
                CSR_WRITE_4(sc, RE_RXCFG, 0xBF00);
                break;
        case 0x24000000:
                sc->re_type = MACFG_41;
                sc->max_jumbo_frame_size = ETHERMTU;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM;
                CSR_WRITE_4(sc, RE_RXCFG, 0xE700);
                break;
        case 0x40900000:
                sc->re_type = MACFG_42;
                sc->max_jumbo_frame_size = ETHERMTU;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM;
                CSR_WRITE_4(sc, RE_RXCFG, 0xE700);
                break;
        case 0x40A00000:
        case 0x40B00000:
        case 0x40C00000:
                sc->re_type = MACFG_43;
                sc->max_jumbo_frame_size = ETHERMTU;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM;
                CSR_WRITE_4(sc, RE_RXCFG, 0xE700);
                break;
        case 0x48000000:
                sc->re_type = MACFG_50;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM | RL_FLAG_MAGIC_PACKET_V2;
                CSR_WRITE_4(sc, RE_RXCFG, 0xBF00);
                break;
        case 0x48100000:
                sc->re_type = MACFG_51;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM | RL_FLAG_MAGIC_PACKET_V2;
                CSR_WRITE_4(sc, RE_RXCFG, 0xBF00);
                break;
        case 0x48800000:
                sc->re_type = MACFG_52;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM | RL_FLAG_MAGIC_PACKET_V2;
                CSR_WRITE_4(sc, RE_RXCFG, 0xBF00);
                break;
        case 0x44000000:
                sc->re_type = MACFG_53;
                sc->max_jumbo_frame_size = ETHERMTU;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM | RL_FLAG_MAGIC_PACKET_V2;
                CSR_WRITE_4(sc, RE_RXCFG, 0xE700);
                break;
        case 0x44800000:
                sc->re_type = MACFG_54;
                sc->max_jumbo_frame_size = ETHERMTU;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM;
                CSR_WRITE_4(sc, RE_RXCFG, 0xE700);
                break;
        case 0x44900000:
                sc->re_type = MACFG_55;
                sc->max_jumbo_frame_size = ETHERMTU;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM;
                CSR_WRITE_4(sc, RE_RXCFG, 0xE700);
                break;
        case 0x4C000000:
                sc->re_type = MACFG_56;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM |
                                   RL_FLAG_8168G_PLUS | RL_FLAG_MAGIC_PACKET_V2;
                CSR_WRITE_4(sc, RE_RXCFG, 0x8F00);
                break;
        case 0x4C100000:
                sc->re_type = MACFG_57;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM |
                                   RL_FLAG_8168G_PLUS | RL_FLAG_MAGIC_PACKET_V2;
                CSR_WRITE_4(sc, RE_RXCFG, 0x8F00);
                break;
        case 0x50800000:
                sc->re_type = MACFG_58;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM |
                                   RL_FLAG_8168G_PLUS | RL_FLAG_MAGIC_PACKET_V2;
                CSR_WRITE_4(sc, RE_RXCFG, 0x8F00);
                break;
        case 0x50900000:
                sc->re_type = MACFG_59;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM |
                                   RL_FLAG_8168G_PLUS | RL_FLAG_MAGIC_PACKET_V2;
                CSR_WRITE_4(sc, RE_RXCFG, 0x8F00);
                break;
        case 0x5C800000:
                sc->re_type = MACFG_60;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM |
                                   RL_FLAG_8168G_PLUS | RL_FLAG_MAGIC_PACKET_V2;
                CSR_WRITE_4(sc, RE_RXCFG, 0x8F00);
                break;
        case 0x50000000:
                sc->re_type = MACFG_61;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM |
                                   RL_FLAG_8168G_PLUS | RL_FLAG_MAGIC_PACKET_V2;
                CSR_WRITE_4(sc, RE_RXCFG, 0x8F00);
                break;
        case 0x50100000:
                sc->re_type = MACFG_62;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM |
                                   RL_FLAG_8168G_PLUS | RL_FLAG_MAGIC_PACKET_V2;
                CSR_WRITE_4(sc, RE_RXCFG, 0x8F00);
                break;
        case 0x50200000:
                sc->re_type = MACFG_67;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM |
                                   RL_FLAG_8168G_PLUS | RL_FLAG_MAGIC_PACKET_V2;
                CSR_WRITE_4(sc, RE_RXCFG, 0x8F00);
                break;
        case 0x28800000:
                sc->re_type = MACFG_63;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM;
                CSR_WRITE_4(sc, RE_RXCFG, 0x8700);
                break;
        case 0x28900000:
                sc->re_type = MACFG_64;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM;
                CSR_WRITE_4(sc, RE_RXCFG, 0x8700);
                break;
        case 0x28A00000:
                sc->re_type = MACFG_65;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM;
                CSR_WRITE_4(sc, RE_RXCFG, 0x8700);
                break;
        case 0x28B00000:
                sc->re_type = MACFG_66;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM;
                CSR_WRITE_4(sc, RE_RXCFG, 0x8700);
                break;
        case 0x54000000:
                sc->re_type = MACFG_68;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM |
                                   RL_FLAG_8168G_PLUS | RL_FLAG_MAGIC_PACKET_V2;
                CSR_WRITE_4(sc, RE_RXCFG, 0x8F00);
                break;
        case 0x54100000:
                sc->re_type = MACFG_69;
                if ((re_mac_ocp_read(sc, 0xD006) & 0xFF00) == 0x0100)
                        sc->re_type = MACFG_74;
                else if ((re_mac_ocp_read(sc, 0xD006) & 0xFF00) == 0x0300)
                        sc->re_type = MACFG_75;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM |
                                   RL_FLAG_8168G_PLUS | RL_FLAG_MAGIC_PACKET_V2;
                CSR_WRITE_4(sc, RE_RXCFG, 0x8F00);
                break;
        case 0x6C000000:
                sc->re_type = MACFG_76;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_8168G_PLUS |
                                   RL_FLAG_MAGIC_PACKET_V2;
                CSR_WRITE_4(sc, RE_RXCFG, 0x8F00);
                break;
        case 0x54900000:
                sc->re_type = MACFG_70;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM |
                                   RL_FLAG_8168G_PLUS | RL_FLAG_MAGIC_PACKET_V2;
                CSR_WRITE_4(sc, RE_RXCFG, 0x8F00);
                break;
        case 0x54A00000:
                sc->re_type = MACFG_71;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM |
                                   RL_FLAG_8168G_PLUS | RL_FLAG_MAGIC_PACKET_V2;
                CSR_WRITE_4(sc, RE_RXCFG, 0x8F00);
                break;
        case 0x54B00000:
                sc->re_type = MACFG_72;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM |
                                   RL_FLAG_8168G_PLUS | RL_FLAG_MAGIC_PACKET_V2;
                CSR_WRITE_4(sc, RE_RXCFG, 0x8F00);
                break;
        case 0x54C00000:
                sc->re_type = MACFG_73;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM |
                                   RL_FLAG_8168G_PLUS | RL_FLAG_MAGIC_PACKET_V2;
                CSR_WRITE_4(sc, RE_RXCFG, 0x8F00);
                break;
        case 0x60800000:
                sc->re_type = MACFG_80;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM |
                                   RL_FLAG_8168G_PLUS | RL_FLAG_MAGIC_PACKET_V3;
                CSR_WRITE_4(sc, RE_RXCFG, 0x40C00400);
                break;
        case 0x60900000:
                sc->re_type = MACFG_81;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM |
                                   RL_FLAG_8168G_PLUS | RL_FLAG_MAGIC_PACKET_V3;
                CSR_WRITE_4(sc, RE_RXCFG, 0x40C00400);
                break;
        case 0x64000000:
                sc->re_type = MACFG_82;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM |
                                   RL_FLAG_8168G_PLUS | RL_FLAG_MAGIC_PACKET_V3;
                CSR_WRITE_4(sc, RE_RXCFG, 0x40C00C00);
                break;
        case 0x64100000:
                sc->re_type = MACFG_83;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM |
                                   RL_FLAG_8168G_PLUS | RL_FLAG_MAGIC_PACKET_V3;
                CSR_WRITE_4(sc, RE_RXCFG, 0x40C00C00);
                break;
        case 0x68000000:
                sc->re_type = MACFG_84;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM |
                                   RL_FLAG_8168G_PLUS | RL_FLAG_MAGIC_PACKET_V3;
                CSR_WRITE_4(sc, RE_RXCFG, 0x40C00C00);
                break;
        case 0x68100000:
                sc->re_type = MACFG_85;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM |
                                   RL_FLAG_8168G_PLUS | RL_FLAG_MAGIC_PACKET_V3;
                CSR_WRITE_4(sc, RE_RXCFG, 0x40C00C00);
                break;
        case 0x68800000:
                sc->re_type = MACFG_86;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM |
                                   RL_FLAG_8168G_PLUS | RL_FLAG_MAGIC_PACKET_V3;
                CSR_WRITE_4(sc, RE_RXCFG, 0x40C00C00);
                break;
        case 0x68900000:
                sc->re_type = MACFG_87;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM |
                                   RL_FLAG_8168G_PLUS | RL_FLAG_MAGIC_PACKET_V3;
                CSR_WRITE_4(sc, RE_RXCFG, 0x40C00C00);
                break;
        case 0x70800000:
                sc->re_type = MACFG_88;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM |
                                   RL_FLAG_8168G_PLUS | RL_FLAG_MAGIC_PACKET_V3;
                CSR_WRITE_4(sc, RE_RXCFG, 0x40C00C00);
                break;
        case 0x64800000:
                sc->re_type = MACFG_90;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM |
                                   RL_FLAG_8168G_PLUS | RL_FLAG_MAGIC_PACKET_V3;
                CSR_WRITE_4(sc, RE_RXCFG, 0x40C00F00);
                break;
        case 0x64900000:
                sc->re_type = MACFG_91;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM |
                                   RL_FLAG_8168G_PLUS | RL_FLAG_MAGIC_PACKET_V3;
                CSR_WRITE_4(sc, RE_RXCFG, 0x40C00D00);
                break;
        case 0x64A00000:
                sc->re_type = MACFG_92;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM |
                                   RL_FLAG_8168G_PLUS | RL_FLAG_MAGIC_PACKET_V3;
                CSR_WRITE_4(sc, RE_RXCFG, 0x40C00D00);
                break;
        case 0x6C800000:
                sc->re_type = MACFG_100;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM |
                                   RL_FLAG_8168G_PLUS | RL_FLAG_MAGIC_PACKET_V3;
                CSR_WRITE_4(sc, RE_RXCFG, 0x40C00D00);
                break;
        case 0x6C900000:
                sc->re_type = MACFG_101;
                sc->max_jumbo_frame_size = Jumbo_Frame_9k;
                sc->re_if_flags |= RL_FLAG_DESCV2 | RL_FLAG_PHYWAKE_PM |
                                   RL_FLAG_8168G_PLUS | RL_FLAG_MAGIC_PACKET_V3;
                CSR_WRITE_4(sc, RE_RXCFG, 0x40C00D00);
                break;
        default:
                device_printf(dev,"unknown device\n");
                sc->re_type = MACFG_FF;
                error = ENXIO;
                break;
        }

        switch(sc->re_device_id) {
        case RT_DEVICEID_8169:
        case RT_DEVICEID_8169SC:
        case RT_DEVICEID_8168:
        case RT_DEVICEID_8161:
        case RT_DEVICEID_8162:
        case RT_DEVICEID_8125:
        case RT_DEVICEID_3000:
        case RT_DEVICEID_8126:
        case RT_DEVICEID_8127:
                //do nothing
                break;
        default:
                sc->max_jumbo_frame_size = ETHERMTU;
                break;
        }

        return error;
}

static bool
re_is_allow_access_dash_ocp(struct re_softc *sc)
{
        bool allow_access = false;
        u_int16_t mac_ocp_data;

        if (!HW_SUPPORT_OCP_CHANNEL(sc))
                goto exit;

        switch (sc->re_type) {
        case MACFG_80:
        case MACFG_81:
                mac_ocp_data = re_mac_ocp_read(sc, 0xd460);
                if (mac_ocp_data == 0xffff || !(mac_ocp_data & BIT_0))
                        goto exit;
                break;
        case MACFG_84:
        case MACFG_85:
                mac_ocp_data = re_mac_ocp_read(sc, 0xd4c0);
                if (mac_ocp_data == 0xffff || (mac_ocp_data & BIT_3))
                        goto exit;
                break;
        default:
                break;
        }

        allow_access = true;

exit:
        return allow_access;
}

static void re_init_software_variable(struct re_softc *sc)
{
        switch(sc->re_device_id) {
        case RT_DEVICEID_8168:
        case RT_DEVICEID_8161:
        case RT_DEVICEID_8162:
        case RT_DEVICEID_8136:
        case RT_DEVICEID_8125:
        case RT_DEVICEID_3000:
        case RT_DEVICEID_8126:
        case RT_DEVICEID_8127:
                sc->re_if_flags |= RL_FLAG_PCIE;
                break;
        }

        sc->re_rx_mbuf_sz = sc->max_jumbo_frame_size + ETHER_VLAN_ENCAP_LEN + ETHER_HDR_LEN + ETHER_CRC_LEN + RE_ETHER_ALIGN + 1;

        if (sc->re_rx_mbuf_sz > max_rx_mbuf_sz) {
                sc->max_jumbo_frame_size -= (sc->re_rx_mbuf_sz - max_rx_mbuf_sz);
                sc->re_rx_mbuf_sz = max_rx_mbuf_sz;
        }

        switch(sc->re_type) {
        case MACFG_63:
        case MACFG_64:
        case MACFG_65:
        case MACFG_66:
                sc->HwSuppDashVer = 1;
                break;
        case MACFG_61:
        case MACFG_62:
        case MACFG_67:
                sc->HwSuppDashVer = 2;
                break;
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
                sc->HwSuppDashVer = 3;
                break;
        case MACFG_80:
        case MACFG_81: {
                u_int8_t tmpUchar;
                tmpUchar = (u_int8_t)re_mac_ocp_read(sc, 0xD006);
                if (tmpUchar == 0x02 || tmpUchar == 0x04)
                        sc->HwSuppDashVer = 2;
        }
        break;
        case MACFG_84:
        case MACFG_85:
                sc->HwSuppDashVer = 4;
                break;
        default:
                sc->HwSuppDashVer = 0;
                break;
        }

        switch(sc->re_type) {
        case MACFG_63:
        case MACFG_64:
        case MACFG_65:
        case MACFG_66:
                sc->HwSuppOcpChannelVer = 1;
                break;
        case MACFG_61:
        case MACFG_62:
        case MACFG_67:
        case MACFG_84:
        case MACFG_85:
        case MACFG_88:
                sc->HwSuppOcpChannelVer = 2;
                break;
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
                sc->HwSuppOcpChannelVer = 3;
                break;
        case MACFG_80:
        case MACFG_81:
                if (sc->HwSuppDashVer > 0)
                        sc->HwSuppOcpChannelVer = 2;
                break;
        default:
                sc->HwSuppOcpChannelVer = 0;
                break;
        }

        switch(sc->re_type) {
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
                sc->HwPkgDet = re_mac_ocp_read(sc, 0xDC00);
                sc->HwPkgDet = (sc->HwPkgDet >> 3) & 0x0F;
                break;
        }

        switch(sc->re_type) {
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
                if (sc->HwPkgDet == 0x06) {
                        u_int8_t tmpUchar = re_eri_read(sc, 0xE6, 1, ERIAR_ExGMAC);
                        if (tmpUchar == 0x02)
                                sc->hw_hw_supp_serdes_phy_ver = 1;
                        else if (tmpUchar == 0x00)
                                sc->hw_hw_supp_serdes_phy_ver = 2;
                }
                break;
        }

        if (HW_SUPP_SERDES_PHY(sc))
                eee_enable = 0;

        if (HW_DASH_SUPPORT_DASH(sc)) {
                sc->AllowAccessDashOcp = re_is_allow_access_dash_ocp(sc);
                sc->re_dash = re_check_dash(sc);
        }

        if (sc->re_dash) {
#if defined(__amd64__) || defined(__i386__)
                if (HW_DASH_SUPPORT_TYPE_3(sc)) {
                        u_int64_t CmacMemPhysAddress;
                        bus_space_handle_t cmac_ioaddr;

                        CmacMemPhysAddress = re_csi_other_fun_read(sc, 0, 0xf018);
                        if (!(CmacMemPhysAddress & BIT_0)) {
                                if (CmacMemPhysAddress & BIT_2)
                                        CmacMemPhysAddress |=  (u_int64_t)re_csi_other_fun_read(sc, 0, 0xf01c) << 32;

                                CmacMemPhysAddress &=  0xFFFFFFF0;
                                /* ioremap MMIO region */
                                sc->re_mapped_cmac_tag = X86_BUS_SPACE_MEM;
                                if (bus_space_map(sc->re_mapped_cmac_tag, CmacMemPhysAddress, RE_REGS_SIZE, 0,
                                                  &cmac_ioaddr))
                                        sc->re_dash = 0;
                                else
                                        sc->re_mapped_cmac_handle = cmac_ioaddr;
                        }
                }
#else
                sc->re_dash = 0;
#endif
        }

        switch(sc->re_type) {
        case MACFG_61:
        case MACFG_62:
        case MACFG_67:
                sc->re_cmac_handle = sc->re_bhandle;
                sc->re_cmac_tag = sc->re_btag;
                break;
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
                sc->re_cmac_handle = sc->re_mapped_cmac_handle;
                sc->re_cmac_tag = sc->re_mapped_cmac_tag;
                break;
        }

        switch (sc->re_type) {
        case MACFG_80:
        case MACFG_81:
                sc->HwSuppIsrVer = 1;
                break;
        case MACFG_82:
        case MACFG_83:
        case MACFG_90:
                sc->HwSuppIsrVer = 2;
                break;
        case MACFG_91:
        case MACFG_92:
                sc->HwSuppIsrVer = 3;
                break;
        case MACFG_84:
        case MACFG_85:
                sc->HwSuppIsrVer = 4;
                break;
        case MACFG_88:
                sc->HwSuppIsrVer = 7;
                break;
        case MACFG_100:
        case MACFG_101:
                sc->HwSuppIsrVer = 6;
                break;
        case MACFG_86:
        case MACFG_87:
                sc->HwSuppIsrVer = 5;
                break;
        default:
                sc->HwSuppIsrVer = 0;
                break;
        }

        if (sc->HwSuppIsrVer > 1)
                sc->use_new_intr_mapping = 1;
        else
                sc->use_new_intr_mapping = 0;

        switch(sc->re_type) {
        case MACFG_14:
        case MACFG_15:
        case MACFG_16:
        case MACFG_17:
        case MACFG_18:
        case MACFG_19:
        case MACFG_31:
        case MACFG_32:
        case MACFG_33:
        case MACFG_41:
        case MACFG_63:
        case MACFG_64:
        case MACFG_65:
        case MACFG_66:
                sc->re_efuse_ver = EFUSE_SUPPORT_V1;
                break;
        case MACFG_36:
        case MACFG_37:
        case MACFG_42:
        case MACFG_43:
        case MACFG_54:
        case MACFG_55:
                sc->re_efuse_ver = EFUSE_SUPPORT_V2;
                break;
        case MACFG_38:
        case MACFG_39:
        case MACFG_50:
        case MACFG_51:
        case MACFG_52:
        case MACFG_53:
        case MACFG_56:
        case MACFG_57:
        case MACFG_58:
        case MACFG_59:
        case MACFG_60:
        case MACFG_61:
        case MACFG_62:
        case MACFG_67:
        case MACFG_68:
        case MACFG_69:
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
        case MACFG_74:
        case MACFG_75:
        case MACFG_76:
                sc->re_efuse_ver = EFUSE_SUPPORT_V3;
                break;
        case MACFG_80:
        case MACFG_81:
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                sc->re_efuse_ver = EFUSE_SUPPORT_V4;
                break;
        default:
                sc->re_efuse_ver = EFUSE_NOT_SUPPORT;
                break;
        }

        switch(sc->re_type) {
        case MACFG_69:
        case MACFG_76: {
                u_int16_t ioffset_p3, ioffset_p2, ioffset_p1, ioffset_p0;
                u_int16_t TmpUshort;

                re_mac_ocp_write(sc, 0xDD02, 0x807D);

                TmpUshort = re_mac_ocp_read(sc, 0xDD02);
                ioffset_p3 = ((TmpUshort & BIT_7) >>7);
                ioffset_p3 <<= 3;
                TmpUshort = re_mac_ocp_read(sc, 0xDD00);

                ioffset_p3 |= ((TmpUshort & (BIT_15 | BIT_14 | BIT_13))>>13);

                ioffset_p2 = ((TmpUshort & (BIT_12|BIT_11|BIT_10|BIT_9))>>9);
                ioffset_p1 = ((TmpUshort & (BIT_8|BIT_7|BIT_6|BIT_5))>>5);

                ioffset_p0 = ((TmpUshort & BIT_4) >>4);
                ioffset_p0 <<= 3;
                ioffset_p0 |= (TmpUshort & (BIT_2| BIT_1 | BIT_0));

                if ((ioffset_p3 == 0x0F) && (ioffset_p2 == 0x0F) && (ioffset_p1 == 0x0F) && (ioffset_p0 == 0x0F)) {
                        sc->RequireAdcBiasPatch = FALSE;
                } else {
                        sc->RequireAdcBiasPatch = TRUE;
                        sc->AdcBiasPatchIoffset = (ioffset_p3<<12)|(ioffset_p2<<8)|(ioffset_p1<<4)|(ioffset_p0);
                }
        }
        break;
        }

        switch(sc->re_type) {
        case MACFG_68:
        case MACFG_69:
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
        case MACFG_75:
        case MACFG_76: {
                u_int16_t rg_saw_cnt;

                re_mdio_write(sc, 0x1F, 0x0C42);
                rg_saw_cnt = re_mdio_read(sc, 0x13);
                rg_saw_cnt &= ~(BIT_15|BIT_14);
                re_mdio_write(sc, 0x1F, 0x0000);

                if (rg_saw_cnt > 0) {
                        sc->SwrCnt1msIni = 16000000/rg_saw_cnt;
                        sc->SwrCnt1msIni &= 0x0FFF;

                        sc->RequireAdjustUpsTxLinkPulseTiming = TRUE;
                }
        }
        break;
        }

#ifdef ENABLE_FIBER_SUPPORT
        re_check_hw_fiber_mode_support(sc);
#endif //ENABLE_FIBER_SUPPORT

        switch (sc->re_type) {
        case MACFG_74:
        case MACFG_75:
                sc->RequiredSecLanDonglePatch = FALSE;
                break;
        }

        switch(sc->re_type) {
        case MACFG_31:
        case MACFG_32:
        case MACFG_33:
        case MACFG_36:
        case MACFG_37:
        case MACFG_38:
        case MACFG_39:
        case MACFG_42:
        case MACFG_43:
        case MACFG_50:
        case MACFG_51:
        case MACFG_52:
        case MACFG_53:
        case MACFG_54:
        case MACFG_55:
        case MACFG_56:
        case MACFG_57:
        case MACFG_58:
        case MACFG_59:
        case MACFG_60:
        case MACFG_61:
        case MACFG_62:
        case MACFG_63:
        case MACFG_64:
        case MACFG_65:
        case MACFG_66:
        case MACFG_67:
        case MACFG_68:
        case MACFG_69:
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
        case MACFG_74:
        case MACFG_75:
        case MACFG_76:
        case MACFG_80:
        case MACFG_81:
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                sc->re_hw_enable_msi_msix = TRUE;
                break;
        }

        switch(sc->re_type) {
        case MACFG_3:
        case MACFG_4:
        case MACFG_5:
        case MACFG_6:
        case MACFG_11:
        case MACFG_12:
        case MACFG_13:
        case MACFG_21:
        case MACFG_22:
        case MACFG_23:
        case MACFG_24:
        case MACFG_25:
        case MACFG_26:
        case MACFG_27:
        case MACFG_28:
        case MACFG_41:
        case MACFG_42:
        case MACFG_43:
        case MACFG_54:
        case MACFG_55:
                sc->re_coalesce_tx_pkt = TRUE;
                break;
        }

        switch(sc->re_type) {
        case MACFG_36:
        case MACFG_37:
        case MACFG_38:
        case MACFG_39:
        case MACFG_42:
        case MACFG_43:
        case MACFG_50:
        case MACFG_51:
        case MACFG_52:
        case MACFG_53:
        case MACFG_54:
        case MACFG_55:
        case MACFG_56:
        case MACFG_57:
        case MACFG_58:
        case MACFG_59:
        case MACFG_60:
        case MACFG_61:
        case MACFG_62:
        case MACFG_67:
        case MACFG_68:
        case MACFG_69:
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
        case MACFG_74:
        case MACFG_75:
        case MACFG_76:
        case MACFG_80:
        case MACFG_81:
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                sc->re_hw_supp_now_is_oob_ver = 1;
                break;
        }

        switch (sc->re_type) {
        case MACFG_36:
        case MACFG_37:
                sc->re_sw_ram_code_ver = NIC_RAMCODE_VERSION_8168E;
                break;
        case MACFG_38:
        case MACFG_39:
                sc->re_sw_ram_code_ver = NIC_RAMCODE_VERSION_8168EVL;
                break;
        case MACFG_50:
        case MACFG_51:
                sc->re_sw_ram_code_ver = NIC_RAMCODE_VERSION_8168F;
                break;
        case MACFG_52:
                sc->re_sw_ram_code_ver = NIC_RAMCODE_VERSION_8411;
                break;
        case MACFG_56:
        case MACFG_57:
                sc->re_sw_ram_code_ver = NIC_RAMCODE_VERSION_8168G;
                break;
        case MACFG_58:
        case MACFG_59:
                sc->re_sw_ram_code_ver = NIC_RAMCODE_VERSION_8168GU;
                break;
        case MACFG_60:
                sc->re_sw_ram_code_ver = NIC_RAMCODE_VERSION_8411B;
                break;
        case MACFG_61:
        case MACFG_62:
        case MACFG_67:
                sc->re_sw_ram_code_ver = NIC_RAMCODE_VERSION_8168EP;
                break;
        case MACFG_68:
        case MACFG_69:
        case MACFG_76:
                sc->re_sw_ram_code_ver = NIC_RAMCODE_VERSION_8168H;
                break;
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
                sc->re_sw_ram_code_ver = NIC_RAMCODE_VERSION_8168FP;
                break;
        case MACFG_74:
                sc->re_sw_ram_code_ver = NIC_RAMCODE_VERSION_8168H_6838;
                break;
        case MACFG_75:
                sc->re_sw_ram_code_ver = NIC_RAMCODE_VERSION_8168H_6878B;
                break;
        case MACFG_80:
                sc->re_sw_ram_code_ver = NIC_RAMCODE_VERSION_8125A_REV_A;
                break;
        case MACFG_81:
                sc->re_sw_ram_code_ver = NIC_RAMCODE_VERSION_8125A_REV_B;
                break;
        case MACFG_82:
                sc->re_sw_ram_code_ver = NIC_RAMCODE_VERSION_8125B_REV_A;
                break;
        case MACFG_83:
                sc->re_sw_ram_code_ver = NIC_RAMCODE_VERSION_8125B_REV_B;
                break;
        case MACFG_84:
                sc->re_sw_ram_code_ver = NIC_RAMCODE_VERSION_8125BP_REV_A;
                break;
        case MACFG_85:
                sc->re_sw_ram_code_ver = NIC_RAMCODE_VERSION_8125BP_REV_B;
                break;
        case MACFG_86:
                sc->re_sw_ram_code_ver = NIC_RAMCODE_VERSION_8125D_REV_A;
                break;
        case MACFG_87:
                sc->re_sw_ram_code_ver = NIC_RAMCODE_VERSION_8125D_REV_B;
                break;
        case MACFG_88:
                sc->re_sw_ram_code_ver = NIC_RAMCODE_VERSION_8125CP_REV_A;
                break;
        case MACFG_90:
                sc->re_sw_ram_code_ver = NIC_RAMCODE_VERSION_8126A_REV_A;
                break;
        case MACFG_91:
                sc->re_sw_ram_code_ver = NIC_RAMCODE_VERSION_8126A_REV_B;
                break;
        case MACFG_92:
                sc->re_sw_ram_code_ver = NIC_RAMCODE_VERSION_8126A_REV_C;
                break;
        case MACFG_100:
                sc->re_sw_ram_code_ver = NIC_RAMCODE_VERSION_8127;
                break;
        case MACFG_101:
                sc->re_sw_ram_code_ver = NIC_RAMCODE_VERSION_8127_REV_A;
                break;
        }

        switch (sc->re_type) {
        case MACFG_81:
                if ((re_mac_ocp_read(sc, 0xD442) & BIT_5) &&
                    (re_real_ocp_phy_read(sc, 0xD068) & BIT_1)
                   ) {
                        sc->RequirePhyMdiSwapPatch = TRUE;
                }
                break;
        }

        switch (sc->re_type) {
        case MACFG_80:
        case MACFG_81:
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                sc->HwSuppExtendTallyCounterVer = 1;
                break;
        }

        switch (sc->re_type) {
        case MACFG_38:
        case MACFG_39:
        case MACFG_50:
        case MACFG_51:
        case MACFG_52:
        case MACFG_56:
        case MACFG_57:
        case MACFG_58:
        case MACFG_59:
        case MACFG_60:
        case MACFG_61:
        case MACFG_62:
        case MACFG_67:
        case MACFG_68:
        case MACFG_69:
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
        case MACFG_74:
        case MACFG_75:
        case MACFG_76:
                sc->HwSuppMacMcuVer = 1;
                break;
        case MACFG_80:
        case MACFG_81:
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                sc->HwSuppMacMcuVer = 2;
                break;
        }

        switch (sc->re_type) {
        case MACFG_80:
        case MACFG_81:
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                sc->MacMcuPageSize = RTL8125_MAC_MCU_PAGE_SIZE;
                break;
        }

        switch (sc->re_type) {
        case MACFG_86:
        case MACFG_87:
                sc->RequiredPfmPatch = TRUE;
                break;
        }

        sc->re_8169_MacVersion = (CSR_READ_4(sc, RE_TXCFG)&0x7c800000)>>25;		/* Get bit 26~30 	*/
        sc->re_8169_MacVersion |= ((CSR_READ_4(sc, RE_TXCFG)&0x00800000)!=0 ? 1:0);	/* Get bit 23 		*/
        DBGPRINT1(sc->re_unit,"8169 Mac Version %d",sc->re_8169_MacVersion);

        /* Rtl8169s single chip detected */
        if (sc->re_type == MACFG_3) {
                RE_LOCK(sc);
                sc->re_8169_PhyVersion=(re_mdio_read(sc, 0x03)&0x000f);
                DBGPRINT1(sc->re_unit,"8169 Phy Version %d",sc->re_8169_PhyVersion);
                RE_UNLOCK(sc);
        }

        sc->link_state = LINK_STATE_UNKNOWN;

#ifdef ENABLE_FIBER_SUPPORT
        if (HW_FIBER_MODE_ENABLED(sc))
                re_set_fiber_mode_software_variable(sc);
#endif //ENABLE_FIBER_SUPPORT
}

static void re_enable_ocp_phy_power_saving(struct re_softc *sc)
{
        u_int16_t val;

        if (sc->re_type == MACFG_59 || sc->re_type == MACFG_60 ||
            sc->re_type == MACFG_62 || sc->re_type == MACFG_67 ||
            sc->re_type == MACFG_68 || sc->re_type == MACFG_69 ||
            sc->re_type == MACFG_70 || sc->re_type == MACFG_71 ||
            sc->re_type == MACFG_72 || sc->re_type == MACFG_73 ||
            sc->re_type == MACFG_74 || sc->re_type == MACFG_75 ||
            sc->re_type == MACFG_76) {
                val = re_ocp_phy_read(sc, 0x0C41, 0x13);
                if (val != 0x0050) {
                        re_set_phy_mcu_patch_request(sc);
                        re_ocp_phy_write(sc, 0x0C41, 0x13, 0x0000);
                        re_ocp_phy_write(sc, 0x0C41, 0x13, 0x0050);
                        re_clear_phy_mcu_patch_request(sc);
                }
        } else if (sc->re_type == MACFG_80 || sc->re_type == MACFG_81) {
                val = re_real_ocp_phy_read(sc, 0xC416);
                if (val != 0x0050) {
                        re_set_phy_mcu_patch_request(sc);
                        re_real_ocp_phy_write(sc, 0xC416, 0x0000);
                        re_real_ocp_phy_write(sc, 0xC416, 0x0050);
                        re_clear_phy_mcu_patch_request(sc);
                }
        }
}

static void re_disable_ocp_phy_power_saving(struct re_softc *sc)
{
        u_int16_t val;

        if (sc->re_type == MACFG_59 || sc->re_type == MACFG_60 ||
            sc->re_type == MACFG_62 || sc->re_type == MACFG_67 ||
            sc->re_type == MACFG_68 || sc->re_type == MACFG_69 ||
            sc->re_type == MACFG_70 || sc->re_type == MACFG_71 ||
            sc->re_type == MACFG_72 || sc->re_type == MACFG_73 ||
            sc->re_type == MACFG_74 || sc->re_type == MACFG_75 ||
            sc->re_type == MACFG_76) {
                val = re_ocp_phy_read(sc, 0x0C41, 0x13);
                if (val != 0x0500) {
                        re_set_phy_mcu_patch_request(sc);
                        re_ocp_phy_write(sc, 0x0C41, 0x13, 0x0000);
                        re_ocp_phy_write(sc, 0x0C41, 0x13, 0x0500);
                        re_clear_phy_mcu_patch_request(sc);
                }
        } else if (sc->re_type == MACFG_80 || sc->re_type == MACFG_81) {
                val = re_real_ocp_phy_read(sc, 0xC416);
                if (val != 0x0500) {
                        re_set_phy_mcu_patch_request(sc);
                        re_real_ocp_phy_write(sc, 0xC416, 0x0000);
                        re_real_ocp_phy_write(sc, 0xC416, 0x0500);
                        re_clear_phy_mcu_patch_request(sc);
                }
        }
}

static void re_hw_d3_para(struct re_softc *sc)
{
        switch (sc->re_type) {
        case MACFG_59:
        case MACFG_60:
        case MACFG_62:
        case MACFG_67:
        case MACFG_68:
        case MACFG_69:
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
        case MACFG_74:
        case MACFG_75:
        case MACFG_76:
        case MACFG_80:
        case MACFG_81:
                re_disable_ocp_phy_power_saving(sc);
                break;
        }
}

static void
re_add_sysctls(struct re_softc *sc)
{
        struct sysctl_ctx_list  *ctx;
        struct sysctl_oid_list  *children;

        ctx = device_get_sysctl_ctx(sc->dev);
        children = SYSCTL_CHILDREN(device_get_sysctl_tree(sc->dev));

#ifndef CTLFLAG_NEEDGIANT
#define CTLFLAG_NEEDGIANT 0
#endif

        SYSCTL_ADD_PROC(ctx, children, OID_AUTO, "driver_var",
                        CTLTYPE_INT | CTLFLAG_RW | CTLFLAG_NEEDGIANT, sc, 0,
                        re_sysctl_driver_variable, "I", "Driver Variables Information");

        SYSCTL_ADD_PROC(ctx, children, OID_AUTO, "stats",
                        CTLTYPE_INT | CTLFLAG_RW | CTLFLAG_NEEDGIANT, sc, 0,
                        re_sysctl_stats, "I", "Statistics Information");

        SYSCTL_ADD_PROC(ctx, children, OID_AUTO, "registers",
                        CTLTYPE_INT | CTLFLAG_RW | CTLFLAG_NEEDGIANT, sc, 0,
                        re_sysctl_registers, "I", "MAC IO Information");

        SYSCTL_ADD_PROC(ctx, children, OID_AUTO, "registers2",
                        CTLTYPE_INT | CTLFLAG_RW | CTLFLAG_NEEDGIANT, sc, 0,
                        re_sysctl_registers2, "I", "MAC IO 0x0000 to 0x4000 Information");

        SYSCTL_ADD_PROC(ctx, children, OID_AUTO, "registers3",
                        CTLTYPE_INT | CTLFLAG_RW | CTLFLAG_NEEDGIANT, sc, 0,
                        re_sysctl_registers3, "I", "MAC IO 0x4000 to 0x8000 Information");

        SYSCTL_ADD_PROC(ctx, children, OID_AUTO, "registers4",
                        CTLTYPE_INT | CTLFLAG_RW | CTLFLAG_NEEDGIANT, sc, 0,
                        re_sysctl_registers4, "I", "MAC IO 0x8000 to 0xC000 Information");

        SYSCTL_ADD_PROC(ctx, children, OID_AUTO, "registers5",
                        CTLTYPE_INT | CTLFLAG_RW | CTLFLAG_NEEDGIANT, sc, 0,
                        re_sysctl_registers5, "I", "MAC IO 0xC000 to 0x10000 Information");

        SYSCTL_ADD_PROC(ctx, children, OID_AUTO, "eth_phy",
                        CTLTYPE_INT | CTLFLAG_RW | CTLFLAG_NEEDGIANT, sc, 0,
                        re_sysctl_eth_phy, "I", "Ethernet PHY Information");

        SYSCTL_ADD_PROC(ctx, children, OID_AUTO, "rx_desc",
                        CTLTYPE_INT | CTLFLAG_RW | CTLFLAG_NEEDGIANT, sc, 0,
                        re_sysctl_dump_rx_desc, "I", "RX Descriptor Information");

        SYSCTL_ADD_PROC(ctx, children, OID_AUTO, "tx_desc",
                        CTLTYPE_INT | CTLFLAG_RW | CTLFLAG_NEEDGIANT, sc, 0,
                        re_sysctl_dump_tx_desc, "I", "TX Descriptor Information");

        if ((sc->re_if_flags & RL_FLAG_PCIE) != 0) {
                SYSCTL_ADD_PROC(ctx, children, OID_AUTO, "pcie_phy",
                                CTLTYPE_INT | CTLFLAG_RW | CTLFLAG_NEEDGIANT, sc, 0,
                                re_sysctl_pcie_phy, "I", "PCIE PHY Information");

                SYSCTL_ADD_PROC(ctx, children, OID_AUTO, "ext_regs",
                                CTLTYPE_INT | CTLFLAG_RW | CTLFLAG_NEEDGIANT, sc, 0,
                                re_sysctl_extended_registers, "I", "Extended Registers Information");

                SYSCTL_ADD_PROC(ctx, children, OID_AUTO, "pci_regs",
                                CTLTYPE_INT | CTLFLAG_RW | CTLFLAG_NEEDGIANT, sc, 0,
                                re_sysctl_pci_registers, "I", "PCI Configuration Information");

                SYSCTL_ADD_PROC(ctx, children, OID_AUTO, "msix_tbl",
                                CTLTYPE_INT | CTLFLAG_RW | CTLFLAG_NEEDGIANT, sc, 0,
                                re_sysctl_msix_tbl, "I", "MSIX Table Information");
        }

        switch (sc->re_type) {
        case MACFG_80:
        case MACFG_81:
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                SYSCTL_ADD_PROC(ctx, children, OID_AUTO, "eee",
                                CTLTYPE_INT | CTLFLAG_RW | CTLFLAG_NEEDGIANT, sc, 0,
                                re_sysctl_eee, "I", "EEE Information");
                break;
        default:
                break;
        }
}

static int
re_sysctl_driver_variable(SYSCTL_HANDLER_ARGS)
{
        struct re_softc         *sc;
        int                     error, result;

        result = -1;
        error = sysctl_handle_int(oidp, &result, 0, req);
        if (error || req->newptr == NULL)
                return (error);

        if (result == 1) {
                sc = (struct re_softc *)arg1;
                RE_LOCK(sc);

                printf("%s Driver Variables:\n", device_get_nameunit(sc->dev));

                printf("driver version\t%s\n", RE_VERSION);
                printf("if_drv_flags\t0x%08x\n", sc->re_ifp->if_drv_flags);
                printf("re_type\t%d\n", sc->re_type);
                printf("re_res_id\t%d\n", sc->re_res_id);
                printf("re_res_type\t%d\n", sc->re_res_type);
                printf("re_8169_MacVersion\t%d\n", sc->re_8169_MacVersion);
                printf("re_8169_PhyVersion\t%d\n", sc->re_8169_PhyVersion);
                printf("tx buffer numer\t%d\n", RE_TX_BUF_NUM);
                printf("rx buffer numer\t%d\n", RE_RX_BUF_NUM);
                for (int i = 0; i < RL_RX_QUEUE_NUM; i++)
                        printf("rx_cur_index%d\t%d\n", i, sc->re_desc.rx_cur_index[i]);
                for (int i = 0; i < RL_TX_QUEUE_NUM; i++) {
                        printf("tx_cur_index%d\t%d\n", i, sc->re_desc.tx_cur_index[i]);
                        printf("tx_last_index%d\t%d\n", i, sc->re_desc.tx_last_index[i]);
                }
                printf("rx_fifo_overflow\t%d\n", sc->rx_fifo_overflow);
                printf("driver_detach\t%d\n", sc->driver_detach);
                printf("interface name\tre%d\n", sc->re_unit);
                printf("re_revid\t0x%02x\n", sc->re_revid);
                printf("re_vendor_id\t0x%04x\n", sc->re_vendor_id);
                printf("re_device_id\t0x%04x\n", sc->re_device_id);
                printf("re_subvendor_id\t0x%04x\n", sc->re_subvendor_id);
                printf("re_subdevice_id\t0x%04x\n", sc->re_subdevice_id);
                printf("max_jumbo_frame_size\t%d\n", sc->max_jumbo_frame_size);
                printf("re_rx_mbuf_sz\t%d\n", sc->re_rx_mbuf_sz);
                printf("re_rx_desc_buf_sz\t%d\n", sc->re_rx_desc_buf_sz);
                printf("re_if_flags\t0x%08x\n", sc->re_if_flags);
                printf("re_tx_cstag\t%d\n", sc->re_tx_cstag);
                printf("re_rx_cstag\t%d\n", sc->re_rx_cstag);
                printf("RequireAdcBiasPatch\t%d\n", sc->RequireAdcBiasPatch);
                printf("RequireAdjustUpsTxLinkPulseTiming\t%d\n", sc->RequireAdjustUpsTxLinkPulseTiming);
                printf("RequiredSecLanDonglePatch\t%d\n", sc->RequiredSecLanDonglePatch);
                printf("RequiredPfmPatch\t%d\n", sc->RequiredPfmPatch);
                printf("RequirePhyMdiSwapPatch\t%d\n", sc->RequirePhyMdiSwapPatch);
                printf("re_efuse_ver\t%d\n", sc->re_efuse_ver);
                printf("re_sw_ram_code_ver\t0x%x\n", sc->re_sw_ram_code_ver);
                printf("re_hw_ram_code_ver\t0x%x\n", sc->re_hw_ram_code_ver);
                printf("cur_page\t0x%x\n", sc->cur_page);
                printf("phy_reg_anlpar\t0x%x\n", sc->phy_reg_anlpar);
                printf("re_hw_enable_msi_msix\t%d\n", sc->re_hw_enable_msi_msix);
                printf("re_coalesce_tx_pkt\t%d\n", sc->re_coalesce_tx_pkt);
                printf("link_state\t%s\n", sc->link_state==2?"up":(sc->link_state==1?"down":"unknown"));
                printf("prohibit_access_reg\t%d\n", sc->prohibit_access_reg);
                printf("re_hw_supp_now_is_oob_ver\t%d\n", sc->re_hw_supp_now_is_oob_ver);
                printf("hw_hw_supp_serdes_phy_ver\t%d\n", sc->hw_hw_supp_serdes_phy_ver);
                printf("HwSuppDashVer\t%d\n", sc->HwSuppDashVer);
                printf("re_dash\t%d\n", sc->re_dash);
                printf("re_dash_fw_ver\t0x%08x\n", sc->re_dash_fw_ver);
                printf("HwPkgDet\t%d\n", sc->HwPkgDet);
                printf("HwFiberModeVer\t%d\n", sc->HwFiberModeVer);
                printf("HwFiberStat\t%d\n", sc->HwFiberStat);
                printf("HwSuppExtendTallyCounterVer\t%d\n", sc->HwSuppExtendTallyCounterVer);
                printf("HwSuppMacMcuVer\t%d\n", sc->HwSuppMacMcuVer);
                printf("MacMcuPageSize\t%d\n", sc->MacMcuPageSize);
                for (int i = 0; i < RL_RX_QUEUE_NUM; i++)
                        printf("rx_desc_tag maxsize%d\t%zd\n", i, sc->re_desc.rx_desc_tag[i]->common.maxsize);
                for (int i = 0; i < RL_TX_QUEUE_NUM; i++)
                        printf("tx_desc_tag maxsize%d\t%zd\n", i, sc->re_desc.tx_desc_tag[i]->common.maxsize);
                printf("re_tally maxsize\t%zd\n", sc->re_tally.re_stag->common.maxsize);
                printf("random_mac\t%d\n", sc->random_mac);
                printf("org_mac_addr\t%6D\n", sc->org_mac_addr, ":");
#if OS_VER < VERSION(6,0)
                printf("dev_addr\t%6D\n", (char *)&sc->arpcom.ac_enaddr, ":");
#elif OS_VER < VERSION(7,0)
                printf("dev_addr\t%6D\n", IFP2ENADDR(sc->re_ifp), ":");
#else
                printf("dev_addr\t%6D\n", IF_LLADDR(sc->re_ifp), ":");
#endif
                printf("msi_disable\t%d\n", msi_disable);
                printf("msix_disable\t%d\n", msix_disable);
                printf("eee_enable\t%d\n", eee_enable);
                printf("prefer_iomap\t%d\n", prefer_iomap);
                printf("phy_power_saving\t%d\n", phy_power_saving);
                printf("phy_mdix_mode\t%d\n", phy_mdix_mode);
                printf("s5wol\t%d\n", s5wol);
                printf("s0_magic_packet\t%d\n", s0_magic_packet);
                printf("config_soc_lan\t%d\n", config_soc_lan);
                printf("interrupt_mitigation\t%d\n", interrupt_mitigation);
                printf("re_lro_entry_count\t%d\n", re_lro_entry_count);
                printf("re_lro_mbufq_depth\t%d\n", re_lro_mbufq_depth);
                printf("use_new_intr_mapping\t%d\n", sc->use_new_intr_mapping);
                printf("msic\t%d\n", pci_msi_count(sc->dev));
                printf("msixc\t%d\n", pci_msix_count(sc->dev));

                RE_UNLOCK(sc);
        }

        return (error);
}

static int
re_sysctl_stats(SYSCTL_HANDLER_ARGS)
{
        struct re_softc         *sc;
        struct re_stats         *stats;
        int                     error, i, result;
        bool                    extend_stats;

        result = -1;
        error = sysctl_handle_int(oidp, &result, 0, req);
        if (error || req->newptr == NULL)
                return (error);

        if (result == 1) {
                sc = (struct re_softc *)arg1;
                RE_LOCK(sc);
                extend_stats = false;
                if (sc->HwSuppExtendTallyCounterVer > 0)
                        extend_stats = true;
                if ((sc->re_ifp->if_drv_flags & IFF_DRV_RUNNING) == 0) {
                        RE_UNLOCK(sc);
                        goto done;
                }

                if (extend_stats)
                        re_set_mac_ocp_bit(sc, 0xEA84, (BIT_1 | BIT_0));

                bus_dmamap_sync(sc->re_tally.re_stag,
                                sc->re_tally.re_smap, BUS_DMASYNC_PREREAD);
                CSR_WRITE_4(sc, RE_DUMPSTATS_HI,
                            RL_ADDR_HI(sc->re_tally.re_stats_addr));
                CSR_WRITE_4(sc, RE_DUMPSTATS_LO,
                            RL_ADDR_LO(sc->re_tally.re_stats_addr));
                CSR_WRITE_4(sc, RE_DUMPSTATS_LO,
                            RL_ADDR_LO(sc->re_tally.re_stats_addr |
                                       RE_DUMPSTATS_START));
                for (i = RE_TIMEOUT; i > 0; i--) {
                        if ((CSR_READ_4(sc, RE_DUMPSTATS_LO) &
                             RE_DUMPSTATS_START) == 0)
                                break;
                        DELAY(1000);
                }
                bus_dmamap_sync(sc->re_tally.re_stag,
                                sc->re_tally.re_smap, BUS_DMASYNC_POSTREAD);

                if (extend_stats)
                        re_clear_mac_ocp_bit(sc, 0xEA84, (BIT_1 | BIT_0));

                RE_UNLOCK(sc);
                if (i == 0) {
                        device_printf(sc->dev,
                                      "DUMP statistics request timed out\n");
                        return (ETIMEDOUT);
                }
done:
                stats = sc->re_tally.re_stats;
                printf("%s statistics:\n", device_get_nameunit(sc->dev));
                printf("Tx frames : %ju\n",
                       (uintmax_t)le64toh(stats->re_tx_pkts));
                printf("Rx frames : %ju\n",
                       (uintmax_t)le64toh(stats->re_rx_pkts));
                printf("Tx errors : %ju\n",
                       (uintmax_t)le64toh(stats->re_tx_errs));
                printf("Rx errors : %u\n",
                       le32toh(stats->re_rx_errs));
                printf("Rx missed frames : %u\n",
                       (uint32_t)le16toh(stats->re_missed_pkts));
                printf("Rx frame alignment errs : %u\n",
                       (uint32_t)le16toh(stats->re_rx_framealign_errs));
                printf("Tx single collisions : %u\n",
                       le32toh(stats->re_tx_onecoll));
                printf("Tx multiple collisions : %u\n",
                       le32toh(stats->re_tx_multicolls));
                printf("Rx unicast frames : %ju\n",
                       (uintmax_t)le64toh(stats->re_rx_ucasts));
                printf("Rx broadcast frames : %ju\n",
                       (uintmax_t)le64toh(stats->re_rx_bcasts));
                printf("Rx multicast frames : %u\n",
                       le32toh(stats->re_rx_mcasts));
                printf("Tx aborts : %u\n",
                       (uint32_t)le16toh(stats->re_tx_aborts));
                printf("Tx underruns : %u\n",
                       (uint32_t)le16toh(stats->re_rx_underruns));

                if (extend_stats) {
                        printf("%s extend statistics:\n", device_get_nameunit(sc->dev));
                        printf("Tx octets : %ju\n",
                               (uintmax_t)le64toh(stats->re_tx_octets));
                        printf("Rx octets : %ju\n",
                               (uintmax_t)le64toh(stats->re_rx_octets));
                        printf("Rx multicast64 : %ju\n",
                               (uintmax_t)le64toh(stats->re_rx_multicast64));
                        printf("Rx unicast64 : %ju\n",
                               (uintmax_t)le64toh(stats->re_tx_unicast64));
                        printf("Tx broadcast64 : %ju\n",
                               (uintmax_t)le64toh(stats->re_tx_broadcast64));
                        printf("Tx multicast64 : %ju\n",
                               (uintmax_t)le64toh(stats->re_tx_multicast64));
                        printf("Tx pause on frames : %u\n",
                               (uint32_t)le32toh(stats->re_tx_pause_on));
                        printf("TTx pause off frames : %u\n",
                               (uint32_t)le32toh(stats->re_tx_pause_off));
                        printf("Tx pause all frames : %u\n",
                               (uint32_t)le32toh(stats->re_tx_pause_all));
                        printf("Tx deferred frames : %u\n",
                               (uint32_t)le32toh(stats->re_tx_deferred));
                        printf("Tx late collisions : %u\n",
                               (uint32_t)le32toh(stats->re_tx_late_collision));
                        printf("Tx all collisions : %u\n",
                               (uint32_t)le32toh(stats->re_tx_all_collision));
                        printf("Tx aborts32 : %u\n",
                               (uint32_t)le32toh(stats->re_tx_aborted32));
                        printf("Rx alignment errs32 : %u\n",
                               (uint32_t)le32toh(stats->re_align_errors32));
                        printf("Rx frame too long : %u\n",
                               (uint32_t)le32toh(stats->re_rx_frame_too_long));
                        printf("Rx runt : %u\n",
                               (uint32_t)le32toh(stats->re_rx_runt));
                        printf("Rx pause on frames : %u\n",
                               (uint32_t)le32toh(stats->re_rx_pause_on));
                        printf("Rx pause off frames : %u\n",
                               (uint32_t)le32toh(stats->re_rx_pause_off));
                        printf("Rx pause all frames : %u\n",
                               (uint32_t)le32toh(stats->re_rx_pause_all));
                        printf("Rx unknown opcode : %u\n",
                               (uint32_t)le32toh(stats->re_rx_unknown_opcode));
                        printf("Rx mac error : %u\n",
                               (uint32_t)le32toh(stats->re_rx_mac_error));
                        printf("Tx underruns32 : %u\n",
                               (uint32_t)le32toh(stats->re_tx_underrun32));
                        printf("Rx mac missed : %u\n",
                               (uint32_t)le32toh(stats->re_rx_mac_missed));
                        printf("Rx tcam drops : %u\n",
                               (uint32_t)le32toh(stats->re_rx_tcam_dropped));
                        printf("Tx desc unavailable : %u\n",
                               (uint32_t)le32toh(stats->re_tdu));
                        printf("Rx desc unavailable : %u\n",
                               (uint32_t)le32toh(stats->re_rdu));
                }
        }

        return (error);
}

static void
re_printf_macio(struct re_softc *sc,
                u_int32_t start,
                u_int32_t end)
{
        int i, n;

        printf("\n%s mac io start:0x%05x end:0x%05x:\n",
               device_get_nameunit(sc->dev),
               start, end);

        for (n=start; n<end;) {
                printf("\n0x%02x:\t", n);

                for (i=0; i<16 && n<end; i++, n++)
                        printf("%02x ", CSR_READ_1(sc, n));
        }

        return;
}

static int
re_sysctl_registers(SYSCTL_HANDLER_ARGS)
{
        struct re_softc         *sc;
        int                     error, max, result;

        result = -1;
        error = sysctl_handle_int(oidp, &result, 0, req);
        if (error || req->newptr == NULL)
                return (error);

        if (result == 1) {
                sc = (struct re_softc *)arg1;
                RE_LOCK(sc);

                max = min(256, rman_get_size(sc->re_res));
                re_printf_macio(sc, 0, max);

                RE_UNLOCK(sc);
        }

        return (error);
}

static int
re_sysctl_registers2(SYSCTL_HANDLER_ARGS)
{
        struct re_softc         *sc;
        int                     error, max, result;

        result = -1;
        error = sysctl_handle_int(oidp, &result, 0, req);
        if (error || req->newptr == NULL)
                return (error);

        if (result == 1) {
                sc = (struct re_softc *)arg1;
                RE_LOCK(sc);

                max = min(0x4000, rman_get_size(sc->re_res));
                re_printf_macio(sc, 0, max);

                RE_UNLOCK(sc);
        }

        return (error);
}

static int
re_sysctl_registers3(SYSCTL_HANDLER_ARGS)
{
        struct re_softc         *sc;
        int                     error, max, result;

        result = -1;
        error = sysctl_handle_int(oidp, &result, 0, req);
        if (error || req->newptr == NULL)
                return (error);

        if (result == 1) {
                sc = (struct re_softc *)arg1;
                RE_LOCK(sc);

                max = min(0x8000, rman_get_size(sc->re_res));
                re_printf_macio(sc, 0x4000, max);

                RE_UNLOCK(sc);
        }

        return (error);
}

static int
re_sysctl_registers4(SYSCTL_HANDLER_ARGS)
{
        struct re_softc         *sc;
        int                     error, max, result;

        result = -1;
        error = sysctl_handle_int(oidp, &result, 0, req);
        if (error || req->newptr == NULL)
                return (error);

        if (result == 1) {
                sc = (struct re_softc *)arg1;
                RE_LOCK(sc);

                max = min(0xC000, rman_get_size(sc->re_res));
                re_printf_macio(sc, 0x8000, max);

                RE_UNLOCK(sc);
        }

        return (error);
}

static int
re_sysctl_registers5(SYSCTL_HANDLER_ARGS)
{
        struct re_softc         *sc;
        int                     error, max, result;

        result = -1;
        error = sysctl_handle_int(oidp, &result, 0, req);
        if (error || req->newptr == NULL)
                return (error);

        if (result == 1) {
                sc = (struct re_softc *)arg1;
                RE_LOCK(sc);

                max = min(0x10000, rman_get_size(sc->re_res));
                re_printf_macio(sc, 0xC000, max);

                RE_UNLOCK(sc);
        }

        return (error);
}

static int
re_sysctl_eth_phy(SYSCTL_HANDLER_ARGS)
{
        struct re_softc         *sc;
        int                     error, i, n, max, result;

        result = -1;
        error = sysctl_handle_int(oidp, &result, 0, req);
        if (error || req->newptr == NULL)
                return (error);

        if (result == 1) {
                sc = (struct re_softc *)arg1;
                RE_LOCK(sc);

                printf("%s ethernet phy:\n", device_get_nameunit(sc->dev));

                max = 16;
                re_mdio_write(sc, 0x1F, 0x0000);
                for (n=0; n<max;) {
                        printf("\n0x%02x:\t", n);

                        for (i = 0; i < 8 && n < max; i++, n++)
                                printf("%04x ", re_mdio_read(sc, n));
                }

                RE_UNLOCK(sc);
        }

        return (error);
}

static void
re_dump_desc(void *desc_base, uint32_t alloc_size)
{
        uint32_t *pdword;
        int i;

        if (desc_base == NULL ||
            alloc_size == 0)
                return;

        pdword = (uint32_t*)desc_base;
        for (i=0; i<(alloc_size/4); i++) {
                if (!(i % 4))
                        printf("\n%04x ", i);
                printf("%08x ", pdword[i]);
        }

        printf("\n");
        return;
}

static int
re_sysctl_dump_rx_desc(SYSCTL_HANDLER_ARGS)
{
        struct re_softc         *sc;
        bus_size_t              rx_list_size;
        int                     error, result;

        result = -1;
        error = sysctl_handle_int(oidp, &result, 0, req);
        if (error || req->newptr == NULL)
                return (error);

        if (result == 1) {
                sc = (struct re_softc *)arg1;
                RE_LOCK(sc);

                for (int i = 0; i < RL_RX_QUEUE_NUM; i++) {
                        printf("%s rx desc(%d):%d\n", device_get_nameunit(sc->dev),
                               i, RE_RX_BUF_NUM);
                        rx_list_size = sc->re_desc.rx_desc_tag[i]->common.maxsize;
                        re_dump_desc((void*)sc->re_desc.rx_desc[i], rx_list_size);
                }

                RE_UNLOCK(sc);
        }

        return (error);
}

static int
re_sysctl_dump_tx_desc(SYSCTL_HANDLER_ARGS)
{
        struct re_softc         *sc;
        bus_size_t              tx_list_size;
        int                     error, result;

        result = -1;
        error = sysctl_handle_int(oidp, &result, 0, req);
        if (error || req->newptr == NULL)
                return (error);

        if (result == 1) {
                sc = (struct re_softc *)arg1;
                RE_LOCK(sc);

                for (int i = 0; i < RL_TX_QUEUE_NUM; i++) {
                        printf("%s tx desc(%d):%d\n", device_get_nameunit(sc->dev),
                               i, RE_TX_BUF_NUM);
                        tx_list_size = sc->re_desc.tx_desc_tag[i]->common.maxsize;
                        re_dump_desc((void*)sc->re_desc.tx_desc[i], tx_list_size);
                }

                RE_UNLOCK(sc);
        }

        return (error);
}

static int
re_sysctl_pcie_phy(SYSCTL_HANDLER_ARGS)
{
        struct re_softc         *sc;
        int                     error, i, n, max, result;

        result = -1;
        error = sysctl_handle_int(oidp, &result, 0, req);
        if (error || req->newptr == NULL)
                return (error);

        if (result == 1) {
                sc = (struct re_softc *)arg1;
                RE_LOCK(sc);

                printf("%s pcie phy:\n", device_get_nameunit(sc->dev));

                max = 31;
                for (n=0; n<max;) {
                        printf("\n0x%02x:\t", n);

                        for (i = 0; i < 8 && n < max; i++, n++)
                                printf("%04x ", re_ephy_read(sc, n));
                }

                RE_UNLOCK(sc);
        }

        return (error);
}

static int
re_sysctl_extended_registers(SYSCTL_HANDLER_ARGS)
{
        struct re_softc         *sc;
        int                     error, i, n, max, result;

        result = -1;
        error = sysctl_handle_int(oidp, &result, 0, req);
        if (error || req->newptr == NULL)
                return (error);

        if (result == 1) {
                sc = (struct re_softc *)arg1;
                RE_LOCK(sc);

                printf("%s extended registers:\n", device_get_nameunit(sc->dev));

                max = 0x100;
                for (n=0; n<max;) {
                        printf("\n0x%02x:\t", n);

                        for (i = 0; i < 4 && n < max; i++, n+=4)
                                printf("%08x ", re_eri_read(sc, n, 4, ERIAR_ExGMAC));
                }

                RE_UNLOCK(sc);
        }

        return (error);
}

static int
re_sysctl_pci_registers(SYSCTL_HANDLER_ARGS)
{
        struct re_softc         *sc;
        int                     error, i, n, max, result;

        result = -1;
        error = sysctl_handle_int(oidp, &result, 0, req);
        if (error || req->newptr == NULL)
                return (error);

        if (result == 1) {
                sc = (struct re_softc *)arg1;
                RE_LOCK(sc);

                printf("%s pci registers:\n", device_get_nameunit(sc->dev));

                max = 0x100;
                for (n=0; n<max;) {
                        printf("\n0x%03x:\t", n);

                        for (i = 0; i < 4 && n < max; i++, n+=4)
                                printf("%08x ", re_csi_read(sc, n | 0xF000));
                }

                n = 0x108;
                printf("\n0x%03x:\t%08x ", n, re_csi_read(sc, n | 0xF000));

                n = 0x110;
                printf("\n0x%03x:\t%08x ", n, re_csi_read(sc, n | 0xF000));

                switch(sc->re_type) {
                case MACFG_62:
                case MACFG_67:
                case MACFG_68:
                case MACFG_69:
                case MACFG_74:
                case MACFG_75:
                case MACFG_76:
                        n = 0x180;
                        break;
                case MACFG_70:
                case MACFG_71:
                case MACFG_72:
                case MACFG_73:
                case MACFG_82:
                case MACFG_83:
                        n = 0x214;
                        break;
                case MACFG_80:
                case MACFG_81:
                        n = 0x264;
                        break;
                case MACFG_84:
                case MACFG_85:
                case MACFG_86:
                case MACFG_87:
                        n = 0x210;
                        break;
                case MACFG_88:
                        n = 0x184;
                        break;
                case MACFG_90:
                case MACFG_91:
                case MACFG_92:
                case MACFG_100:
                case MACFG_101:
                        n = 0x22C;
                        break;
                default:
                        n =  0;
                        break;
                }
                if (n > 0)
                        printf("\n0x%03x:\t%08x ", n, re_csi_other_fun_read(sc, 0, n | 0xF000));

                n = 0x70c;
                printf("\n0x%03x:\t%08x ", n, re_csi_read(sc, n | 0xF000));

                RE_UNLOCK(sc);
        }

        return (error);
}

static void
re_dump_msix_tbl(struct re_softc *sc)
{
        int msixc;
        int i, j;

        msixc = pci_msix_count(sc->dev);
        if (msixc == 0)
                return;
        for (i=0; i<msixc; i++) {
                printf("\n0x%04x:\t", i);

                for (j=0; j<4; j++)
                        printf("%08x ",
                               RE_MSIX_TBL_READ_4(
                                       sc, i*0x10 + 4 * j));
        }
}

static int
re_sysctl_msix_tbl(SYSCTL_HANDLER_ARGS)
{
        struct re_softc         *sc;
        int                     error, result;

        result = -1;
        error = sysctl_handle_int(oidp, &result, 0, req);
        if (error || req->newptr == NULL)
                return (error);

        sc = (struct re_softc *)arg1;

        if (sc->re_res_pba == NULL)
                return EPERM;

        if (result == 1) {

                RE_LOCK(sc);

                printf("%s msix table:\n", device_get_nameunit(sc->dev));

                re_dump_msix_tbl(sc);

                RE_UNLOCK(sc);
        }

        return (error);
}

static int
re_sysctl_eee(SYSCTL_HANDLER_ARGS)
{
        struct re_softc         *sc;
        int                     error, result;

        result = -1;
        error = sysctl_handle_int(oidp, &result, 0, req);
        if (error || req->newptr == NULL)
                return (error);

        sc = (struct re_softc *)arg1;

        if (result == 1) {
                uint16_t val, adv, adv2, lp_adv, lp_adv2;

                RE_LOCK(sc);

                val = re_mac_ocp_read(sc, 0xE040);
                val &= BIT_1 | BIT_0;

                adv = re_real_ocp_phy_read(sc, 0xA5D0);
                adv2 = re_real_ocp_phy_read(sc, 0xA6D4);
                lp_adv = re_real_ocp_phy_read(sc, 0xA5D2);
                lp_adv2 = re_real_ocp_phy_read(sc, 0xA6D0);

                printf("%s EEE setting:\n", device_get_nameunit(sc->dev));
                printf("EEE status: %s - %s\n", (!!val)?"enabled":"disabled",
                       ((adv & lp_adv) || (adv2 & lp_adv2))?"active":"inactive");

                printf("Advertised EEE link modes:\n");
                if (adv & BIT_1)
                        printf("IFM_100_TX \n");
                if (adv & BIT_2)
                        printf("IFM_1000_T \n");
                if (adv2 & BIT_0)
                        printf("IFM_2500_T \n");
                if (adv2 & BIT_1)
                        printf("IFM_5000_T \n");
                if (adv & BIT_3)
                        printf("IFM_10G_T \n");

                printf("Link partner advertised EEE link modes:\n");
                if (lp_adv & BIT_1)
                        printf("IFM_100_TX \n");
                if (lp_adv & BIT_2)
                        printf("IFM_1000_T \n");
                if (lp_adv2 & BIT_0)
                        printf("IFM_2500_T \n");
                if (lp_adv2 & BIT_1)
                        printf("IFM_5000_T \n");
                if (lp_adv & BIT_3)
                        printf("IFM_10G_T \n");

                RE_UNLOCK(sc);
        }

        return (error);
}

static void
re_setup_msix_tbl(struct re_softc *sc)
{
        u_int32_t msix_tbl[4];
        int msixc;
        int i, j;

        msixc = pci_msix_count(sc->dev);
        if (msixc == 0)
                return;

        for (i=0; i<4; i++)
                msix_tbl[i] = RE_MSIX_TBL_READ_4(sc, 4*i);

        for (i=1; i<msixc; i++) {
                for (j=0; j<4; j++)
                        RE_MSIX_TBL_WRITE_4(sc, i*0x10 + 4 * j, msix_tbl[j]);
        }
}

/*
* Attach the interface. Allocate softc structures, do ifmedia
* setup and ethernet/BPF attach.
*/
static int re_attach(device_t dev)
{
        /*int			s;*/
        bus_size_t		rx_list_size, tx_list_size;
        u_char			eaddr[ETHER_ADDR_LEN];
        u_int32_t		command;
        struct re_softc		*sc;
        struct ifnet		*ifp;
        int			unit, error = 0, rid;
        int     reg;
        int		msic=0, msixc=0;

        /*s = splimp();*/

        sc = device_get_softc(dev);
        unit = device_get_unit(dev);
        bzero(sc, sizeof(struct re_softc));
        RE_LOCK_INIT(sc,device_get_nameunit(dev));
        sc->dev = dev;

        sc->driver_detach = 0;

        sc->re_vendor_id  = pci_get_vendor(dev);
        sc->re_device_id = pci_get_device(dev);
        sc->re_subvendor_id = pci_get_subvendor(dev);
        sc->re_subdevice_id = pci_get_subdevice(dev);
        sc->re_revid = pci_get_revid(dev);
        pci_enable_busmaster(dev);

        /*
         * Map control/status registers.
         */
        command = pci_read_config(dev, PCIR_COMMAND, 4);
        command |= (PCIM_CMD_PORTEN|PCIM_CMD_MEMEN|PCIM_CMD_BUSMASTEREN);
        pci_write_config(dev, PCIR_COMMAND, command, 4);

        if (prefer_iomap == 0) {
                sc->re_res_id = PCIR_BAR(2);
                sc->re_res_type = SYS_RES_MEMORY;
                /* PCI NIC use different BARs. */
                if (sc->re_device_id == RT_DEVICEID_8169 || sc->re_device_id == RT_DEVICEID_8169SC)
                        sc->re_res_id = PCIR_BAR(1);
        } else {
                sc->re_res_id = PCIR_BAR(0);
                sc->re_res_type = SYS_RES_IOPORT;
        }
        sc->re_res = bus_alloc_resource(dev, sc->re_res_type, &sc->re_res_id,
                                        0, ~0, 1, RF_ACTIVE);
        if (sc->re_res == NULL && prefer_iomap == 0) {
                sc->re_res_id = PCIR_BAR(0);
                sc->re_res_type = SYS_RES_IOPORT;
                sc->re_res = bus_alloc_resource(dev, sc->re_res_type, &sc->re_res_id,
                                                0, ~0, 1, RF_ACTIVE);
        }

        if (sc->re_res == NULL) {
                device_printf(dev,"couldn't map ports/memory\n");
                error = ENXIO;
                goto fail;
        }

        if (sc->re_res_type == SYS_RES_IOPORT)
                device_printf(dev, "Using I/O Ports\n");
        else
                device_printf(dev, "Using Memory Mapping!\n");

        sc->re_btag = rman_get_bustag(sc->re_res);
        sc->re_bhandle = rman_get_bushandle(sc->re_res);

        error = re_check_mac_version(sc);

        if (error) {
                goto fail;
        }

        re_init_software_variable(sc);

#if OS_VER >= VERSION(7,0)
        msic = pci_msi_count(dev);
        msixc = pci_msix_count(dev);
        if (pci_find_cap(dev, PCIY_EXPRESS, &reg) == 0) {
                sc->re_if_flags |= RL_FLAG_PCIE;
                sc->re_expcap = reg;
        } else {
                sc->re_if_flags &= ~RL_FLAG_PCIE;
                sc->re_expcap = 0;
        }

        //device_printf(dev, "MSI count : %d\n", msic);
        //device_printf(dev, "MSI-X count : %d\n", msixc);
        if (sc->re_hw_enable_msi_msix == FALSE) {
                msixc = 0;
                msic = 0;
        }
        if (msix_disable > 0)
                msixc = 0;
        if (msi_disable > 0)
                msic = 0;

        /* Prefer MSI-X to MSI. */
        if (msixc > 0) {
                rid = PCIR_BAR(4);
                msixc = RL_MSI_MESSAGES;
                sc->re_res_pba = bus_alloc_resource_any(dev, SYS_RES_MEMORY,
                                                        &rid, RF_ACTIVE);
                if (sc->re_res_pba == NULL) {
                        device_printf(dev,
                                      "could not allocate MSI-X PBA resource\n");
                }
                if (sc->re_res_pba != NULL &&
                    pci_alloc_msix(dev, &msixc) == 0) {
                        if (msixc == RL_MSI_MESSAGES) {
                                device_printf(dev, "Using %d MSI-X message\n",
                                              msixc);
                                sc->re_if_flags |= RL_FLAG_MSIX;
                        } else
                                pci_release_msi(dev);
                }
                if ((sc->re_if_flags & RL_FLAG_MSIX) == 0) {
                        if (sc->re_res_pba != NULL)
                                bus_release_resource(dev, SYS_RES_MEMORY, rid,
                                                     sc->re_res_pba);
                        sc->re_res_pba = NULL;
                        msixc = 0;
                }

                if (sc->re_res_pba != NULL) {
                        sc->re_msix_tbl_tag = rman_get_bustag(sc->re_res_pba);
                        sc->re_msix_tbl_handle = rman_get_bushandle(sc->re_res_pba);
                }
        }

        /* Prefer MSI to INTx. */
        if (msixc == 0 && msic > 0) {
                msic = RL_MSI_MESSAGES;
                if (pci_alloc_msi(dev, &msic) == 0) {
                        if (msic == RL_MSI_MESSAGES) {
                                device_printf(dev, "Using %d MSI message\n",
                                              msic);
                                sc->re_if_flags |= RL_FLAG_MSI;
                        } else
                                pci_release_msi(dev);
                }
                if ((sc->re_if_flags & RL_FLAG_MSI) == 0)
                        msic = 0;
        }
#endif //OS_VER >= VERSION(7,0)

        if ((sc->re_if_flags & (RL_FLAG_MSI | RL_FLAG_MSIX)) == 0) {
                rid = 0;
                sc->re_irq = bus_alloc_resource(dev, SYS_RES_IRQ, &rid, 0, ~0, 1,
                                                RF_SHAREABLE | RF_ACTIVE);

                if (sc->re_irq == NULL) {
                        device_printf(dev,"couldn't map interrupt\n");
                        error = ENXIO;
                        goto fail;
                }
                device_printf(dev, "Using line-based interrupt\n");
        } else {
                rid = 1;
                sc->re_irq = bus_alloc_resource_any(dev,
                                                    SYS_RES_IRQ, &rid, RF_ACTIVE);
                if (sc->re_irq == NULL) {
                        device_printf(dev,
                                      "couldn't allocate IRQ resources for "
                                      "message %d\n", rid);
                        error = ENXIO;
                        goto fail;
                }
        }

#if OS_VER >= VERSION(7,3)
        /* Disable ASPM L0S/L1 and Clock Request. */
        if (sc->re_expcap != 0) {
                u_int32_t		cap, ctl;
                cap = pci_read_config(dev, sc->re_expcap +
                                      RE_PCIER_LINK_CAP, 2);
                if ((cap & RE_PCIEM_LINK_CAP_ASPM) != 0) {
                        ctl = pci_read_config(dev, sc->re_expcap +
                                              RE_PCIER_LINK_CTL, 2);
                        if ((ctl & 0x0103) != 0) {
                                ctl &= ~0x0103;
                                pci_write_config(dev, sc->re_expcap +
                                                 RE_PCIER_LINK_CTL, ctl, 2);
                                device_printf(dev, "ASPM disabled\n");
                        }
                } else
                        device_printf(dev, "no ASPM capability\n");
        }
#endif //OS_VER >= VERSION(7,3)

        re_init_timer(sc);

        RE_LOCK(sc);
        re_exit_oob(sc);
        re_hw_init(sc);
        RE_UNLOCK(sc);

        /*
         * Reset the adapter. Only take the lock here as it's needed in
         * order to call re_reset().
         */
        RE_LOCK(sc);
        re_reset(sc);
        RE_UNLOCK(sc);

        sc->re_unit = unit;

        if (sc->re_type == MACFG_3) {	/* Change PCI Latency time*/
                pci_write_config(dev, RE_PCI_LATENCY_TIMER, 0x40, 1);
        }

        error = bus_dma_tag_create(
#if OS_VER < VERSION(7,0)
                        NULL,
#else
                        bus_get_dma_tag(dev),		/* parent */
#endif
                        1, 0,		/* alignment, boundary */
                        BUS_SPACE_MAXADDR,		/* lowaddr */
                        BUS_SPACE_MAXADDR,		/* highaddr */
                        NULL, NULL,			/* filter, filterarg */
                        BUS_SPACE_MAXSIZE_32BIT,	/* maxsize */
                        0,				/* nsegments */
                        BUS_SPACE_MAXSIZE_32BIT,	/* maxsegsize */
                        0,				/* flags */
                        NULL, NULL,			/* lockfunc, lockarg */
                        &sc->re_parent_tag);

        rx_list_size = sizeof(union RxDesc) * (RE_RX_BUF_NUM + 1);
        for (int i = 0; i < RL_RX_QUEUE_NUM; i++) {
                error = bus_dma_tag_create(
                                sc->re_parent_tag,
                                RE_DESC_ALIGN, 0,		/* alignment, boundary */
                                BUS_SPACE_MAXADDR,		/* lowaddr */
                                BUS_SPACE_MAXADDR,		/* highaddr */
                                NULL, NULL,			/* filter, filterarg */
                                rx_list_size,				/* maxsize */
                                1,				/* nsegments */
                                rx_list_size,				/* maxsegsize */
                                0,				/* flags */
                                NULL, NULL,			/* lockfunc, lockarg */
                                &sc->re_desc.rx_desc_tag[i]);
                if (error) {
                        device_printf(dev,"bus_dma_tag_create fail\n");
                        goto fail;
                }

                error = bus_dmamem_alloc(sc->re_desc.rx_desc_tag[i],
                                         (void**) &sc->re_desc.rx_desc[i],
                                         BUS_DMA_WAITOK|BUS_DMA_COHERENT|BUS_DMA_ZERO,
                                         &sc->re_desc.rx_desc_dmamap[i]);
                if (error) {
                        device_printf(dev,"bus_dmamem_alloc fail\n");
                        goto fail;
                }
        }

        tx_list_size = sizeof(union TxDesc) * (RE_TX_BUF_NUM + 1);
        for (int i=0; i<RL_TX_QUEUE_NUM; i++) {
                error = bus_dma_tag_create(
                                sc->re_parent_tag,
                                RE_DESC_ALIGN, 0,		/* alignment, boundary */
                                BUS_SPACE_MAXADDR,		/* lowaddr */
                                BUS_SPACE_MAXADDR,		/* highaddr */
                                NULL, NULL,			/* filter, filterarg */
                                tx_list_size,				/* maxsize */
                                1,				/* nsegments */
                                tx_list_size,				/* maxsegsize */
                                0,				/* flags */
                                NULL, NULL,			/* lockfunc, lockarg */
                                &sc->re_desc.tx_desc_tag[i]);
                if (error) {
                        device_printf(dev,"bus_dma_tag_create fail\n");
                        goto fail;
                }

                error = bus_dmamem_alloc(sc->re_desc.tx_desc_tag[i],
                                         (void**) &sc->re_desc.tx_desc[i],
                                         BUS_DMA_WAITOK|BUS_DMA_COHERENT|BUS_DMA_ZERO,
                                         &sc->re_desc.tx_desc_dmamap[i]);

                if (error) {
                        device_printf(dev,"bus_dmamem_alloc fail\n");
                        goto fail;
                }
        }

        sc->re_tx_cstag =1;
        sc->re_rx_cstag =1;

        error = re_alloc_stats(dev, sc);
        if (error)
                goto fail;
        re_add_sysctls(sc);

#if OS_VER < VERSION(6,0)
        ifp = &sc->arpcom.ac_if;
#else
        ifp = sc->re_ifp = if_alloc(IFT_ETHER);
        if (ifp == NULL) {
                device_printf(dev, "can not if_alloc()\n");
                error = ENOSPC;
                goto fail;
        }
#endif
        ifp->if_softc = sc;
#if OS_VER < VERSION(5,3)
        ifp->if_unit = unit;
        ifp->if_name = "re";
#else
        if_initname(ifp, device_get_name(dev), device_get_unit(dev));
#endif
        ifp->if_mtu = ETHERMTU;
        ifp->if_flags = IFF_BROADCAST | IFF_SIMPLEX | IFF_MULTICAST;
        ifp->if_ioctl = re_ioctl;
        ifp->if_output = ether_output;
        ifp->if_start = re_start;
#if OS_VER < VERSION(7,0)
        ifp->if_watchdog = re_watchdog;
#endif
        if ((sc->re_type == MACFG_24) || (sc->re_type == MACFG_25) || (sc->re_type == MACFG_26))
                ifp->if_hwassist |= CSUM_TCP | CSUM_UDP;
        else
                ifp->if_hwassist |= RE_CSUM_FEATURES;
        ifp->if_capabilities = IFCAP_HWCSUM | IFCAP_HWCSUM_IPV6;
        /* TSO capability setup */
        if (sc->re_if_flags & RL_FLAG_8168G_PLUS) {
                ifp->if_hwassist |= CSUM_TSO;
                ifp->if_capabilities |= IFCAP_TSO;
        }
        /* RTL8169/RTL8101E/RTL8168B not support TSO v6 */
        if (!(sc->re_if_flags & RL_FLAG_DESCV2)) {
                ifp->if_hwassist &= ~(CSUM_IP6_TSO |
                                      CSUM_TCP_IPV6 |
                                      CSUM_UDP_IPV6);
                ifp->if_capabilities &= ~(IFCAP_TSO6 | IFCAP_HWCSUM_IPV6);
        }
        ifp->if_init = re_init;
        /* VLAN capability setup */
        ifp->if_capabilities |= IFCAP_VLAN_MTU | IFCAP_VLAN_HWTAGGING;
        /* LRO capability setup */
        ifp->if_capabilities |= IFCAP_LRO;

        /* Enable WOL if PM is supported. */
        if (pci_find_cap(sc->dev, PCIY_PMG, &reg) == 0)
                ifp->if_capabilities |= IFCAP_WOL;
        ifp->if_capenable = ifp->if_capabilities;
        ifp->if_capenable &= ~(IFCAP_WOL_UCAST | IFCAP_WOL_MCAST);
        /*
         * Default disable ipv6 tso.
         */
        ifp->if_hwassist &= ~CSUM_IP6_TSO;
        ifp->if_capenable &= ~IFCAP_TSO6;

        /* Not enable LRO for OS version lower than 11.0 */
#if OS_VER < VERSION(11,0)
        ifp->if_capenable &= ~IFCAP_LRO;
#endif
        /* Get station address. */
        RE_LOCK(sc);
        re_get_hw_mac_address(sc, eaddr);
        RE_UNLOCK(sc);

        /*
         * A RealTek chip was detected. Inform the world.
         */
        device_printf(dev,"version:%s\n", RE_VERSION);
        device_printf(dev,"Ethernet address: %6D\n", eaddr, ":");
        if (HW_DASH_SUPPORT_DASH(sc)) {
                device_printf(dev,"DASH status: %s\n", sc->re_dash?"enabled":"disabled");
                if (sc->re_dash)
                        device_printf(dev,"DASH FW: 0x%08x\n", sc->re_dash_fw_ver);
        }
        printf("\nThis product is covered by one or more of the following patents: \
           \nUS6,570,884, US6,115,776, and US6,327,625.\n");

#if OS_VER < VERSION(6,0)
        bcopy(eaddr, (char *)&sc->arpcom.ac_enaddr, ETHER_ADDR_LEN);
#endif
        bcopy(eaddr, (char *)&sc->org_mac_addr, ETHER_ADDR_LEN);

        RE_LOCK(sc);
        re_phy_power_up(dev);
        re_hw_phy_config(sc);
        re_hw_ephy_config(sc);
        re_clrwol(sc);

        set_rxbufsize(sc);
        error =re_alloc_buf(sc);

        if (error) {
                RE_UNLOCK(sc);
                goto fail;
        }

        /* Init descriptors. */
        re_var_init(sc);

        RE_UNLOCK(sc);

        switch(sc->re_device_id) {
        case RT_DEVICEID_8127:
                ifp->if_baudrate = 100000000000;
                break;
        case RT_DEVICEID_8126:
                ifp->if_baudrate = 50000000000;
                break;
        case RT_DEVICEID_8125:
        case RT_DEVICEID_3000:
                ifp->if_baudrate = 25000000000;
                break;
        case RT_DEVICEID_8169:
        case RT_DEVICEID_8169SC:
        case RT_DEVICEID_8168:
        case RT_DEVICEID_8161:
        case RT_DEVICEID_8162:
                ifp->if_baudrate = 1000000000;
                break;
        default:
                ifp->if_baudrate = 100000000;
                break;
        }
        IFQ_SET_MAXLEN(&ifp->if_snd, IFQ_MAXLEN);
        ifp->if_snd.ifq_drv_maxlen = IFQ_MAXLEN;
        IFQ_SET_READY(&ifp->if_snd);

        switch (sc->re_type) {
        case MACFG_80:
        case MACFG_81:
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                sc->ifmedia_upd = re_ifmedia_upd_8125;
                sc->ifmedia_sts = re_ifmedia_sts_8125;
                sc->intr = re_intr_8125;
                sc->int_task = re_int_task_8125;
                sc->int_task_poll = re_int_task_8125_poll;
                sc->hw_start_unlock = re_hw_start_unlock_8125;
                break;
        default:
                sc->ifmedia_upd = re_ifmedia_upd;
                sc->ifmedia_sts = re_ifmedia_sts;
                sc->intr = re_intr;
                sc->int_task = re_int_task;
                sc->int_task_poll = re_int_task_poll;
                sc->hw_start_unlock = re_hw_start_unlock;
                break;
        }

        error = re_config_soft_lro(sc);

        if (error)
                goto fail;

        /*
        * Specify the media types supported by this adapter and register
        * callbacks to update media and link information
        */
        ifmedia_init(&sc->media, IFM_IMASK, sc->ifmedia_upd, sc->ifmedia_sts);
        ifmedia_add(&sc->media, IFM_ETHER | IFM_10_T, 0, NULL);
        ifmedia_add(&sc->media, IFM_ETHER | IFM_10_T | IFM_FDX, 0, NULL);
        ifmedia_add(&sc->media, IFM_ETHER | IFM_100_TX, 0, NULL);
        ifmedia_add(&sc->media, IFM_ETHER | IFM_100_TX | IFM_FDX, 0, NULL);
        switch(sc->re_device_id) {
        case RT_DEVICEID_8125:
        case RT_DEVICEID_3000:
        case RT_DEVICEID_8126:
        case RT_DEVICEID_8127:
        case RT_DEVICEID_8169:
        case RT_DEVICEID_8169SC:
        case RT_DEVICEID_8168:
        case RT_DEVICEID_8161:
        case RT_DEVICEID_8162:
                ifmedia_add(&sc->media, IFM_ETHER | IFM_1000_T | IFM_FDX, 0, NULL);
                break;
        default:
                break;
        }
        switch(sc->re_device_id) {
        case RT_DEVICEID_8125:
        case RT_DEVICEID_3000:
        case RT_DEVICEID_8126:
        case RT_DEVICEID_8127:
                ifmedia_add(&sc->media, IFM_ETHER | IFM_2500_T | IFM_FDX, 0, NULL);
                break;
        default:
                break;
        }
        switch(sc->re_device_id) {
        case RT_DEVICEID_8126:
        case RT_DEVICEID_8127:
                ifmedia_add(&sc->media, IFM_ETHER | IFM_5000_T | IFM_FDX, 0, NULL);
                break;
        default:
                break;
        }
        switch(sc->re_device_id) {
        case RT_DEVICEID_8127:
                ifmedia_add(&sc->media, IFM_ETHER | IFM_10G_T | IFM_FDX, 0, NULL);
                break;
        default:
                break;
        }
        ifmedia_add(&sc->media, IFM_ETHER | IFM_AUTO, 0, NULL);
        ifmedia_set(&sc->media, IFM_ETHER | IFM_AUTO);
        sc->media.ifm_media = IFM_ETHER | IFM_AUTO;

#if OS_VER >= VERSION(13,0)
        NET_TASK_INIT(&sc->re_inttask, 0, sc->int_task, sc);
        NET_TASK_INIT(&sc->re_inttask_poll, 0, sc->int_task_poll, sc);
#elif OS_VER>=VERSION(7,0)
        TASK_INIT(&sc->re_inttask, 0, sc->int_task, sc);
        TASK_INIT(&sc->re_inttask_poll, 0, sc->int_task_poll, sc);
#endif

#if OS_VER>=VERSION(7,0)
        sc->re_tq = taskqueue_create_fast("re_taskq", M_WAITOK,
                                          taskqueue_thread_enqueue, &sc->re_tq);
        if (sc->re_tq == NULL) {
                error = ENOMEM;
                goto fail_intr;
        }
        error = taskqueue_start_threads(&sc->re_tq, 1, PI_NET, "%s taskq",
                                        device_get_nameunit(dev));
        if (error) goto fail_intr;
#endif

#if OS_VER < VERSION(7,0)
        error = bus_setup_intr(dev, sc->re_irq, INTR_TYPE_NET,
                               sc->intr, sc, &sc->re_intrhand);
#else
        error = bus_setup_intr(dev, sc->re_irq, INTR_TYPE_NET|INTR_MPSAFE,
                               sc->intr, NULL, sc, &sc->re_intrhand);
#endif

        if (error) goto fail_intr;

        RE_LOCK(sc);
        re_setup_msix_tbl(sc);
        sc->ifmedia_upd(ifp);
        RE_UNLOCK(sc);

        /*
         * Call MI attach routine.
         */
        /*#if OS_VER < VERSION(5, 1)*/
#if OS_VER < VERSION(4,9)
        ether_ifattach(ifp, ETHER_BPF_SUPPORTED);
#else
        ether_ifattach(ifp, eaddr);
#endif

fail_intr:
        if (error) {
                device_printf(dev, "couldn't set up interrupt handler\n");
#if OS_VER < VERSION(4,9)
                ether_ifdetach(ifp, ETHER_BPF_SUPPORTED);
#else
                ether_ifdetach(ifp);
#endif
        }

fail:
        if (error)
                re_detach(dev);

        return(error);
}

static int re_detach(device_t dev)
{
        struct re_softc		*sc;
        struct ifnet		*ifp;
        /*int			s;*/
        int			i;
        int			rid;

        /*s = splimp();*/

        sc = device_get_softc(dev);

        ifp = RE_GET_IFNET(sc);

        re_free_soft_lro(sc);

        /* These should only be active if attach succeeded */
        if (device_is_attached(dev)) {
                RE_LOCK(sc);
                re_stop(sc);
                RE_UNLOCK(sc);
#if OS_VER < VERSION(4,9)
                ether_ifdetach(ifp, ETHER_BPF_SUPPORTED);
#else
                ether_ifdetach(ifp);
#endif
        }

        if (HW_DASH_SUPPORT_DASH(sc) && sc->re_res) {
                RE_LOCK(sc);
                re_driver_stop(sc);
                RE_UNLOCK(sc);
        }

#if OS_VER>=VERSION(7,0)
        if (sc->re_tq) {
                taskqueue_drain(sc->re_tq, &sc->re_inttask);
                taskqueue_drain(sc->re_tq, &sc->re_inttask_poll);
                taskqueue_free(sc->re_tq);
        }
#endif

        bus_generic_detach(dev);

        sc->driver_detach = 1;

        if (sc->re_intrhand)
                bus_teardown_intr(dev, sc->re_irq, sc->re_intrhand);

#if OS_VER>=VERSION(6,0)
        if (ifp)
                if_free(ifp);
#endif

        if ((sc->re_if_flags & (RL_FLAG_MSI | RL_FLAG_MSIX)) == 0)
                rid = 0;
        else
                rid = 1;
        if (sc->re_irq) {
                bus_release_resource(dev, SYS_RES_IRQ, rid, sc->re_irq);
                sc->re_irq = NULL;
        }
        if ((sc->re_if_flags & (RL_FLAG_MSI | RL_FLAG_MSIX)) != 0)
                pci_release_msi(dev);
        if (sc->re_res_pba) {
                rid = PCIR_BAR(4);
                bus_release_resource(dev, SYS_RES_MEMORY, rid, sc->re_res_pba);
        }
        if (sc->re_res)
                bus_release_resource(dev, sc->re_res_type, sc->re_res_id, sc->re_res);

        if (HW_DASH_SUPPORT_TYPE_3(sc) && sc->re_dash)
                bus_space_unmap(sc->re_cmac_tag, sc->re_mapped_cmac_handle, RE_REGS_SIZE);

        for (int j = 0; j < RL_RX_QUEUE_NUM; j++) {
                if (sc->re_desc.re_rx_mtag[j]) {
                        for (i = 0; i < RE_RX_BUF_NUM; i++) {
                                if (sc->re_desc.rx_buf[j][i]!=NULL) {
                                        bus_dmamap_sync(sc->re_desc.re_rx_mtag[j],
                                                        sc->re_desc.re_rx_dmamap[j][i],
                                                        BUS_DMASYNC_POSTREAD);
                                        bus_dmamap_unload(sc->re_desc.re_rx_mtag[j],
                                                          sc->re_desc.re_rx_dmamap[j][i]);
                                        bus_dmamap_destroy(sc->re_desc.re_rx_mtag[j],
                                                           sc->re_desc.re_rx_dmamap[j][i]);
                                        m_freem(sc->re_desc.rx_buf[j][i]);
                                        sc->re_desc.rx_buf[j][i] =NULL;
                                }
                        }
                        bus_dma_tag_destroy(sc->re_desc.re_rx_mtag[j]);
                        sc->re_desc.re_rx_mtag[j] =0;
                }
        }

        for (int j = 0; j < RL_TX_QUEUE_NUM; j++) {
                if (sc->re_desc.re_tx_mtag[j]) {
                        for (int i = 0; i < RE_TX_BUF_NUM; i++) {
                                bus_dmamap_destroy(sc->re_desc.re_tx_mtag[j],
                                                   sc->re_desc.re_tx_dmamap[j][i]);
                        }
                        bus_dma_tag_destroy(sc->re_desc.re_tx_mtag[j]);
                        sc->re_desc.re_tx_mtag[j] =0;
                }
        }

        for (int i = 0; i < RL_RX_QUEUE_NUM; i++) {
                if (sc->re_desc.rx_desc_tag[i]) {
                        bus_dmamap_sync(sc->re_desc.rx_desc_tag[i],
                                        sc->re_desc.rx_desc_dmamap[i],
                                        BUS_DMASYNC_POSTREAD|BUS_DMASYNC_POSTWRITE);
                        bus_dmamap_unload(sc->re_desc.rx_desc_tag[i],
                                          sc->re_desc.rx_desc_dmamap[i]);
                        bus_dmamem_free(sc->re_desc.rx_desc_tag[i],
                                        sc->re_desc.rx_desc[i],
                                        sc->re_desc.rx_desc_dmamap[i]);
                        bus_dma_tag_destroy(sc->re_desc.rx_desc_tag[i]);
                }
        }

        for (int i=0; i<RL_TX_QUEUE_NUM; i++) {
                if (sc->re_desc.tx_desc_tag[i]) {
                        bus_dmamap_sync(sc->re_desc.tx_desc_tag[i],
                                        sc->re_desc.tx_desc_dmamap[i],
                                        BUS_DMASYNC_POSTREAD|BUS_DMASYNC_POSTWRITE);
                        bus_dmamap_unload(sc->re_desc.tx_desc_tag[i],
                                          sc->re_desc.tx_desc_dmamap[i]);
                        bus_dmamem_free(sc->re_desc.tx_desc_tag[i],
                                        sc->re_desc.tx_desc[i],
                                        sc->re_desc.tx_desc_dmamap[i]);
                        bus_dma_tag_destroy(sc->re_desc.tx_desc_tag[i]);
                }
        }

        /* Unload and free the stats buffer and map */

        if (sc->re_tally.re_stag) {
                if (sc->re_tally.re_stats_addr)
                        bus_dmamap_unload(sc->re_tally.re_stag,
                                          sc->re_tally.re_smap);
                if (sc->re_tally.re_stats)
                        bus_dmamem_free(sc->re_tally.re_stag,
                                        sc->re_tally.re_stats, sc->re_tally.re_smap);
                bus_dma_tag_destroy(sc->re_tally.re_stag);
        }

        if (sc->re_parent_tag) {
                bus_dma_tag_destroy(sc->re_parent_tag);
        }

        /*splx(s);*/
        RE_LOCK_DESTROY(sc);

        return(0);
}

static void
re_link_state_change(struct ifnet *ifp, int link_state)
{
#if OS_VER>=VERSION(6,0)
        if_link_state_change(ifp, link_state);
#else
        ifp->if_link_state = link_state
#endif
}

/*
  * Device suspend routine.  Stop the interface and save some PCI
  * settings in case the BIOS doesn't restore them properly on
  * resume.
  */
static int
re_suspend(device_t dev)
{
        struct re_softc         *sc;
        struct ifnet            *ifp;

        sc = device_get_softc(dev);
        RE_LOCK(sc);
        ifp = RE_GET_IFNET(sc);
        sc->re_link_chg_det = 0;
        sc->phy_reg_anlpar = re_get_phy_lp_ability(sc);
        re_stop(sc);
        re_hw_d3_para(sc);
        re_setwol(sc);
        if (HW_DASH_SUPPORT_DASH(sc))
                re_driver_stop(sc);
        sc->suspended = 1;
        sc->link_state = LINK_STATE_UNKNOWN;
        re_link_state_change(ifp, sc->link_state);
        pci_set_powerstate(dev, PCI_POWERSTATE_D3);
        sc->prohibit_access_reg = 1;
        RE_UNLOCK(sc);

        return (0);
}

/*
 * Device resume routine.  Restore some PCI settings in case the BIOS
 * doesn't, re-enable busmastering, and restart the interface if
 * appropriate.
 */
static int
re_resume(device_t dev)
{
        struct re_softc         *sc;
        struct ifnet            *ifp;

        sc = device_get_softc(dev);

        RE_LOCK(sc);

        ifp = RE_GET_IFNET(sc);

        sc->prohibit_access_reg = 0;

        re_setup_msix_tbl(sc);

        re_exit_oob(sc);

        re_hw_init(sc);

        re_reset(sc);

        re_phy_power_up(dev);

        re_hw_phy_config(sc);

        /*
         * Clear WOL matching such that normal Rx filtering
         * wouldn't interfere with WOL patterns.
         */
        re_clrwol(sc);

        RE_UNLOCK(sc);

        RE_LOCK(sc);
        sc->ifmedia_upd(ifp);
        sc->suspended = 0;
        if (ifp->if_flags & IFF_UP) {
                sc->re_link_chg_det = 1;
                re_start_timer(sc);
        }
        RE_UNLOCK(sc);

        return (0);
}

static void
re_clear_set_ephy_bit(
        struct re_softc *sc,
        u_int16_t   addr,
        u_int16_t   clearmask,
        u_int16_t   setmask
)
{
        u_int16_t EphyValue;

        EphyValue = re_ephy_read(sc, addr);
        EphyValue &= ~clearmask;
        EphyValue |= setmask;
        re_ephy_write(sc, addr, EphyValue);
}

static void
re_clear_ephy_bit(
        struct re_softc *sc,
        u_int16_t   addr,
        u_int16_t   mask
)
{
        re_clear_set_ephy_bit(sc,
                              addr,
                              mask,
                              0
                             );
}

static void
re_set_ephy_bit(
        struct re_softc *sc,
        u_int16_t   addr,
        u_int16_t   mask
)
{
        re_clear_set_ephy_bit(sc,
                              addr,
                              0,
                              mask
                             );
}

static void
_re_set_l1_l0s_entry_latency(struct re_softc *sc, u_int8_t setting)
{
        //Set PCI configuration space offset 0x79 to setting
        u_int32_t data32;

        setting &= 0x3f;

        data32 = re_csi_read(sc, 0x870c);
        data32 &= 0xc0ffffff;
        data32 |= (setting << 24);
        re_csi_write(sc, 0x870c, data32);
}

static void
re_set_l1_l0s_entry_latency(struct re_softc *sc, u_int32_t macver)
{
        switch(macver) {
        case 0x34800000:
        case 0x24800000:
        case 0x24000000:
                _re_set_l1_l0s_entry_latency(sc, 0x3f);
                break;
        case 0x3C000000:
        case 0x3C800000:
        case 0x28000000:
        case 0x28800000:
        case 0x2C000000:
        case 0x2C800000:
        case 0x48000000:
        case 0x48800000:
        case 0x4C000000:
        case 0x50800000:
        case 0x5C800000:
        case 0x54000000:
        case 0x6C000000:
        case 0x50000000:
        case 0x54800000:
        case 0x60800000:
        case 0x64000000:
        case 0x64800000:
        case 0x68000000:
        case 0x68800000:
        case 0x70800000:
        case 0x6C800000:
                _re_set_l1_l0s_entry_latency(sc, 0x27);
                break;
        default:
                break;
        }
}

static void
_re_set_mrrs(struct re_softc *sc, u_int8_t setting)
{
        u_int8_t data8;

        setting &= 0x70;
        data8 = pci_read_config(sc->dev, 0x79, 1);
        data8 &= ~0x70;
        data8 |= setting;
        pci_write_config(sc->dev, 0x79, data8, 1);
}

static void
re_set_mrrs(struct re_softc *sc, u_int32_t macver)
{
        struct ifnet		*ifp;

        if (config_soc_lan)
                return;

        ifp = RE_GET_IFNET(sc);

        switch(macver) {
        case 0x3C000000:
        case 0x3C800000:
        case 0x28000000:
        case 0x28800000:
        case 0x2C000000:
                if (ifp->if_mtu > ETHERMTU)
                        _re_set_mrrs(sc, 0x20);
                else
                        _re_set_mrrs(sc, 0x40);
                break;
        case 0x34800000:
        case 0x24800000:
        case 0x2C800000:
        case 0x24000000:
        case 0x48000000:
        case 0x48800000:
        case 0x4C000000:
        case 0x50800000:
        case 0x5C800000:
        case 0x54000000:
        case 0x6C000000:
        case 0x50000000:
        case 0x54800000:
        case 0x60800000:
        case 0x64000000:
        case 0x64800000:
        case 0x68000000:
        case 0x68800000:
        case 0x70800000:
        case 0x6C800000:
                _re_set_mrrs(sc, 0x40);
                break;
        default:
                break;
        }
}

static void
re_disable_l1_timeout(struct re_softc *sc)
{
        re_csi_write(sc, 0x890, re_csi_read(sc, 0x890) & ~BIT_0);
}

/*
 * Stop all chip I/O so that the kernel's probe routines don't
 * get confused by errant DMAs when rebooting.
 */
static int re_shutdown(device_t dev)	/* The same with re_stop(sc) */
{
        struct re_softc		*sc;

        sc = device_get_softc(dev);

        RE_LOCK(sc);
        sc->re_link_chg_det = 0;
        sc->phy_reg_anlpar = re_get_phy_lp_ability(sc);
        re_stop(sc);
        RE_UNLOCK(sc);

        RE_LOCK(sc);
        re_hw_d3_para(sc);
        if (s5wol == 0) {
                re_phy_power_down(dev);
        } else {
                struct ifnet            *ifp;
                ifp = RE_GET_IFNET(sc);
                ifp->if_capenable = IFCAP_WOL_MAGIC;
                re_setwol(sc);
        }

        if (HW_DASH_SUPPORT_DASH(sc))
                re_driver_stop(sc);
        RE_UNLOCK(sc);

        return 0;
}

static void re_set_eee_lpi_timer(struct re_softc *sc)
{
        struct ifnet		*ifp;

        ifp = RE_GET_IFNET(sc);

        switch (sc->re_type) {
        case MACFG_68:
        case MACFG_69:
        case MACFG_74:
        case MACFG_75:
        case MACFG_76:
                re_mac_ocp_write(sc, RE_EEE_TXIDLE_TIMER_8168, ifp->if_mtu + ETHER_HDR_LEN + 0x20);
                break;
        case MACFG_80:
        case MACFG_81:
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                CSR_WRITE_2(sc, RE_EEE_TXIDLE_TIMER_8125, ifp->if_mtu + ETHER_HDR_LEN + 0x20);
                break;
        default:
                break;
        }
}

static void re_set_pfm_patch(struct re_softc *sc, bool enable)
{
        if (!sc->RequiredPfmPatch)
                goto exit;

        if (enable) {
                re_set_mac_ocp_bit(sc, 0xD3F0, BIT_0);
                re_set_mac_ocp_bit(sc, 0xD3F2, BIT_0);
                re_set_mac_ocp_bit(sc, 0xE85A, BIT_6);
        } else {
                re_clear_mac_ocp_bit(sc, 0xD3F0, BIT_0);
                re_clear_mac_ocp_bit(sc, 0xD3F2, BIT_0);
                re_clear_mac_ocp_bit(sc, 0xE85A, BIT_6);
        }

exit:
        return;
}

static void re_clear_ephy_ext_addr(struct re_softc *sc)
{
        switch (sc->re_type) {
        case MACFG_74:
        case MACFG_75:
                re_clear_mac_ocp_bit(sc, 0xDE28, (BIT_1 | BIT_0));
                break;
        case MACFG_100:
        case MACFG_101:
                _re_ephy_write(sc, RE_EPHY_EXT_ADDR, 0x0000);
                break;
        default:
                break;
        }
}

static void re_hw_start_unlock(struct re_softc *sc)
{
        struct ifnet		*ifp;
        u_int32_t		macver;
        u_int8_t		data8;
        u_int16_t		data16 = 0;
        u_int32_t		Data32;

        ifp = RE_GET_IFNET(sc);

        /* Init descriptors. */
        re_var_init(sc);

        re_enable_cfg9346_write(sc);

        switch(sc->re_type) {
        case MACFG_36:
        case MACFG_37:
        case MACFG_38:
        case MACFG_39:
        case MACFG_42:
        case MACFG_43:
        case MACFG_50:
        case MACFG_51:
        case MACFG_52:
        case MACFG_53:
        case MACFG_54:
        case MACFG_55:
        case MACFG_56:
        case MACFG_57:
        case MACFG_58:
        case MACFG_59:
        case MACFG_60:
        case MACFG_61:
        case MACFG_62:
        case MACFG_67:
        case MACFG_68:
        case MACFG_69:
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
        case MACFG_74:
        case MACFG_75:
        case MACFG_76:
                _re_enable_aspm_clkreq_lock(sc, 0);
                re_enable_force_clkreq(sc, 0);
                break;
        }

        /*disable Link Down Power Saving(non-LDPS)*/
        /*CSR_WRITE_1(sc, RE_LDPS, 0x05);*/
        /*ldps= CSR_READ_1(sc, RE_LDPS);*/

        re_set_eee_lpi_timer(sc);

        CSR_WRITE_2(sc, RE_CPlusCmd, 0x2060);
        if (interrupt_mitigation)
                CSR_WRITE_2(sc, RE_IntrMitigate, 0x5f51);
        else
                CSR_WRITE_2(sc, RE_IntrMitigate, 0x0000);

        CSR_WRITE_1(sc, RE_MTPS, 0x3f);

        if (sc->re_device_id == RT_DEVICEID_8169 || sc->re_device_id == RT_DEVICEID_8169SC) {
                //do nothing
        } else {
                /* Set the initial TX configuration.*/
                CSR_WRITE_4(sc, RE_TXCFG, RE_TXCFG_CONFIG);
        }

        macver = CSR_READ_4(sc, RE_TXCFG) & 0xFC800000;

        re_set_l1_l0s_entry_latency(sc, macver);

        re_set_mrrs(sc, macver);

        if (macver == 0x00800000 || macver == 0x04000000 || macver == 0x10000000) {
                CSR_WRITE_2(sc, RE_CPlusCmd, 0x0063| ((sc->re_type == MACFG_3 && sc->re_8169_MacVersion==1) ? 0x4008:0));
        } else if (macver == 0x18000000 || macver == 0x98000000) {
                CSR_WRITE_2(sc, RE_CPlusCmd, 0x0068);
                CSR_WRITE_2(sc, RE_IntrMitigate, 0x0000);
        } else if (macver == 0x30000000) {
                CSR_WRITE_2 (sc, RE_CPlusCmd, 0x2060);
                CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) & ~BIT_0);

                if (ifp->if_mtu > ETHERMTU) {
                        data8 = pci_read_config(sc->dev, 0x69, 1);
                        data8 &= ~0x70;
                        data8 |= 0x28;
                        pci_write_config(sc->dev, 0x69, data8, 1);
                } else {
                        data8 = pci_read_config(sc->dev, 0x69, 1);
                        data8 &= ~0x70;
                        data8 |= 0x58;
                        pci_write_config(sc->dev, 0x69, data8, 1);
                }
        } else if (macver == 0x38000000) {
                CSR_WRITE_2 (sc, RE_CPlusCmd, 0x2060);
                CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) & ~BIT_0);

                if (ifp->if_mtu > ETHERMTU) {
                        data8 = pci_read_config(sc->dev, 0x69, 1);
                        data8 &= ~0x70;
                        data8 |= 0x28;
                        pci_write_config(sc->dev, 0x69, data8, 1);
                        CSR_WRITE_1(sc, RE_CFG4, CSR_READ_1(sc, RE_CFG4) | BIT_0);
                } else {
                        data8 = pci_read_config(sc->dev, 0x69, 1);
                        data8 &= ~0x70;
                        data8 |= 0x58;
                        pci_write_config(sc->dev, 0x69, data8, 1);
                        CSR_WRITE_1(sc, RE_CFG4, CSR_READ_1(sc, RE_CFG4) & ~ BIT_0);
                }
        } else if (macver == 0x34000000 || macver == 0xB4000000) {
                CSR_WRITE_2 (sc, RE_CPlusCmd, 0x2060);
        } else if (macver == 0x34800000 || macver == 0x24800000) {
                if (pci_read_config(sc->dev, 0x81, 1) == 1) {
                        CSR_WRITE_1(sc, RE_DBG_reg, 0x98);
                        CSR_WRITE_1(sc, RE_CFG2, CSR_READ_1(sc, RE_CFG2) | 0x80);
                        CSR_WRITE_1(sc, RE_CFG4, CSR_READ_1(sc, RE_CFG4) | 0x04);
                        pci_write_config(sc->dev, 0x81, 1, 1);
                }
                CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) & ~BIT_0);

                CSR_WRITE_2 (sc, RE_CPlusCmd, 0x2060);

                if (sc->re_type == MACFG_14) {
                        CSR_WRITE_1(sc,RE_CFG1, 0x0f);
                } else if (sc->re_type == MACFG_15) {
                        CSR_WRITE_1(sc,RE_CFG1, 0x0f);
                } else if (sc->re_type == MACFG_18) {
                        CSR_WRITE_1(sc, 0xF5, CSR_READ_1(sc, 0xF5)|0x04);
                } else if (sc->re_type == MACFG_19) {
                        CSR_WRITE_1(sc, 0xF4, CSR_READ_1(sc, 0xF4)|0x08);
                        CSR_WRITE_1(sc, 0xF5, CSR_READ_1(sc, 0xF5)|0x04);
                }
        } else if (macver == 0x3C000000) {
                //disable clock request.
                pci_write_config(sc->dev, 0x81, 0, 1);

                CSR_WRITE_1(sc, RE_CFG1, CSR_READ_1(sc, RE_CFG1)|0x10);
                CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) & ~BIT_0);

                CSR_WRITE_2 (sc, RE_CPlusCmd, 0x2060);
                if (sc->re_type == MACFG_24) {
                        /*set mac register offset 0xd1 to 0xf8*/
                        CSR_WRITE_1(sc, RE_DBG_reg, 0xF8);

                        if (ifp->if_mtu > ETHERMTU) {
                                CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) | BIT_2); //Jumbo_en0
                                CSR_WRITE_1(sc, RE_CFG4, CSR_READ_1(sc, RE_CFG4) | (1 << 1)); //Jumbo_en1

                                ifp->if_capenable &= ~IFCAP_HWCSUM;
                                CSR_WRITE_2 (sc, RE_CPlusCmd,CSR_READ_2(sc, RE_CPlusCmd) & ~RL_RxChkSum);
                                ifp->if_hwassist &= ~RE_CSUM_FEATURES;
                        } else {
                                CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) & ~BIT_2); //Jumbo_en0
                                CSR_WRITE_1(sc, RE_CFG4, CSR_READ_1(sc, RE_CFG4) & ~(1 << 1)); //Jumbo_en1

                                if (sc->re_tx_cstag) {
                                        ifp->if_capenable |= IFCAP_TXCSUM;
                                        if ((sc->re_type == MACFG_24) || (sc->re_type == MACFG_25) || (sc->re_type == MACFG_26))
                                                ifp->if_hwassist |= CSUM_TCP | CSUM_UDP;
                                        else
                                                ifp->if_hwassist |= RE_CSUM_FEATURES;
                                }
                                if (sc->re_rx_cstag) {
                                        ifp->if_capenable |= IFCAP_RXCSUM;
                                        CSR_WRITE_2 (sc, RE_CPlusCmd,CSR_READ_2(sc, RE_CPlusCmd) |RL_RxChkSum);
                                }
                        }
                } else if (sc->re_type == MACFG_25) {
                        if (ifp->if_mtu > ETHERMTU) {
                                CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) | BIT_2); //Jumbo_en0
                                CSR_WRITE_1(sc, RE_CFG4, CSR_READ_1(sc, RE_CFG4) | (1<<1)); //Jumbo_en1

                                ifp->if_capenable &= ~IFCAP_HWCSUM;
                                CSR_WRITE_2 (sc, RE_CPlusCmd,CSR_READ_2(sc, RE_CPlusCmd) & ~RL_RxChkSum);
                                ifp->if_hwassist &= ~RE_CSUM_FEATURES;
                        } else {
                                CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) & ~BIT_2); //Jumbo_en0
                                CSR_WRITE_1(sc, RE_CFG4, CSR_READ_1(sc, RE_CFG4) & ~(1<<1)); //Jumbo_en1

                                if (sc->re_tx_cstag) {
                                        ifp->if_capenable |= IFCAP_TXCSUM;
                                        if ((sc->re_type == MACFG_24) || (sc->re_type == MACFG_25) || (sc->re_type == MACFG_26))
                                                ifp->if_hwassist |= CSUM_TCP | CSUM_UDP;
                                        else
                                                ifp->if_hwassist |= RE_CSUM_FEATURES;
                                }
                                if (sc->re_rx_cstag) {
                                        ifp->if_capenable |= IFCAP_RXCSUM;
                                        CSR_WRITE_2 (sc, RE_CPlusCmd,CSR_READ_2(sc, RE_CPlusCmd) |RL_RxChkSum);
                                }
                        }
                } else if (sc->re_type == MACFG_26) {
                        if (ifp->if_mtu > ETHERMTU) {
                                CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) | BIT_2); //Jumbo_en0
                                CSR_WRITE_1(sc, RE_CFG4, CSR_READ_1(sc, RE_CFG4) | (1<<1)); //Jumbo_en1

                                ifp->if_capenable &= ~IFCAP_HWCSUM;
                                CSR_WRITE_2 (sc, RE_CPlusCmd,CSR_READ_2(sc, RE_CPlusCmd) & ~RL_RxChkSum);
                                ifp->if_hwassist &= ~RE_CSUM_FEATURES;
                        } else {
                                CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) & ~BIT_2); //Jumbo_en0
                                CSR_WRITE_1(sc, RE_CFG4, CSR_READ_1(sc, RE_CFG4) & ~(1<<1)); //Jumbo_en1

                                if (sc->re_tx_cstag) {
                                        ifp->if_capenable |= IFCAP_TXCSUM;
                                        if ((sc->re_type == MACFG_24) || (sc->re_type == MACFG_25) || (sc->re_type == MACFG_26))
                                                ifp->if_hwassist |= CSUM_TCP | CSUM_UDP;
                                        else
                                                ifp->if_hwassist |= RE_CSUM_FEATURES;
                                }
                                if (sc->re_rx_cstag) {
                                        ifp->if_capenable |= IFCAP_RXCSUM;
                                        CSR_WRITE_2 (sc, RE_CPlusCmd,CSR_READ_2(sc, RE_CPlusCmd) |RL_RxChkSum);
                                }
                        }
                }
        } else if (macver == 0x3C800000) {
                //disable clock request.
                pci_write_config(sc->dev, 0x81, 0x00, 1);

                re_eri_write(sc, 0x1EC, 1, 0x07, ERIAR_ASF);

                CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) & ~BIT_0);
                CSR_WRITE_2 (sc, RE_CPlusCmd, 0x2060);
                if (sc->re_type == MACFG_28)
                        CSR_WRITE_1(sc, 0xD1, 0x20);

                if (ifp->if_mtu > ETHERMTU) {
                        CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) | BIT_2); //Jumbo_en0
                        CSR_WRITE_1(sc, RE_CFG4, CSR_READ_1(sc, RE_CFG4) | (1<<1)); //Jumbo_en1

                        ifp->if_capenable &= ~IFCAP_HWCSUM;
                        CSR_WRITE_2 (sc, RE_CPlusCmd,CSR_READ_2(sc, RE_CPlusCmd) & ~RL_RxChkSum);
                        ifp->if_hwassist &= ~RE_CSUM_FEATURES;
                } else {
                        CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) & ~BIT_2); //Jumbo_en0
                        CSR_WRITE_1(sc, RE_CFG4, CSR_READ_1(sc, RE_CFG4) & ~(1<<1)); //Jumbo_en1

                        if (sc->re_tx_cstag) {
                                ifp->if_capenable |= IFCAP_TXCSUM;
                                ifp->if_hwassist |= RE_CSUM_FEATURES;
                        }
                        if (sc->re_rx_cstag) {
                                ifp->if_capenable |= IFCAP_RXCSUM;
                                CSR_WRITE_2 (sc, RE_CPlusCmd,CSR_READ_2(sc, RE_CPlusCmd) |RL_RxChkSum);
                        }
                }
        } else if (macver == 0x28000000) {
                //disable clock request.
                pci_write_config(sc->dev, 0x81, 0x00, 1);

                CSR_WRITE_1(sc, RE_TDFNR, 0x8);

                CSR_WRITE_1(sc, 0xD1, CSR_READ_1(sc, 0xD1) | 0x02);

                CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) & ~BIT_0);
                CSR_WRITE_1(sc, RE_DBG_reg, CSR_READ_1(sc, RE_DBG_reg)|0x82);

                CSR_WRITE_2 (sc, RE_CPlusCmd, 0x2060);

                if (ifp->if_mtu > ETHERMTU) {
                        CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) | BIT_2); //Jumbo_en0
                        CSR_WRITE_1(sc, RE_CFG4, CSR_READ_1(sc, RE_CFG4) | (1<<1)); //Jumbo_en1

                        ifp->if_capenable &= ~IFCAP_HWCSUM;
                        CSR_WRITE_2 (sc, RE_CPlusCmd,CSR_READ_2(sc, RE_CPlusCmd) & ~RL_RxChkSum);
                        ifp->if_hwassist &= ~RE_CSUM_FEATURES;
                } else {
                        CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) & ~BIT_2); //Jumbo_en0
                        CSR_WRITE_1(sc, RE_CFG4, CSR_READ_1(sc, RE_CFG4) & ~(1<<1)); //Jumbo_en1

                        if (sc->re_tx_cstag) {
                                ifp->if_capenable |= IFCAP_TXCSUM;
                                ifp->if_hwassist |= RE_CSUM_FEATURES;
                        }
                        if (sc->re_rx_cstag) {
                                ifp->if_capenable |= IFCAP_RXCSUM;
                                CSR_WRITE_2 (sc, RE_CPlusCmd,CSR_READ_2(sc, RE_CPlusCmd) |RL_RxChkSum);
                        }
                }

                if (sc->re_type == MACFG_31)
                        CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) & ~(1<<4));
                else if (sc->re_type == MACFG_33)
                        CSR_WRITE_1(sc, RE_CFG1, CSR_READ_1(sc, RE_CFG1)|0x10);
        } else if (macver == 0x28800000) {
                /* disable clock request. */
                pci_write_config(sc->dev, 0x81, 0x00, 1);

                CSR_WRITE_1(sc, RE_TDFNR, 0x8);
                if (sc->re_dash &&
                    (sc->re_type == MACFG_63 || sc->re_type == MACFG_64))
                        CSR_WRITE_1(sc, RE_TDFNR, 0x1);

                CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) & ~BIT_0);
                CSR_WRITE_1(sc, RE_DBG_reg, CSR_READ_1(sc, RE_DBG_reg)|0x82);

                CSR_WRITE_2 (sc, RE_CPlusCmd, 0x2060);

                if (ifp->if_mtu > ETHERMTU) {
                        CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) | BIT_2); //Jumbo_en0
                        CSR_WRITE_1(sc, RE_CFG4, CSR_READ_1(sc, RE_CFG4) | (1<<1)); //Jumbo_en1

                        ifp->if_capenable &= ~IFCAP_HWCSUM;
                        CSR_WRITE_2 (sc, RE_CPlusCmd,CSR_READ_2(sc, RE_CPlusCmd) & ~RL_RxChkSum);
                        ifp->if_hwassist &= ~RE_CSUM_FEATURES;
                } else {
                        CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) & ~BIT_2); //Jumbo_en0
                        CSR_WRITE_1(sc, RE_CFG4, CSR_READ_1(sc, RE_CFG4) & ~(1<<1)); //Jumbo_en1

                        if (sc->re_tx_cstag) {
                                ifp->if_capenable |= IFCAP_TXCSUM;
                                ifp->if_hwassist |= RE_CSUM_FEATURES;
                        }
                        if (sc->re_rx_cstag) {
                                ifp->if_capenable |= IFCAP_RXCSUM;
                                CSR_WRITE_2 (sc, RE_CPlusCmd,CSR_READ_2(sc, RE_CPlusCmd) |RL_RxChkSum);
                        }
                }
        } else if (macver == 0x2C000000) {
                /* disable clock request. */
                pci_write_config(sc->dev, 0x81, 0x00, 1);

                CSR_WRITE_1(sc, 0xF3, CSR_READ_1(sc, 0xF3)|0x20);
                CSR_WRITE_1(sc, 0xF3, CSR_READ_1(sc, 0xF3)& ~0x20);

                CSR_WRITE_1(sc, 0xD0, CSR_READ_1(sc, 0xD0)|0xC0);
                CSR_WRITE_1(sc, 0xF1, CSR_READ_1(sc, 0xF1)|0x73);
                CSR_WRITE_1(sc, RE_CFG5, (CSR_READ_1(sc, RE_CFG5)& ~0x08));
                CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) & ~BIT_0);

                CSR_WRITE_1(sc, RE_TDFNR, 0x8);

                if (sc->re_type == MACFG_36 || sc->re_type == MACFG_37) {
                        if (ifp->if_mtu > ETHERMTU) {
                                CSR_WRITE_1 (sc, RE_MTPS, 0x24);
                                CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) | BIT_2);
                                CSR_WRITE_1(sc, RE_CFG4, CSR_READ_1(sc, RE_CFG4) |0x01);

                                ifp->if_capenable &= ~IFCAP_HWCSUM;
                                CSR_WRITE_2 (sc, RE_CPlusCmd,CSR_READ_2(sc, RE_CPlusCmd) & ~RL_RxChkSum);
                                ifp->if_hwassist &= ~RE_CSUM_FEATURES;
                        } else {
                                CSR_WRITE_1 (sc, RE_MTPS, 0x0c);
                                CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) & ~BIT_2);
                                CSR_WRITE_1(sc, RE_CFG4, CSR_READ_1(sc, RE_CFG4) & ~0x01);

                                if (sc->re_tx_cstag) {
                                        ifp->if_capenable |= IFCAP_TXCSUM;
                                        ifp->if_hwassist |= RE_CSUM_FEATURES;
                                }
                                if (sc->re_rx_cstag) {
                                        ifp->if_capenable |= IFCAP_RXCSUM;
                                        CSR_WRITE_2 (sc, RE_CPlusCmd,CSR_READ_2(sc, RE_CPlusCmd) |RL_RxChkSum);
                                }
                        }
                }
        } else if (macver == 0x2C800000) {
                /* disable clock request. */
                pci_write_config(sc->dev, 0x81, 0x00, 1);

                CSR_WRITE_1(sc, RE_TDFNR, 0x8);

                re_eri_write(sc, 0xC0, 2, 0x0000, ERIAR_ExGMAC);
                re_eri_write(sc, 0xB8, 4, 0x00000000, ERIAR_ExGMAC);
                re_eri_write(sc, 0xC8, 4, 0x00100002, ERIAR_ExGMAC);
                re_eri_write(sc, 0xE8, 4, 0x00100006, ERIAR_ExGMAC);

                Data32 = re_eri_read(sc, 0xD4, 4, ERIAR_ExGMAC);
                Data32 |= (BIT_8 | BIT_9 | BIT_10 | BIT_11 | BIT_12);
                re_eri_write(sc, 0xD4, 4, Data32, ERIAR_ExGMAC);
                if (sc ->re_type == MACFG_39) {
                        Data32 = re_eri_read(sc, 0x1B0, 4, ERIAR_ExGMAC);
                        Data32 |= BIT_4;
                        re_eri_write(sc, 0x1B0, 4, Data32, ERIAR_ExGMAC);
                        re_eri_write(sc, 0xCC, 4, 0x00000050, ERIAR_ExGMAC);
                        re_eri_write(sc, 0xD0, 4, 0x07ff0060, ERIAR_ExGMAC);
                }

                CSR_WRITE_4(sc, RE_TXCFG, CSR_READ_4(sc, RE_TXCFG) |BIT_7);
                CSR_WRITE_1(sc, 0xD3, CSR_READ_1(sc, 0xD3) & ~BIT_7);
                CSR_WRITE_1(sc, 0x1B, CSR_READ_1(sc, 0x1B) & ~0x07);

                CSR_WRITE_2 (sc, RE_CPlusCmd, 0x2060);

                if (sc ->re_type == MACFG_38) {
                        CSR_WRITE_4(sc, 0xB0, 0xEE480010);
                        CSR_WRITE_1(sc, 0x1A, CSR_READ_1(sc, 0x1A) & ~(BIT_2 |BIT_3));
                        re_eri_write(sc, 0x1DC, 1, 0x64, ERIAR_ExGMAC);
                }

                CSR_WRITE_1(sc, RE_CFG2, CSR_READ_1(sc, RE_CFG2) | BIT_5);
                CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) & ~BIT_0);

                CSR_WRITE_1(sc, 0xD0, CSR_READ_1(sc, 0xD0) | BIT_6);
                CSR_WRITE_1(sc, 0xF2, CSR_READ_1(sc, 0xF2) | BIT_6);

                if (ifp->if_mtu > ETHERMTU)
                        CSR_WRITE_1 (sc, RE_MTPS, 0x27);
                ifp->if_capenable &= ~IFCAP_HWCSUM;
                ifp->if_hwassist &= ~RE_CSUM_FEATURES;
        } else if (macver == 0x24000000) {
                if (pci_read_config(sc->dev, 0x81, 1)==1) {
                        CSR_WRITE_1(sc, RE_DBG_reg, 0x98);
                        CSR_WRITE_1(sc, RE_CFG2, CSR_READ_1(sc, RE_CFG2) | 0x80);
                        CSR_WRITE_1(sc, RE_CFG4, CSR_READ_1(sc, RE_CFG4) | 0x04);
                        pci_write_config(sc->dev, 0x81, 1, 1);
                }

                CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) & ~BIT_0);

                CSR_WRITE_2 (sc, RE_CPlusCmd, 0x2060);
        } else if (macver == 0x40800000) {
                CSR_WRITE_1(sc, 0xF2, CSR_READ_1(sc, 0xF2) | 0x80);
                CSR_WRITE_1(sc, 0xF1, CSR_READ_1(sc, 0xF1) | 0x28);
                CSR_WRITE_1(sc, 0xD3, CSR_READ_1(sc, 0xD3) | 0x0C);
                CSR_WRITE_1(sc, 0xD3, CSR_READ_1(sc, 0xD3) & ~BIT_7);
                CSR_WRITE_1(sc, 0xD0, CSR_READ_1(sc, 0xD0) | 0x40);
                CSR_WRITE_2(sc, 0xE0, CSR_READ_2(sc, 0xE0) & ~0xDF9C);
                CSR_WRITE_1(sc, 0xD1, CSR_READ_1(sc, 0xD1) | 0x02);

                CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) & ~BIT_0);
        } else if (macver == 0x44000000) {
                CSR_WRITE_2(sc, 0xE0, CSR_READ_2(sc, 0xE0) & ~0xDF9C);

                re_eri_write(sc, 0xC8, 4, 0x00000002, ERIAR_ExGMAC);
                re_eri_write(sc, 0xE8, 4, 0x00000006, ERIAR_ExGMAC);

                Data32 = re_eri_read(sc, 0xD4, 4, ERIAR_ExGMAC);
                Data32 |= BIT_11 | BIT_10;
                re_eri_write(sc, 0xD4, 4, Data32, ERIAR_ExGMAC);

                CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) & ~BIT_0);

                if (ifp->if_mtu > ETHERMTU)
                        CSR_WRITE_1 (sc, RE_MTPS, 0x27);
        } else if (macver == 0x48000000) {
                CSR_WRITE_1(sc, RE_TDFNR, 0x8);

                Data32 = re_eri_read(sc, 0xD4, 4, ERIAR_ExGMAC);
                Data32 |= (BIT_8 | BIT_9 | BIT_10 | BIT_11 | BIT_12);
                re_eri_write(sc, 0xD4, 4, Data32, ERIAR_ExGMAC);
                re_eri_write(sc, 0xC0, 2, 0x0000, ERIAR_ExGMAC);
                re_eri_write(sc, 0xB8, 4, 0x00000000, ERIAR_ExGMAC);
                re_eri_write(sc, 0xC8, 4, 0x00100002, ERIAR_ExGMAC);
                re_eri_write(sc, 0xE8, 4, 0x00100006, ERIAR_ExGMAC);

                Data32 = re_eri_read(sc, 0x1B0, 4, ERIAR_ExGMAC);
                Data32 |= BIT_4;
                re_eri_write(sc, 0x1B0, 4, Data32, ERIAR_ExGMAC);
                re_eri_write(sc, 0xCC, 4, 0x00000050, ERIAR_ExGMAC);
                re_eri_write(sc, 0xD0, 4, 0x00000060, ERIAR_ExGMAC);
                Data32 = re_eri_read(sc, 0x1D0, 4, ERIAR_ExGMAC);
                Data32 |= BIT_4;
                re_eri_write(sc, 0x1D0, 4, Data32, ERIAR_ExGMAC);

                CSR_WRITE_4(sc, RE_TXCFG, CSR_READ_4(sc, RE_TXCFG) | BIT_7);
                CSR_WRITE_1(sc, 0xD3, CSR_READ_1(sc, 0xD3) & ~BIT_7);
                CSR_WRITE_1(sc, 0x1B, CSR_READ_1(sc, 0x1B) & ~0x07);

                CSR_WRITE_2 (sc, RE_CPlusCmd, 0x2060);

                CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) & ~BIT_0);

                CSR_WRITE_1(sc, 0xD0, CSR_READ_1(sc, 0xD0) | BIT_6);
                CSR_WRITE_1(sc, 0xF2, CSR_READ_1(sc, 0xF2) | BIT_6);

                if (ifp->if_mtu > ETHERMTU)
                        CSR_WRITE_1 (sc, RE_MTPS, 0x27);

                if (ifp->if_mtu > ETHERMTU) {
                        ifp->if_capenable &= ~IFCAP_HWCSUM;
                        ifp->if_hwassist &= ~RE_CSUM_FEATURES;
                } else {
                        if (sc->re_tx_cstag) {
                                ifp->if_capenable |= IFCAP_TXCSUM;
                                ifp->if_hwassist |= RE_CSUM_FEATURES;
                        }
                        if (sc->re_rx_cstag) {
                                ifp->if_capenable |= IFCAP_RXCSUM;
                        }
                }
        } else if (macver == 0x48800000) {
                CSR_WRITE_1(sc, RE_TDFNR, 0x8);

                Data32 = re_eri_read(sc, 0xD4, 4, ERIAR_ExGMAC);
                Data32 |= BIT_11 | BIT_10;
                re_eri_write(sc, 0xD4, 4, Data32, ERIAR_ExGMAC);
                re_eri_write(sc, 0xC0, 2, 0x0000, ERIAR_ExGMAC);
                re_eri_write(sc, 0xB8, 4, 0x00000000, ERIAR_ExGMAC);
                re_eri_write(sc, 0xC8, 4, 0x00100002, ERIAR_ExGMAC);
                re_eri_write(sc, 0xE8, 4, 0x00100006, ERIAR_ExGMAC);

                Data32 = re_eri_read(sc, 0x1B0, 4, ERIAR_ExGMAC);
                Data32 |= BIT_4;
                re_eri_write(sc, 0x1B0, 4, Data32, ERIAR_ExGMAC);
                re_eri_write(sc, 0xCC, 4, 0x00000050, ERIAR_ExGMAC);
                re_eri_write(sc, 0xD0, 4, 0x00000060, ERIAR_ExGMAC);
                Data32 = re_eri_read(sc, 0x1D0, 4, ERIAR_ExGMAC);
                Data32 |= BIT_4;
                re_eri_write(sc, 0x1D0, 4, Data32, ERIAR_ExGMAC);

                CSR_WRITE_4(sc, RE_TXCFG, CSR_READ_4(sc, RE_TXCFG) | BIT_7);
                CSR_WRITE_1(sc, 0xD3, CSR_READ_1(sc, 0xD3) & ~BIT_7);
                //CSR_WRITE_1(sc, 0x1B, CSR_READ_1(sc, 0x1B) & ~0x07);

                CSR_WRITE_2 (sc, RE_CPlusCmd, 0x2060);

                CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) & ~BIT_0);

                CSR_WRITE_1(sc, 0xD0, CSR_READ_1(sc, 0xD0) | BIT_6);
                CSR_WRITE_1(sc, 0xF2, CSR_READ_1(sc, 0xF2) | BIT_6);

                if (ifp->if_mtu > ETHERMTU)
                        CSR_WRITE_1 (sc, RE_MTPS, 0x27);

                if (ifp->if_mtu > ETHERMTU) {
                        ifp->if_capenable &= ~IFCAP_HWCSUM;
                        ifp->if_hwassist &= ~RE_CSUM_FEATURES;
                } else {
                        if (sc->re_tx_cstag) {
                                ifp->if_capenable |= IFCAP_TXCSUM;
                                ifp->if_hwassist |= RE_CSUM_FEATURES;
                        }
                        if (sc->re_rx_cstag) {
                                ifp->if_capenable |= IFCAP_RXCSUM;
                        }
                }
        } else if (macver == 0x44800000) {
                CSR_WRITE_1(sc, 0xF2, CSR_READ_1(sc, 0xF2) | 0x80);
                CSR_WRITE_1(sc, 0xF1, CSR_READ_1(sc, 0xF1) | 0x28);
                CSR_WRITE_1(sc, 0xD3, CSR_READ_1(sc, 0xD3) | 0x0C);
                CSR_WRITE_1(sc, 0xD3, CSR_READ_1(sc, 0xD3) & ~BIT_7);
                CSR_WRITE_1(sc, 0xD0, CSR_READ_1(sc, 0xD0) | 0x40);
                CSR_WRITE_2(sc, 0xE0, CSR_READ_2(sc, 0xE0) & ~0xDF9C);

                CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) & ~BIT_0);
        } else if (macver == 0x4C000000 || macver == 0x50800000 ||
                   macver == 0x5C800000 || macver == 0x54000000 ||
                   macver == 0x6C000000) {
                CSR_WRITE_1(sc, RE_CFG2, CSR_READ_1(sc, RE_CFG2) | BIT_5);

                if (sc->re_type == MACFG_59) {
                        re_mac_ocp_write(sc, 0xD3C0, 0x0B00);
                        re_mac_ocp_write(sc, 0xD3C2, 0x0000);
                }

                if (sc->re_type == MACFG_68 || sc->re_type == MACFG_69 ||
                    sc->re_type == MACFG_74 || sc->re_type == MACFG_75 ||
                    sc->re_type == MACFG_76) {
                        re_mac_ocp_write(sc, 0xD400, re_mac_ocp_read(sc, 0xD400) & ~(BIT_0));

                        data16 = re_mac_ocp_read(sc, 0xE63E);
                        data16 &= ~(BIT_3 | BIT_2 | BIT_1);
                        re_mac_ocp_write(sc, 0xE63E, data16);
                        data16 |= (BIT_0);
                        re_mac_ocp_write(sc, 0xE63E, data16);
                        data16 &= ~(BIT_0);
                        re_mac_ocp_write(sc, 0xE63E, data16);
                        re_mac_ocp_write(sc, 0xC094, 0x0);
                        re_mac_ocp_write(sc, 0xC09E, 0x0);

                        re_mac_ocp_write(sc, 0xE098, 0x0AA2);
                }
                CSR_WRITE_1(sc, RE_TDFNR, 0x4);

                if (sc->re_type == MACFG_56 || sc->re_type == MACFG_57) {
                        Data32 = re_csi_read(sc, 0x2710);
                        Data32 &=0xFFFF0FFF;
                        Data32 |= (0x04 << 12);
                        re_csi_write(sc, 0x2710, Data32);
                }

                Data32 = re_eri_read(sc, 0xD4, 4, ERIAR_ExGMAC);
                Data32 |= (BIT_7 | BIT_8 | BIT_9 | BIT_10 | BIT_11 | BIT_12);
                re_eri_write(sc, 0xD4, 4, Data32, ERIAR_ExGMAC);

                if (sc->re_type == MACFG_68 || sc->re_type == MACFG_69 ||
                    sc->re_type == MACFG_74 || sc->re_type == MACFG_75 ||
                    sc->re_type == MACFG_76) {
                        Data32 = re_eri_read(sc, 0xDC, 4, ERIAR_ExGMAC);
                        Data32 |= (BIT_2| BIT_3 | BIT_4);
                        re_eri_write(sc, 0xDC, 4, Data32, ERIAR_ExGMAC);
                }

                re_eri_write(sc, 0xC8, 4, 0x00080002, ERIAR_ExGMAC);
                re_eri_write(sc, 0xCC, 2, 0x38, ERIAR_ExGMAC);
                re_eri_write(sc, 0xD0, 2, 0x48, ERIAR_ExGMAC);
                re_eri_write(sc, 0xE8, 4, 0x00100006, ERIAR_ExGMAC);

                if (sc->re_type == MACFG_68 || sc->re_type == MACFG_69 ||
                    sc->re_type == MACFG_74 || sc->re_type == MACFG_75 ||
                    sc->re_type == MACFG_76) {
                        re_mac_ocp_write(sc, 0xE054, 0x0000);

                        Data32 = re_eri_read(sc, 0x5F0, 4, ERIAR_ExGMAC);
                        Data32 &= ~(BIT_11 | BIT_10 | BIT_9 | BIT_8 | BIT_3 | BIT_2 | BIT_1 | BIT_0);
                        re_eri_write(sc, 0x5F0, 4, Data32, ERIAR_ExGMAC);
                } else {
                        re_eri_write(sc, 0x5F0, 2, 0x4F87, ERIAR_ExGMAC);
                }

                if (sc->re_type == MACFG_74 || sc->re_type == MACFG_75)
                        re_set_mac_ocp_bit(sc, 0xD438, (BIT_1 | BIT_0));

                Data32 = re_eri_read(sc, 0x2FC, 4, ERIAR_ExGMAC);
                Data32 &= ~(BIT_0 | BIT_1 | BIT_2);
                Data32 |= (BIT_0);
                re_eri_write(sc, 0x2FC, 4, Data32, ERIAR_ExGMAC);

                Data32 = re_eri_read(sc, 0x1B0, 4, ERIAR_ExGMAC);
                Data32 &= ~BIT_12;
                re_eri_write(sc, 0x1B0, 4, Data32, ERIAR_ExGMAC);

                CSR_WRITE_4(sc, RE_TXCFG, CSR_READ_4(sc, RE_TXCFG) | BIT_7);
                CSR_WRITE_1(sc, 0xD3, CSR_READ_1(sc, 0xD3) & ~BIT_7);
                CSR_WRITE_1(sc, 0x1B, CSR_READ_1(sc, 0x1B) & ~0x07);

                CSR_WRITE_2 (sc, RE_CPlusCmd, 0x2060);

                if (sc->re_type == MACFG_74) {
                        re_clear_mac_ocp_bit(sc, 0xD438, BIT_2);

                        re_clear_ephy_ext_addr(sc);

                        re_set_mac_ocp_bit(sc, 0xD438, BIT_2);
                } else if (sc->re_type == MACFG_75) {
                        re_clear_mac_ocp_bit(sc, 0xD438, BIT_2);

                        re_clear_ephy_ext_addr(sc);

                        re_set_mac_ocp_bit(sc, 0xD438, BIT_2);
                }

                if (sc->re_type == MACFG_60) {
                        data16 = re_mac_ocp_read(sc, 0xD3C0);
                        data16 &= 0xF000;
                        data16 |= 0x0FFF;
                        re_mac_ocp_write(sc, 0xD3C0, data16);

                        data16 = re_mac_ocp_read(sc, 0xD3C2);
                        data16 &= 0xFF00;
                        re_mac_ocp_write(sc, 0xD3C2, data16);

                        data16 = re_mac_ocp_read(sc, 0xD3C4);
                        data16 |= (BIT_0);
                        re_mac_ocp_write(sc, 0xD3C4, data16);
                } else if (sc->re_type == MACFG_68 || sc->re_type == MACFG_69 ||
                           sc->re_type == MACFG_76) {
                        if (sc->RequireAdjustUpsTxLinkPulseTiming) {
                                data16 = re_mac_ocp_read(sc, 0xD412);
                                data16 &= ~(0x0FFF);
                                data16 |= sc->SwrCnt1msIni;
                                re_mac_ocp_write(sc, 0xD412, data16);
                        }

                        data16 = re_mac_ocp_read(sc, 0xE056);
                        data16 &= ~(BIT_7 | BIT_6 | BIT_5 | BIT_4);
                        re_mac_ocp_write(sc, 0xE056, data16);

                        data16 = re_mac_ocp_read(sc, 0xE052);
                        data16 &= ~(BIT_15 | BIT_14 | BIT_13 | BIT_3);
                        data16 |= BIT_15;
                        re_mac_ocp_write(sc, 0xE052, data16);

                        data16 = re_mac_ocp_read(sc, 0xD420);
                        data16 &= ~(BIT_11 | BIT_10 | BIT_9 | BIT_8 | BIT_7 | BIT_6 | BIT_5 | BIT_4 | BIT_3 | BIT_2 | BIT_1 | BIT_0);
                        data16 |= 0x45F;
                        re_mac_ocp_write(sc, 0xD420, data16);

                        data16 = re_mac_ocp_read(sc, 0xE0D6);
                        data16 &= ~(BIT_8 | BIT_7 | BIT_6 | BIT_5 | BIT_4 | BIT_3 | BIT_2 | BIT_1 | BIT_0);
                        data16 |= 0x17F;
                        re_mac_ocp_write(sc, 0xE0D6, data16);
                }

                CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) & ~BIT_0);

                CSR_WRITE_1(sc, 0xD0, CSR_READ_1(sc, 0xD0) | BIT_6);
                CSR_WRITE_1(sc, 0xF2, CSR_READ_1(sc, 0xF2) | BIT_6);

                CSR_WRITE_1(sc, 0xF2, CSR_READ_1(sc, 0xF2) & ~BIT_3);

                if (ifp->if_mtu > ETHERMTU)
                        CSR_WRITE_1 (sc, RE_MTPS, 0x27);

                if (sc->re_type == MACFG_56 || sc->re_type == MACFG_57 ||
                    sc->re_type == MACFG_58 || sc->re_type == MACFG_59) {
                        re_mac_ocp_write(sc, 0xC140, 0xFFFF);
                } else if (sc->re_type == MACFG_68 || sc->re_type == MACFG_69 ||
                           sc->re_type == MACFG_74 || sc->re_type == MACFG_75 ||
                           sc->re_type == MACFG_76) {
                        re_mac_ocp_write(sc, 0xC140, 0xFFFF);
                        re_mac_ocp_write(sc, 0xC142, 0xFFFF);
                }

                if (ifp->if_mtu > ETHERMTU) {
                        ifp->if_capenable &= ~IFCAP_HWCSUM;
                        ifp->if_hwassist &= ~RE_CSUM_FEATURES;
                } else {
                        if (sc->re_tx_cstag) {
                                ifp->if_capenable |= IFCAP_TXCSUM;
                                ifp->if_hwassist |= RE_CSUM_FEATURES;
                        }
                        if (sc->re_rx_cstag) {
                                ifp->if_capenable |= IFCAP_RXCSUM;
                        }
                }
        } else if (macver == 0x50000000) {
                Data32 = re_eri_read(sc, 0xD4, 4, ERIAR_ExGMAC);
                Data32 |= BIT_7 | BIT_8 | BIT_9 | BIT_10 | BIT_11 | BIT_12;
                re_eri_write(sc, 0xD4, 4, Data32, ERIAR_ExGMAC);

                re_eri_write(sc, 0xC8, 4, 0x00080002, ERIAR_ExGMAC);
                re_eri_write(sc, 0xCC, 2, 0x2F, ERIAR_ExGMAC);
                re_eri_write(sc, 0xD0, 2, 0x5F, ERIAR_ExGMAC);
                re_eri_write(sc, 0xE8, 4, 0x00100006, ERIAR_ExGMAC);

                if (sc->re_type == MACFG_62 || sc->re_type == MACFG_67) {
                        OOB_mutex_lock(sc);
                        re_eri_write(sc, 0x5F0, 4, 0x4F87, ERIAR_ExGMAC);
                        OOB_mutex_unlock(sc);
                }

                Data32 = re_eri_read(sc, 0x2FC, 4, ERIAR_ExGMAC);
                Data32 &= ~(BIT_0 | BIT_1 | BIT_2);
                Data32 |= (BIT_0 | BIT_1);
                re_eri_write(sc, 0x2FC, 4, Data32, ERIAR_ExGMAC);

                Data32 = re_eri_read(sc, 0x1B0, 4, ERIAR_ExGMAC);
                Data32 &= ~BIT_12;
                re_eri_write(sc, 0x1B0, 4, Data32, ERIAR_ExGMAC);

                CSR_WRITE_4(sc, RE_TXCFG, CSR_READ_4(sc, RE_TXCFG) | BIT_7);
                CSR_WRITE_1(sc, 0xD3, CSR_READ_1(sc, 0xD3) & ~BIT_7);
                CSR_WRITE_1(sc, 0x1B, CSR_READ_1(sc, 0x1B) & ~0x07);

                CSR_WRITE_2 (sc, RE_CPlusCmd, 0x2060);

                CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) & ~BIT_0);

                CSR_WRITE_1(sc, 0xD0, CSR_READ_1(sc, 0xD0) | BIT_6);
                CSR_WRITE_1(sc, 0xF2, CSR_READ_1(sc, 0xF2) | BIT_6);

                CSR_WRITE_1(sc, 0xF2, CSR_READ_1(sc, 0xF2) & ~BIT_3);

                if (ifp->if_mtu > ETHERMTU)
                        CSR_WRITE_1 (sc, RE_MTPS, 0x27);

                if (sc->re_type == MACFG_67) {
                        data16 = re_mac_ocp_read(sc, 0xD3E2);
                        data16 &= 0xF000;
                        data16 |= 0xAFD;
                        re_mac_ocp_write(sc, 0xD3E2, data16);

                        data16 = re_mac_ocp_read(sc, 0xD3E4);
                        data16 &= 0xFF00;
                        re_mac_ocp_write(sc, 0xD3E4, data16);

                        data16 = re_mac_ocp_read(sc, 0xE860);
                        data16 |= BIT_7;
                        re_mac_ocp_write(sc, 0xE860, data16);
                }

                re_mac_ocp_write(sc, 0xC140, 0xFFFF);
                re_mac_ocp_write(sc, 0xC142, 0xFFFF);

                if (ifp->if_mtu > ETHERMTU) {
                        ifp->if_capenable &= ~IFCAP_HWCSUM;
                        ifp->if_hwassist &= ~RE_CSUM_FEATURES;
                } else {
                        if (sc->re_tx_cstag) {
                                ifp->if_capenable |= IFCAP_TXCSUM;
                                ifp->if_hwassist |= RE_CSUM_FEATURES;
                        }
                        if (sc->re_rx_cstag) {
                                ifp->if_capenable |= IFCAP_RXCSUM;
                        }
                }
        } else if (macver == 0x54800000) {
                re_mac_ocp_write(sc, 0xE098, 0xC302);

                re_mac_ocp_write(sc, 0xD400, re_mac_ocp_read(sc, 0xD400) & ~(BIT_0));

                if (sc->RequireAdjustUpsTxLinkPulseTiming) {
                        data16 = re_mac_ocp_read(sc, 0xD412);
                        data16 &= ~(0x0FFF);
                        data16 |= sc->SwrCnt1msIni;
                        re_mac_ocp_write(sc, 0xD412, data16);
                }

                data16 = re_mac_ocp_read(sc, 0xE056);
                data16 &= ~(BIT_7 | BIT_6 | BIT_5 | BIT_4);
                if (sc->HwPkgDet == 0x0F)
                        data16 |= (BIT_6 | BIT_5 | BIT_4);
                re_mac_ocp_write(sc, 0xE056, data16);
                if (FALSE == HW_SUPP_SERDES_PHY(sc))
                        re_mac_ocp_write(sc, 0xEA80, 0x0003);
                else
                        re_mac_ocp_write(sc, 0xEA80, 0x0000);

                OOB_mutex_lock(sc);
                data16 = re_mac_ocp_read(sc, 0xE052);
                data16 &= ~(BIT_3 | BIT_0);
                if (sc->HwPkgDet == 0x0F)
                        data16 |= BIT_0;
                re_mac_ocp_write(sc, 0xE052, data16);
                OOB_mutex_unlock(sc);

                data16 = re_mac_ocp_read(sc, 0xD420);
                data16 &= ~(BIT_11 | BIT_10 | BIT_9 | BIT_8 | BIT_7 | BIT_6 | BIT_5 | BIT_4 | BIT_3 | BIT_2 | BIT_1 | BIT_0);
                data16 |= 0x45F;
                re_mac_ocp_write(sc, 0xD420, data16);

                CSR_WRITE_1(sc, RE_TDFNR, 0x4);

                data16 = re_mac_ocp_read(sc, 0xE63E);
                data16 &= ~(BIT_3 | BIT_2 | BIT_1);
                re_mac_ocp_write(sc, 0xE63E, data16);
                data16 |= (BIT_0);
                re_mac_ocp_write(sc, 0xE63E, data16);
                data16 &= ~(BIT_0);
                re_mac_ocp_write(sc, 0xE63E, data16);
                re_mac_ocp_write(sc, 0xC094, 0x0);
                re_mac_ocp_write(sc, 0xC09E, 0x0);

                Data32 = re_eri_read(sc, 0xD4, 4, ERIAR_ExGMAC);
                Data32 |= BIT_7 | BIT_8 | BIT_9 | BIT_10 | BIT_11 | BIT_12;
                if (sc->re_type == MACFG_71 || sc->re_type == MACFG_72 ||
                    sc->re_type == MACFG_73)
                        Data32 |= BIT_4;
                re_eri_write(sc, 0xD4, 4, Data32, ERIAR_ExGMAC);

                Data32 = re_eri_read(sc, 0xDC, 4, ERIAR_ExGMAC);
                Data32 |= (BIT_2| BIT_3);
                re_eri_write(sc, 0xDC, 4, Data32, ERIAR_ExGMAC);

                re_eri_write(sc, 0xC8, 4, 0x00080002, ERIAR_ExGMAC);
                re_eri_write(sc, 0xCC, 2, 0x2F, ERIAR_ExGMAC);
                re_eri_write(sc, 0xD0, 2, 0x5F, ERIAR_ExGMAC);
                re_eri_write(sc, 0xE8, 4, 0x00100006, ERIAR_ExGMAC);

                OOB_mutex_lock(sc);
                if (sc->HwPkgDet == 0x0F)
                        re_eri_write(sc, 0x5F0, 2, 0x4F00, ERIAR_ExGMAC);
                else
                        re_eri_write(sc, 0x5F0, 2, 0x4000, ERIAR_ExGMAC);
                OOB_mutex_unlock(sc);

                Data32 = re_eri_read(sc, 0x2FC, 4, ERIAR_ExGMAC);
                Data32 &= ~(BIT_0 | BIT_1);
                Data32 |= (BIT_0);
                re_eri_write(sc, 0x2FC, 4, Data32, ERIAR_ExGMAC);

                Data32 = re_eri_read(sc, 0x1B0, 4, ERIAR_ExGMAC);
                Data32 &= ~BIT_12;
                re_eri_write(sc, 0x1B0, 4, Data32, ERIAR_ExGMAC);

                Data32 = re_eri_read(sc, 0x1D0, 1, ERIAR_ExGMAC);
                Data32 &= ~BIT_1;
                re_eri_write(sc, 0x1D0, 1, Data32, ERIAR_ExGMAC);

                re_eri_write(sc, 0xC0, 2, 0x0000, ERIAR_ExGMAC);
                re_eri_write(sc, 0xB8, 4, 0x00000000, ERIAR_ExGMAC);

                CSR_WRITE_4(sc, RE_TXCFG, CSR_READ_4(sc, RE_TXCFG) | BIT_7);
                CSR_WRITE_1(sc, 0xD3, CSR_READ_1(sc, 0xD3) & ~BIT_7);
                CSR_WRITE_1(sc, 0x1B, CSR_READ_1(sc, 0x1B) & ~0x07);

                CSR_WRITE_2 (sc, RE_CPlusCmd, 0x2060);

                CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) & ~BIT_0);

                if (FALSE == HW_SUPP_SERDES_PHY(sc)) {
                        CSR_WRITE_1(sc, 0xD0, CSR_READ_1(sc, 0xD0) | BIT_6);
                        CSR_WRITE_1(sc, 0xF2, CSR_READ_1(sc, 0xF2) | BIT_6);
                        CSR_WRITE_1(sc, 0xD0, CSR_READ_1(sc, 0xD0) | BIT_7);
                } else {
                        CSR_WRITE_1(sc, 0xD0, CSR_READ_1(sc, 0xD0) & ~BIT_6);
                        CSR_WRITE_1(sc, 0xF2, CSR_READ_1(sc, 0xF2) & ~BIT_6);
                        CSR_WRITE_1(sc, 0xD0, CSR_READ_1(sc, 0xD0) & ~BIT_7);
                }

                CSR_WRITE_1(sc, 0xF2, CSR_READ_1(sc, 0xF2) & ~BIT_3);

                if (ifp->if_mtu > ETHERMTU)
                        CSR_WRITE_1 (sc, RE_MTPS, 0x27);

                re_mac_ocp_write(sc, 0xC140, 0xFFFF);
                re_mac_ocp_write(sc, 0xC142, 0xFFFF);

                if (ifp->if_mtu > ETHERMTU) {
                        ifp->if_capenable &= ~IFCAP_HWCSUM;
                        ifp->if_hwassist &= ~RE_CSUM_FEATURES;
                } else {
                        if (sc->re_tx_cstag) {
                                ifp->if_capenable |= IFCAP_TXCSUM;
                                ifp->if_hwassist |= RE_CSUM_FEATURES;
                        }
                        if (sc->re_rx_cstag) {
                                ifp->if_capenable |= IFCAP_RXCSUM;
                        }
                }
        }

        if (!((sc->re_if_flags & RL_FLAG_DESCV2) &&
              (sc->re_if_flags & RL_FLAG_8168G_PLUS)))
                ifp->if_hwassist &= ~(CSUM_TCP_IPV6 | CSUM_UDP_IPV6);

        //clear io_rdy_l23
        switch (sc->re_type) {
        case MACFG_42:
        case MACFG_43:
        case MACFG_52:
        case MACFG_53:
        case MACFG_54:
        case MACFG_55:
        case MACFG_56:
        case MACFG_57:
        case MACFG_58:
        case MACFG_59:
        case MACFG_60:
        case MACFG_61:
        case MACFG_62:
        case MACFG_67:
        case MACFG_68:
        case MACFG_69:
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
        case MACFG_74:
        case MACFG_75:
        case MACFG_76:
                CSR_WRITE_1(sc, RE_CFG3, CSR_READ_1(sc, RE_CFG3) & ~BIT_1);
                break;
        }

        switch(sc->re_type) {
        case MACFG_36:
        case MACFG_37:
        case MACFG_38:
        case MACFG_39:
        case MACFG_42:
        case MACFG_43:
        case MACFG_50:
        case MACFG_51:
        case MACFG_52:
        case MACFG_53:
        case MACFG_54:
        case MACFG_55:
        case MACFG_56:
        case MACFG_57:
        case MACFG_58:
        case MACFG_59:
        case MACFG_60:
        case MACFG_61:
        case MACFG_62:
        case MACFG_67:
        case MACFG_68:
        case MACFG_69:
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
        case MACFG_74:
        case MACFG_75:
        case MACFG_76:
                _re_enable_aspm_clkreq_lock(sc, 1);
                re_enable_force_clkreq(sc, 0);
                break;
        }

        //clear wol
        re_clrwol(sc);

        data16 = CSR_READ_2(sc, RE_CPlusCmd);
        if ((ifp->if_capenable & IFCAP_VLAN_HWTAGGING) != 0)
                data16 |= RL_CPLUSCMD_VLANSTRIP;
        else
                data16 &= ~RL_CPLUSCMD_VLANSTRIP;

        if ((ifp->if_capenable & IFCAP_RXCSUM) != 0)
                data16 |= RL_RxChkSum;
        else
                data16 &= ~RL_RxChkSum;
        CSR_WRITE_2 (sc, RE_CPlusCmd, data16);

        re_disable_cfg9346_write(sc);
        //CSR_WRITE_1(sc, 0xec, 0x3f);

        if (sc->re_device_id == RT_DEVICEID_8169 || sc->re_device_id == RT_DEVICEID_8169SC) {
                /* Enable transmit and receive.*/
                CSR_WRITE_1(sc, RE_COMMAND, RE_CMD_TX_ENB | RE_CMD_RX_ENB);

                /* Set the initial TX configuration.*/
                CSR_WRITE_4(sc, RE_TXCFG, RE_TXCFG_CONFIG);

                /* Set the initial RX configuration.*/
                /*
                 * Program the multicast filter, if necessary.
                 */
                re_set_rx_packet_filter(sc);
        } else {
                /* Set the initial RX configuration.*/
                /*
                 * Program the multicast filter, if necessary.
                 */
                re_set_rx_packet_filter(sc);

                /* Enable transmit and receive.*/
                CSR_WRITE_1(sc, RE_COMMAND, RE_CMD_TX_ENB | RE_CMD_RX_ENB);
        }

        ifp->if_drv_flags |= IFF_DRV_RUNNING;
        ifp->if_drv_flags &= ~IFF_DRV_OACTIVE;

        /*
        * Enable interrupts.
        */
        CSR_WRITE_2(sc, RE_IMR, RE_INTRS);
}

static void re_init_unlock(void *xsc)  	/* Software & Hardware Initialize */
{
        struct re_softc		*sc = xsc;
        struct ifnet		*ifp;
#if OS_VER < VERSION(6,0)
        int			i;
#endif
        union {
                uint32_t align_dummy;
                u_char eaddr[ETHER_ADDR_LEN];
        } eaddr;

        ifp = RE_GET_IFNET(sc);

        /*
         * Cancel pending I/O and free all RX/TX buffers.
         */
        re_stop(sc);

        /*
        * Disable TSO if interface MTU size is greater than MSS
        * allowed in controller.
        */
        if (ifp->if_mtu > ETHERMTU) {
                ifp->if_capenable &= ~(IFCAP_TSO | IFCAP_VLAN_HWTSO);
                ifp->if_hwassist &= ~CSUM_TSO;
        }

        /* Copy MAC address on stack to align. */
#if OS_VER < VERSION(6,0)
        bcopy((char *)&sc->arpcom.ac_enaddr, eaddr.eaddr, ETHER_ADDR_LEN);
#elif OS_VER < VERSION(7,0)
        bcopy(IFP2ENADDR(ifp), eaddr.eaddr, ETHER_ADDR_LEN);
#else
        bcopy(IF_LLADDR(ifp), eaddr.eaddr, ETHER_ADDR_LEN);
#endif

        /* Init our MAC address */
        re_rar_set(sc, eaddr.eaddr);

        sc->hw_start_unlock(sc);

        return;
}


static void re_init_locked(void *xsc)  	/* Software & Hardware Initialize */
{
        struct re_softc		*sc = xsc;
        struct ifnet		*ifp;

        ifp = RE_GET_IFNET(sc);

        if (re_link_ok(sc)) {
                sc->link_state = LINK_STATE_UP;
                re_link_state_change(ifp, sc->link_state);
                re_link_on_patch(sc);
        }

        sc->re_link_chg_det = 1;
        re_start_timer(sc);
}

static void re_init(void *xsc)  	/* Software & Hardware Initialize */
{
        struct re_softc		*sc = xsc;

        RE_LOCK(sc);
        re_init_locked(sc);
        RE_UNLOCK(sc);
}

static u_int32_t re_get_default_imr_8125(struct re_softc *sc)
{
        if (sc->use_new_intr_mapping) {
                switch (sc->HwSuppIsrVer) {
                case 2:
                        return RE_INTRS_8125B;
                case 3:
                        return RE_INTRS_8126;
                case 4:
                        return RE_INTRS_8125BP;
                case 5:
                        return RE_INTRS_8125D;
                case 6:
                        return RE_INTRS_8127;
                case 7:
                        return RE_INTRS_8125CP;
                default:
                        device_printf(sc->dev,"invalid isr version \n");
                        return 0;
                }
        } else
                return RE_INTRS;
}

static u_int32_t re_get_isr_8125(struct re_softc *sc)
{
        if (sc->use_new_intr_mapping)
                return CSR_READ_4(sc, RE_ISR_V2_8125);
        else
                return CSR_READ_4(sc, RE_ISR0_8125);
}

static void re_set_isr_8125(struct re_softc *sc, u_int32_t val)
{
        if (sc->use_new_intr_mapping)
                CSR_WRITE_4(sc, RE_ISR_V2_8125, val);
        else
                CSR_WRITE_4(sc, RE_ISR0_8125, val);
}

static void re_enable_imr_8125(struct re_softc *sc)
{
        if (sc->use_new_intr_mapping)
                CSR_WRITE_4(sc, RE_IMR_V2_SET_REG_8125, re_get_default_imr_8125(sc));
        else
                CSR_WRITE_4(sc, RE_IMR0_8125, RE_INTRS);
}

static void re_disable_imr_8125(struct re_softc *sc)
{
        if (sc->use_new_intr_mapping) {
                CSR_WRITE_4(sc, RE_IMR_V2_CLEAR_REG_8125, 0xffffffff);
                if (sc->HwSuppIsrVer > 3)
                        CSR_WRITE_4(sc, RE_IMR_V4_L2_CLEAR_REG_8125, 0xffffffff);
        } else
                CSR_WRITE_4(sc, RE_IMR0_8125, 0x00000000);
}

static void re_hw_start_unlock_8125(struct re_softc *sc)
{
        struct ifnet		*ifp;
        u_int32_t		macver;
        u_int16_t		data16 = 0;
        u_int8_t 		data8;

        ifp = RE_GET_IFNET(sc);

        /* Init descriptors. */
        re_var_init(sc);

        re_enable_cfg9346_write(sc);

        _re_enable_aspm_clkreq_lock(sc, 0);
        re_enable_force_clkreq(sc, 0);

        re_set_eee_lpi_timer(sc);

        CSR_WRITE_2(sc, RE_CPlusCmd, 0x2060);

        /* Set the initial TX configuration.*/
        CSR_WRITE_4(sc, RE_TXCFG, RE_TXCFG_CONFIG);

        macver = CSR_READ_4(sc, RE_TXCFG) & 0xFC800000;

        re_set_l1_l0s_entry_latency(sc, macver);

        re_set_mrrs(sc, macver);

        if (macver == 0x60800000 || macver == 0x64000000 ||
            macver == 0x64800000 || macver == 0x68000000 ||
            macver == 0x68800000 || macver == 0x70800000 ||
            macver == 0x6C800000) {
                CSR_WRITE_1(sc, RE_CFG2, CSR_READ_1(sc, RE_CFG2) | BIT_5);

                re_mac_ocp_write(sc, 0xE098, 0xC302);

                if (macver == 0x64800000)
                        re_disable_l1_timeout(sc);

                if (macver == 0x60800000 || macver == 0x64000000)
                        CSR_WRITE_2(sc, 0x382, 0x221B);

                CSR_WRITE_1(sc, 0x4500, 0x00);
                CSR_WRITE_2(sc, 0x4800, 0x0000);

                CSR_WRITE_1(sc, RE_CFG1, CSR_READ_1(sc, RE_CFG1) & ~0x10);

                CSR_WRITE_1(sc, 0xF2, CSR_READ_1(sc, 0xF2) & ~BIT_3);

                CSR_WRITE_1(sc, RE_TDFNR, 0x10);

                re_mac_ocp_write(sc, 0xC140, 0xFFFF);
                re_mac_ocp_write(sc, 0xC142, 0xFFFF);

                //old tx desc format
                data16 = re_mac_ocp_read(sc, 0xEB58);
                if (sc->re_type == MACFG_91 || sc->re_type == MACFG_92 ||
                    sc->re_type == MACFG_100 || sc->re_type == MACFG_101)
                        data16 &= ~(BIT_0 | BIT_1);
                else
                        data16 &= ~(BIT_0);
                re_mac_ocp_write(sc, 0xEB58, data16);

                if (sc->re_type == MACFG_84 || sc->re_type == MACFG_85 ||
                    sc->re_type == MACFG_86 || sc->re_type == MACFG_87 ||
                    sc->re_type == MACFG_88 || sc->re_type == MACFG_91 ||
                    sc->re_type == MACFG_92 || sc->re_type == MACFG_100 ||
                    sc->re_type == MACFG_101)
                        CSR_WRITE_1(sc, 0xD8, CSR_READ_1(sc, 0xD8) & ~BIT_1);

                /* Disable descfifo_release_option for tx close mode */
                if (sc->re_type == MACFG_100 || sc->re_type == MACFG_101)
                        CSR_WRITE_1(sc, 0x20E4, CSR_READ_1(sc, 0x20E4) & ~BIT_2);

                if (sc->re_type == MACFG_88 || sc->re_type == MACFG_101) {
                        re_clear_mac_ocp_bit(sc, 0xE00C, BIT_12);

                        re_clear_mac_ocp_bit(sc, 0xC0C2, BIT_6);
                }

                data16 = re_mac_ocp_read(sc, 0xE614);
                if (sc->re_type == MACFG_100 || sc->re_type == MACFG_101) {
                        data16 &= ~(BIT_11 | BIT_10 | BIT_9 | BIT_8);
                        data16 |= (15 << 8);
                } else {
                        data16 &= ~(BIT_10 | BIT_9 | BIT_8);
                        if (sc->re_type == MACFG_82 || sc->re_type == MACFG_83)
                                data16 |= (2 << 8);
                        else if (sc->re_type == MACFG_100 || sc->re_type == MACFG_101)
                                data16 |= (4 << 8);
                        else
                                data16 |= (3 << 8);
                }
                re_mac_ocp_write(sc, 0xE614, data16);

                data16 = re_mac_ocp_read(sc, 0xE63E);
                data16 &= ~(BIT_11 | BIT_10);
                re_mac_ocp_write(sc, 0xE63E, data16);

                data16 = re_mac_ocp_read(sc, 0xE63E);
                data16 &= ~(BIT_5 | BIT_4);
                data16 |= (0x02 << 4);
                re_mac_ocp_write(sc, 0xE63E, data16);

                data16 = re_mac_ocp_read(sc, 0xC0B4);
                data16 |= (BIT_3|BIT_2);
                re_mac_ocp_write(sc, 0xC0B4, data16);

                data16 = re_mac_ocp_read(sc, 0xEB6A);
                data16 &= ~(BIT_7 | BIT_6 | BIT_5 | BIT_4 | BIT_3 | BIT_2 | BIT_1 | BIT_0);
                data16 |= (BIT_5 | BIT_4 | BIT_1 | BIT_0);
                re_mac_ocp_write(sc, 0xEB6A, data16);

                data16 = re_mac_ocp_read(sc, 0xEB50);
                data16 &= ~(BIT_9 | BIT_8 | BIT_7 | BIT_6 | BIT_5);
                data16 |= (BIT_6);
                re_mac_ocp_write(sc, 0xEB50, data16);

                data16 = re_mac_ocp_read(sc, 0xE056);
                data16 &= ~(BIT_7 | BIT_6 | BIT_5 | BIT_4);
                //data16 |= (BIT_4 | BIT_5);
                re_mac_ocp_write(sc, 0xE056, data16);

                CSR_WRITE_1(sc, 0xD0, CSR_READ_1(sc, 0xD0) | BIT_7);

                data16 = re_mac_ocp_read(sc, 0xE040);
                data16 &= ~(BIT_12);
                re_mac_ocp_write(sc, 0xE040, data16);

                data16 = re_mac_ocp_read(sc, 0xEA1C);
                data16 &= ~(BIT_1 | BIT_0);
                data16 |= (BIT_0);
                re_mac_ocp_write(sc, 0xEA1C, data16);

                if (HW_DASH_SUPPORT_DASH(sc))
                        OOB_mutex_lock(sc);

                if (sc->re_type == MACFG_86 || sc->re_type == MACFG_87)
                        re_mac_ocp_write(sc, 0xE0C0, 0x4403);
                else
                        re_mac_ocp_write(sc, 0xE0C0, 0x4000);

                re_set_mac_ocp_bit(sc, 0xE052, (BIT_6 | BIT_5));
                re_clear_mac_ocp_bit(sc, 0xE052, BIT_3 | BIT_7);

                if (HW_DASH_SUPPORT_DASH(sc))
                        OOB_mutex_unlock(sc);

                data16 = re_mac_ocp_read(sc, 0xC0AC);
                data16 |= (BIT_7 | BIT_8 | BIT_9 | BIT_10 | BIT_11 | BIT_12);
                if (macver == 0x60800000)
                        data16 &= ~(BIT_7);
                re_mac_ocp_write(sc, 0xC0AC, data16);

                data16 = re_mac_ocp_read(sc, 0xD430);
                data16 &= ~(BIT_11 | BIT_10 | BIT_9 | BIT_8 | BIT_7 | BIT_6 | BIT_5 | BIT_4 | BIT_3 | BIT_2 | BIT_1 | BIT_0);
                data16 |= 0x45F;
                re_mac_ocp_write(sc, 0xD430, data16);

                //re_mac_ocp_write(sc, 0xE0C0, 0x4F87);
                CSR_WRITE_1(sc, 0xD0, CSR_READ_1(sc, 0xD0) | (BIT_6 | BIT_7));

                if (sc->re_type == MACFG_80 || sc->re_type == MACFG_81)
                        CSR_WRITE_1(sc, 0xD3, CSR_READ_1(sc, 0xD3) | BIT_0);

                if (sc->re_type != MACFG_86 && sc->re_type != MACFG_87)
                        re_mac_ocp_write(sc, 0xE080, re_mac_ocp_read(sc, 0xE080)&~BIT_1);

                data16 = re_mac_ocp_read(sc, 0xEA1C);
                data16 &= ~(BIT_2);
                if (sc->re_type == MACFG_91 || sc->re_type == MACFG_92 ||
                    sc->re_type == MACFG_100 || sc->re_type == MACFG_101)
                        data16 &= ~(BIT_9 | BIT_8);
                re_mac_ocp_write(sc, 0xEA1C, data16);

                re_set_mac_ocp_bit(sc, 0xEB54, BIT_0);
                DELAY(1);
                re_clear_mac_ocp_bit(sc, 0xEB54, BIT_0);
                CSR_WRITE_2(sc, 0x1880, CSR_READ_2(sc, 0x1880) & ~(BIT_4 | BIT_5));

                if (macver == 0x60800000 || macver == 0x68000000 ||
                    macver == 0x68800000 || macver == 0x70800000 ||
                    macver == 0x6C800000) {
                        for (int i=0xA00; i<0xB00; i+=4)
                                CSR_WRITE_4(sc, i, 0x0000);
                } else {
                        for (int i = 0xA00; i < 0xA80; i += 4)
                                CSR_WRITE_4(sc, i, 0x0000);
                }

                if (macver == 0x60800000) {
                        //do nothing
                } else {
                        data8 = CSR_READ_1(sc, RE_INT_CFG0_8125);
                        data8 &= ~(RTL8125_INT_CFG0_ENABLE_8125 |
                                   RTL8125_INT_CFG0_TIMEOUT0_BYPASS |
                                   RTL8125_INT_CFG0_MITIGATION_BYPASS);
                        if (sc->re_type == MACFG_91 || sc->re_type == MACFG_92 ||
                            sc->re_type == MACFG_100 || sc->re_type == MACFG_101)
                                data8 &= ~RTL8126_INT_CFG0_RDU_BYPASS;
                        if (sc->re_type == MACFG_84 || sc->re_type == MACFG_85 ||
                            sc->re_type == MACFG_86 || sc->re_type == MACFG_87)
                                data8 &= ~RTL8125_INT_CFG0_MSIX_ENTRY_NUM_MODE;
                        if (sc->re_type == MACFG_88 || sc->re_type == MACFG_100 ||
                            sc->re_type == MACFG_101)
                                data8 &= ~(RTL8125_INT_CFG0_AUTO_CLR_IMR |
                                           RTL8125_INT_CFG0_AVOID_MISS_INTR);
                        /* New intr mapping */
                        if (sc->use_new_intr_mapping)
                                data8 |= RTL8125_INT_CFG0_ENABLE_8125;
                        CSR_WRITE_1(sc, RE_INT_CFG0_8125, data8);

                        CSR_WRITE_2(sc, RE_INT_CFG1_8125, 0x0000);
                }

                if (macver == 0x64800000 || macver == 0x68000000 ||
                    macver == 0x68800000 || macver == 0x70800000 ||
                    macver == 0x6C800000) {
                        data16 = CSR_READ_2(sc, RE_AVB_CTRL);
                        data16 &= ~RE_DVLAN_MODE_ENABLE;
                        CSR_WRITE_2(sc, RE_AVB_CTRL, data16);
                }

                if (sc->re_tx_cstag) {
                        ifp->if_capenable |= IFCAP_TXCSUM;
                        ifp->if_hwassist |= RE_CSUM_FEATURES;
                }
                if (sc->re_rx_cstag) {
                        ifp->if_capenable |= IFCAP_RXCSUM;
                }
        }

        if (sc->RequiredPfmPatch)
                re_set_pfm_patch(sc, 0);

        _re_enable_aspm_clkreq_lock(sc, 1);
        re_enable_force_clkreq(sc, 0);

        //clear wol
        re_clrwol(sc);

        //Interrupt Mitigation
        if (macver == 0x68000000 || macver == 0x68800000 ||
            macver == 0x70800000 || macver == 0x6C800000)
                CSR_WRITE_4(sc, 0x0A00, 0x00140014);
        else
                CSR_WRITE_4(sc, 0x0A00, 0x00630063);

        if ((ifp->if_capenable & IFCAP_VLAN_HWTAGGING) != 0)
                CSR_WRITE_4(sc, RE_RXCFG, CSR_READ_4(sc, RE_RXCFG) | (BIT_22 | BIT_23));
        else
                CSR_WRITE_4(sc, RE_RXCFG, CSR_READ_4(sc, RE_RXCFG) & ~(BIT_22 | BIT_23));

        data16 = CSR_READ_2(sc, RE_CPlusCmd);
        if ((ifp->if_capenable & IFCAP_RXCSUM) != 0)
                data16 |= RL_RxChkSum;
        else
                data16 &= ~RL_RxChkSum;
        CSR_WRITE_2 (sc, RE_CPlusCmd, data16);

        re_disable_cfg9346_write(sc);

        /* Set the initial RX configuration.*/
        /*
         * Program the multicast filter, if necessary.
         */
        re_set_rx_packet_filter(sc);

        /* Enable transmit and receive.*/
        CSR_WRITE_1(sc, RE_COMMAND, RE_CMD_TX_ENB | RE_CMD_RX_ENB);

        ifp->if_drv_flags |= IFF_DRV_RUNNING;
        ifp->if_drv_flags &= ~IFF_DRV_OACTIVE;

        /*
        * Enable interrupts.
        */
        re_enable_imr_8125(sc);
}

/*
 * Initialize the transmit descriptors.
 */
static int re_var_init(struct re_softc *sc)
{
        int			i;
        union RxDesc *rxptr;
        union TxDesc *txptr;

        for (int j = 0; j < RL_RX_QUEUE_NUM; j++) {
                struct re_dma_map_arg re_dm_arg = {sc, j};
                sc->re_desc.rx_cur_index[j] = 0;
                rxptr = sc->re_desc.rx_desc[j];
                for (i = 0; i < RE_RX_BUF_NUM; i++) {
                        memset(&rxptr[i], 0, sizeof(union RxDesc));

                        /* Init the RX buffer pointer register. */
                        bus_dmamap_load(sc->re_desc.re_rx_mtag[j],
                                        sc->re_desc.re_rx_dmamap[j][i],
                                        sc->re_desc.rx_buf[j][i]->m_data, sc->re_rx_desc_buf_sz,
                                        re_rx_dma_map_buf,
                                        &rxptr[i],
                                        0);
                        bus_dmamap_sync(sc->re_desc.re_rx_mtag[j],
                                        sc->re_desc.re_rx_dmamap[j][i],
                                        BUS_DMASYNC_PREREAD);

                        rxptr[i].ul[0] = htole32(sc->re_rx_desc_buf_sz);
                        if (i == (RE_RX_BUF_NUM - 1))
                                rxptr[i].ul[0] |= htole32(RL_RDESC_CMD_EOR);
                        rxptr[i].ul[0] |= htole32(RL_RDESC_CMD_OWN);
                }

                bus_dmamap_load(sc->re_desc.rx_desc_tag[j],
                                sc->re_desc.rx_desc_dmamap[j],
                                sc->re_desc.rx_desc[j],
                                sizeof(union RxDesc)*RE_RX_BUF_NUM,
                                re_dma_map_rxdesc,
                                &re_dm_arg,
                                0);
                bus_dmamap_sync(sc->re_desc.rx_desc_tag[j],
                                sc->re_desc.rx_desc_dmamap[j],
                                BUS_DMASYNC_PREREAD|BUS_DMASYNC_PREWRITE);
        }

        for (int j=0; j<RL_TX_QUEUE_NUM; j++) {
                struct re_dma_map_arg re_dm_arg = {sc, j};
                sc->re_desc.tx_cur_index[j] = 0;
                sc->re_desc.tx_last_index[j] = 0;
                txptr = sc->re_desc.tx_desc[j];
                for (i = 0; i < RE_TX_BUF_NUM; i++) {
                        memset(&txptr[i], 0, sizeof(union TxDesc));
                        if (i == (RE_TX_BUF_NUM - 1))
                                txptr[i].ul[0] = htole32(RL_TDESC_CMD_EOR);
                }

                bus_dmamap_load(sc->re_desc.tx_desc_tag[j],
                                sc->re_desc.tx_desc_dmamap[j],
                                sc->re_desc.tx_desc[j],
                                sizeof(union TxDesc) * RE_TX_BUF_NUM,
                                re_dma_map_txdesc,
                                &re_dm_arg,
                                0);
                bus_dmamap_sync(sc->re_desc.tx_desc_tag[j],
                                sc->re_desc.tx_desc_dmamap[j],
                                BUS_DMASYNC_PREREAD|BUS_DMASYNC_PREWRITE);
        }

        return 0;
}

static void re_reset(struct re_softc *sc)
{
        register int		i;

        re_clear_all_rx_packet_filter(sc);

        switch (sc->re_type) {
        case MACFG_3:
        case MACFG_4:
        case MACFG_5:
        case MACFG_6:
                DELAY(10000);
                break;
        case MACFG_11:
        case MACFG_12:
        case MACFG_13:
        case MACFG_14:
        case MACFG_15:
        case MACFG_16:
        case MACFG_17:
        case MACFG_18:
        case MACFG_19:
        case MACFG_21:
        case MACFG_22:
        case MACFG_23:
        case MACFG_24:
        case MACFG_25:
        case MACFG_26:
        case MACFG_27:
        case MACFG_28:
        case MACFG_31:
        case MACFG_32:
        case MACFG_33:
        case MACFG_36:
        case MACFG_37:
        case MACFG_41:
        case MACFG_42:
        case MACFG_43:
        case MACFG_54:
        case MACFG_55:
        case MACFG_63:
        case MACFG_64:
        case MACFG_65:
        case MACFG_66:
                CSR_WRITE_1(sc, RE_COMMAND, 0x8C);
                break;
        case MACFG_38:
        case MACFG_39:
        case MACFG_50:
        case MACFG_51:
        case MACFG_52:
        case MACFG_53:
                CSR_WRITE_1(sc, RE_COMMAND, 0x8C);
                while (!(CSR_READ_4(sc,RE_TXCFG) & BIT_11)) DELAY(100);
                break;
        case MACFG_56:
        case MACFG_57:
        case MACFG_58:
        case MACFG_59:
        case MACFG_60:
        case MACFG_61:
        case MACFG_62:
        case MACFG_67:
        case MACFG_68:
        case MACFG_69:
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
        case MACFG_74:
        case MACFG_75:
        case MACFG_76:
        case MACFG_80:
        case MACFG_81:
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                DELAY(2000);
                break;
        default:
                DELAY(10000);
                break;
        }
        DELAY(200);
        CSR_WRITE_1(sc, RE_COMMAND, RE_CMD_RESET);

        for (i = 0; i < RE_TIMEOUT; i++) {
                DELAY(10);
                if (!(CSR_READ_1(sc, RE_COMMAND) & RE_CMD_RESET))
                        break;
        }

        if (i == RE_TIMEOUT)
                device_printf(sc->dev,"reset never completed!\n");

        return;
}

static u_int8_t re_link_ok(struct re_softc *sc)
{
        u_int8_t	retval;

        retval = (CSR_READ_1(sc, RE_PHY_STATUS) & RL_PHY_STATUS_LINK_STS) ? 1 : 0;

        return retval;
}

static u_int8_t
re_set_wol_linkspeed(struct re_softc *sc)
{
        u_int8_t wol_link_speed = 0xff;

        if (HW_SUPP_SERDES_PHY(sc))
                goto exit;

#ifdef ENABLE_FIBER_SUPPORT
        if (HW_FIBER_MODE_ENABLED(sc))
                goto exit;
#endif //ENABLE_FIBER_SUPPORT

        re_mdio_write(sc, 0x1F, 0x0000);

        wol_link_speed = RE_WOL_LINK_SPEED_100M_FIRST;
        if (!sc->re_dash) {
                u_int16_t anlpar;

                anlpar = sc->phy_reg_anlpar;

                if ((anlpar & ANLPAR_10_FD) || (anlpar & ANLPAR_10)) {
                        wol_link_speed = RE_WOL_LINK_SPEED_10M_FIRST;
                } else {
                        wol_link_speed = RE_WOL_LINK_SPEED_100M_FIRST;
                }
        }

        switch(sc->re_device_id) {
        case RT_DEVICEID_8127:
                re_clear_eth_ocp_phy_bit(sc, 0xA5D4, RTK_ADVERTISE_10GFULL);
        case RT_DEVICEID_8126:
                re_clear_eth_ocp_phy_bit(sc, 0xA5D4, RTK_ADVERTISE_5000FULL);
        /*	FALLTHROUGH */
        case RT_DEVICEID_8125:
        case RT_DEVICEID_3000:
        case RT_DEVICEID_8162:
                re_clear_eth_ocp_phy_bit(sc, 0xA5D4, RTK_ADVERTISE_2500FULL);
        /*	FALLTHROUGH */
        case RT_DEVICEID_8169:
        case RT_DEVICEID_8169SC:
        case RT_DEVICEID_8168:
        case RT_DEVICEID_8161:
                re_mdio_write(sc, MII_100T2CR, re_mdio_read(sc,MII_100T2CR) & ~(GTCR_ADV_1000TFDX|GTCR_ADV_1000THDX));
        /*	FALLTHROUGH */
        default:
                if (wol_link_speed == RE_WOL_LINK_SPEED_10M_FIRST)
                        re_mdio_write(sc, MII_ANAR, re_mdio_read(sc,MII_ANAR) & ~(ANAR_TX_FD | ANAR_TX));
                break;
        }

        switch(sc->re_device_id) {
        case RT_DEVICEID_8127:
        case RT_DEVICEID_8126:
        case RT_DEVICEID_8125:
        case RT_DEVICEID_3000:
        case RT_DEVICEID_8162:
        case RT_DEVICEID_8169:
        case RT_DEVICEID_8169SC:
        case RT_DEVICEID_8168:
        case RT_DEVICEID_8161:
                re_mdio_write(sc, MII_BMCR, BMCR_RESET | BMCR_AUTOEN | BMCR_STARTNEG);
                break;
        default:
                if (sc->re_type == MACFG_36)
                        re_mdio_write(sc, MII_BMCR, BMCR_RESET | BMCR_AUTOEN | BMCR_STARTNEG);
                else
                        re_mdio_write(sc, MII_BMCR, BMCR_AUTOEN | BMCR_STARTNEG);
                break;
        }

exit:
        return wol_link_speed;
}

static void
re_setwol(struct re_softc *sc)
{
        struct ifnet            *ifp;
        int                     pmc;
        uint16_t                pmstat;
        uint8_t                 v;

        RE_LOCK_ASSERT(sc);

        ifp = RE_GET_IFNET(sc);

        if ((ifp->if_capenable & IFCAP_WOL) == 0) {
                re_phy_power_down(sc->dev);
                return;
        }

        if (pci_find_cap(sc->dev, PCIY_PMG, &pmc) != 0)
                return;

        /* Enable config register write. */
        re_enable_cfg9346_write(sc);

        if (ifp->if_capenable & IFCAP_WOL_MAGIC)
                re_enable_magic_packet(sc);
        else
                re_disable_magic_packet(sc);

        v = CSR_READ_1(sc, RE_CFG5);
        v &= ~(RL_CFG5_WOL_BCAST | RL_CFG5_WOL_MCAST | RL_CFG5_WOL_UCAST |
               RL_CFG5_WOL_LANWAKE);

        if ((ifp->if_capenable & IFCAP_WOL_UCAST) != 0)
                v |= RL_CFG5_WOL_UCAST;
        if ((ifp->if_capenable & IFCAP_WOL_MCAST) != 0)
                v |= RL_CFG5_WOL_MCAST | RL_CFG5_WOL_BCAST;
        if ((ifp->if_capenable & IFCAP_WOL) != 0)
                v |= RL_CFG5_WOL_LANWAKE;
        CSR_WRITE_1(sc, RE_CFG5, v);

        /* Config register write done. */
        re_disable_cfg9346_write(sc);

        /*
         * It seems that hardware resets its link speed to 100Mbps in
         * power down mode so switching to 100Mbps in driver is not
         * needed.
         */

        /* Request PME if WOL is requested. */
        pmstat = pci_read_config(sc->dev, pmc + PCIR_POWER_STATUS, 2);
        pmstat &= ~(PCIM_PSTAT_PME | PCIM_PSTAT_PMEENABLE);
        if ((ifp->if_capenable & IFCAP_WOL) != 0)
                pmstat |= PCIM_PSTAT_PME | PCIM_PSTAT_PMEENABLE;
        pci_write_config(sc->dev, pmc + PCIR_POWER_STATUS, pmstat, 2);

        /* Put controller into sleep mode. */
        if ((ifp->if_capenable & IFCAP_WOL) != 0) {
                uint8_t wol_link_speed;
                re_set_rx_packet_filter_in_sleep_state(sc);
                wol_link_speed = re_set_wol_linkspeed(sc);
                if (sc->re_type == MACFG_21 || sc->re_type == MACFG_22)
                        CSR_WRITE_1(sc, RE_COMMAND, RE_CMD_RX_ENB);

                if (sc->RequiredPfmPatch)
                        re_set_pfm_patch(sc,
                                         (wol_link_speed == RE_WOL_LINK_SPEED_10M_FIRST) ?
                                         1 : 0);
        }
}

static void
re_clrwol(struct re_softc *sc)
{
        int                     pmc;
        uint16_t                pmstat;
        uint8_t                 v;

        RE_LOCK_ASSERT(sc);

        if (pci_find_cap(sc->dev, PCIY_PMG, &pmc) != 0)
                return;

        /* Disable PME and clear PME status. */
        pmstat = pci_read_config(sc->dev, pmc + PCIR_POWER_STATUS, 2);
        pmstat &= ~PCIM_PSTAT_PMEENABLE;
        pci_write_config(sc->dev, pmc + PCIR_POWER_STATUS, pmstat, 2);

        /* Enable config register write. */
        re_enable_cfg9346_write(sc);

        v = CSR_READ_1(sc, RE_CFG3);
        v &= ~(RL_CFG3_WOL_LINK);
        CSR_WRITE_1(sc, RE_CFG3, v);

        re_disable_magic_packet(sc);

        v = CSR_READ_1(sc, RE_CFG5);
        v &= ~(RL_CFG5_WOL_BCAST | RL_CFG5_WOL_MCAST | RL_CFG5_WOL_UCAST);
        v &= ~RL_CFG5_WOL_LANWAKE;
        CSR_WRITE_1(sc, RE_CFG5, v);

        /* Config register write done. */
        re_disable_cfg9346_write(sc);
}

/*
 * Stop the adapter and free any mbufs allocated to the
 * RX and TX lists.
 */
static void re_stop(struct re_softc *sc)  	/* Stop Driver */
{
        struct ifnet		*ifp;

        /*	RE_LOCK_ASSERT(sc);*/

        ifp = RE_GET_IFNET(sc);
#if OS_VER < VERSION(9,0)
        ifp->if_timer = 0;
#endif

        re_stop_timer(sc);

        /*
         * Disable accepting frames to put RX MAC into idle state.
         * Otherwise it's possible to get frames while stop command
         * execution is in progress and controller can DMA the frame
         * to already freed RX buffer during that period.
         */
        re_clear_all_rx_packet_filter(sc);

        switch (sc->re_type) {
        case MACFG_80:
        case MACFG_81:
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                re_disable_imr_8125(sc);
                re_set_isr_8125(sc, 0xffffffff);
                break;
        default:
                CSR_WRITE_2(sc, RE_IMR, 0x0000);
                CSR_WRITE_2(sc, RE_ISR, 0xffff);
                break;
        }

        switch (sc->re_type) {
        case MACFG_50:
        case MACFG_51:
        case MACFG_52:
                re_eri_write(sc, 0x1bc, 4, 0x0000001f, ERIAR_ExGMAC);
                re_eri_write(sc, 0x1dc, 4, 0x0000002d, ERIAR_ExGMAC);
                break;
        case MACFG_38:
                re_eri_write(sc, 0x1bc, 4, 0x0000001f, ERIAR_ExGMAC);
                re_eri_write(sc, 0x1dc, 4, 0x0000003f, ERIAR_ExGMAC);
                break;
        }

        switch (sc->re_type) {
        case MACFG_74:
        case MACFG_75:
                re_set_mac_ocp_bit(sc, 0xD438, BIT_3);
                re_set_mac_ocp_bit(sc, 0xD438, BIT_2);
                re_clear_ephy_ext_addr(sc);
                re_set_mac_ocp_bit(sc, 0xD438, (BIT_1 | BIT_0));
                break;
        }

        re_reset(sc);

        /*
         * Free the TX list buffers.
         */
        for (int i = 0; i < RL_TX_QUEUE_NUM; i++) {
                while (sc->re_desc.tx_last_index[i]!=sc->re_desc.tx_cur_index[i]) {
                        u_int32_t entry = sc->re_desc.tx_last_index[i] % RE_TX_BUF_NUM;
                        if (sc->re_desc.re_tx_mtag[i]) {
                                bus_dmamap_sync(sc->re_desc.re_tx_mtag[i],
                                                sc->re_desc.re_tx_dmamap[i][entry],
                                                BUS_DMASYNC_POSTWRITE);
                                bus_dmamap_unload(sc->re_desc.re_tx_mtag[i],
                                                  sc->re_desc.re_tx_dmamap[i][entry]);
                        }

                        if (sc->re_desc.tx_buf[i][entry]!=NULL) {
                                m_freem(sc->re_desc.tx_buf[i][entry]);
                                sc->re_desc.tx_buf[i][entry] = NULL;
                        }
                        sc->re_desc.tx_last_index[i]++;
                }
        }

        ifp->if_drv_flags &= ~(IFF_DRV_RUNNING | IFF_DRV_OACTIVE);

        return;
}

/*
 * Main transmit routine.
 */
static void re_start(struct ifnet *ifp)  	/* Transmit Packet*/
{
        struct re_softc		*sc;

        sc = ifp->if_softc;	/* Paste to ifp in function re_attach(dev) */
        RE_LOCK(sc);
        re_start_locked(ifp, default_tx_qid);
        RE_UNLOCK(sc);
}

static uint16_t
re_get_eth_type(struct mbuf *mb)
{
        struct ether_vlan_header *eh;
        uint16_t eth_type;

        eh = mtod(mb, struct ether_vlan_header *);
        if (mb->m_len < ETHER_HDR_LEN)
                return (0);
        if (eh->evl_encap_proto == htons(ETHERTYPE_VLAN))
                eth_type = ntohs(eh->evl_proto);
        else
                eth_type = ntohs(eh->evl_encap_proto);
        return (eth_type);
}

static int
re_get_l4hdr_offset(struct mbuf *mb)
{
        struct ether_vlan_header *eh;
        //struct tcphdr *th;
        struct ip *ip;
        int ip_hlen;
        struct ip6_hdr *ip6;
        uint16_t eth_type;
        int eth_hdr_len;

        eh = mtod(mb, struct ether_vlan_header *);
        if (mb->m_len < ETHER_HDR_LEN)
                return (0);
        if (eh->evl_encap_proto == htons(ETHERTYPE_VLAN)) {
                eth_type = ntohs(eh->evl_proto);
                eth_hdr_len = ETHER_HDR_LEN + ETHER_VLAN_ENCAP_LEN;
        } else {
                eth_type = ntohs(eh->evl_encap_proto);
                eth_hdr_len = ETHER_HDR_LEN;
        }
        if (mb->m_len < eth_hdr_len)
                return (0);
        switch (eth_type) {
        case ETHERTYPE_IP:
                ip = (struct ip *)(mb->m_data + eth_hdr_len);
                if (mb->m_len < eth_hdr_len + sizeof(*ip))
                        return (0);
                if (ip->ip_p != IPPROTO_TCP && ip->ip_p != IPPROTO_UDP)
                        return (0);
                ip_hlen = ip->ip_hl << 2;
                eth_hdr_len += ip_hlen;
                break;
        case ETHERTYPE_IPV6:
                ip6 = (struct ip6_hdr *)(mb->m_data + eth_hdr_len);
                if (mb->m_len < eth_hdr_len + sizeof(*ip6))
                        return (0);
                if (ip6->ip6_nxt != IPPROTO_TCP && ip6->ip6_nxt != IPPROTO_UDP)
                        return (0);
                eth_hdr_len += sizeof(*ip6);
                break;
        default:
                return (0);
        }
        /*
        if (mb->m_len < eth_hdr_len + sizeof(*th))
        return (0);
        th = (struct tcphdr *)(mb->m_data + eth_hdr_len);
        tcp_hlen = th->th_off << 2;
        eth_hdr_len += tcp_hlen;
        if (mb->m_len < eth_hdr_len)
        return (0);
        */
        return (eth_hdr_len);
}

static void re_start_locked(struct ifnet *ifp, u_int32_t qid)
{
        bus_dma_segment_t	segs[RE_NTXSEGS];
        bus_dmamap_t		map;
        struct re_softc		*sc;
        struct mbuf		*m_head;
        struct mbuf		*m_new;
        uint32_t tx_cur_index;
        uint32_t entry;
        uint32_t first_entry;
        int		queued;
        int		nsegs;
        int		error;
        int		i;

        sc = ifp->if_softc;	/* Paste to ifp in function re_attach(dev) */

        /*	RE_LOCK_ASSERT(sc);*/

        if ((sc->driver_detach == 1) || (sc->rx_fifo_overflow != 0))
                return;

        tx_cur_index = sc->re_desc.tx_cur_index[qid];
        for (queued = 0; !IFQ_DRV_IS_EMPTY(&ifp->if_snd);) {
                int fs = 1, ls = 0;
                uint32_t  opts1;
                uint32_t  opts2;
                IFQ_DRV_DEQUEUE(&ifp->if_snd, m_head);	/* Remove(get) data from system transmit queue */
                if (m_head == NULL) {
                        break;
                }

                if ((sc->re_type == MACFG_80 || sc->re_type == MACFG_81 ||
                    sc->re_type == MACFG_82 || sc->re_type == MACFG_83) &&
                    sc->re_device_id != RT_DEVICEID_3000) {
                        if (re_8125_pad(sc, m_head) != 0) {
                                IFQ_DRV_PREPEND(&ifp->if_snd, m_head);
                                ifp->if_drv_flags |= IFF_DRV_OACTIVE;
                                break;
                        }
                }

                entry = tx_cur_index % RE_TX_BUF_NUM;
                if (sc->re_coalesce_tx_pkt) {
                        if (re_encap(sc, &m_head)) {
                                IFQ_DRV_PREPEND(&ifp->if_snd, m_head);
                                ifp->if_drv_flags |= IFF_DRV_OACTIVE;
                                break;
                        }
                }

                error = bus_dmamap_load_mbuf_sg(sc->re_desc.re_tx_mtag[qid], sc->re_desc.re_tx_dmamap[qid][entry],
                                                m_head, segs, &nsegs, BUS_DMA_NOWAIT);
                if (error == EFBIG) {
                        m_new = m_collapse(m_head, M_NOWAIT, RE_NTXSEGS);
                        if (m_new == NULL) {
                                m_freem(m_head);
                                m_head = NULL;
                                //return (ENOBUFS);
                                break;
                        }
                        m_head = m_new;
                        error = bus_dmamap_load_mbuf_sg(sc->re_desc.re_tx_mtag[qid],
                                                        sc->re_desc.re_tx_dmamap[qid][entry], m_head, segs, &nsegs, BUS_DMA_NOWAIT);
                        if (error != 0) {
                                m_freem(m_head);
                                m_head = NULL;
                                //return (error);
                                break;
                        }
                } else if (error != 0) {
                        //return (error);
                        IFQ_DRV_PREPEND(&ifp->if_snd, m_head);
                        ifp->if_drv_flags |= IFF_DRV_OACTIVE;
                        break;
                }
                if (nsegs == 0) {
                        m_freem(m_head);
                        m_head = NULL;
                        //return (EIO);
                        break;
                }

                /* Check for number of available descriptors. */
                if (CountFreeTxDescNum(&sc->re_desc, qid) < nsegs) {	/* No enough descriptor */
                        bus_dmamap_unload(sc->re_desc.re_tx_mtag[qid], sc->re_desc.re_tx_dmamap[qid][entry]);
                        IFQ_DRV_PREPEND(&ifp->if_snd, m_head);
                        ifp->if_drv_flags |= IFF_DRV_OACTIVE;
                        break;
                }

                bus_dmamap_sync(sc->re_desc.re_tx_mtag[qid], sc->re_desc.re_tx_dmamap[qid][entry],
                                BUS_DMASYNC_PREWRITE);

                first_entry = entry;

                if (ifp->if_bpf) {		/* If there's a BPF listener, bounce a copy of this frame to him. */
                        //printf("If there's a BPF listener, bounce a copy of this frame to him. \n");

                        /*#if OS_VER < VERSION(5, 1)*/
#if OS_VER < VERSION(4,9)
                        bpf_mtap(ifp, m_head);
#else
                        bpf_mtap(ifp->if_bpf, m_head);
#endif
                }

                opts1 = opts2 = 0;
                //hw tso
                if ((m_head->m_pkthdr.csum_flags & CSUM_TSO) != 0) {
                        if ((sc->re_if_flags & RL_FLAG_DESCV2) == 0) {
                                opts2 |= RL_TDESC_CMD_LGSEND |
                                         ((uint32_t)m_head->m_pkthdr.tso_segsz <<
                                          RL_TDESC_CMD_MSSVAL_SHIFT);
                        } else {
                                /*
                                 * RTL8168C/RTL816CP/RTL8111C/RTL8111CP
                                 */
                                const uint16_t l4hoffset =
                                        re_get_l4hdr_offset(m_head);

                                if (re_get_eth_type(m_head) == ETHERTYPE_IPV6)
                                        opts1 |= RL_TDESC_CMD_GTSENDV6;
                                else
                                        opts1 |= RL_TDESC_CMD_GTSENDV4;
                                opts1 |= (l4hoffset << RL_TDESC_CMD_GTSEND_TCPHO_SHIFT);
                                opts2 |= ((uint32_t)m_head->m_pkthdr.tso_segsz <<
                                          RL_TDESC_CMD_MSSVALV2_SHIFT);
                        }
                        goto process_vlan;
                }
                //hw checksum
                if ((m_head->m_pkthdr.csum_flags & RE_CSUM_FEATURES) != 0) {
                        if ((sc->re_if_flags & RL_FLAG_DESCV2) == 0) {
                                opts1 |= RL_IPV4CS1;
                                if (m_head->m_pkthdr.csum_flags & CSUM_TCP)
                                        opts1 |= RL_TCPCS1;
                                if (m_head->m_pkthdr.csum_flags & CSUM_UDP)
                                        opts1 |= RL_UDPCS1;
                        } else {
                                /*
                                 * RTL8168C/RTL816CP/RTL8111C/RTL8111CP
                                 */
                                if (re_get_eth_type(m_head) == ETHERTYPE_IP)
                                        opts2 |= RL_IPV4CS;
                                if (m_head->m_pkthdr.csum_flags &
                                    (CSUM_TCP | CSUM_TCP_IPV6))
                                        opts2 |= RL_TCPCS;
                                else if (m_head->m_pkthdr.csum_flags &
                                         (CSUM_UDP | CSUM_UDP_IPV6))
                                        opts2 |= RL_UDPCS;
                                if (m_head->m_pkthdr.csum_flags &
                                    (CSUM_TCP_IPV6 | CSUM_UDP_IPV6)) {
                                        const uint16_t l4hoffset =
                                                re_get_l4hdr_offset(m_head);

                                        opts2 |= RL_CS_V6F |
                                                 (l4hoffset << RL_TDESC_CMD_CSUM_TCPHO_SHIFT);
                                }
                        }
                        goto process_vlan;
                }
process_vlan:
                //vlan
                if (m_head->m_flags & M_VLANTAG)
                        opts2 |= bswap16(m_head->m_pkthdr.ether_vtag) | RL_TDESC_VLANCTL_TAG;
#ifdef _DEBUG_
                printf("PktLen=%d \n", m_head->m_pkthdr.len);
#endif
                for (i = 0; i < nsegs; i++) {
                        if (segs[i].ds_len == 0)
                                continue;

                        if (i == (nsegs - 1)) {
                                ls=1;

                                /*
                                * Insure that the map for this transmission
                                * is placed at the array index of the last descriptor
                                * in this chain.  (Swap last and first dmamaps.)
                                */
                                map = sc->re_desc.re_tx_dmamap[qid][first_entry];
                                sc->re_desc.re_tx_dmamap[qid][first_entry] = sc->re_desc.re_tx_dmamap[qid][entry];
                                sc->re_desc.re_tx_dmamap[qid][entry] = map;
                                sc->re_desc.tx_buf[qid][entry] = m_head;
                        } else
                                sc->re_desc.tx_buf[qid][entry] = NULL;

                        WritePacket(sc,&segs[i],fs,ls,opts2,opts1, entry, default_tx_qid);

                        fs=0;

                        tx_cur_index++;
                        entry = tx_cur_index % RE_TX_BUF_NUM;
                }
                sc->re_desc.tx_cur_index[qid] = tx_cur_index;
#ifdef _DEBUG_
                printf("\n");
#endif
                queued++;
        }

        sc->re_desc.tx_cur_index[qid] = tx_cur_index;

        if (queued == 0)
                return;

        re_start_tx(sc);

#if OS_VER < VERSION(9,0)
        ifp->if_timer = 5;
#endif

        return;
}

static void _re_start_tx(struct re_softc	*sc)
{
        switch (sc->re_type) {
        case MACFG_80:
        case MACFG_81:
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
                CSR_WRITE_2(sc, RE_TPPOLL_8125, (RE_NPQ_8125 << default_tx_qid));
                break;
        case MACFG_100:
        case MACFG_101:
                CSR_WRITE_4(sc, RE_TPPOLL_8125, (RE_NPQ_8125 << default_tx_qid));
                break;
        default:
                CSR_WRITE_1(sc, RE_TPPOLL, RE_NPQ);
                break;
        }
}

static void re_start_tx(struct re_softc	*sc)
{
        for (int i=0; i<RL_TX_QUEUE_NUM; i++) {
                bus_dmamap_sync(sc->re_desc.tx_desc_tag[i],
                                sc->re_desc.tx_desc_dmamap[i],
                                BUS_DMASYNC_PREREAD|BUS_DMASYNC_PREWRITE);
        }

        _re_start_tx(sc);
}

/*
 * Encapsulate an mbuf chain in a descriptor by coupling the mbuf data
 * pointers to the fragment pointers.
 */
static int re_encap(struct re_softc *sc,struct mbuf **m_head)
{
        struct mbuf		*m_new = NULL;

        m_new = m_defrag(*m_head, M_DONTWAIT);

        if (m_new == NULL) {
                printf("re%d: no memory for tx list", sc->re_unit);
                return (1);
        }

        /* Pad frames to at least 60 bytes. */
        if (m_new->m_pkthdr.len < RE_MIN_FRAMELEN) {	/* Case length < 60 bytes */
                /*
                 * Make security concious people happy: zero out the
                 * bytes in the pad area, since we don't know what
                 * this mbuf cluster buffer's previous user might
                 * have left in it.
                 */
                bzero(mtod(m_new, char *) + m_new->m_pkthdr.len,
                      RE_MIN_FRAMELEN - m_new->m_pkthdr.len);
                m_new->m_pkthdr.len = RE_MIN_FRAMELEN;
                m_new->m_len = m_new->m_pkthdr.len;
        }

        *m_head = m_new;

        return(0);
}

#define MIN_IPV4_PATCH_PKT_LEN (121)
#define MIN_IPV6_PATCH_PKT_LEN (147)
static int re_8125_pad(struct re_softc *sc,struct mbuf *m_head)
{
        uint32_t min_pkt_len;
        uint16_t ether_type;

        if ((m_head->m_pkthdr.csum_flags & (CSUM_TCP | CSUM_UDP)) != 0)
                goto out;

        ether_type = re_get_eth_type(m_head);
        min_pkt_len = RE_MIN_FRAMELEN;
        if (ether_type == ETHERTYPE_IP) {
                struct ip *ip = (struct ip *)mtodo(m_head, ETHER_HDR_LEN);
                if (ip->ip_p == IPPROTO_UDP)
                        min_pkt_len = MIN_IPV4_PATCH_PKT_LEN;
        } else if (ether_type == ETHERTYPE_IPV6) {
                struct ip6_hdr *ip6 = (struct ip6_hdr *)mtodo(m_head, ETHER_HDR_LEN);
                if (ip6->ip6_nxt == IPPROTO_UDP)
                        min_pkt_len = MIN_IPV6_PATCH_PKT_LEN;
        }

        if (m_head->m_pkthdr.len < min_pkt_len) {
                static const uint8_t pad[MIN_IPV4_PATCH_PKT_LEN];
                uint16_t pad_len = min_pkt_len - m_head->m_pkthdr.len;
                if (!m_append(m_head, pad_len, pad))
                        return (1);

                if (ether_type == ETHERTYPE_IP &&
                    m_head->m_pkthdr.csum_flags & CSUM_IP) {
                        struct ip *ip;
                        m_head->m_data += ETHER_HDR_LEN;
                        ip = mtod(m_head, struct ip *);
                        ip->ip_sum = in_cksum(m_head, ip->ip_hl << 2);
                        m_head->m_data -= ETHER_HDR_LEN;
                        m_head->m_pkthdr.csum_flags &= ~CSUM_IP;
                }
        }

out:
        return(0);
}

static void WritePacket(struct re_softc	*sc, bus_dma_segment_t *segs, int fs, int ls, uint32_t opts2, uint32_t opts1, uint32_t entry, uint32_t qid)
{
        union TxDesc *txptr;
        uint32_t status;

        txptr = &(sc->re_desc.tx_desc[qid][entry]);

        status = RL_TDESC_CMD_OWN | opts1 | segs->ds_len;

        if (fs)
                status |= RL_TDESC_CMD_SOF;
        if (ls)
                status |= RL_TDESC_CMD_EOF;
        if (entry == (RE_TX_BUF_NUM - 1))
                status |= RL_TDESC_CMD_EOR;

        txptr->so1.TxBuff = htole64(segs->ds_addr);

        txptr->ul[1] = htole32(opts2);
        /* make sure opts2 is set before opts1 */
        wmb();
        txptr->ul[0] = htole32(status);
}

static uint32_t CountFreeTxDescNum(struct re_descriptor *desc, u_int32_t qid)
{
        uint32_t ret=desc->tx_last_index[qid] + RE_TX_BUF_NUM - desc->tx_cur_index[qid];

        return ret;
}

/*
static int CountMbufNum(struct mbuf *m_head)
{
        int ret=0;
        struct mbuf *ptr = m_head;

        while (ptr!=NULL) {
                if (ptr->m_len >0)
                        ret++;
                ptr=ptr->m_next;
        }

        return ret;
}
*/

#ifdef RE_FIXUP_RX
static __inline void re_fixup_rx(struct mbuf *m)
{
        int                     i;
        uint16_t                *src, *dst;

        src = mtod(m, uint16_t *);
        dst = src - (RE_ETHER_ALIGN - ETHER_ALIGN) / sizeof *src;

        for (i = 0; i < (m->m_len / sizeof(uint16_t) + 1); i++)
                *dst++ = *src++;

        m->m_data -= RE_ETHER_ALIGN - ETHER_ALIGN;
}
#endif

/*
 * A frame was downloaded to the chip. It's safe for us to clean up
 * the list buffers.
 */
static void re_txeof(struct re_softc *sc, u_int32_t qid)  	/* Transmit OK/ERR handler */
{
        union TxDesc *txptr;
        struct ifnet		*ifp;
        u_int32_t           txstat;
        u_int32_t           entry;
        u_int32_t           tx_cur_index;
        u_int32_t           tx_last_index;

        /*	printf("X");*/

        ifp = RE_GET_IFNET(sc);

#if OS_VER < VERSION(9,0)
        /* Clear the timeout timer. */
        ifp->if_timer = 0;
#endif
        tx_cur_index = sc->re_desc.tx_cur_index[qid];
        tx_last_index = sc->re_desc.tx_last_index[qid];
        /* No packet to complete. */
        if (tx_cur_index == tx_last_index)
                return;

        bus_dmamap_sync(sc->re_desc.tx_desc_tag[qid],
                        sc->re_desc.tx_desc_dmamap[qid],
                        BUS_DMASYNC_POSTREAD|BUS_DMASYNC_POSTWRITE);

        while (tx_cur_index != tx_last_index) {
                entry = tx_last_index % RE_TX_BUF_NUM;
                txptr=&(sc->re_desc.tx_desc[qid][entry]);
                txstat = le32toh(txptr->ul[0]);
                if (txstat & RL_TDESC_STAT_OWN)
                        break;
#ifdef _DEBUG_
                printf("**** Tx OK  ****\n");
#endif
                if (sc->re_desc.tx_buf[entry]!=NULL) {
                        bus_dmamap_sync(sc->re_desc.re_tx_mtag[qid],
                                        sc->re_desc.re_tx_dmamap[qid][entry],
                                        BUS_DMASYNC_POSTWRITE);
                        bus_dmamap_unload(sc->re_desc.re_tx_mtag[qid],
                                          sc->re_desc.re_tx_dmamap[qid][entry]);
                        /* Free Current MBuf in a Mbuf list*/
                        m_freem(sc->re_desc.tx_buf[qid][entry]);
                        sc->re_desc.tx_buf[qid][entry] = NULL;
                }

                tx_last_index++;
        }

        if (sc->re_desc.tx_last_index[qid] != tx_last_index) {
                sc->re_desc.tx_last_index[qid] = tx_last_index;
                ifp->if_drv_flags &= ~IFF_DRV_OACTIVE;
        }

        /* prevent tx stop. */
        if (tx_cur_index != tx_last_index)
                _re_start_tx(sc);

        return;
}

#if defined(INET) || defined(INET6)
static int
re_lro_rx(struct re_softc *sc, struct mbuf *m)
{
        struct lro_ctrl *lro;

        lro = &sc->re_lro;

        if (lro->lro_mbuf_max != 0) {
                tcp_lro_queue_mbuf(lro, m);
                return (0);
        }

        return (tcp_lro_rx(lro, m, 0));
}
#endif

static void
re_rxq_input(struct re_softc *sc, struct mbuf *m, bool lro_able)
{
        struct ifnet		*ifp;

        ifp = RE_GET_IFNET(sc);

#if defined(INET) || defined(INET6)
        if ((ifp->if_capenable & IFCAP_LRO) && lro_able) {
                if (re_lro_rx(sc, m) == 0)
                        return;
        }
#endif

        /*#if OS_VER < VERSION(5, 1)*/
#if OS_VER < VERSION(4,9)
        /* Remove header from mbuf and pass it on. */
        m_adj(m, sizeof(struct ether_header));
        ether_input(ifp, eh, m);
#else
        (*ifp->if_input)(ifp, m);
#endif
}

static void
re_drain_soft_lro(struct re_softc *sc)
{
#if defined(INET) || defined(INET6)
#if OS_VER >= VERSION(11,0)
        tcp_lro_flush_all(&sc->re_lro);
#else
        struct lro_entry *queued;

        while ((!SLIST_EMPTY(&lro->lro_active))) {
                queued = SLIST_FIRST(&lro->lro_active);
                SLIST_REMOVE_HEAD(&lro->lro_active, next);
                tcp_lro_flush(lro, queued);
        }
#endif //OS_VER >= VERSION(11,0)
#endif //defined(INET) || defined(INET6)
}

/*
 * A frame has been uploaded: pass the resulting mbuf chain up to
 * the higher level protocols.
 *
 * You know there's something wrong with a PCI bus-master chip design
 * when you have to use m_devget().
 *
 * The receive operation is badly documented in the datasheet, so I'll
 * attempt to document it here. The driver provides a buffer area and
 * places its base address in the RX buffer start address register.
 * The chip then begins copying frames into the RX buffer. Each frame
 * is preceeded by a 32-bit RX status word which specifies the length
 * of the frame and certain other status bits. Each frame (starting with
 * the status word) is also 32-bit aligned. The frame length is in the
 * first 16 bits of the status word; the lower 15 bits correspond with
 * the 'rx status register' mentioned in the datasheet.
 *
 * Note: to make the Alpha happy, the frame payload needs to be aligned
 * on a 32-bit boundary. To achieve this, we cheat a bit by copying from
 * the ring buffer starting at an address two bytes before the actual
 * data location. We can then shave off the first two bytes using m_adj().
 * The reason we do this is because m_devget() doesn't let us specify an
 * offset into the mbuf storage space, so we have to artificially create
 * one. The ring is allocated in such a way that there are a few unused
 * bytes of space preceecing it so that it will be safe for us to do the
 * 2-byte backstep even if reading from the ring at offset 0.
 */
static int _re_rxeof(struct re_softc *sc, u_int32_t qid)  	/* Receive Data OK/ERR handler */
{
        struct mbuf		*m;
        struct ifnet		*ifp;
        union RxDesc *rxptr;
        struct mbuf *buf;
        int size;
        int maxpkt = RE_RX_BUDGET;
        u_int32_t entry;
        u_int32_t pkt_size;
        u_int32_t rx_cur_index;
        u_int32_t opts2,opts1,status;
        bool lro_able;

        /* RE_LOCK_ASSERT(sc);*/

        ifp = RE_GET_IFNET(sc);

        bus_dmamap_sync(sc->re_desc.rx_desc_tag[qid],
                        sc->re_desc.rx_desc_dmamap[qid],
                        BUS_DMASYNC_POSTREAD|BUS_DMASYNC_POSTWRITE);

        rx_cur_index = sc->re_desc.rx_cur_index[qid];
        while (maxpkt > 0) {
                entry = rx_cur_index % RE_RX_BUF_NUM;
                rxptr=&(sc->re_desc.rx_desc[qid][entry]);
                opts1 = le32toh(rxptr->ul[0]);
                lro_able = false;

                /* Check Receive OK */
                if (opts1&RL_RDESC_STAT_OWN) {
                        if (maxpkt != RE_RX_BUDGET)
                                break;

                        if (sc->HwSuppIsrVer > 0)
                                CSR_READ_1(sc, RE_IMR0_8125);
                        else
                                CSR_READ_1(sc, RE_IMR);

                        opts1 = le32toh(rxptr->ul[0]);
                        if (opts1&RL_RDESC_STAT_OWN)
                                break;
                }

                /* Check if this packet is received correctly*/
                if (opts1&RL_RDESC_RES)	/*Check RES bit*/
                        goto drop_packet;

                pkt_size = (opts1&RL_RDESC_STAT_GFRAGLEN)-ETHER_CRC_LEN;
                if (pkt_size > sc->re_rx_mbuf_sz)
                        goto drop_packet;

                /* Wait other fields is dmaed */
                rmb();

                opts2 = le32toh(rxptr->ul[1]);

                //buf = m_getcl(M_DONTWAIT, MT_DATA, M_PKTHDR); /* Alloc a new mbuf */

                if (sc->re_rx_mbuf_sz <= MCLBYTES)
                        size = MCLBYTES;
                else if (sc->re_rx_mbuf_sz <= MJUMPAGESIZE)
                        size = MJUMPAGESIZE;
                else
                        size = MJUM9BYTES;

                buf = m_getjcl(M_DONTWAIT, MT_DATA, M_PKTHDR, size);
                if (buf==NULL)
                        goto drop_packet;

                buf->m_len = buf->m_pkthdr.len = size;
#ifdef RE_FIXUP_RX
                /*
                 * This is part of an evil trick to deal with non-x86 platforms.
                 * The RealTek chip requires RX buffers to be aligned on 64-bit
                 * boundaries, but that will hose non-x86 machines. To get around
                 * this, we leave some empty space at the start of each buffer
                 * and for non-x86 hosts, we copy the buffer back six bytes
                 * to achieve word alignment. This is slightly more efficient
                 * than allocating a new buffer, copying the contents, and
                 * discarding the old buffer.
                 */
                m_adj(buf, RE_ETHER_ALIGN);
#endif

                bus_dmamap_sync(sc->re_desc.re_rx_mtag[qid],
                                sc->re_desc.re_rx_dmamap[qid][entry],
                                BUS_DMASYNC_POSTREAD);
                bus_dmamap_unload(sc->re_desc.re_rx_mtag[qid],
                                  sc->re_desc.re_rx_dmamap[qid][entry]);

                m = sc->re_desc.rx_buf[qid][entry];
                sc->re_desc.rx_buf[qid][entry] = buf;
                m->m_pkthdr.len = m->m_len = pkt_size;
                m->m_pkthdr.rcvif = ifp;

#ifdef RE_FIXUP_RX
                re_fixup_rx(m);
#endif

                //vlan
                if (opts2 & RL_RDESC_VLANCTL_TAG) {
                        m->m_pkthdr.ether_vtag =
                                bswap16((opts2 & RL_RDESC_VLANCTL_DATA));
                        m->m_flags |= M_VLANTAG;
                }
                if (ifp->if_capenable & IFCAP_RXCSUM) {
                        if ((sc->re_if_flags & RL_FLAG_DESCV2) == 0) {
                                u_int32_t proto = opts1 & RL_ProtoMASK;

                                if (proto != 0)
                                        m->m_pkthdr.csum_flags |=  CSUM_IP_CHECKED;
                                if (!(opts1 & RL_IPF))
                                        m->m_pkthdr.csum_flags |= CSUM_IP_VALID;
                                if (((proto == RL_ProtoTCP) && !(opts1 & RL_TCPF)) ||
                                    ((proto == RL_ProtoUDP) && !(opts1 & RL_UDPF))) {
                                        m->m_pkthdr.csum_flags |= CSUM_DATA_VALID|CSUM_PSEUDO_HDR;
                                        m->m_pkthdr.csum_data = 0xffff;
                                        if (proto == RL_ProtoTCP)
                                                lro_able = true;
                                }
                        } else {
                                /*
                                 * RTL8168C/RTL816CP/RTL8111C/RTL8111CP
                                 */
                                if (opts2 & RL_V4F)
                                        m->m_pkthdr.csum_flags |= CSUM_IP_CHECKED;
                                if ((opts2 & RL_V4F) && !(opts1 & RL_IPF))
                                        m->m_pkthdr.csum_flags |= CSUM_IP_VALID;
                                if (((opts1 & RL_TCPT) && !(opts1 & RL_TCPF)) ||
                                    ((opts1 & RL_UDPT) && !(opts1 & RL_UDPF))) {
                                        m->m_pkthdr.csum_flags |= CSUM_DATA_VALID|CSUM_PSEUDO_HDR;
                                        m->m_pkthdr.csum_data = 0xffff;
                                        if (opts1 & RL_TCPT)
                                                lro_able = true;
                                }
                        }
                }

#if OS_VER < VERSION(11,0)
                ifp->if_ipackets++;
#else
                if_inc_counter(ifp, IFCOUNTER_IPACKETS, 1);
#endif
#ifdef _DEBUG_
                printf("Rcv Packet, Len=%d \n", m->m_len);
#endif

                RE_UNLOCK(sc);

                re_rxq_input(sc, m, lro_able);

                RE_LOCK(sc);

                /* Load the map for rx buffer. */
                bus_dmamap_load(sc->re_desc.re_rx_mtag[qid],
                                sc->re_desc.re_rx_dmamap[qid][entry],
                                sc->re_desc.rx_buf[qid][entry]->m_data,
                                sc->re_rx_desc_buf_sz,
                                re_rx_dma_map_buf, rxptr,
                                0);
                bus_dmamap_sync(sc->re_desc.re_rx_mtag[qid],
                                sc->re_desc.re_rx_dmamap[qid][entry],
                                BUS_DMASYNC_PREREAD);
update_desc:
                status = RL_RDESC_CMD_OWN | sc->re_rx_desc_buf_sz;
                if (entry == (RE_RX_BUF_NUM - 1))
                        status |= RL_RDESC_CMD_EOR;

                rxptr->ul[1]=0;
                /* make sure desc is all set before releasing it */
                wmb();
                rxptr->ul[0] = htole32(status);

                rx_cur_index++;

                maxpkt--;

                continue;
drop_packet:
#if OS_VER < VERSION(11,0)
                ifp->if_ierrors++;
#else
                if_inc_counter(ifp, IFCOUNTER_IERRORS, 1);
#endif
                goto update_desc;
        }

        if (sc->re_desc.rx_cur_index[qid] != rx_cur_index) {
                if (ifp->if_capenable & IFCAP_LRO) {
                        RE_UNLOCK(sc);
                        re_drain_soft_lro(sc);
                        RE_LOCK(sc);
                }
                sc->re_desc.rx_cur_index[qid] = rx_cur_index;
                bus_dmamap_sync(sc->re_desc.rx_desc_tag[qid],
                                sc->re_desc.rx_desc_dmamap[qid],
                                BUS_DMASYNC_PREREAD|BUS_DMASYNC_PREWRITE);
        }

        return (RE_RX_BUDGET - maxpkt);
}

static int re_rxeof(struct re_softc *sc)  	/* Receive Data OK/ERR handler */
{
        int done = 0;

        for (int i = 0; i < RL_RX_QUEUE_NUM; i++)
                done += _re_rxeof(sc, i);

        return done;
}

#if OS_VER < VERSION(7,0)
static void re_intr(void *arg)  	/* Interrupt Handler */
#else
static int re_intr(void *arg)  	/* Interrupt Handler */
#endif //OS_VER < VERSION(7,0)
{
        struct re_softc		*sc;

        sc = arg;

        if ((sc->re_if_flags & (RL_FLAG_MSI | RL_FLAG_MSIX)) == 0) {
                if ((CSR_READ_2(sc, RE_ISR) & RE_INTRS) == 0) {
#if OS_VER < VERSION(7,0)
                        return;
#else
                        return (FILTER_STRAY);
#endif
                }
        }

        /* Disable interrupts. */
        CSR_WRITE_2(sc, RE_IMR, 0x0000);

#if OS_VER < VERSION(7,0)
        re_int_task(arg, 0);
#else //OS_VER < VERSION(7,0)
#if OS_VER < VERSION(11,0)
        taskqueue_enqueue_fast(sc->re_tq, &sc->re_inttask);
#else ////OS_VER < VERSION(11,0)
        taskqueue_enqueue(sc->re_tq, &sc->re_inttask);
#endif //OS_VER < VERSION(11,0)
        return (FILTER_HANDLED);
#endif //OS_VER < VERSION(7,0)
}

#if OS_VER < VERSION(7,0)
static void re_intr_8125(void *arg)  	/* Interrupt Handler */
#else
static int re_intr_8125(void *arg)  	/* Interrupt Handler */
#endif //OS_VER < VERSION(7,0)
{
        struct re_softc		*sc;

        sc = arg;

        if ((sc->re_if_flags & (RL_FLAG_MSI | RL_FLAG_MSIX)) == 0) {
                if ((re_get_isr_8125(sc) & re_get_default_imr_8125(sc)) == 0) {
#if OS_VER < VERSION(7,0)
                        return;
#else
                        return (FILTER_STRAY);
#endif
                }
        }

        /* Disable interrupts. */
        re_disable_imr_8125(sc);

#if OS_VER < VERSION(7,0)
        re_int_task_8125(arg, 0);
#else //OS_VER < VERSION(7,0)
#if OS_VER < VERSION(11,0)
        taskqueue_enqueue_fast(sc->re_tq, &sc->re_inttask);
#else ////OS_VER < VERSION(11,0)
        taskqueue_enqueue(sc->re_tq, &sc->re_inttask);
#endif //OS_VER < VERSION(11,0)
        return (FILTER_HANDLED);
#endif //OS_VER < VERSION(7,0)
}

static void re_int_task_poll(void *arg, int npending)
{
        struct re_softc		*sc;
        struct ifnet		*ifp;
        int 				done;

        sc = arg;

        RE_LOCK(sc);

        ifp = RE_GET_IFNET(sc);

        if (sc->suspended ||
            (ifp->if_drv_flags & IFF_DRV_RUNNING) == 0) {
                RE_UNLOCK(sc);
                return;
        }

        done = re_rxeof(sc);

        re_txeof(sc, default_tx_qid);

        if (!IFQ_DRV_IS_EMPTY(&ifp->if_snd))
                re_start_locked(ifp, default_tx_qid);

        RE_UNLOCK(sc);

#if OS_VER>=VERSION(7,0)
        if (done >= RE_RX_BUDGET) {
#if OS_VER < VERSION(11,0)
                taskqueue_enqueue_fast(sc->re_tq, &sc->re_inttask_poll);
#else ////OS_VER < VERSION(11,0)
                taskqueue_enqueue(sc->re_tq, &sc->re_inttask_poll);
#endif //OS_VER < VERSION(11,0)
                return;
        }
#endif //OS_VER>=VERSION(7,0)

        /* Re-enable interrupts. */
        CSR_WRITE_2(sc, RE_IMR, RE_INTRS);
}

static void re_int_task(void *arg, int npending)
{
        struct re_softc		*sc;
        struct ifnet		*ifp;
        int 				done;
        u_int32_t			status;

        sc = arg;

        RE_LOCK(sc);

        ifp = RE_GET_IFNET(sc);

        status = CSR_READ_2(sc, RE_ISR);

        if (status) {
                CSR_WRITE_2(sc, RE_ISR, status & ~RE_ISR_FIFO_OFLOW);
        }

        if (sc->suspended ||
            (ifp->if_drv_flags & IFF_DRV_RUNNING) == 0) {
                RE_UNLOCK(sc);
                return;
        }

        done = re_rxeof(sc);

        if (sc->re_type == MACFG_21) {
                if (status & RE_ISR_FIFO_OFLOW) {
                        sc->rx_fifo_overflow = 1;
                        CSR_WRITE_2(sc, RE_IntrMitigate, 0x0000);
                        CSR_WRITE_4(sc, RE_TIMERCNT, 0x4000);
                        CSR_WRITE_4(sc, RE_TIMERINT, 0x4000);
                } else {
                        sc->rx_fifo_overflow = 0;
                        CSR_WRITE_4(sc,RE_CPlusCmd, 0x51512082);
                }

                if (status & RE_ISR_PCS_TIMEOUT) {
                        if ((status & RE_ISR_FIFO_OFLOW) &&
                            (!(status & (RE_ISR_RX_OK | RE_ISR_TX_OK | RE_ISR_RX_OVERRUN)))) {
                                re_reset(sc);
                                re_init_locked(sc);
                                sc->rx_fifo_overflow = 0;
                                CSR_WRITE_2(sc, RE_ISR, RE_ISR_FIFO_OFLOW);
                        }
                }
        }

        re_txeof(sc, default_tx_qid);

        if (status & RE_ISR_SYSTEM_ERR) {
                re_reset(sc);
                re_init_locked(sc);
        }

        if (!IFQ_DRV_IS_EMPTY(&ifp->if_snd))
                re_start_locked(ifp, default_tx_qid);

        RE_UNLOCK(sc);

#if OS_VER>=VERSION(7,0)
        if (done >= RE_RX_BUDGET) {
#if OS_VER < VERSION(11,0)
                taskqueue_enqueue_fast(sc->re_tq, &sc->re_inttask_poll);
#else ////OS_VER < VERSION(11,0)
                taskqueue_enqueue(sc->re_tq, &sc->re_inttask_poll);
#endif //OS_VER < VERSION(11,0)
                return;
        }
#endif //OS_VER>=VERSION(7,0)

        /* Re-enable interrupts. */
        CSR_WRITE_2(sc, RE_IMR, RE_INTRS);
}

static void re_int_task_8125_poll(void *arg, int npending)
{
        struct re_softc		*sc;
        struct ifnet		*ifp;
        int 				done;

        sc = arg;

        RE_LOCK(sc);

        ifp = RE_GET_IFNET(sc);

        if (sc->suspended ||
            (ifp->if_drv_flags & IFF_DRV_RUNNING) == 0) {
                RE_UNLOCK(sc);
                return;
        }

        done = re_rxeof(sc);

        re_txeof(sc, default_tx_qid);

        if (!IFQ_DRV_IS_EMPTY(&ifp->if_snd))
                re_start_locked(ifp, default_tx_qid);

        RE_UNLOCK(sc);

#if OS_VER>=VERSION(7,0)
        if (done >= RE_RX_BUDGET) {
#if OS_VER < VERSION(11,0)
                taskqueue_enqueue_fast(sc->re_tq, &sc->re_inttask_poll);
#else ////OS_VER < VERSION(11,0)
                taskqueue_enqueue(sc->re_tq, &sc->re_inttask_poll);
#endif //OS_VER < VERSION(11,0)
                return;
        }
#endif //OS_VER>=VERSION(7,0)

        /* Re-enable interrupts. */
        re_enable_imr_8125(sc);
}

static void re_int_task_8125(void *arg, int npending)
{
        struct re_softc		*sc;
        struct ifnet		*ifp;
        int 				done;
        u_int32_t			status;

        sc = arg;

        RE_LOCK(sc);

        ifp = RE_GET_IFNET(sc);

        status = re_get_isr_8125(sc);
        if (!sc->use_new_intr_mapping)
                status &= ~RE_ISR_FIFO_OFLOW;
        re_set_isr_8125(sc, status);

        if (sc->suspended ||
            (ifp->if_drv_flags & IFF_DRV_RUNNING) == 0) {
                RE_UNLOCK(sc);
                return;
        }

        done = re_rxeof(sc);

        re_txeof(sc, default_tx_qid);

        if (!sc->use_new_intr_mapping) {
                if (status & RE_ISR_SYSTEM_ERR) {
                        re_reset(sc);
                        re_init_locked(sc);
                }
        }

        if (!IFQ_DRV_IS_EMPTY(&ifp->if_snd))
                re_start_locked(ifp, default_tx_qid);

        RE_UNLOCK(sc);

#if OS_VER>=VERSION(7,0)
        if (done >= RE_RX_BUDGET) {
#if OS_VER < VERSION(11,0)
                taskqueue_enqueue_fast(sc->re_tq, &sc->re_inttask_poll);
#else ////OS_VER < VERSION(11,0)
                taskqueue_enqueue(sc->re_tq, &sc->re_inttask_poll);
#endif //OS_VER < VERSION(11,0)
                return;
        }
#endif //OS_VER>=VERSION(7,0)

        /* Re-enable interrupts. */
        re_enable_imr_8125(sc);
}

static void re_set_multicast_reg(struct re_softc *sc, u_int32_t mask0,
                                 u_int32_t mask4)
{
        u_int8_t  enable_cfg_reg_write = 0;

        if (sc->re_type == MACFG_5 || sc->re_type == MACFG_6)
                enable_cfg_reg_write = 1;

        if (enable_cfg_reg_write)
                re_enable_cfg9346_write(sc);
        CSR_WRITE_4(sc, RE_MAR0, mask0);
        CSR_WRITE_4(sc, RE_MAR4, mask4);
        if (enable_cfg_reg_write)
                re_disable_cfg9346_write(sc);

        return;
}

static void re_clear_all_rx_packet_filter(struct re_softc *sc)
{
        CSR_WRITE_4(sc, RE_RXCFG, CSR_READ_4(sc, RE_RXCFG) &
                    ~(RE_RXCFG_RX_ALLPHYS | RE_RXCFG_RX_INDIV | RE_RXCFG_RX_MULTI |
                      RE_RXCFG_RX_BROAD | RE_RXCFG_RX_RUNT | RE_RXCFG_RX_ERRPKT));
}

static void re_set_rx_packet_filter_in_sleep_state(struct re_softc *sc)
{
        u_int32_t		rxfilt;

        rxfilt = CSR_READ_4(sc, RE_RXCFG);

        rxfilt &= ~(RE_RXCFG_RX_ALLPHYS | RE_RXCFG_RX_INDIV | RE_RXCFG_RX_MULTI | RE_RXCFG_RX_BROAD | RE_RXCFG_RX_RUNT | RE_RXCFG_RX_ERRPKT);
        rxfilt |= (RE_RXCFG_RX_INDIV | RE_RXCFG_RX_MULTI | RE_RXCFG_RX_BROAD);

        CSR_WRITE_4(sc, RE_RXCFG, rxfilt);

        re_set_multicast_reg(sc, 0xFFFFFFFF, 0xFFFFFFFF);

        return;
}

static void re_set_rx_packet_filter(struct re_softc *sc)
{
        struct ifnet		*ifp;
        u_int32_t		rxfilt;

        ifp = RE_GET_IFNET(sc);

        rxfilt = CSR_READ_4(sc, RE_RXCFG);

        rxfilt |= RE_RXCFG_RX_INDIV;

        if (ifp->if_flags & (IFF_MULTICAST | IFF_ALLMULTI | IFF_PROMISC)) {
                rxfilt |= (RE_RXCFG_RX_ALLPHYS | RE_RXCFG_RX_MULTI);
        } else {
                rxfilt &= ~(RE_RXCFG_RX_MULTI);
        }

        if (ifp->if_flags & IFF_PROMISC) {
                rxfilt |= (RE_RXCFG_RX_ALLPHYS | RE_RXCFG_RX_RUNT | RE_RXCFG_RX_ERRPKT);
        } else {
                rxfilt &= ~(RE_RXCFG_RX_ALLPHYS | RE_RXCFG_RX_RUNT | RE_RXCFG_RX_ERRPKT);
        }

        if (ifp->if_flags & (IFF_BROADCAST | IFF_PROMISC)) {
                rxfilt |= RE_RXCFG_RX_BROAD;
        } else {
                rxfilt &= ~RE_RXCFG_RX_BROAD;
        }

        CSR_WRITE_4(sc, RE_RXCFG, rxfilt);

        re_setmulti(sc);

        return;
}

#if OS_VER >= VERSION(13,0)
static u_int
re_hash_maddr(void *arg, struct sockaddr_dl *sdl, u_int cnt)
{
	uint32_t h, *hashes = arg;

	h = ether_crc32_be(LLADDR(sdl), ETHER_ADDR_LEN) >> 26;
	if (h < 32)
		hashes[0] |= (1 << h);
	else
		hashes[1] |= (1 << (h - 32));

	return (1);
}
#endif

/*
 * Program the 64-bit multicast hash filter.
 */
static void re_setmulti(struct re_softc *sc)
{
        struct ifnet		*ifp;
        int			h = 0;
        u_int32_t		hashes[2] = { 0, 0 };
#if OS_VER < VERSION(13,0)
        struct ifmultiaddr	*ifma;
#endif
        u_int32_t		rxfilt;
        int			mcnt = 0;

        ifp = RE_GET_IFNET(sc);

        rxfilt = CSR_READ_4(sc, RE_RXCFG);

        if (ifp->if_flags & IFF_ALLMULTI || ifp->if_flags & IFF_PROMISC) {
                rxfilt |= RE_RXCFG_RX_MULTI;
                CSR_WRITE_4(sc, RE_RXCFG, rxfilt);
                re_set_multicast_reg(sc, 0xFFFFFFFF, 0xFFFFFFFF);

                return;
        }

        /* now program new ones */
#if OS_VER >= VERSION(13,0)
	mcnt = if_foreach_llmaddr(ifp, re_hash_maddr, hashes);
#else
#if OS_VER >= VERSION(12,0)
        if_maddr_rlock(ifp);
#elif OS_VER > VERSION(6,0)
        IF_ADDR_LOCK(ifp);
#endif
#if OS_VER < VERSION(4,9)
        for (ifma = ifp->if_multiaddrs.lh_first; ifma != NULL;
             ifma = ifma->ifma_link.le_next)
#elif OS_VER < VERSION(12,0)
        TAILQ_FOREACH(ifma,&ifp->if_multiaddrs,ifma_link)
#else
        CK_STAILQ_FOREACH(ifma,&ifp->if_multiaddrs,ifma_link)
#endif
        {
                if (ifma->ifma_addr->sa_family != AF_LINK)
                        continue;
                h = ether_crc32_be(LLADDR((struct sockaddr_dl *)
                                          ifma->ifma_addr), ETHER_ADDR_LEN) >> 26;
                if (h < 32)
                        hashes[0] |= (1 << h);
                else
                        hashes[1] |= (1 << (h - 32));
                mcnt++;
        }
#if OS_VER >= VERSION(12,0)
        if_maddr_runlock(ifp);
#elif OS_VER > VERSION(6,0)
        IF_ADDR_UNLOCK(ifp);
#endif
#endif

        if (mcnt) {
                if ((sc->re_if_flags & RL_FLAG_PCIE) != 0) {
                        h = bswap32(hashes[0]);
                        hashes[0] = bswap32(hashes[1]);
                        hashes[1] = h;
                }
                rxfilt |= RE_RXCFG_RX_MULTI;
        } else
                rxfilt &= ~RE_RXCFG_RX_MULTI;

        CSR_WRITE_4(sc, RE_RXCFG, rxfilt);
        re_set_multicast_reg(sc, hashes[0], hashes[1]);

        return;
}

static int re_ioctl(struct ifnet *ifp, u_long command, caddr_t data)
{
        struct re_softc		*sc = ifp->if_softc;
        struct ifreq		*ifr = (struct ifreq *) data;
        /*int			s;*/
        int			error = 0;
        int mask, reinit;
        /*s = splimp();*/

        switch(command) {
        case SIOCSIFADDR:
        case SIOCGIFADDR:
                error = ether_ioctl(ifp, command, data);

                break;
        case SIOCSIFMTU:

                //printf("before mtu =%d\n",(int)ifp->if_mtu);
                if (ifr->ifr_mtu > sc->max_jumbo_frame_size) {
                        error = EINVAL;
                        break;
                }
                RE_LOCK(sc);
                if (ifp->if_mtu != ifr->ifr_mtu) {
                        ifp->if_mtu = ifr->ifr_mtu;
                        //if running
                        if (ifp->if_drv_flags & IFF_DRV_RUNNING) {
                                //printf("set mtu when running\n");
                                re_stop(sc);

                                re_release_buf(sc);
                                set_rxbufsize(sc);
                                error = re_alloc_buf(sc);

                                if (error == 0) {
                                        re_init_locked(sc);
                                }
                        } else {
                                //if not running
                                re_release_buf(sc);
                                set_rxbufsize(sc);
                                error = re_alloc_buf(sc);
                                if (error == 0) {
                                        /* Init descriptors. */
                                        re_var_init(sc);
                                }

                        }

                        if (ifp->if_mtu > ETHERMTU) {
                                ifp->if_capenable &= ~(IFCAP_TSO |
                                                       IFCAP_VLAN_HWTSO);
                                ifp->if_hwassist &= ~CSUM_TSO;
                        }
                        //	printf("after mtu =%d\n",(int)ifp->if_mtu);
                }
                RE_UNLOCK(sc);
                break;
        case SIOCSIFFLAGS:
                RE_LOCK(sc);
                if (ifp->if_flags & IFF_UP) {
                        re_init_locked(sc);
                } else if (ifp->if_drv_flags & IFF_DRV_RUNNING) {
                        re_stop(sc);
                }
                error = 0;
                RE_UNLOCK(sc);
                break;
        case SIOCADDMULTI:
        case SIOCDELMULTI:
                RE_LOCK(sc);
                re_set_rx_packet_filter(sc);
                RE_UNLOCK(sc);
                error = 0;
                break;
        case SIOCGIFMEDIA:
        case SIOCSIFMEDIA:
                error = ifmedia_ioctl(ifp, ifr, &sc->media, command);
                break;
        case SIOCSIFCAP:
                mask = ifr->ifr_reqcap ^ ifp->if_capenable;
                reinit = 0;

                if ((mask & IFCAP_TXCSUM) != 0 && (ifp->if_capabilities & IFCAP_TXCSUM) != 0) {
                        ifp->if_capenable ^= IFCAP_TXCSUM;
                        if ((ifp->if_capenable & IFCAP_TXCSUM) != 0)  {
                                if ((sc->re_type == MACFG_24) || (sc->re_type == MACFG_25) || (sc->re_type == MACFG_26))
                                        ifp->if_hwassist |= CSUM_TCP | CSUM_UDP;
                                else
                                        ifp->if_hwassist |= RE_CSUM_FEATURES_IPV4;
                        } else
                                ifp->if_hwassist &= ~RE_CSUM_FEATURES_IPV4;
                        reinit = 1;
                }

                if ((mask & IFCAP_TXCSUM_IPV6) != 0 && (ifp->if_capabilities & IFCAP_TXCSUM_IPV6) != 0) {
                        ifp->if_capenable ^= IFCAP_TXCSUM_IPV6;
                        if ((ifp->if_capenable & IFCAP_TXCSUM_IPV6) != 0)  {
                                ifp->if_hwassist |= RE_CSUM_FEATURES_IPV6;
                        } else
                                ifp->if_hwassist &= ~RE_CSUM_FEATURES_IPV6;

                        if (!((sc->re_if_flags & RL_FLAG_DESCV2) &&
                              (sc->re_if_flags & RL_FLAG_8168G_PLUS)))
                                ifp->if_hwassist &= ~RE_CSUM_FEATURES_IPV6;
                        reinit = 1;
                }

                if ((mask & IFCAP_RXCSUM) != 0 &&
                    (ifp->if_capabilities & IFCAP_RXCSUM) != 0) {
                        ifp->if_capenable ^= IFCAP_RXCSUM;
                        reinit = 1;
                }

                if ((mask & IFCAP_RXCSUM_IPV6) != 0 &&
                    (ifp->if_capabilities & IFCAP_RXCSUM_IPV6) != 0) {
                        ifp->if_capenable ^= IFCAP_RXCSUM_IPV6;
                        reinit = 1;
                }

                if ((ifp->if_mtu <= ETHERMTU) || ((sc->re_type>= MACFG_3) &&(sc->re_type <=MACFG_6)) || ((sc->re_type>= MACFG_21) && (sc->re_type <=MACFG_23))) {
                        if (ifp->if_capenable & IFCAP_TXCSUM)
                                sc->re_tx_cstag = 1;
                        else
                                sc->re_tx_cstag = 0;

                        if (ifp->if_capenable & (IFCAP_RXCSUM | IFCAP_RXCSUM_IPV6))
                                sc->re_rx_cstag = 1;
                        else
                                sc->re_rx_cstag = 0;
                }

                if ((mask & IFCAP_TSO4) != 0 &&
                    (ifp->if_capabilities & IFCAP_TSO4) != 0) {
                        ifp->if_capenable ^= IFCAP_TSO4;
                        if ((IFCAP_TSO4 & ifp->if_capenable) != 0)
                                ifp->if_hwassist |= CSUM_IP_TSO;
                        else
                                ifp->if_hwassist &= ~CSUM_IP_TSO;
                        if (ifp->if_mtu > ETHERMTU) {
                                ifp->if_capenable &= ~IFCAP_TSO4;
                                ifp->if_hwassist &= ~CSUM_IP_TSO;
                        }
                }
                /*
                if ((mask & IFCAP_TSO6) != 0 &&
                (ifp->if_capabilities & IFCAP_TSO6) != 0) {
                ifp->if_capenable ^= IFCAP_TSO6;
                if ((IFCAP_TSO6 & ifp->if_capenable) != 0)
                ifp->if_hwassist |= CSUM_IP6_TSO;
                else
                ifp->if_hwassist &= ~CSUM_IP6_TSO;
                if (ifp->if_mtu > ETHERMTU) {
                ifp->if_capenable &= ~IFCAP_TSO6;
                ifp->if_hwassist &= ~CSUM_IP6_TSO;
                }
                if (ifp->if_mtu > ETHERMTU) {
                ifp->if_capenable &= ~IFCAP_TSO6;
                ifp->if_hwassist &= ~CSUM_IP6_TSO;
                }
                }
                */
                if ((mask & IFCAP_VLAN_HWTSO) != 0 &&
                    (ifp->if_capabilities & IFCAP_VLAN_HWTSO) != 0)
                        ifp->if_capenable ^= IFCAP_VLAN_HWTSO;
                if ((mask & IFCAP_VLAN_HWTAGGING) != 0 &&
                    (ifp->if_capabilities & IFCAP_VLAN_HWTAGGING) != 0) {
                        ifp->if_capenable ^= IFCAP_VLAN_HWTAGGING;
                        /* TSO over VLAN requires VLAN hardware tagging. */
                        //if ((ifp->if_capenable & IFCAP_VLAN_HWTAGGING) == 0)
                        //	ifp->if_capenable &= ~IFCAP_VLAN_HWTSO;
                        reinit = 1;
                }
                if (mask & IFCAP_LRO)
                        ifp->if_capenable ^= IFCAP_LRO;

                if ((mask & IFCAP_WOL) != 0 &&
                    (ifp->if_capabilities & IFCAP_WOL) != 0) {
                        if ((mask & IFCAP_WOL_UCAST) != 0)
                                ifp->if_capenable ^= IFCAP_WOL_UCAST;
                        if ((mask & IFCAP_WOL_MCAST) != 0)
                                ifp->if_capenable ^= IFCAP_WOL_MCAST;
                        if ((mask & IFCAP_WOL_MAGIC) != 0)
                                ifp->if_capenable ^= IFCAP_WOL_MAGIC;
                }
                if (reinit && ifp->if_drv_flags & IFF_DRV_RUNNING) {
                        ifp->if_drv_flags &= ~IFF_DRV_RUNNING;
                        re_init(sc);
                }
                VLAN_CAPABILITIES(ifp);
                break;
        default:
                error = EINVAL;
                break;
        }

        /*(void)splx(s);*/

        return(error);
}

static void re_link_on_patch(struct re_softc *sc)
{
        struct ifnet		*ifp;

        ifp = RE_GET_IFNET(sc);

        if (sc->re_type == MACFG_50 || sc->re_type == MACFG_51 || sc->re_type == MACFG_52) {
                if (CSR_READ_1(sc, RE_PHY_STATUS) & RL_PHY_STATUS_1000MF) {
                        re_eri_write(sc, 0x1bc, 4, 0x00000011, ERIAR_ExGMAC);
                        re_eri_write(sc, 0x1dc, 4, 0x0000001f, ERIAR_ExGMAC);
                } else if (CSR_READ_1(sc, RE_PHY_STATUS) & RL_PHY_STATUS_100M) {
                        re_eri_write(sc, 0x1bc, 4, 0x0000001f, ERIAR_ExGMAC);
                        re_eri_write(sc, 0x1dc, 4, 0x0000001f, ERIAR_ExGMAC);
                } else {
                        re_eri_write(sc, 0x1bc, 4, 0x0000001f, ERIAR_ExGMAC);
                        re_eri_write(sc, 0x1dc, 4, 0x0000002d, ERIAR_ExGMAC);
                }
        } else if ((sc->re_type == MACFG_38 || sc->re_type == MACFG_39) && (ifp->if_flags & IFF_UP)) {
                if (sc->re_type == MACFG_38 && (CSR_READ_1(sc, RE_PHY_STATUS) & RL_PHY_STATUS_10M)) {
                        CSR_WRITE_4(sc, RE_RXCFG, CSR_READ_4(sc, RE_RXCFG) | RE_RXCFG_RX_ALLPHYS);
                } else if (sc->re_type == MACFG_39) {
                        if (CSR_READ_1(sc, RE_PHY_STATUS) & RL_PHY_STATUS_1000MF) {
                                re_eri_write(sc, 0x1bc, 4, 0x00000011, ERIAR_ExGMAC);
                                re_eri_write(sc, 0x1dc, 4, 0x00000005, ERIAR_ExGMAC);
                        } else if (CSR_READ_1(sc, RE_PHY_STATUS) & RL_PHY_STATUS_100M) {
                                re_eri_write(sc, 0x1bc, 4, 0x0000001f, ERIAR_ExGMAC);
                                re_eri_write(sc, 0x1dc, 4, 0x00000005, ERIAR_ExGMAC);
                        } else {
                                re_eri_write(sc, 0x1bc, 4, 0x0000001f, ERIAR_ExGMAC);
                                re_eri_write(sc, 0x1dc, 4, 0x0000003f, ERIAR_ExGMAC);
                        }
                }
        } else if ((sc->re_type == MACFG_36 || sc->re_type == MACFG_37) && eee_enable == 1) {
                /*Full -Duplex  mode*/
                if (CSR_READ_1(sc, RE_PHY_STATUS) & RL_PHY_STATUS_FULL_DUP) {
                        re_mdio_write(sc, 0x1F, 0x0006);
                        re_mdio_write(sc, 0x00, 0x5a30);
                        re_mdio_write(sc, 0x1F, 0x0000);
                        if (CSR_READ_1(sc, RE_PHY_STATUS) & (RL_PHY_STATUS_10M | RL_PHY_STATUS_100M))
                                CSR_WRITE_4(sc, RE_TXCFG, (CSR_READ_4(sc, RE_TXCFG) & ~BIT_19) | BIT_25);

                } else {
                        re_mdio_write(sc, 0x1F, 0x0006);
                        re_mdio_write(sc, 0x00, 0x5a00);
                        re_mdio_write(sc, 0x1F, 0x0000);
                        if (CSR_READ_1(sc, RE_PHY_STATUS) & (RL_PHY_STATUS_10M | RL_PHY_STATUS_100M))
                                CSR_WRITE_4(sc, RE_TXCFG, (CSR_READ_4(sc, RE_TXCFG) & ~BIT_19) | RE_TXCFG_IFG);
                }
        } else if ((sc->re_type == MACFG_56 || sc->re_type == MACFG_57 ||
                    sc->re_type == MACFG_58 || sc->re_type == MACFG_59 ||
                    sc->re_type == MACFG_60 || sc->re_type == MACFG_61 ||
                    sc->re_type == MACFG_62 || sc->re_type == MACFG_67 ||
                    sc->re_type == MACFG_68 || sc->re_type == MACFG_69 ||
                    sc->re_type == MACFG_70 || sc->re_type == MACFG_71 ||
                    sc->re_type == MACFG_72 || sc->re_type == MACFG_73 ||
                    sc->re_type == MACFG_74 || sc->re_type == MACFG_75 ||
                    sc->re_type == MACFG_76 || sc->re_type == MACFG_80 ||
                    sc->re_type == MACFG_81 || sc->re_type == MACFG_82 ||
                    sc->re_type == MACFG_83 || sc->re_type == MACFG_84 ||
                    sc->re_type == MACFG_85 || sc->re_type == MACFG_86 ||
                    sc->re_type == MACFG_87 || sc->re_type == MACFG_88 ||
                    sc->re_type == MACFG_90 || sc->re_type == MACFG_91 ||
                    sc->re_type == MACFG_92 || sc->re_type == MACFG_100 ||
                    sc->re_type == MACFG_101) &&
                   (ifp->if_flags & IFF_UP)) {
                if (CSR_READ_1(sc, RE_PHY_STATUS) & RL_PHY_STATUS_FULL_DUP)
                        CSR_WRITE_4(sc, RE_TXCFG, (CSR_READ_4(sc, RE_TXCFG) | (BIT_24 | BIT_25)) & ~BIT_19);
                else
                        CSR_WRITE_4(sc, RE_TXCFG, (CSR_READ_4(sc, RE_TXCFG) | BIT_25) & ~(BIT_19 | BIT_24));
        }

        if (sc->re_type == MACFG_56 || sc->re_type == MACFG_57 ||
            sc->re_type == MACFG_61 || sc->re_type == MACFG_62) {
                /*half mode*/
                if (!(CSR_READ_1(sc, RE_PHY_STATUS) & RL_PHY_STATUS_FULL_DUP)) {
                        re_mdio_write(sc, 0x1F, 0x0000);
                        re_mdio_write(sc, MII_ANAR, re_mdio_read(sc, MII_ANAR)&~(ANAR_FC |ANAR_PAUSE_ASYM));
                }
        }

        if (CSR_READ_1(sc, RE_PHY_STATUS) & RL_PHY_STATUS_10M) {
                if (sc->re_type == MACFG_70 || sc->re_type == MACFG_71 ||
                    sc->re_type == MACFG_72 || sc->re_type == MACFG_73) {
                        uint32_t Data32;

                        Data32 = re_eri_read(sc, 0x1D0, 1, ERIAR_ExGMAC);
                        Data32 |= BIT_1;
                        re_eri_write(sc, 0x1D0, 1, Data32, ERIAR_ExGMAC);
                } else if (sc->re_type == MACFG_80 || sc->re_type == MACFG_81 ||
                           sc->re_type == MACFG_82 || sc->re_type == MACFG_83 ||
                           sc->re_type == MACFG_84 || sc->re_type == MACFG_85 ||
                           sc->re_type == MACFG_88 || sc->re_type == MACFG_90 ||
                           sc->re_type == MACFG_91 || sc->re_type == MACFG_92 ||
                           sc->re_type == MACFG_100 || sc->re_type == MACFG_101) {
                        re_mac_ocp_write(sc, 0xE080, re_mac_ocp_read(sc, 0xE080)|BIT_1);
                }
        }

        if (sc->RequiredPfmPatch)
                re_set_pfm_patch(sc, (CSR_READ_1(sc, RE_PHY_STATUS) & RL_PHY_STATUS_10M) ? 1 : 0);

        re_init_unlock(sc);
}

static void re_link_down_patch(struct re_softc *sc)
{
        if (sc->RequiredPfmPatch)
                re_set_pfm_patch(sc, 1);

        re_txeof(sc, default_tx_qid);
        re_rxeof(sc);
        re_stop(sc);
}

/*
 * Check Link Status.
 */
static void re_check_link_status(struct re_softc *sc)
{
        u_int8_t	link_state;
        struct ifnet		*ifp;

        ifp = RE_GET_IFNET(sc);

        if (re_link_ok(sc)) {
                link_state = LINK_STATE_UP;
        } else {
                link_state = LINK_STATE_DOWN;
        }

        if (link_state != sc->link_state) {
                sc->link_state = link_state;
                if (link_state == LINK_STATE_UP) {
                        re_link_on_patch(sc);
                        re_link_state_change(ifp, LINK_STATE_UP);
                } else {
                        re_link_state_change(ifp, LINK_STATE_DOWN);
                        re_link_down_patch(sc);
                }
        }
}

static void re_init_timer(struct re_softc *sc)
{
#ifdef RE_USE_NEW_CALLOUT_FUN
        callout_init(&sc->re_stat_ch, CALLOUT_MPSAFE);
#else
        callout_handle_init(&sc->re_stat_ch);
#endif
}

static void re_stop_timer(struct re_softc *sc)
{
#ifdef RE_USE_NEW_CALLOUT_FUN
        callout_stop(&sc->re_stat_ch);
#else
        untimeout(re_tick, sc, sc->re_stat_ch);
#endif
}

static void re_start_timer(struct re_softc *sc)
{
#ifdef RE_USE_NEW_CALLOUT_FUN
        callout_reset(&sc->re_stat_ch, hz, re_tick, sc);
#else
        re_stop_timer(sc);
        sc->re_stat_ch = timeout(re_tick, sc, hz);
#endif
}

static void re_tick(void *xsc)
{
        /*called per second*/
        struct re_softc		*sc;
        int			s;

        s = splimp();

        sc = xsc;
        /*mii = device_get_softc(sc->re_miibus);

        mii_tick(mii);*/

        splx(s);

        RE_LOCK(sc);

        if (sc->re_link_chg_det == 1) {
                re_check_link_status(sc);
                re_start_timer(sc);
        }

        RE_UNLOCK(sc);

        return;
}

#if OS_VER < VERSION(7,0)
static void re_watchdog(ifp)
struct ifnet		*ifp;
{
        struct re_softc		*sc;

        sc = ifp->if_softc;

        printf("re%d: watchdog timeout\n", sc->re_unit);
#if OS_VER < VERSION(11,0)
        ifp->if_oerrors++;
#else
        if_inc_counter(ifp, IFCOUNTER_OERRORS, 1);
#endif

        re_txeof(sc, default_tx_qid);
        re_rxeof(sc);
        re_init(sc);

        return;
}
#endif

/*
 * Set media options.
 */
static int re_ifmedia_upd(struct ifnet *ifp)
{
        struct re_softc	*sc = ifp->if_softc;
        struct ifmedia	*ifm = &sc->media;
        int anar;
        int gbcr;

        if (IFM_TYPE(ifm->ifm_media) != IFM_ETHER)
                return(EINVAL);

        if (sc->re_type == MACFG_68 || sc->re_type == MACFG_69 ||
            sc->re_type == MACFG_70 || sc->re_type == MACFG_71 ||
            sc->re_type == MACFG_72 || sc->re_type == MACFG_73 ||
            sc->re_type == MACFG_74 || sc->re_type == MACFG_75) {
                //Disable Giga Lite
                re_mdio_write(sc, 0x1F, 0x0A42);
                re_clear_eth_phy_bit(sc, 0x14, BIT_9);
                if (sc->re_type == MACFG_70 || sc->re_type == MACFG_71 ||
                    sc->re_type == MACFG_72 || sc->re_type == MACFG_73)
                        re_clear_eth_phy_bit(sc, 0x14, BIT_7);
                re_mdio_write(sc, 0x1F, 0x0A40);
                re_mdio_write(sc, 0x1F, 0x0000);
        }


        switch (IFM_SUBTYPE(ifm->ifm_media)) {
        case IFM_AUTO:
                anar = ANAR_TX_FD |
                       ANAR_TX |
                       ANAR_10_FD |
                       ANAR_10;
                gbcr = GTCR_ADV_1000TFDX |
                       GTCR_ADV_1000THDX;
                break;
        case IFM_1000_SX:
#if OS_VER < 500000
        case IFM_1000_TX:
#else
        case IFM_1000_T:
#endif
                anar = ANAR_TX_FD |
                       ANAR_TX |
                       ANAR_10_FD |
                       ANAR_10;
                gbcr = GTCR_ADV_1000TFDX |
                       GTCR_ADV_1000THDX;
                break;
        case IFM_100_TX:
                gbcr = re_mdio_read(sc, MII_100T2CR) &
                       ~(GTCR_ADV_1000TFDX | GTCR_ADV_1000THDX);
                if ((ifm->ifm_media & IFM_GMASK) == IFM_FDX) {
                        anar = ANAR_TX_FD |
                               ANAR_TX |
                               ANAR_10_FD |
                               ANAR_10;
                } else {
                        anar = ANAR_TX |
                               ANAR_10_FD |
                               ANAR_10;
                }
                break;
        case IFM_10_T:
                gbcr = re_mdio_read(sc, MII_100T2CR) &
                       ~(GTCR_ADV_1000TFDX | GTCR_ADV_1000THDX);
                if ((ifm->ifm_media & IFM_GMASK) == IFM_FDX) {
                        anar = ANAR_10_FD |
                               ANAR_10;
                } else {
                        anar = ANAR_10;
                }

                if (sc->re_type == MACFG_13) {
                        re_mdio_write(sc, MII_BMCR, 0x8000);
                }

                break;
        default:
                printf("re%d: Unsupported media type\n", sc->re_unit);
                return(0);
        }

        if (sc->re_device_id==RT_DEVICEID_8162)
                re_clear_eth_ocp_phy_bit(sc, 0xA5D4, RTK_ADVERTISE_2500FULL);

        re_mdio_write(sc, 0x1F, 0x0000);
        if (sc->re_device_id==RT_DEVICEID_8169 || sc->re_device_id==RT_DEVICEID_8169SC ||
            sc->re_device_id==RT_DEVICEID_8168 || sc->re_device_id==RT_DEVICEID_8161 ||
            sc->re_device_id==RT_DEVICEID_8162) {
                re_mdio_write(sc, MII_ANAR, anar | ANAR_FC | ANAR_PAUSE_ASYM);
                re_mdio_write(sc, MII_100T2CR, gbcr);
                re_mdio_write(sc, MII_BMCR, BMCR_RESET | BMCR_AUTOEN | BMCR_STARTNEG);
        } else if (sc->re_type == MACFG_36) {
                re_mdio_write(sc, MII_ANAR, anar | ANAR_FC | ANAR_PAUSE_ASYM);
                re_mdio_write(sc, MII_BMCR, BMCR_RESET | BMCR_AUTOEN | BMCR_STARTNEG);
        } else {
                re_mdio_write(sc, MII_ANAR, anar | 1);
                re_mdio_write(sc, MII_BMCR, BMCR_AUTOEN | BMCR_STARTNEG);
        }

        return(0);
}

static int re_ifmedia_upd_8125(struct ifnet *ifp)
{
        struct re_softc	*sc = ifp->if_softc;
        struct ifmedia	*ifm = &sc->media;
        int anar;
        int gbcr;
        int cr2500;

        if (IFM_TYPE(ifm->ifm_media) != IFM_ETHER)
                return(EINVAL);

        //Disable Giga Lite
        switch(sc->re_device_id) {
        case RT_DEVICEID_8127:
                re_clear_eth_ocp_phy_bit(sc, 0xA5EA, BIT_2);
        /* FALLTHROUGH */
        case RT_DEVICEID_8126:
                re_clear_eth_ocp_phy_bit(sc, 0xA5EA, BIT_1);
        /* FALLTHROUGH */
        default:
                re_clear_eth_ocp_phy_bit(sc, 0xA428, BIT_9);
                re_clear_eth_ocp_phy_bit(sc, 0xA5EA, BIT_0);
                break;
        }

        cr2500 = re_real_ocp_phy_read(sc, 0xA5D4) &
                 ~(RTK_ADVERTISE_2500FULL | RTK_ADVERTISE_5000FULL |
                   RTK_ADVERTISE_10GFULL);
        gbcr = re_mdio_read(sc, MII_100T2CR) &
               ~(GTCR_ADV_1000TFDX | GTCR_ADV_1000THDX);
        anar = re_mdio_read(sc, MII_ANAR) &
               ~(ANAR_10 | ANAR_10_FD | ANAR_TX | ANAR_TX_FD | ANAR_FC | ANAR_PAUSE_ASYM);

        switch (IFM_SUBTYPE(ifm->ifm_media)) {
        case IFM_AUTO:
        case IFM_10G_T:
                if (sc->re_device_id == RT_DEVICEID_8127)
                        cr2500 |= RTK_ADVERTISE_10GFULL;
        /*	FALLTHROUGH */
        case IFM_5000_T:
                if (sc->re_device_id == RT_DEVICEID_8127 || sc->re_device_id==RT_DEVICEID_8126)
                        cr2500 |= RTK_ADVERTISE_5000FULL;
        /*	FALLTHROUGH */
        case IFM_2500_SX:
        case IFM_2500_X:
        case IFM_2500_T:
                cr2500 |= RTK_ADVERTISE_2500FULL;
        /*	FALLTHROUGH */
        case MACFG_80:
        case IFM_1000_SX:
#if OS_VER < 500000
        case IFM_1000_TX:
#else
        case IFM_1000_T:
#endif
                gbcr |= GTCR_ADV_1000TFDX;
                anar |= ANAR_TX_FD;
        /*	FALLTHROUGH */
        case IFM_100_TX:
                anar |= ANAR_TX | ANAR_10_FD;
                if ((ifm->ifm_media & IFM_GMASK) == IFM_FDX)
                        anar |= ANAR_TX_FD;
        /*	FALLTHROUGH */
        case IFM_10_T:
                anar |= ANAR_10;
                if ((ifm->ifm_media & IFM_GMASK) == IFM_FDX)
                        anar |= ANAR_10_FD;

                if (sc->re_type == MACFG_13) {
                        re_mdio_write(sc, MII_BMCR, 0x8000);
                }
                break;
        default:
                printf("re%d: Unsupported media type\n", sc->re_unit);
                return(0);
        }

        re_mdio_write(sc, 0x1F, 0x0000);
        re_real_ocp_phy_write(sc, 0xA5D4, cr2500);
        re_mdio_write(sc, MII_ANAR, anar | ANAR_FC | ANAR_PAUSE_ASYM);
        re_mdio_write(sc, MII_100T2CR, gbcr);
        re_mdio_write(sc, MII_BMCR, BMCR_RESET | BMCR_AUTOEN | BMCR_STARTNEG);

        return(0);
}

/*
 * Report current media status.
 */
static void re_ifmedia_sts(struct ifnet *ifp, struct ifmediareq *ifmr)
{
        struct re_softc		*sc;

        sc = ifp->if_softc;

        RE_LOCK(sc);

        ifmr->ifm_status = IFM_AVALID;
        ifmr->ifm_active = IFM_ETHER;

        if (re_link_ok(sc)) {
                unsigned char msr;

                ifmr->ifm_status |= IFM_ACTIVE;

                msr = CSR_READ_1(sc, RE_PHY_STATUS);
                if (msr & RL_PHY_STATUS_FULL_DUP)
                        ifmr->ifm_active |= IFM_FDX;
                else
                        ifmr->ifm_active |= IFM_HDX;

                if (msr & RL_PHY_STATUS_10M)
                        ifmr->ifm_active |= IFM_10_T;
                else if (msr & RL_PHY_STATUS_100M)
                        ifmr->ifm_active |= IFM_100_TX;
                else if (msr & RL_PHY_STATUS_1000MF)
                        ifmr->ifm_active |= IFM_1000_T;
        }

        RE_UNLOCK(sc);

        return;
}

static void re_ifmedia_sts_8125(struct ifnet *ifp, struct ifmediareq *ifmr)
{
        struct re_softc		*sc;

        sc = ifp->if_softc;

        RE_LOCK(sc);

        ifmr->ifm_status = IFM_AVALID;
        ifmr->ifm_active = IFM_ETHER;

        if (re_link_ok(sc)) {
                u_int32_t msr;

                ifmr->ifm_status |= IFM_ACTIVE;

                msr = CSR_READ_4(sc, RE_PHY_STATUS);

                if (msr & RL_PHY_STATUS_FULL_DUP)
                        ifmr->ifm_active |= IFM_FDX;
                else
                        ifmr->ifm_active |= IFM_HDX;

                if (msr & RL_PHY_STATUS_10M)
                        ifmr->ifm_active |= IFM_10_T;
                else if (msr & RL_PHY_STATUS_100M)
                        ifmr->ifm_active |= IFM_100_TX;
                else if (msr & RL_PHY_STATUS_1000MF)
                        ifmr->ifm_active |= IFM_1000_T;
                else if (msr & RL_PHY_STATUS_500MF)
                        ifmr->ifm_active |= IFM_1000_T;
                else if (msr & RL_PHY_STATUS_1250MF)
                        ifmr->ifm_active |= IFM_1000_T;
                else if (msr & RL_PHY_STATUS_2500MF)
                        ifmr->ifm_active |= IFM_2500_T;
                else if (msr & RL_PHY_STATUS_5000MF_LITE)
                        ifmr->ifm_active |= IFM_2500_T;
                else if (msr & RL_PHY_STATUS_5000MF)
                        ifmr->ifm_active |= IFM_5000_T;
                else if (msr & RL_PHY_STATUS_10000MF_LITE)
                        ifmr->ifm_active |= IFM_5000_T;
                else if (msr & RL_PHY_STATUS_10000MF)
                        ifmr->ifm_active |= IFM_10G_T;

                if (msr & RL_PHY_STATUS_TX_FLOW_CTRL)
                        ifmr->ifm_active |= IFM_ETH_TXPAUSE;
                if (msr & RL_PHY_STATUS_RX_FLOW_CTRL)
                        ifmr->ifm_active |= IFM_ETH_RXPAUSE;
        }

        RE_UNLOCK(sc);

        return;
}

static bool re_is_advanced_eee_enabled(struct re_softc *sc)
{
        switch (sc->re_type) {
        case MACFG_58:
        case MACFG_59:
        case MACFG_60:
        case MACFG_61:
        case MACFG_62:
        case MACFG_67:
        case MACFG_68:
        case MACFG_69:
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
        case MACFG_74:
        case MACFG_75:
        case MACFG_80:
        case MACFG_81:
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                if (re_real_ocp_phy_read(sc, 0xA430) & BIT_15)
                        return true;
                break;
        default:
                break;
        }

        return false;
}

static void _re_disable_advanced_eee(struct re_softc *sc)
{
        u_int16_t data;
        bool lock;

        if (re_is_advanced_eee_enabled(sc))
                lock = true;
        else
                lock = false;

        if (lock)
                re_set_phy_mcu_patch_request(sc);

        switch (sc->re_type) {
        case MACFG_59:
                re_eri_write(sc, 0x1EA, 1, 0x00, ERIAR_ExGMAC);

                re_mdio_write(sc, 0x1F, 0x0A42);
                data = re_mdio_read(sc, 0x16);
                data &= ~(BIT_1);
                re_mdio_write(sc, 0x16, data);
                re_mdio_write(sc, 0x1F, 0x0000);
                break;
        case MACFG_60:
                data = re_mac_ocp_read(sc, 0xE052);
                data &= ~(BIT_0);
                re_mac_ocp_write(sc, 0xE052, data);

                re_mdio_write(sc, 0x1F, 0x0A42);
                data = re_mdio_read(sc, 0x16);
                data &= ~(BIT_1);
                re_mdio_write(sc, 0x16, data);
                re_mdio_write(sc, 0x1F, 0x0000);
                break;
        case MACFG_61:
        case MACFG_62:
        case MACFG_67:
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
                data = re_mac_ocp_read(sc, 0xE052);
                data &= ~(BIT_0);
                re_mac_ocp_write(sc, 0xE052, data);
                break;
        case MACFG_68:
        case MACFG_69:
        case MACFG_74:
        case MACFG_75:
                data = re_mac_ocp_read(sc, 0xE052);
                data &= ~(BIT_0);
                re_mac_ocp_write(sc, 0xE052, data);

                re_mdio_write(sc, 0x1F, 0x0A43);
                data = re_mdio_read(sc, 0x10) & ~(BIT_15);
                re_mdio_write(sc, 0x10, data);

                re_mdio_write(sc, 0x1F, 0x0A44);
                data = re_mdio_read(sc, 0x11) & ~(BIT_12 | BIT_13 | BIT_14);
                re_mdio_write(sc, 0x11, data);
                re_mdio_write(sc, 0x1f, 0x0000);
                break;
        case MACFG_80:
        case MACFG_81:
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                re_clear_mac_ocp_bit(sc, 0xE052, BIT_0);
                re_clear_eth_ocp_phy_bit(sc, 0xA442, BIT_12 | BIT_13);
                re_clear_eth_ocp_phy_bit(sc, 0xA430, BIT_15);
                break;
        }

        if (lock)
                re_clear_phy_mcu_patch_request(sc);
}

static void re_disable_advanced_eee(struct re_softc *sc)
{
        if (sc->HwSuppDashVer > 1)
                OOB_mutex_lock(sc);

        _re_disable_advanced_eee(sc);

        if (sc->HwSuppDashVer > 1)
                OOB_mutex_unlock(sc);
}

static int re_enable_eee(struct re_softc *sc)
{
        int ret;
        u_int16_t data;

        ret = 0;
        switch (sc->re_type) {
        case MACFG_42:
        case MACFG_43:
                re_eri_write(sc, 0x1B0, 2, 0xED03, ERIAR_ExGMAC);
                re_mdio_write(sc, 0x1F, 0x0004);
                if (CSR_READ_1(sc, 0xEF) & 0x02) {
                        re_mdio_write(sc, 0x10, 0x731F);
                        re_mdio_write(sc, 0x19, 0x7630);
                } else {
                        re_mdio_write(sc, 0x10, 0x711F);
                        re_mdio_write(sc, 0x19, 0x7030);
                }
                re_mdio_write(sc, 0x1A, 0x1506);
                re_mdio_write(sc, 0x1B, 0x0551);
                re_mdio_write(sc, 0x1F, 0x0000);
                re_mdio_write(sc, 0x0D, 0x0007);
                re_mdio_write(sc, 0x0E, 0x003C);
                re_mdio_write(sc, 0x0D, 0x4007);
                re_mdio_write(sc, 0x0E, 0x0002);
                re_mdio_write(sc, 0x0D, 0x0000);

                re_mdio_write(sc, 0x1F, 0x0000);
                re_mdio_write(sc, 0x0D, 0x0003);
                re_mdio_write(sc, 0x0E, 0x0015);
                re_mdio_write(sc, 0x0D, 0x4003);
                re_mdio_write(sc, 0x0E, 0x0002);
                re_mdio_write(sc, 0x0D, 0x0000);

                re_mdio_write(sc, MII_BMCR, BMCR_AUTOEN | BMCR_STARTNEG);
                break;

        case MACFG_53:
        case MACFG_54:
        case MACFG_55:
                re_eri_write(sc, 0x1B0, 2, 0xED03, ERIAR_ExGMAC);
                re_mdio_write(sc, 0x1F, 0x0004);
                re_mdio_write(sc, 0x10, 0x731F);
                re_mdio_write(sc, 0x19, 0x7630);
                re_mdio_write(sc, 0x1A, 0x1506);
                re_mdio_write(sc, 0x1F, 0x0000);
                re_mdio_write(sc, 0x0D, 0x0007);
                re_mdio_write(sc, 0x0E, 0x003C);
                re_mdio_write(sc, 0x0D, 0x4007);
                re_mdio_write(sc, 0x0E, 0x0002);
                re_mdio_write(sc, 0x0D, 0x0000);

                re_mdio_write(sc, MII_BMCR, BMCR_AUTOEN | BMCR_STARTNEG);
                break;

        case MACFG_36:
        case MACFG_37:
                re_mdio_write(sc, 0x1F, 0x0007);
                re_mdio_write(sc, 0x1E, 0x0020);
                data = re_mdio_read(sc, 0x15) | 0x0100;
                re_mdio_write(sc, 0x15, data);
                re_mdio_write(sc, 0x1F, 0x0006);
                re_mdio_write(sc, 0x00, 0x5A30);
                re_mdio_write(sc, 0x1F, 0x0000);
                re_mdio_write(sc, 0x0D, 0x0007);
                re_mdio_write(sc, 0x0E, 0x003C);
                re_mdio_write(sc, 0x0D, 0x4007);
                re_mdio_write(sc, 0x0E, 0x0006);
                re_mdio_write(sc, 0x0D, 0x0000);
                if ((CSR_READ_1(sc, RE_CFG4)&RL_CFG4_CUSTOMIZED_LED) && (CSR_READ_1(sc, RE_MACDBG) & BIT_7)) {
                        re_mdio_write(sc, 0x1F, 0x0005);
                        re_mdio_write(sc, 0x05, 0x8AC8);
                        re_mdio_write(sc, 0x06, CSR_READ_1(sc, RE_CUSTOM_LED));
                        re_mdio_write(sc, 0x05, 0x8B82);
                        data = re_mdio_read(sc, 0x06) | 0x0010;
                        re_mdio_write(sc, 0x05, 0x8B82);
                        re_mdio_write(sc, 0x06, data);
                        re_mdio_write(sc, 0x1F, 0x0000);
                }
                break;

        case MACFG_50:
        case MACFG_51:
        case MACFG_52:
                data = re_eri_read(sc, 0x1B0, 4, ERIAR_ExGMAC) | 0x0003;
                re_eri_write(sc, 0x1B0, 4, data, ERIAR_ExGMAC);
                re_mdio_write(sc, 0x1F, 0x0007);
                re_mdio_write(sc, 0x1E, 0x0020);
                data = re_mdio_read(sc, 0x15)|0x0100;
                re_mdio_write(sc, 0x15, data);
                re_mdio_write(sc, 0x1F, 0x0005);
                re_mdio_write(sc, 0x05, 0x8B85);
                data = re_mdio_read(sc, 0x06)|0x2000;
                re_mdio_write(sc, 0x06, data);
                re_mdio_write(sc, 0x1F, 0x0000);
                re_mdio_write(sc, 0x0D, 0x0007);
                re_mdio_write(sc, 0x0E, 0x003C);
                re_mdio_write(sc, 0x0D, 0x4007);
                re_mdio_write(sc, 0x0E, 0x0006);
                re_mdio_write(sc, 0x1D, 0x0000);
                break;

        case MACFG_38:
        case MACFG_39:
                data = re_eri_read(sc, 0x1B0, 4, ERIAR_ExGMAC);
                data |= BIT_1 | BIT_0;
                re_eri_write(sc, 0x1B0, 4, data, ERIAR_ExGMAC);
                re_mdio_write(sc, 0x1F, 0x0004);
                re_mdio_write(sc, 0x1F, 0x0007);
                re_mdio_write(sc, 0x1e, 0x0020);
                data = re_mdio_read(sc, 0x15);
                data |= BIT_8;
                re_mdio_write(sc, 0x15, data);
                re_mdio_write(sc, 0x1F, 0x0002);
                re_mdio_write(sc, 0x1F, 0x0005);
                re_mdio_write(sc, 0x05, 0x8B85);
                data = re_mdio_read(sc, 0x06);
                data |= BIT_13;
                re_mdio_write(sc, 0x06, data);
                re_mdio_write(sc, 0x1F, 0x0000);
                re_mdio_write(sc, 0x0D, 0x0007);
                re_mdio_write(sc, 0x0E, 0x003C);
                re_mdio_write(sc, 0x0D, 0x4007);
                re_mdio_write(sc, 0x0E, 0x0006);
                re_mdio_write(sc, 0x0D, 0x0000);
                break;

        case MACFG_56:
        case MACFG_57:
        case MACFG_58:
        case MACFG_59:
        case MACFG_60:
        case MACFG_61:
        case MACFG_62:
        case MACFG_67:
        case MACFG_68:
        case MACFG_69:
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
        case MACFG_74:
        case MACFG_75:
                data = re_eri_read(sc, 0x1B0, 4, ERIAR_ExGMAC);
                data |= BIT_1 | BIT_0;
                re_eri_write(sc, 0x1B0, 4, data, ERIAR_ExGMAC);
                re_mdio_write(sc, 0x1F, 0x0A43);
                data = re_mdio_read(sc, 0x11);
                re_mdio_write(sc, 0x11, data | BIT_4);
                re_mdio_write(sc, 0x1F, 0x0A5D);
                re_mdio_write(sc, 0x10, 0x0006);
                re_mdio_write(sc, 0x1F, 0x0000);
                break;

        case MACFG_80:
        case MACFG_81:
                re_set_mac_ocp_bit(sc, 0xE040, (BIT_1|BIT_0));
                re_set_mac_ocp_bit(sc, 0xEB62, (BIT_2|BIT_1));

                re_set_eth_ocp_phy_bit(sc, 0xA432, BIT_4);
                re_set_eth_ocp_phy_bit(sc, 0xA5D0, (BIT_2 | BIT_1));
                re_clear_eth_ocp_phy_bit(sc, 0xA6D4, BIT_0);

                re_clear_eth_ocp_phy_bit(sc, 0xA6D8, BIT_4);
                re_clear_eth_ocp_phy_bit(sc, 0xA428, BIT_7);
                re_clear_eth_ocp_phy_bit(sc, 0xA4A2, BIT_9);
                break;

        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
                re_set_mac_ocp_bit(sc, 0xE040, (BIT_1|BIT_0));

                re_set_eth_ocp_phy_bit(sc, 0xA432, BIT_4);

                re_set_eth_ocp_phy_bit(sc, 0xA5D0, (BIT_2 | BIT_1));
                re_clear_eth_ocp_phy_bit(sc, 0xA6D4, BIT_0);

                re_clear_eth_ocp_phy_bit(sc, 0xA6D8, BIT_4);
                re_clear_eth_ocp_phy_bit(sc, 0xA428, BIT_7);
                re_clear_eth_ocp_phy_bit(sc, 0xA4A2, BIT_9);
                break;

        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                re_set_mac_ocp_bit(sc, 0xE040, (BIT_1|BIT_0));

                re_set_eth_ocp_phy_bit(sc, 0xA5D0, (BIT_2 | BIT_1));
                re_clear_eth_ocp_phy_bit(sc, 0xA6D4, (BIT_1|BIT_0));

                re_clear_eth_ocp_phy_bit(sc, 0xA428, BIT_7);
                re_clear_eth_ocp_phy_bit(sc, 0xA4A2, BIT_9);
                break;

        default:
                ret = -EOPNOTSUPP;
                break;
        }

        switch (sc->re_type) {
        case MACFG_68:
        case MACFG_69:
        case MACFG_74:
        case MACFG_75:
                re_mdio_write(sc, 0x1F, 0x0A4A);
                re_set_eth_phy_bit(sc, 0x11, BIT_9);
                re_mdio_write(sc, 0x1F, 0x0A42);
                re_set_eth_phy_bit(sc, 0x14, BIT_7);
                re_mdio_write(sc, 0x1F, 0x0000);
                break;
        }

        /*Advanced EEE*/
        re_disable_advanced_eee(sc);

        return ret;
}

static int re_disable_eee(struct re_softc *sc)
{
        int ret;
        u_int16_t data;

        ret = 0;
        switch (sc->re_type) {
        case MACFG_42:
        case MACFG_43:
                re_eri_write(sc, 0x1B0, 2, 0, ERIAR_ExGMAC);
                re_mdio_write(sc, 0x1F, 0x0004);
                re_mdio_write(sc, 0x10, 0x401F);
                re_mdio_write(sc, 0x19, 0x7030);

                re_mdio_write(sc, 0x1F, 0x0000);
                re_mdio_write(sc, 0x0D, 0x0007);
                re_mdio_write(sc, 0x0E, 0x003C);
                re_mdio_write(sc, 0x0D, 0x4007);
                re_mdio_write(sc, 0x0E, 0x0000);
                re_mdio_write(sc, 0x0D, 0x0000);

                re_mdio_write(sc, 0x1F, 0x0000);
                re_mdio_write(sc, 0x0D, 0x0003);
                re_mdio_write(sc, 0x0E, 0x0015);
                re_mdio_write(sc, 0x0D, 0x4003);
                re_mdio_write(sc, 0x0E, 0x0000);
                re_mdio_write(sc, 0x0D, 0x0000);

                re_mdio_write(sc, MII_BMCR, BMCR_AUTOEN | BMCR_STARTNEG);
                break;

        case MACFG_53:
                re_eri_write(sc, 0x1B0, 2, 0, ERIAR_ExGMAC);
                re_mdio_write(sc, 0x1F, 0x0004);
                re_mdio_write(sc, 0x10, 0x401F);
                re_mdio_write(sc, 0x19, 0x7030);

                re_mdio_write(sc, 0x1F, 0x0000);
                re_mdio_write(sc, 0x0D, 0x0007);
                re_mdio_write(sc, 0x0E, 0x003C);
                re_mdio_write(sc, 0x0D, 0x4007);
                re_mdio_write(sc, 0x0E, 0x0000);
                re_mdio_write(sc, 0x0D, 0x0000);

                re_mdio_write(sc, MII_BMCR, BMCR_AUTOEN | BMCR_STARTNEG);
                break;

        case MACFG_54:
        case MACFG_55:
                re_eri_write(sc, 0x1B0, 2, 0, ERIAR_ExGMAC);
                re_mdio_write(sc, 0x1F, 0x0004);
                re_mdio_write(sc, 0x10, 0xC07F);
                re_mdio_write(sc, 0x19, 0x7030);
                re_mdio_write(sc, 0x1F, 0x0000);

                re_mdio_write(sc, 0x1F, 0x0000);
                re_mdio_write(sc, 0x0D, 0x0007);
                re_mdio_write(sc, 0x0E, 0x003C);
                re_mdio_write(sc, 0x0D, 0x4007);
                re_mdio_write(sc, 0x0E, 0x0000);
                re_mdio_write(sc, 0x0D, 0x0000);

                re_mdio_write(sc, MII_BMCR, BMCR_AUTOEN | BMCR_STARTNEG);
                break;

        case MACFG_36:
        case MACFG_37:
                re_mdio_write(sc, 0x1F, 0x0007);
                re_mdio_write(sc, 0x1E, 0x0020);
                data = re_mdio_read(sc, 0x15) & ~0x0100;
                re_mdio_write(sc, 0x15, data);
                re_mdio_write(sc, 0x1F, 0x0006);
                re_mdio_write(sc, 0x00, 0x5A00);
                re_mdio_write(sc, 0x1F, 0x0000);
                re_mdio_write(sc, 0x0D, 0x0007);
                re_mdio_write(sc, 0x0E, 0x003C);
                re_mdio_write(sc, 0x0D, 0x4007);
                re_mdio_write(sc, 0x0E, 0x0000);
                re_mdio_write(sc, 0x0D, 0x0000);
                re_mdio_write(sc, 0x1F, 0x0000);
                if (CSR_READ_1(sc, RE_CFG4) & RL_CFG4_CUSTOMIZED_LED) {
                        re_mdio_write(sc, 0x1F, 0x0005);
                        re_mdio_write(sc, 0x05, 0x8B82);
                        data = re_mdio_read(sc, 0x06) & ~0x0010;
                        re_mdio_write(sc, 0x05, 0x8B82);
                        re_mdio_write(sc, 0x06, data);
                        re_mdio_write(sc, 0x1F, 0x0000);
                }
                break;

        case MACFG_50:
        case MACFG_51:
        case MACFG_52:
                data = re_eri_read(sc, 0x1B0, 4, ERIAR_ExGMAC)& ~0x0003;
                re_eri_write(sc, 0x1B0, 4, data, ERIAR_ExGMAC);
                re_mdio_write(sc, 0x1F, 0x0005);
                re_mdio_write(sc, 0x05, 0x8B85);
                data = re_mdio_read(sc, 0x06) & ~0x2000;
                re_mdio_write(sc, 0x06, data);
                re_mdio_write(sc, 0x1F, 0x0007);
                re_mdio_write(sc, 0x1E, 0x0020);
                data = re_mdio_read(sc, 0x15) & ~0x0100;
                re_mdio_write(sc, 0x15, data);
                re_mdio_write(sc, 0x1F, 0x0000);
                re_mdio_write(sc, 0x0D, 0x0007);
                re_mdio_write(sc, 0x0E, 0x003C);
                re_mdio_write(sc, 0x0D, 0x4007);
                re_mdio_write(sc, 0x0E, 0x0000);
                re_mdio_write(sc, 0x0D, 0x0000);
                re_mdio_write(sc, 0x1F, 0x0000);
                break;

        case MACFG_38:
        case MACFG_39:
                data = re_eri_read(sc, 0x1B0, 4, ERIAR_ExGMAC);
                data &= ~(BIT_1 | BIT_0);
                re_eri_write(sc, 0x1B0, 4, data, ERIAR_ExGMAC);
                re_mdio_write(sc, 0x1F, 0x0005);
                re_mdio_write(sc, 0x05, 0x8B85);
                data = re_mdio_read(sc, 0x06);
                data &= ~BIT_13;
                re_mdio_write(sc, 0x06, data);
                re_mdio_write(sc, 0x1F, 0x0004);
                re_mdio_write(sc, 0x1F, 0x0007);
                re_mdio_write(sc, 0x1e, 0x0020);
                data = re_mdio_read(sc, 0x15);
                data &= ~BIT_8;
                re_mdio_write(sc, 0x15, data);
                re_mdio_write(sc, 0x1F, 0x0002);
                re_mdio_write(sc, 0x1F, 0x0000);
                re_mdio_write(sc, 0x0D, 0x0007);
                re_mdio_write(sc, 0x0E, 0x003C);
                re_mdio_write(sc, 0x0D, 0x4007);
                re_mdio_write(sc, 0x0E, 0x0000);
                re_mdio_write(sc, 0x0D, 0x0000);
                re_mdio_write(sc, 0x1F, 0x0000);
                break;

        case MACFG_56:
        case MACFG_57:
        case MACFG_58:
        case MACFG_59:
        case MACFG_60:
        case MACFG_61:
        case MACFG_62:
        case MACFG_67:
        case MACFG_68:
        case MACFG_69:
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
        case MACFG_74:
        case MACFG_75:
                data = re_eri_read(sc, 0x1B0, 4, ERIAR_ExGMAC);
                data &= ~(BIT_1 | BIT_0);
                re_eri_write(sc, 0x1B0, 4, data, ERIAR_ExGMAC);
                re_mdio_write(sc, 0x1F, 0x0A43);
                data = re_mdio_read(sc, 0x11);
                if (sc->re_type == MACFG_75)
                        re_mdio_write(sc, 0x11, data | BIT_4);
                else
                        re_mdio_write(sc, 0x11, data & ~BIT_4);
                re_mdio_write(sc, 0x1F, 0x0A5D);
                re_mdio_write(sc, 0x10, 0x0000);
                re_mdio_write(sc, 0x1F, 0x0000);
                break;

        case MACFG_80:
        case MACFG_81:
                re_clear_mac_ocp_bit(sc, 0xE040, (BIT_1|BIT_0));
                re_clear_mac_ocp_bit(sc, 0xEB62, (BIT_2|BIT_1));

                re_clear_eth_ocp_phy_bit(sc, 0xA432, BIT_4);
                re_clear_eth_ocp_phy_bit(sc, 0xA5D0, (BIT_2 | BIT_1));
                re_clear_eth_ocp_phy_bit(sc, 0xA6D4, BIT_0);

                re_clear_eth_ocp_phy_bit(sc, 0xA6D8, BIT_4);
                re_clear_eth_ocp_phy_bit(sc, 0xA428, BIT_7);
                re_clear_eth_ocp_phy_bit(sc, 0xA4A2, BIT_9);
                break;

        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
                re_clear_mac_ocp_bit(sc, 0xE040, (BIT_1|BIT_0));

                re_set_eth_ocp_phy_bit(sc, 0xA432, BIT_4);

                re_clear_eth_ocp_phy_bit(sc, 0xA5D0, (BIT_2 | BIT_1));
                re_clear_eth_ocp_phy_bit(sc, 0xA6D4, BIT_0);

                re_clear_eth_ocp_phy_bit(sc, 0xA6D8, BIT_4);
                re_clear_eth_ocp_phy_bit(sc, 0xA428, BIT_7);
                re_clear_eth_ocp_phy_bit(sc, 0xA4A2, BIT_9);
                break;

        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                re_clear_mac_ocp_bit(sc, 0xE040, (BIT_1|BIT_0));

                re_clear_eth_ocp_phy_bit(sc, 0xA5D0, (BIT_2 | BIT_1));
                re_clear_eth_ocp_phy_bit(sc, 0xA6D4, (BIT_0 | BIT_1));

                re_clear_eth_ocp_phy_bit(sc, 0xA428, BIT_7);
                re_clear_eth_ocp_phy_bit(sc, 0xA4A2, BIT_9);
                break;

        default:
                ret = -EOPNOTSUPP;
                break;
        }

        switch (sc->re_type) {
        case MACFG_68:
        case MACFG_69:
        case MACFG_74:
        case MACFG_75:
                re_mdio_write(sc, 0x1F, 0x0A42);
                re_clear_eth_phy_bit(sc, 0x14, BIT_7);
                re_mdio_write(sc, 0x1F, 0x0A4A);
                re_clear_eth_phy_bit(sc, 0x11, BIT_9);
                re_mdio_write(sc, 0x1F, 0x0000);
                break;
        }

        /*Advanced EEE*/
        re_disable_advanced_eee(sc);

        return ret;
}

static int re_phy_ram_code_check(struct re_softc *sc)
{
        u_int16_t PhyRegValue;
        int retval = TRUE;

        switch(sc->re_type) {
        case MACFG_56:
                re_mdio_write(sc, 0x1f, 0x0A40);
                PhyRegValue = re_mdio_read(sc, 0x10);
                PhyRegValue &= ~(BIT_11);
                re_mdio_write(sc, 0x10, PhyRegValue);


                re_mdio_write(sc, 0x1f, 0x0A00);
                PhyRegValue = re_mdio_read(sc, 0x10);
                PhyRegValue &= ~(BIT_12 | BIT_13 | BIT_14 | BIT_15);
                re_mdio_write(sc, 0x10, PhyRegValue);

                re_mdio_write(sc, 0x1f, 0x0A43);
                re_mdio_write(sc, 0x13, 0x8010);
                PhyRegValue = re_mdio_read(sc, 0x14);
                PhyRegValue &= ~(BIT_11);
                re_mdio_write(sc, 0x14, PhyRegValue);

                re_set_phy_mcu_patch_request(sc);

                re_mdio_write(sc, 0x1f, 0x0A40);
                re_mdio_write(sc, 0x10, 0x0140);

                re_mdio_write(sc, 0x1f, 0x0A4A);
                PhyRegValue = re_mdio_read(sc, 0x13);
                PhyRegValue &= ~(BIT_6);
                PhyRegValue |= (BIT_7);
                re_mdio_write(sc, 0x13, PhyRegValue);

                re_mdio_write(sc, 0x1f, 0x0A44);
                PhyRegValue = re_mdio_read(sc, 0x14);
                PhyRegValue |= (BIT_2);
                re_mdio_write(sc, 0x14, PhyRegValue);

                re_mdio_write(sc, 0x1f, 0x0A50);
                PhyRegValue = re_mdio_read(sc, 0x11);
                PhyRegValue |= (BIT_11|BIT_12);
                re_mdio_write(sc, 0x11, PhyRegValue);

                re_clear_phy_mcu_patch_request(sc);

                re_mdio_write(sc, 0x1f, 0x0A40);
                re_mdio_write(sc, 0x10, 0x1040);

                re_mdio_write(sc, 0x1f, 0x0A4A);
                PhyRegValue = re_mdio_read(sc, 0x13);
                PhyRegValue &= ~(BIT_6|BIT_7);
                re_mdio_write(sc, 0x13, PhyRegValue);

                re_mdio_write(sc, 0x1f, 0x0A44);
                PhyRegValue = re_mdio_read(sc, 0x14);
                PhyRegValue &= ~(BIT_2);
                re_mdio_write(sc, 0x14, PhyRegValue);

                re_mdio_write(sc, 0x1f, 0x0A50);
                PhyRegValue = re_mdio_read(sc, 0x11);
                PhyRegValue &= ~(BIT_11|BIT_12);
                re_mdio_write(sc, 0x11, PhyRegValue);

                re_mdio_write(sc, 0x1f, 0x0A43);
                re_mdio_write(sc, 0x13, 0x8010);
                PhyRegValue = re_mdio_read(sc, 0x14);
                PhyRegValue |= (BIT_11);
                re_mdio_write(sc, 0x14, PhyRegValue);

                re_set_phy_mcu_patch_request(sc);

                re_mdio_write(sc, 0x1f, 0x0A20);
                PhyRegValue = re_mdio_read(sc, 0x13);
                if (PhyRegValue & BIT_11) {
                        if (PhyRegValue & BIT_10) {
                                retval = FALSE;
                        }
                }

                re_clear_phy_mcu_patch_request(sc);

                //delay 2ms
                DELAY(2000);
                break;
        default:
                break;
        }

        re_mdio_write(sc, 0x1F, 0x0000);

        return retval;
}

static void re_set_phy_ram_code_check_fail_flag(struct re_softc *sc)
{
        u_int16_t TmpUshort;

        switch(sc->re_type) {
        case MACFG_56:
                TmpUshort = re_mac_ocp_read(sc, 0xD3C0);
                TmpUshort |= BIT_0;
                re_mac_ocp_write(sc, 0xD3C0, TmpUshort);
                break;
        }
}

static int re_hw_phy_mcu_code_ver_matched(struct re_softc *sc)
{
        int ram_code_ver_match = 0;

        switch (sc->re_type) {
        case MACFG_36:
        case MACFG_37:
                re_mdio_write(sc, 0x1F, 0x0005);
                re_mdio_write(sc, 0x05, 0x8B60);
                sc->re_hw_ram_code_ver = re_mdio_read(sc, 0x06);
                re_mdio_write(sc, 0x1F, 0x0000);
                break;
        case MACFG_38:
        case MACFG_39:
        case MACFG_50:
        case MACFG_51:
        case MACFG_52:
                re_mdio_write(sc, 0x1F, 0x0005);
                re_mdio_write(sc, 0x05, 0x8B30);
                sc->re_hw_ram_code_ver = re_mdio_read(sc, 0x06);
                re_mdio_write(sc, 0x1F, 0x0000);
                break;
        case MACFG_56:
        case MACFG_57:
        case MACFG_58:
        case MACFG_59:
        case MACFG_60:
        case MACFG_61:
        case MACFG_62:
        case MACFG_67:
        case MACFG_68:
        case MACFG_69:
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
        case MACFG_74:
        case MACFG_75:
        case MACFG_76:
                re_mdio_write(sc, 0x1F, 0x0A43);
                re_mdio_write(sc, 0x13, 0x801E);
                sc->re_hw_ram_code_ver = re_mdio_read(sc, 0x14);
                re_mdio_write(sc, 0x1F, 0x0000);
                break;
        case MACFG_80:
        case MACFG_81:
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                re_real_ocp_phy_write(sc, 0xA436, 0x801E);
                sc->re_hw_ram_code_ver = re_real_ocp_phy_read(sc, 0xA438);
                break;
        default:
                sc->re_hw_ram_code_ver = ~0;
                break;
        }

        if (sc->re_hw_ram_code_ver == sc->re_sw_ram_code_ver)
                ram_code_ver_match = 1;

        return ram_code_ver_match;
}

static void re_write_hw_phy_mcu_code_ver(struct re_softc *sc)
{
        switch (sc->re_type) {
        case MACFG_36:
        case MACFG_37:
                re_mdio_write(sc, 0x1F, 0x0005);
                re_mdio_write(sc, 0x05, 0x8B60);
                re_mdio_write(sc, 0x06, sc->re_sw_ram_code_ver);
                re_mdio_write(sc, 0x1F, 0x0000);
                sc->re_hw_ram_code_ver = sc->re_sw_ram_code_ver;
                break;
        case MACFG_38:
        case MACFG_39:
        case MACFG_50:
        case MACFG_51:
        case MACFG_52:
                re_mdio_write(sc, 0x1F, 0x0005);
                re_mdio_write(sc, 0x05, 0x8B30);
                re_mdio_write(sc, 0x06, sc->re_sw_ram_code_ver);
                re_mdio_write(sc, 0x1F, 0x0000);
                sc->re_hw_ram_code_ver = sc->re_sw_ram_code_ver;
                break;
        case MACFG_56:
        case MACFG_57:
        case MACFG_58:
        case MACFG_59:
        case MACFG_60:
        case MACFG_61:
        case MACFG_62:
        case MACFG_67:
        case MACFG_68:
        case MACFG_69:
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
        case MACFG_74:
        case MACFG_75:
        case MACFG_76:
                re_mdio_write(sc, 0x1F, 0x0A43);
                re_mdio_write(sc, 0x13, 0x801E);
                re_mdio_write(sc, 0x14, sc->re_sw_ram_code_ver);
                re_mdio_write(sc, 0x1F, 0x0000);
                sc->re_hw_ram_code_ver = sc->re_sw_ram_code_ver;
                break;
        case MACFG_80:
        case MACFG_81:
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                re_real_ocp_phy_write(sc, 0xA436, 0x801E);
                re_real_ocp_phy_write(sc, 0xA438, sc->re_sw_ram_code_ver);
                sc->re_hw_ram_code_ver = sc->re_sw_ram_code_ver;
                break;
        }
}

static void
re_acquire_phy_mcu_patch_key_lock(struct re_softc *sc)
{
        u_int16_t PatchKey;

        switch (sc->re_type) {
        case MACFG_80:
                PatchKey = 0x8600;
                break;
        case MACFG_81:
                PatchKey = 0x8601;
                break;
        case MACFG_82:
                PatchKey = 0x3700;
                break;
        case MACFG_83:
                PatchKey = 0x3701;
                break;
        default:
                return;
        }
        re_real_ocp_phy_write(sc, 0xA436, 0x8024);
        re_real_ocp_phy_write(sc, 0xA438, PatchKey);
        re_real_ocp_phy_write(sc, 0xA436, 0xB82E);
        re_real_ocp_phy_write(sc, 0xA438, 0x0001);
}

static void
re_release_phy_mcu_patch_key_lock(struct re_softc *sc)
{
        switch (sc->re_type) {
        case MACFG_80:
        case MACFG_81:
        case MACFG_82:
        case MACFG_83:
                re_real_ocp_phy_write(sc, 0xA436, 0x0000);
                re_real_ocp_phy_write(sc, 0xA438, 0x0000);
                re_clear_eth_ocp_phy_bit(sc, 0xB82E, BIT_0);
                re_real_ocp_phy_write(sc, 0xA436, 0x8024);
                re_real_ocp_phy_write(sc, 0xA438, 0x0000);
                break;
        default:
                break;
        }
}

bool
re_set_phy_mcu_patch_request(struct re_softc *sc)
{
        u_int16_t PhyRegValue;
        u_int16_t WaitCount = 0;
        bool bSuccess = TRUE;

        switch (sc->re_type) {
        case MACFG_56:
        case MACFG_57:
        case MACFG_58:
        case MACFG_59:
        case MACFG_60:
        case MACFG_61:
        case MACFG_62:
        case MACFG_67:
        case MACFG_68:
        case MACFG_69:
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
        case MACFG_74:
        case MACFG_75:
        case MACFG_76:
                re_mdio_write(sc, 0x1f, 0x0B82);
                re_set_eth_phy_bit(sc, 0x10, BIT_4);

                re_mdio_write(sc, 0x1f, 0x0B80);
                WaitCount = 0;
                do {
                        PhyRegValue = re_mdio_read(sc, 0x10);
                        DELAY(50);
                        DELAY(50);
                        WaitCount++;
                } while (!(PhyRegValue & BIT_6) && (WaitCount < 1000));

                if (!(PhyRegValue & BIT_6) && (WaitCount == 1000)) bSuccess = FALSE;

                re_mdio_write(sc, 0x1f, 0x0000);
                break;
        case MACFG_80:
        case MACFG_81:
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                re_set_eth_ocp_phy_bit(sc, 0xB820, BIT_4);

                WaitCount = 0;
                do {
                        PhyRegValue = re_real_ocp_phy_read(sc, 0xB800);
                        DELAY(50);
                        DELAY(50);
                        WaitCount++;
                } while (!(PhyRegValue & BIT_6) && (WaitCount < 1000));

                if (!(PhyRegValue & BIT_6) && (WaitCount == 1000)) bSuccess = FALSE;

                break;
        }

        return bSuccess;
}

bool
re_clear_phy_mcu_patch_request(struct re_softc *sc)
{
        u_int16_t PhyRegValue;
        u_int16_t WaitCount = 0;
        bool bSuccess = TRUE;

        switch (sc->re_type) {
        case MACFG_56:
        case MACFG_57:
        case MACFG_58:
        case MACFG_59:
        case MACFG_60:
        case MACFG_61:
        case MACFG_62:
        case MACFG_67:
        case MACFG_68:
        case MACFG_69:
        case MACFG_70:
        case MACFG_71:
        case MACFG_72:
        case MACFG_73:
        case MACFG_74:
        case MACFG_75:
        case MACFG_76:
                re_mdio_write(sc, 0x1f, 0x0B82);
                re_clear_eth_phy_bit(sc, 0x10, BIT_4);

                re_mdio_write(sc, 0x1f, 0x0B80);
                WaitCount = 0;
                do {
                        PhyRegValue = re_mdio_read(sc, 0x10);
                        DELAY(50);
                        DELAY(50);
                        WaitCount++;
                } while ((PhyRegValue & BIT_6) && (WaitCount < 1000));

                if ((PhyRegValue & BIT_6) && (WaitCount == 1000)) bSuccess = FALSE;

                re_mdio_write(sc, 0x1f, 0x0000);
                break;
        case MACFG_80:
        case MACFG_81:
        case MACFG_82:
        case MACFG_83:
        case MACFG_84:
        case MACFG_85:
        case MACFG_86:
        case MACFG_87:
        case MACFG_88:
        case MACFG_90:
        case MACFG_91:
        case MACFG_92:
        case MACFG_100:
        case MACFG_101:
                re_clear_eth_ocp_phy_bit(sc, 0xB820, BIT_4);

                WaitCount = 0;
                do {
                        PhyRegValue = re_real_ocp_phy_read(sc, 0xB800);
                        DELAY(50);
                        DELAY(50);
                        WaitCount++;
                } while ((PhyRegValue & BIT_6) && (WaitCount < 1000));

                if ((PhyRegValue & BIT_6) && (WaitCount == 1000)) bSuccess = FALSE;

                break;
        }

        return bSuccess;
}

static void
re_set_phy_mcu_ram_code(struct re_softc *sc, const u_int16_t *ramcode, u_int16_t codesize)
{
        u_int16_t i;
        u_int16_t addr;
        u_int16_t val;

        if (ramcode == NULL || codesize % 2) {
                goto out;
        }

        for (i = 0; i < codesize; i += 2) {
                addr = ramcode[i];
                val = ramcode[i + 1];
                if (addr == 0xFFFF && val == 0xFFFF) {
                        break;
                }
                re_real_ocp_phy_write(sc, addr, val);
        }

out:
        return;
}

static void re_set_phy_mcu_8168e_1(struct re_softc *sc)
{
        u_int16_t PhyRegValue;
        int i;

        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x00, 0x1800);
        re_mdio_write(sc, 0x1f, 0x0007);
        re_mdio_write(sc, 0x1e, 0x0023);
        re_mdio_write(sc, 0x17, 0x0117);
        re_mdio_write(sc, 0x1f, 0x0007);
        re_mdio_write(sc, 0x1E, 0x002C);
        re_mdio_write(sc, 0x1B, 0x5000);
        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x16, 0x4104);
        for (i=0; i<200; i++) {
                DELAY(100);
                PhyRegValue = re_mdio_read(sc, 0x1E);
                PhyRegValue &= 0x03FF;
                if (PhyRegValue== 0x000C)
                        break;
        }
        re_mdio_write(sc, 0x1f, 0x0005);
        for (i=0; i<200; i++) {
                DELAY(100);
                PhyRegValue = re_mdio_read(sc, 0x07);
                if ((PhyRegValue&0x0020)==0)
                        break;
        }
        PhyRegValue = re_mdio_read(sc, 0x07);
        if (PhyRegValue & 0x0020) {
                re_mdio_write(sc, 0x1f, 0x0007);
                re_mdio_write(sc, 0x1e, 0x00a1);
                re_mdio_write(sc, 0x17, 0x1000);
                re_mdio_write(sc, 0x17, 0x0000);
                re_mdio_write(sc, 0x17, 0x2000);
                re_mdio_write(sc, 0x1e, 0x002f);
                re_mdio_write(sc, 0x18, 0x9bfb);
                re_mdio_write(sc, 0x1f, 0x0005);
                re_mdio_write(sc, 0x07, 0x0000);
                re_mdio_write(sc, 0x1f, 0x0000);
        }

        re_mdio_write(sc, 0x1f, 0x0005);
        re_mdio_write(sc, 0x05, 0xfff6);
        re_mdio_write(sc, 0x06, 0x0080);
        PhyRegValue = re_mdio_read(sc, 0x00);
        PhyRegValue &= ~(BIT_7);
        re_mdio_write(sc, 0x00, PhyRegValue);
        re_mdio_write(sc, 0x1f, 0x0002);
        PhyRegValue = re_mdio_read(sc, 0x08);
        PhyRegValue &= ~(BIT_7);
        re_mdio_write(sc, 0x08, PhyRegValue);
        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x1f, 0x0007);
        re_mdio_write(sc, 0x1e, 0x0023);
        re_mdio_write(sc, 0x16, 0x0306);
        re_mdio_write(sc, 0x16, 0x0307);
        re_mdio_write(sc, 0x15, 0x000e);
        re_mdio_write(sc, 0x19, 0x000a);
        re_mdio_write(sc, 0x15, 0x0010);
        re_mdio_write(sc, 0x19, 0x0008);
        re_mdio_write(sc, 0x15, 0x0018);
        re_mdio_write(sc, 0x19, 0x4801);
        re_mdio_write(sc, 0x15, 0x0019);
        re_mdio_write(sc, 0x19, 0x6801);
        re_mdio_write(sc, 0x15, 0x001a);
        re_mdio_write(sc, 0x19, 0x66a1);
        re_mdio_write(sc, 0x15, 0x001f);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0020);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0021);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0022);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0023);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0024);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0025);
        re_mdio_write(sc, 0x19, 0x64a1);
        re_mdio_write(sc, 0x15, 0x0026);
        re_mdio_write(sc, 0x19, 0x40ea);
        re_mdio_write(sc, 0x15, 0x0027);
        re_mdio_write(sc, 0x19, 0x4503);
        re_mdio_write(sc, 0x15, 0x0028);
        re_mdio_write(sc, 0x19, 0x9f00);
        re_mdio_write(sc, 0x15, 0x0029);
        re_mdio_write(sc, 0x19, 0xa631);
        re_mdio_write(sc, 0x15, 0x002a);
        re_mdio_write(sc, 0x19, 0x9717);
        re_mdio_write(sc, 0x15, 0x002b);
        re_mdio_write(sc, 0x19, 0x302c);
        re_mdio_write(sc, 0x15, 0x002c);
        re_mdio_write(sc, 0x19, 0x4802);
        re_mdio_write(sc, 0x15, 0x002d);
        re_mdio_write(sc, 0x19, 0x58da);
        re_mdio_write(sc, 0x15, 0x002e);
        re_mdio_write(sc, 0x19, 0x400d);
        re_mdio_write(sc, 0x15, 0x002f);
        re_mdio_write(sc, 0x19, 0x4488);
        re_mdio_write(sc, 0x15, 0x0030);
        re_mdio_write(sc, 0x19, 0x9e00);
        re_mdio_write(sc, 0x15, 0x0031);
        re_mdio_write(sc, 0x19, 0x63c8);
        re_mdio_write(sc, 0x15, 0x0032);
        re_mdio_write(sc, 0x19, 0x6481);
        re_mdio_write(sc, 0x15, 0x0033);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0034);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0035);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0036);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0037);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0038);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0039);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x003a);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x003b);
        re_mdio_write(sc, 0x19, 0x63e8);
        re_mdio_write(sc, 0x15, 0x003c);
        re_mdio_write(sc, 0x19, 0x7d00);
        re_mdio_write(sc, 0x15, 0x003d);
        re_mdio_write(sc, 0x19, 0x59d4);
        re_mdio_write(sc, 0x15, 0x003e);
        re_mdio_write(sc, 0x19, 0x63f8);
        re_mdio_write(sc, 0x15, 0x0040);
        re_mdio_write(sc, 0x19, 0x64a1);
        re_mdio_write(sc, 0x15, 0x0041);
        re_mdio_write(sc, 0x19, 0x30de);
        re_mdio_write(sc, 0x15, 0x0044);
        re_mdio_write(sc, 0x19, 0x480f);
        re_mdio_write(sc, 0x15, 0x0045);
        re_mdio_write(sc, 0x19, 0x6800);
        re_mdio_write(sc, 0x15, 0x0046);
        re_mdio_write(sc, 0x19, 0x6680);
        re_mdio_write(sc, 0x15, 0x0047);
        re_mdio_write(sc, 0x19, 0x7c10);
        re_mdio_write(sc, 0x15, 0x0048);
        re_mdio_write(sc, 0x19, 0x63c8);
        re_mdio_write(sc, 0x15, 0x0049);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x004a);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x004b);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x004c);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x004d);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x004e);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x004f);
        re_mdio_write(sc, 0x19, 0x40ea);
        re_mdio_write(sc, 0x15, 0x0050);
        re_mdio_write(sc, 0x19, 0x4503);
        re_mdio_write(sc, 0x15, 0x0051);
        re_mdio_write(sc, 0x19, 0x58ca);
        re_mdio_write(sc, 0x15, 0x0052);
        re_mdio_write(sc, 0x19, 0x63c8);
        re_mdio_write(sc, 0x15, 0x0053);
        re_mdio_write(sc, 0x19, 0x63d8);
        re_mdio_write(sc, 0x15, 0x0054);
        re_mdio_write(sc, 0x19, 0x66a0);
        re_mdio_write(sc, 0x15, 0x0055);
        re_mdio_write(sc, 0x19, 0x9f00);
        re_mdio_write(sc, 0x15, 0x0056);
        re_mdio_write(sc, 0x19, 0x3000);
        re_mdio_write(sc, 0x15, 0x006E);
        re_mdio_write(sc, 0x19, 0x9afa);
        re_mdio_write(sc, 0x15, 0x00a1);
        re_mdio_write(sc, 0x19, 0x3044);
        re_mdio_write(sc, 0x15, 0x00ab);
        re_mdio_write(sc, 0x19, 0x5820);
        re_mdio_write(sc, 0x15, 0x00ac);
        re_mdio_write(sc, 0x19, 0x5e04);
        re_mdio_write(sc, 0x15, 0x00ad);
        re_mdio_write(sc, 0x19, 0xb60c);
        re_mdio_write(sc, 0x15, 0x00af);
        re_mdio_write(sc, 0x19, 0x000a);
        re_mdio_write(sc, 0x15, 0x00b2);
        re_mdio_write(sc, 0x19, 0x30b9);
        re_mdio_write(sc, 0x15, 0x00b9);
        re_mdio_write(sc, 0x19, 0x4408);
        re_mdio_write(sc, 0x15, 0x00ba);
        re_mdio_write(sc, 0x19, 0x480b);
        re_mdio_write(sc, 0x15, 0x00bb);
        re_mdio_write(sc, 0x19, 0x5e00);
        re_mdio_write(sc, 0x15, 0x00bc);
        re_mdio_write(sc, 0x19, 0x405f);
        re_mdio_write(sc, 0x15, 0x00bd);
        re_mdio_write(sc, 0x19, 0x4448);
        re_mdio_write(sc, 0x15, 0x00be);
        re_mdio_write(sc, 0x19, 0x4020);
        re_mdio_write(sc, 0x15, 0x00bf);
        re_mdio_write(sc, 0x19, 0x4468);
        re_mdio_write(sc, 0x15, 0x00c0);
        re_mdio_write(sc, 0x19, 0x9c02);
        re_mdio_write(sc, 0x15, 0x00c1);
        re_mdio_write(sc, 0x19, 0x58a0);
        re_mdio_write(sc, 0x15, 0x00c2);
        re_mdio_write(sc, 0x19, 0xb605);
        re_mdio_write(sc, 0x15, 0x00c3);
        re_mdio_write(sc, 0x19, 0xc0d3);
        re_mdio_write(sc, 0x15, 0x00c4);
        re_mdio_write(sc, 0x19, 0x00e6);
        re_mdio_write(sc, 0x15, 0x00c5);
        re_mdio_write(sc, 0x19, 0xdaec);
        re_mdio_write(sc, 0x15, 0x00c6);
        re_mdio_write(sc, 0x19, 0x00fa);
        re_mdio_write(sc, 0x15, 0x00c7);
        re_mdio_write(sc, 0x19, 0x9df9);
        re_mdio_write(sc, 0x15, 0x00c8);
        re_mdio_write(sc, 0x19, 0x307a);
        re_mdio_write(sc, 0x15, 0x0112);
        re_mdio_write(sc, 0x19, 0x6421);
        re_mdio_write(sc, 0x15, 0x0113);
        re_mdio_write(sc, 0x19, 0x7c08);
        re_mdio_write(sc, 0x15, 0x0114);
        re_mdio_write(sc, 0x19, 0x63f0);
        re_mdio_write(sc, 0x15, 0x0115);
        re_mdio_write(sc, 0x19, 0x4003);
        re_mdio_write(sc, 0x15, 0x0116);
        re_mdio_write(sc, 0x19, 0x4418);
        re_mdio_write(sc, 0x15, 0x0117);
        re_mdio_write(sc, 0x19, 0x9b00);
        re_mdio_write(sc, 0x15, 0x0118);
        re_mdio_write(sc, 0x19, 0x6461);
        re_mdio_write(sc, 0x15, 0x0119);
        re_mdio_write(sc, 0x19, 0x64e1);
        re_mdio_write(sc, 0x15, 0x011a);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0150);
        re_mdio_write(sc, 0x19, 0x7c80);
        re_mdio_write(sc, 0x15, 0x0151);
        re_mdio_write(sc, 0x19, 0x6461);
        re_mdio_write(sc, 0x15, 0x0152);
        re_mdio_write(sc, 0x19, 0x4003);
        re_mdio_write(sc, 0x15, 0x0153);
        re_mdio_write(sc, 0x19, 0x4540);
        re_mdio_write(sc, 0x15, 0x0154);
        re_mdio_write(sc, 0x19, 0x9f00);
        re_mdio_write(sc, 0x15, 0x0155);
        re_mdio_write(sc, 0x19, 0x9d00);
        re_mdio_write(sc, 0x15, 0x0156);
        re_mdio_write(sc, 0x19, 0x7c40);
        re_mdio_write(sc, 0x15, 0x0157);
        re_mdio_write(sc, 0x19, 0x6421);
        re_mdio_write(sc, 0x15, 0x0158);
        re_mdio_write(sc, 0x19, 0x7c80);
        re_mdio_write(sc, 0x15, 0x0159);
        re_mdio_write(sc, 0x19, 0x64a1);
        re_mdio_write(sc, 0x15, 0x015a);
        re_mdio_write(sc, 0x19, 0x30fe);
        re_mdio_write(sc, 0x15, 0x021e);
        re_mdio_write(sc, 0x19, 0x5410);
        re_mdio_write(sc, 0x15, 0x0225);
        re_mdio_write(sc, 0x19, 0x5400);
        re_mdio_write(sc, 0x15, 0x023D);
        re_mdio_write(sc, 0x19, 0x4050);
        re_mdio_write(sc, 0x15, 0x0295);
        re_mdio_write(sc, 0x19, 0x6c08);
        re_mdio_write(sc, 0x15, 0x02bd);
        re_mdio_write(sc, 0x19, 0xa523);
        re_mdio_write(sc, 0x15, 0x02be);
        re_mdio_write(sc, 0x19, 0x32ca);
        re_mdio_write(sc, 0x15, 0x02ca);
        re_mdio_write(sc, 0x19, 0x48b3);
        re_mdio_write(sc, 0x15, 0x02cb);
        re_mdio_write(sc, 0x19, 0x4020);
        re_mdio_write(sc, 0x15, 0x02cc);
        re_mdio_write(sc, 0x19, 0x4823);
        re_mdio_write(sc, 0x15, 0x02cd);
        re_mdio_write(sc, 0x19, 0x4510);
        re_mdio_write(sc, 0x15, 0x02ce);
        re_mdio_write(sc, 0x19, 0xb63a);
        re_mdio_write(sc, 0x15, 0x02cf);
        re_mdio_write(sc, 0x19, 0x7dc8);
        re_mdio_write(sc, 0x15, 0x02d6);
        re_mdio_write(sc, 0x19, 0x9bf8);
        re_mdio_write(sc, 0x15, 0x02d8);
        re_mdio_write(sc, 0x19, 0x85f6);
        re_mdio_write(sc, 0x15, 0x02d9);
        re_mdio_write(sc, 0x19, 0x32e0);
        re_mdio_write(sc, 0x15, 0x02e0);
        re_mdio_write(sc, 0x19, 0x4834);
        re_mdio_write(sc, 0x15, 0x02e1);
        re_mdio_write(sc, 0x19, 0x6c08);
        re_mdio_write(sc, 0x15, 0x02e2);
        re_mdio_write(sc, 0x19, 0x4020);
        re_mdio_write(sc, 0x15, 0x02e3);
        re_mdio_write(sc, 0x19, 0x4824);
        re_mdio_write(sc, 0x15, 0x02e4);
        re_mdio_write(sc, 0x19, 0x4520);
        re_mdio_write(sc, 0x15, 0x02e5);
        re_mdio_write(sc, 0x19, 0x4008);
        re_mdio_write(sc, 0x15, 0x02e6);
        re_mdio_write(sc, 0x19, 0x4560);
        re_mdio_write(sc, 0x15, 0x02e7);
        re_mdio_write(sc, 0x19, 0x9d04);
        re_mdio_write(sc, 0x15, 0x02e8);
        re_mdio_write(sc, 0x19, 0x48c4);
        re_mdio_write(sc, 0x15, 0x02e9);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x02ea);
        re_mdio_write(sc, 0x19, 0x4844);
        re_mdio_write(sc, 0x15, 0x02eb);
        re_mdio_write(sc, 0x19, 0x7dc8);
        re_mdio_write(sc, 0x15, 0x02f0);
        re_mdio_write(sc, 0x19, 0x9cf7);
        re_mdio_write(sc, 0x15, 0x02f1);
        re_mdio_write(sc, 0x19, 0xdf94);
        re_mdio_write(sc, 0x15, 0x02f2);
        re_mdio_write(sc, 0x19, 0x0002);
        re_mdio_write(sc, 0x15, 0x02f3);
        re_mdio_write(sc, 0x19, 0x6810);
        re_mdio_write(sc, 0x15, 0x02f4);
        re_mdio_write(sc, 0x19, 0xb614);
        re_mdio_write(sc, 0x15, 0x02f5);
        re_mdio_write(sc, 0x19, 0xc42b);
        re_mdio_write(sc, 0x15, 0x02f6);
        re_mdio_write(sc, 0x19, 0x00d4);
        re_mdio_write(sc, 0x15, 0x02f7);
        re_mdio_write(sc, 0x19, 0xc455);
        re_mdio_write(sc, 0x15, 0x02f8);
        re_mdio_write(sc, 0x19, 0x0093);
        re_mdio_write(sc, 0x15, 0x02f9);
        re_mdio_write(sc, 0x19, 0x92ee);
        re_mdio_write(sc, 0x15, 0x02fa);
        re_mdio_write(sc, 0x19, 0xefed);
        re_mdio_write(sc, 0x15, 0x02fb);
        re_mdio_write(sc, 0x19, 0x3312);
        re_mdio_write(sc, 0x15, 0x0312);
        re_mdio_write(sc, 0x19, 0x49b5);
        re_mdio_write(sc, 0x15, 0x0313);
        re_mdio_write(sc, 0x19, 0x7d00);
        re_mdio_write(sc, 0x15, 0x0314);
        re_mdio_write(sc, 0x19, 0x4d00);
        re_mdio_write(sc, 0x15, 0x0315);
        re_mdio_write(sc, 0x19, 0x6810);
        re_mdio_write(sc, 0x15, 0x031e);
        re_mdio_write(sc, 0x19, 0x404f);
        re_mdio_write(sc, 0x15, 0x031f);
        re_mdio_write(sc, 0x19, 0x44c8);
        re_mdio_write(sc, 0x15, 0x0320);
        re_mdio_write(sc, 0x19, 0xd64f);
        re_mdio_write(sc, 0x15, 0x0321);
        re_mdio_write(sc, 0x19, 0x00e7);
        re_mdio_write(sc, 0x15, 0x0322);
        re_mdio_write(sc, 0x19, 0x7c08);
        re_mdio_write(sc, 0x15, 0x0323);
        re_mdio_write(sc, 0x19, 0x8203);
        re_mdio_write(sc, 0x15, 0x0324);
        re_mdio_write(sc, 0x19, 0x4d48);
        re_mdio_write(sc, 0x15, 0x0325);
        re_mdio_write(sc, 0x19, 0x3327);
        re_mdio_write(sc, 0x15, 0x0326);
        re_mdio_write(sc, 0x19, 0x4d40);
        re_mdio_write(sc, 0x15, 0x0327);
        re_mdio_write(sc, 0x19, 0xc8d7);
        re_mdio_write(sc, 0x15, 0x0328);
        re_mdio_write(sc, 0x19, 0x0003);
        re_mdio_write(sc, 0x15, 0x0329);
        re_mdio_write(sc, 0x19, 0x7c20);
        re_mdio_write(sc, 0x15, 0x032a);
        re_mdio_write(sc, 0x19, 0x4c20);
        re_mdio_write(sc, 0x15, 0x032b);
        re_mdio_write(sc, 0x19, 0xc8ed);
        re_mdio_write(sc, 0x15, 0x032c);
        re_mdio_write(sc, 0x19, 0x00f4);
        re_mdio_write(sc, 0x15, 0x032d);
        re_mdio_write(sc, 0x19, 0x82b3);
        re_mdio_write(sc, 0x15, 0x032e);
        re_mdio_write(sc, 0x19, 0xd11d);
        re_mdio_write(sc, 0x15, 0x032f);
        re_mdio_write(sc, 0x19, 0x00b1);
        re_mdio_write(sc, 0x15, 0x0330);
        re_mdio_write(sc, 0x19, 0xde18);
        re_mdio_write(sc, 0x15, 0x0331);
        re_mdio_write(sc, 0x19, 0x0008);
        re_mdio_write(sc, 0x15, 0x0332);
        re_mdio_write(sc, 0x19, 0x91ee);
        re_mdio_write(sc, 0x15, 0x0333);
        re_mdio_write(sc, 0x19, 0x3339);
        re_mdio_write(sc, 0x15, 0x033a);
        re_mdio_write(sc, 0x19, 0x4064);
        re_mdio_write(sc, 0x15, 0x0340);
        re_mdio_write(sc, 0x19, 0x9e06);
        re_mdio_write(sc, 0x15, 0x0341);
        re_mdio_write(sc, 0x19, 0x7c08);
        re_mdio_write(sc, 0x15, 0x0342);
        re_mdio_write(sc, 0x19, 0x8203);
        re_mdio_write(sc, 0x15, 0x0343);
        re_mdio_write(sc, 0x19, 0x4d48);
        re_mdio_write(sc, 0x15, 0x0344);
        re_mdio_write(sc, 0x19, 0x3346);
        re_mdio_write(sc, 0x15, 0x0345);
        re_mdio_write(sc, 0x19, 0x4d40);
        re_mdio_write(sc, 0x15, 0x0346);
        re_mdio_write(sc, 0x19, 0xd11d);
        re_mdio_write(sc, 0x15, 0x0347);
        re_mdio_write(sc, 0x19, 0x0099);
        re_mdio_write(sc, 0x15, 0x0348);
        re_mdio_write(sc, 0x19, 0xbb17);
        re_mdio_write(sc, 0x15, 0x0349);
        re_mdio_write(sc, 0x19, 0x8102);
        re_mdio_write(sc, 0x15, 0x034a);
        re_mdio_write(sc, 0x19, 0x334d);
        re_mdio_write(sc, 0x15, 0x034b);
        re_mdio_write(sc, 0x19, 0xa22c);
        re_mdio_write(sc, 0x15, 0x034c);
        re_mdio_write(sc, 0x19, 0x3397);
        re_mdio_write(sc, 0x15, 0x034d);
        re_mdio_write(sc, 0x19, 0x91f2);
        re_mdio_write(sc, 0x15, 0x034e);
        re_mdio_write(sc, 0x19, 0xc218);
        re_mdio_write(sc, 0x15, 0x034f);
        re_mdio_write(sc, 0x19, 0x00f0);
        re_mdio_write(sc, 0x15, 0x0350);
        re_mdio_write(sc, 0x19, 0x3397);
        re_mdio_write(sc, 0x15, 0x0351);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0364);
        re_mdio_write(sc, 0x19, 0xbc05);
        re_mdio_write(sc, 0x15, 0x0367);
        re_mdio_write(sc, 0x19, 0xa1fc);
        re_mdio_write(sc, 0x15, 0x0368);
        re_mdio_write(sc, 0x19, 0x3377);
        re_mdio_write(sc, 0x15, 0x0369);
        re_mdio_write(sc, 0x19, 0x328b);
        re_mdio_write(sc, 0x15, 0x036a);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0377);
        re_mdio_write(sc, 0x19, 0x4b97);
        re_mdio_write(sc, 0x15, 0x0378);
        re_mdio_write(sc, 0x19, 0x6818);
        re_mdio_write(sc, 0x15, 0x0379);
        re_mdio_write(sc, 0x19, 0x4b07);
        re_mdio_write(sc, 0x15, 0x037a);
        re_mdio_write(sc, 0x19, 0x40ac);
        re_mdio_write(sc, 0x15, 0x037b);
        re_mdio_write(sc, 0x19, 0x4445);
        re_mdio_write(sc, 0x15, 0x037c);
        re_mdio_write(sc, 0x19, 0x404e);
        re_mdio_write(sc, 0x15, 0x037d);
        re_mdio_write(sc, 0x19, 0x4461);
        re_mdio_write(sc, 0x15, 0x037e);
        re_mdio_write(sc, 0x19, 0x9c09);
        re_mdio_write(sc, 0x15, 0x037f);
        re_mdio_write(sc, 0x19, 0x63da);
        re_mdio_write(sc, 0x15, 0x0380);
        re_mdio_write(sc, 0x19, 0x5440);
        re_mdio_write(sc, 0x15, 0x0381);
        re_mdio_write(sc, 0x19, 0x4b98);
        re_mdio_write(sc, 0x15, 0x0382);
        re_mdio_write(sc, 0x19, 0x7c60);
        re_mdio_write(sc, 0x15, 0x0383);
        re_mdio_write(sc, 0x19, 0x4c00);
        re_mdio_write(sc, 0x15, 0x0384);
        re_mdio_write(sc, 0x19, 0x4b08);
        re_mdio_write(sc, 0x15, 0x0385);
        re_mdio_write(sc, 0x19, 0x63d8);
        re_mdio_write(sc, 0x15, 0x0386);
        re_mdio_write(sc, 0x19, 0x338d);
        re_mdio_write(sc, 0x15, 0x0387);
        re_mdio_write(sc, 0x19, 0xd64f);
        re_mdio_write(sc, 0x15, 0x0388);
        re_mdio_write(sc, 0x19, 0x0080);
        re_mdio_write(sc, 0x15, 0x0389);
        re_mdio_write(sc, 0x19, 0x820c);
        re_mdio_write(sc, 0x15, 0x038a);
        re_mdio_write(sc, 0x19, 0xa10b);
        re_mdio_write(sc, 0x15, 0x038b);
        re_mdio_write(sc, 0x19, 0x9df3);
        re_mdio_write(sc, 0x15, 0x038c);
        re_mdio_write(sc, 0x19, 0x3395);
        re_mdio_write(sc, 0x15, 0x038d);
        re_mdio_write(sc, 0x19, 0xd64f);
        re_mdio_write(sc, 0x15, 0x038e);
        re_mdio_write(sc, 0x19, 0x00f9);
        re_mdio_write(sc, 0x15, 0x038f);
        re_mdio_write(sc, 0x19, 0xc017);
        re_mdio_write(sc, 0x15, 0x0390);
        re_mdio_write(sc, 0x19, 0x0005);
        re_mdio_write(sc, 0x15, 0x0391);
        re_mdio_write(sc, 0x19, 0x6c0b);
        re_mdio_write(sc, 0x15, 0x0392);
        re_mdio_write(sc, 0x19, 0xa103);
        re_mdio_write(sc, 0x15, 0x0393);
        re_mdio_write(sc, 0x19, 0x6c08);
        re_mdio_write(sc, 0x15, 0x0394);
        re_mdio_write(sc, 0x19, 0x9df9);
        re_mdio_write(sc, 0x15, 0x0395);
        re_mdio_write(sc, 0x19, 0x6c08);
        re_mdio_write(sc, 0x15, 0x0396);
        re_mdio_write(sc, 0x19, 0x3397);
        re_mdio_write(sc, 0x15, 0x0399);
        re_mdio_write(sc, 0x19, 0x6810);
        re_mdio_write(sc, 0x15, 0x03a4);
        re_mdio_write(sc, 0x19, 0x7c08);
        re_mdio_write(sc, 0x15, 0x03a5);
        re_mdio_write(sc, 0x19, 0x8203);
        re_mdio_write(sc, 0x15, 0x03a6);
        re_mdio_write(sc, 0x19, 0x4d08);
        re_mdio_write(sc, 0x15, 0x03a7);
        re_mdio_write(sc, 0x19, 0x33a9);
        re_mdio_write(sc, 0x15, 0x03a8);
        re_mdio_write(sc, 0x19, 0x4d00);
        re_mdio_write(sc, 0x15, 0x03a9);
        re_mdio_write(sc, 0x19, 0x9bfa);
        re_mdio_write(sc, 0x15, 0x03aa);
        re_mdio_write(sc, 0x19, 0x33b6);
        re_mdio_write(sc, 0x15, 0x03bb);
        re_mdio_write(sc, 0x19, 0x4056);
        re_mdio_write(sc, 0x15, 0x03bc);
        re_mdio_write(sc, 0x19, 0x44e9);
        re_mdio_write(sc, 0x15, 0x03bd);
        re_mdio_write(sc, 0x19, 0x405e);
        re_mdio_write(sc, 0x15, 0x03be);
        re_mdio_write(sc, 0x19, 0x44f8);
        re_mdio_write(sc, 0x15, 0x03bf);
        re_mdio_write(sc, 0x19, 0xd64f);
        re_mdio_write(sc, 0x15, 0x03c0);
        re_mdio_write(sc, 0x19, 0x0037);
        re_mdio_write(sc, 0x15, 0x03c1);
        re_mdio_write(sc, 0x19, 0xbd37);
        re_mdio_write(sc, 0x15, 0x03c2);
        re_mdio_write(sc, 0x19, 0x9cfd);
        re_mdio_write(sc, 0x15, 0x03c3);
        re_mdio_write(sc, 0x19, 0xc639);
        re_mdio_write(sc, 0x15, 0x03c4);
        re_mdio_write(sc, 0x19, 0x0011);
        re_mdio_write(sc, 0x15, 0x03c5);
        re_mdio_write(sc, 0x19, 0x9b03);
        re_mdio_write(sc, 0x15, 0x03c6);
        re_mdio_write(sc, 0x19, 0x7c01);
        re_mdio_write(sc, 0x15, 0x03c7);
        re_mdio_write(sc, 0x19, 0x4c01);
        re_mdio_write(sc, 0x15, 0x03c8);
        re_mdio_write(sc, 0x19, 0x9e03);
        re_mdio_write(sc, 0x15, 0x03c9);
        re_mdio_write(sc, 0x19, 0x7c20);
        re_mdio_write(sc, 0x15, 0x03ca);
        re_mdio_write(sc, 0x19, 0x4c20);
        re_mdio_write(sc, 0x15, 0x03cb);
        re_mdio_write(sc, 0x19, 0x9af4);
        re_mdio_write(sc, 0x15, 0x03cc);
        re_mdio_write(sc, 0x19, 0x7c12);
        re_mdio_write(sc, 0x15, 0x03cd);
        re_mdio_write(sc, 0x19, 0x4c52);
        re_mdio_write(sc, 0x15, 0x03ce);
        re_mdio_write(sc, 0x19, 0x4470);
        re_mdio_write(sc, 0x15, 0x03cf);
        re_mdio_write(sc, 0x19, 0x7c12);
        re_mdio_write(sc, 0x15, 0x03d0);
        re_mdio_write(sc, 0x19, 0x4c40);
        re_mdio_write(sc, 0x15, 0x03d1);
        re_mdio_write(sc, 0x19, 0x33bf);
        re_mdio_write(sc, 0x15, 0x03d6);
        re_mdio_write(sc, 0x19, 0x4047);
        re_mdio_write(sc, 0x15, 0x03d7);
        re_mdio_write(sc, 0x19, 0x4469);
        re_mdio_write(sc, 0x15, 0x03d8);
        re_mdio_write(sc, 0x19, 0x492b);
        re_mdio_write(sc, 0x15, 0x03d9);
        re_mdio_write(sc, 0x19, 0x4479);
        re_mdio_write(sc, 0x15, 0x03da);
        re_mdio_write(sc, 0x19, 0x7c09);
        re_mdio_write(sc, 0x15, 0x03db);
        re_mdio_write(sc, 0x19, 0x8203);
        re_mdio_write(sc, 0x15, 0x03dc);
        re_mdio_write(sc, 0x19, 0x4d48);
        re_mdio_write(sc, 0x15, 0x03dd);
        re_mdio_write(sc, 0x19, 0x33df);
        re_mdio_write(sc, 0x15, 0x03de);
        re_mdio_write(sc, 0x19, 0x4d40);
        re_mdio_write(sc, 0x15, 0x03df);
        re_mdio_write(sc, 0x19, 0xd64f);
        re_mdio_write(sc, 0x15, 0x03e0);
        re_mdio_write(sc, 0x19, 0x0017);
        re_mdio_write(sc, 0x15, 0x03e1);
        re_mdio_write(sc, 0x19, 0xbd17);
        re_mdio_write(sc, 0x15, 0x03e2);
        re_mdio_write(sc, 0x19, 0x9b03);
        re_mdio_write(sc, 0x15, 0x03e3);
        re_mdio_write(sc, 0x19, 0x7c20);
        re_mdio_write(sc, 0x15, 0x03e4);
        re_mdio_write(sc, 0x19, 0x4c20);
        re_mdio_write(sc, 0x15, 0x03e5);
        re_mdio_write(sc, 0x19, 0x88f5);
        re_mdio_write(sc, 0x15, 0x03e6);
        re_mdio_write(sc, 0x19, 0xc428);
        re_mdio_write(sc, 0x15, 0x03e7);
        re_mdio_write(sc, 0x19, 0x0008);
        re_mdio_write(sc, 0x15, 0x03e8);
        re_mdio_write(sc, 0x19, 0x9af2);
        re_mdio_write(sc, 0x15, 0x03e9);
        re_mdio_write(sc, 0x19, 0x7c12);
        re_mdio_write(sc, 0x15, 0x03ea);
        re_mdio_write(sc, 0x19, 0x4c52);
        re_mdio_write(sc, 0x15, 0x03eb);
        re_mdio_write(sc, 0x19, 0x4470);
        re_mdio_write(sc, 0x15, 0x03ec);
        re_mdio_write(sc, 0x19, 0x7c12);
        re_mdio_write(sc, 0x15, 0x03ed);
        re_mdio_write(sc, 0x19, 0x4c40);
        re_mdio_write(sc, 0x15, 0x03ee);
        re_mdio_write(sc, 0x19, 0x33da);
        re_mdio_write(sc, 0x15, 0x03ef);
        re_mdio_write(sc, 0x19, 0x3312);
        re_mdio_write(sc, 0x16, 0x0306);
        re_mdio_write(sc, 0x16, 0x0300);
        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x17, 0x2179);
        re_mdio_write(sc, 0x1f, 0x0007);
        re_mdio_write(sc, 0x1e, 0x0040);
        re_mdio_write(sc, 0x18, 0x0645);
        re_mdio_write(sc, 0x19, 0xe200);
        re_mdio_write(sc, 0x18, 0x0655);
        re_mdio_write(sc, 0x19, 0x9000);
        re_mdio_write(sc, 0x18, 0x0d05);
        re_mdio_write(sc, 0x19, 0xbe00);
        re_mdio_write(sc, 0x18, 0x0d15);
        re_mdio_write(sc, 0x19, 0xd300);
        re_mdio_write(sc, 0x18, 0x0d25);
        re_mdio_write(sc, 0x19, 0xfe00);
        re_mdio_write(sc, 0x18, 0x0d35);
        re_mdio_write(sc, 0x19, 0x4000);
        re_mdio_write(sc, 0x18, 0x0d45);
        re_mdio_write(sc, 0x19, 0x7f00);
        re_mdio_write(sc, 0x18, 0x0d55);
        re_mdio_write(sc, 0x19, 0x1000);
        re_mdio_write(sc, 0x18, 0x0d65);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x18, 0x0d75);
        re_mdio_write(sc, 0x19, 0x8200);
        re_mdio_write(sc, 0x18, 0x0d85);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x18, 0x0d95);
        re_mdio_write(sc, 0x19, 0x7000);
        re_mdio_write(sc, 0x18, 0x0da5);
        re_mdio_write(sc, 0x19, 0x0f00);
        re_mdio_write(sc, 0x18, 0x0db5);
        re_mdio_write(sc, 0x19, 0x0100);
        re_mdio_write(sc, 0x18, 0x0dc5);
        re_mdio_write(sc, 0x19, 0x9b00);
        re_mdio_write(sc, 0x18, 0x0dd5);
        re_mdio_write(sc, 0x19, 0x7f00);
        re_mdio_write(sc, 0x18, 0x0de5);
        re_mdio_write(sc, 0x19, 0xe000);
        re_mdio_write(sc, 0x18, 0x0df5);
        re_mdio_write(sc, 0x19, 0xef00);
        re_mdio_write(sc, 0x18, 0x16d5);
        re_mdio_write(sc, 0x19, 0xe200);
        re_mdio_write(sc, 0x18, 0x16e5);
        re_mdio_write(sc, 0x19, 0xab00);
        re_mdio_write(sc, 0x18, 0x2904);
        re_mdio_write(sc, 0x19, 0x4000);
        re_mdio_write(sc, 0x18, 0x2914);
        re_mdio_write(sc, 0x19, 0x7f00);
        re_mdio_write(sc, 0x18, 0x2924);
        re_mdio_write(sc, 0x19, 0x0100);
        re_mdio_write(sc, 0x18, 0x2934);
        re_mdio_write(sc, 0x19, 0x2000);
        re_mdio_write(sc, 0x18, 0x2944);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x18, 0x2954);
        re_mdio_write(sc, 0x19, 0x4600);
        re_mdio_write(sc, 0x18, 0x2964);
        re_mdio_write(sc, 0x19, 0xfc00);
        re_mdio_write(sc, 0x18, 0x2974);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x18, 0x2984);
        re_mdio_write(sc, 0x19, 0x5000);
        re_mdio_write(sc, 0x18, 0x2994);
        re_mdio_write(sc, 0x19, 0x9d00);
        re_mdio_write(sc, 0x18, 0x29a4);
        re_mdio_write(sc, 0x19, 0xff00);
        re_mdio_write(sc, 0x18, 0x29b4);
        re_mdio_write(sc, 0x19, 0x4000);
        re_mdio_write(sc, 0x18, 0x29c4);
        re_mdio_write(sc, 0x19, 0x7f00);
        re_mdio_write(sc, 0x18, 0x29d4);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x18, 0x29e4);
        re_mdio_write(sc, 0x19, 0x2000);
        re_mdio_write(sc, 0x18, 0x29f4);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x18, 0x2a04);
        re_mdio_write(sc, 0x19, 0xe600);
        re_mdio_write(sc, 0x18, 0x2a14);
        re_mdio_write(sc, 0x19, 0xff00);
        re_mdio_write(sc, 0x18, 0x2a24);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x18, 0x2a34);
        re_mdio_write(sc, 0x19, 0x5000);
        re_mdio_write(sc, 0x18, 0x2a44);
        re_mdio_write(sc, 0x19, 0x8500);
        re_mdio_write(sc, 0x18, 0x2a54);
        re_mdio_write(sc, 0x19, 0x7f00);
        re_mdio_write(sc, 0x18, 0x2a64);
        re_mdio_write(sc, 0x19, 0xac00);
        re_mdio_write(sc, 0x18, 0x2a74);
        re_mdio_write(sc, 0x19, 0x0800);
        re_mdio_write(sc, 0x18, 0x2a84);
        re_mdio_write(sc, 0x19, 0xfc00);
        re_mdio_write(sc, 0x18, 0x2a94);
        re_mdio_write(sc, 0x19, 0xe000);
        re_mdio_write(sc, 0x18, 0x2aa4);
        re_mdio_write(sc, 0x19, 0x7400);
        re_mdio_write(sc, 0x18, 0x2ab4);
        re_mdio_write(sc, 0x19, 0x4000);
        re_mdio_write(sc, 0x18, 0x2ac4);
        re_mdio_write(sc, 0x19, 0x7f00);
        re_mdio_write(sc, 0x18, 0x2ad4);
        re_mdio_write(sc, 0x19, 0x0100);
        re_mdio_write(sc, 0x18, 0x2ae4);
        re_mdio_write(sc, 0x19, 0xff00);
        re_mdio_write(sc, 0x18, 0x2af4);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x18, 0x2b04);
        re_mdio_write(sc, 0x19, 0x4400);
        re_mdio_write(sc, 0x18, 0x2b14);
        re_mdio_write(sc, 0x19, 0xfc00);
        re_mdio_write(sc, 0x18, 0x2b24);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x18, 0x2b34);
        re_mdio_write(sc, 0x19, 0x4000);
        re_mdio_write(sc, 0x18, 0x2b44);
        re_mdio_write(sc, 0x19, 0x9d00);
        re_mdio_write(sc, 0x18, 0x2b54);
        re_mdio_write(sc, 0x19, 0xff00);
        re_mdio_write(sc, 0x18, 0x2b64);
        re_mdio_write(sc, 0x19, 0x4000);
        re_mdio_write(sc, 0x18, 0x2b74);
        re_mdio_write(sc, 0x19, 0x7f00);
        re_mdio_write(sc, 0x18, 0x2b84);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x18, 0x2b94);
        re_mdio_write(sc, 0x19, 0xff00);
        re_mdio_write(sc, 0x18, 0x2ba4);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x18, 0x2bb4);
        re_mdio_write(sc, 0x19, 0xfc00);
        re_mdio_write(sc, 0x18, 0x2bc4);
        re_mdio_write(sc, 0x19, 0xff00);
        re_mdio_write(sc, 0x18, 0x2bd4);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x18, 0x2be4);
        re_mdio_write(sc, 0x19, 0x4000);
        re_mdio_write(sc, 0x18, 0x2bf4);
        re_mdio_write(sc, 0x19, 0x8900);
        re_mdio_write(sc, 0x18, 0x2c04);
        re_mdio_write(sc, 0x19, 0x8300);
        re_mdio_write(sc, 0x18, 0x2c14);
        re_mdio_write(sc, 0x19, 0xe000);
        re_mdio_write(sc, 0x18, 0x2c24);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x18, 0x2c34);
        re_mdio_write(sc, 0x19, 0xac00);
        re_mdio_write(sc, 0x18, 0x2c44);
        re_mdio_write(sc, 0x19, 0x0800);
        re_mdio_write(sc, 0x18, 0x2c54);
        re_mdio_write(sc, 0x19, 0xfa00);
        re_mdio_write(sc, 0x18, 0x2c64);
        re_mdio_write(sc, 0x19, 0xe100);
        re_mdio_write(sc, 0x18, 0x2c74);
        re_mdio_write(sc, 0x19, 0x7f00);
        re_mdio_write(sc, 0x18, 0x0001);
        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x17, 0x2100);
        re_mdio_write(sc, 0x1f, 0x0005);
        re_mdio_write(sc, 0x05, 0xfff6);
        re_mdio_write(sc, 0x06, 0x0080);
        re_mdio_write(sc, 0x05, 0x8b88);
        re_mdio_write(sc, 0x06, 0x0000);
        re_mdio_write(sc, 0x06, 0x0000);
        re_mdio_write(sc, 0x06, 0x0000);
        re_mdio_write(sc, 0x06, 0x0000);
        re_mdio_write(sc, 0x05, 0x8000);
        re_mdio_write(sc, 0x06, 0xd480);
        re_mdio_write(sc, 0x06, 0xc1e4);
        re_mdio_write(sc, 0x06, 0x8b9a);
        re_mdio_write(sc, 0x06, 0xe58b);
        re_mdio_write(sc, 0x06, 0x9bee);
        re_mdio_write(sc, 0x06, 0x8b83);
        re_mdio_write(sc, 0x06, 0x41bf);
        re_mdio_write(sc, 0x06, 0x8b88);
        re_mdio_write(sc, 0x06, 0xec00);
        re_mdio_write(sc, 0x06, 0x19a9);
        re_mdio_write(sc, 0x06, 0x8b90);
        re_mdio_write(sc, 0x06, 0xf9ee);
        re_mdio_write(sc, 0x06, 0xfff6);
        re_mdio_write(sc, 0x06, 0x00ee);
        re_mdio_write(sc, 0x06, 0xfff7);
        re_mdio_write(sc, 0x06, 0xffe0);
        re_mdio_write(sc, 0x06, 0xe140);
        re_mdio_write(sc, 0x06, 0xe1e1);
        re_mdio_write(sc, 0x06, 0x41f7);
        re_mdio_write(sc, 0x06, 0x2ff6);
        re_mdio_write(sc, 0x06, 0x28e4);
        re_mdio_write(sc, 0x06, 0xe140);
        re_mdio_write(sc, 0x06, 0xe5e1);
        re_mdio_write(sc, 0x06, 0x41f7);
        re_mdio_write(sc, 0x06, 0x0002);
        re_mdio_write(sc, 0x06, 0x020c);
        re_mdio_write(sc, 0x06, 0x0202);
        re_mdio_write(sc, 0x06, 0x1d02);
        re_mdio_write(sc, 0x06, 0x0230);
        re_mdio_write(sc, 0x06, 0x0202);
        re_mdio_write(sc, 0x06, 0x4002);
        re_mdio_write(sc, 0x06, 0x028b);
        re_mdio_write(sc, 0x06, 0x0280);
        re_mdio_write(sc, 0x06, 0x6c02);
        re_mdio_write(sc, 0x06, 0x8085);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x88e1);
        re_mdio_write(sc, 0x06, 0x8b89);
        re_mdio_write(sc, 0x06, 0x1e01);
        re_mdio_write(sc, 0x06, 0xe18b);
        re_mdio_write(sc, 0x06, 0x8a1e);
        re_mdio_write(sc, 0x06, 0x01e1);
        re_mdio_write(sc, 0x06, 0x8b8b);
        re_mdio_write(sc, 0x06, 0x1e01);
        re_mdio_write(sc, 0x06, 0xe18b);
        re_mdio_write(sc, 0x06, 0x8c1e);
        re_mdio_write(sc, 0x06, 0x01e1);
        re_mdio_write(sc, 0x06, 0x8b8d);
        re_mdio_write(sc, 0x06, 0x1e01);
        re_mdio_write(sc, 0x06, 0xe18b);
        re_mdio_write(sc, 0x06, 0x8e1e);
        re_mdio_write(sc, 0x06, 0x01a0);
        re_mdio_write(sc, 0x06, 0x00c7);
        re_mdio_write(sc, 0x06, 0xaec3);
        re_mdio_write(sc, 0x06, 0xf8e0);
        re_mdio_write(sc, 0x06, 0x8b8d);
        re_mdio_write(sc, 0x06, 0xad20);
        re_mdio_write(sc, 0x06, 0x10ee);
        re_mdio_write(sc, 0x06, 0x8b8d);
        re_mdio_write(sc, 0x06, 0x0002);
        re_mdio_write(sc, 0x06, 0x1310);
        re_mdio_write(sc, 0x06, 0x021f);
        re_mdio_write(sc, 0x06, 0x9d02);
        re_mdio_write(sc, 0x06, 0x1f0c);
        re_mdio_write(sc, 0x06, 0x0227);
        re_mdio_write(sc, 0x06, 0x49fc);
        re_mdio_write(sc, 0x06, 0x04f8);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x8ead);
        re_mdio_write(sc, 0x06, 0x200b);
        re_mdio_write(sc, 0x06, 0xf620);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x8e02);
        re_mdio_write(sc, 0x06, 0x830e);
        re_mdio_write(sc, 0x06, 0x021b);
        re_mdio_write(sc, 0x06, 0x67ad);
        re_mdio_write(sc, 0x06, 0x2211);
        re_mdio_write(sc, 0x06, 0xf622);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x8e02);
        re_mdio_write(sc, 0x06, 0x2ba5);
        re_mdio_write(sc, 0x06, 0x022a);
        re_mdio_write(sc, 0x06, 0x2402);
        re_mdio_write(sc, 0x06, 0x80c6);
        re_mdio_write(sc, 0x06, 0x022a);
        re_mdio_write(sc, 0x06, 0xf0ad);
        re_mdio_write(sc, 0x06, 0x2511);
        re_mdio_write(sc, 0x06, 0xf625);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x8e02);
        re_mdio_write(sc, 0x06, 0x8226);
        re_mdio_write(sc, 0x06, 0x0204);
        re_mdio_write(sc, 0x06, 0x0302);
        re_mdio_write(sc, 0x06, 0x19cc);
        re_mdio_write(sc, 0x06, 0x022b);
        re_mdio_write(sc, 0x06, 0x5bfc);
        re_mdio_write(sc, 0x06, 0x04ee);
        re_mdio_write(sc, 0x06, 0x8b8d);
        re_mdio_write(sc, 0x06, 0x0105);
        re_mdio_write(sc, 0x06, 0xf8e0);
        re_mdio_write(sc, 0x06, 0x8b83);
        re_mdio_write(sc, 0x06, 0xad24);
        re_mdio_write(sc, 0x06, 0x44e0);
        re_mdio_write(sc, 0x06, 0xe022);
        re_mdio_write(sc, 0x06, 0xe1e0);
        re_mdio_write(sc, 0x06, 0x23ad);
        re_mdio_write(sc, 0x06, 0x223b);
        re_mdio_write(sc, 0x06, 0xe08a);
        re_mdio_write(sc, 0x06, 0xbea0);
        re_mdio_write(sc, 0x06, 0x0005);
        re_mdio_write(sc, 0x06, 0x0228);
        re_mdio_write(sc, 0x06, 0xdeae);
        re_mdio_write(sc, 0x06, 0x42a0);
        re_mdio_write(sc, 0x06, 0x0105);
        re_mdio_write(sc, 0x06, 0x0228);
        re_mdio_write(sc, 0x06, 0xf1ae);
        re_mdio_write(sc, 0x06, 0x3aa0);
        re_mdio_write(sc, 0x06, 0x0205);
        re_mdio_write(sc, 0x06, 0x0281);
        re_mdio_write(sc, 0x06, 0x25ae);
        re_mdio_write(sc, 0x06, 0x32a0);
        re_mdio_write(sc, 0x06, 0x0305);
        re_mdio_write(sc, 0x06, 0x0229);
        re_mdio_write(sc, 0x06, 0x9aae);
        re_mdio_write(sc, 0x06, 0x2aa0);
        re_mdio_write(sc, 0x06, 0x0405);
        re_mdio_write(sc, 0x06, 0x0229);
        re_mdio_write(sc, 0x06, 0xaeae);
        re_mdio_write(sc, 0x06, 0x22a0);
        re_mdio_write(sc, 0x06, 0x0505);
        re_mdio_write(sc, 0x06, 0x0229);
        re_mdio_write(sc, 0x06, 0xd7ae);
        re_mdio_write(sc, 0x06, 0x1aa0);
        re_mdio_write(sc, 0x06, 0x0605);
        re_mdio_write(sc, 0x06, 0x0229);
        re_mdio_write(sc, 0x06, 0xfeae);
        re_mdio_write(sc, 0x06, 0x12ee);
        re_mdio_write(sc, 0x06, 0x8ac0);
        re_mdio_write(sc, 0x06, 0x00ee);
        re_mdio_write(sc, 0x06, 0x8ac1);
        re_mdio_write(sc, 0x06, 0x00ee);
        re_mdio_write(sc, 0x06, 0x8ac6);
        re_mdio_write(sc, 0x06, 0x00ee);
        re_mdio_write(sc, 0x06, 0x8abe);
        re_mdio_write(sc, 0x06, 0x00ae);
        re_mdio_write(sc, 0x06, 0x00fc);
        re_mdio_write(sc, 0x06, 0x04f8);
        re_mdio_write(sc, 0x06, 0x022a);
        re_mdio_write(sc, 0x06, 0x67e0);
        re_mdio_write(sc, 0x06, 0xe022);
        re_mdio_write(sc, 0x06, 0xe1e0);
        re_mdio_write(sc, 0x06, 0x230d);
        re_mdio_write(sc, 0x06, 0x0658);
        re_mdio_write(sc, 0x06, 0x03a0);
        re_mdio_write(sc, 0x06, 0x0202);
        re_mdio_write(sc, 0x06, 0xae2d);
        re_mdio_write(sc, 0x06, 0xa001);
        re_mdio_write(sc, 0x06, 0x02ae);
        re_mdio_write(sc, 0x06, 0x2da0);
        re_mdio_write(sc, 0x06, 0x004d);
        re_mdio_write(sc, 0x06, 0xe0e2);
        re_mdio_write(sc, 0x06, 0x00e1);
        re_mdio_write(sc, 0x06, 0xe201);
        re_mdio_write(sc, 0x06, 0xad24);
        re_mdio_write(sc, 0x06, 0x44e0);
        re_mdio_write(sc, 0x06, 0x8ac2);
        re_mdio_write(sc, 0x06, 0xe48a);
        re_mdio_write(sc, 0x06, 0xc4e0);
        re_mdio_write(sc, 0x06, 0x8ac3);
        re_mdio_write(sc, 0x06, 0xe48a);
        re_mdio_write(sc, 0x06, 0xc5ee);
        re_mdio_write(sc, 0x06, 0x8abe);
        re_mdio_write(sc, 0x06, 0x03e0);
        re_mdio_write(sc, 0x06, 0x8b83);
        re_mdio_write(sc, 0x06, 0xad25);
        re_mdio_write(sc, 0x06, 0x3aee);
        re_mdio_write(sc, 0x06, 0x8abe);
        re_mdio_write(sc, 0x06, 0x05ae);
        re_mdio_write(sc, 0x06, 0x34e0);
        re_mdio_write(sc, 0x06, 0x8ace);
        re_mdio_write(sc, 0x06, 0xae03);
        re_mdio_write(sc, 0x06, 0xe08a);
        re_mdio_write(sc, 0x06, 0xcfe1);
        re_mdio_write(sc, 0x06, 0x8ac2);
        re_mdio_write(sc, 0x06, 0x4905);
        re_mdio_write(sc, 0x06, 0xe58a);
        re_mdio_write(sc, 0x06, 0xc4e1);
        re_mdio_write(sc, 0x06, 0x8ac3);
        re_mdio_write(sc, 0x06, 0x4905);
        re_mdio_write(sc, 0x06, 0xe58a);
        re_mdio_write(sc, 0x06, 0xc5ee);
        re_mdio_write(sc, 0x06, 0x8abe);
        re_mdio_write(sc, 0x06, 0x0502);
        re_mdio_write(sc, 0x06, 0x2ab6);
        re_mdio_write(sc, 0x06, 0xac20);
        re_mdio_write(sc, 0x06, 0x1202);
        re_mdio_write(sc, 0x06, 0x819b);
        re_mdio_write(sc, 0x06, 0xac20);
        re_mdio_write(sc, 0x06, 0x0cee);
        re_mdio_write(sc, 0x06, 0x8ac1);
        re_mdio_write(sc, 0x06, 0x00ee);
        re_mdio_write(sc, 0x06, 0x8ac6);
        re_mdio_write(sc, 0x06, 0x00ee);
        re_mdio_write(sc, 0x06, 0x8abe);
        re_mdio_write(sc, 0x06, 0x02fc);
        re_mdio_write(sc, 0x06, 0x04d0);
        re_mdio_write(sc, 0x06, 0x0002);
        re_mdio_write(sc, 0x06, 0x81ad);
        re_mdio_write(sc, 0x06, 0x590f);
        re_mdio_write(sc, 0x06, 0x3902);
        re_mdio_write(sc, 0x06, 0xaa04);
        re_mdio_write(sc, 0x06, 0xd001);
        re_mdio_write(sc, 0x06, 0xae02);
        re_mdio_write(sc, 0x06, 0xd000);
        re_mdio_write(sc, 0x06, 0x04f9);
        re_mdio_write(sc, 0x06, 0xfae2);
        re_mdio_write(sc, 0x06, 0xe2d2);
        re_mdio_write(sc, 0x06, 0xe3e2);
        re_mdio_write(sc, 0x06, 0xd3f9);
        re_mdio_write(sc, 0x06, 0x5af7);
        re_mdio_write(sc, 0x06, 0xe6e2);
        re_mdio_write(sc, 0x06, 0xd2e7);
        re_mdio_write(sc, 0x06, 0xe2d3);
        re_mdio_write(sc, 0x06, 0xe2e0);
        re_mdio_write(sc, 0x06, 0x2ce3);
        re_mdio_write(sc, 0x06, 0xe02d);
        re_mdio_write(sc, 0x06, 0xf95b);
        re_mdio_write(sc, 0x06, 0xe01e);
        re_mdio_write(sc, 0x06, 0x30e6);
        re_mdio_write(sc, 0x06, 0xe02c);
        re_mdio_write(sc, 0x06, 0xe7e0);
        re_mdio_write(sc, 0x06, 0x2de2);
        re_mdio_write(sc, 0x06, 0xe2cc);
        re_mdio_write(sc, 0x06, 0xe3e2);
        re_mdio_write(sc, 0x06, 0xcdf9);
        re_mdio_write(sc, 0x06, 0x5a0f);
        re_mdio_write(sc, 0x06, 0x6a50);
        re_mdio_write(sc, 0x06, 0xe6e2);
        re_mdio_write(sc, 0x06, 0xcce7);
        re_mdio_write(sc, 0x06, 0xe2cd);
        re_mdio_write(sc, 0x06, 0xe0e0);
        re_mdio_write(sc, 0x06, 0x3ce1);
        re_mdio_write(sc, 0x06, 0xe03d);
        re_mdio_write(sc, 0x06, 0xef64);
        re_mdio_write(sc, 0x06, 0xfde0);
        re_mdio_write(sc, 0x06, 0xe2cc);
        re_mdio_write(sc, 0x06, 0xe1e2);
        re_mdio_write(sc, 0x06, 0xcd58);
        re_mdio_write(sc, 0x06, 0x0f5a);
        re_mdio_write(sc, 0x06, 0xf01e);
        re_mdio_write(sc, 0x06, 0x02e4);
        re_mdio_write(sc, 0x06, 0xe2cc);
        re_mdio_write(sc, 0x06, 0xe5e2);
        re_mdio_write(sc, 0x06, 0xcdfd);
        re_mdio_write(sc, 0x06, 0xe0e0);
        re_mdio_write(sc, 0x06, 0x2ce1);
        re_mdio_write(sc, 0x06, 0xe02d);
        re_mdio_write(sc, 0x06, 0x59e0);
        re_mdio_write(sc, 0x06, 0x5b1f);
        re_mdio_write(sc, 0x06, 0x1e13);
        re_mdio_write(sc, 0x06, 0xe4e0);
        re_mdio_write(sc, 0x06, 0x2ce5);
        re_mdio_write(sc, 0x06, 0xe02d);
        re_mdio_write(sc, 0x06, 0xfde0);
        re_mdio_write(sc, 0x06, 0xe2d2);
        re_mdio_write(sc, 0x06, 0xe1e2);
        re_mdio_write(sc, 0x06, 0xd358);
        re_mdio_write(sc, 0x06, 0xf75a);
        re_mdio_write(sc, 0x06, 0x081e);
        re_mdio_write(sc, 0x06, 0x02e4);
        re_mdio_write(sc, 0x06, 0xe2d2);
        re_mdio_write(sc, 0x06, 0xe5e2);
        re_mdio_write(sc, 0x06, 0xd3ef);
        re_mdio_write(sc, 0x06, 0x46fe);
        re_mdio_write(sc, 0x06, 0xfd04);
        re_mdio_write(sc, 0x06, 0xf8f9);
        re_mdio_write(sc, 0x06, 0xfaef);
        re_mdio_write(sc, 0x06, 0x69e0);
        re_mdio_write(sc, 0x06, 0xe022);
        re_mdio_write(sc, 0x06, 0xe1e0);
        re_mdio_write(sc, 0x06, 0x2358);
        re_mdio_write(sc, 0x06, 0xc4e1);
        re_mdio_write(sc, 0x06, 0x8b6e);
        re_mdio_write(sc, 0x06, 0x1f10);
        re_mdio_write(sc, 0x06, 0x9e58);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x6ead);
        re_mdio_write(sc, 0x06, 0x2222);
        re_mdio_write(sc, 0x06, 0xac27);
        re_mdio_write(sc, 0x06, 0x55ac);
        re_mdio_write(sc, 0x06, 0x2602);
        re_mdio_write(sc, 0x06, 0xae1a);
        re_mdio_write(sc, 0x06, 0xd106);
        re_mdio_write(sc, 0x06, 0xbf3b);
        re_mdio_write(sc, 0x06, 0xba02);
        re_mdio_write(sc, 0x06, 0x2dc1);
        re_mdio_write(sc, 0x06, 0xd107);
        re_mdio_write(sc, 0x06, 0xbf3b);
        re_mdio_write(sc, 0x06, 0xbd02);
        re_mdio_write(sc, 0x06, 0x2dc1);
        re_mdio_write(sc, 0x06, 0xd107);
        re_mdio_write(sc, 0x06, 0xbf3b);
        re_mdio_write(sc, 0x06, 0xc002);
        re_mdio_write(sc, 0x06, 0x2dc1);
        re_mdio_write(sc, 0x06, 0xae30);
        re_mdio_write(sc, 0x06, 0xd103);
        re_mdio_write(sc, 0x06, 0xbf3b);
        re_mdio_write(sc, 0x06, 0xc302);
        re_mdio_write(sc, 0x06, 0x2dc1);
        re_mdio_write(sc, 0x06, 0xd100);
        re_mdio_write(sc, 0x06, 0xbf3b);
        re_mdio_write(sc, 0x06, 0xc602);
        re_mdio_write(sc, 0x06, 0x2dc1);
        re_mdio_write(sc, 0x06, 0xd100);
        re_mdio_write(sc, 0x06, 0xbf82);
        re_mdio_write(sc, 0x06, 0xca02);
        re_mdio_write(sc, 0x06, 0x2dc1);
        re_mdio_write(sc, 0x06, 0xd10f);
        re_mdio_write(sc, 0x06, 0xbf3b);
        re_mdio_write(sc, 0x06, 0xba02);
        re_mdio_write(sc, 0x06, 0x2dc1);
        re_mdio_write(sc, 0x06, 0xd101);
        re_mdio_write(sc, 0x06, 0xbf3b);
        re_mdio_write(sc, 0x06, 0xbd02);
        re_mdio_write(sc, 0x06, 0x2dc1);
        re_mdio_write(sc, 0x06, 0xd101);
        re_mdio_write(sc, 0x06, 0xbf3b);
        re_mdio_write(sc, 0x06, 0xc002);
        re_mdio_write(sc, 0x06, 0x2dc1);
        re_mdio_write(sc, 0x06, 0xef96);
        re_mdio_write(sc, 0x06, 0xfefd);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xd100);
        re_mdio_write(sc, 0x06, 0xbf3b);
        re_mdio_write(sc, 0x06, 0xc302);
        re_mdio_write(sc, 0x06, 0x2dc1);
        re_mdio_write(sc, 0x06, 0xd011);
        re_mdio_write(sc, 0x06, 0x022b);
        re_mdio_write(sc, 0x06, 0xfb59);
        re_mdio_write(sc, 0x06, 0x03ef);
        re_mdio_write(sc, 0x06, 0x01d1);
        re_mdio_write(sc, 0x06, 0x00a0);
        re_mdio_write(sc, 0x06, 0x0002);
        re_mdio_write(sc, 0x06, 0xd101);
        re_mdio_write(sc, 0x06, 0xbf3b);
        re_mdio_write(sc, 0x06, 0xc602);
        re_mdio_write(sc, 0x06, 0x2dc1);
        re_mdio_write(sc, 0x06, 0xd111);
        re_mdio_write(sc, 0x06, 0xad20);
        re_mdio_write(sc, 0x06, 0x020c);
        re_mdio_write(sc, 0x06, 0x11ad);
        re_mdio_write(sc, 0x06, 0x2102);
        re_mdio_write(sc, 0x06, 0x0c12);
        re_mdio_write(sc, 0x06, 0xbf82);
        re_mdio_write(sc, 0x06, 0xca02);
        re_mdio_write(sc, 0x06, 0x2dc1);
        re_mdio_write(sc, 0x06, 0xaec8);
        re_mdio_write(sc, 0x06, 0x70e4);
        re_mdio_write(sc, 0x06, 0x2602);
        re_mdio_write(sc, 0x06, 0x82d1);
        re_mdio_write(sc, 0x06, 0x05f8);
        re_mdio_write(sc, 0x06, 0xfaef);
        re_mdio_write(sc, 0x06, 0x69e0);
        re_mdio_write(sc, 0x06, 0xe2fe);
        re_mdio_write(sc, 0x06, 0xe1e2);
        re_mdio_write(sc, 0x06, 0xffad);
        re_mdio_write(sc, 0x06, 0x2d1a);
        re_mdio_write(sc, 0x06, 0xe0e1);
        re_mdio_write(sc, 0x06, 0x4ee1);
        re_mdio_write(sc, 0x06, 0xe14f);
        re_mdio_write(sc, 0x06, 0xac2d);
        re_mdio_write(sc, 0x06, 0x22f6);
        re_mdio_write(sc, 0x06, 0x0302);
        re_mdio_write(sc, 0x06, 0x033b);
        re_mdio_write(sc, 0x06, 0xf703);
        re_mdio_write(sc, 0x06, 0xf706);
        re_mdio_write(sc, 0x06, 0xbf84);
        re_mdio_write(sc, 0x06, 0x4402);
        re_mdio_write(sc, 0x06, 0x2d21);
        re_mdio_write(sc, 0x06, 0xae11);
        re_mdio_write(sc, 0x06, 0xe0e1);
        re_mdio_write(sc, 0x06, 0x4ee1);
        re_mdio_write(sc, 0x06, 0xe14f);
        re_mdio_write(sc, 0x06, 0xad2d);
        re_mdio_write(sc, 0x06, 0x08bf);
        re_mdio_write(sc, 0x06, 0x844f);
        re_mdio_write(sc, 0x06, 0x022d);
        re_mdio_write(sc, 0x06, 0x21f6);
        re_mdio_write(sc, 0x06, 0x06ef);
        re_mdio_write(sc, 0x06, 0x96fe);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xf8fa);
        re_mdio_write(sc, 0x06, 0xef69);
        re_mdio_write(sc, 0x06, 0x0283);
        re_mdio_write(sc, 0x06, 0x4502);
        re_mdio_write(sc, 0x06, 0x83a2);
        re_mdio_write(sc, 0x06, 0xe0e0);
        re_mdio_write(sc, 0x06, 0x00e1);
        re_mdio_write(sc, 0x06, 0xe001);
        re_mdio_write(sc, 0x06, 0xad27);
        re_mdio_write(sc, 0x06, 0x1fd1);
        re_mdio_write(sc, 0x06, 0x01bf);
        re_mdio_write(sc, 0x06, 0x843b);
        re_mdio_write(sc, 0x06, 0x022d);
        re_mdio_write(sc, 0x06, 0xc1e0);
        re_mdio_write(sc, 0x06, 0xe020);
        re_mdio_write(sc, 0x06, 0xe1e0);
        re_mdio_write(sc, 0x06, 0x21ad);
        re_mdio_write(sc, 0x06, 0x200e);
        re_mdio_write(sc, 0x06, 0xd100);
        re_mdio_write(sc, 0x06, 0xbf84);
        re_mdio_write(sc, 0x06, 0x3b02);
        re_mdio_write(sc, 0x06, 0x2dc1);
        re_mdio_write(sc, 0x06, 0xbf3b);
        re_mdio_write(sc, 0x06, 0x9602);
        re_mdio_write(sc, 0x06, 0x2d21);
        re_mdio_write(sc, 0x06, 0xef96);
        re_mdio_write(sc, 0x06, 0xfefc);
        re_mdio_write(sc, 0x06, 0x04f8);
        re_mdio_write(sc, 0x06, 0xf9fa);
        re_mdio_write(sc, 0x06, 0xef69);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x87ad);
        re_mdio_write(sc, 0x06, 0x204c);
        re_mdio_write(sc, 0x06, 0xd200);
        re_mdio_write(sc, 0x06, 0xe0e2);
        re_mdio_write(sc, 0x06, 0x0058);
        re_mdio_write(sc, 0x06, 0x010c);
        re_mdio_write(sc, 0x06, 0x021e);
        re_mdio_write(sc, 0x06, 0x20e0);
        re_mdio_write(sc, 0x06, 0xe000);
        re_mdio_write(sc, 0x06, 0x5810);
        re_mdio_write(sc, 0x06, 0x1e20);
        re_mdio_write(sc, 0x06, 0xe0e0);
        re_mdio_write(sc, 0x06, 0x3658);
        re_mdio_write(sc, 0x06, 0x031e);
        re_mdio_write(sc, 0x06, 0x20e0);
        re_mdio_write(sc, 0x06, 0xe022);
        re_mdio_write(sc, 0x06, 0xe1e0);
        re_mdio_write(sc, 0x06, 0x2358);
        re_mdio_write(sc, 0x06, 0xe01e);
        re_mdio_write(sc, 0x06, 0x20e0);
        re_mdio_write(sc, 0x06, 0x8b64);
        re_mdio_write(sc, 0x06, 0x1f02);
        re_mdio_write(sc, 0x06, 0x9e22);
        re_mdio_write(sc, 0x06, 0xe68b);
        re_mdio_write(sc, 0x06, 0x64ad);
        re_mdio_write(sc, 0x06, 0x3214);
        re_mdio_write(sc, 0x06, 0xad34);
        re_mdio_write(sc, 0x06, 0x11ef);
        re_mdio_write(sc, 0x06, 0x0258);
        re_mdio_write(sc, 0x06, 0x039e);
        re_mdio_write(sc, 0x06, 0x07ad);
        re_mdio_write(sc, 0x06, 0x3508);
        re_mdio_write(sc, 0x06, 0x5ac0);
        re_mdio_write(sc, 0x06, 0x9f04);
        re_mdio_write(sc, 0x06, 0xd101);
        re_mdio_write(sc, 0x06, 0xae02);
        re_mdio_write(sc, 0x06, 0xd100);
        re_mdio_write(sc, 0x06, 0xbf84);
        re_mdio_write(sc, 0x06, 0x3e02);
        re_mdio_write(sc, 0x06, 0x2dc1);
        re_mdio_write(sc, 0x06, 0xef96);
        re_mdio_write(sc, 0x06, 0xfefd);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xf8f9);
        re_mdio_write(sc, 0x06, 0xfbe0);
        re_mdio_write(sc, 0x06, 0x8b85);
        re_mdio_write(sc, 0x06, 0xad25);
        re_mdio_write(sc, 0x06, 0x22e0);
        re_mdio_write(sc, 0x06, 0xe022);
        re_mdio_write(sc, 0x06, 0xe1e0);
        re_mdio_write(sc, 0x06, 0x23e2);
        re_mdio_write(sc, 0x06, 0xe036);
        re_mdio_write(sc, 0x06, 0xe3e0);
        re_mdio_write(sc, 0x06, 0x375a);
        re_mdio_write(sc, 0x06, 0xc40d);
        re_mdio_write(sc, 0x06, 0x0158);
        re_mdio_write(sc, 0x06, 0x021e);
        re_mdio_write(sc, 0x06, 0x20e3);
        re_mdio_write(sc, 0x06, 0x8ae7);
        re_mdio_write(sc, 0x06, 0xac31);
        re_mdio_write(sc, 0x06, 0x60ac);
        re_mdio_write(sc, 0x06, 0x3a08);
        re_mdio_write(sc, 0x06, 0xac3e);
        re_mdio_write(sc, 0x06, 0x26ae);
        re_mdio_write(sc, 0x06, 0x67af);
        re_mdio_write(sc, 0x06, 0x8437);
        re_mdio_write(sc, 0x06, 0xad37);
        re_mdio_write(sc, 0x06, 0x61e0);
        re_mdio_write(sc, 0x06, 0x8ae8);
        re_mdio_write(sc, 0x06, 0x10e4);
        re_mdio_write(sc, 0x06, 0x8ae8);
        re_mdio_write(sc, 0x06, 0xe18a);
        re_mdio_write(sc, 0x06, 0xe91b);
        re_mdio_write(sc, 0x06, 0x109e);
        re_mdio_write(sc, 0x06, 0x02ae);
        re_mdio_write(sc, 0x06, 0x51d1);
        re_mdio_write(sc, 0x06, 0x00bf);
        re_mdio_write(sc, 0x06, 0x8441);
        re_mdio_write(sc, 0x06, 0x022d);
        re_mdio_write(sc, 0x06, 0xc1ee);
        re_mdio_write(sc, 0x06, 0x8ae8);
        re_mdio_write(sc, 0x06, 0x00ae);
        re_mdio_write(sc, 0x06, 0x43ad);
        re_mdio_write(sc, 0x06, 0x3627);
        re_mdio_write(sc, 0x06, 0xe08a);
        re_mdio_write(sc, 0x06, 0xeee1);
        re_mdio_write(sc, 0x06, 0x8aef);
        re_mdio_write(sc, 0x06, 0xef74);
        re_mdio_write(sc, 0x06, 0xe08a);
        re_mdio_write(sc, 0x06, 0xeae1);
        re_mdio_write(sc, 0x06, 0x8aeb);
        re_mdio_write(sc, 0x06, 0x1b74);
        re_mdio_write(sc, 0x06, 0x9e2e);
        re_mdio_write(sc, 0x06, 0x14e4);
        re_mdio_write(sc, 0x06, 0x8aea);
        re_mdio_write(sc, 0x06, 0xe58a);
        re_mdio_write(sc, 0x06, 0xebef);
        re_mdio_write(sc, 0x06, 0x74e0);
        re_mdio_write(sc, 0x06, 0x8aee);
        re_mdio_write(sc, 0x06, 0xe18a);
        re_mdio_write(sc, 0x06, 0xef1b);
        re_mdio_write(sc, 0x06, 0x479e);
        re_mdio_write(sc, 0x06, 0x0fae);
        re_mdio_write(sc, 0x06, 0x19ee);
        re_mdio_write(sc, 0x06, 0x8aea);
        re_mdio_write(sc, 0x06, 0x00ee);
        re_mdio_write(sc, 0x06, 0x8aeb);
        re_mdio_write(sc, 0x06, 0x00ae);
        re_mdio_write(sc, 0x06, 0x0fac);
        re_mdio_write(sc, 0x06, 0x390c);
        re_mdio_write(sc, 0x06, 0xd101);
        re_mdio_write(sc, 0x06, 0xbf84);
        re_mdio_write(sc, 0x06, 0x4102);
        re_mdio_write(sc, 0x06, 0x2dc1);
        re_mdio_write(sc, 0x06, 0xee8a);
        re_mdio_write(sc, 0x06, 0xe800);
        re_mdio_write(sc, 0x06, 0xe68a);
        re_mdio_write(sc, 0x06, 0xe7ff);
        re_mdio_write(sc, 0x06, 0xfdfc);
        re_mdio_write(sc, 0x06, 0x0400);
        re_mdio_write(sc, 0x06, 0xe234);
        re_mdio_write(sc, 0x06, 0xcce2);
        re_mdio_write(sc, 0x06, 0x0088);
        re_mdio_write(sc, 0x06, 0xe200);
        re_mdio_write(sc, 0x06, 0xa725);
        re_mdio_write(sc, 0x06, 0xe50a);
        re_mdio_write(sc, 0x06, 0x1de5);
        re_mdio_write(sc, 0x06, 0x0a2c);
        re_mdio_write(sc, 0x06, 0xe50a);
        re_mdio_write(sc, 0x06, 0x6de5);
        re_mdio_write(sc, 0x06, 0x0a1d);
        re_mdio_write(sc, 0x06, 0xe50a);
        re_mdio_write(sc, 0x06, 0x1ce5);
        re_mdio_write(sc, 0x06, 0x0a2d);
        re_mdio_write(sc, 0x06, 0xa755);
        re_mdio_write(sc, 0x05, 0x8b64);
        re_mdio_write(sc, 0x06, 0x0000);
        re_mdio_write(sc, 0x05, 0x8b94);
        re_mdio_write(sc, 0x06, 0x82cd);
        re_mdio_write(sc, 0x05, 0x8b85);
        re_mdio_write(sc, 0x06, 0x2000);
        re_mdio_write(sc, 0x05, 0x8aee);
        re_mdio_write(sc, 0x06, 0x03b8);
        re_mdio_write(sc, 0x05, 0x8ae8);
        re_mdio_write(sc, 0x06, 0x0002);
        PhyRegValue = re_mdio_read(sc, 0x01);
        PhyRegValue |= BIT_0;
        re_mdio_write(sc, 0x01, PhyRegValue);
        PhyRegValue = re_mdio_read(sc, 0x00);
        PhyRegValue |= BIT_0;
        re_mdio_write(sc, 0x00, PhyRegValue);
        re_mdio_write(sc, 0x1f, 0x0);
        re_mdio_write(sc, 0x1f, 0x0005);
        for (i=0; i<200; i++) {
                DELAY(100);
                PhyRegValue = re_mdio_read(sc, 0x00);
                if (PhyRegValue&0x0080)
                        break;
        }
        re_mdio_write(sc, 0x1f, 0x0007);
        re_mdio_write(sc, 0x1e, 0x0023);
        PhyRegValue = re_mdio_read(sc, 0x17);
        PhyRegValue &= ~(BIT_0);
        if (sc->RequiredSecLanDonglePatch)
                PhyRegValue &= ~(BIT_2);
        re_mdio_write(sc, 0x17, PhyRegValue);
        re_mdio_write(sc, 0x1f, 0x0007);
        re_mdio_write(sc, 0x1e, 0x0028);
        re_mdio_write(sc, 0x15, 0x0010);
        re_mdio_write(sc, 0x1f, 0x0007);
        re_mdio_write(sc, 0x1e, 0x0041);
        re_mdio_write(sc, 0x15, 0x0802);
        re_mdio_write(sc, 0x16, 0x2185);
        re_mdio_write(sc, 0x1f, 0x0000);
}

static void re_set_phy_mcu_8168e_2(struct re_softc *sc)
{
        u_int16_t PhyRegValue;
        int	i;

        if (re_efuse_read(sc, 0x22) == 0x0c) {
                re_mdio_write(sc, 0x1f, 0x0000);
                re_mdio_write(sc, 0x00, 0x1800);
                re_mdio_write(sc, 0x1f, 0x0007);
                re_mdio_write(sc, 0x1e, 0x0023);
                re_mdio_write(sc, 0x17, 0x0117);
                re_mdio_write(sc, 0x1f, 0x0007);
                re_mdio_write(sc, 0x1E, 0x002C);
                re_mdio_write(sc, 0x1B, 0x5000);
                re_mdio_write(sc, 0x1f, 0x0000);
                re_mdio_write(sc, 0x16, 0x4104);
                for (i=0; i<200; i++) {
                        DELAY(100);
                        PhyRegValue = re_mdio_read(sc, 0x1E);
                        PhyRegValue &= 0x03FF;
                        if (PhyRegValue== 0x000C)
                                break;
                }
                re_mdio_write(sc, 0x1f, 0x0005);
                for (i=0; i<200; i++) {
                        DELAY(100);
                        PhyRegValue = re_mdio_read(sc, 0x07);
                        if ((PhyRegValue&0x0020)==0)
                                break;
                }
                PhyRegValue = re_mdio_read(sc, 0x07);
                if (PhyRegValue & 0x0020) {
                        re_mdio_write(sc, 0x1f, 0x0007);
                        re_mdio_write(sc, 0x1e, 0x00a1);
                        re_mdio_write(sc, 0x17, 0x1000);
                        re_mdio_write(sc, 0x17, 0x0000);
                        re_mdio_write(sc, 0x17, 0x2000);
                        re_mdio_write(sc, 0x1e, 0x002f);
                        re_mdio_write(sc, 0x18, 0x9bfb);
                        re_mdio_write(sc, 0x1f, 0x0005);
                        re_mdio_write(sc, 0x07, 0x0000);
                        re_mdio_write(sc, 0x1f, 0x0000);
                }
                re_mdio_write(sc, 0x1f, 0x0005);
                re_mdio_write(sc, 0x05, 0xfff6);
                re_mdio_write(sc, 0x06, 0x0080);
                PhyRegValue = re_mdio_read(sc, 0x00);
                PhyRegValue &= ~(BIT_7);
                re_mdio_write(sc, 0x00, PhyRegValue);
                re_mdio_write(sc, 0x1f, 0x0002);
                PhyRegValue = re_mdio_read(sc, 0x08);
                PhyRegValue &= ~(BIT_7);
                re_mdio_write(sc, 0x08, PhyRegValue);
                re_mdio_write(sc, 0x1f, 0x0000);
                re_mdio_write(sc, 0x1f, 0x0007);
                re_mdio_write(sc, 0x1e, 0x0023);
                re_mdio_write(sc, 0x16, 0x0306);
                re_mdio_write(sc, 0x16, 0x0307);
                re_mdio_write(sc, 0x15, 0x000e);
                re_mdio_write(sc, 0x19, 0x000a);
                re_mdio_write(sc, 0x15, 0x0010);
                re_mdio_write(sc, 0x19, 0x0008);
                re_mdio_write(sc, 0x15, 0x0018);
                re_mdio_write(sc, 0x19, 0x4801);
                re_mdio_write(sc, 0x15, 0x0019);
                re_mdio_write(sc, 0x19, 0x6801);
                re_mdio_write(sc, 0x15, 0x001a);
                re_mdio_write(sc, 0x19, 0x66a1);
                re_mdio_write(sc, 0x15, 0x001f);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0020);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0021);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0022);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0023);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0024);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0025);
                re_mdio_write(sc, 0x19, 0x64a1);
                re_mdio_write(sc, 0x15, 0x0026);
                re_mdio_write(sc, 0x19, 0x40ea);
                re_mdio_write(sc, 0x15, 0x0027);
                re_mdio_write(sc, 0x19, 0x4503);
                re_mdio_write(sc, 0x15, 0x0028);
                re_mdio_write(sc, 0x19, 0x9f00);
                re_mdio_write(sc, 0x15, 0x0029);
                re_mdio_write(sc, 0x19, 0xa631);
                re_mdio_write(sc, 0x15, 0x002a);
                re_mdio_write(sc, 0x19, 0x9717);
                re_mdio_write(sc, 0x15, 0x002b);
                re_mdio_write(sc, 0x19, 0x302c);
                re_mdio_write(sc, 0x15, 0x002c);
                re_mdio_write(sc, 0x19, 0x4802);
                re_mdio_write(sc, 0x15, 0x002d);
                re_mdio_write(sc, 0x19, 0x58da);
                re_mdio_write(sc, 0x15, 0x002e);
                re_mdio_write(sc, 0x19, 0x400d);
                re_mdio_write(sc, 0x15, 0x002f);
                re_mdio_write(sc, 0x19, 0x4488);
                re_mdio_write(sc, 0x15, 0x0030);
                re_mdio_write(sc, 0x19, 0x9e00);
                re_mdio_write(sc, 0x15, 0x0031);
                re_mdio_write(sc, 0x19, 0x63c8);
                re_mdio_write(sc, 0x15, 0x0032);
                re_mdio_write(sc, 0x19, 0x6481);
                re_mdio_write(sc, 0x15, 0x0033);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0034);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0035);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0036);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0037);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0038);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0039);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x003a);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x003b);
                re_mdio_write(sc, 0x19, 0x63e8);
                re_mdio_write(sc, 0x15, 0x003c);
                re_mdio_write(sc, 0x19, 0x7d00);
                re_mdio_write(sc, 0x15, 0x003d);
                re_mdio_write(sc, 0x19, 0x59d4);
                re_mdio_write(sc, 0x15, 0x003e);
                re_mdio_write(sc, 0x19, 0x63f8);
                re_mdio_write(sc, 0x15, 0x0040);
                re_mdio_write(sc, 0x19, 0x64a1);
                re_mdio_write(sc, 0x15, 0x0041);
                re_mdio_write(sc, 0x19, 0x30de);
                re_mdio_write(sc, 0x15, 0x0044);
                re_mdio_write(sc, 0x19, 0x480f);
                re_mdio_write(sc, 0x15, 0x0045);
                re_mdio_write(sc, 0x19, 0x6800);
                re_mdio_write(sc, 0x15, 0x0046);
                re_mdio_write(sc, 0x19, 0x6680);
                re_mdio_write(sc, 0x15, 0x0047);
                re_mdio_write(sc, 0x19, 0x7c10);
                re_mdio_write(sc, 0x15, 0x0048);
                re_mdio_write(sc, 0x19, 0x63c8);
                re_mdio_write(sc, 0x15, 0x0049);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x004a);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x004b);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x004c);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x004d);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x004e);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x004f);
                re_mdio_write(sc, 0x19, 0x40ea);
                re_mdio_write(sc, 0x15, 0x0050);
                re_mdio_write(sc, 0x19, 0x4503);
                re_mdio_write(sc, 0x15, 0x0051);
                re_mdio_write(sc, 0x19, 0x58ca);
                re_mdio_write(sc, 0x15, 0x0052);
                re_mdio_write(sc, 0x19, 0x63c8);
                re_mdio_write(sc, 0x15, 0x0053);
                re_mdio_write(sc, 0x19, 0x63d8);
                re_mdio_write(sc, 0x15, 0x0054);
                re_mdio_write(sc, 0x19, 0x66a0);
                re_mdio_write(sc, 0x15, 0x0055);
                re_mdio_write(sc, 0x19, 0x9f00);
                re_mdio_write(sc, 0x15, 0x0056);
                re_mdio_write(sc, 0x19, 0x3000);
                re_mdio_write(sc, 0x15, 0x00a1);
                re_mdio_write(sc, 0x19, 0x3044);
                re_mdio_write(sc, 0x15, 0x00ab);
                re_mdio_write(sc, 0x19, 0x5820);
                re_mdio_write(sc, 0x15, 0x00ac);
                re_mdio_write(sc, 0x19, 0x5e04);
                re_mdio_write(sc, 0x15, 0x00ad);
                re_mdio_write(sc, 0x19, 0xb60c);
                re_mdio_write(sc, 0x15, 0x00af);
                re_mdio_write(sc, 0x19, 0x000a);
                re_mdio_write(sc, 0x15, 0x00b2);
                re_mdio_write(sc, 0x19, 0x30b9);
                re_mdio_write(sc, 0x15, 0x00b9);
                re_mdio_write(sc, 0x19, 0x4408);
                re_mdio_write(sc, 0x15, 0x00ba);
                re_mdio_write(sc, 0x19, 0x480b);
                re_mdio_write(sc, 0x15, 0x00bb);
                re_mdio_write(sc, 0x19, 0x5e00);
                re_mdio_write(sc, 0x15, 0x00bc);
                re_mdio_write(sc, 0x19, 0x405f);
                re_mdio_write(sc, 0x15, 0x00bd);
                re_mdio_write(sc, 0x19, 0x4448);
                re_mdio_write(sc, 0x15, 0x00be);
                re_mdio_write(sc, 0x19, 0x4020);
                re_mdio_write(sc, 0x15, 0x00bf);
                re_mdio_write(sc, 0x19, 0x4468);
                re_mdio_write(sc, 0x15, 0x00c0);
                re_mdio_write(sc, 0x19, 0x9c02);
                re_mdio_write(sc, 0x15, 0x00c1);
                re_mdio_write(sc, 0x19, 0x58a0);
                re_mdio_write(sc, 0x15, 0x00c2);
                re_mdio_write(sc, 0x19, 0xb605);
                re_mdio_write(sc, 0x15, 0x00c3);
                re_mdio_write(sc, 0x19, 0xc0d3);
                re_mdio_write(sc, 0x15, 0x00c4);
                re_mdio_write(sc, 0x19, 0x00e6);
                re_mdio_write(sc, 0x15, 0x00c5);
                re_mdio_write(sc, 0x19, 0xdaec);
                re_mdio_write(sc, 0x15, 0x00c6);
                re_mdio_write(sc, 0x19, 0x00fa);
                re_mdio_write(sc, 0x15, 0x00c7);
                re_mdio_write(sc, 0x19, 0x9df9);
                re_mdio_write(sc, 0x15, 0x0112);
                re_mdio_write(sc, 0x19, 0x6421);
                re_mdio_write(sc, 0x15, 0x0113);
                re_mdio_write(sc, 0x19, 0x7c08);
                re_mdio_write(sc, 0x15, 0x0114);
                re_mdio_write(sc, 0x19, 0x63f0);
                re_mdio_write(sc, 0x15, 0x0115);
                re_mdio_write(sc, 0x19, 0x4003);
                re_mdio_write(sc, 0x15, 0x0116);
                re_mdio_write(sc, 0x19, 0x4418);
                re_mdio_write(sc, 0x15, 0x0117);
                re_mdio_write(sc, 0x19, 0x9b00);
                re_mdio_write(sc, 0x15, 0x0118);
                re_mdio_write(sc, 0x19, 0x6461);
                re_mdio_write(sc, 0x15, 0x0119);
                re_mdio_write(sc, 0x19, 0x64e1);
                re_mdio_write(sc, 0x15, 0x011a);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0150);
                re_mdio_write(sc, 0x19, 0x7c80);
                re_mdio_write(sc, 0x15, 0x0151);
                re_mdio_write(sc, 0x19, 0x6461);
                re_mdio_write(sc, 0x15, 0x0152);
                re_mdio_write(sc, 0x19, 0x4003);
                re_mdio_write(sc, 0x15, 0x0153);
                re_mdio_write(sc, 0x19, 0x4540);
                re_mdio_write(sc, 0x15, 0x0154);
                re_mdio_write(sc, 0x19, 0x9f00);
                re_mdio_write(sc, 0x15, 0x0155);
                re_mdio_write(sc, 0x19, 0x9d00);
                re_mdio_write(sc, 0x15, 0x0156);
                re_mdio_write(sc, 0x19, 0x7c40);
                re_mdio_write(sc, 0x15, 0x0157);
                re_mdio_write(sc, 0x19, 0x6421);
                re_mdio_write(sc, 0x15, 0x0158);
                re_mdio_write(sc, 0x19, 0x7c80);
                re_mdio_write(sc, 0x15, 0x0159);
                re_mdio_write(sc, 0x19, 0x64a1);
                re_mdio_write(sc, 0x15, 0x015a);
                re_mdio_write(sc, 0x19, 0x30fe);
                re_mdio_write(sc, 0x15, 0x029c);
                re_mdio_write(sc, 0x19, 0x0070);
                re_mdio_write(sc, 0x15, 0x02b2);
                re_mdio_write(sc, 0x19, 0x005a);
                re_mdio_write(sc, 0x15, 0x02bd);
                re_mdio_write(sc, 0x19, 0xa522);
                re_mdio_write(sc, 0x15, 0x02ce);
                re_mdio_write(sc, 0x19, 0xb63e);
                re_mdio_write(sc, 0x15, 0x02d9);
                re_mdio_write(sc, 0x19, 0x32df);
                re_mdio_write(sc, 0x15, 0x02df);
                re_mdio_write(sc, 0x19, 0x4500);
                re_mdio_write(sc, 0x15, 0x02e7);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x02f4);
                re_mdio_write(sc, 0x19, 0xb618);
                re_mdio_write(sc, 0x15, 0x02fb);
                re_mdio_write(sc, 0x19, 0xb900);
                re_mdio_write(sc, 0x15, 0x02fc);
                re_mdio_write(sc, 0x19, 0x49b5);
                re_mdio_write(sc, 0x15, 0x02fd);
                re_mdio_write(sc, 0x19, 0x6812);
                re_mdio_write(sc, 0x15, 0x02fe);
                re_mdio_write(sc, 0x19, 0x66a0);
                re_mdio_write(sc, 0x15, 0x02ff);
                re_mdio_write(sc, 0x19, 0x9900);
                re_mdio_write(sc, 0x15, 0x0300);
                re_mdio_write(sc, 0x19, 0x64a0);
                re_mdio_write(sc, 0x15, 0x0301);
                re_mdio_write(sc, 0x19, 0x3316);
                re_mdio_write(sc, 0x15, 0x0308);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x030c);
                re_mdio_write(sc, 0x19, 0x3000);
                re_mdio_write(sc, 0x15, 0x0312);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0313);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0314);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0315);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0316);
                re_mdio_write(sc, 0x19, 0x49b5);
                re_mdio_write(sc, 0x15, 0x0317);
                re_mdio_write(sc, 0x19, 0x7d00);
                re_mdio_write(sc, 0x15, 0x0318);
                re_mdio_write(sc, 0x19, 0x4d00);
                re_mdio_write(sc, 0x15, 0x0319);
                re_mdio_write(sc, 0x19, 0x6810);
                re_mdio_write(sc, 0x15, 0x031a);
                re_mdio_write(sc, 0x19, 0x6c08);
                re_mdio_write(sc, 0x15, 0x031b);
                re_mdio_write(sc, 0x19, 0x4925);
                re_mdio_write(sc, 0x15, 0x031c);
                re_mdio_write(sc, 0x19, 0x403b);
                re_mdio_write(sc, 0x15, 0x031d);
                re_mdio_write(sc, 0x19, 0xa602);
                re_mdio_write(sc, 0x15, 0x031e);
                re_mdio_write(sc, 0x19, 0x402f);
                re_mdio_write(sc, 0x15, 0x031f);
                re_mdio_write(sc, 0x19, 0x4484);
                re_mdio_write(sc, 0x15, 0x0320);
                re_mdio_write(sc, 0x19, 0x40c8);
                re_mdio_write(sc, 0x15, 0x0321);
                re_mdio_write(sc, 0x19, 0x44c4);
                re_mdio_write(sc, 0x15, 0x0322);
                re_mdio_write(sc, 0x19, 0x404f);
                re_mdio_write(sc, 0x15, 0x0323);
                re_mdio_write(sc, 0x19, 0x44c8);
                re_mdio_write(sc, 0x15, 0x0324);
                re_mdio_write(sc, 0x19, 0xd64f);
                re_mdio_write(sc, 0x15, 0x0325);
                re_mdio_write(sc, 0x19, 0x00e7);
                re_mdio_write(sc, 0x15, 0x0326);
                re_mdio_write(sc, 0x19, 0x7c08);
                re_mdio_write(sc, 0x15, 0x0327);
                re_mdio_write(sc, 0x19, 0x8203);
                re_mdio_write(sc, 0x15, 0x0328);
                re_mdio_write(sc, 0x19, 0x4d48);
                re_mdio_write(sc, 0x15, 0x0329);
                re_mdio_write(sc, 0x19, 0x332b);
                re_mdio_write(sc, 0x15, 0x032a);
                re_mdio_write(sc, 0x19, 0x4d40);
                re_mdio_write(sc, 0x15, 0x032c);
                re_mdio_write(sc, 0x19, 0x00f8);
                re_mdio_write(sc, 0x15, 0x032d);
                re_mdio_write(sc, 0x19, 0x82b2);
                re_mdio_write(sc, 0x15, 0x032f);
                re_mdio_write(sc, 0x19, 0x00b0);
                re_mdio_write(sc, 0x15, 0x0332);
                re_mdio_write(sc, 0x19, 0x91f2);
                re_mdio_write(sc, 0x15, 0x033f);
                re_mdio_write(sc, 0x19, 0xb6cd);
                re_mdio_write(sc, 0x15, 0x0340);
                re_mdio_write(sc, 0x19, 0x9e01);
                re_mdio_write(sc, 0x15, 0x0341);
                re_mdio_write(sc, 0x19, 0xd11d);
                re_mdio_write(sc, 0x15, 0x0342);
                re_mdio_write(sc, 0x19, 0x009d);
                re_mdio_write(sc, 0x15, 0x0343);
                re_mdio_write(sc, 0x19, 0xbb1c);
                re_mdio_write(sc, 0x15, 0x0344);
                re_mdio_write(sc, 0x19, 0x8102);
                re_mdio_write(sc, 0x15, 0x0345);
                re_mdio_write(sc, 0x19, 0x3348);
                re_mdio_write(sc, 0x15, 0x0346);
                re_mdio_write(sc, 0x19, 0xa231);
                re_mdio_write(sc, 0x15, 0x0347);
                re_mdio_write(sc, 0x19, 0x335b);
                re_mdio_write(sc, 0x15, 0x0348);
                re_mdio_write(sc, 0x19, 0x91f7);
                re_mdio_write(sc, 0x15, 0x0349);
                re_mdio_write(sc, 0x19, 0xc218);
                re_mdio_write(sc, 0x15, 0x034a);
                re_mdio_write(sc, 0x19, 0x00f5);
                re_mdio_write(sc, 0x15, 0x034b);
                re_mdio_write(sc, 0x19, 0x335b);
                re_mdio_write(sc, 0x15, 0x034c);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x034d);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x034e);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x034f);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0350);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x035b);
                re_mdio_write(sc, 0x19, 0xa23c);
                re_mdio_write(sc, 0x15, 0x035c);
                re_mdio_write(sc, 0x19, 0x7c08);
                re_mdio_write(sc, 0x15, 0x035d);
                re_mdio_write(sc, 0x19, 0x4c00);
                re_mdio_write(sc, 0x15, 0x035e);
                re_mdio_write(sc, 0x19, 0x3397);
                re_mdio_write(sc, 0x15, 0x0363);
                re_mdio_write(sc, 0x19, 0xb6a9);
                re_mdio_write(sc, 0x15, 0x0366);
                re_mdio_write(sc, 0x19, 0x00f5);
                re_mdio_write(sc, 0x15, 0x0382);
                re_mdio_write(sc, 0x19, 0x7c40);
                re_mdio_write(sc, 0x15, 0x0388);
                re_mdio_write(sc, 0x19, 0x0084);
                re_mdio_write(sc, 0x15, 0x0389);
                re_mdio_write(sc, 0x19, 0xdd17);
                re_mdio_write(sc, 0x15, 0x038a);
                re_mdio_write(sc, 0x19, 0x000b);
                re_mdio_write(sc, 0x15, 0x038b);
                re_mdio_write(sc, 0x19, 0xa10a);
                re_mdio_write(sc, 0x15, 0x038c);
                re_mdio_write(sc, 0x19, 0x337e);
                re_mdio_write(sc, 0x15, 0x038d);
                re_mdio_write(sc, 0x19, 0x6c0b);
                re_mdio_write(sc, 0x15, 0x038e);
                re_mdio_write(sc, 0x19, 0xa107);
                re_mdio_write(sc, 0x15, 0x038f);
                re_mdio_write(sc, 0x19, 0x6c08);
                re_mdio_write(sc, 0x15, 0x0390);
                re_mdio_write(sc, 0x19, 0xc017);
                re_mdio_write(sc, 0x15, 0x0391);
                re_mdio_write(sc, 0x19, 0x0004);
                re_mdio_write(sc, 0x15, 0x0392);
                re_mdio_write(sc, 0x19, 0xd64f);
                re_mdio_write(sc, 0x15, 0x0393);
                re_mdio_write(sc, 0x19, 0x00f4);
                re_mdio_write(sc, 0x15, 0x0397);
                re_mdio_write(sc, 0x19, 0x4098);
                re_mdio_write(sc, 0x15, 0x0398);
                re_mdio_write(sc, 0x19, 0x4408);
                re_mdio_write(sc, 0x15, 0x0399);
                re_mdio_write(sc, 0x19, 0x55bf);
                re_mdio_write(sc, 0x15, 0x039a);
                re_mdio_write(sc, 0x19, 0x4bb9);
                re_mdio_write(sc, 0x15, 0x039b);
                re_mdio_write(sc, 0x19, 0x6810);
                re_mdio_write(sc, 0x15, 0x039c);
                re_mdio_write(sc, 0x19, 0x4b29);
                re_mdio_write(sc, 0x15, 0x039d);
                re_mdio_write(sc, 0x19, 0x4041);
                re_mdio_write(sc, 0x15, 0x039e);
                re_mdio_write(sc, 0x19, 0x442a);
                re_mdio_write(sc, 0x15, 0x039f);
                re_mdio_write(sc, 0x19, 0x4029);
                re_mdio_write(sc, 0x15, 0x03aa);
                re_mdio_write(sc, 0x19, 0x33b8);
                re_mdio_write(sc, 0x15, 0x03b6);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x03b7);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x03b8);
                re_mdio_write(sc, 0x19, 0x543f);
                re_mdio_write(sc, 0x15, 0x03b9);
                re_mdio_write(sc, 0x19, 0x499a);
                re_mdio_write(sc, 0x15, 0x03ba);
                re_mdio_write(sc, 0x19, 0x7c40);
                re_mdio_write(sc, 0x15, 0x03bb);
                re_mdio_write(sc, 0x19, 0x4c40);
                re_mdio_write(sc, 0x15, 0x03bc);
                re_mdio_write(sc, 0x19, 0x490a);
                re_mdio_write(sc, 0x15, 0x03bd);
                re_mdio_write(sc, 0x19, 0x405e);
                re_mdio_write(sc, 0x15, 0x03c2);
                re_mdio_write(sc, 0x19, 0x9a03);
                re_mdio_write(sc, 0x15, 0x03c4);
                re_mdio_write(sc, 0x19, 0x0015);
                re_mdio_write(sc, 0x15, 0x03c5);
                re_mdio_write(sc, 0x19, 0x9e03);
                re_mdio_write(sc, 0x15, 0x03c8);
                re_mdio_write(sc, 0x19, 0x9cf7);
                re_mdio_write(sc, 0x15, 0x03c9);
                re_mdio_write(sc, 0x19, 0x7c12);
                re_mdio_write(sc, 0x15, 0x03ca);
                re_mdio_write(sc, 0x19, 0x4c52);
                re_mdio_write(sc, 0x15, 0x03cb);
                re_mdio_write(sc, 0x19, 0x4458);
                re_mdio_write(sc, 0x15, 0x03cd);
                re_mdio_write(sc, 0x19, 0x4c40);
                re_mdio_write(sc, 0x15, 0x03ce);
                re_mdio_write(sc, 0x19, 0x33bf);
                re_mdio_write(sc, 0x15, 0x03cf);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x03d0);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x03d1);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x03d5);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x03d6);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x03d7);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x03d8);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x03d9);
                re_mdio_write(sc, 0x19, 0x49bb);
                re_mdio_write(sc, 0x15, 0x03da);
                re_mdio_write(sc, 0x19, 0x4478);
                re_mdio_write(sc, 0x15, 0x03db);
                re_mdio_write(sc, 0x19, 0x492b);
                re_mdio_write(sc, 0x15, 0x03dc);
                re_mdio_write(sc, 0x19, 0x7c01);
                re_mdio_write(sc, 0x15, 0x03dd);
                re_mdio_write(sc, 0x19, 0x4c00);
                re_mdio_write(sc, 0x15, 0x03de);
                re_mdio_write(sc, 0x19, 0xbd1a);
                re_mdio_write(sc, 0x15, 0x03df);
                re_mdio_write(sc, 0x19, 0xc428);
                re_mdio_write(sc, 0x15, 0x03e0);
                re_mdio_write(sc, 0x19, 0x0008);
                re_mdio_write(sc, 0x15, 0x03e1);
                re_mdio_write(sc, 0x19, 0x9cfd);
                re_mdio_write(sc, 0x15, 0x03e2);
                re_mdio_write(sc, 0x19, 0x7c12);
                re_mdio_write(sc, 0x15, 0x03e3);
                re_mdio_write(sc, 0x19, 0x4c52);
                re_mdio_write(sc, 0x15, 0x03e4);
                re_mdio_write(sc, 0x19, 0x4458);
                re_mdio_write(sc, 0x15, 0x03e5);
                re_mdio_write(sc, 0x19, 0x7c12);
                re_mdio_write(sc, 0x15, 0x03e6);
                re_mdio_write(sc, 0x19, 0x4c40);
                re_mdio_write(sc, 0x15, 0x03e7);
                re_mdio_write(sc, 0x19, 0x33de);
                re_mdio_write(sc, 0x15, 0x03e8);
                re_mdio_write(sc, 0x19, 0xc218);
                re_mdio_write(sc, 0x15, 0x03e9);
                re_mdio_write(sc, 0x19, 0x0002);
                re_mdio_write(sc, 0x15, 0x03ea);
                re_mdio_write(sc, 0x19, 0x32df);
                re_mdio_write(sc, 0x15, 0x03eb);
                re_mdio_write(sc, 0x19, 0x3316);
                re_mdio_write(sc, 0x15, 0x03ec);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x03ed);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x03ee);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x03ef);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x03f7);
                re_mdio_write(sc, 0x19, 0x330c);
                re_mdio_write(sc, 0x16, 0x0306);
                re_mdio_write(sc, 0x16, 0x0300);
                re_mdio_write(sc, 0x1f, 0x0005);
                re_mdio_write(sc, 0x05, 0xfff6);
                re_mdio_write(sc, 0x06, 0x0080);
                re_mdio_write(sc, 0x05, 0x8000);
                re_mdio_write(sc, 0x06, 0x0280);
                re_mdio_write(sc, 0x06, 0x48f7);
                re_mdio_write(sc, 0x06, 0x00e0);
                re_mdio_write(sc, 0x06, 0xfff7);
                re_mdio_write(sc, 0x06, 0xa080);
                re_mdio_write(sc, 0x06, 0x02ae);
                re_mdio_write(sc, 0x06, 0xf602);
                re_mdio_write(sc, 0x06, 0x0200);
                re_mdio_write(sc, 0x06, 0x0280);
                re_mdio_write(sc, 0x06, 0x9002);
                re_mdio_write(sc, 0x06, 0x0224);
                re_mdio_write(sc, 0x06, 0x0202);
                re_mdio_write(sc, 0x06, 0x3402);
                re_mdio_write(sc, 0x06, 0x027f);
                re_mdio_write(sc, 0x06, 0x0280);
                re_mdio_write(sc, 0x06, 0xa602);
                re_mdio_write(sc, 0x06, 0x80bf);
                re_mdio_write(sc, 0x06, 0xe08b);
                re_mdio_write(sc, 0x06, 0x88e1);
                re_mdio_write(sc, 0x06, 0x8b89);
                re_mdio_write(sc, 0x06, 0x1e01);
                re_mdio_write(sc, 0x06, 0xe18b);
                re_mdio_write(sc, 0x06, 0x8a1e);
                re_mdio_write(sc, 0x06, 0x01e1);
                re_mdio_write(sc, 0x06, 0x8b8b);
                re_mdio_write(sc, 0x06, 0x1e01);
                re_mdio_write(sc, 0x06, 0xe18b);
                re_mdio_write(sc, 0x06, 0x8c1e);
                re_mdio_write(sc, 0x06, 0x01e1);
                re_mdio_write(sc, 0x06, 0x8b8d);
                re_mdio_write(sc, 0x06, 0x1e01);
                re_mdio_write(sc, 0x06, 0xe18b);
                re_mdio_write(sc, 0x06, 0x8e1e);
                re_mdio_write(sc, 0x06, 0x01a0);
                re_mdio_write(sc, 0x06, 0x00c7);
                re_mdio_write(sc, 0x06, 0xaebb);
                re_mdio_write(sc, 0x06, 0xee8a);
                re_mdio_write(sc, 0x06, 0xe600);
                re_mdio_write(sc, 0x06, 0xee8a);
                re_mdio_write(sc, 0x06, 0xee03);
                re_mdio_write(sc, 0x06, 0xee8a);
                re_mdio_write(sc, 0x06, 0xefb8);
                re_mdio_write(sc, 0x06, 0xee8a);
                re_mdio_write(sc, 0x06, 0xe902);
                re_mdio_write(sc, 0x06, 0xee8b);
                re_mdio_write(sc, 0x06, 0x8285);
                re_mdio_write(sc, 0x06, 0xee8b);
                re_mdio_write(sc, 0x06, 0x8520);
                re_mdio_write(sc, 0x06, 0xee8b);
                re_mdio_write(sc, 0x06, 0x8701);
                re_mdio_write(sc, 0x06, 0xd481);
                re_mdio_write(sc, 0x06, 0x35e4);
                re_mdio_write(sc, 0x06, 0x8b94);
                re_mdio_write(sc, 0x06, 0xe58b);
                re_mdio_write(sc, 0x06, 0x95bf);
                re_mdio_write(sc, 0x06, 0x8b88);
                re_mdio_write(sc, 0x06, 0xec00);
                re_mdio_write(sc, 0x06, 0x19a9);
                re_mdio_write(sc, 0x06, 0x8b90);
                re_mdio_write(sc, 0x06, 0xf9ee);
                re_mdio_write(sc, 0x06, 0xfff6);
                re_mdio_write(sc, 0x06, 0x00ee);
                re_mdio_write(sc, 0x06, 0xfff7);
                re_mdio_write(sc, 0x06, 0xffe0);
                re_mdio_write(sc, 0x06, 0xe140);
                re_mdio_write(sc, 0x06, 0xe1e1);
                re_mdio_write(sc, 0x06, 0x41f7);
                re_mdio_write(sc, 0x06, 0x2ff6);
                re_mdio_write(sc, 0x06, 0x28e4);
                re_mdio_write(sc, 0x06, 0xe140);
                re_mdio_write(sc, 0x06, 0xe5e1);
                re_mdio_write(sc, 0x06, 0x4104);
                re_mdio_write(sc, 0x06, 0xf8e0);
                re_mdio_write(sc, 0x06, 0x8b89);
                re_mdio_write(sc, 0x06, 0xad20);
                re_mdio_write(sc, 0x06, 0x0dee);
                re_mdio_write(sc, 0x06, 0x8b89);
                re_mdio_write(sc, 0x06, 0x0002);
                re_mdio_write(sc, 0x06, 0x82f4);
                re_mdio_write(sc, 0x06, 0x021f);
                re_mdio_write(sc, 0x06, 0x4102);
                re_mdio_write(sc, 0x06, 0x2812);
                re_mdio_write(sc, 0x06, 0xfc04);
                re_mdio_write(sc, 0x06, 0xf8e0);
                re_mdio_write(sc, 0x06, 0x8b8d);
                re_mdio_write(sc, 0x06, 0xad20);
                re_mdio_write(sc, 0x06, 0x10ee);
                re_mdio_write(sc, 0x06, 0x8b8d);
                re_mdio_write(sc, 0x06, 0x0002);
                re_mdio_write(sc, 0x06, 0x139d);
                re_mdio_write(sc, 0x06, 0x0281);
                re_mdio_write(sc, 0x06, 0xd602);
                re_mdio_write(sc, 0x06, 0x1f99);
                re_mdio_write(sc, 0x06, 0x0227);
                re_mdio_write(sc, 0x06, 0xeafc);
                re_mdio_write(sc, 0x06, 0x04f8);
                re_mdio_write(sc, 0x06, 0xe08b);
                re_mdio_write(sc, 0x06, 0x8ead);
                re_mdio_write(sc, 0x06, 0x2014);
                re_mdio_write(sc, 0x06, 0xf620);
                re_mdio_write(sc, 0x06, 0xe48b);
                re_mdio_write(sc, 0x06, 0x8e02);
                re_mdio_write(sc, 0x06, 0x8104);
                re_mdio_write(sc, 0x06, 0x021b);
                re_mdio_write(sc, 0x06, 0xf402);
                re_mdio_write(sc, 0x06, 0x2c9c);
                re_mdio_write(sc, 0x06, 0x0281);
                re_mdio_write(sc, 0x06, 0x7902);
                re_mdio_write(sc, 0x06, 0x8443);
                re_mdio_write(sc, 0x06, 0xad22);
                re_mdio_write(sc, 0x06, 0x11f6);
                re_mdio_write(sc, 0x06, 0x22e4);
                re_mdio_write(sc, 0x06, 0x8b8e);
                re_mdio_write(sc, 0x06, 0x022c);
                re_mdio_write(sc, 0x06, 0x4602);
                re_mdio_write(sc, 0x06, 0x2ac5);
                re_mdio_write(sc, 0x06, 0x0229);
                re_mdio_write(sc, 0x06, 0x2002);
                re_mdio_write(sc, 0x06, 0x2b91);
                re_mdio_write(sc, 0x06, 0xad25);
                re_mdio_write(sc, 0x06, 0x11f6);
                re_mdio_write(sc, 0x06, 0x25e4);
                re_mdio_write(sc, 0x06, 0x8b8e);
                re_mdio_write(sc, 0x06, 0x0284);
                re_mdio_write(sc, 0x06, 0xe202);
                re_mdio_write(sc, 0x06, 0x043a);
                re_mdio_write(sc, 0x06, 0x021a);
                re_mdio_write(sc, 0x06, 0x5902);
                re_mdio_write(sc, 0x06, 0x2bfc);
                re_mdio_write(sc, 0x06, 0xfc04);
                re_mdio_write(sc, 0x06, 0xf8fa);
                re_mdio_write(sc, 0x06, 0xef69);
                re_mdio_write(sc, 0x06, 0xe0e0);
                re_mdio_write(sc, 0x06, 0x00e1);
                re_mdio_write(sc, 0x06, 0xe001);
                re_mdio_write(sc, 0x06, 0xad27);
                re_mdio_write(sc, 0x06, 0x1fd1);
                re_mdio_write(sc, 0x06, 0x01bf);
                re_mdio_write(sc, 0x06, 0x8638);
                re_mdio_write(sc, 0x06, 0x022f);
                re_mdio_write(sc, 0x06, 0x50e0);
                re_mdio_write(sc, 0x06, 0xe020);
                re_mdio_write(sc, 0x06, 0xe1e0);
                re_mdio_write(sc, 0x06, 0x21ad);
                re_mdio_write(sc, 0x06, 0x200e);
                re_mdio_write(sc, 0x06, 0xd100);
                re_mdio_write(sc, 0x06, 0xbf86);
                re_mdio_write(sc, 0x06, 0x3802);
                re_mdio_write(sc, 0x06, 0x2f50);
                re_mdio_write(sc, 0x06, 0xbf3d);
                re_mdio_write(sc, 0x06, 0x3902);
                re_mdio_write(sc, 0x06, 0x2eb0);
                re_mdio_write(sc, 0x06, 0xef96);
                re_mdio_write(sc, 0x06, 0xfefc);
                re_mdio_write(sc, 0x06, 0x0402);
                re_mdio_write(sc, 0x06, 0x8591);
                re_mdio_write(sc, 0x06, 0x0281);
                re_mdio_write(sc, 0x06, 0x3c05);
                re_mdio_write(sc, 0x06, 0xf8fa);
                re_mdio_write(sc, 0x06, 0xef69);
                re_mdio_write(sc, 0x06, 0xe0e2);
                re_mdio_write(sc, 0x06, 0xfee1);
                re_mdio_write(sc, 0x06, 0xe2ff);
                re_mdio_write(sc, 0x06, 0xad2d);
                re_mdio_write(sc, 0x06, 0x1ae0);
                re_mdio_write(sc, 0x06, 0xe14e);
                re_mdio_write(sc, 0x06, 0xe1e1);
                re_mdio_write(sc, 0x06, 0x4fac);
                re_mdio_write(sc, 0x06, 0x2d22);
                re_mdio_write(sc, 0x06, 0xf603);
                re_mdio_write(sc, 0x06, 0x0203);
                re_mdio_write(sc, 0x06, 0x36f7);
                re_mdio_write(sc, 0x06, 0x03f7);
                re_mdio_write(sc, 0x06, 0x06bf);
                re_mdio_write(sc, 0x06, 0x8622);
                re_mdio_write(sc, 0x06, 0x022e);
                re_mdio_write(sc, 0x06, 0xb0ae);
                re_mdio_write(sc, 0x06, 0x11e0);
                re_mdio_write(sc, 0x06, 0xe14e);
                re_mdio_write(sc, 0x06, 0xe1e1);
                re_mdio_write(sc, 0x06, 0x4fad);
                re_mdio_write(sc, 0x06, 0x2d08);
                re_mdio_write(sc, 0x06, 0xbf86);
                re_mdio_write(sc, 0x06, 0x2d02);
                re_mdio_write(sc, 0x06, 0x2eb0);
                re_mdio_write(sc, 0x06, 0xf606);
                re_mdio_write(sc, 0x06, 0xef96);
                re_mdio_write(sc, 0x06, 0xfefc);
                re_mdio_write(sc, 0x06, 0x04f8);
                re_mdio_write(sc, 0x06, 0xf9fa);
                re_mdio_write(sc, 0x06, 0xef69);
                re_mdio_write(sc, 0x06, 0xe08b);
                re_mdio_write(sc, 0x06, 0x87ad);
                re_mdio_write(sc, 0x06, 0x204c);
                re_mdio_write(sc, 0x06, 0xd200);
                re_mdio_write(sc, 0x06, 0xe0e2);
                re_mdio_write(sc, 0x06, 0x0058);
                re_mdio_write(sc, 0x06, 0x010c);
                re_mdio_write(sc, 0x06, 0x021e);
                re_mdio_write(sc, 0x06, 0x20e0);
                re_mdio_write(sc, 0x06, 0xe000);
                re_mdio_write(sc, 0x06, 0x5810);
                re_mdio_write(sc, 0x06, 0x1e20);
                re_mdio_write(sc, 0x06, 0xe0e0);
                re_mdio_write(sc, 0x06, 0x3658);
                re_mdio_write(sc, 0x06, 0x031e);
                re_mdio_write(sc, 0x06, 0x20e0);
                re_mdio_write(sc, 0x06, 0xe022);
                re_mdio_write(sc, 0x06, 0xe1e0);
                re_mdio_write(sc, 0x06, 0x2358);
                re_mdio_write(sc, 0x06, 0xe01e);
                re_mdio_write(sc, 0x06, 0x20e0);
                re_mdio_write(sc, 0x06, 0x8ae6);
                re_mdio_write(sc, 0x06, 0x1f02);
                re_mdio_write(sc, 0x06, 0x9e22);
                re_mdio_write(sc, 0x06, 0xe68a);
                re_mdio_write(sc, 0x06, 0xe6ad);
                re_mdio_write(sc, 0x06, 0x3214);
                re_mdio_write(sc, 0x06, 0xad34);
                re_mdio_write(sc, 0x06, 0x11ef);
                re_mdio_write(sc, 0x06, 0x0258);
                re_mdio_write(sc, 0x06, 0x039e);
                re_mdio_write(sc, 0x06, 0x07ad);
                re_mdio_write(sc, 0x06, 0x3508);
                re_mdio_write(sc, 0x06, 0x5ac0);
                re_mdio_write(sc, 0x06, 0x9f04);
                re_mdio_write(sc, 0x06, 0xd101);
                re_mdio_write(sc, 0x06, 0xae02);
                re_mdio_write(sc, 0x06, 0xd100);
                re_mdio_write(sc, 0x06, 0xbf86);
                re_mdio_write(sc, 0x06, 0x3e02);
                re_mdio_write(sc, 0x06, 0x2f50);
                re_mdio_write(sc, 0x06, 0xef96);
                re_mdio_write(sc, 0x06, 0xfefd);
                re_mdio_write(sc, 0x06, 0xfc04);
                re_mdio_write(sc, 0x06, 0xf8f9);
                re_mdio_write(sc, 0x06, 0xfae0);
                re_mdio_write(sc, 0x06, 0x8b81);
                re_mdio_write(sc, 0x06, 0xac26);
                re_mdio_write(sc, 0x06, 0x0ee0);
                re_mdio_write(sc, 0x06, 0x8b81);
                re_mdio_write(sc, 0x06, 0xac21);
                re_mdio_write(sc, 0x06, 0x08e0);
                re_mdio_write(sc, 0x06, 0x8b87);
                re_mdio_write(sc, 0x06, 0xac24);
                re_mdio_write(sc, 0x06, 0x02ae);
                re_mdio_write(sc, 0x06, 0x6bee);
                re_mdio_write(sc, 0x06, 0xe0ea);
                re_mdio_write(sc, 0x06, 0x00ee);
                re_mdio_write(sc, 0x06, 0xe0eb);
                re_mdio_write(sc, 0x06, 0x00e2);
                re_mdio_write(sc, 0x06, 0xe07c);
                re_mdio_write(sc, 0x06, 0xe3e0);
                re_mdio_write(sc, 0x06, 0x7da5);
                re_mdio_write(sc, 0x06, 0x1111);
                re_mdio_write(sc, 0x06, 0x15d2);
                re_mdio_write(sc, 0x06, 0x60d6);
                re_mdio_write(sc, 0x06, 0x6666);
                re_mdio_write(sc, 0x06, 0x0207);
                re_mdio_write(sc, 0x06, 0xf9d2);
                re_mdio_write(sc, 0x06, 0xa0d6);
                re_mdio_write(sc, 0x06, 0xaaaa);
                re_mdio_write(sc, 0x06, 0x0207);
                re_mdio_write(sc, 0x06, 0xf902);
                re_mdio_write(sc, 0x06, 0x825c);
                re_mdio_write(sc, 0x06, 0xae44);
                re_mdio_write(sc, 0x06, 0xa566);
                re_mdio_write(sc, 0x06, 0x6602);
                re_mdio_write(sc, 0x06, 0xae38);
                re_mdio_write(sc, 0x06, 0xa5aa);
                re_mdio_write(sc, 0x06, 0xaa02);
                re_mdio_write(sc, 0x06, 0xae32);
                re_mdio_write(sc, 0x06, 0xeee0);
                re_mdio_write(sc, 0x06, 0xea04);
                re_mdio_write(sc, 0x06, 0xeee0);
                re_mdio_write(sc, 0x06, 0xeb06);
                re_mdio_write(sc, 0x06, 0xe2e0);
                re_mdio_write(sc, 0x06, 0x7ce3);
                re_mdio_write(sc, 0x06, 0xe07d);
                re_mdio_write(sc, 0x06, 0xe0e0);
                re_mdio_write(sc, 0x06, 0x38e1);
                re_mdio_write(sc, 0x06, 0xe039);
                re_mdio_write(sc, 0x06, 0xad2e);
                re_mdio_write(sc, 0x06, 0x21ad);
                re_mdio_write(sc, 0x06, 0x3f13);
                re_mdio_write(sc, 0x06, 0xe0e4);
                re_mdio_write(sc, 0x06, 0x14e1);
                re_mdio_write(sc, 0x06, 0xe415);
                re_mdio_write(sc, 0x06, 0x6880);
                re_mdio_write(sc, 0x06, 0xe4e4);
                re_mdio_write(sc, 0x06, 0x14e5);
                re_mdio_write(sc, 0x06, 0xe415);
                re_mdio_write(sc, 0x06, 0x0282);
                re_mdio_write(sc, 0x06, 0x5cae);
                re_mdio_write(sc, 0x06, 0x0bac);
                re_mdio_write(sc, 0x06, 0x3e02);
                re_mdio_write(sc, 0x06, 0xae06);
                re_mdio_write(sc, 0x06, 0x0282);
                re_mdio_write(sc, 0x06, 0x8602);
                re_mdio_write(sc, 0x06, 0x82b0);
                re_mdio_write(sc, 0x06, 0xfefd);
                re_mdio_write(sc, 0x06, 0xfc04);
                re_mdio_write(sc, 0x06, 0xf8e1);
                re_mdio_write(sc, 0x06, 0x8b2e);
                re_mdio_write(sc, 0x06, 0xe08b);
                re_mdio_write(sc, 0x06, 0x81ad);
                re_mdio_write(sc, 0x06, 0x2605);
                re_mdio_write(sc, 0x06, 0x0221);
                re_mdio_write(sc, 0x06, 0xf3f7);
                re_mdio_write(sc, 0x06, 0x28e0);
                re_mdio_write(sc, 0x06, 0x8b81);
                re_mdio_write(sc, 0x06, 0xad21);
                re_mdio_write(sc, 0x06, 0x0502);
                re_mdio_write(sc, 0x06, 0x22f8);
                re_mdio_write(sc, 0x06, 0xf729);
                re_mdio_write(sc, 0x06, 0xe08b);
                re_mdio_write(sc, 0x06, 0x87ad);
                re_mdio_write(sc, 0x06, 0x2405);
                re_mdio_write(sc, 0x06, 0x0282);
                re_mdio_write(sc, 0x06, 0xebf7);
                re_mdio_write(sc, 0x06, 0x2ae5);
                re_mdio_write(sc, 0x06, 0x8b2e);
                re_mdio_write(sc, 0x06, 0xfc04);
                re_mdio_write(sc, 0x06, 0xf8e0);
                re_mdio_write(sc, 0x06, 0x8b81);
                re_mdio_write(sc, 0x06, 0xad26);
                re_mdio_write(sc, 0x06, 0x0302);
                re_mdio_write(sc, 0x06, 0x2134);
                re_mdio_write(sc, 0x06, 0xe08b);
                re_mdio_write(sc, 0x06, 0x81ad);
                re_mdio_write(sc, 0x06, 0x2109);
                re_mdio_write(sc, 0x06, 0xe08b);
                re_mdio_write(sc, 0x06, 0x2eac);
                re_mdio_write(sc, 0x06, 0x2003);
                re_mdio_write(sc, 0x06, 0x0283);
                re_mdio_write(sc, 0x06, 0x52e0);
                re_mdio_write(sc, 0x06, 0x8b87);
                re_mdio_write(sc, 0x06, 0xad24);
                re_mdio_write(sc, 0x06, 0x09e0);
                re_mdio_write(sc, 0x06, 0x8b2e);
                re_mdio_write(sc, 0x06, 0xac21);
                re_mdio_write(sc, 0x06, 0x0302);
                re_mdio_write(sc, 0x06, 0x8337);
                re_mdio_write(sc, 0x06, 0xfc04);
                re_mdio_write(sc, 0x06, 0xf8e1);
                re_mdio_write(sc, 0x06, 0x8b2e);
                re_mdio_write(sc, 0x06, 0xe08b);
                re_mdio_write(sc, 0x06, 0x81ad);
                re_mdio_write(sc, 0x06, 0x2608);
                re_mdio_write(sc, 0x06, 0xe085);
                re_mdio_write(sc, 0x06, 0xd2ad);
                re_mdio_write(sc, 0x06, 0x2502);
                re_mdio_write(sc, 0x06, 0xf628);
                re_mdio_write(sc, 0x06, 0xe08b);
                re_mdio_write(sc, 0x06, 0x81ad);
                re_mdio_write(sc, 0x06, 0x210a);
                re_mdio_write(sc, 0x06, 0xe086);
                re_mdio_write(sc, 0x06, 0x0af6);
                re_mdio_write(sc, 0x06, 0x27a0);
                re_mdio_write(sc, 0x06, 0x0502);
                re_mdio_write(sc, 0x06, 0xf629);
                re_mdio_write(sc, 0x06, 0xe08b);
                re_mdio_write(sc, 0x06, 0x87ad);
                re_mdio_write(sc, 0x06, 0x2408);
                re_mdio_write(sc, 0x06, 0xe08a);
                re_mdio_write(sc, 0x06, 0xedad);
                re_mdio_write(sc, 0x06, 0x2002);
                re_mdio_write(sc, 0x06, 0xf62a);
                re_mdio_write(sc, 0x06, 0xe58b);
                re_mdio_write(sc, 0x06, 0x2ea1);
                re_mdio_write(sc, 0x06, 0x0003);
                re_mdio_write(sc, 0x06, 0x0221);
                re_mdio_write(sc, 0x06, 0x11fc);
                re_mdio_write(sc, 0x06, 0x04ee);
                re_mdio_write(sc, 0x06, 0x8aed);
                re_mdio_write(sc, 0x06, 0x00ee);
                re_mdio_write(sc, 0x06, 0x8aec);
                re_mdio_write(sc, 0x06, 0x0004);
                re_mdio_write(sc, 0x06, 0xf8e0);
                re_mdio_write(sc, 0x06, 0x8b87);
                re_mdio_write(sc, 0x06, 0xad24);
                re_mdio_write(sc, 0x06, 0x3ae0);
                re_mdio_write(sc, 0x06, 0xe0ea);
                re_mdio_write(sc, 0x06, 0xe1e0);
                re_mdio_write(sc, 0x06, 0xeb58);
                re_mdio_write(sc, 0x06, 0xf8d1);
                re_mdio_write(sc, 0x06, 0x01e4);
                re_mdio_write(sc, 0x06, 0xe0ea);
                re_mdio_write(sc, 0x06, 0xe5e0);
                re_mdio_write(sc, 0x06, 0xebe0);
                re_mdio_write(sc, 0x06, 0xe07c);
                re_mdio_write(sc, 0x06, 0xe1e0);
                re_mdio_write(sc, 0x06, 0x7d5c);
                re_mdio_write(sc, 0x06, 0x00ff);
                re_mdio_write(sc, 0x06, 0x3c00);
                re_mdio_write(sc, 0x06, 0x1eab);
                re_mdio_write(sc, 0x06, 0x1ce0);
                re_mdio_write(sc, 0x06, 0xe04c);
                re_mdio_write(sc, 0x06, 0xe1e0);
                re_mdio_write(sc, 0x06, 0x4d58);
                re_mdio_write(sc, 0x06, 0xc1e4);
                re_mdio_write(sc, 0x06, 0xe04c);
                re_mdio_write(sc, 0x06, 0xe5e0);
                re_mdio_write(sc, 0x06, 0x4de0);
                re_mdio_write(sc, 0x06, 0xe0ee);
                re_mdio_write(sc, 0x06, 0xe1e0);
                re_mdio_write(sc, 0x06, 0xef69);
                re_mdio_write(sc, 0x06, 0x3ce4);
                re_mdio_write(sc, 0x06, 0xe0ee);
                re_mdio_write(sc, 0x06, 0xe5e0);
                re_mdio_write(sc, 0x06, 0xeffc);
                re_mdio_write(sc, 0x06, 0x04f8);
                re_mdio_write(sc, 0x06, 0xe08b);
                re_mdio_write(sc, 0x06, 0x87ad);
                re_mdio_write(sc, 0x06, 0x2412);
                re_mdio_write(sc, 0x06, 0xe0e0);
                re_mdio_write(sc, 0x06, 0xeee1);
                re_mdio_write(sc, 0x06, 0xe0ef);
                re_mdio_write(sc, 0x06, 0x59c3);
                re_mdio_write(sc, 0x06, 0xe4e0);
                re_mdio_write(sc, 0x06, 0xeee5);
                re_mdio_write(sc, 0x06, 0xe0ef);
                re_mdio_write(sc, 0x06, 0xee8a);
                re_mdio_write(sc, 0x06, 0xed01);
                re_mdio_write(sc, 0x06, 0xfc04);
                re_mdio_write(sc, 0x06, 0xf8e0);
                re_mdio_write(sc, 0x06, 0x8b81);
                re_mdio_write(sc, 0x06, 0xac25);
                re_mdio_write(sc, 0x06, 0x0502);
                re_mdio_write(sc, 0x06, 0x8363);
                re_mdio_write(sc, 0x06, 0xae03);
                re_mdio_write(sc, 0x06, 0x0225);
                re_mdio_write(sc, 0x06, 0x16fc);
                re_mdio_write(sc, 0x06, 0x04f8);
                re_mdio_write(sc, 0x06, 0xf9fa);
                re_mdio_write(sc, 0x06, 0xef69);
                re_mdio_write(sc, 0x06, 0xfae0);
                re_mdio_write(sc, 0x06, 0x860a);
                re_mdio_write(sc, 0x06, 0xa000);
                re_mdio_write(sc, 0x06, 0x19e0);
                re_mdio_write(sc, 0x06, 0x860b);
                re_mdio_write(sc, 0x06, 0xe18b);
                re_mdio_write(sc, 0x06, 0x331b);
                re_mdio_write(sc, 0x06, 0x109e);
                re_mdio_write(sc, 0x06, 0x04aa);
                re_mdio_write(sc, 0x06, 0x02ae);
                re_mdio_write(sc, 0x06, 0x06ee);
                re_mdio_write(sc, 0x06, 0x860a);
                re_mdio_write(sc, 0x06, 0x01ae);
                re_mdio_write(sc, 0x06, 0xe602);
                re_mdio_write(sc, 0x06, 0x241e);
                re_mdio_write(sc, 0x06, 0xae14);
                re_mdio_write(sc, 0x06, 0xa001);
                re_mdio_write(sc, 0x06, 0x1402);
                re_mdio_write(sc, 0x06, 0x2426);
                re_mdio_write(sc, 0x06, 0xbf26);
                re_mdio_write(sc, 0x06, 0x6d02);
                re_mdio_write(sc, 0x06, 0x2eb0);
                re_mdio_write(sc, 0x06, 0xee86);
                re_mdio_write(sc, 0x06, 0x0b00);
                re_mdio_write(sc, 0x06, 0xee86);
                re_mdio_write(sc, 0x06, 0x0a02);
                re_mdio_write(sc, 0x06, 0xaf84);
                re_mdio_write(sc, 0x06, 0x3ca0);
                re_mdio_write(sc, 0x06, 0x0252);
                re_mdio_write(sc, 0x06, 0xee86);
                re_mdio_write(sc, 0x06, 0x0400);
                re_mdio_write(sc, 0x06, 0xee86);
                re_mdio_write(sc, 0x06, 0x0500);
                re_mdio_write(sc, 0x06, 0xe086);
                re_mdio_write(sc, 0x06, 0x0be1);
                re_mdio_write(sc, 0x06, 0x8b32);
                re_mdio_write(sc, 0x06, 0x1b10);
                re_mdio_write(sc, 0x06, 0x9e04);
                re_mdio_write(sc, 0x06, 0xaa02);
                re_mdio_write(sc, 0x06, 0xaecb);
                re_mdio_write(sc, 0x06, 0xee86);
                re_mdio_write(sc, 0x06, 0x0b00);
                re_mdio_write(sc, 0x06, 0x0224);
                re_mdio_write(sc, 0x06, 0x3ae2);
                re_mdio_write(sc, 0x06, 0x8604);
                re_mdio_write(sc, 0x06, 0xe386);
                re_mdio_write(sc, 0x06, 0x05ef);
                re_mdio_write(sc, 0x06, 0x65e2);
                re_mdio_write(sc, 0x06, 0x8606);
                re_mdio_write(sc, 0x06, 0xe386);
                re_mdio_write(sc, 0x06, 0x071b);
                re_mdio_write(sc, 0x06, 0x56aa);
                re_mdio_write(sc, 0x06, 0x0eef);
                re_mdio_write(sc, 0x06, 0x56e6);
                re_mdio_write(sc, 0x06, 0x8606);
                re_mdio_write(sc, 0x06, 0xe786);
                re_mdio_write(sc, 0x06, 0x07e2);
                re_mdio_write(sc, 0x06, 0x8609);
                re_mdio_write(sc, 0x06, 0xe686);
                re_mdio_write(sc, 0x06, 0x08e0);
                re_mdio_write(sc, 0x06, 0x8609);
                re_mdio_write(sc, 0x06, 0xa000);
                re_mdio_write(sc, 0x06, 0x07ee);
                re_mdio_write(sc, 0x06, 0x860a);
                re_mdio_write(sc, 0x06, 0x03af);
                re_mdio_write(sc, 0x06, 0x8369);
                re_mdio_write(sc, 0x06, 0x0224);
                re_mdio_write(sc, 0x06, 0x8e02);
                re_mdio_write(sc, 0x06, 0x2426);
                re_mdio_write(sc, 0x06, 0xae48);
                re_mdio_write(sc, 0x06, 0xa003);
                re_mdio_write(sc, 0x06, 0x21e0);
                re_mdio_write(sc, 0x06, 0x8608);
                re_mdio_write(sc, 0x06, 0xe186);
                re_mdio_write(sc, 0x06, 0x091b);
                re_mdio_write(sc, 0x06, 0x019e);
                re_mdio_write(sc, 0x06, 0x0caa);
                re_mdio_write(sc, 0x06, 0x0502);
                re_mdio_write(sc, 0x06, 0x249d);
                re_mdio_write(sc, 0x06, 0xaee7);
                re_mdio_write(sc, 0x06, 0x0224);
                re_mdio_write(sc, 0x06, 0x8eae);
                re_mdio_write(sc, 0x06, 0xe2ee);
                re_mdio_write(sc, 0x06, 0x860a);
                re_mdio_write(sc, 0x06, 0x04ee);
                re_mdio_write(sc, 0x06, 0x860b);
                re_mdio_write(sc, 0x06, 0x00af);
                re_mdio_write(sc, 0x06, 0x8369);
                re_mdio_write(sc, 0x06, 0xa004);
                re_mdio_write(sc, 0x06, 0x15e0);
                re_mdio_write(sc, 0x06, 0x860b);
                re_mdio_write(sc, 0x06, 0xe18b);
                re_mdio_write(sc, 0x06, 0x341b);
                re_mdio_write(sc, 0x06, 0x109e);
                re_mdio_write(sc, 0x06, 0x05aa);
                re_mdio_write(sc, 0x06, 0x03af);
                re_mdio_write(sc, 0x06, 0x8383);
                re_mdio_write(sc, 0x06, 0xee86);
                re_mdio_write(sc, 0x06, 0x0a05);
                re_mdio_write(sc, 0x06, 0xae0c);
                re_mdio_write(sc, 0x06, 0xa005);
                re_mdio_write(sc, 0x06, 0x02ae);
                re_mdio_write(sc, 0x06, 0x0702);
                re_mdio_write(sc, 0x06, 0x2309);
                re_mdio_write(sc, 0x06, 0xee86);
                re_mdio_write(sc, 0x06, 0x0a00);
                re_mdio_write(sc, 0x06, 0xfeef);
                re_mdio_write(sc, 0x06, 0x96fe);
                re_mdio_write(sc, 0x06, 0xfdfc);
                re_mdio_write(sc, 0x06, 0x04f8);
                re_mdio_write(sc, 0x06, 0xf9fa);
                re_mdio_write(sc, 0x06, 0xef69);
                re_mdio_write(sc, 0x06, 0xfbe0);
                re_mdio_write(sc, 0x06, 0x8b85);
                re_mdio_write(sc, 0x06, 0xad25);
                re_mdio_write(sc, 0x06, 0x22e0);
                re_mdio_write(sc, 0x06, 0xe022);
                re_mdio_write(sc, 0x06, 0xe1e0);
                re_mdio_write(sc, 0x06, 0x23e2);
                re_mdio_write(sc, 0x06, 0xe036);
                re_mdio_write(sc, 0x06, 0xe3e0);
                re_mdio_write(sc, 0x06, 0x375a);
                re_mdio_write(sc, 0x06, 0xc40d);
                re_mdio_write(sc, 0x06, 0x0158);
                re_mdio_write(sc, 0x06, 0x021e);
                re_mdio_write(sc, 0x06, 0x20e3);
                re_mdio_write(sc, 0x06, 0x8ae7);
                re_mdio_write(sc, 0x06, 0xac31);
                re_mdio_write(sc, 0x06, 0x60ac);
                re_mdio_write(sc, 0x06, 0x3a08);
                re_mdio_write(sc, 0x06, 0xac3e);
                re_mdio_write(sc, 0x06, 0x26ae);
                re_mdio_write(sc, 0x06, 0x67af);
                re_mdio_write(sc, 0x06, 0x84db);
                re_mdio_write(sc, 0x06, 0xad37);
                re_mdio_write(sc, 0x06, 0x61e0);
                re_mdio_write(sc, 0x06, 0x8ae8);
                re_mdio_write(sc, 0x06, 0x10e4);
                re_mdio_write(sc, 0x06, 0x8ae8);
                re_mdio_write(sc, 0x06, 0xe18a);
                re_mdio_write(sc, 0x06, 0xe91b);
                re_mdio_write(sc, 0x06, 0x109e);
                re_mdio_write(sc, 0x06, 0x02ae);
                re_mdio_write(sc, 0x06, 0x51d1);
                re_mdio_write(sc, 0x06, 0x00bf);
                re_mdio_write(sc, 0x06, 0x863b);
                re_mdio_write(sc, 0x06, 0x022f);
                re_mdio_write(sc, 0x06, 0x50ee);
                re_mdio_write(sc, 0x06, 0x8ae8);
                re_mdio_write(sc, 0x06, 0x00ae);
                re_mdio_write(sc, 0x06, 0x43ad);
                re_mdio_write(sc, 0x06, 0x3627);
                re_mdio_write(sc, 0x06, 0xe08a);
                re_mdio_write(sc, 0x06, 0xeee1);
                re_mdio_write(sc, 0x06, 0x8aef);
                re_mdio_write(sc, 0x06, 0xef74);
                re_mdio_write(sc, 0x06, 0xe08a);
                re_mdio_write(sc, 0x06, 0xeae1);
                re_mdio_write(sc, 0x06, 0x8aeb);
                re_mdio_write(sc, 0x06, 0x1b74);
                re_mdio_write(sc, 0x06, 0x9e2e);
                re_mdio_write(sc, 0x06, 0x14e4);
                re_mdio_write(sc, 0x06, 0x8aea);
                re_mdio_write(sc, 0x06, 0xe58a);
                re_mdio_write(sc, 0x06, 0xebef);
                re_mdio_write(sc, 0x06, 0x74e0);
                re_mdio_write(sc, 0x06, 0x8aee);
                re_mdio_write(sc, 0x06, 0xe18a);
                re_mdio_write(sc, 0x06, 0xef1b);
                re_mdio_write(sc, 0x06, 0x479e);
                re_mdio_write(sc, 0x06, 0x0fae);
                re_mdio_write(sc, 0x06, 0x19ee);
                re_mdio_write(sc, 0x06, 0x8aea);
                re_mdio_write(sc, 0x06, 0x00ee);
                re_mdio_write(sc, 0x06, 0x8aeb);
                re_mdio_write(sc, 0x06, 0x00ae);
                re_mdio_write(sc, 0x06, 0x0fac);
                re_mdio_write(sc, 0x06, 0x390c);
                re_mdio_write(sc, 0x06, 0xd101);
                re_mdio_write(sc, 0x06, 0xbf86);
                re_mdio_write(sc, 0x06, 0x3b02);
                re_mdio_write(sc, 0x06, 0x2f50);
                re_mdio_write(sc, 0x06, 0xee8a);
                re_mdio_write(sc, 0x06, 0xe800);
                re_mdio_write(sc, 0x06, 0xe68a);
                re_mdio_write(sc, 0x06, 0xe7ff);
                re_mdio_write(sc, 0x06, 0xef96);
                re_mdio_write(sc, 0x06, 0xfefd);
                re_mdio_write(sc, 0x06, 0xfc04);
                re_mdio_write(sc, 0x06, 0xf8f9);
                re_mdio_write(sc, 0x06, 0xfaef);
                re_mdio_write(sc, 0x06, 0x69e0);
                re_mdio_write(sc, 0x06, 0xe022);
                re_mdio_write(sc, 0x06, 0xe1e0);
                re_mdio_write(sc, 0x06, 0x2358);
                re_mdio_write(sc, 0x06, 0xc4e1);
                re_mdio_write(sc, 0x06, 0x8b6e);
                re_mdio_write(sc, 0x06, 0x1f10);
                re_mdio_write(sc, 0x06, 0x9e24);
                re_mdio_write(sc, 0x06, 0xe48b);
                re_mdio_write(sc, 0x06, 0x6ead);
                re_mdio_write(sc, 0x06, 0x2218);
                re_mdio_write(sc, 0x06, 0xac27);
                re_mdio_write(sc, 0x06, 0x0dac);
                re_mdio_write(sc, 0x06, 0x2605);
                re_mdio_write(sc, 0x06, 0x0203);
                re_mdio_write(sc, 0x06, 0x8fae);
                re_mdio_write(sc, 0x06, 0x1302);
                re_mdio_write(sc, 0x06, 0x03c8);
                re_mdio_write(sc, 0x06, 0xae0e);
                re_mdio_write(sc, 0x06, 0x0203);
                re_mdio_write(sc, 0x06, 0xe102);
                re_mdio_write(sc, 0x06, 0x8520);
                re_mdio_write(sc, 0x06, 0xae06);
                re_mdio_write(sc, 0x06, 0x0203);
                re_mdio_write(sc, 0x06, 0x8f02);
                re_mdio_write(sc, 0x06, 0x8566);
                re_mdio_write(sc, 0x06, 0xef96);
                re_mdio_write(sc, 0x06, 0xfefd);
                re_mdio_write(sc, 0x06, 0xfc04);
                re_mdio_write(sc, 0x06, 0xf8fa);
                re_mdio_write(sc, 0x06, 0xef69);
                re_mdio_write(sc, 0x06, 0xe08b);
                re_mdio_write(sc, 0x06, 0x82ad);
                re_mdio_write(sc, 0x06, 0x2737);
                re_mdio_write(sc, 0x06, 0xbf86);
                re_mdio_write(sc, 0x06, 0x4402);
                re_mdio_write(sc, 0x06, 0x2f23);
                re_mdio_write(sc, 0x06, 0xac28);
                re_mdio_write(sc, 0x06, 0x2ed1);
                re_mdio_write(sc, 0x06, 0x01bf);
                re_mdio_write(sc, 0x06, 0x8647);
                re_mdio_write(sc, 0x06, 0x022f);
                re_mdio_write(sc, 0x06, 0x50bf);
                re_mdio_write(sc, 0x06, 0x8641);
                re_mdio_write(sc, 0x06, 0x022f);
                re_mdio_write(sc, 0x06, 0x23e5);
                re_mdio_write(sc, 0x06, 0x8af0);
                re_mdio_write(sc, 0x06, 0xe0e0);
                re_mdio_write(sc, 0x06, 0x22e1);
                re_mdio_write(sc, 0x06, 0xe023);
                re_mdio_write(sc, 0x06, 0xac2e);
                re_mdio_write(sc, 0x06, 0x04d1);
                re_mdio_write(sc, 0x06, 0x01ae);
                re_mdio_write(sc, 0x06, 0x02d1);
                re_mdio_write(sc, 0x06, 0x00bf);
                re_mdio_write(sc, 0x06, 0x8641);
                re_mdio_write(sc, 0x06, 0x022f);
                re_mdio_write(sc, 0x06, 0x50d1);
                re_mdio_write(sc, 0x06, 0x01bf);
                re_mdio_write(sc, 0x06, 0x8644);
                re_mdio_write(sc, 0x06, 0x022f);
                re_mdio_write(sc, 0x06, 0x50ef);
                re_mdio_write(sc, 0x06, 0x96fe);
                re_mdio_write(sc, 0x06, 0xfc04);
                re_mdio_write(sc, 0x06, 0xf8fa);
                re_mdio_write(sc, 0x06, 0xef69);
                re_mdio_write(sc, 0x06, 0xbf86);
                re_mdio_write(sc, 0x06, 0x4702);
                re_mdio_write(sc, 0x06, 0x2f23);
                re_mdio_write(sc, 0x06, 0xad28);
                re_mdio_write(sc, 0x06, 0x19d1);
                re_mdio_write(sc, 0x06, 0x00bf);
                re_mdio_write(sc, 0x06, 0x8644);
                re_mdio_write(sc, 0x06, 0x022f);
                re_mdio_write(sc, 0x06, 0x50e1);
                re_mdio_write(sc, 0x06, 0x8af0);
                re_mdio_write(sc, 0x06, 0xbf86);
                re_mdio_write(sc, 0x06, 0x4102);
                re_mdio_write(sc, 0x06, 0x2f50);
                re_mdio_write(sc, 0x06, 0xd100);
                re_mdio_write(sc, 0x06, 0xbf86);
                re_mdio_write(sc, 0x06, 0x4702);
                re_mdio_write(sc, 0x06, 0x2f50);
                re_mdio_write(sc, 0x06, 0xef96);
                re_mdio_write(sc, 0x06, 0xfefc);
                re_mdio_write(sc, 0x06, 0x04f8);
                re_mdio_write(sc, 0x06, 0xe0e2);
                re_mdio_write(sc, 0x06, 0xfee1);
                re_mdio_write(sc, 0x06, 0xe2ff);
                re_mdio_write(sc, 0x06, 0xad2e);
                re_mdio_write(sc, 0x06, 0x63e0);
                re_mdio_write(sc, 0x06, 0xe038);
                re_mdio_write(sc, 0x06, 0xe1e0);
                re_mdio_write(sc, 0x06, 0x39ad);
                re_mdio_write(sc, 0x06, 0x2f10);
                re_mdio_write(sc, 0x06, 0xe0e0);
                re_mdio_write(sc, 0x06, 0x34e1);
                re_mdio_write(sc, 0x06, 0xe035);
                re_mdio_write(sc, 0x06, 0xf726);
                re_mdio_write(sc, 0x06, 0xe4e0);
                re_mdio_write(sc, 0x06, 0x34e5);
                re_mdio_write(sc, 0x06, 0xe035);
                re_mdio_write(sc, 0x06, 0xae0e);
                re_mdio_write(sc, 0x06, 0xe0e2);
                re_mdio_write(sc, 0x06, 0xd6e1);
                re_mdio_write(sc, 0x06, 0xe2d7);
                re_mdio_write(sc, 0x06, 0xf728);
                re_mdio_write(sc, 0x06, 0xe4e2);
                re_mdio_write(sc, 0x06, 0xd6e5);
                re_mdio_write(sc, 0x06, 0xe2d7);
                re_mdio_write(sc, 0x06, 0xe0e2);
                re_mdio_write(sc, 0x06, 0x34e1);
                re_mdio_write(sc, 0x06, 0xe235);
                re_mdio_write(sc, 0x06, 0xf72b);
                re_mdio_write(sc, 0x06, 0xe4e2);
                re_mdio_write(sc, 0x06, 0x34e5);
                re_mdio_write(sc, 0x06, 0xe235);
                re_mdio_write(sc, 0x06, 0xd07d);
                re_mdio_write(sc, 0x06, 0xb0fe);
                re_mdio_write(sc, 0x06, 0xe0e2);
                re_mdio_write(sc, 0x06, 0x34e1);
                re_mdio_write(sc, 0x06, 0xe235);
                re_mdio_write(sc, 0x06, 0xf62b);
                re_mdio_write(sc, 0x06, 0xe4e2);
                re_mdio_write(sc, 0x06, 0x34e5);
                re_mdio_write(sc, 0x06, 0xe235);
                re_mdio_write(sc, 0x06, 0xe0e0);
                re_mdio_write(sc, 0x06, 0x34e1);
                re_mdio_write(sc, 0x06, 0xe035);
                re_mdio_write(sc, 0x06, 0xf626);
                re_mdio_write(sc, 0x06, 0xe4e0);
                re_mdio_write(sc, 0x06, 0x34e5);
                re_mdio_write(sc, 0x06, 0xe035);
                re_mdio_write(sc, 0x06, 0xe0e2);
                re_mdio_write(sc, 0x06, 0xd6e1);
                re_mdio_write(sc, 0x06, 0xe2d7);
                re_mdio_write(sc, 0x06, 0xf628);
                re_mdio_write(sc, 0x06, 0xe4e2);
                re_mdio_write(sc, 0x06, 0xd6e5);
                re_mdio_write(sc, 0x06, 0xe2d7);
                re_mdio_write(sc, 0x06, 0xfc04);
                re_mdio_write(sc, 0x06, 0xae20);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0xa725);
                re_mdio_write(sc, 0x06, 0xe50a);
                re_mdio_write(sc, 0x06, 0x1de5);
                re_mdio_write(sc, 0x06, 0x0a2c);
                re_mdio_write(sc, 0x06, 0xe50a);
                re_mdio_write(sc, 0x06, 0x6de5);
                re_mdio_write(sc, 0x06, 0x0a1d);
                re_mdio_write(sc, 0x06, 0xe50a);
                re_mdio_write(sc, 0x06, 0x1ce5);
                re_mdio_write(sc, 0x06, 0x0a2d);
                re_mdio_write(sc, 0x06, 0xa755);
                re_mdio_write(sc, 0x06, 0x00e2);
                re_mdio_write(sc, 0x06, 0x3488);
                re_mdio_write(sc, 0x06, 0xe200);
                re_mdio_write(sc, 0x06, 0xcce2);
                re_mdio_write(sc, 0x06, 0x0055);
                re_mdio_write(sc, 0x06, 0xe020);
                re_mdio_write(sc, 0x06, 0x55e2);
                re_mdio_write(sc, 0x06, 0xd600);
                re_mdio_write(sc, 0x06, 0xe24a);
                PhyRegValue = re_mdio_read(sc, 0x01);
                PhyRegValue |= BIT_0;
                re_mdio_write(sc, 0x01, PhyRegValue);
                PhyRegValue = re_mdio_read(sc, 0x00);
                PhyRegValue |= BIT_0;
                re_mdio_write(sc, 0x00, PhyRegValue);
                re_mdio_write(sc, 0x1f, 0x0000);
                re_mdio_write(sc, 0x1f, 0x0000);
                re_mdio_write(sc, 0x17, 0x2179);
                re_mdio_write(sc, 0x1f, 0x0001);
                re_mdio_write(sc, 0x10, 0xf274);
                re_mdio_write(sc, 0x1f, 0x0007);
                re_mdio_write(sc, 0x1e, 0x0042);
                re_mdio_write(sc, 0x15, 0x0f00);
                re_mdio_write(sc, 0x15, 0x0f00);
                re_mdio_write(sc, 0x16, 0x7408);
                re_mdio_write(sc, 0x15, 0x0e00);
                re_mdio_write(sc, 0x15, 0x0f00);
                re_mdio_write(sc, 0x15, 0x0f01);
                re_mdio_write(sc, 0x16, 0x4000);
                re_mdio_write(sc, 0x15, 0x0e01);
                re_mdio_write(sc, 0x15, 0x0f01);
                re_mdio_write(sc, 0x15, 0x0f02);
                re_mdio_write(sc, 0x16, 0x9400);
                re_mdio_write(sc, 0x15, 0x0e02);
                re_mdio_write(sc, 0x15, 0x0f02);
                re_mdio_write(sc, 0x15, 0x0f03);
                re_mdio_write(sc, 0x16, 0x7408);
                re_mdio_write(sc, 0x15, 0x0e03);
                re_mdio_write(sc, 0x15, 0x0f03);
                re_mdio_write(sc, 0x15, 0x0f04);
                re_mdio_write(sc, 0x16, 0x4008);
                re_mdio_write(sc, 0x15, 0x0e04);
                re_mdio_write(sc, 0x15, 0x0f04);
                re_mdio_write(sc, 0x15, 0x0f05);
                re_mdio_write(sc, 0x16, 0x9400);
                re_mdio_write(sc, 0x15, 0x0e05);
                re_mdio_write(sc, 0x15, 0x0f05);
                re_mdio_write(sc, 0x15, 0x0f06);
                re_mdio_write(sc, 0x16, 0x0803);
                re_mdio_write(sc, 0x15, 0x0e06);
                re_mdio_write(sc, 0x15, 0x0f06);
                re_mdio_write(sc, 0x15, 0x0d00);
                re_mdio_write(sc, 0x15, 0x0100);
                re_mdio_write(sc, 0x1f, 0x0001);
                re_mdio_write(sc, 0x10, 0xf074);
                re_mdio_write(sc, 0x1f, 0x0000);
                re_mdio_write(sc, 0x17, 0x2149);
                re_mdio_write(sc, 0x1f, 0x0005);
                for (i=0; i<200; i++) {
                        DELAY(100);
                        PhyRegValue = re_mdio_read(sc, 0x00);
                        if (PhyRegValue&0x0080)
                                break;
                }
                re_mdio_write(sc, 0x1f, 0x0007);
                re_mdio_write(sc, 0x1e, 0x0023);
                PhyRegValue = re_mdio_read(sc, 0x17);
                PhyRegValue &= ~(BIT_0);
                if (sc->RequiredSecLanDonglePatch)
                        PhyRegValue &= ~(BIT_2);
                re_mdio_write(sc, 0x17, PhyRegValue);
                re_mdio_write(sc, 0x1f, 0x0000);
                re_mdio_write(sc, 0x1f, 0x0007);
                re_mdio_write(sc, 0x1e, 0x0023);
                PhyRegValue = re_mdio_read(sc, 0x17);
                PhyRegValue |= BIT_14;
                re_mdio_write(sc, 0x17, PhyRegValue);
                re_mdio_write(sc, 0x1e, 0x0020);
                PhyRegValue = re_mdio_read(sc, 0x1b);
                PhyRegValue |= BIT_7;
                re_mdio_write(sc, 0x1b, PhyRegValue);
                re_mdio_write(sc, 0x1e, 0x0041);
                re_mdio_write(sc, 0x15, 0x0e02);
                re_mdio_write(sc, 0x1e, 0x0028);
                PhyRegValue = re_mdio_read(sc, 0x19);
                PhyRegValue |= BIT_15;
                re_mdio_write(sc, 0x19, PhyRegValue);
                re_mdio_write(sc, 0x1f, 0x0000);
        } else {
                re_mdio_write(sc, 0x1f, 0x0000);
                re_mdio_write(sc, 0x00, 0x1800);
                re_mdio_write(sc, 0x1f, 0x0007);
                re_mdio_write(sc, 0x1e, 0x0023);
                re_mdio_write(sc, 0x17, 0x0117);
                re_mdio_write(sc, 0x1f, 0x0007);
                re_mdio_write(sc, 0x1E, 0x002C);
                re_mdio_write(sc, 0x1B, 0x5000);
                re_mdio_write(sc, 0x1f, 0x0000);
                re_mdio_write(sc, 0x16, 0x4104);
                for (i = 0; i < 200; i++) {
                        DELAY(100);
                        PhyRegValue = re_mdio_read(sc, 0x1E);
                        PhyRegValue &= 0x03FF;
                        if (PhyRegValue==0x000C)
                                break;
                }
                re_mdio_write(sc, 0x1f, 0x0005);
                for (i = 0; i < 200; i++) {
                        DELAY(100);
                        PhyRegValue = re_mdio_read(sc, 0x07);
                        if ((PhyRegValue & BIT_5) == 0)
                                break;
                }
                PhyRegValue = re_mdio_read(sc, 0x07);
                if (PhyRegValue & BIT_5) {
                        re_mdio_write(sc, 0x1f, 0x0007);
                        re_mdio_write(sc, 0x1e, 0x00a1);
                        re_mdio_write(sc, 0x17, 0x1000);
                        re_mdio_write(sc, 0x17, 0x0000);
                        re_mdio_write(sc, 0x17, 0x2000);
                        re_mdio_write(sc, 0x1e, 0x002f);
                        re_mdio_write(sc, 0x18, 0x9bfb);
                        re_mdio_write(sc, 0x1f, 0x0005);
                        re_mdio_write(sc, 0x07, 0x0000);
                        re_mdio_write(sc, 0x1f, 0x0000);
                }
                re_mdio_write(sc, 0x1f, 0x0005);
                re_mdio_write(sc, 0x05, 0xfff6);
                re_mdio_write(sc, 0x06, 0x0080);
                PhyRegValue = re_mdio_read(sc, 0x00);
                PhyRegValue &= ~(BIT_7);
                re_mdio_write(sc, 0x00, PhyRegValue);
                re_mdio_write(sc, 0x1f, 0x0002);
                PhyRegValue = re_mdio_read(sc, 0x08);
                PhyRegValue &= ~(BIT_7);
                re_mdio_write(sc, 0x08, PhyRegValue);
                re_mdio_write(sc, 0x1f, 0x0000);

                re_mdio_write(sc, 0x1f, 0x0007);
                re_mdio_write(sc, 0x1e, 0x0023);
                re_mdio_write(sc, 0x16, 0x0306);
                re_mdio_write(sc, 0x16, 0x0307);
                re_mdio_write(sc, 0x15, 0x000e);
                re_mdio_write(sc, 0x19, 0x000a);
                re_mdio_write(sc, 0x15, 0x0010);
                re_mdio_write(sc, 0x19, 0x0008);
                re_mdio_write(sc, 0x15, 0x0018);
                re_mdio_write(sc, 0x19, 0x4801);
                re_mdio_write(sc, 0x15, 0x0019);
                re_mdio_write(sc, 0x19, 0x6801);
                re_mdio_write(sc, 0x15, 0x001a);
                re_mdio_write(sc, 0x19, 0x66a1);
                re_mdio_write(sc, 0x15, 0x001f);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0020);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0021);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0022);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0023);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0024);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0025);
                re_mdio_write(sc, 0x19, 0x64a1);
                re_mdio_write(sc, 0x15, 0x0026);
                re_mdio_write(sc, 0x19, 0x40ea);
                re_mdio_write(sc, 0x15, 0x0027);
                re_mdio_write(sc, 0x19, 0x4503);
                re_mdio_write(sc, 0x15, 0x0028);
                re_mdio_write(sc, 0x19, 0x9f00);
                re_mdio_write(sc, 0x15, 0x0029);
                re_mdio_write(sc, 0x19, 0xa631);
                re_mdio_write(sc, 0x15, 0x002a);
                re_mdio_write(sc, 0x19, 0x9717);
                re_mdio_write(sc, 0x15, 0x002b);
                re_mdio_write(sc, 0x19, 0x302c);
                re_mdio_write(sc, 0x15, 0x002c);
                re_mdio_write(sc, 0x19, 0x4802);
                re_mdio_write(sc, 0x15, 0x002d);
                re_mdio_write(sc, 0x19, 0x58da);
                re_mdio_write(sc, 0x15, 0x002e);
                re_mdio_write(sc, 0x19, 0x400d);
                re_mdio_write(sc, 0x15, 0x002f);
                re_mdio_write(sc, 0x19, 0x4488);
                re_mdio_write(sc, 0x15, 0x0030);
                re_mdio_write(sc, 0x19, 0x9e00);
                re_mdio_write(sc, 0x15, 0x0031);
                re_mdio_write(sc, 0x19, 0x63c8);
                re_mdio_write(sc, 0x15, 0x0032);
                re_mdio_write(sc, 0x19, 0x6481);
                re_mdio_write(sc, 0x15, 0x0033);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0034);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0035);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0036);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0037);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0038);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0039);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x003a);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x003b);
                re_mdio_write(sc, 0x19, 0x63e8);
                re_mdio_write(sc, 0x15, 0x003c);
                re_mdio_write(sc, 0x19, 0x7d00);
                re_mdio_write(sc, 0x15, 0x003d);
                re_mdio_write(sc, 0x19, 0x59d4);
                re_mdio_write(sc, 0x15, 0x003e);
                re_mdio_write(sc, 0x19, 0x63f8);
                re_mdio_write(sc, 0x15, 0x0040);
                re_mdio_write(sc, 0x19, 0x64a1);
                re_mdio_write(sc, 0x15, 0x0041);
                re_mdio_write(sc, 0x19, 0x30de);
                re_mdio_write(sc, 0x15, 0x0044);
                re_mdio_write(sc, 0x19, 0x480f);
                re_mdio_write(sc, 0x15, 0x0045);
                re_mdio_write(sc, 0x19, 0x6800);
                re_mdio_write(sc, 0x15, 0x0046);
                re_mdio_write(sc, 0x19, 0x6680);
                re_mdio_write(sc, 0x15, 0x0047);
                re_mdio_write(sc, 0x19, 0x7c10);
                re_mdio_write(sc, 0x15, 0x0048);
                re_mdio_write(sc, 0x19, 0x63c8);
                re_mdio_write(sc, 0x15, 0x0049);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x004a);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x004b);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x004c);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x004d);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x004e);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x004f);
                re_mdio_write(sc, 0x19, 0x40ea);
                re_mdio_write(sc, 0x15, 0x0050);
                re_mdio_write(sc, 0x19, 0x4503);
                re_mdio_write(sc, 0x15, 0x0051);
                re_mdio_write(sc, 0x19, 0x58ca);
                re_mdio_write(sc, 0x15, 0x0052);
                re_mdio_write(sc, 0x19, 0x63c8);
                re_mdio_write(sc, 0x15, 0x0053);
                re_mdio_write(sc, 0x19, 0x63d8);
                re_mdio_write(sc, 0x15, 0x0054);
                re_mdio_write(sc, 0x19, 0x66a0);
                re_mdio_write(sc, 0x15, 0x0055);
                re_mdio_write(sc, 0x19, 0x9f00);
                re_mdio_write(sc, 0x15, 0x0056);
                re_mdio_write(sc, 0x19, 0x3000);
                re_mdio_write(sc, 0x15, 0x00a1);
                re_mdio_write(sc, 0x19, 0x3044);
                re_mdio_write(sc, 0x15, 0x00ab);
                re_mdio_write(sc, 0x19, 0x5820);
                re_mdio_write(sc, 0x15, 0x00ac);
                re_mdio_write(sc, 0x19, 0x5e04);
                re_mdio_write(sc, 0x15, 0x00ad);
                re_mdio_write(sc, 0x19, 0xb60c);
                re_mdio_write(sc, 0x15, 0x00af);
                re_mdio_write(sc, 0x19, 0x000a);
                re_mdio_write(sc, 0x15, 0x00b2);
                re_mdio_write(sc, 0x19, 0x30b9);
                re_mdio_write(sc, 0x15, 0x00b9);
                re_mdio_write(sc, 0x19, 0x4408);
                re_mdio_write(sc, 0x15, 0x00ba);
                re_mdio_write(sc, 0x19, 0x480b);
                re_mdio_write(sc, 0x15, 0x00bb);
                re_mdio_write(sc, 0x19, 0x5e00);
                re_mdio_write(sc, 0x15, 0x00bc);
                re_mdio_write(sc, 0x19, 0x405f);
                re_mdio_write(sc, 0x15, 0x00bd);
                re_mdio_write(sc, 0x19, 0x4448);
                re_mdio_write(sc, 0x15, 0x00be);
                re_mdio_write(sc, 0x19, 0x4020);
                re_mdio_write(sc, 0x15, 0x00bf);
                re_mdio_write(sc, 0x19, 0x4468);
                re_mdio_write(sc, 0x15, 0x00c0);
                re_mdio_write(sc, 0x19, 0x9c02);
                re_mdio_write(sc, 0x15, 0x00c1);
                re_mdio_write(sc, 0x19, 0x58a0);
                re_mdio_write(sc, 0x15, 0x00c2);
                re_mdio_write(sc, 0x19, 0xb605);
                re_mdio_write(sc, 0x15, 0x00c3);
                re_mdio_write(sc, 0x19, 0xc0d3);
                re_mdio_write(sc, 0x15, 0x00c4);
                re_mdio_write(sc, 0x19, 0x00e6);
                re_mdio_write(sc, 0x15, 0x00c5);
                re_mdio_write(sc, 0x19, 0xdaec);
                re_mdio_write(sc, 0x15, 0x00c6);
                re_mdio_write(sc, 0x19, 0x00fa);
                re_mdio_write(sc, 0x15, 0x00c7);
                re_mdio_write(sc, 0x19, 0x9df9);
                re_mdio_write(sc, 0x15, 0x0112);
                re_mdio_write(sc, 0x19, 0x6421);
                re_mdio_write(sc, 0x15, 0x0113);
                re_mdio_write(sc, 0x19, 0x7c08);
                re_mdio_write(sc, 0x15, 0x0114);
                re_mdio_write(sc, 0x19, 0x63f0);
                re_mdio_write(sc, 0x15, 0x0115);
                re_mdio_write(sc, 0x19, 0x4003);
                re_mdio_write(sc, 0x15, 0x0116);
                re_mdio_write(sc, 0x19, 0x4418);
                re_mdio_write(sc, 0x15, 0x0117);
                re_mdio_write(sc, 0x19, 0x9b00);
                re_mdio_write(sc, 0x15, 0x0118);
                re_mdio_write(sc, 0x19, 0x6461);
                re_mdio_write(sc, 0x15, 0x0119);
                re_mdio_write(sc, 0x19, 0x64e1);
                re_mdio_write(sc, 0x15, 0x011a);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0150);
                re_mdio_write(sc, 0x19, 0x7c80);
                re_mdio_write(sc, 0x15, 0x0151);
                re_mdio_write(sc, 0x19, 0x6461);
                re_mdio_write(sc, 0x15, 0x0152);
                re_mdio_write(sc, 0x19, 0x4003);
                re_mdio_write(sc, 0x15, 0x0153);
                re_mdio_write(sc, 0x19, 0x4540);
                re_mdio_write(sc, 0x15, 0x0154);
                re_mdio_write(sc, 0x19, 0x9f00);
                re_mdio_write(sc, 0x15, 0x0155);
                re_mdio_write(sc, 0x19, 0x9d00);
                re_mdio_write(sc, 0x15, 0x0156);
                re_mdio_write(sc, 0x19, 0x7c40);
                re_mdio_write(sc, 0x15, 0x0157);
                re_mdio_write(sc, 0x19, 0x6421);
                re_mdio_write(sc, 0x15, 0x0158);
                re_mdio_write(sc, 0x19, 0x7c80);
                re_mdio_write(sc, 0x15, 0x0159);
                re_mdio_write(sc, 0x19, 0x64a1);
                re_mdio_write(sc, 0x15, 0x015a);
                re_mdio_write(sc, 0x19, 0x30fe);
                re_mdio_write(sc, 0x15, 0x029c);
                re_mdio_write(sc, 0x19, 0x0070);
                re_mdio_write(sc, 0x15, 0x02b2);
                re_mdio_write(sc, 0x19, 0x005a);
                re_mdio_write(sc, 0x15, 0x02bd);
                re_mdio_write(sc, 0x19, 0xa522);
                re_mdio_write(sc, 0x15, 0x02ce);
                re_mdio_write(sc, 0x19, 0xb63e);
                re_mdio_write(sc, 0x15, 0x02d9);
                re_mdio_write(sc, 0x19, 0x32df);
                re_mdio_write(sc, 0x15, 0x02df);
                re_mdio_write(sc, 0x19, 0x4500);
                re_mdio_write(sc, 0x15, 0x02f4);
                re_mdio_write(sc, 0x19, 0xb618);
                re_mdio_write(sc, 0x15, 0x02fb);
                re_mdio_write(sc, 0x19, 0xb900);
                re_mdio_write(sc, 0x15, 0x02fc);
                re_mdio_write(sc, 0x19, 0x49b5);
                re_mdio_write(sc, 0x15, 0x02fd);
                re_mdio_write(sc, 0x19, 0x6812);
                re_mdio_write(sc, 0x15, 0x02fe);
                re_mdio_write(sc, 0x19, 0x66a0);
                re_mdio_write(sc, 0x15, 0x02ff);
                re_mdio_write(sc, 0x19, 0x9900);
                re_mdio_write(sc, 0x15, 0x0300);
                re_mdio_write(sc, 0x19, 0x64a0);
                re_mdio_write(sc, 0x15, 0x0301);
                re_mdio_write(sc, 0x19, 0x3316);
                re_mdio_write(sc, 0x15, 0x0308);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x030c);
                re_mdio_write(sc, 0x19, 0x3000);
                re_mdio_write(sc, 0x15, 0x0312);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0313);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0314);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0315);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0316);
                re_mdio_write(sc, 0x19, 0x49b5);
                re_mdio_write(sc, 0x15, 0x0317);
                re_mdio_write(sc, 0x19, 0x7d00);
                re_mdio_write(sc, 0x15, 0x0318);
                re_mdio_write(sc, 0x19, 0x4d00);
                re_mdio_write(sc, 0x15, 0x0319);
                re_mdio_write(sc, 0x19, 0x6810);
                re_mdio_write(sc, 0x15, 0x031a);
                re_mdio_write(sc, 0x19, 0x6c08);
                re_mdio_write(sc, 0x15, 0x031b);
                re_mdio_write(sc, 0x19, 0x4925);
                re_mdio_write(sc, 0x15, 0x031c);
                re_mdio_write(sc, 0x19, 0x403b);
                re_mdio_write(sc, 0x15, 0x031d);
                re_mdio_write(sc, 0x19, 0xa602);
                re_mdio_write(sc, 0x15, 0x031e);
                re_mdio_write(sc, 0x19, 0x402f);
                re_mdio_write(sc, 0x15, 0x031f);
                re_mdio_write(sc, 0x19, 0x4484);
                re_mdio_write(sc, 0x15, 0x0320);
                re_mdio_write(sc, 0x19, 0x40c8);
                re_mdio_write(sc, 0x15, 0x0321);
                re_mdio_write(sc, 0x19, 0x44c4);
                re_mdio_write(sc, 0x15, 0x0322);
                re_mdio_write(sc, 0x19, 0x404f);
                re_mdio_write(sc, 0x15, 0x0323);
                re_mdio_write(sc, 0x19, 0x44c8);
                re_mdio_write(sc, 0x15, 0x0324);
                re_mdio_write(sc, 0x19, 0xd64f);
                re_mdio_write(sc, 0x15, 0x0325);
                re_mdio_write(sc, 0x19, 0x00e7);
                re_mdio_write(sc, 0x15, 0x0326);
                re_mdio_write(sc, 0x19, 0x7c08);
                re_mdio_write(sc, 0x15, 0x0327);
                re_mdio_write(sc, 0x19, 0x8203);
                re_mdio_write(sc, 0x15, 0x0328);
                re_mdio_write(sc, 0x19, 0x4d48);
                re_mdio_write(sc, 0x15, 0x0329);
                re_mdio_write(sc, 0x19, 0x332b);
                re_mdio_write(sc, 0x15, 0x032a);
                re_mdio_write(sc, 0x19, 0x4d40);
                re_mdio_write(sc, 0x15, 0x032c);
                re_mdio_write(sc, 0x19, 0x00f8);
                re_mdio_write(sc, 0x15, 0x032d);
                re_mdio_write(sc, 0x19, 0x82b2);
                re_mdio_write(sc, 0x15, 0x032f);
                re_mdio_write(sc, 0x19, 0x00b0);
                re_mdio_write(sc, 0x15, 0x0332);
                re_mdio_write(sc, 0x19, 0x91f2);
                re_mdio_write(sc, 0x15, 0x033f);
                re_mdio_write(sc, 0x19, 0xb6cd);
                re_mdio_write(sc, 0x15, 0x0340);
                re_mdio_write(sc, 0x19, 0x9e01);
                re_mdio_write(sc, 0x15, 0x0341);
                re_mdio_write(sc, 0x19, 0xd11d);
                re_mdio_write(sc, 0x15, 0x0342);
                re_mdio_write(sc, 0x19, 0x009d);
                re_mdio_write(sc, 0x15, 0x0343);
                re_mdio_write(sc, 0x19, 0xbb1c);
                re_mdio_write(sc, 0x15, 0x0344);
                re_mdio_write(sc, 0x19, 0x8102);
                re_mdio_write(sc, 0x15, 0x0345);
                re_mdio_write(sc, 0x19, 0x3348);
                re_mdio_write(sc, 0x15, 0x0346);
                re_mdio_write(sc, 0x19, 0xa231);
                re_mdio_write(sc, 0x15, 0x0347);
                re_mdio_write(sc, 0x19, 0x335b);
                re_mdio_write(sc, 0x15, 0x0348);
                re_mdio_write(sc, 0x19, 0x91f7);
                re_mdio_write(sc, 0x15, 0x0349);
                re_mdio_write(sc, 0x19, 0xc218);
                re_mdio_write(sc, 0x15, 0x034a);
                re_mdio_write(sc, 0x19, 0x00f5);
                re_mdio_write(sc, 0x15, 0x034b);
                re_mdio_write(sc, 0x19, 0x335b);
                re_mdio_write(sc, 0x15, 0x034c);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x034d);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x034e);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x034f);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x0350);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x035b);
                re_mdio_write(sc, 0x19, 0xa23c);
                re_mdio_write(sc, 0x15, 0x035c);
                re_mdio_write(sc, 0x19, 0x7c08);
                re_mdio_write(sc, 0x15, 0x035d);
                re_mdio_write(sc, 0x19, 0x4c00);
                re_mdio_write(sc, 0x15, 0x035e);
                re_mdio_write(sc, 0x19, 0x3397);
                re_mdio_write(sc, 0x15, 0x0363);
                re_mdio_write(sc, 0x19, 0xb6a9);
                re_mdio_write(sc, 0x15, 0x0366);
                re_mdio_write(sc, 0x19, 0x00f5);
                re_mdio_write(sc, 0x15, 0x0382);
                re_mdio_write(sc, 0x19, 0x7c40);
                re_mdio_write(sc, 0x15, 0x0388);
                re_mdio_write(sc, 0x19, 0x0084);
                re_mdio_write(sc, 0x15, 0x0389);
                re_mdio_write(sc, 0x19, 0xdd17);
                re_mdio_write(sc, 0x15, 0x038a);
                re_mdio_write(sc, 0x19, 0x000b);
                re_mdio_write(sc, 0x15, 0x038b);
                re_mdio_write(sc, 0x19, 0xa10a);
                re_mdio_write(sc, 0x15, 0x038c);
                re_mdio_write(sc, 0x19, 0x337e);
                re_mdio_write(sc, 0x15, 0x038d);
                re_mdio_write(sc, 0x19, 0x6c0b);
                re_mdio_write(sc, 0x15, 0x038e);
                re_mdio_write(sc, 0x19, 0xa107);
                re_mdio_write(sc, 0x15, 0x038f);
                re_mdio_write(sc, 0x19, 0x6c08);
                re_mdio_write(sc, 0x15, 0x0390);
                re_mdio_write(sc, 0x19, 0xc017);
                re_mdio_write(sc, 0x15, 0x0391);
                re_mdio_write(sc, 0x19, 0x0004);
                re_mdio_write(sc, 0x15, 0x0392);
                re_mdio_write(sc, 0x19, 0xd64f);
                re_mdio_write(sc, 0x15, 0x0393);
                re_mdio_write(sc, 0x19, 0x00f4);
                re_mdio_write(sc, 0x15, 0x0397);
                re_mdio_write(sc, 0x19, 0x4098);
                re_mdio_write(sc, 0x15, 0x0398);
                re_mdio_write(sc, 0x19, 0x4408);
                re_mdio_write(sc, 0x15, 0x0399);
                re_mdio_write(sc, 0x19, 0x55bf);
                re_mdio_write(sc, 0x15, 0x039a);
                re_mdio_write(sc, 0x19, 0x4bb9);
                re_mdio_write(sc, 0x15, 0x039b);
                re_mdio_write(sc, 0x19, 0x6810);
                re_mdio_write(sc, 0x15, 0x039c);
                re_mdio_write(sc, 0x19, 0x4b29);
                re_mdio_write(sc, 0x15, 0x039d);
                re_mdio_write(sc, 0x19, 0x4041);
                re_mdio_write(sc, 0x15, 0x039e);
                re_mdio_write(sc, 0x19, 0x442a);
                re_mdio_write(sc, 0x15, 0x039f);
                re_mdio_write(sc, 0x19, 0x4029);
                re_mdio_write(sc, 0x15, 0x03aa);
                re_mdio_write(sc, 0x19, 0x33b8);
                re_mdio_write(sc, 0x15, 0x03b6);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x03b7);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x03b8);
                re_mdio_write(sc, 0x19, 0x543f);
                re_mdio_write(sc, 0x15, 0x03b9);
                re_mdio_write(sc, 0x19, 0x499a);
                re_mdio_write(sc, 0x15, 0x03ba);
                re_mdio_write(sc, 0x19, 0x7c40);
                re_mdio_write(sc, 0x15, 0x03bb);
                re_mdio_write(sc, 0x19, 0x4c40);
                re_mdio_write(sc, 0x15, 0x03bc);
                re_mdio_write(sc, 0x19, 0x490a);
                re_mdio_write(sc, 0x15, 0x03bd);
                re_mdio_write(sc, 0x19, 0x405e);
                re_mdio_write(sc, 0x15, 0x03c2);
                re_mdio_write(sc, 0x19, 0x9a03);
                re_mdio_write(sc, 0x15, 0x03c4);
                re_mdio_write(sc, 0x19, 0x0015);
                re_mdio_write(sc, 0x15, 0x03c5);
                re_mdio_write(sc, 0x19, 0x9e03);
                re_mdio_write(sc, 0x15, 0x03c8);
                re_mdio_write(sc, 0x19, 0x9cf7);
                re_mdio_write(sc, 0x15, 0x03c9);
                re_mdio_write(sc, 0x19, 0x7c12);
                re_mdio_write(sc, 0x15, 0x03ca);
                re_mdio_write(sc, 0x19, 0x4c52);
                re_mdio_write(sc, 0x15, 0x03cb);
                re_mdio_write(sc, 0x19, 0x4458);
                re_mdio_write(sc, 0x15, 0x03cd);
                re_mdio_write(sc, 0x19, 0x4c40);
                re_mdio_write(sc, 0x15, 0x03ce);
                re_mdio_write(sc, 0x19, 0x33bf);
                re_mdio_write(sc, 0x15, 0x03cf);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x03d0);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x03d1);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x03d5);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x03d6);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x03d7);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x03d8);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x03d9);
                re_mdio_write(sc, 0x19, 0x49bb);
                re_mdio_write(sc, 0x15, 0x03da);
                re_mdio_write(sc, 0x19, 0x4478);
                re_mdio_write(sc, 0x15, 0x03db);
                re_mdio_write(sc, 0x19, 0x492b);
                re_mdio_write(sc, 0x15, 0x03dc);
                re_mdio_write(sc, 0x19, 0x7c01);
                re_mdio_write(sc, 0x15, 0x03dd);
                re_mdio_write(sc, 0x19, 0x4c00);
                re_mdio_write(sc, 0x15, 0x03de);
                re_mdio_write(sc, 0x19, 0xbd1a);
                re_mdio_write(sc, 0x15, 0x03df);
                re_mdio_write(sc, 0x19, 0xc428);
                re_mdio_write(sc, 0x15, 0x03e0);
                re_mdio_write(sc, 0x19, 0x0008);
                re_mdio_write(sc, 0x15, 0x03e1);
                re_mdio_write(sc, 0x19, 0x9cfd);
                re_mdio_write(sc, 0x15, 0x03e2);
                re_mdio_write(sc, 0x19, 0x7c12);
                re_mdio_write(sc, 0x15, 0x03e3);
                re_mdio_write(sc, 0x19, 0x4c52);
                re_mdio_write(sc, 0x15, 0x03e4);
                re_mdio_write(sc, 0x19, 0x4458);
                re_mdio_write(sc, 0x15, 0x03e5);
                re_mdio_write(sc, 0x19, 0x7c12);
                re_mdio_write(sc, 0x15, 0x03e6);
                re_mdio_write(sc, 0x19, 0x4c40);
                re_mdio_write(sc, 0x15, 0x03e7);
                re_mdio_write(sc, 0x19, 0x33de);
                re_mdio_write(sc, 0x15, 0x03e8);
                re_mdio_write(sc, 0x19, 0xc218);
                re_mdio_write(sc, 0x15, 0x03e9);
                re_mdio_write(sc, 0x19, 0x0002);
                re_mdio_write(sc, 0x15, 0x03ea);
                re_mdio_write(sc, 0x19, 0x32df);
                re_mdio_write(sc, 0x15, 0x03eb);
                re_mdio_write(sc, 0x19, 0x3316);
                re_mdio_write(sc, 0x15, 0x03ec);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x03ed);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x03ee);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x03ef);
                re_mdio_write(sc, 0x19, 0x0000);
                re_mdio_write(sc, 0x15, 0x03f7);
                re_mdio_write(sc, 0x19, 0x330c);
                re_mdio_write(sc, 0x16, 0x0306);
                re_mdio_write(sc, 0x16, 0x0300);

                re_mdio_write(sc, 0x1f, 0x0005);
                re_mdio_write(sc, 0x05, 0xfff6);
                re_mdio_write(sc, 0x06, 0x0080);
                re_mdio_write(sc, 0x05, 0x8000);
                re_mdio_write(sc, 0x06, 0x0280);
                re_mdio_write(sc, 0x06, 0x48f7);
                re_mdio_write(sc, 0x06, 0x00e0);
                re_mdio_write(sc, 0x06, 0xfff7);
                re_mdio_write(sc, 0x06, 0xa080);
                re_mdio_write(sc, 0x06, 0x02ae);
                re_mdio_write(sc, 0x06, 0xf602);
                re_mdio_write(sc, 0x06, 0x0200);
                re_mdio_write(sc, 0x06, 0x0280);
                re_mdio_write(sc, 0x06, 0x9002);
                re_mdio_write(sc, 0x06, 0x0224);
                re_mdio_write(sc, 0x06, 0x0202);
                re_mdio_write(sc, 0x06, 0x3402);
                re_mdio_write(sc, 0x06, 0x027f);
                re_mdio_write(sc, 0x06, 0x0280);
                re_mdio_write(sc, 0x06, 0xa602);
                re_mdio_write(sc, 0x06, 0x80bf);
                re_mdio_write(sc, 0x06, 0xe08b);
                re_mdio_write(sc, 0x06, 0x88e1);
                re_mdio_write(sc, 0x06, 0x8b89);
                re_mdio_write(sc, 0x06, 0x1e01);
                re_mdio_write(sc, 0x06, 0xe18b);
                re_mdio_write(sc, 0x06, 0x8a1e);
                re_mdio_write(sc, 0x06, 0x01e1);
                re_mdio_write(sc, 0x06, 0x8b8b);
                re_mdio_write(sc, 0x06, 0x1e01);
                re_mdio_write(sc, 0x06, 0xe18b);
                re_mdio_write(sc, 0x06, 0x8c1e);
                re_mdio_write(sc, 0x06, 0x01e1);
                re_mdio_write(sc, 0x06, 0x8b8d);
                re_mdio_write(sc, 0x06, 0x1e01);
                re_mdio_write(sc, 0x06, 0xe18b);
                re_mdio_write(sc, 0x06, 0x8e1e);
                re_mdio_write(sc, 0x06, 0x01a0);
                re_mdio_write(sc, 0x06, 0x00c7);
                re_mdio_write(sc, 0x06, 0xaebb);
                re_mdio_write(sc, 0x06, 0xee8a);
                re_mdio_write(sc, 0x06, 0xe600);
                re_mdio_write(sc, 0x06, 0xee8a);
                re_mdio_write(sc, 0x06, 0xee03);
                re_mdio_write(sc, 0x06, 0xee8a);
                re_mdio_write(sc, 0x06, 0xefb8);
                re_mdio_write(sc, 0x06, 0xee8a);
                re_mdio_write(sc, 0x06, 0xe902);
                re_mdio_write(sc, 0x06, 0xee8b);
                re_mdio_write(sc, 0x06, 0x8285);
                re_mdio_write(sc, 0x06, 0xee8b);
                re_mdio_write(sc, 0x06, 0x8520);
                re_mdio_write(sc, 0x06, 0xee8b);
                re_mdio_write(sc, 0x06, 0x8701);
                re_mdio_write(sc, 0x06, 0xd481);
                re_mdio_write(sc, 0x06, 0x35e4);
                re_mdio_write(sc, 0x06, 0x8b94);
                re_mdio_write(sc, 0x06, 0xe58b);
                re_mdio_write(sc, 0x06, 0x95bf);
                re_mdio_write(sc, 0x06, 0x8b88);
                re_mdio_write(sc, 0x06, 0xec00);
                re_mdio_write(sc, 0x06, 0x19a9);
                re_mdio_write(sc, 0x06, 0x8b90);
                re_mdio_write(sc, 0x06, 0xf9ee);
                re_mdio_write(sc, 0x06, 0xfff6);
                re_mdio_write(sc, 0x06, 0x00ee);
                re_mdio_write(sc, 0x06, 0xfff7);
                re_mdio_write(sc, 0x06, 0xffe0);
                re_mdio_write(sc, 0x06, 0xe140);
                re_mdio_write(sc, 0x06, 0xe1e1);
                re_mdio_write(sc, 0x06, 0x41f7);
                re_mdio_write(sc, 0x06, 0x2ff6);
                re_mdio_write(sc, 0x06, 0x28e4);
                re_mdio_write(sc, 0x06, 0xe140);
                re_mdio_write(sc, 0x06, 0xe5e1);
                re_mdio_write(sc, 0x06, 0x4104);
                re_mdio_write(sc, 0x06, 0xf8e0);
                re_mdio_write(sc, 0x06, 0x8b89);
                re_mdio_write(sc, 0x06, 0xad20);
                re_mdio_write(sc, 0x06, 0x0dee);
                re_mdio_write(sc, 0x06, 0x8b89);
                re_mdio_write(sc, 0x06, 0x0002);
                re_mdio_write(sc, 0x06, 0x82f4);
                re_mdio_write(sc, 0x06, 0x021f);
                re_mdio_write(sc, 0x06, 0x4102);
                re_mdio_write(sc, 0x06, 0x2812);
                re_mdio_write(sc, 0x06, 0xfc04);
                re_mdio_write(sc, 0x06, 0xf8e0);
                re_mdio_write(sc, 0x06, 0x8b8d);
                re_mdio_write(sc, 0x06, 0xad20);
                re_mdio_write(sc, 0x06, 0x10ee);
                re_mdio_write(sc, 0x06, 0x8b8d);
                re_mdio_write(sc, 0x06, 0x0002);
                re_mdio_write(sc, 0x06, 0x139d);
                re_mdio_write(sc, 0x06, 0x0281);
                re_mdio_write(sc, 0x06, 0xd602);
                re_mdio_write(sc, 0x06, 0x1f99);
                re_mdio_write(sc, 0x06, 0x0227);
                re_mdio_write(sc, 0x06, 0xeafc);
                re_mdio_write(sc, 0x06, 0x04f8);
                re_mdio_write(sc, 0x06, 0xe08b);
                re_mdio_write(sc, 0x06, 0x8ead);
                re_mdio_write(sc, 0x06, 0x2014);
                re_mdio_write(sc, 0x06, 0xf620);
                re_mdio_write(sc, 0x06, 0xe48b);
                re_mdio_write(sc, 0x06, 0x8e02);
                re_mdio_write(sc, 0x06, 0x8104);
                re_mdio_write(sc, 0x06, 0x021b);
                re_mdio_write(sc, 0x06, 0xf402);
                re_mdio_write(sc, 0x06, 0x2c9c);
                re_mdio_write(sc, 0x06, 0x0281);
                re_mdio_write(sc, 0x06, 0x7902);
                re_mdio_write(sc, 0x06, 0x8443);
                re_mdio_write(sc, 0x06, 0xad22);
                re_mdio_write(sc, 0x06, 0x11f6);
                re_mdio_write(sc, 0x06, 0x22e4);
                re_mdio_write(sc, 0x06, 0x8b8e);
                re_mdio_write(sc, 0x06, 0x022c);
                re_mdio_write(sc, 0x06, 0x4602);
                re_mdio_write(sc, 0x06, 0x2ac5);
                re_mdio_write(sc, 0x06, 0x0229);
                re_mdio_write(sc, 0x06, 0x2002);
                re_mdio_write(sc, 0x06, 0x2b91);
                re_mdio_write(sc, 0x06, 0xad25);
                re_mdio_write(sc, 0x06, 0x11f6);
                re_mdio_write(sc, 0x06, 0x25e4);
                re_mdio_write(sc, 0x06, 0x8b8e);
                re_mdio_write(sc, 0x06, 0x0284);
                re_mdio_write(sc, 0x06, 0xe202);
                re_mdio_write(sc, 0x06, 0x043a);
                re_mdio_write(sc, 0x06, 0x021a);
                re_mdio_write(sc, 0x06, 0x5902);
                re_mdio_write(sc, 0x06, 0x2bfc);
                re_mdio_write(sc, 0x06, 0xfc04);
                re_mdio_write(sc, 0x06, 0xf8fa);
                re_mdio_write(sc, 0x06, 0xef69);
                re_mdio_write(sc, 0x06, 0xe0e0);
                re_mdio_write(sc, 0x06, 0x00e1);
                re_mdio_write(sc, 0x06, 0xe001);
                re_mdio_write(sc, 0x06, 0xad27);
                re_mdio_write(sc, 0x06, 0x1fd1);
                re_mdio_write(sc, 0x06, 0x01bf);
                re_mdio_write(sc, 0x06, 0x8638);
                re_mdio_write(sc, 0x06, 0x022f);
                re_mdio_write(sc, 0x06, 0x50e0);
                re_mdio_write(sc, 0x06, 0xe020);
                re_mdio_write(sc, 0x06, 0xe1e0);
                re_mdio_write(sc, 0x06, 0x21ad);
                re_mdio_write(sc, 0x06, 0x200e);
                re_mdio_write(sc, 0x06, 0xd100);
                re_mdio_write(sc, 0x06, 0xbf86);
                re_mdio_write(sc, 0x06, 0x3802);
                re_mdio_write(sc, 0x06, 0x2f50);
                re_mdio_write(sc, 0x06, 0xbf3d);
                re_mdio_write(sc, 0x06, 0x3902);
                re_mdio_write(sc, 0x06, 0x2eb0);
                re_mdio_write(sc, 0x06, 0xef96);
                re_mdio_write(sc, 0x06, 0xfefc);
                re_mdio_write(sc, 0x06, 0x0402);
                re_mdio_write(sc, 0x06, 0x8591);
                re_mdio_write(sc, 0x06, 0x0281);
                re_mdio_write(sc, 0x06, 0x3c05);
                re_mdio_write(sc, 0x06, 0xf8fa);
                re_mdio_write(sc, 0x06, 0xef69);
                re_mdio_write(sc, 0x06, 0xe0e2);
                re_mdio_write(sc, 0x06, 0xfee1);
                re_mdio_write(sc, 0x06, 0xe2ff);
                re_mdio_write(sc, 0x06, 0xad2d);
                re_mdio_write(sc, 0x06, 0x1ae0);
                re_mdio_write(sc, 0x06, 0xe14e);
                re_mdio_write(sc, 0x06, 0xe1e1);
                re_mdio_write(sc, 0x06, 0x4fac);
                re_mdio_write(sc, 0x06, 0x2d22);
                re_mdio_write(sc, 0x06, 0xf603);
                re_mdio_write(sc, 0x06, 0x0203);
                re_mdio_write(sc, 0x06, 0x36f7);
                re_mdio_write(sc, 0x06, 0x03f7);
                re_mdio_write(sc, 0x06, 0x06bf);
                re_mdio_write(sc, 0x06, 0x8622);
                re_mdio_write(sc, 0x06, 0x022e);
                re_mdio_write(sc, 0x06, 0xb0ae);
                re_mdio_write(sc, 0x06, 0x11e0);
                re_mdio_write(sc, 0x06, 0xe14e);
                re_mdio_write(sc, 0x06, 0xe1e1);
                re_mdio_write(sc, 0x06, 0x4fad);
                re_mdio_write(sc, 0x06, 0x2d08);
                re_mdio_write(sc, 0x06, 0xbf86);
                re_mdio_write(sc, 0x06, 0x2d02);
                re_mdio_write(sc, 0x06, 0x2eb0);
                re_mdio_write(sc, 0x06, 0xf606);
                re_mdio_write(sc, 0x06, 0xef96);
                re_mdio_write(sc, 0x06, 0xfefc);
                re_mdio_write(sc, 0x06, 0x04f8);
                re_mdio_write(sc, 0x06, 0xf9fa);
                re_mdio_write(sc, 0x06, 0xef69);
                re_mdio_write(sc, 0x06, 0xe08b);
                re_mdio_write(sc, 0x06, 0x87ad);
                re_mdio_write(sc, 0x06, 0x204c);
                re_mdio_write(sc, 0x06, 0xd200);
                re_mdio_write(sc, 0x06, 0xe0e2);
                re_mdio_write(sc, 0x06, 0x0058);
                re_mdio_write(sc, 0x06, 0x010c);
                re_mdio_write(sc, 0x06, 0x021e);
                re_mdio_write(sc, 0x06, 0x20e0);
                re_mdio_write(sc, 0x06, 0xe000);
                re_mdio_write(sc, 0x06, 0x5810);
                re_mdio_write(sc, 0x06, 0x1e20);
                re_mdio_write(sc, 0x06, 0xe0e0);
                re_mdio_write(sc, 0x06, 0x3658);
                re_mdio_write(sc, 0x06, 0x031e);
                re_mdio_write(sc, 0x06, 0x20e0);
                re_mdio_write(sc, 0x06, 0xe022);
                re_mdio_write(sc, 0x06, 0xe1e0);
                re_mdio_write(sc, 0x06, 0x2358);
                re_mdio_write(sc, 0x06, 0xe01e);
                re_mdio_write(sc, 0x06, 0x20e0);
                re_mdio_write(sc, 0x06, 0x8ae6);
                re_mdio_write(sc, 0x06, 0x1f02);
                re_mdio_write(sc, 0x06, 0x9e22);
                re_mdio_write(sc, 0x06, 0xe68a);
                re_mdio_write(sc, 0x06, 0xe6ad);
                re_mdio_write(sc, 0x06, 0x3214);
                re_mdio_write(sc, 0x06, 0xad34);
                re_mdio_write(sc, 0x06, 0x11ef);
                re_mdio_write(sc, 0x06, 0x0258);
                re_mdio_write(sc, 0x06, 0x039e);
                re_mdio_write(sc, 0x06, 0x07ad);
                re_mdio_write(sc, 0x06, 0x3508);
                re_mdio_write(sc, 0x06, 0x5ac0);
                re_mdio_write(sc, 0x06, 0x9f04);
                re_mdio_write(sc, 0x06, 0xd101);
                re_mdio_write(sc, 0x06, 0xae02);
                re_mdio_write(sc, 0x06, 0xd100);
                re_mdio_write(sc, 0x06, 0xbf86);
                re_mdio_write(sc, 0x06, 0x3e02);
                re_mdio_write(sc, 0x06, 0x2f50);
                re_mdio_write(sc, 0x06, 0xef96);
                re_mdio_write(sc, 0x06, 0xfefd);
                re_mdio_write(sc, 0x06, 0xfc04);
                re_mdio_write(sc, 0x06, 0xf8f9);
                re_mdio_write(sc, 0x06, 0xfae0);
                re_mdio_write(sc, 0x06, 0x8b81);
                re_mdio_write(sc, 0x06, 0xac26);
                re_mdio_write(sc, 0x06, 0x0ee0);
                re_mdio_write(sc, 0x06, 0x8b81);
                re_mdio_write(sc, 0x06, 0xac21);
                re_mdio_write(sc, 0x06, 0x08e0);
                re_mdio_write(sc, 0x06, 0x8b87);
                re_mdio_write(sc, 0x06, 0xac24);
                re_mdio_write(sc, 0x06, 0x02ae);
                re_mdio_write(sc, 0x06, 0x6bee);
                re_mdio_write(sc, 0x06, 0xe0ea);
                re_mdio_write(sc, 0x06, 0x00ee);
                re_mdio_write(sc, 0x06, 0xe0eb);
                re_mdio_write(sc, 0x06, 0x00e2);
                re_mdio_write(sc, 0x06, 0xe07c);
                re_mdio_write(sc, 0x06, 0xe3e0);
                re_mdio_write(sc, 0x06, 0x7da5);
                re_mdio_write(sc, 0x06, 0x1111);
                re_mdio_write(sc, 0x06, 0x15d2);
                re_mdio_write(sc, 0x06, 0x60d6);
                re_mdio_write(sc, 0x06, 0x6666);
                re_mdio_write(sc, 0x06, 0x0207);
                re_mdio_write(sc, 0x06, 0xf9d2);
                re_mdio_write(sc, 0x06, 0xa0d6);
                re_mdio_write(sc, 0x06, 0xaaaa);
                re_mdio_write(sc, 0x06, 0x0207);
                re_mdio_write(sc, 0x06, 0xf902);
                re_mdio_write(sc, 0x06, 0x825c);
                re_mdio_write(sc, 0x06, 0xae44);
                re_mdio_write(sc, 0x06, 0xa566);
                re_mdio_write(sc, 0x06, 0x6602);
                re_mdio_write(sc, 0x06, 0xae38);
                re_mdio_write(sc, 0x06, 0xa5aa);
                re_mdio_write(sc, 0x06, 0xaa02);
                re_mdio_write(sc, 0x06, 0xae32);
                re_mdio_write(sc, 0x06, 0xeee0);
                re_mdio_write(sc, 0x06, 0xea04);
                re_mdio_write(sc, 0x06, 0xeee0);
                re_mdio_write(sc, 0x06, 0xeb06);
                re_mdio_write(sc, 0x06, 0xe2e0);
                re_mdio_write(sc, 0x06, 0x7ce3);
                re_mdio_write(sc, 0x06, 0xe07d);
                re_mdio_write(sc, 0x06, 0xe0e0);
                re_mdio_write(sc, 0x06, 0x38e1);
                re_mdio_write(sc, 0x06, 0xe039);
                re_mdio_write(sc, 0x06, 0xad2e);
                re_mdio_write(sc, 0x06, 0x21ad);
                re_mdio_write(sc, 0x06, 0x3f13);
                re_mdio_write(sc, 0x06, 0xe0e4);
                re_mdio_write(sc, 0x06, 0x14e1);
                re_mdio_write(sc, 0x06, 0xe415);
                re_mdio_write(sc, 0x06, 0x6880);
                re_mdio_write(sc, 0x06, 0xe4e4);
                re_mdio_write(sc, 0x06, 0x14e5);
                re_mdio_write(sc, 0x06, 0xe415);
                re_mdio_write(sc, 0x06, 0x0282);
                re_mdio_write(sc, 0x06, 0x5cae);
                re_mdio_write(sc, 0x06, 0x0bac);
                re_mdio_write(sc, 0x06, 0x3e02);
                re_mdio_write(sc, 0x06, 0xae06);
                re_mdio_write(sc, 0x06, 0x0282);
                re_mdio_write(sc, 0x06, 0x8602);
                re_mdio_write(sc, 0x06, 0x82b0);
                re_mdio_write(sc, 0x06, 0xfefd);
                re_mdio_write(sc, 0x06, 0xfc04);
                re_mdio_write(sc, 0x06, 0xf8e1);
                re_mdio_write(sc, 0x06, 0x8b2e);
                re_mdio_write(sc, 0x06, 0xe08b);
                re_mdio_write(sc, 0x06, 0x81ad);
                re_mdio_write(sc, 0x06, 0x2605);
                re_mdio_write(sc, 0x06, 0x0221);
                re_mdio_write(sc, 0x06, 0xf3f7);
                re_mdio_write(sc, 0x06, 0x28e0);
                re_mdio_write(sc, 0x06, 0x8b81);
                re_mdio_write(sc, 0x06, 0xad21);
                re_mdio_write(sc, 0x06, 0x0502);
                re_mdio_write(sc, 0x06, 0x22f8);
                re_mdio_write(sc, 0x06, 0xf729);
                re_mdio_write(sc, 0x06, 0xe08b);
                re_mdio_write(sc, 0x06, 0x87ad);
                re_mdio_write(sc, 0x06, 0x2405);
                re_mdio_write(sc, 0x06, 0x0282);
                re_mdio_write(sc, 0x06, 0xebf7);
                re_mdio_write(sc, 0x06, 0x2ae5);
                re_mdio_write(sc, 0x06, 0x8b2e);
                re_mdio_write(sc, 0x06, 0xfc04);
                re_mdio_write(sc, 0x06, 0xf8e0);
                re_mdio_write(sc, 0x06, 0x8b81);
                re_mdio_write(sc, 0x06, 0xad26);
                re_mdio_write(sc, 0x06, 0x0302);
                re_mdio_write(sc, 0x06, 0x2134);
                re_mdio_write(sc, 0x06, 0xe08b);
                re_mdio_write(sc, 0x06, 0x81ad);
                re_mdio_write(sc, 0x06, 0x2109);
                re_mdio_write(sc, 0x06, 0xe08b);
                re_mdio_write(sc, 0x06, 0x2eac);
                re_mdio_write(sc, 0x06, 0x2003);
                re_mdio_write(sc, 0x06, 0x0283);
                re_mdio_write(sc, 0x06, 0x52e0);
                re_mdio_write(sc, 0x06, 0x8b87);
                re_mdio_write(sc, 0x06, 0xad24);
                re_mdio_write(sc, 0x06, 0x09e0);
                re_mdio_write(sc, 0x06, 0x8b2e);
                re_mdio_write(sc, 0x06, 0xac21);
                re_mdio_write(sc, 0x06, 0x0302);
                re_mdio_write(sc, 0x06, 0x8337);
                re_mdio_write(sc, 0x06, 0xfc04);
                re_mdio_write(sc, 0x06, 0xf8e1);
                re_mdio_write(sc, 0x06, 0x8b2e);
                re_mdio_write(sc, 0x06, 0xe08b);
                re_mdio_write(sc, 0x06, 0x81ad);
                re_mdio_write(sc, 0x06, 0x2608);
                re_mdio_write(sc, 0x06, 0xe085);
                re_mdio_write(sc, 0x06, 0xd2ad);
                re_mdio_write(sc, 0x06, 0x2502);
                re_mdio_write(sc, 0x06, 0xf628);
                re_mdio_write(sc, 0x06, 0xe08b);
                re_mdio_write(sc, 0x06, 0x81ad);
                re_mdio_write(sc, 0x06, 0x210a);
                re_mdio_write(sc, 0x06, 0xe086);
                re_mdio_write(sc, 0x06, 0x0af6);
                re_mdio_write(sc, 0x06, 0x27a0);
                re_mdio_write(sc, 0x06, 0x0502);
                re_mdio_write(sc, 0x06, 0xf629);
                re_mdio_write(sc, 0x06, 0xe08b);
                re_mdio_write(sc, 0x06, 0x87ad);
                re_mdio_write(sc, 0x06, 0x2408);
                re_mdio_write(sc, 0x06, 0xe08a);
                re_mdio_write(sc, 0x06, 0xedad);
                re_mdio_write(sc, 0x06, 0x2002);
                re_mdio_write(sc, 0x06, 0xf62a);
                re_mdio_write(sc, 0x06, 0xe58b);
                re_mdio_write(sc, 0x06, 0x2ea1);
                re_mdio_write(sc, 0x06, 0x0003);
                re_mdio_write(sc, 0x06, 0x0221);
                re_mdio_write(sc, 0x06, 0x11fc);
                re_mdio_write(sc, 0x06, 0x04ee);
                re_mdio_write(sc, 0x06, 0x8aed);
                re_mdio_write(sc, 0x06, 0x00ee);
                re_mdio_write(sc, 0x06, 0x8aec);
                re_mdio_write(sc, 0x06, 0x0004);
                re_mdio_write(sc, 0x06, 0xf8e0);
                re_mdio_write(sc, 0x06, 0x8b87);
                re_mdio_write(sc, 0x06, 0xad24);
                re_mdio_write(sc, 0x06, 0x3ae0);
                re_mdio_write(sc, 0x06, 0xe0ea);
                re_mdio_write(sc, 0x06, 0xe1e0);
                re_mdio_write(sc, 0x06, 0xeb58);
                re_mdio_write(sc, 0x06, 0xf8d1);
                re_mdio_write(sc, 0x06, 0x01e4);
                re_mdio_write(sc, 0x06, 0xe0ea);
                re_mdio_write(sc, 0x06, 0xe5e0);
                re_mdio_write(sc, 0x06, 0xebe0);
                re_mdio_write(sc, 0x06, 0xe07c);
                re_mdio_write(sc, 0x06, 0xe1e0);
                re_mdio_write(sc, 0x06, 0x7d5c);
                re_mdio_write(sc, 0x06, 0x00ff);
                re_mdio_write(sc, 0x06, 0x3c00);
                re_mdio_write(sc, 0x06, 0x1eab);
                re_mdio_write(sc, 0x06, 0x1ce0);
                re_mdio_write(sc, 0x06, 0xe04c);
                re_mdio_write(sc, 0x06, 0xe1e0);
                re_mdio_write(sc, 0x06, 0x4d58);
                re_mdio_write(sc, 0x06, 0xc1e4);
                re_mdio_write(sc, 0x06, 0xe04c);
                re_mdio_write(sc, 0x06, 0xe5e0);
                re_mdio_write(sc, 0x06, 0x4de0);
                re_mdio_write(sc, 0x06, 0xe0ee);
                re_mdio_write(sc, 0x06, 0xe1e0);
                re_mdio_write(sc, 0x06, 0xef69);
                re_mdio_write(sc, 0x06, 0x3ce4);
                re_mdio_write(sc, 0x06, 0xe0ee);
                re_mdio_write(sc, 0x06, 0xe5e0);
                re_mdio_write(sc, 0x06, 0xeffc);
                re_mdio_write(sc, 0x06, 0x04f8);
                re_mdio_write(sc, 0x06, 0xe08b);
                re_mdio_write(sc, 0x06, 0x87ad);
                re_mdio_write(sc, 0x06, 0x2412);
                re_mdio_write(sc, 0x06, 0xe0e0);
                re_mdio_write(sc, 0x06, 0xeee1);
                re_mdio_write(sc, 0x06, 0xe0ef);
                re_mdio_write(sc, 0x06, 0x59c3);
                re_mdio_write(sc, 0x06, 0xe4e0);
                re_mdio_write(sc, 0x06, 0xeee5);
                re_mdio_write(sc, 0x06, 0xe0ef);
                re_mdio_write(sc, 0x06, 0xee8a);
                re_mdio_write(sc, 0x06, 0xed01);
                re_mdio_write(sc, 0x06, 0xfc04);
                re_mdio_write(sc, 0x06, 0xf8e0);
                re_mdio_write(sc, 0x06, 0x8b81);
                re_mdio_write(sc, 0x06, 0xac25);
                re_mdio_write(sc, 0x06, 0x0502);
                re_mdio_write(sc, 0x06, 0x8363);
                re_mdio_write(sc, 0x06, 0xae03);
                re_mdio_write(sc, 0x06, 0x0225);
                re_mdio_write(sc, 0x06, 0x16fc);
                re_mdio_write(sc, 0x06, 0x04f8);
                re_mdio_write(sc, 0x06, 0xf9fa);
                re_mdio_write(sc, 0x06, 0xef69);
                re_mdio_write(sc, 0x06, 0xfae0);
                re_mdio_write(sc, 0x06, 0x860a);
                re_mdio_write(sc, 0x06, 0xa000);
                re_mdio_write(sc, 0x06, 0x19e0);
                re_mdio_write(sc, 0x06, 0x860b);
                re_mdio_write(sc, 0x06, 0xe18b);
                re_mdio_write(sc, 0x06, 0x331b);
                re_mdio_write(sc, 0x06, 0x109e);
                re_mdio_write(sc, 0x06, 0x04aa);
                re_mdio_write(sc, 0x06, 0x02ae);
                re_mdio_write(sc, 0x06, 0x06ee);
                re_mdio_write(sc, 0x06, 0x860a);
                re_mdio_write(sc, 0x06, 0x01ae);
                re_mdio_write(sc, 0x06, 0xe602);
                re_mdio_write(sc, 0x06, 0x241e);
                re_mdio_write(sc, 0x06, 0xae14);
                re_mdio_write(sc, 0x06, 0xa001);
                re_mdio_write(sc, 0x06, 0x1402);
                re_mdio_write(sc, 0x06, 0x2426);
                re_mdio_write(sc, 0x06, 0xbf26);
                re_mdio_write(sc, 0x06, 0x6d02);
                re_mdio_write(sc, 0x06, 0x2eb0);
                re_mdio_write(sc, 0x06, 0xee86);
                re_mdio_write(sc, 0x06, 0x0b00);
                re_mdio_write(sc, 0x06, 0xee86);
                re_mdio_write(sc, 0x06, 0x0a02);
                re_mdio_write(sc, 0x06, 0xaf84);
                re_mdio_write(sc, 0x06, 0x3ca0);
                re_mdio_write(sc, 0x06, 0x0252);
                re_mdio_write(sc, 0x06, 0xee86);
                re_mdio_write(sc, 0x06, 0x0400);
                re_mdio_write(sc, 0x06, 0xee86);
                re_mdio_write(sc, 0x06, 0x0500);
                re_mdio_write(sc, 0x06, 0xe086);
                re_mdio_write(sc, 0x06, 0x0be1);
                re_mdio_write(sc, 0x06, 0x8b32);
                re_mdio_write(sc, 0x06, 0x1b10);
                re_mdio_write(sc, 0x06, 0x9e04);
                re_mdio_write(sc, 0x06, 0xaa02);
                re_mdio_write(sc, 0x06, 0xaecb);
                re_mdio_write(sc, 0x06, 0xee86);
                re_mdio_write(sc, 0x06, 0x0b00);
                re_mdio_write(sc, 0x06, 0x0224);
                re_mdio_write(sc, 0x06, 0x3ae2);
                re_mdio_write(sc, 0x06, 0x8604);
                re_mdio_write(sc, 0x06, 0xe386);
                re_mdio_write(sc, 0x06, 0x05ef);
                re_mdio_write(sc, 0x06, 0x65e2);
                re_mdio_write(sc, 0x06, 0x8606);
                re_mdio_write(sc, 0x06, 0xe386);
                re_mdio_write(sc, 0x06, 0x071b);
                re_mdio_write(sc, 0x06, 0x56aa);
                re_mdio_write(sc, 0x06, 0x0eef);
                re_mdio_write(sc, 0x06, 0x56e6);
                re_mdio_write(sc, 0x06, 0x8606);
                re_mdio_write(sc, 0x06, 0xe786);
                re_mdio_write(sc, 0x06, 0x07e2);
                re_mdio_write(sc, 0x06, 0x8609);
                re_mdio_write(sc, 0x06, 0xe686);
                re_mdio_write(sc, 0x06, 0x08e0);
                re_mdio_write(sc, 0x06, 0x8609);
                re_mdio_write(sc, 0x06, 0xa000);
                re_mdio_write(sc, 0x06, 0x07ee);
                re_mdio_write(sc, 0x06, 0x860a);
                re_mdio_write(sc, 0x06, 0x03af);
                re_mdio_write(sc, 0x06, 0x8369);
                re_mdio_write(sc, 0x06, 0x0224);
                re_mdio_write(sc, 0x06, 0x8e02);
                re_mdio_write(sc, 0x06, 0x2426);
                re_mdio_write(sc, 0x06, 0xae48);
                re_mdio_write(sc, 0x06, 0xa003);
                re_mdio_write(sc, 0x06, 0x21e0);
                re_mdio_write(sc, 0x06, 0x8608);
                re_mdio_write(sc, 0x06, 0xe186);
                re_mdio_write(sc, 0x06, 0x091b);
                re_mdio_write(sc, 0x06, 0x019e);
                re_mdio_write(sc, 0x06, 0x0caa);
                re_mdio_write(sc, 0x06, 0x0502);
                re_mdio_write(sc, 0x06, 0x249d);
                re_mdio_write(sc, 0x06, 0xaee7);
                re_mdio_write(sc, 0x06, 0x0224);
                re_mdio_write(sc, 0x06, 0x8eae);
                re_mdio_write(sc, 0x06, 0xe2ee);
                re_mdio_write(sc, 0x06, 0x860a);
                re_mdio_write(sc, 0x06, 0x04ee);
                re_mdio_write(sc, 0x06, 0x860b);
                re_mdio_write(sc, 0x06, 0x00af);
                re_mdio_write(sc, 0x06, 0x8369);
                re_mdio_write(sc, 0x06, 0xa004);
                re_mdio_write(sc, 0x06, 0x15e0);
                re_mdio_write(sc, 0x06, 0x860b);
                re_mdio_write(sc, 0x06, 0xe18b);
                re_mdio_write(sc, 0x06, 0x341b);
                re_mdio_write(sc, 0x06, 0x109e);
                re_mdio_write(sc, 0x06, 0x05aa);
                re_mdio_write(sc, 0x06, 0x03af);
                re_mdio_write(sc, 0x06, 0x8383);
                re_mdio_write(sc, 0x06, 0xee86);
                re_mdio_write(sc, 0x06, 0x0a05);
                re_mdio_write(sc, 0x06, 0xae0c);
                re_mdio_write(sc, 0x06, 0xa005);
                re_mdio_write(sc, 0x06, 0x02ae);
                re_mdio_write(sc, 0x06, 0x0702);
                re_mdio_write(sc, 0x06, 0x2309);
                re_mdio_write(sc, 0x06, 0xee86);
                re_mdio_write(sc, 0x06, 0x0a00);
                re_mdio_write(sc, 0x06, 0xfeef);
                re_mdio_write(sc, 0x06, 0x96fe);
                re_mdio_write(sc, 0x06, 0xfdfc);
                re_mdio_write(sc, 0x06, 0x04f8);
                re_mdio_write(sc, 0x06, 0xf9fa);
                re_mdio_write(sc, 0x06, 0xef69);
                re_mdio_write(sc, 0x06, 0xfbe0);
                re_mdio_write(sc, 0x06, 0x8b85);
                re_mdio_write(sc, 0x06, 0xad25);
                re_mdio_write(sc, 0x06, 0x22e0);
                re_mdio_write(sc, 0x06, 0xe022);
                re_mdio_write(sc, 0x06, 0xe1e0);
                re_mdio_write(sc, 0x06, 0x23e2);
                re_mdio_write(sc, 0x06, 0xe036);
                re_mdio_write(sc, 0x06, 0xe3e0);
                re_mdio_write(sc, 0x06, 0x375a);
                re_mdio_write(sc, 0x06, 0xc40d);
                re_mdio_write(sc, 0x06, 0x0158);
                re_mdio_write(sc, 0x06, 0x021e);
                re_mdio_write(sc, 0x06, 0x20e3);
                re_mdio_write(sc, 0x06, 0x8ae7);
                re_mdio_write(sc, 0x06, 0xac31);
                re_mdio_write(sc, 0x06, 0x60ac);
                re_mdio_write(sc, 0x06, 0x3a08);
                re_mdio_write(sc, 0x06, 0xac3e);
                re_mdio_write(sc, 0x06, 0x26ae);
                re_mdio_write(sc, 0x06, 0x67af);
                re_mdio_write(sc, 0x06, 0x84db);
                re_mdio_write(sc, 0x06, 0xad37);
                re_mdio_write(sc, 0x06, 0x61e0);
                re_mdio_write(sc, 0x06, 0x8ae8);
                re_mdio_write(sc, 0x06, 0x10e4);
                re_mdio_write(sc, 0x06, 0x8ae8);
                re_mdio_write(sc, 0x06, 0xe18a);
                re_mdio_write(sc, 0x06, 0xe91b);
                re_mdio_write(sc, 0x06, 0x109e);
                re_mdio_write(sc, 0x06, 0x02ae);
                re_mdio_write(sc, 0x06, 0x51d1);
                re_mdio_write(sc, 0x06, 0x00bf);
                re_mdio_write(sc, 0x06, 0x863b);
                re_mdio_write(sc, 0x06, 0x022f);
                re_mdio_write(sc, 0x06, 0x50ee);
                re_mdio_write(sc, 0x06, 0x8ae8);
                re_mdio_write(sc, 0x06, 0x00ae);
                re_mdio_write(sc, 0x06, 0x43ad);
                re_mdio_write(sc, 0x06, 0x3627);
                re_mdio_write(sc, 0x06, 0xe08a);
                re_mdio_write(sc, 0x06, 0xeee1);
                re_mdio_write(sc, 0x06, 0x8aef);
                re_mdio_write(sc, 0x06, 0xef74);
                re_mdio_write(sc, 0x06, 0xe08a);
                re_mdio_write(sc, 0x06, 0xeae1);
                re_mdio_write(sc, 0x06, 0x8aeb);
                re_mdio_write(sc, 0x06, 0x1b74);
                re_mdio_write(sc, 0x06, 0x9e2e);
                re_mdio_write(sc, 0x06, 0x14e4);
                re_mdio_write(sc, 0x06, 0x8aea);
                re_mdio_write(sc, 0x06, 0xe58a);
                re_mdio_write(sc, 0x06, 0xebef);
                re_mdio_write(sc, 0x06, 0x74e0);
                re_mdio_write(sc, 0x06, 0x8aee);
                re_mdio_write(sc, 0x06, 0xe18a);
                re_mdio_write(sc, 0x06, 0xef1b);
                re_mdio_write(sc, 0x06, 0x479e);
                re_mdio_write(sc, 0x06, 0x0fae);
                re_mdio_write(sc, 0x06, 0x19ee);
                re_mdio_write(sc, 0x06, 0x8aea);
                re_mdio_write(sc, 0x06, 0x00ee);
                re_mdio_write(sc, 0x06, 0x8aeb);
                re_mdio_write(sc, 0x06, 0x00ae);
                re_mdio_write(sc, 0x06, 0x0fac);
                re_mdio_write(sc, 0x06, 0x390c);
                re_mdio_write(sc, 0x06, 0xd101);
                re_mdio_write(sc, 0x06, 0xbf86);
                re_mdio_write(sc, 0x06, 0x3b02);
                re_mdio_write(sc, 0x06, 0x2f50);
                re_mdio_write(sc, 0x06, 0xee8a);
                re_mdio_write(sc, 0x06, 0xe800);
                re_mdio_write(sc, 0x06, 0xe68a);
                re_mdio_write(sc, 0x06, 0xe7ff);
                re_mdio_write(sc, 0x06, 0xef96);
                re_mdio_write(sc, 0x06, 0xfefd);
                re_mdio_write(sc, 0x06, 0xfc04);
                re_mdio_write(sc, 0x06, 0xf8f9);
                re_mdio_write(sc, 0x06, 0xfaef);
                re_mdio_write(sc, 0x06, 0x69e0);
                re_mdio_write(sc, 0x06, 0xe022);
                re_mdio_write(sc, 0x06, 0xe1e0);
                re_mdio_write(sc, 0x06, 0x2358);
                re_mdio_write(sc, 0x06, 0xc4e1);
                re_mdio_write(sc, 0x06, 0x8b6e);
                re_mdio_write(sc, 0x06, 0x1f10);
                re_mdio_write(sc, 0x06, 0x9e24);
                re_mdio_write(sc, 0x06, 0xe48b);
                re_mdio_write(sc, 0x06, 0x6ead);
                re_mdio_write(sc, 0x06, 0x2218);
                re_mdio_write(sc, 0x06, 0xac27);
                re_mdio_write(sc, 0x06, 0x0dac);
                re_mdio_write(sc, 0x06, 0x2605);
                re_mdio_write(sc, 0x06, 0x0203);
                re_mdio_write(sc, 0x06, 0x8fae);
                re_mdio_write(sc, 0x06, 0x1302);
                re_mdio_write(sc, 0x06, 0x03c8);
                re_mdio_write(sc, 0x06, 0xae0e);
                re_mdio_write(sc, 0x06, 0x0203);
                re_mdio_write(sc, 0x06, 0xe102);
                re_mdio_write(sc, 0x06, 0x8520);
                re_mdio_write(sc, 0x06, 0xae06);
                re_mdio_write(sc, 0x06, 0x0203);
                re_mdio_write(sc, 0x06, 0x8f02);
                re_mdio_write(sc, 0x06, 0x8566);
                re_mdio_write(sc, 0x06, 0xef96);
                re_mdio_write(sc, 0x06, 0xfefd);
                re_mdio_write(sc, 0x06, 0xfc04);
                re_mdio_write(sc, 0x06, 0xf8fa);
                re_mdio_write(sc, 0x06, 0xef69);
                re_mdio_write(sc, 0x06, 0xe08b);
                re_mdio_write(sc, 0x06, 0x82ad);
                re_mdio_write(sc, 0x06, 0x2737);
                re_mdio_write(sc, 0x06, 0xbf86);
                re_mdio_write(sc, 0x06, 0x4402);
                re_mdio_write(sc, 0x06, 0x2f23);
                re_mdio_write(sc, 0x06, 0xac28);
                re_mdio_write(sc, 0x06, 0x2ed1);
                re_mdio_write(sc, 0x06, 0x01bf);
                re_mdio_write(sc, 0x06, 0x8647);
                re_mdio_write(sc, 0x06, 0x022f);
                re_mdio_write(sc, 0x06, 0x50bf);
                re_mdio_write(sc, 0x06, 0x8641);
                re_mdio_write(sc, 0x06, 0x022f);
                re_mdio_write(sc, 0x06, 0x23e5);
                re_mdio_write(sc, 0x06, 0x8af0);
                re_mdio_write(sc, 0x06, 0xe0e0);
                re_mdio_write(sc, 0x06, 0x22e1);
                re_mdio_write(sc, 0x06, 0xe023);
                re_mdio_write(sc, 0x06, 0xac2e);
                re_mdio_write(sc, 0x06, 0x04d1);
                re_mdio_write(sc, 0x06, 0x01ae);
                re_mdio_write(sc, 0x06, 0x02d1);
                re_mdio_write(sc, 0x06, 0x00bf);
                re_mdio_write(sc, 0x06, 0x8641);
                re_mdio_write(sc, 0x06, 0x022f);
                re_mdio_write(sc, 0x06, 0x50d1);
                re_mdio_write(sc, 0x06, 0x01bf);
                re_mdio_write(sc, 0x06, 0x8644);
                re_mdio_write(sc, 0x06, 0x022f);
                re_mdio_write(sc, 0x06, 0x50ef);
                re_mdio_write(sc, 0x06, 0x96fe);
                re_mdio_write(sc, 0x06, 0xfc04);
                re_mdio_write(sc, 0x06, 0xf8fa);
                re_mdio_write(sc, 0x06, 0xef69);
                re_mdio_write(sc, 0x06, 0xbf86);
                re_mdio_write(sc, 0x06, 0x4702);
                re_mdio_write(sc, 0x06, 0x2f23);
                re_mdio_write(sc, 0x06, 0xad28);
                re_mdio_write(sc, 0x06, 0x19d1);
                re_mdio_write(sc, 0x06, 0x00bf);
                re_mdio_write(sc, 0x06, 0x8644);
                re_mdio_write(sc, 0x06, 0x022f);
                re_mdio_write(sc, 0x06, 0x50e1);
                re_mdio_write(sc, 0x06, 0x8af0);
                re_mdio_write(sc, 0x06, 0xbf86);
                re_mdio_write(sc, 0x06, 0x4102);
                re_mdio_write(sc, 0x06, 0x2f50);
                re_mdio_write(sc, 0x06, 0xd100);
                re_mdio_write(sc, 0x06, 0xbf86);
                re_mdio_write(sc, 0x06, 0x4702);
                re_mdio_write(sc, 0x06, 0x2f50);
                re_mdio_write(sc, 0x06, 0xef96);
                re_mdio_write(sc, 0x06, 0xfefc);
                re_mdio_write(sc, 0x06, 0x04f8);
                re_mdio_write(sc, 0x06, 0xe0e2);
                re_mdio_write(sc, 0x06, 0xfee1);
                re_mdio_write(sc, 0x06, 0xe2ff);
                re_mdio_write(sc, 0x06, 0xad2e);
                re_mdio_write(sc, 0x06, 0x63e0);
                re_mdio_write(sc, 0x06, 0xe038);
                re_mdio_write(sc, 0x06, 0xe1e0);
                re_mdio_write(sc, 0x06, 0x39ad);
                re_mdio_write(sc, 0x06, 0x2f10);
                re_mdio_write(sc, 0x06, 0xe0e0);
                re_mdio_write(sc, 0x06, 0x34e1);
                re_mdio_write(sc, 0x06, 0xe035);
                re_mdio_write(sc, 0x06, 0xf726);
                re_mdio_write(sc, 0x06, 0xe4e0);
                re_mdio_write(sc, 0x06, 0x34e5);
                re_mdio_write(sc, 0x06, 0xe035);
                re_mdio_write(sc, 0x06, 0xae0e);
                re_mdio_write(sc, 0x06, 0xe0e2);
                re_mdio_write(sc, 0x06, 0xd6e1);
                re_mdio_write(sc, 0x06, 0xe2d7);
                re_mdio_write(sc, 0x06, 0xf728);
                re_mdio_write(sc, 0x06, 0xe4e2);
                re_mdio_write(sc, 0x06, 0xd6e5);
                re_mdio_write(sc, 0x06, 0xe2d7);
                re_mdio_write(sc, 0x06, 0xe0e2);
                re_mdio_write(sc, 0x06, 0x34e1);
                re_mdio_write(sc, 0x06, 0xe235);
                re_mdio_write(sc, 0x06, 0xf72b);
                re_mdio_write(sc, 0x06, 0xe4e2);
                re_mdio_write(sc, 0x06, 0x34e5);
                re_mdio_write(sc, 0x06, 0xe235);
                re_mdio_write(sc, 0x06, 0xd07d);
                re_mdio_write(sc, 0x06, 0xb0fe);
                re_mdio_write(sc, 0x06, 0xe0e2);
                re_mdio_write(sc, 0x06, 0x34e1);
                re_mdio_write(sc, 0x06, 0xe235);
                re_mdio_write(sc, 0x06, 0xf62b);
                re_mdio_write(sc, 0x06, 0xe4e2);
                re_mdio_write(sc, 0x06, 0x34e5);
                re_mdio_write(sc, 0x06, 0xe235);
                re_mdio_write(sc, 0x06, 0xe0e0);
                re_mdio_write(sc, 0x06, 0x34e1);
                re_mdio_write(sc, 0x06, 0xe035);
                re_mdio_write(sc, 0x06, 0xf626);
                re_mdio_write(sc, 0x06, 0xe4e0);
                re_mdio_write(sc, 0x06, 0x34e5);
                re_mdio_write(sc, 0x06, 0xe035);
                re_mdio_write(sc, 0x06, 0xe0e2);
                re_mdio_write(sc, 0x06, 0xd6e1);
                re_mdio_write(sc, 0x06, 0xe2d7);
                re_mdio_write(sc, 0x06, 0xf628);
                re_mdio_write(sc, 0x06, 0xe4e2);
                re_mdio_write(sc, 0x06, 0xd6e5);
                re_mdio_write(sc, 0x06, 0xe2d7);
                re_mdio_write(sc, 0x06, 0xfc04);
                re_mdio_write(sc, 0x06, 0xae20);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0x0000);
                re_mdio_write(sc, 0x06, 0xa725);
                re_mdio_write(sc, 0x06, 0xe50a);
                re_mdio_write(sc, 0x06, 0x1de5);
                re_mdio_write(sc, 0x06, 0x0a2c);
                re_mdio_write(sc, 0x06, 0xe50a);
                re_mdio_write(sc, 0x06, 0x6de5);
                re_mdio_write(sc, 0x06, 0x0a1d);
                re_mdio_write(sc, 0x06, 0xe50a);
                re_mdio_write(sc, 0x06, 0x1ce5);
                re_mdio_write(sc, 0x06, 0x0a2d);
                re_mdio_write(sc, 0x06, 0xa755);
                re_mdio_write(sc, 0x06, 0x00e2);
                re_mdio_write(sc, 0x06, 0x3488);
                re_mdio_write(sc, 0x06, 0xe200);
                re_mdio_write(sc, 0x06, 0xcce2);
                re_mdio_write(sc, 0x06, 0x0055);
                re_mdio_write(sc, 0x06, 0xe020);
                re_mdio_write(sc, 0x06, 0x55e2);
                re_mdio_write(sc, 0x06, 0xd600);
                re_mdio_write(sc, 0x06, 0xe24a);
                PhyRegValue = re_mdio_read(sc, 0x01);
                PhyRegValue |= BIT_0;
                re_mdio_write(sc, 0x01, PhyRegValue);
                PhyRegValue = re_mdio_read(sc, 0x00);
                PhyRegValue |= BIT_0;
                re_mdio_write(sc, 0x00, PhyRegValue);
                re_mdio_write(sc, 0x1f, 0x0000);

                re_mdio_write(sc, 0x1f, 0x0005);
                for (i = 0; i < 200; i++) {
                        DELAY(100);
                        PhyRegValue = re_mdio_read(sc, 0x00);
                        if (PhyRegValue & BIT_7)
                                break;
                }
                re_mdio_write(sc, 0x1f, 0x0007);
                re_mdio_write(sc, 0x1e, 0x0023);
                PhyRegValue = re_mdio_read(sc, 0x17);
                PhyRegValue &= ~(BIT_0);
                if (sc->RequiredSecLanDonglePatch)
                        PhyRegValue &= ~(BIT_2);
                re_mdio_write(sc, 0x17, PhyRegValue);
                re_mdio_write(sc, 0x1f, 0x0000);
        }
}

static void re_set_phy_mcu_8168evl_1(struct re_softc *sc)
{
        u_int16_t PhyRegValue;
        int i;

        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x00, 0x1800);
        PhyRegValue= re_mdio_read(sc, 0x15);
        PhyRegValue &= ~BIT_12;
        re_mdio_write(sc, 0x15, PhyRegValue);
        DELAY(200);
        DELAY(200);
        re_mdio_write(sc, 0x1f, 0x0004);
        re_mdio_write(sc, 0x1f, 0x0007);
        re_mdio_write(sc, 0x1e, 0x0023);
        PhyRegValue = re_mdio_read(sc, 0x17);
        if ((PhyRegValue & BIT_11) == 0x0000) {
                PhyRegValue |= BIT_0;
                re_mdio_write(sc, 0x17, PhyRegValue);
                for (i = 0; i < 200; i++) {
                        DELAY(100);
                        PhyRegValue = re_mdio_read(sc, 0x17);
                        if (PhyRegValue & BIT_11)
                                break;
                }
        }
        PhyRegValue = re_mdio_read(sc, 0x17);
        PhyRegValue |= BIT_11;
        re_mdio_write(sc, 0x17,PhyRegValue);
        re_mdio_write(sc, 0x1f, 0x0004);
        re_mdio_write(sc, 0x1f, 0x0007);
        re_mdio_write(sc, 0x1E, 0x002C);
        re_mdio_write(sc, 0x1B, 0x5000);
        re_mdio_write(sc, 0x1E, 0x002d);
        re_mdio_write(sc, 0x19, 0x0004);
        re_mdio_write(sc, 0x1f, 0x0002);
        re_mdio_write(sc, 0x1f, 0x0000);
        for (i = 0; i < 200; i++) {
                DELAY(100);
                PhyRegValue= re_mdio_read(sc, 0x1E);
                if ((PhyRegValue& 0x03FF) == 0x0014)
                        break;
        }
        re_mdio_write(sc, 0x1f, 0x0005);
        for (i = 0; i < 200; i++) {
                DELAY(100);
                PhyRegValue= re_mdio_read(sc, 0x07);
                if ((PhyRegValue& BIT_5) == 0)
                        break;
        }
        PhyRegValue = re_mdio_read(sc, 0x07);
        if (PhyRegValue & BIT_5) {
                re_mdio_write(sc, 0x1f, 0x0004);
                re_mdio_write(sc, 0x1f, 0x0007);
                re_mdio_write(sc, 0x1e, 0x00a1);
                re_mdio_write(sc, 0x17, 0x1000);
                re_mdio_write(sc, 0x17, 0x0000);
                re_mdio_write(sc, 0x17, 0x2000);
                re_mdio_write(sc, 0x1e, 0x002f);
                re_mdio_write(sc, 0x18, 0x9bfb);
                re_mdio_write(sc, 0x1f, 0x0005);
                re_mdio_write(sc, 0x07, 0x0000);
                re_mdio_write(sc, 0x1f, 0x0002);
                re_mdio_write(sc, 0x1f, 0x0000);
        }
        re_mdio_write(sc, 0x1f, 0x0005);
        re_mdio_write(sc, 0x05, 0xfff6);
        re_mdio_write(sc, 0x06, 0x0080);
        PhyRegValue = re_mdio_read(sc, 0x00);
        PhyRegValue &= ~BIT_7;
        re_mdio_write(sc, 0x00, PhyRegValue);
        re_mdio_write(sc, 0x1f, 0x0004);
        re_mdio_write(sc, 0x1f, 0x0007);
        re_mdio_write(sc, 0x1e, 0x0023);
        re_mdio_write(sc, 0x16, 0x0306);
        re_mdio_write(sc, 0x16, 0x0307);
        re_mdio_write(sc, 0x15, 0x0000);
        re_mdio_write(sc, 0x19, 0x407d);
        re_mdio_write(sc, 0x15, 0x0001);
        re_mdio_write(sc, 0x19, 0x440f);
        re_mdio_write(sc, 0x15, 0x0002);
        re_mdio_write(sc, 0x19, 0x7c03);
        re_mdio_write(sc, 0x15, 0x0003);
        re_mdio_write(sc, 0x19, 0x6c03);
        re_mdio_write(sc, 0x15, 0x0004);
        re_mdio_write(sc, 0x19, 0xc4d5);
        re_mdio_write(sc, 0x15, 0x0005);
        re_mdio_write(sc, 0x19, 0x00ff);
        re_mdio_write(sc, 0x15, 0x0006);
        re_mdio_write(sc, 0x19, 0x74f0);
        re_mdio_write(sc, 0x15, 0x0007);
        re_mdio_write(sc, 0x19, 0x4880);
        re_mdio_write(sc, 0x15, 0x0008);
        re_mdio_write(sc, 0x19, 0x4c00);
        re_mdio_write(sc, 0x15, 0x0009);
        re_mdio_write(sc, 0x19, 0x4800);
        re_mdio_write(sc, 0x15, 0x000a);
        re_mdio_write(sc, 0x19, 0x5000);
        re_mdio_write(sc, 0x15, 0x000b);
        re_mdio_write(sc, 0x19, 0x4400);
        re_mdio_write(sc, 0x15, 0x000c);
        re_mdio_write(sc, 0x19, 0x7801);
        re_mdio_write(sc, 0x15, 0x000d);
        re_mdio_write(sc, 0x19, 0x4000);
        re_mdio_write(sc, 0x15, 0x000e);
        re_mdio_write(sc, 0x19, 0x7800);
        re_mdio_write(sc, 0x15, 0x000f);
        re_mdio_write(sc, 0x19, 0x7010);
        re_mdio_write(sc, 0x15, 0x0010);
        re_mdio_write(sc, 0x19, 0x6804);
        re_mdio_write(sc, 0x15, 0x0011);
        re_mdio_write(sc, 0x19, 0x64a0);
        re_mdio_write(sc, 0x15, 0x0012);
        re_mdio_write(sc, 0x19, 0x63da);
        re_mdio_write(sc, 0x15, 0x0013);
        re_mdio_write(sc, 0x19, 0x63d8);
        re_mdio_write(sc, 0x15, 0x0014);
        re_mdio_write(sc, 0x19, 0x6f05);
        re_mdio_write(sc, 0x15, 0x0015);
        re_mdio_write(sc, 0x19, 0x5420);
        re_mdio_write(sc, 0x15, 0x0016);
        re_mdio_write(sc, 0x19, 0x58ce);
        re_mdio_write(sc, 0x15, 0x0017);
        re_mdio_write(sc, 0x19, 0x5cf3);
        re_mdio_write(sc, 0x15, 0x0018);
        re_mdio_write(sc, 0x19, 0xb600);
        re_mdio_write(sc, 0x15, 0x0019);
        re_mdio_write(sc, 0x19, 0xc659);
        re_mdio_write(sc, 0x15, 0x001a);
        re_mdio_write(sc, 0x19, 0x0018);
        re_mdio_write(sc, 0x15, 0x001b);
        re_mdio_write(sc, 0x19, 0xc403);
        re_mdio_write(sc, 0x15, 0x001c);
        re_mdio_write(sc, 0x19, 0x0016);
        re_mdio_write(sc, 0x15, 0x001d);
        re_mdio_write(sc, 0x19, 0xaa05);
        re_mdio_write(sc, 0x15, 0x001e);
        re_mdio_write(sc, 0x19, 0xc503);
        re_mdio_write(sc, 0x15, 0x001f);
        re_mdio_write(sc, 0x19, 0x0003);
        re_mdio_write(sc, 0x15, 0x0020);
        re_mdio_write(sc, 0x19, 0x89f8);
        re_mdio_write(sc, 0x15, 0x0021);
        re_mdio_write(sc, 0x19, 0x32ae);
        re_mdio_write(sc, 0x15, 0x0022);
        re_mdio_write(sc, 0x19, 0x7c03);
        re_mdio_write(sc, 0x15, 0x0023);
        re_mdio_write(sc, 0x19, 0x6c03);
        re_mdio_write(sc, 0x15, 0x0024);
        re_mdio_write(sc, 0x19, 0x7c03);
        re_mdio_write(sc, 0x15, 0x0025);
        re_mdio_write(sc, 0x19, 0x6801);
        re_mdio_write(sc, 0x15, 0x0026);
        re_mdio_write(sc, 0x19, 0x66a0);
        re_mdio_write(sc, 0x15, 0x0027);
        re_mdio_write(sc, 0x19, 0xa300);
        re_mdio_write(sc, 0x15, 0x0028);
        re_mdio_write(sc, 0x19, 0x64a0);
        re_mdio_write(sc, 0x15, 0x0029);
        re_mdio_write(sc, 0x19, 0x76f0);
        re_mdio_write(sc, 0x15, 0x002a);
        re_mdio_write(sc, 0x19, 0x7670);
        re_mdio_write(sc, 0x15, 0x002b);
        re_mdio_write(sc, 0x19, 0x7630);
        re_mdio_write(sc, 0x15, 0x002c);
        re_mdio_write(sc, 0x19, 0x31a6);
        re_mdio_write(sc, 0x15, 0x002d);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x002e);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x002f);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0030);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0031);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0032);
        re_mdio_write(sc, 0x19, 0x4801);
        re_mdio_write(sc, 0x15, 0x0033);
        re_mdio_write(sc, 0x19, 0x6803);
        re_mdio_write(sc, 0x15, 0x0034);
        re_mdio_write(sc, 0x19, 0x66a1);
        re_mdio_write(sc, 0x15, 0x0035);
        re_mdio_write(sc, 0x19, 0x7c03);
        re_mdio_write(sc, 0x15, 0x0036);
        re_mdio_write(sc, 0x19, 0x6c03);
        re_mdio_write(sc, 0x15, 0x0037);
        re_mdio_write(sc, 0x19, 0xa300);
        re_mdio_write(sc, 0x15, 0x0038);
        re_mdio_write(sc, 0x19, 0x64a1);
        re_mdio_write(sc, 0x15, 0x0039);
        re_mdio_write(sc, 0x19, 0x7c08);
        re_mdio_write(sc, 0x15, 0x003a);
        re_mdio_write(sc, 0x19, 0x74f8);
        re_mdio_write(sc, 0x15, 0x003b);
        re_mdio_write(sc, 0x19, 0x63d0);
        re_mdio_write(sc, 0x15, 0x003c);
        re_mdio_write(sc, 0x19, 0x7ff0);
        re_mdio_write(sc, 0x15, 0x003d);
        re_mdio_write(sc, 0x19, 0x77f0);
        re_mdio_write(sc, 0x15, 0x003e);
        re_mdio_write(sc, 0x19, 0x7ff0);
        re_mdio_write(sc, 0x15, 0x003f);
        re_mdio_write(sc, 0x19, 0x7750);
        re_mdio_write(sc, 0x15, 0x0040);
        re_mdio_write(sc, 0x19, 0x63d8);
        re_mdio_write(sc, 0x15, 0x0041);
        re_mdio_write(sc, 0x19, 0x7cf0);
        re_mdio_write(sc, 0x15, 0x0042);
        re_mdio_write(sc, 0x19, 0x7708);
        re_mdio_write(sc, 0x15, 0x0043);
        re_mdio_write(sc, 0x19, 0xa654);
        re_mdio_write(sc, 0x15, 0x0044);
        re_mdio_write(sc, 0x19, 0x304a);
        re_mdio_write(sc, 0x15, 0x0045);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0046);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0047);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0048);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0049);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x004a);
        re_mdio_write(sc, 0x19, 0x4802);
        re_mdio_write(sc, 0x15, 0x004b);
        re_mdio_write(sc, 0x19, 0x4003);
        re_mdio_write(sc, 0x15, 0x004c);
        re_mdio_write(sc, 0x19, 0x4440);
        re_mdio_write(sc, 0x15, 0x004d);
        re_mdio_write(sc, 0x19, 0x63c8);
        re_mdio_write(sc, 0x15, 0x004e);
        re_mdio_write(sc, 0x19, 0x6481);
        re_mdio_write(sc, 0x15, 0x004f);
        re_mdio_write(sc, 0x19, 0x9d00);
        re_mdio_write(sc, 0x15, 0x0050);
        re_mdio_write(sc, 0x19, 0x63e8);
        re_mdio_write(sc, 0x15, 0x0051);
        re_mdio_write(sc, 0x19, 0x7d00);
        re_mdio_write(sc, 0x15, 0x0052);
        re_mdio_write(sc, 0x19, 0x5900);
        re_mdio_write(sc, 0x15, 0x0053);
        re_mdio_write(sc, 0x19, 0x63f8);
        re_mdio_write(sc, 0x15, 0x0054);
        re_mdio_write(sc, 0x19, 0x64a1);
        re_mdio_write(sc, 0x15, 0x0055);
        re_mdio_write(sc, 0x19, 0x3116);
        re_mdio_write(sc, 0x15, 0x0056);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0057);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0058);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0059);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x005a);
        re_mdio_write(sc, 0x19, 0x7c03);
        re_mdio_write(sc, 0x15, 0x005b);
        re_mdio_write(sc, 0x19, 0x6c03);
        re_mdio_write(sc, 0x15, 0x005c);
        re_mdio_write(sc, 0x19, 0x7c08);
        re_mdio_write(sc, 0x15, 0x005d);
        re_mdio_write(sc, 0x19, 0x6000);
        re_mdio_write(sc, 0x15, 0x005e);
        re_mdio_write(sc, 0x19, 0x59ce);
        re_mdio_write(sc, 0x15, 0x005f);
        re_mdio_write(sc, 0x19, 0x4400);
        re_mdio_write(sc, 0x15, 0x0060);
        re_mdio_write(sc, 0x19, 0x7d00);
        re_mdio_write(sc, 0x15, 0x0061);
        re_mdio_write(sc, 0x19, 0x72b0);
        re_mdio_write(sc, 0x15, 0x0062);
        re_mdio_write(sc, 0x19, 0x400e);
        re_mdio_write(sc, 0x15, 0x0063);
        re_mdio_write(sc, 0x19, 0x4440);
        re_mdio_write(sc, 0x15, 0x0064);
        re_mdio_write(sc, 0x19, 0x9d00);
        re_mdio_write(sc, 0x15, 0x0065);
        re_mdio_write(sc, 0x19, 0x7f00);
        re_mdio_write(sc, 0x15, 0x0066);
        re_mdio_write(sc, 0x19, 0x70b0);
        re_mdio_write(sc, 0x15, 0x0067);
        re_mdio_write(sc, 0x19, 0x7c08);
        re_mdio_write(sc, 0x15, 0x0068);
        re_mdio_write(sc, 0x19, 0x6008);
        re_mdio_write(sc, 0x15, 0x0069);
        re_mdio_write(sc, 0x19, 0x7cf0);
        re_mdio_write(sc, 0x15, 0x006a);
        re_mdio_write(sc, 0x19, 0x7750);
        re_mdio_write(sc, 0x15, 0x006b);
        re_mdio_write(sc, 0x19, 0x4007);
        re_mdio_write(sc, 0x15, 0x006c);
        re_mdio_write(sc, 0x19, 0x4500);
        re_mdio_write(sc, 0x15, 0x006d);
        re_mdio_write(sc, 0x19, 0x4023);
        re_mdio_write(sc, 0x15, 0x006e);
        re_mdio_write(sc, 0x19, 0x4580);
        re_mdio_write(sc, 0x15, 0x006f);
        re_mdio_write(sc, 0x19, 0x9f00);
        re_mdio_write(sc, 0x15, 0x0070);
        re_mdio_write(sc, 0x19, 0xcd78);
        re_mdio_write(sc, 0x15, 0x0071);
        re_mdio_write(sc, 0x19, 0x0003);
        re_mdio_write(sc, 0x15, 0x0072);
        re_mdio_write(sc, 0x19, 0xbe02);
        re_mdio_write(sc, 0x15, 0x0073);
        re_mdio_write(sc, 0x19, 0x3070);
        re_mdio_write(sc, 0x15, 0x0074);
        re_mdio_write(sc, 0x19, 0x7cf0);
        re_mdio_write(sc, 0x15, 0x0075);
        re_mdio_write(sc, 0x19, 0x77f0);
        re_mdio_write(sc, 0x15, 0x0076);
        re_mdio_write(sc, 0x19, 0x4400);
        re_mdio_write(sc, 0x15, 0x0077);
        re_mdio_write(sc, 0x19, 0x4007);
        re_mdio_write(sc, 0x15, 0x0078);
        re_mdio_write(sc, 0x19, 0x4500);
        re_mdio_write(sc, 0x15, 0x0079);
        re_mdio_write(sc, 0x19, 0x4023);
        re_mdio_write(sc, 0x15, 0x007a);
        re_mdio_write(sc, 0x19, 0x4580);
        re_mdio_write(sc, 0x15, 0x007b);
        re_mdio_write(sc, 0x19, 0x9f00);
        re_mdio_write(sc, 0x15, 0x007c);
        re_mdio_write(sc, 0x19, 0xce80);
        re_mdio_write(sc, 0x15, 0x007d);
        re_mdio_write(sc, 0x19, 0x0004);
        re_mdio_write(sc, 0x15, 0x007e);
        re_mdio_write(sc, 0x19, 0xce80);
        re_mdio_write(sc, 0x15, 0x007f);
        re_mdio_write(sc, 0x19, 0x0002);
        re_mdio_write(sc, 0x15, 0x0080);
        re_mdio_write(sc, 0x19, 0x307c);
        re_mdio_write(sc, 0x15, 0x0081);
        re_mdio_write(sc, 0x19, 0x4400);
        re_mdio_write(sc, 0x15, 0x0082);
        re_mdio_write(sc, 0x19, 0x480f);
        re_mdio_write(sc, 0x15, 0x0083);
        re_mdio_write(sc, 0x19, 0x6802);
        re_mdio_write(sc, 0x15, 0x0084);
        re_mdio_write(sc, 0x19, 0x6680);
        re_mdio_write(sc, 0x15, 0x0085);
        re_mdio_write(sc, 0x19, 0x7c10);
        re_mdio_write(sc, 0x15, 0x0086);
        re_mdio_write(sc, 0x19, 0x6010);
        re_mdio_write(sc, 0x15, 0x0087);
        re_mdio_write(sc, 0x19, 0x400a);
        re_mdio_write(sc, 0x15, 0x0088);
        re_mdio_write(sc, 0x19, 0x4580);
        re_mdio_write(sc, 0x15, 0x0089);
        re_mdio_write(sc, 0x19, 0x9e00);
        re_mdio_write(sc, 0x15, 0x008a);
        re_mdio_write(sc, 0x19, 0x7d00);
        re_mdio_write(sc, 0x15, 0x008b);
        re_mdio_write(sc, 0x19, 0x5800);
        re_mdio_write(sc, 0x15, 0x008c);
        re_mdio_write(sc, 0x19, 0x63c8);
        re_mdio_write(sc, 0x15, 0x008d);
        re_mdio_write(sc, 0x19, 0x63d8);
        re_mdio_write(sc, 0x15, 0x008e);
        re_mdio_write(sc, 0x19, 0x66a0);
        re_mdio_write(sc, 0x15, 0x008f);
        re_mdio_write(sc, 0x19, 0x8300);
        re_mdio_write(sc, 0x15, 0x0090);
        re_mdio_write(sc, 0x19, 0x7ff0);
        re_mdio_write(sc, 0x15, 0x0091);
        re_mdio_write(sc, 0x19, 0x74f0);
        re_mdio_write(sc, 0x15, 0x0092);
        re_mdio_write(sc, 0x19, 0x3006);
        re_mdio_write(sc, 0x15, 0x0093);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0094);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0095);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0096);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0097);
        re_mdio_write(sc, 0x19, 0x4803);
        re_mdio_write(sc, 0x15, 0x0098);
        re_mdio_write(sc, 0x19, 0x7c03);
        re_mdio_write(sc, 0x15, 0x0099);
        re_mdio_write(sc, 0x19, 0x6c03);
        re_mdio_write(sc, 0x15, 0x009a);
        re_mdio_write(sc, 0x19, 0xa203);
        re_mdio_write(sc, 0x15, 0x009b);
        re_mdio_write(sc, 0x19, 0x64b1);
        re_mdio_write(sc, 0x15, 0x009c);
        re_mdio_write(sc, 0x19, 0x309e);
        re_mdio_write(sc, 0x15, 0x009d);
        re_mdio_write(sc, 0x19, 0x64b3);
        re_mdio_write(sc, 0x15, 0x009e);
        re_mdio_write(sc, 0x19, 0x4030);
        re_mdio_write(sc, 0x15, 0x009f);
        re_mdio_write(sc, 0x19, 0x440e);
        re_mdio_write(sc, 0x15, 0x00a0);
        re_mdio_write(sc, 0x19, 0x4020);
        re_mdio_write(sc, 0x15, 0x00a1);
        re_mdio_write(sc, 0x19, 0x4419);
        re_mdio_write(sc, 0x15, 0x00a2);
        re_mdio_write(sc, 0x19, 0x7801);
        re_mdio_write(sc, 0x15, 0x00a3);
        re_mdio_write(sc, 0x19, 0xc520);
        re_mdio_write(sc, 0x15, 0x00a4);
        re_mdio_write(sc, 0x19, 0x000b);
        re_mdio_write(sc, 0x15, 0x00a5);
        re_mdio_write(sc, 0x19, 0x4020);
        re_mdio_write(sc, 0x15, 0x00a6);
        re_mdio_write(sc, 0x19, 0x7800);
        re_mdio_write(sc, 0x15, 0x00a7);
        re_mdio_write(sc, 0x19, 0x58a4);
        re_mdio_write(sc, 0x15, 0x00a8);
        re_mdio_write(sc, 0x19, 0x63da);
        re_mdio_write(sc, 0x15, 0x00a9);
        re_mdio_write(sc, 0x19, 0x5cb0);
        re_mdio_write(sc, 0x15, 0x00aa);
        re_mdio_write(sc, 0x19, 0x7d00);
        re_mdio_write(sc, 0x15, 0x00ab);
        re_mdio_write(sc, 0x19, 0x72b0);
        re_mdio_write(sc, 0x15, 0x00ac);
        re_mdio_write(sc, 0x19, 0x7f00);
        re_mdio_write(sc, 0x15, 0x00ad);
        re_mdio_write(sc, 0x19, 0x70b0);
        re_mdio_write(sc, 0x15, 0x00ae);
        re_mdio_write(sc, 0x19, 0x30b8);
        re_mdio_write(sc, 0x15, 0x00AF);
        re_mdio_write(sc, 0x19, 0x4060);
        re_mdio_write(sc, 0x15, 0x00B0);
        re_mdio_write(sc, 0x19, 0x7800);
        re_mdio_write(sc, 0x15, 0x00B1);
        re_mdio_write(sc, 0x19, 0x7e00);
        re_mdio_write(sc, 0x15, 0x00B2);
        re_mdio_write(sc, 0x19, 0x72B0);
        re_mdio_write(sc, 0x15, 0x00B3);
        re_mdio_write(sc, 0x19, 0x7F00);
        re_mdio_write(sc, 0x15, 0x00B4);
        re_mdio_write(sc, 0x19, 0x73B0);
        re_mdio_write(sc, 0x15, 0x00b5);
        re_mdio_write(sc, 0x19, 0x58a0);
        re_mdio_write(sc, 0x15, 0x00b6);
        re_mdio_write(sc, 0x19, 0x63d2);
        re_mdio_write(sc, 0x15, 0x00b7);
        re_mdio_write(sc, 0x19, 0x5c00);
        re_mdio_write(sc, 0x15, 0x00b8);
        re_mdio_write(sc, 0x19, 0x5780);
        re_mdio_write(sc, 0x15, 0x00b9);
        re_mdio_write(sc, 0x19, 0xb60d);
        re_mdio_write(sc, 0x15, 0x00ba);
        re_mdio_write(sc, 0x19, 0x9bff);
        re_mdio_write(sc, 0x15, 0x00bb);
        re_mdio_write(sc, 0x19, 0x7c03);
        re_mdio_write(sc, 0x15, 0x00bc);
        re_mdio_write(sc, 0x19, 0x6001);
        re_mdio_write(sc, 0x15, 0x00bd);
        re_mdio_write(sc, 0x19, 0xc020);
        re_mdio_write(sc, 0x15, 0x00be);
        re_mdio_write(sc, 0x19, 0x002b);
        re_mdio_write(sc, 0x15, 0x00bf);
        re_mdio_write(sc, 0x19, 0xc137);
        re_mdio_write(sc, 0x15, 0x00c0);
        re_mdio_write(sc, 0x19, 0x0006);
        re_mdio_write(sc, 0x15, 0x00c1);
        re_mdio_write(sc, 0x19, 0x9af8);
        re_mdio_write(sc, 0x15, 0x00c2);
        re_mdio_write(sc, 0x19, 0x30c6);
        re_mdio_write(sc, 0x15, 0x00c3);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x00c4);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x00c5);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x00c6);
        re_mdio_write(sc, 0x19, 0x7d00);
        re_mdio_write(sc, 0x15, 0x00c7);
        re_mdio_write(sc, 0x19, 0x70b0);
        re_mdio_write(sc, 0x15, 0x00c8);
        re_mdio_write(sc, 0x19, 0x4400);
        re_mdio_write(sc, 0x15, 0x00c9);
        re_mdio_write(sc, 0x19, 0x4804);
        re_mdio_write(sc, 0x15, 0x00ca);
        re_mdio_write(sc, 0x19, 0x7c80);
        re_mdio_write(sc, 0x15, 0x00cb);
        re_mdio_write(sc, 0x19, 0x5c80);
        re_mdio_write(sc, 0x15, 0x00cc);
        re_mdio_write(sc, 0x19, 0x4010);
        re_mdio_write(sc, 0x15, 0x00cd);
        re_mdio_write(sc, 0x19, 0x4415);
        re_mdio_write(sc, 0x15, 0x00ce);
        re_mdio_write(sc, 0x19, 0x9b00);
        re_mdio_write(sc, 0x15, 0x00cf);
        re_mdio_write(sc, 0x19, 0x7f00);
        re_mdio_write(sc, 0x15, 0x00d0);
        re_mdio_write(sc, 0x19, 0x70b0);
        re_mdio_write(sc, 0x15, 0x00d1);
        re_mdio_write(sc, 0x19, 0x3177);
        re_mdio_write(sc, 0x15, 0x00d2);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x00d3);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x00d4);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x00d5);
        re_mdio_write(sc, 0x19, 0x4808);
        re_mdio_write(sc, 0x15, 0x00d6);
        re_mdio_write(sc, 0x19, 0x4007);
        re_mdio_write(sc, 0x15, 0x00d7);
        re_mdio_write(sc, 0x19, 0x4420);
        re_mdio_write(sc, 0x15, 0x00d8);
        re_mdio_write(sc, 0x19, 0x63d8);
        re_mdio_write(sc, 0x15, 0x00d9);
        re_mdio_write(sc, 0x19, 0xb608);
        re_mdio_write(sc, 0x15, 0x00da);
        re_mdio_write(sc, 0x19, 0xbcbd);
        re_mdio_write(sc, 0x15, 0x00db);
        re_mdio_write(sc, 0x19, 0xc60b);
        re_mdio_write(sc, 0x15, 0x00dc);
        re_mdio_write(sc, 0x19, 0x00fd);
        re_mdio_write(sc, 0x15, 0x00dd);
        re_mdio_write(sc, 0x19, 0x30e1);
        re_mdio_write(sc, 0x15, 0x00de);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x00df);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x00e0);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x00e1);
        re_mdio_write(sc, 0x19, 0x4809);
        re_mdio_write(sc, 0x15, 0x00e2);
        re_mdio_write(sc, 0x19, 0x7e40);
        re_mdio_write(sc, 0x15, 0x00e3);
        re_mdio_write(sc, 0x19, 0x5a40);
        re_mdio_write(sc, 0x15, 0x00e4);
        re_mdio_write(sc, 0x19, 0x305a);
        re_mdio_write(sc, 0x15, 0x00e5);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x00e6);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x00e7);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x00e8);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x00e9);
        re_mdio_write(sc, 0x19, 0x480a);
        re_mdio_write(sc, 0x15, 0x00ea);
        re_mdio_write(sc, 0x19, 0x5820);
        re_mdio_write(sc, 0x15, 0x00eb);
        re_mdio_write(sc, 0x19, 0x6c03);
        re_mdio_write(sc, 0x15, 0x00ec);
        re_mdio_write(sc, 0x19, 0xb60a);
        re_mdio_write(sc, 0x15, 0x00ed);
        re_mdio_write(sc, 0x19, 0xda07);
        re_mdio_write(sc, 0x15, 0x00ee);
        re_mdio_write(sc, 0x19, 0x0008);
        re_mdio_write(sc, 0x15, 0x00ef);
        re_mdio_write(sc, 0x19, 0xc60b);
        re_mdio_write(sc, 0x15, 0x00f0);
        re_mdio_write(sc, 0x19, 0x00fc);
        re_mdio_write(sc, 0x15, 0x00f1);
        re_mdio_write(sc, 0x19, 0x30f6);
        re_mdio_write(sc, 0x15, 0x00f2);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x00f3);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x00f4);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x00f5);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x00f6);
        re_mdio_write(sc, 0x19, 0x4408);
        re_mdio_write(sc, 0x15, 0x00f7);
        re_mdio_write(sc, 0x19, 0x480b);
        re_mdio_write(sc, 0x15, 0x00f8);
        re_mdio_write(sc, 0x19, 0x6f03);
        re_mdio_write(sc, 0x15, 0x00f9);
        re_mdio_write(sc, 0x19, 0x405f);
        re_mdio_write(sc, 0x15, 0x00fa);
        re_mdio_write(sc, 0x19, 0x4448);
        re_mdio_write(sc, 0x15, 0x00fb);
        re_mdio_write(sc, 0x19, 0x4020);
        re_mdio_write(sc, 0x15, 0x00fc);
        re_mdio_write(sc, 0x19, 0x4468);
        re_mdio_write(sc, 0x15, 0x00fd);
        re_mdio_write(sc, 0x19, 0x9c03);
        re_mdio_write(sc, 0x15, 0x00fe);
        re_mdio_write(sc, 0x19, 0x6f07);
        re_mdio_write(sc, 0x15, 0x00ff);
        re_mdio_write(sc, 0x19, 0x58a0);
        re_mdio_write(sc, 0x15, 0x0100);
        re_mdio_write(sc, 0x19, 0xd6d1);
        re_mdio_write(sc, 0x15, 0x0101);
        re_mdio_write(sc, 0x19, 0x0004);
        re_mdio_write(sc, 0x15, 0x0102);
        re_mdio_write(sc, 0x19, 0xc137);
        re_mdio_write(sc, 0x15, 0x0103);
        re_mdio_write(sc, 0x19, 0x0002);
        re_mdio_write(sc, 0x15, 0x0104);
        re_mdio_write(sc, 0x19, 0xa0e5);
        re_mdio_write(sc, 0x15, 0x0105);
        re_mdio_write(sc, 0x19, 0x9df8);
        re_mdio_write(sc, 0x15, 0x0106);
        re_mdio_write(sc, 0x19, 0x30c6);
        re_mdio_write(sc, 0x15, 0x0107);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0108);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0109);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x010a);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x010b);
        re_mdio_write(sc, 0x19, 0x4808);
        re_mdio_write(sc, 0x15, 0x010c);
        re_mdio_write(sc, 0x19, 0xc32d);
        re_mdio_write(sc, 0x15, 0x010d);
        re_mdio_write(sc, 0x19, 0x0003);
        re_mdio_write(sc, 0x15, 0x010e);
        re_mdio_write(sc, 0x19, 0xc8b3);
        re_mdio_write(sc, 0x15, 0x010f);
        re_mdio_write(sc, 0x19, 0x00fc);
        re_mdio_write(sc, 0x15, 0x0110);
        re_mdio_write(sc, 0x19, 0x4400);
        re_mdio_write(sc, 0x15, 0x0111);
        re_mdio_write(sc, 0x19, 0x3116);
        re_mdio_write(sc, 0x15, 0x0112);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0113);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0114);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0115);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0116);
        re_mdio_write(sc, 0x19, 0x4803);
        re_mdio_write(sc, 0x15, 0x0117);
        re_mdio_write(sc, 0x19, 0x7c03);
        re_mdio_write(sc, 0x15, 0x0118);
        re_mdio_write(sc, 0x19, 0x6c02);
        re_mdio_write(sc, 0x15, 0x0119);
        re_mdio_write(sc, 0x19, 0x7c04);
        re_mdio_write(sc, 0x15, 0x011a);
        re_mdio_write(sc, 0x19, 0x6000);
        re_mdio_write(sc, 0x15, 0x011b);
        re_mdio_write(sc, 0x19, 0x5cf7);
        re_mdio_write(sc, 0x15, 0x011c);
        re_mdio_write(sc, 0x19, 0x7c2a);
        re_mdio_write(sc, 0x15, 0x011d);
        re_mdio_write(sc, 0x19, 0x5800);
        re_mdio_write(sc, 0x15, 0x011e);
        re_mdio_write(sc, 0x19, 0x5400);
        re_mdio_write(sc, 0x15, 0x011f);
        re_mdio_write(sc, 0x19, 0x7c08);
        re_mdio_write(sc, 0x15, 0x0120);
        re_mdio_write(sc, 0x19, 0x74f0);
        re_mdio_write(sc, 0x15, 0x0121);
        re_mdio_write(sc, 0x19, 0x4019);
        re_mdio_write(sc, 0x15, 0x0122);
        re_mdio_write(sc, 0x19, 0x440d);
        re_mdio_write(sc, 0x15, 0x0123);
        re_mdio_write(sc, 0x19, 0xb6c1);
        re_mdio_write(sc, 0x15, 0x0124);
        re_mdio_write(sc, 0x19, 0xc05b);
        re_mdio_write(sc, 0x15, 0x0125);
        re_mdio_write(sc, 0x19, 0x00bf);
        re_mdio_write(sc, 0x15, 0x0126);
        re_mdio_write(sc, 0x19, 0xc025);
        re_mdio_write(sc, 0x15, 0x0127);
        re_mdio_write(sc, 0x19, 0x00bd);
        re_mdio_write(sc, 0x15, 0x0128);
        re_mdio_write(sc, 0x19, 0xc603);
        re_mdio_write(sc, 0x15, 0x0129);
        re_mdio_write(sc, 0x19, 0x00bb);
        re_mdio_write(sc, 0x15, 0x012a);
        re_mdio_write(sc, 0x19, 0x8805);
        re_mdio_write(sc, 0x15, 0x012b);
        re_mdio_write(sc, 0x19, 0x7801);
        re_mdio_write(sc, 0x15, 0x012c);
        re_mdio_write(sc, 0x19, 0x4001);
        re_mdio_write(sc, 0x15, 0x012d);
        re_mdio_write(sc, 0x19, 0x7800);
        re_mdio_write(sc, 0x15, 0x012e);
        re_mdio_write(sc, 0x19, 0xa3dd);
        re_mdio_write(sc, 0x15, 0x012f);
        re_mdio_write(sc, 0x19, 0x7c03);
        re_mdio_write(sc, 0x15, 0x0130);
        re_mdio_write(sc, 0x19, 0x6c03);
        re_mdio_write(sc, 0x15, 0x0131);
        re_mdio_write(sc, 0x19, 0x8407);
        re_mdio_write(sc, 0x15, 0x0132);
        re_mdio_write(sc, 0x19, 0x7c03);
        re_mdio_write(sc, 0x15, 0x0133);
        re_mdio_write(sc, 0x19, 0x6c02);
        re_mdio_write(sc, 0x15, 0x0134);
        re_mdio_write(sc, 0x19, 0xd9b8);
        re_mdio_write(sc, 0x15, 0x0135);
        re_mdio_write(sc, 0x19, 0x0003);
        re_mdio_write(sc, 0x15, 0x0136);
        re_mdio_write(sc, 0x19, 0xc240);
        re_mdio_write(sc, 0x15, 0x0137);
        re_mdio_write(sc, 0x19, 0x0015);
        re_mdio_write(sc, 0x15, 0x0138);
        re_mdio_write(sc, 0x19, 0x7c03);
        re_mdio_write(sc, 0x15, 0x0139);
        re_mdio_write(sc, 0x19, 0x6c02);
        re_mdio_write(sc, 0x15, 0x013a);
        re_mdio_write(sc, 0x19, 0x9ae9);
        re_mdio_write(sc, 0x15, 0x013b);
        re_mdio_write(sc, 0x19, 0x3140);
        re_mdio_write(sc, 0x15, 0x013c);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x013d);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x013e);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x013f);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0140);
        re_mdio_write(sc, 0x19, 0x4807);
        re_mdio_write(sc, 0x15, 0x0141);
        re_mdio_write(sc, 0x19, 0x4004);
        re_mdio_write(sc, 0x15, 0x0142);
        re_mdio_write(sc, 0x19, 0x4410);
        re_mdio_write(sc, 0x15, 0x0143);
        re_mdio_write(sc, 0x19, 0x7c0c);
        re_mdio_write(sc, 0x15, 0x0144);
        re_mdio_write(sc, 0x19, 0x600c);
        re_mdio_write(sc, 0x15, 0x0145);
        re_mdio_write(sc, 0x19, 0x9b00);
        re_mdio_write(sc, 0x15, 0x0146);
        re_mdio_write(sc, 0x19, 0xa68f);
        re_mdio_write(sc, 0x15, 0x0147);
        re_mdio_write(sc, 0x19, 0x3116);
        re_mdio_write(sc, 0x15, 0x0148);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0149);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x014a);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x014b);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x014c);
        re_mdio_write(sc, 0x19, 0x4804);
        re_mdio_write(sc, 0x15, 0x014d);
        re_mdio_write(sc, 0x19, 0x54c0);
        re_mdio_write(sc, 0x15, 0x014e);
        re_mdio_write(sc, 0x19, 0xb703);
        re_mdio_write(sc, 0x15, 0x014f);
        re_mdio_write(sc, 0x19, 0x5cff);
        re_mdio_write(sc, 0x15, 0x0150);
        re_mdio_write(sc, 0x19, 0x315f);
        re_mdio_write(sc, 0x15, 0x0151);
        re_mdio_write(sc, 0x19, 0x7c08);
        re_mdio_write(sc, 0x15, 0x0152);
        re_mdio_write(sc, 0x19, 0x74f8);
        re_mdio_write(sc, 0x15, 0x0153);
        re_mdio_write(sc, 0x19, 0x6421);
        re_mdio_write(sc, 0x15, 0x0154);
        re_mdio_write(sc, 0x19, 0x7c08);
        re_mdio_write(sc, 0x15, 0x0155);
        re_mdio_write(sc, 0x19, 0x6000);
        re_mdio_write(sc, 0x15, 0x0156);
        re_mdio_write(sc, 0x19, 0x4003);
        re_mdio_write(sc, 0x15, 0x0157);
        re_mdio_write(sc, 0x19, 0x4418);
        re_mdio_write(sc, 0x15, 0x0158);
        re_mdio_write(sc, 0x19, 0x9b00);
        re_mdio_write(sc, 0x15, 0x0159);
        re_mdio_write(sc, 0x19, 0x6461);
        re_mdio_write(sc, 0x15, 0x015a);
        re_mdio_write(sc, 0x19, 0x64e1);
        re_mdio_write(sc, 0x15, 0x015b);
        re_mdio_write(sc, 0x19, 0x7c20);
        re_mdio_write(sc, 0x15, 0x015c);
        re_mdio_write(sc, 0x19, 0x5820);
        re_mdio_write(sc, 0x15, 0x015d);
        re_mdio_write(sc, 0x19, 0x5ccf);
        re_mdio_write(sc, 0x15, 0x015e);
        re_mdio_write(sc, 0x19, 0x7050);
        re_mdio_write(sc, 0x15, 0x015f);
        re_mdio_write(sc, 0x19, 0xd9b8);
        re_mdio_write(sc, 0x15, 0x0160);
        re_mdio_write(sc, 0x19, 0x0008);
        re_mdio_write(sc, 0x15, 0x0161);
        re_mdio_write(sc, 0x19, 0xdab1);
        re_mdio_write(sc, 0x15, 0x0162);
        re_mdio_write(sc, 0x19, 0x0015);
        re_mdio_write(sc, 0x15, 0x0163);
        re_mdio_write(sc, 0x19, 0xc244);
        re_mdio_write(sc, 0x15, 0x0164);
        re_mdio_write(sc, 0x19, 0x0013);
        re_mdio_write(sc, 0x15, 0x0165);
        re_mdio_write(sc, 0x19, 0xc021);
        re_mdio_write(sc, 0x15, 0x0166);
        re_mdio_write(sc, 0x19, 0x00f9);
        re_mdio_write(sc, 0x15, 0x0167);
        re_mdio_write(sc, 0x19, 0x3177);
        re_mdio_write(sc, 0x15, 0x0168);
        re_mdio_write(sc, 0x19, 0x5cf7);
        re_mdio_write(sc, 0x15, 0x0169);
        re_mdio_write(sc, 0x19, 0x4010);
        re_mdio_write(sc, 0x15, 0x016a);
        re_mdio_write(sc, 0x19, 0x4428);
        re_mdio_write(sc, 0x15, 0x016b);
        re_mdio_write(sc, 0x19, 0x9c00);
        re_mdio_write(sc, 0x15, 0x016c);
        re_mdio_write(sc, 0x19, 0x7c08);
        re_mdio_write(sc, 0x15, 0x016d);
        re_mdio_write(sc, 0x19, 0x6008);
        re_mdio_write(sc, 0x15, 0x016e);
        re_mdio_write(sc, 0x19, 0x7c08);
        re_mdio_write(sc, 0x15, 0x016f);
        re_mdio_write(sc, 0x19, 0x74f0);
        re_mdio_write(sc, 0x15, 0x0170);
        re_mdio_write(sc, 0x19, 0x6461);
        re_mdio_write(sc, 0x15, 0x0171);
        re_mdio_write(sc, 0x19, 0x6421);
        re_mdio_write(sc, 0x15, 0x0172);
        re_mdio_write(sc, 0x19, 0x64a1);
        re_mdio_write(sc, 0x15, 0x0173);
        re_mdio_write(sc, 0x19, 0x3116);
        re_mdio_write(sc, 0x15, 0x0174);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0175);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0176);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0177);
        re_mdio_write(sc, 0x19, 0x4805);
        re_mdio_write(sc, 0x15, 0x0178);
        re_mdio_write(sc, 0x19, 0xa103);
        re_mdio_write(sc, 0x15, 0x0179);
        re_mdio_write(sc, 0x19, 0x7c02);
        re_mdio_write(sc, 0x15, 0x017a);
        re_mdio_write(sc, 0x19, 0x6002);
        re_mdio_write(sc, 0x15, 0x017b);
        re_mdio_write(sc, 0x19, 0x7e00);
        re_mdio_write(sc, 0x15, 0x017c);
        re_mdio_write(sc, 0x19, 0x5400);
        re_mdio_write(sc, 0x15, 0x017d);
        re_mdio_write(sc, 0x19, 0x7c6b);
        re_mdio_write(sc, 0x15, 0x017e);
        re_mdio_write(sc, 0x19, 0x5c63);
        re_mdio_write(sc, 0x15, 0x017f);
        re_mdio_write(sc, 0x19, 0x407d);
        re_mdio_write(sc, 0x15, 0x0180);
        re_mdio_write(sc, 0x19, 0xa602);
        re_mdio_write(sc, 0x15, 0x0181);
        re_mdio_write(sc, 0x19, 0x4001);
        re_mdio_write(sc, 0x15, 0x0182);
        re_mdio_write(sc, 0x19, 0x4420);
        re_mdio_write(sc, 0x15, 0x0183);
        re_mdio_write(sc, 0x19, 0x4020);
        re_mdio_write(sc, 0x15, 0x0184);
        re_mdio_write(sc, 0x19, 0x44a1);
        re_mdio_write(sc, 0x15, 0x0185);
        re_mdio_write(sc, 0x19, 0xd6e0);
        re_mdio_write(sc, 0x15, 0x0186);
        re_mdio_write(sc, 0x19, 0x0009);
        re_mdio_write(sc, 0x15, 0x0187);
        re_mdio_write(sc, 0x19, 0x9efe);
        re_mdio_write(sc, 0x15, 0x0188);
        re_mdio_write(sc, 0x19, 0x7c02);
        re_mdio_write(sc, 0x15, 0x0189);
        re_mdio_write(sc, 0x19, 0x6000);
        re_mdio_write(sc, 0x15, 0x018a);
        re_mdio_write(sc, 0x19, 0x9c00);
        re_mdio_write(sc, 0x15, 0x018b);
        re_mdio_write(sc, 0x19, 0x318f);
        re_mdio_write(sc, 0x15, 0x018c);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x018d);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x018e);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x018f);
        re_mdio_write(sc, 0x19, 0x4806);
        re_mdio_write(sc, 0x15, 0x0190);
        re_mdio_write(sc, 0x19, 0x7c10);
        re_mdio_write(sc, 0x15, 0x0191);
        re_mdio_write(sc, 0x19, 0x5c10);
        re_mdio_write(sc, 0x15, 0x0192);
        re_mdio_write(sc, 0x19, 0x40fa);
        re_mdio_write(sc, 0x15, 0x0193);
        re_mdio_write(sc, 0x19, 0xa602);
        re_mdio_write(sc, 0x15, 0x0194);
        re_mdio_write(sc, 0x19, 0x4010);
        re_mdio_write(sc, 0x15, 0x0195);
        re_mdio_write(sc, 0x19, 0x4440);
        re_mdio_write(sc, 0x15, 0x0196);
        re_mdio_write(sc, 0x19, 0x9d00);
        re_mdio_write(sc, 0x15, 0x0197);
        re_mdio_write(sc, 0x19, 0x7c80);
        re_mdio_write(sc, 0x15, 0x0198);
        re_mdio_write(sc, 0x19, 0x6400);
        re_mdio_write(sc, 0x15, 0x0199);
        re_mdio_write(sc, 0x19, 0x4003);
        re_mdio_write(sc, 0x15, 0x019a);
        re_mdio_write(sc, 0x19, 0x4540);
        re_mdio_write(sc, 0x15, 0x019b);
        re_mdio_write(sc, 0x19, 0x7c08);
        re_mdio_write(sc, 0x15, 0x019c);
        re_mdio_write(sc, 0x19, 0x6008);
        re_mdio_write(sc, 0x15, 0x019d);
        re_mdio_write(sc, 0x19, 0x9f00);
        re_mdio_write(sc, 0x15, 0x019e);
        re_mdio_write(sc, 0x19, 0x7c40);
        re_mdio_write(sc, 0x15, 0x019f);
        re_mdio_write(sc, 0x19, 0x6400);
        re_mdio_write(sc, 0x15, 0x01a0);
        re_mdio_write(sc, 0x19, 0x7c80);
        re_mdio_write(sc, 0x15, 0x01a1);
        re_mdio_write(sc, 0x19, 0x6480);
        re_mdio_write(sc, 0x15, 0x01a2);
        re_mdio_write(sc, 0x19, 0x3140);
        re_mdio_write(sc, 0x15, 0x01a3);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x01a4);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x01a5);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x01a6);
        re_mdio_write(sc, 0x19, 0x4400);
        re_mdio_write(sc, 0x15, 0x01a7);
        re_mdio_write(sc, 0x19, 0x7c0b);
        re_mdio_write(sc, 0x15, 0x01a8);
        re_mdio_write(sc, 0x19, 0x6c01);
        re_mdio_write(sc, 0x15, 0x01a9);
        re_mdio_write(sc, 0x19, 0x64a8);
        re_mdio_write(sc, 0x15, 0x01aa);
        re_mdio_write(sc, 0x19, 0x6800);
        re_mdio_write(sc, 0x15, 0x01ab);
        re_mdio_write(sc, 0x19, 0x5cf0);
        re_mdio_write(sc, 0x15, 0x01ac);
        re_mdio_write(sc, 0x19, 0x588f);
        re_mdio_write(sc, 0x15, 0x01ad);
        re_mdio_write(sc, 0x19, 0xb628);
        re_mdio_write(sc, 0x15, 0x01ae);
        re_mdio_write(sc, 0x19, 0xc053);
        re_mdio_write(sc, 0x15, 0x01af);
        re_mdio_write(sc, 0x19, 0x0026);
        re_mdio_write(sc, 0x15, 0x01b0);
        re_mdio_write(sc, 0x19, 0xc02d);
        re_mdio_write(sc, 0x15, 0x01b1);
        re_mdio_write(sc, 0x19, 0x0024);
        re_mdio_write(sc, 0x15, 0x01b2);
        re_mdio_write(sc, 0x19, 0xc603);
        re_mdio_write(sc, 0x15, 0x01b3);
        re_mdio_write(sc, 0x19, 0x0022);
        re_mdio_write(sc, 0x15, 0x01b4);
        re_mdio_write(sc, 0x19, 0x8cf9);
        re_mdio_write(sc, 0x15, 0x01b5);
        re_mdio_write(sc, 0x19, 0x31ba);
        re_mdio_write(sc, 0x15, 0x01b6);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x01b7);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x01b8);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x01b9);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x01ba);
        re_mdio_write(sc, 0x19, 0x4400);
        re_mdio_write(sc, 0x15, 0x01bb);
        re_mdio_write(sc, 0x19, 0x5420);
        re_mdio_write(sc, 0x15, 0x01bc);
        re_mdio_write(sc, 0x19, 0x4811);
        re_mdio_write(sc, 0x15, 0x01bd);
        re_mdio_write(sc, 0x19, 0x5000);
        re_mdio_write(sc, 0x15, 0x01be);
        re_mdio_write(sc, 0x19, 0x4801);
        re_mdio_write(sc, 0x15, 0x01bf);
        re_mdio_write(sc, 0x19, 0x6800);
        re_mdio_write(sc, 0x15, 0x01c0);
        re_mdio_write(sc, 0x19, 0x31f5);
        re_mdio_write(sc, 0x15, 0x01c1);
        re_mdio_write(sc, 0x19, 0xb614);
        re_mdio_write(sc, 0x15, 0x01c2);
        re_mdio_write(sc, 0x19, 0x8ce4);
        re_mdio_write(sc, 0x15, 0x01c3);
        re_mdio_write(sc, 0x19, 0xb30c);
        re_mdio_write(sc, 0x15, 0x01c4);
        re_mdio_write(sc, 0x19, 0x7c03);
        re_mdio_write(sc, 0x15, 0x01c5);
        re_mdio_write(sc, 0x19, 0x6c02);
        re_mdio_write(sc, 0x15, 0x01c6);
        re_mdio_write(sc, 0x19, 0x8206);
        re_mdio_write(sc, 0x15, 0x01c7);
        re_mdio_write(sc, 0x19, 0x7c03);
        re_mdio_write(sc, 0x15, 0x01c8);
        re_mdio_write(sc, 0x19, 0x6c00);
        re_mdio_write(sc, 0x15, 0x01c9);
        re_mdio_write(sc, 0x19, 0x7c04);
        re_mdio_write(sc, 0x15, 0x01ca);
        re_mdio_write(sc, 0x19, 0x7404);
        re_mdio_write(sc, 0x15, 0x01cb);
        re_mdio_write(sc, 0x19, 0x31c0);
        re_mdio_write(sc, 0x15, 0x01cc);
        re_mdio_write(sc, 0x19, 0x7c04);
        re_mdio_write(sc, 0x15, 0x01cd);
        re_mdio_write(sc, 0x19, 0x7400);
        re_mdio_write(sc, 0x15, 0x01ce);
        re_mdio_write(sc, 0x19, 0x31c0);
        re_mdio_write(sc, 0x15, 0x01cf);
        re_mdio_write(sc, 0x19, 0x8df1);
        re_mdio_write(sc, 0x15, 0x01d0);
        re_mdio_write(sc, 0x19, 0x3248);
        re_mdio_write(sc, 0x15, 0x01d1);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x01d2);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x01d3);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x01d4);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x01d5);
        re_mdio_write(sc, 0x19, 0x4400);
        re_mdio_write(sc, 0x15, 0x01d6);
        re_mdio_write(sc, 0x19, 0x7c03);
        re_mdio_write(sc, 0x15, 0x01d7);
        re_mdio_write(sc, 0x19, 0x6c03);
        re_mdio_write(sc, 0x15, 0x01d8);
        re_mdio_write(sc, 0x19, 0x7670);
        re_mdio_write(sc, 0x15, 0x01d9);
        re_mdio_write(sc, 0x19, 0x4023);
        re_mdio_write(sc, 0x15, 0x01da);
        re_mdio_write(sc, 0x19, 0x4500);
        re_mdio_write(sc, 0x15, 0x01db);
        re_mdio_write(sc, 0x19, 0x4069);
        re_mdio_write(sc, 0x15, 0x01dc);
        re_mdio_write(sc, 0x19, 0x4580);
        re_mdio_write(sc, 0x15, 0x01dd);
        re_mdio_write(sc, 0x19, 0x9f00);
        re_mdio_write(sc, 0x15, 0x01de);
        re_mdio_write(sc, 0x19, 0xcff5);
        re_mdio_write(sc, 0x15, 0x01df);
        re_mdio_write(sc, 0x19, 0x00ff);
        re_mdio_write(sc, 0x15, 0x01e0);
        re_mdio_write(sc, 0x19, 0x76f0);
        re_mdio_write(sc, 0x15, 0x01e1);
        re_mdio_write(sc, 0x19, 0x4400);
        re_mdio_write(sc, 0x15, 0x01e2);
        re_mdio_write(sc, 0x19, 0x4023);
        re_mdio_write(sc, 0x15, 0x01e3);
        re_mdio_write(sc, 0x19, 0x4500);
        re_mdio_write(sc, 0x15, 0x01e4);
        re_mdio_write(sc, 0x19, 0x4069);
        re_mdio_write(sc, 0x15, 0x01e5);
        re_mdio_write(sc, 0x19, 0x4580);
        re_mdio_write(sc, 0x15, 0x01e6);
        re_mdio_write(sc, 0x19, 0x9f00);
        re_mdio_write(sc, 0x15, 0x01e7);
        re_mdio_write(sc, 0x19, 0xd0f5);
        re_mdio_write(sc, 0x15, 0x01e8);
        re_mdio_write(sc, 0x19, 0x00ff);
        re_mdio_write(sc, 0x15, 0x01e9);
        re_mdio_write(sc, 0x19, 0x4400);
        re_mdio_write(sc, 0x15, 0x01ea);
        re_mdio_write(sc, 0x19, 0x7c03);
        re_mdio_write(sc, 0x15, 0x01eb);
        re_mdio_write(sc, 0x19, 0x6800);
        re_mdio_write(sc, 0x15, 0x01ec);
        re_mdio_write(sc, 0x19, 0x66a0);
        re_mdio_write(sc, 0x15, 0x01ed);
        re_mdio_write(sc, 0x19, 0x8300);
        re_mdio_write(sc, 0x15, 0x01ee);
        re_mdio_write(sc, 0x19, 0x74f0);
        re_mdio_write(sc, 0x15, 0x01ef);
        re_mdio_write(sc, 0x19, 0x3006);
        re_mdio_write(sc, 0x15, 0x01f0);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x01f1);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x01f2);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x01f3);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x01f4);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x01f5);
        re_mdio_write(sc, 0x19, 0x7c03);
        re_mdio_write(sc, 0x15, 0x01f6);
        re_mdio_write(sc, 0x19, 0x6c02);
        re_mdio_write(sc, 0x15, 0x01f7);
        re_mdio_write(sc, 0x19, 0x409d);
        re_mdio_write(sc, 0x15, 0x01f8);
        re_mdio_write(sc, 0x19, 0x7c87);
        re_mdio_write(sc, 0x15, 0x01f9);
        re_mdio_write(sc, 0x19, 0xae14);
        re_mdio_write(sc, 0x15, 0x01fa);
        re_mdio_write(sc, 0x19, 0x4400);
        re_mdio_write(sc, 0x15, 0x01fb);
        re_mdio_write(sc, 0x19, 0x7c40);
        re_mdio_write(sc, 0x15, 0x01fc);
        re_mdio_write(sc, 0x19, 0x6800);
        re_mdio_write(sc, 0x15, 0x01fd);
        re_mdio_write(sc, 0x19, 0x7801);
        re_mdio_write(sc, 0x15, 0x01fe);
        re_mdio_write(sc, 0x19, 0x980e);
        re_mdio_write(sc, 0x15, 0x01ff);
        re_mdio_write(sc, 0x19, 0x930c);
        re_mdio_write(sc, 0x15, 0x0200);
        re_mdio_write(sc, 0x19, 0x9206);
        re_mdio_write(sc, 0x15, 0x0201);
        re_mdio_write(sc, 0x19, 0x4002);
        re_mdio_write(sc, 0x15, 0x0202);
        re_mdio_write(sc, 0x19, 0x7800);
        re_mdio_write(sc, 0x15, 0x0203);
        re_mdio_write(sc, 0x19, 0x588f);
        re_mdio_write(sc, 0x15, 0x0204);
        re_mdio_write(sc, 0x19, 0x5520);
        re_mdio_write(sc, 0x15, 0x0205);
        re_mdio_write(sc, 0x19, 0x320c);
        re_mdio_write(sc, 0x15, 0x0206);
        re_mdio_write(sc, 0x19, 0x4000);
        re_mdio_write(sc, 0x15, 0x0207);
        re_mdio_write(sc, 0x19, 0x7800);
        re_mdio_write(sc, 0x15, 0x0208);
        re_mdio_write(sc, 0x19, 0x588d);
        re_mdio_write(sc, 0x15, 0x0209);
        re_mdio_write(sc, 0x19, 0x5500);
        re_mdio_write(sc, 0x15, 0x020a);
        re_mdio_write(sc, 0x19, 0x320c);
        re_mdio_write(sc, 0x15, 0x020b);
        re_mdio_write(sc, 0x19, 0x4002);
        re_mdio_write(sc, 0x15, 0x020c);
        re_mdio_write(sc, 0x19, 0x3220);
        re_mdio_write(sc, 0x15, 0x020d);
        re_mdio_write(sc, 0x19, 0x4480);
        re_mdio_write(sc, 0x15, 0x020e);
        re_mdio_write(sc, 0x19, 0x9e03);
        re_mdio_write(sc, 0x15, 0x020f);
        re_mdio_write(sc, 0x19, 0x7c40);
        re_mdio_write(sc, 0x15, 0x0210);
        re_mdio_write(sc, 0x19, 0x6840);
        re_mdio_write(sc, 0x15, 0x0211);
        re_mdio_write(sc, 0x19, 0x7801);
        re_mdio_write(sc, 0x15, 0x0212);
        re_mdio_write(sc, 0x19, 0x980e);
        re_mdio_write(sc, 0x15, 0x0213);
        re_mdio_write(sc, 0x19, 0x930c);
        re_mdio_write(sc, 0x15, 0x0214);
        re_mdio_write(sc, 0x19, 0x9206);
        re_mdio_write(sc, 0x15, 0x0215);
        re_mdio_write(sc, 0x19, 0x4000);
        re_mdio_write(sc, 0x15, 0x0216);
        re_mdio_write(sc, 0x19, 0x7800);
        re_mdio_write(sc, 0x15, 0x0217);
        re_mdio_write(sc, 0x19, 0x588f);
        re_mdio_write(sc, 0x15, 0x0218);
        re_mdio_write(sc, 0x19, 0x5520);
        re_mdio_write(sc, 0x15, 0x0219);
        re_mdio_write(sc, 0x19, 0x3220);
        re_mdio_write(sc, 0x15, 0x021a);
        re_mdio_write(sc, 0x19, 0x4002);
        re_mdio_write(sc, 0x15, 0x021b);
        re_mdio_write(sc, 0x19, 0x7800);
        re_mdio_write(sc, 0x15, 0x021c);
        re_mdio_write(sc, 0x19, 0x588d);
        re_mdio_write(sc, 0x15, 0x021d);
        re_mdio_write(sc, 0x19, 0x5540);
        re_mdio_write(sc, 0x15, 0x021e);
        re_mdio_write(sc, 0x19, 0x3220);
        re_mdio_write(sc, 0x15, 0x021f);
        re_mdio_write(sc, 0x19, 0x4000);
        re_mdio_write(sc, 0x15, 0x0220);
        re_mdio_write(sc, 0x19, 0x7800);
        re_mdio_write(sc, 0x15, 0x0221);
        re_mdio_write(sc, 0x19, 0x7c03);
        re_mdio_write(sc, 0x15, 0x0222);
        re_mdio_write(sc, 0x19, 0x6c00);
        re_mdio_write(sc, 0x15, 0x0223);
        re_mdio_write(sc, 0x19, 0x3231);
        re_mdio_write(sc, 0x15, 0x0224);
        re_mdio_write(sc, 0x19, 0xab06);
        re_mdio_write(sc, 0x15, 0x0225);
        re_mdio_write(sc, 0x19, 0xbf08);
        re_mdio_write(sc, 0x15, 0x0226);
        re_mdio_write(sc, 0x19, 0x4076);
        re_mdio_write(sc, 0x15, 0x0227);
        re_mdio_write(sc, 0x19, 0x7d07);
        re_mdio_write(sc, 0x15, 0x0228);
        re_mdio_write(sc, 0x19, 0x4502);
        re_mdio_write(sc, 0x15, 0x0229);
        re_mdio_write(sc, 0x19, 0x3231);
        re_mdio_write(sc, 0x15, 0x022a);
        re_mdio_write(sc, 0x19, 0x7d80);
        re_mdio_write(sc, 0x15, 0x022b);
        re_mdio_write(sc, 0x19, 0x5180);
        re_mdio_write(sc, 0x15, 0x022c);
        re_mdio_write(sc, 0x19, 0x322f);
        re_mdio_write(sc, 0x15, 0x022d);
        re_mdio_write(sc, 0x19, 0x7d80);
        re_mdio_write(sc, 0x15, 0x022e);
        re_mdio_write(sc, 0x19, 0x5000);
        re_mdio_write(sc, 0x15, 0x022f);
        re_mdio_write(sc, 0x19, 0x7d07);
        re_mdio_write(sc, 0x15, 0x0230);
        re_mdio_write(sc, 0x19, 0x4402);
        re_mdio_write(sc, 0x15, 0x0231);
        re_mdio_write(sc, 0x19, 0x7c03);
        re_mdio_write(sc, 0x15, 0x0232);
        re_mdio_write(sc, 0x19, 0x6c02);
        re_mdio_write(sc, 0x15, 0x0233);
        re_mdio_write(sc, 0x19, 0x7c03);
        re_mdio_write(sc, 0x15, 0x0234);
        re_mdio_write(sc, 0x19, 0xb309);
        re_mdio_write(sc, 0x15, 0x0235);
        re_mdio_write(sc, 0x19, 0xb204);
        re_mdio_write(sc, 0x15, 0x0236);
        re_mdio_write(sc, 0x19, 0xb105);
        re_mdio_write(sc, 0x15, 0x0237);
        re_mdio_write(sc, 0x19, 0x6c00);
        re_mdio_write(sc, 0x15, 0x0238);
        re_mdio_write(sc, 0x19, 0x31c1);
        re_mdio_write(sc, 0x15, 0x0239);
        re_mdio_write(sc, 0x19, 0x6c00);
        re_mdio_write(sc, 0x15, 0x023a);
        re_mdio_write(sc, 0x19, 0x3261);
        re_mdio_write(sc, 0x15, 0x023b);
        re_mdio_write(sc, 0x19, 0x6c00);
        re_mdio_write(sc, 0x15, 0x023c);
        re_mdio_write(sc, 0x19, 0x3250);
        re_mdio_write(sc, 0x15, 0x023d);
        re_mdio_write(sc, 0x19, 0xb203);
        re_mdio_write(sc, 0x15, 0x023e);
        re_mdio_write(sc, 0x19, 0x6c00);
        re_mdio_write(sc, 0x15, 0x023f);
        re_mdio_write(sc, 0x19, 0x327a);
        re_mdio_write(sc, 0x15, 0x0240);
        re_mdio_write(sc, 0x19, 0x6c00);
        re_mdio_write(sc, 0x15, 0x0241);
        re_mdio_write(sc, 0x19, 0x3293);
        re_mdio_write(sc, 0x15, 0x0242);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0243);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0244);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0245);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0246);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0247);
        re_mdio_write(sc, 0x19, 0x32a3);
        re_mdio_write(sc, 0x15, 0x0248);
        re_mdio_write(sc, 0x19, 0x5520);
        re_mdio_write(sc, 0x15, 0x0249);
        re_mdio_write(sc, 0x19, 0x403d);
        re_mdio_write(sc, 0x15, 0x024a);
        re_mdio_write(sc, 0x19, 0x440c);
        re_mdio_write(sc, 0x15, 0x024b);
        re_mdio_write(sc, 0x19, 0x4812);
        re_mdio_write(sc, 0x15, 0x024c);
        re_mdio_write(sc, 0x19, 0x5001);
        re_mdio_write(sc, 0x15, 0x024d);
        re_mdio_write(sc, 0x19, 0x4802);
        re_mdio_write(sc, 0x15, 0x024e);
        re_mdio_write(sc, 0x19, 0x6880);
        re_mdio_write(sc, 0x15, 0x024f);
        re_mdio_write(sc, 0x19, 0x31f5);
        re_mdio_write(sc, 0x15, 0x0250);
        re_mdio_write(sc, 0x19, 0xb685);
        re_mdio_write(sc, 0x15, 0x0251);
        re_mdio_write(sc, 0x19, 0x801c);
        re_mdio_write(sc, 0x15, 0x0252);
        re_mdio_write(sc, 0x19, 0xbaf5);
        re_mdio_write(sc, 0x15, 0x0253);
        re_mdio_write(sc, 0x19, 0xc07c);
        re_mdio_write(sc, 0x15, 0x0254);
        re_mdio_write(sc, 0x19, 0x00fb);
        re_mdio_write(sc, 0x15, 0x0255);
        re_mdio_write(sc, 0x19, 0x325a);
        re_mdio_write(sc, 0x15, 0x0256);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0257);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0258);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0259);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x025a);
        re_mdio_write(sc, 0x19, 0x481a);
        re_mdio_write(sc, 0x15, 0x025b);
        re_mdio_write(sc, 0x19, 0x5001);
        re_mdio_write(sc, 0x15, 0x025c);
        re_mdio_write(sc, 0x19, 0x401b);
        re_mdio_write(sc, 0x15, 0x025d);
        re_mdio_write(sc, 0x19, 0x480a);
        re_mdio_write(sc, 0x15, 0x025e);
        re_mdio_write(sc, 0x19, 0x4418);
        re_mdio_write(sc, 0x15, 0x025f);
        re_mdio_write(sc, 0x19, 0x6900);
        re_mdio_write(sc, 0x15, 0x0260);
        re_mdio_write(sc, 0x19, 0x31f5);
        re_mdio_write(sc, 0x15, 0x0261);
        re_mdio_write(sc, 0x19, 0xb64b);
        re_mdio_write(sc, 0x15, 0x0262);
        re_mdio_write(sc, 0x19, 0xdb00);
        re_mdio_write(sc, 0x15, 0x0263);
        re_mdio_write(sc, 0x19, 0x0048);
        re_mdio_write(sc, 0x15, 0x0264);
        re_mdio_write(sc, 0x19, 0xdb7d);
        re_mdio_write(sc, 0x15, 0x0265);
        re_mdio_write(sc, 0x19, 0x0002);
        re_mdio_write(sc, 0x15, 0x0266);
        re_mdio_write(sc, 0x19, 0xa0fa);
        re_mdio_write(sc, 0x15, 0x0267);
        re_mdio_write(sc, 0x19, 0x4408);
        re_mdio_write(sc, 0x15, 0x0268);
        re_mdio_write(sc, 0x19, 0x3248);
        re_mdio_write(sc, 0x15, 0x0269);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x026a);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x026b);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x026c);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x026d);
        re_mdio_write(sc, 0x19, 0xb806);
        re_mdio_write(sc, 0x15, 0x026e);
        re_mdio_write(sc, 0x19, 0x588d);
        re_mdio_write(sc, 0x15, 0x026f);
        re_mdio_write(sc, 0x19, 0x5500);
        re_mdio_write(sc, 0x15, 0x0270);
        re_mdio_write(sc, 0x19, 0x7801);
        re_mdio_write(sc, 0x15, 0x0271);
        re_mdio_write(sc, 0x19, 0x4002);
        re_mdio_write(sc, 0x15, 0x0272);
        re_mdio_write(sc, 0x19, 0x7800);
        re_mdio_write(sc, 0x15, 0x0273);
        re_mdio_write(sc, 0x19, 0x4814);
        re_mdio_write(sc, 0x15, 0x0274);
        re_mdio_write(sc, 0x19, 0x500b);
        re_mdio_write(sc, 0x15, 0x0275);
        re_mdio_write(sc, 0x19, 0x4804);
        re_mdio_write(sc, 0x15, 0x0276);
        re_mdio_write(sc, 0x19, 0x40c4);
        re_mdio_write(sc, 0x15, 0x0277);
        re_mdio_write(sc, 0x19, 0x4425);
        re_mdio_write(sc, 0x15, 0x0278);
        re_mdio_write(sc, 0x19, 0x6a00);
        re_mdio_write(sc, 0x15, 0x0279);
        re_mdio_write(sc, 0x19, 0x31f5);
        re_mdio_write(sc, 0x15, 0x027a);
        re_mdio_write(sc, 0x19, 0xb632);
        re_mdio_write(sc, 0x15, 0x027b);
        re_mdio_write(sc, 0x19, 0xdc03);
        re_mdio_write(sc, 0x15, 0x027c);
        re_mdio_write(sc, 0x19, 0x0027);
        re_mdio_write(sc, 0x15, 0x027d);
        re_mdio_write(sc, 0x19, 0x80fc);
        re_mdio_write(sc, 0x15, 0x027e);
        re_mdio_write(sc, 0x19, 0x3283);
        re_mdio_write(sc, 0x15, 0x027f);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0280);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0281);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0282);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0283);
        re_mdio_write(sc, 0x19, 0xb806);
        re_mdio_write(sc, 0x15, 0x0284);
        re_mdio_write(sc, 0x19, 0x588f);
        re_mdio_write(sc, 0x15, 0x0285);
        re_mdio_write(sc, 0x19, 0x5520);
        re_mdio_write(sc, 0x15, 0x0286);
        re_mdio_write(sc, 0x19, 0x7801);
        re_mdio_write(sc, 0x15, 0x0287);
        re_mdio_write(sc, 0x19, 0x4000);
        re_mdio_write(sc, 0x15, 0x0288);
        re_mdio_write(sc, 0x19, 0x7800);
        re_mdio_write(sc, 0x15, 0x0289);
        re_mdio_write(sc, 0x19, 0x4818);
        re_mdio_write(sc, 0x15, 0x028a);
        re_mdio_write(sc, 0x19, 0x5051);
        re_mdio_write(sc, 0x15, 0x028b);
        re_mdio_write(sc, 0x19, 0x4808);
        re_mdio_write(sc, 0x15, 0x028c);
        re_mdio_write(sc, 0x19, 0x4050);
        re_mdio_write(sc, 0x15, 0x028d);
        re_mdio_write(sc, 0x19, 0x4462);
        re_mdio_write(sc, 0x15, 0x028e);
        re_mdio_write(sc, 0x19, 0x40c4);
        re_mdio_write(sc, 0x15, 0x028f);
        re_mdio_write(sc, 0x19, 0x4473);
        re_mdio_write(sc, 0x15, 0x0290);
        re_mdio_write(sc, 0x19, 0x5041);
        re_mdio_write(sc, 0x15, 0x0291);
        re_mdio_write(sc, 0x19, 0x6b00);
        re_mdio_write(sc, 0x15, 0x0292);
        re_mdio_write(sc, 0x19, 0x31f5);
        re_mdio_write(sc, 0x15, 0x0293);
        re_mdio_write(sc, 0x19, 0xb619);
        re_mdio_write(sc, 0x15, 0x0294);
        re_mdio_write(sc, 0x19, 0x80d9);
        re_mdio_write(sc, 0x15, 0x0295);
        re_mdio_write(sc, 0x19, 0xbd06);
        re_mdio_write(sc, 0x15, 0x0296);
        re_mdio_write(sc, 0x19, 0xbb0d);
        re_mdio_write(sc, 0x15, 0x0297);
        re_mdio_write(sc, 0x19, 0xaf14);
        re_mdio_write(sc, 0x15, 0x0298);
        re_mdio_write(sc, 0x19, 0x8efa);
        re_mdio_write(sc, 0x15, 0x0299);
        re_mdio_write(sc, 0x19, 0x5049);
        re_mdio_write(sc, 0x15, 0x029a);
        re_mdio_write(sc, 0x19, 0x3248);
        re_mdio_write(sc, 0x15, 0x029b);
        re_mdio_write(sc, 0x19, 0x4c10);
        re_mdio_write(sc, 0x15, 0x029c);
        re_mdio_write(sc, 0x19, 0x44b0);
        re_mdio_write(sc, 0x15, 0x029d);
        re_mdio_write(sc, 0x19, 0x4c00);
        re_mdio_write(sc, 0x15, 0x029e);
        re_mdio_write(sc, 0x19, 0x3292);
        re_mdio_write(sc, 0x15, 0x029f);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x02a0);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x02a1);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x02a2);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x02a3);
        re_mdio_write(sc, 0x19, 0x481f);
        re_mdio_write(sc, 0x15, 0x02a4);
        re_mdio_write(sc, 0x19, 0x5005);
        re_mdio_write(sc, 0x15, 0x02a5);
        re_mdio_write(sc, 0x19, 0x480f);
        re_mdio_write(sc, 0x15, 0x02a6);
        re_mdio_write(sc, 0x19, 0xac00);
        re_mdio_write(sc, 0x15, 0x02a7);
        re_mdio_write(sc, 0x19, 0x31a6);
        re_mdio_write(sc, 0x15, 0x02a8);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x02a9);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x02aa);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x02ab);
        re_mdio_write(sc, 0x19, 0x31ba);
        re_mdio_write(sc, 0x15, 0x02ac);
        re_mdio_write(sc, 0x19, 0x31d5);
        re_mdio_write(sc, 0x15, 0x02ad);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x02ae);
        re_mdio_write(sc, 0x19, 0x5cf0);
        re_mdio_write(sc, 0x15, 0x02af);
        re_mdio_write(sc, 0x19, 0x588c);
        re_mdio_write(sc, 0x15, 0x02b0);
        re_mdio_write(sc, 0x19, 0x542f);
        re_mdio_write(sc, 0x15, 0x02b1);
        re_mdio_write(sc, 0x19, 0x7ffb);
        re_mdio_write(sc, 0x15, 0x02b2);
        re_mdio_write(sc, 0x19, 0x6ff8);
        re_mdio_write(sc, 0x15, 0x02b3);
        re_mdio_write(sc, 0x19, 0x64a4);
        re_mdio_write(sc, 0x15, 0x02b4);
        re_mdio_write(sc, 0x19, 0x64a0);
        re_mdio_write(sc, 0x15, 0x02b5);
        re_mdio_write(sc, 0x19, 0x6800);
        re_mdio_write(sc, 0x15, 0x02b6);
        re_mdio_write(sc, 0x19, 0x4400);
        re_mdio_write(sc, 0x15, 0x02b7);
        re_mdio_write(sc, 0x19, 0x4020);
        re_mdio_write(sc, 0x15, 0x02b8);
        re_mdio_write(sc, 0x19, 0x4480);
        re_mdio_write(sc, 0x15, 0x02b9);
        re_mdio_write(sc, 0x19, 0x9e00);
        re_mdio_write(sc, 0x15, 0x02ba);
        re_mdio_write(sc, 0x19, 0x4891);
        re_mdio_write(sc, 0x15, 0x02bb);
        re_mdio_write(sc, 0x19, 0x4cc0);
        re_mdio_write(sc, 0x15, 0x02bc);
        re_mdio_write(sc, 0x19, 0x4801);
        re_mdio_write(sc, 0x15, 0x02bd);
        re_mdio_write(sc, 0x19, 0xa609);
        re_mdio_write(sc, 0x15, 0x02be);
        re_mdio_write(sc, 0x19, 0xd64f);
        re_mdio_write(sc, 0x15, 0x02bf);
        re_mdio_write(sc, 0x19, 0x004e);
        re_mdio_write(sc, 0x15, 0x02c0);
        re_mdio_write(sc, 0x19, 0x87fe);
        re_mdio_write(sc, 0x15, 0x02c1);
        re_mdio_write(sc, 0x19, 0x32c6);
        re_mdio_write(sc, 0x15, 0x02c2);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x02c3);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x02c4);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x02c5);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x02c6);
        re_mdio_write(sc, 0x19, 0x48b2);
        re_mdio_write(sc, 0x15, 0x02c7);
        re_mdio_write(sc, 0x19, 0x4020);
        re_mdio_write(sc, 0x15, 0x02c8);
        re_mdio_write(sc, 0x19, 0x4822);
        re_mdio_write(sc, 0x15, 0x02c9);
        re_mdio_write(sc, 0x19, 0x4488);
        re_mdio_write(sc, 0x15, 0x02ca);
        re_mdio_write(sc, 0x19, 0xd64f);
        re_mdio_write(sc, 0x15, 0x02cb);
        re_mdio_write(sc, 0x19, 0x0042);
        re_mdio_write(sc, 0x15, 0x02cc);
        re_mdio_write(sc, 0x19, 0x8203);
        re_mdio_write(sc, 0x15, 0x02cd);
        re_mdio_write(sc, 0x19, 0x4cc8);
        re_mdio_write(sc, 0x15, 0x02ce);
        re_mdio_write(sc, 0x19, 0x32d0);
        re_mdio_write(sc, 0x15, 0x02cf);
        re_mdio_write(sc, 0x19, 0x4cc0);
        re_mdio_write(sc, 0x15, 0x02d0);
        re_mdio_write(sc, 0x19, 0xc4d4);
        re_mdio_write(sc, 0x15, 0x02d1);
        re_mdio_write(sc, 0x19, 0x00f9);
        re_mdio_write(sc, 0x15, 0x02d2);
        re_mdio_write(sc, 0x19, 0xa51a);
        re_mdio_write(sc, 0x15, 0x02d3);
        re_mdio_write(sc, 0x19, 0x32d9);
        re_mdio_write(sc, 0x15, 0x02d4);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x02d5);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x02d6);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x02d7);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x02d8);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x02d9);
        re_mdio_write(sc, 0x19, 0x48b3);
        re_mdio_write(sc, 0x15, 0x02da);
        re_mdio_write(sc, 0x19, 0x4020);
        re_mdio_write(sc, 0x15, 0x02db);
        re_mdio_write(sc, 0x19, 0x4823);
        re_mdio_write(sc, 0x15, 0x02dc);
        re_mdio_write(sc, 0x19, 0x4410);
        re_mdio_write(sc, 0x15, 0x02dd);
        re_mdio_write(sc, 0x19, 0xb630);
        re_mdio_write(sc, 0x15, 0x02de);
        re_mdio_write(sc, 0x19, 0x7dc8);
        re_mdio_write(sc, 0x15, 0x02df);
        re_mdio_write(sc, 0x19, 0x8203);
        re_mdio_write(sc, 0x15, 0x02e0);
        re_mdio_write(sc, 0x19, 0x4c48);
        re_mdio_write(sc, 0x15, 0x02e1);
        re_mdio_write(sc, 0x19, 0x32e3);
        re_mdio_write(sc, 0x15, 0x02e2);
        re_mdio_write(sc, 0x19, 0x4c40);
        re_mdio_write(sc, 0x15, 0x02e3);
        re_mdio_write(sc, 0x19, 0x9bfa);
        re_mdio_write(sc, 0x15, 0x02e4);
        re_mdio_write(sc, 0x19, 0x84ca);
        re_mdio_write(sc, 0x15, 0x02e5);
        re_mdio_write(sc, 0x19, 0x85f8);
        re_mdio_write(sc, 0x15, 0x02e6);
        re_mdio_write(sc, 0x19, 0x32ec);
        re_mdio_write(sc, 0x15, 0x02e7);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x02e8);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x02e9);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x02ea);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x02eb);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x02ec);
        re_mdio_write(sc, 0x19, 0x48d4);
        re_mdio_write(sc, 0x15, 0x02ed);
        re_mdio_write(sc, 0x19, 0x4020);
        re_mdio_write(sc, 0x15, 0x02ee);
        re_mdio_write(sc, 0x19, 0x4844);
        re_mdio_write(sc, 0x15, 0x02ef);
        re_mdio_write(sc, 0x19, 0x4420);
        re_mdio_write(sc, 0x15, 0x02f0);
        re_mdio_write(sc, 0x19, 0x6800);
        re_mdio_write(sc, 0x15, 0x02f1);
        re_mdio_write(sc, 0x19, 0x7dc0);
        re_mdio_write(sc, 0x15, 0x02f2);
        re_mdio_write(sc, 0x19, 0x4c40);
        re_mdio_write(sc, 0x15, 0x02f3);
        re_mdio_write(sc, 0x19, 0x7c0b);
        re_mdio_write(sc, 0x15, 0x02f4);
        re_mdio_write(sc, 0x19, 0x6c08);
        re_mdio_write(sc, 0x15, 0x02f5);
        re_mdio_write(sc, 0x19, 0x3311);
        re_mdio_write(sc, 0x15, 0x02f6);
        re_mdio_write(sc, 0x19, 0x9cfd);
        re_mdio_write(sc, 0x15, 0x02f7);
        re_mdio_write(sc, 0x19, 0xb616);
        re_mdio_write(sc, 0x15, 0x02f8);
        re_mdio_write(sc, 0x19, 0xc42b);
        re_mdio_write(sc, 0x15, 0x02f9);
        re_mdio_write(sc, 0x19, 0x00e0);
        re_mdio_write(sc, 0x15, 0x02fa);
        re_mdio_write(sc, 0x19, 0xc455);
        re_mdio_write(sc, 0x15, 0x02fb);
        re_mdio_write(sc, 0x19, 0x00b3);
        re_mdio_write(sc, 0x15, 0x02fc);
        re_mdio_write(sc, 0x19, 0xb20a);
        re_mdio_write(sc, 0x15, 0x02fd);
        re_mdio_write(sc, 0x19, 0x7c03);
        re_mdio_write(sc, 0x15, 0x02fe);
        re_mdio_write(sc, 0x19, 0x6c02);
        re_mdio_write(sc, 0x15, 0x02ff);
        re_mdio_write(sc, 0x19, 0x8204);
        re_mdio_write(sc, 0x15, 0x0300);
        re_mdio_write(sc, 0x19, 0x7c04);
        re_mdio_write(sc, 0x15, 0x0301);
        re_mdio_write(sc, 0x19, 0x7404);
        re_mdio_write(sc, 0x15, 0x0302);
        re_mdio_write(sc, 0x19, 0x32f3);
        re_mdio_write(sc, 0x15, 0x0303);
        re_mdio_write(sc, 0x19, 0x7c04);
        re_mdio_write(sc, 0x15, 0x0304);
        re_mdio_write(sc, 0x19, 0x7400);
        re_mdio_write(sc, 0x15, 0x0305);
        re_mdio_write(sc, 0x19, 0x32f3);
        re_mdio_write(sc, 0x15, 0x0306);
        re_mdio_write(sc, 0x19, 0xefed);
        re_mdio_write(sc, 0x15, 0x0307);
        re_mdio_write(sc, 0x19, 0x3342);
        re_mdio_write(sc, 0x15, 0x0308);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0309);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x030a);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x030b);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x030c);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x030d);
        re_mdio_write(sc, 0x19, 0x3006);
        re_mdio_write(sc, 0x15, 0x030e);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x030f);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0310);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0311);
        re_mdio_write(sc, 0x19, 0x7c08);
        re_mdio_write(sc, 0x15, 0x0312);
        re_mdio_write(sc, 0x19, 0xa207);
        re_mdio_write(sc, 0x15, 0x0313);
        re_mdio_write(sc, 0x19, 0x4c00);
        re_mdio_write(sc, 0x15, 0x0314);
        re_mdio_write(sc, 0x19, 0x3322);
        re_mdio_write(sc, 0x15, 0x0315);
        re_mdio_write(sc, 0x19, 0x4041);
        re_mdio_write(sc, 0x15, 0x0316);
        re_mdio_write(sc, 0x19, 0x7d07);
        re_mdio_write(sc, 0x15, 0x0317);
        re_mdio_write(sc, 0x19, 0x4502);
        re_mdio_write(sc, 0x15, 0x0318);
        re_mdio_write(sc, 0x19, 0x3322);
        re_mdio_write(sc, 0x15, 0x0319);
        re_mdio_write(sc, 0x19, 0x4c08);
        re_mdio_write(sc, 0x15, 0x031a);
        re_mdio_write(sc, 0x19, 0x3322);
        re_mdio_write(sc, 0x15, 0x031b);
        re_mdio_write(sc, 0x19, 0x7d80);
        re_mdio_write(sc, 0x15, 0x031c);
        re_mdio_write(sc, 0x19, 0x5180);
        re_mdio_write(sc, 0x15, 0x031d);
        re_mdio_write(sc, 0x19, 0x3320);
        re_mdio_write(sc, 0x15, 0x031e);
        re_mdio_write(sc, 0x19, 0x7d80);
        re_mdio_write(sc, 0x15, 0x031f);
        re_mdio_write(sc, 0x19, 0x5000);
        re_mdio_write(sc, 0x15, 0x0320);
        re_mdio_write(sc, 0x19, 0x7d07);
        re_mdio_write(sc, 0x15, 0x0321);
        re_mdio_write(sc, 0x19, 0x4402);
        re_mdio_write(sc, 0x15, 0x0322);
        re_mdio_write(sc, 0x19, 0x7c03);
        re_mdio_write(sc, 0x15, 0x0323);
        re_mdio_write(sc, 0x19, 0x6c02);
        re_mdio_write(sc, 0x15, 0x0324);
        re_mdio_write(sc, 0x19, 0x7c03);
        re_mdio_write(sc, 0x15, 0x0325);
        re_mdio_write(sc, 0x19, 0xb30c);
        re_mdio_write(sc, 0x15, 0x0326);
        re_mdio_write(sc, 0x19, 0xb206);
        re_mdio_write(sc, 0x15, 0x0327);
        re_mdio_write(sc, 0x19, 0xb103);
        re_mdio_write(sc, 0x15, 0x0328);
        re_mdio_write(sc, 0x19, 0x6c00);
        re_mdio_write(sc, 0x15, 0x0329);
        re_mdio_write(sc, 0x19, 0x32f6);
        re_mdio_write(sc, 0x15, 0x032a);
        re_mdio_write(sc, 0x19, 0x6c00);
        re_mdio_write(sc, 0x15, 0x032b);
        re_mdio_write(sc, 0x19, 0x3352);
        re_mdio_write(sc, 0x15, 0x032c);
        re_mdio_write(sc, 0x19, 0xb103);
        re_mdio_write(sc, 0x15, 0x032d);
        re_mdio_write(sc, 0x19, 0x6c00);
        re_mdio_write(sc, 0x15, 0x032e);
        re_mdio_write(sc, 0x19, 0x336a);
        re_mdio_write(sc, 0x15, 0x032f);
        re_mdio_write(sc, 0x19, 0x6c00);
        re_mdio_write(sc, 0x15, 0x0330);
        re_mdio_write(sc, 0x19, 0x3382);
        re_mdio_write(sc, 0x15, 0x0331);
        re_mdio_write(sc, 0x19, 0xb206);
        re_mdio_write(sc, 0x15, 0x0332);
        re_mdio_write(sc, 0x19, 0xb103);
        re_mdio_write(sc, 0x15, 0x0333);
        re_mdio_write(sc, 0x19, 0x6c00);
        re_mdio_write(sc, 0x15, 0x0334);
        re_mdio_write(sc, 0x19, 0x3395);
        re_mdio_write(sc, 0x15, 0x0335);
        re_mdio_write(sc, 0x19, 0x6c00);
        re_mdio_write(sc, 0x15, 0x0336);
        re_mdio_write(sc, 0x19, 0x33c6);
        re_mdio_write(sc, 0x15, 0x0337);
        re_mdio_write(sc, 0x19, 0xb103);
        re_mdio_write(sc, 0x15, 0x0338);
        re_mdio_write(sc, 0x19, 0x6c00);
        re_mdio_write(sc, 0x15, 0x0339);
        re_mdio_write(sc, 0x19, 0x33d7);
        re_mdio_write(sc, 0x15, 0x033a);
        re_mdio_write(sc, 0x19, 0x6c00);
        re_mdio_write(sc, 0x15, 0x033b);
        re_mdio_write(sc, 0x19, 0x33f2);
        re_mdio_write(sc, 0x15, 0x033c);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x033d);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x033e);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x033f);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0340);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0341);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0342);
        re_mdio_write(sc, 0x19, 0x49b5);
        re_mdio_write(sc, 0x15, 0x0343);
        re_mdio_write(sc, 0x19, 0x7d00);
        re_mdio_write(sc, 0x15, 0x0344);
        re_mdio_write(sc, 0x19, 0x4d00);
        re_mdio_write(sc, 0x15, 0x0345);
        re_mdio_write(sc, 0x19, 0x6880);
        re_mdio_write(sc, 0x15, 0x0346);
        re_mdio_write(sc, 0x19, 0x7c08);
        re_mdio_write(sc, 0x15, 0x0347);
        re_mdio_write(sc, 0x19, 0x6c08);
        re_mdio_write(sc, 0x15, 0x0348);
        re_mdio_write(sc, 0x19, 0x4925);
        re_mdio_write(sc, 0x15, 0x0349);
        re_mdio_write(sc, 0x19, 0x403b);
        re_mdio_write(sc, 0x15, 0x034a);
        re_mdio_write(sc, 0x19, 0xa602);
        re_mdio_write(sc, 0x15, 0x034b);
        re_mdio_write(sc, 0x19, 0x402f);
        re_mdio_write(sc, 0x15, 0x034c);
        re_mdio_write(sc, 0x19, 0x4484);
        re_mdio_write(sc, 0x15, 0x034d);
        re_mdio_write(sc, 0x19, 0x40c8);
        re_mdio_write(sc, 0x15, 0x034e);
        re_mdio_write(sc, 0x19, 0x44c4);
        re_mdio_write(sc, 0x15, 0x034f);
        re_mdio_write(sc, 0x19, 0xd64f);
        re_mdio_write(sc, 0x15, 0x0350);
        re_mdio_write(sc, 0x19, 0x00bd);
        re_mdio_write(sc, 0x15, 0x0351);
        re_mdio_write(sc, 0x19, 0x3311);
        re_mdio_write(sc, 0x15, 0x0352);
        re_mdio_write(sc, 0x19, 0xc8ed);
        re_mdio_write(sc, 0x15, 0x0353);
        re_mdio_write(sc, 0x19, 0x00fc);
        re_mdio_write(sc, 0x15, 0x0354);
        re_mdio_write(sc, 0x19, 0x8221);
        re_mdio_write(sc, 0x15, 0x0355);
        re_mdio_write(sc, 0x19, 0xd11d);
        re_mdio_write(sc, 0x15, 0x0356);
        re_mdio_write(sc, 0x19, 0x001f);
        re_mdio_write(sc, 0x15, 0x0357);
        re_mdio_write(sc, 0x19, 0xde18);
        re_mdio_write(sc, 0x15, 0x0358);
        re_mdio_write(sc, 0x19, 0x0008);
        re_mdio_write(sc, 0x15, 0x0359);
        re_mdio_write(sc, 0x19, 0x91f6);
        re_mdio_write(sc, 0x15, 0x035a);
        re_mdio_write(sc, 0x19, 0x3360);
        re_mdio_write(sc, 0x15, 0x035b);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x035c);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x035d);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x035e);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x035f);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0360);
        re_mdio_write(sc, 0x19, 0x4bb6);
        re_mdio_write(sc, 0x15, 0x0361);
        re_mdio_write(sc, 0x19, 0x4064);
        re_mdio_write(sc, 0x15, 0x0362);
        re_mdio_write(sc, 0x19, 0x4b26);
        re_mdio_write(sc, 0x15, 0x0363);
        re_mdio_write(sc, 0x19, 0x4410);
        re_mdio_write(sc, 0x15, 0x0364);
        re_mdio_write(sc, 0x19, 0x4006);
        re_mdio_write(sc, 0x15, 0x0365);
        re_mdio_write(sc, 0x19, 0x4490);
        re_mdio_write(sc, 0x15, 0x0366);
        re_mdio_write(sc, 0x19, 0x6900);
        re_mdio_write(sc, 0x15, 0x0367);
        re_mdio_write(sc, 0x19, 0xb6a6);
        re_mdio_write(sc, 0x15, 0x0368);
        re_mdio_write(sc, 0x19, 0x9e02);
        re_mdio_write(sc, 0x15, 0x0369);
        re_mdio_write(sc, 0x19, 0x3311);
        re_mdio_write(sc, 0x15, 0x036a);
        re_mdio_write(sc, 0x19, 0xd11d);
        re_mdio_write(sc, 0x15, 0x036b);
        re_mdio_write(sc, 0x19, 0x000a);
        re_mdio_write(sc, 0x15, 0x036c);
        re_mdio_write(sc, 0x19, 0xbb0f);
        re_mdio_write(sc, 0x15, 0x036d);
        re_mdio_write(sc, 0x19, 0x8102);
        re_mdio_write(sc, 0x15, 0x036e);
        re_mdio_write(sc, 0x19, 0x3371);
        re_mdio_write(sc, 0x15, 0x036f);
        re_mdio_write(sc, 0x19, 0xa21e);
        re_mdio_write(sc, 0x15, 0x0370);
        re_mdio_write(sc, 0x19, 0x33b6);
        re_mdio_write(sc, 0x15, 0x0371);
        re_mdio_write(sc, 0x19, 0x91f6);
        re_mdio_write(sc, 0x15, 0x0372);
        re_mdio_write(sc, 0x19, 0xc218);
        re_mdio_write(sc, 0x15, 0x0373);
        re_mdio_write(sc, 0x19, 0x00f4);
        re_mdio_write(sc, 0x15, 0x0374);
        re_mdio_write(sc, 0x19, 0x33b6);
        re_mdio_write(sc, 0x15, 0x0375);
        re_mdio_write(sc, 0x19, 0x32ec);
        re_mdio_write(sc, 0x15, 0x0376);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0377);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0378);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x0379);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x037a);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x037b);
        re_mdio_write(sc, 0x19, 0x4b97);
        re_mdio_write(sc, 0x15, 0x037c);
        re_mdio_write(sc, 0x19, 0x402b);
        re_mdio_write(sc, 0x15, 0x037d);
        re_mdio_write(sc, 0x19, 0x4b07);
        re_mdio_write(sc, 0x15, 0x037e);
        re_mdio_write(sc, 0x19, 0x4422);
        re_mdio_write(sc, 0x15, 0x037f);
        re_mdio_write(sc, 0x19, 0x6980);
        re_mdio_write(sc, 0x15, 0x0380);
        re_mdio_write(sc, 0x19, 0xb608);
        re_mdio_write(sc, 0x15, 0x0381);
        re_mdio_write(sc, 0x19, 0x3311);
        re_mdio_write(sc, 0x15, 0x0382);
        re_mdio_write(sc, 0x19, 0xbc05);
        re_mdio_write(sc, 0x15, 0x0383);
        re_mdio_write(sc, 0x19, 0xc21c);
        re_mdio_write(sc, 0x15, 0x0384);
        re_mdio_write(sc, 0x19, 0x0032);
        re_mdio_write(sc, 0x15, 0x0385);
        re_mdio_write(sc, 0x19, 0xa1fb);
        re_mdio_write(sc, 0x15, 0x0386);
        re_mdio_write(sc, 0x19, 0x338d);
        re_mdio_write(sc, 0x15, 0x0387);
        re_mdio_write(sc, 0x19, 0x32ae);
        re_mdio_write(sc, 0x15, 0x0388);
        re_mdio_write(sc, 0x19, 0x330d);
        re_mdio_write(sc, 0x15, 0x0389);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x038a);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x038b);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x038c);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x038d);
        re_mdio_write(sc, 0x19, 0x4b97);
        re_mdio_write(sc, 0x15, 0x038e);
        re_mdio_write(sc, 0x19, 0x6a08);
        re_mdio_write(sc, 0x15, 0x038f);
        re_mdio_write(sc, 0x19, 0x4b07);
        re_mdio_write(sc, 0x15, 0x0390);
        re_mdio_write(sc, 0x19, 0x40ac);
        re_mdio_write(sc, 0x15, 0x0391);
        re_mdio_write(sc, 0x19, 0x4445);
        re_mdio_write(sc, 0x15, 0x0392);
        re_mdio_write(sc, 0x19, 0x404e);
        re_mdio_write(sc, 0x15, 0x0393);
        re_mdio_write(sc, 0x19, 0x4461);
        re_mdio_write(sc, 0x15, 0x0394);
        re_mdio_write(sc, 0x19, 0x3311);
        re_mdio_write(sc, 0x15, 0x0395);
        re_mdio_write(sc, 0x19, 0x9c0a);
        re_mdio_write(sc, 0x15, 0x0396);
        re_mdio_write(sc, 0x19, 0x63da);
        re_mdio_write(sc, 0x15, 0x0397);
        re_mdio_write(sc, 0x19, 0x6f0c);
        re_mdio_write(sc, 0x15, 0x0398);
        re_mdio_write(sc, 0x19, 0x5440);
        re_mdio_write(sc, 0x15, 0x0399);
        re_mdio_write(sc, 0x19, 0x4b98);
        re_mdio_write(sc, 0x15, 0x039a);
        re_mdio_write(sc, 0x19, 0x7c40);
        re_mdio_write(sc, 0x15, 0x039b);
        re_mdio_write(sc, 0x19, 0x4c00);
        re_mdio_write(sc, 0x15, 0x039c);
        re_mdio_write(sc, 0x19, 0x4b08);
        re_mdio_write(sc, 0x15, 0x039d);
        re_mdio_write(sc, 0x19, 0x63d8);
        re_mdio_write(sc, 0x15, 0x039e);
        re_mdio_write(sc, 0x19, 0x33a5);
        re_mdio_write(sc, 0x15, 0x039f);
        re_mdio_write(sc, 0x19, 0xd64f);
        re_mdio_write(sc, 0x15, 0x03a0);
        re_mdio_write(sc, 0x19, 0x00e8);
        re_mdio_write(sc, 0x15, 0x03a1);
        re_mdio_write(sc, 0x19, 0x820e);
        re_mdio_write(sc, 0x15, 0x03a2);
        re_mdio_write(sc, 0x19, 0xa10d);
        re_mdio_write(sc, 0x15, 0x03a3);
        re_mdio_write(sc, 0x19, 0x9df1);
        re_mdio_write(sc, 0x15, 0x03a4);
        re_mdio_write(sc, 0x19, 0x33af);
        re_mdio_write(sc, 0x15, 0x03a5);
        re_mdio_write(sc, 0x19, 0xd64f);
        re_mdio_write(sc, 0x15, 0x03a6);
        re_mdio_write(sc, 0x19, 0x00f9);
        re_mdio_write(sc, 0x15, 0x03a7);
        re_mdio_write(sc, 0x19, 0xc017);
        re_mdio_write(sc, 0x15, 0x03a8);
        re_mdio_write(sc, 0x19, 0x0007);
        re_mdio_write(sc, 0x15, 0x03a9);
        re_mdio_write(sc, 0x19, 0x7c03);
        re_mdio_write(sc, 0x15, 0x03aa);
        re_mdio_write(sc, 0x19, 0x6c03);
        re_mdio_write(sc, 0x15, 0x03ab);
        re_mdio_write(sc, 0x19, 0xa104);
        re_mdio_write(sc, 0x15, 0x03ac);
        re_mdio_write(sc, 0x19, 0x7c03);
        re_mdio_write(sc, 0x15, 0x03ad);
        re_mdio_write(sc, 0x19, 0x6c00);
        re_mdio_write(sc, 0x15, 0x03ae);
        re_mdio_write(sc, 0x19, 0x9df7);
        re_mdio_write(sc, 0x15, 0x03af);
        re_mdio_write(sc, 0x19, 0x7c03);
        re_mdio_write(sc, 0x15, 0x03b0);
        re_mdio_write(sc, 0x19, 0x6c08);
        re_mdio_write(sc, 0x15, 0x03b1);
        re_mdio_write(sc, 0x19, 0x33b6);
        re_mdio_write(sc, 0x15, 0x03b2);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x03b3);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x03b4);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x03b5);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x03b6);
        re_mdio_write(sc, 0x19, 0x55af);
        re_mdio_write(sc, 0x15, 0x03b7);
        re_mdio_write(sc, 0x19, 0x7ff0);
        re_mdio_write(sc, 0x15, 0x03b8);
        re_mdio_write(sc, 0x19, 0x6ff0);
        re_mdio_write(sc, 0x15, 0x03b9);
        re_mdio_write(sc, 0x19, 0x4bb9);
        re_mdio_write(sc, 0x15, 0x03ba);
        re_mdio_write(sc, 0x19, 0x6a80);
        re_mdio_write(sc, 0x15, 0x03bb);
        re_mdio_write(sc, 0x19, 0x4b29);
        re_mdio_write(sc, 0x15, 0x03bc);
        re_mdio_write(sc, 0x19, 0x4041);
        re_mdio_write(sc, 0x15, 0x03bd);
        re_mdio_write(sc, 0x19, 0x440a);
        re_mdio_write(sc, 0x15, 0x03be);
        re_mdio_write(sc, 0x19, 0x4029);
        re_mdio_write(sc, 0x15, 0x03bf);
        re_mdio_write(sc, 0x19, 0x4418);
        re_mdio_write(sc, 0x15, 0x03c0);
        re_mdio_write(sc, 0x19, 0x4090);
        re_mdio_write(sc, 0x15, 0x03c1);
        re_mdio_write(sc, 0x19, 0x4438);
        re_mdio_write(sc, 0x15, 0x03c2);
        re_mdio_write(sc, 0x19, 0x40c4);
        re_mdio_write(sc, 0x15, 0x03c3);
        re_mdio_write(sc, 0x19, 0x447b);
        re_mdio_write(sc, 0x15, 0x03c4);
        re_mdio_write(sc, 0x19, 0xb6c4);
        re_mdio_write(sc, 0x15, 0x03c5);
        re_mdio_write(sc, 0x19, 0x3311);
        re_mdio_write(sc, 0x15, 0x03c6);
        re_mdio_write(sc, 0x19, 0x9bfe);
        re_mdio_write(sc, 0x15, 0x03c7);
        re_mdio_write(sc, 0x19, 0x33cc);
        re_mdio_write(sc, 0x15, 0x03c8);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x03c9);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x03ca);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x03cb);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x03cc);
        re_mdio_write(sc, 0x19, 0x542f);
        re_mdio_write(sc, 0x15, 0x03cd);
        re_mdio_write(sc, 0x19, 0x499a);
        re_mdio_write(sc, 0x15, 0x03ce);
        re_mdio_write(sc, 0x19, 0x7c40);
        re_mdio_write(sc, 0x15, 0x03cf);
        re_mdio_write(sc, 0x19, 0x4c40);
        re_mdio_write(sc, 0x15, 0x03d0);
        re_mdio_write(sc, 0x19, 0x490a);
        re_mdio_write(sc, 0x15, 0x03d1);
        re_mdio_write(sc, 0x19, 0x405e);
        re_mdio_write(sc, 0x15, 0x03d2);
        re_mdio_write(sc, 0x19, 0x44f8);
        re_mdio_write(sc, 0x15, 0x03d3);
        re_mdio_write(sc, 0x19, 0x6b00);
        re_mdio_write(sc, 0x15, 0x03d4);
        re_mdio_write(sc, 0x19, 0xd64f);
        re_mdio_write(sc, 0x15, 0x03d5);
        re_mdio_write(sc, 0x19, 0x0028);
        re_mdio_write(sc, 0x15, 0x03d6);
        re_mdio_write(sc, 0x19, 0x3311);
        re_mdio_write(sc, 0x15, 0x03d7);
        re_mdio_write(sc, 0x19, 0xbd27);
        re_mdio_write(sc, 0x15, 0x03d8);
        re_mdio_write(sc, 0x19, 0x9cfc);
        re_mdio_write(sc, 0x15, 0x03d9);
        re_mdio_write(sc, 0x19, 0xc639);
        re_mdio_write(sc, 0x15, 0x03da);
        re_mdio_write(sc, 0x19, 0x000f);
        re_mdio_write(sc, 0x15, 0x03db);
        re_mdio_write(sc, 0x19, 0x9e03);
        re_mdio_write(sc, 0x15, 0x03dc);
        re_mdio_write(sc, 0x19, 0x7c01);
        re_mdio_write(sc, 0x15, 0x03dd);
        re_mdio_write(sc, 0x19, 0x4c01);
        re_mdio_write(sc, 0x15, 0x03de);
        re_mdio_write(sc, 0x19, 0x9af6);
        re_mdio_write(sc, 0x15, 0x03df);
        re_mdio_write(sc, 0x19, 0x7c12);
        re_mdio_write(sc, 0x15, 0x03e0);
        re_mdio_write(sc, 0x19, 0x4c52);
        re_mdio_write(sc, 0x15, 0x03e1);
        re_mdio_write(sc, 0x19, 0x4470);
        re_mdio_write(sc, 0x15, 0x03e2);
        re_mdio_write(sc, 0x19, 0x7c12);
        re_mdio_write(sc, 0x15, 0x03e3);
        re_mdio_write(sc, 0x19, 0x4c40);
        re_mdio_write(sc, 0x15, 0x03e4);
        re_mdio_write(sc, 0x19, 0x33d4);
        re_mdio_write(sc, 0x15, 0x03e5);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x03e6);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x03e7);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x03e8);
        re_mdio_write(sc, 0x19, 0x0000);
        re_mdio_write(sc, 0x15, 0x03e9);
        re_mdio_write(sc, 0x19, 0x49bb);
        re_mdio_write(sc, 0x15, 0x03ea);
        re_mdio_write(sc, 0x19, 0x4478);
        re_mdio_write(sc, 0x15, 0x03eb);
        re_mdio_write(sc, 0x19, 0x492b);
        re_mdio_write(sc, 0x15, 0x03ec);
        re_mdio_write(sc, 0x19, 0x6b80);
        re_mdio_write(sc, 0x15, 0x03ed);
        re_mdio_write(sc, 0x19, 0x7c01);
        re_mdio_write(sc, 0x15, 0x03ee);
        re_mdio_write(sc, 0x19, 0x4c00);
        re_mdio_write(sc, 0x15, 0x03ef);
        re_mdio_write(sc, 0x19, 0xd64f);
        re_mdio_write(sc, 0x15, 0x03f0);
        re_mdio_write(sc, 0x19, 0x000d);
        re_mdio_write(sc, 0x15, 0x03f1);
        re_mdio_write(sc, 0x19, 0x3311);
        re_mdio_write(sc, 0x15, 0x03f2);
        re_mdio_write(sc, 0x19, 0xbd0c);
        re_mdio_write(sc, 0x15, 0x03f3);
        re_mdio_write(sc, 0x19, 0xc428);
        re_mdio_write(sc, 0x15, 0x03f4);
        re_mdio_write(sc, 0x19, 0x0008);
        re_mdio_write(sc, 0x15, 0x03f5);
        re_mdio_write(sc, 0x19, 0x9afa);
        re_mdio_write(sc, 0x15, 0x03f6);
        re_mdio_write(sc, 0x19, 0x7c12);
        re_mdio_write(sc, 0x15, 0x03f7);
        re_mdio_write(sc, 0x19, 0x4c52);
        re_mdio_write(sc, 0x15, 0x03f8);
        re_mdio_write(sc, 0x19, 0x4470);
        re_mdio_write(sc, 0x15, 0x03f9);
        re_mdio_write(sc, 0x19, 0x7c12);
        re_mdio_write(sc, 0x15, 0x03fa);
        re_mdio_write(sc, 0x19, 0x4c40);
        re_mdio_write(sc, 0x15, 0x03fb);
        re_mdio_write(sc, 0x19, 0x33ef);
        re_mdio_write(sc, 0x15, 0x03fc);
        re_mdio_write(sc, 0x19, 0x3342);
        re_mdio_write(sc, 0x15, 0x03fd);
        re_mdio_write(sc, 0x19, 0x330d);
        re_mdio_write(sc, 0x15, 0x03fe);
        re_mdio_write(sc, 0x19, 0x32ae);
        re_mdio_write(sc, 0x15, 0x0000);
        re_mdio_write(sc, 0x16, 0x0306);
        re_mdio_write(sc, 0x16, 0x0300);
        re_mdio_write(sc, 0x1f, 0x0002);
        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x1f, 0x0005);
        re_mdio_write(sc, 0x05, 0xfff6);
        re_mdio_write(sc, 0x06, 0x0080);
        re_mdio_write(sc, 0x05, 0x8000);
        re_mdio_write(sc, 0x06, 0x0280);
        re_mdio_write(sc, 0x06, 0x48f7);
        re_mdio_write(sc, 0x06, 0x00e0);
        re_mdio_write(sc, 0x06, 0xfff7);
        re_mdio_write(sc, 0x06, 0xa080);
        re_mdio_write(sc, 0x06, 0x02ae);
        re_mdio_write(sc, 0x06, 0xf602);
        re_mdio_write(sc, 0x06, 0x0112);
        re_mdio_write(sc, 0x06, 0x0201);
        re_mdio_write(sc, 0x06, 0x1f02);
        re_mdio_write(sc, 0x06, 0x012c);
        re_mdio_write(sc, 0x06, 0x0201);
        re_mdio_write(sc, 0x06, 0x3c02);
        re_mdio_write(sc, 0x06, 0x0156);
        re_mdio_write(sc, 0x06, 0x0201);
        re_mdio_write(sc, 0x06, 0x6d02);
        re_mdio_write(sc, 0x06, 0x809d);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x88e1);
        re_mdio_write(sc, 0x06, 0x8b89);
        re_mdio_write(sc, 0x06, 0x1e01);
        re_mdio_write(sc, 0x06, 0xe18b);
        re_mdio_write(sc, 0x06, 0x8a1e);
        re_mdio_write(sc, 0x06, 0x01e1);
        re_mdio_write(sc, 0x06, 0x8b8b);
        re_mdio_write(sc, 0x06, 0x1e01);
        re_mdio_write(sc, 0x06, 0xe18b);
        re_mdio_write(sc, 0x06, 0x8c1e);
        re_mdio_write(sc, 0x06, 0x01e1);
        re_mdio_write(sc, 0x06, 0x8b8d);
        re_mdio_write(sc, 0x06, 0x1e01);
        re_mdio_write(sc, 0x06, 0xe18b);
        re_mdio_write(sc, 0x06, 0x8e1e);
        re_mdio_write(sc, 0x06, 0x01a0);
        re_mdio_write(sc, 0x06, 0x00c7);
        re_mdio_write(sc, 0x06, 0xaebb);
        re_mdio_write(sc, 0x06, 0xd100);
        re_mdio_write(sc, 0x06, 0xbf82);
        re_mdio_write(sc, 0x06, 0xc702);
        re_mdio_write(sc, 0x06, 0x320a);
        re_mdio_write(sc, 0x06, 0xd105);
        re_mdio_write(sc, 0x06, 0xbf82);
        re_mdio_write(sc, 0x06, 0xcd02);
        re_mdio_write(sc, 0x06, 0x320a);
        re_mdio_write(sc, 0x06, 0xd100);
        re_mdio_write(sc, 0x06, 0xbf82);
        re_mdio_write(sc, 0x06, 0xca02);
        re_mdio_write(sc, 0x06, 0x320a);
        re_mdio_write(sc, 0x06, 0xd105);
        re_mdio_write(sc, 0x06, 0xbf82);
        re_mdio_write(sc, 0x06, 0xd002);
        re_mdio_write(sc, 0x06, 0x320a);
        re_mdio_write(sc, 0x06, 0xd481);
        re_mdio_write(sc, 0x06, 0xc9e4);
        re_mdio_write(sc, 0x06, 0x8b90);
        re_mdio_write(sc, 0x06, 0xe58b);
        re_mdio_write(sc, 0x06, 0x91d4);
        re_mdio_write(sc, 0x06, 0x81b8);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x92e5);
        re_mdio_write(sc, 0x06, 0x8b93);
        re_mdio_write(sc, 0x06, 0xbf8b);
        re_mdio_write(sc, 0x06, 0x88ec);
        re_mdio_write(sc, 0x06, 0x0019);
        re_mdio_write(sc, 0x06, 0xa98b);
        re_mdio_write(sc, 0x06, 0x90f9);
        re_mdio_write(sc, 0x06, 0xeeff);
        re_mdio_write(sc, 0x06, 0xf600);
        re_mdio_write(sc, 0x06, 0xeeff);
        re_mdio_write(sc, 0x06, 0xf7fc);
        re_mdio_write(sc, 0x06, 0xd100);
        re_mdio_write(sc, 0x06, 0xbf82);
        re_mdio_write(sc, 0x06, 0xc102);
        re_mdio_write(sc, 0x06, 0x320a);
        re_mdio_write(sc, 0x06, 0xd101);
        re_mdio_write(sc, 0x06, 0xbf82);
        re_mdio_write(sc, 0x06, 0xc402);
        re_mdio_write(sc, 0x06, 0x320a);
        re_mdio_write(sc, 0x06, 0x04f8);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x8ead);
        re_mdio_write(sc, 0x06, 0x201a);
        re_mdio_write(sc, 0x06, 0xf620);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x8e02);
        re_mdio_write(sc, 0x06, 0x824b);
        re_mdio_write(sc, 0x06, 0x0281);
        re_mdio_write(sc, 0x06, 0x1902);
        re_mdio_write(sc, 0x06, 0x2c9d);
        re_mdio_write(sc, 0x06, 0x0203);
        re_mdio_write(sc, 0x06, 0x9602);
        re_mdio_write(sc, 0x06, 0x0473);
        re_mdio_write(sc, 0x06, 0x022e);
        re_mdio_write(sc, 0x06, 0x3902);
        re_mdio_write(sc, 0x06, 0x044d);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x8ead);
        re_mdio_write(sc, 0x06, 0x210b);
        re_mdio_write(sc, 0x06, 0xf621);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x8e02);
        re_mdio_write(sc, 0x06, 0x0416);
        re_mdio_write(sc, 0x06, 0x021b);
        re_mdio_write(sc, 0x06, 0xa4e0);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0xad22);
        re_mdio_write(sc, 0x06, 0x05f6);
        re_mdio_write(sc, 0x06, 0x22e4);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x8ead);
        re_mdio_write(sc, 0x06, 0x2305);
        re_mdio_write(sc, 0x06, 0xf623);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x8ee0);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0xad24);
        re_mdio_write(sc, 0x06, 0x05f6);
        re_mdio_write(sc, 0x06, 0x24e4);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x8ead);
        re_mdio_write(sc, 0x06, 0x2505);
        re_mdio_write(sc, 0x06, 0xf625);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x8ee0);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0xad26);
        re_mdio_write(sc, 0x06, 0x08f6);
        re_mdio_write(sc, 0x06, 0x26e4);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0x0281);
        re_mdio_write(sc, 0x06, 0xdae0);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0xad27);
        re_mdio_write(sc, 0x06, 0x05f6);
        re_mdio_write(sc, 0x06, 0x27e4);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0x0203);
        re_mdio_write(sc, 0x06, 0x5cfc);
        re_mdio_write(sc, 0x06, 0x04f8);
        re_mdio_write(sc, 0x06, 0xfaef);
        re_mdio_write(sc, 0x06, 0x69e0);
        re_mdio_write(sc, 0x06, 0x8b85);
        re_mdio_write(sc, 0x06, 0xad21);
        re_mdio_write(sc, 0x06, 0x57e0);
        re_mdio_write(sc, 0x06, 0xe022);
        re_mdio_write(sc, 0x06, 0xe1e0);
        re_mdio_write(sc, 0x06, 0x2358);
        re_mdio_write(sc, 0x06, 0xc059);
        re_mdio_write(sc, 0x06, 0x021e);
        re_mdio_write(sc, 0x06, 0x01e1);
        re_mdio_write(sc, 0x06, 0x8b3c);
        re_mdio_write(sc, 0x06, 0x1f10);
        re_mdio_write(sc, 0x06, 0x9e44);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x3cad);
        re_mdio_write(sc, 0x06, 0x211d);
        re_mdio_write(sc, 0x06, 0xe18b);
        re_mdio_write(sc, 0x06, 0x84f7);
        re_mdio_write(sc, 0x06, 0x29e5);
        re_mdio_write(sc, 0x06, 0x8b84);
        re_mdio_write(sc, 0x06, 0xac27);
        re_mdio_write(sc, 0x06, 0x0dac);
        re_mdio_write(sc, 0x06, 0x2605);
        re_mdio_write(sc, 0x06, 0x0281);
        re_mdio_write(sc, 0x06, 0x7fae);
        re_mdio_write(sc, 0x06, 0x2b02);
        re_mdio_write(sc, 0x06, 0x2c23);
        re_mdio_write(sc, 0x06, 0xae26);
        re_mdio_write(sc, 0x06, 0x022c);
        re_mdio_write(sc, 0x06, 0x41ae);
        re_mdio_write(sc, 0x06, 0x21e0);
        re_mdio_write(sc, 0x06, 0x8b87);
        re_mdio_write(sc, 0x06, 0xad22);
        re_mdio_write(sc, 0x06, 0x18e0);
        re_mdio_write(sc, 0x06, 0xfff7);
        re_mdio_write(sc, 0x06, 0x58fc);
        re_mdio_write(sc, 0x06, 0xe4ff);
        re_mdio_write(sc, 0x06, 0xf7d1);
        re_mdio_write(sc, 0x06, 0x00bf);
        re_mdio_write(sc, 0x06, 0x2eee);
        re_mdio_write(sc, 0x06, 0x0232);
        re_mdio_write(sc, 0x06, 0x0ad1);
        re_mdio_write(sc, 0x06, 0x00bf);
        re_mdio_write(sc, 0x06, 0x82e8);
        re_mdio_write(sc, 0x06, 0x0232);
        re_mdio_write(sc, 0x06, 0x0a02);
        re_mdio_write(sc, 0x06, 0x2bdf);
        re_mdio_write(sc, 0x06, 0xef96);
        re_mdio_write(sc, 0x06, 0xfefc);
        re_mdio_write(sc, 0x06, 0x04d0);
        re_mdio_write(sc, 0x06, 0x0202);
        re_mdio_write(sc, 0x06, 0x1e97);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x87ad);
        re_mdio_write(sc, 0x06, 0x2228);
        re_mdio_write(sc, 0x06, 0xd100);
        re_mdio_write(sc, 0x06, 0xbf82);
        re_mdio_write(sc, 0x06, 0xd302);
        re_mdio_write(sc, 0x06, 0x320a);
        re_mdio_write(sc, 0x06, 0xd10c);
        re_mdio_write(sc, 0x06, 0xbf82);
        re_mdio_write(sc, 0x06, 0xd602);
        re_mdio_write(sc, 0x06, 0x320a);
        re_mdio_write(sc, 0x06, 0xd104);
        re_mdio_write(sc, 0x06, 0xbf82);
        re_mdio_write(sc, 0x06, 0xd902);
        re_mdio_write(sc, 0x06, 0x320a);
        re_mdio_write(sc, 0x06, 0xd101);
        re_mdio_write(sc, 0x06, 0xbf82);
        re_mdio_write(sc, 0x06, 0xe802);
        re_mdio_write(sc, 0x06, 0x320a);
        re_mdio_write(sc, 0x06, 0xe0ff);
        re_mdio_write(sc, 0x06, 0xf768);
        re_mdio_write(sc, 0x06, 0x03e4);
        re_mdio_write(sc, 0x06, 0xfff7);
        re_mdio_write(sc, 0x06, 0xd004);
        re_mdio_write(sc, 0x06, 0x0228);
        re_mdio_write(sc, 0x06, 0x7a04);
        re_mdio_write(sc, 0x06, 0xf8e0);
        re_mdio_write(sc, 0x06, 0xe234);
        re_mdio_write(sc, 0x06, 0xe1e2);
        re_mdio_write(sc, 0x06, 0x35f6);
        re_mdio_write(sc, 0x06, 0x2be4);
        re_mdio_write(sc, 0x06, 0xe234);
        re_mdio_write(sc, 0x06, 0xe5e2);
        re_mdio_write(sc, 0x06, 0x35fc);
        re_mdio_write(sc, 0x06, 0x05f8);
        re_mdio_write(sc, 0x06, 0xe0e2);
        re_mdio_write(sc, 0x06, 0x34e1);
        re_mdio_write(sc, 0x06, 0xe235);
        re_mdio_write(sc, 0x06, 0xf72b);
        re_mdio_write(sc, 0x06, 0xe4e2);
        re_mdio_write(sc, 0x06, 0x34e5);
        re_mdio_write(sc, 0x06, 0xe235);
        re_mdio_write(sc, 0x06, 0xfc05);
        re_mdio_write(sc, 0x06, 0xf8f9);
        re_mdio_write(sc, 0x06, 0xfaef);
        re_mdio_write(sc, 0x06, 0x69ac);
        re_mdio_write(sc, 0x06, 0x1b4c);
        re_mdio_write(sc, 0x06, 0xbf2e);
        re_mdio_write(sc, 0x06, 0x3002);
        re_mdio_write(sc, 0x06, 0x31dd);
        re_mdio_write(sc, 0x06, 0xef01);
        re_mdio_write(sc, 0x06, 0xe28a);
        re_mdio_write(sc, 0x06, 0x76e4);
        re_mdio_write(sc, 0x06, 0x8a76);
        re_mdio_write(sc, 0x06, 0x1f12);
        re_mdio_write(sc, 0x06, 0x9e3a);
        re_mdio_write(sc, 0x06, 0xef12);
        re_mdio_write(sc, 0x06, 0x5907);
        re_mdio_write(sc, 0x06, 0x9f12);
        re_mdio_write(sc, 0x06, 0xf8e0);
        re_mdio_write(sc, 0x06, 0x8b40);
        re_mdio_write(sc, 0x06, 0xf721);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x40d0);
        re_mdio_write(sc, 0x06, 0x0302);
        re_mdio_write(sc, 0x06, 0x287a);
        re_mdio_write(sc, 0x06, 0x0282);
        re_mdio_write(sc, 0x06, 0x34fc);
        re_mdio_write(sc, 0x06, 0xa000);
        re_mdio_write(sc, 0x06, 0x1002);
        re_mdio_write(sc, 0x06, 0x2dc3);
        re_mdio_write(sc, 0x06, 0x022e);
        re_mdio_write(sc, 0x06, 0x21e0);
        re_mdio_write(sc, 0x06, 0x8b40);
        re_mdio_write(sc, 0x06, 0xf621);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x40ae);
        re_mdio_write(sc, 0x06, 0x0fbf);
        re_mdio_write(sc, 0x06, 0x3fa5);
        re_mdio_write(sc, 0x06, 0x0231);
        re_mdio_write(sc, 0x06, 0x6cbf);
        re_mdio_write(sc, 0x06, 0x3fa2);
        re_mdio_write(sc, 0x06, 0x0231);
        re_mdio_write(sc, 0x06, 0x6c02);
        re_mdio_write(sc, 0x06, 0x2dc3);
        re_mdio_write(sc, 0x06, 0xef96);
        re_mdio_write(sc, 0x06, 0xfefd);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xf8e0);
        re_mdio_write(sc, 0x06, 0xe2f4);
        re_mdio_write(sc, 0x06, 0xe1e2);
        re_mdio_write(sc, 0x06, 0xf5e4);
        re_mdio_write(sc, 0x06, 0x8a78);
        re_mdio_write(sc, 0x06, 0xe58a);
        re_mdio_write(sc, 0x06, 0x79ee);
        re_mdio_write(sc, 0x06, 0xe2f4);
        re_mdio_write(sc, 0x06, 0xd8ee);
        re_mdio_write(sc, 0x06, 0xe2f5);
        re_mdio_write(sc, 0x06, 0x20fc);
        re_mdio_write(sc, 0x06, 0x04f8);
        re_mdio_write(sc, 0x06, 0xf9fa);
        re_mdio_write(sc, 0x06, 0xef69);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x87ad);
        re_mdio_write(sc, 0x06, 0x2065);
        re_mdio_write(sc, 0x06, 0xd200);
        re_mdio_write(sc, 0x06, 0xbf2e);
        re_mdio_write(sc, 0x06, 0xe802);
        re_mdio_write(sc, 0x06, 0x31dd);
        re_mdio_write(sc, 0x06, 0x1e21);
        re_mdio_write(sc, 0x06, 0xbf82);
        re_mdio_write(sc, 0x06, 0xdf02);
        re_mdio_write(sc, 0x06, 0x31dd);
        re_mdio_write(sc, 0x06, 0x0c11);
        re_mdio_write(sc, 0x06, 0x1e21);
        re_mdio_write(sc, 0x06, 0xbf82);
        re_mdio_write(sc, 0x06, 0xe202);
        re_mdio_write(sc, 0x06, 0x31dd);
        re_mdio_write(sc, 0x06, 0x0c12);
        re_mdio_write(sc, 0x06, 0x1e21);
        re_mdio_write(sc, 0x06, 0xbf82);
        re_mdio_write(sc, 0x06, 0xe502);
        re_mdio_write(sc, 0x06, 0x31dd);
        re_mdio_write(sc, 0x06, 0x0c13);
        re_mdio_write(sc, 0x06, 0x1e21);
        re_mdio_write(sc, 0x06, 0xbf1f);
        re_mdio_write(sc, 0x06, 0x5302);
        re_mdio_write(sc, 0x06, 0x31dd);
        re_mdio_write(sc, 0x06, 0x0c14);
        re_mdio_write(sc, 0x06, 0x1e21);
        re_mdio_write(sc, 0x06, 0xbf82);
        re_mdio_write(sc, 0x06, 0xeb02);
        re_mdio_write(sc, 0x06, 0x31dd);
        re_mdio_write(sc, 0x06, 0x0c16);
        re_mdio_write(sc, 0x06, 0x1e21);
        re_mdio_write(sc, 0x06, 0xe083);
        re_mdio_write(sc, 0x06, 0xe01f);
        re_mdio_write(sc, 0x06, 0x029e);
        re_mdio_write(sc, 0x06, 0x22e6);
        re_mdio_write(sc, 0x06, 0x83e0);
        re_mdio_write(sc, 0x06, 0xad31);
        re_mdio_write(sc, 0x06, 0x14ad);
        re_mdio_write(sc, 0x06, 0x3011);
        re_mdio_write(sc, 0x06, 0xef02);
        re_mdio_write(sc, 0x06, 0x580c);
        re_mdio_write(sc, 0x06, 0x9e07);
        re_mdio_write(sc, 0x06, 0xad36);
        re_mdio_write(sc, 0x06, 0x085a);
        re_mdio_write(sc, 0x06, 0x309f);
        re_mdio_write(sc, 0x06, 0x04d1);
        re_mdio_write(sc, 0x06, 0x01ae);
        re_mdio_write(sc, 0x06, 0x02d1);
        re_mdio_write(sc, 0x06, 0x00bf);
        re_mdio_write(sc, 0x06, 0x82dc);
        re_mdio_write(sc, 0x06, 0x0232);
        re_mdio_write(sc, 0x06, 0x0aef);
        re_mdio_write(sc, 0x06, 0x96fe);
        re_mdio_write(sc, 0x06, 0xfdfc);
        re_mdio_write(sc, 0x06, 0x0400);
        re_mdio_write(sc, 0x06, 0xe140);
        re_mdio_write(sc, 0x06, 0x77e1);
        re_mdio_write(sc, 0x06, 0x4010);
        re_mdio_write(sc, 0x06, 0xe150);
        re_mdio_write(sc, 0x06, 0x32e1);
        re_mdio_write(sc, 0x06, 0x5030);
        re_mdio_write(sc, 0x06, 0xe144);
        re_mdio_write(sc, 0x06, 0x74e1);
        re_mdio_write(sc, 0x06, 0x44bb);
        re_mdio_write(sc, 0x06, 0xe2d2);
        re_mdio_write(sc, 0x06, 0x40e0);
        re_mdio_write(sc, 0x06, 0x2cfc);
        re_mdio_write(sc, 0x06, 0xe2cc);
        re_mdio_write(sc, 0x06, 0xcce2);
        re_mdio_write(sc, 0x06, 0x00cc);
        re_mdio_write(sc, 0x06, 0xe000);
        re_mdio_write(sc, 0x06, 0x99e0);
        re_mdio_write(sc, 0x06, 0x3688);
        re_mdio_write(sc, 0x06, 0xe036);
        re_mdio_write(sc, 0x06, 0x99e1);
        re_mdio_write(sc, 0x06, 0x40dd);
        re_mdio_write(sc, 0x06, 0xe022);
        re_mdio_write(sc, 0x05, 0xe142);
        PhyRegValue = re_mdio_read(sc, 0x06);
        PhyRegValue |= BIT_0;
        re_mdio_write(sc, 0x06, PhyRegValue);
        re_mdio_write(sc, 0x05, 0xe140);
        PhyRegValue = re_mdio_read(sc, 0x06);
        PhyRegValue |= BIT_0;
        re_mdio_write(sc, 0x06, PhyRegValue);
        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x1f, 0x0005);
        for (i = 0; i < 200; i++) {
                DELAY(100);
                PhyRegValue = re_mdio_read(sc, 0x00);
                if (PhyRegValue & BIT_7)
                        break;
        }

        re_mdio_write(sc, 0x1F, 0x0001);
        re_mdio_write(sc, 0x0B, 0x6C14);
        re_mdio_write(sc, 0x14, 0x7F3D);
        re_mdio_write(sc, 0x1C, 0xFAFE);
        re_mdio_write(sc, 0x08, 0x07C5);
        re_mdio_write(sc, 0x10, 0xF090);
        re_mdio_write(sc, 0x1F, 0x0003);
        re_mdio_write(sc, 0x14, 0x641A);
        re_mdio_write(sc, 0x1A, 0x0606);
        re_mdio_write(sc, 0x12, 0xF480);
        re_mdio_write(sc, 0x13, 0x0747);
        re_mdio_write(sc, 0x1F, 0x0000);

        re_mdio_write(sc, 0x1F, 0x0003);
        re_mdio_write(sc, 0x0D, 0x0207);
        re_mdio_write(sc, 0x02, 0x5FD0);
        re_mdio_write(sc, 0x1F, 0x0000);

        re_mdio_write(sc, 0x1F, 0x0003);
        re_mdio_write(sc, 0x09, 0xA20F);
        re_mdio_write(sc, 0x1F, 0x0000);

        re_mdio_write(sc, 0x1f, 0x0003);
        PhyRegValue = re_mdio_read(sc, 0x19);
        PhyRegValue &= ~BIT_0;
        re_mdio_write(sc, 0x19, PhyRegValue);
        PhyRegValue = re_mdio_read(sc, 0x10);
        PhyRegValue &= ~BIT_10;
        re_mdio_write(sc, 0x10, PhyRegValue);
        re_mdio_write(sc, 0x1f, 0x0000);

        re_mdio_write(sc, 0x1f, 0x0004);
        re_mdio_write(sc, 0x1f, 0x0007);
        re_mdio_write(sc, 0x1e, 0x0023);
        PhyRegValue = re_mdio_read(sc, 0x17);
        PhyRegValue &= ~BIT_0;
        PhyRegValue |= BIT_2;
        re_mdio_write(sc, 0x17, PhyRegValue);
        re_mdio_write(sc, 0x1f, 0x0002);
        re_mdio_write(sc, 0x1f, 0x0000);
}

static void re_set_phy_mcu_8168evl_2(struct re_softc *sc)
{
        u_int16_t PhyRegValue;
        int i;

        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x00, 0x1800);
        PhyRegValue = re_mdio_read(sc, 0x15);
        PhyRegValue &= ~(BIT_12);
        re_mdio_write(sc, 0x15, PhyRegValue);
        re_mdio_write(sc, 0x00, 0x4800);
        re_mdio_write(sc, 0x1f, 0x0007);
        re_mdio_write(sc, 0x1e, 0x002f);
        for (i = 0; i < 1000; i++) {
                if (re_mdio_read(sc, 0x1c) & BIT_7)
                        break;
                DELAY(100);
        }
        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x00, 0x1800);
        re_mdio_write(sc, 0x1f, 0x0007);
        re_mdio_write(sc, 0x1e, 0x0023);
        for (i = 0; i < 200; i++) {
                if ((re_mdio_read(sc, 0x17) & BIT_0) == 0)
                        break;
                DELAY(100);
        }
        re_mdio_write(sc, 0x1f, 0x0005);
        re_mdio_write(sc, 0x05, 0xfff6);
        re_mdio_write(sc, 0x06, 0x0080);
        re_mdio_write(sc, 0x1f, 0x0007);
        re_mdio_write(sc, 0x1e, 0x0023);
        re_mdio_write(sc, 0x16, 0x0306);
        re_mdio_write(sc, 0x16, 0x0307);
        re_mdio_write(sc, 0x15, 0x00AF);
        re_mdio_write(sc, 0x19, 0x4060);
        re_mdio_write(sc, 0x15, 0x00B0);
        re_mdio_write(sc, 0x19, 0x7800);
        re_mdio_write(sc, 0x15, 0x00B1);
        re_mdio_write(sc, 0x19, 0x7e00);
        re_mdio_write(sc, 0x15, 0x00B2);
        re_mdio_write(sc, 0x19, 0x72B0);
        re_mdio_write(sc, 0x15, 0x00B3);
        re_mdio_write(sc, 0x19, 0x7F00);
        re_mdio_write(sc, 0x15, 0x00B4);
        re_mdio_write(sc, 0x19, 0x73B0);
        re_mdio_write(sc, 0x15, 0x0101);
        re_mdio_write(sc, 0x19, 0x0005);
        re_mdio_write(sc, 0x15, 0x0103);
        re_mdio_write(sc, 0x19, 0x0003);
        re_mdio_write(sc, 0x15, 0x0105);
        re_mdio_write(sc, 0x19, 0x30FD);
        re_mdio_write(sc, 0x15, 0x0106);
        re_mdio_write(sc, 0x19, 0x9DF7);
        re_mdio_write(sc, 0x15, 0x0107);
        re_mdio_write(sc, 0x19, 0x30C6);
        re_mdio_write(sc, 0x15, 0x0098);
        re_mdio_write(sc, 0x19, 0x7c0b);
        re_mdio_write(sc, 0x15, 0x0099);
        re_mdio_write(sc, 0x19, 0x6c0b);
        re_mdio_write(sc, 0x15, 0x00eb);
        re_mdio_write(sc, 0x19, 0x6c0b);
        re_mdio_write(sc, 0x15, 0x00f8);
        re_mdio_write(sc, 0x19, 0x6f0b);
        re_mdio_write(sc, 0x15, 0x00fe);
        re_mdio_write(sc, 0x19, 0x6f0f);
        re_mdio_write(sc, 0x15, 0x00db);
        re_mdio_write(sc, 0x19, 0x6f09);
        re_mdio_write(sc, 0x15, 0x00dc);
        re_mdio_write(sc, 0x19, 0xaefd);
        re_mdio_write(sc, 0x15, 0x00dd);
        re_mdio_write(sc, 0x19, 0x6f0b);
        re_mdio_write(sc, 0x15, 0x00de);
        re_mdio_write(sc, 0x19, 0xc60b);
        re_mdio_write(sc, 0x15, 0x00df);
        re_mdio_write(sc, 0x19, 0x00fa);
        re_mdio_write(sc, 0x15, 0x00e0);
        re_mdio_write(sc, 0x19, 0x30e1);
        re_mdio_write(sc, 0x15, 0x020c);
        re_mdio_write(sc, 0x19, 0x3224);
        re_mdio_write(sc, 0x15, 0x020e);
        re_mdio_write(sc, 0x19, 0x9813);
        re_mdio_write(sc, 0x15, 0x020f);
        re_mdio_write(sc, 0x19, 0x7801);
        re_mdio_write(sc, 0x15, 0x0210);
        re_mdio_write(sc, 0x19, 0x930f);
        re_mdio_write(sc, 0x15, 0x0211);
        re_mdio_write(sc, 0x19, 0x9206);
        re_mdio_write(sc, 0x15, 0x0212);
        re_mdio_write(sc, 0x19, 0x4002);
        re_mdio_write(sc, 0x15, 0x0213);
        re_mdio_write(sc, 0x19, 0x7800);
        re_mdio_write(sc, 0x15, 0x0214);
        re_mdio_write(sc, 0x19, 0x588f);
        re_mdio_write(sc, 0x15, 0x0215);
        re_mdio_write(sc, 0x19, 0x5520);
        re_mdio_write(sc, 0x15, 0x0216);
        re_mdio_write(sc, 0x19, 0x3224);
        re_mdio_write(sc, 0x15, 0x0217);
        re_mdio_write(sc, 0x19, 0x4002);
        re_mdio_write(sc, 0x15, 0x0218);
        re_mdio_write(sc, 0x19, 0x7800);
        re_mdio_write(sc, 0x15, 0x0219);
        re_mdio_write(sc, 0x19, 0x588d);
        re_mdio_write(sc, 0x15, 0x021a);
        re_mdio_write(sc, 0x19, 0x5540);
        re_mdio_write(sc, 0x15, 0x021b);
        re_mdio_write(sc, 0x19, 0x9e03);
        re_mdio_write(sc, 0x15, 0x021c);
        re_mdio_write(sc, 0x19, 0x7c40);
        re_mdio_write(sc, 0x15, 0x021d);
        re_mdio_write(sc, 0x19, 0x6840);
        re_mdio_write(sc, 0x15, 0x021e);
        re_mdio_write(sc, 0x19, 0x3224);
        re_mdio_write(sc, 0x15, 0x021f);
        re_mdio_write(sc, 0x19, 0x4002);
        re_mdio_write(sc, 0x15, 0x0220);
        re_mdio_write(sc, 0x19, 0x3224);
        re_mdio_write(sc, 0x15, 0x0221);
        re_mdio_write(sc, 0x19, 0x9e03);
        re_mdio_write(sc, 0x15, 0x0222);
        re_mdio_write(sc, 0x19, 0x7c40);
        re_mdio_write(sc, 0x15, 0x0223);
        re_mdio_write(sc, 0x19, 0x6840);
        re_mdio_write(sc, 0x15, 0x0224);
        re_mdio_write(sc, 0x19, 0x7800);
        re_mdio_write(sc, 0x15, 0x0225);
        re_mdio_write(sc, 0x19, 0x3231);
        re_mdio_write(sc, 0x15, 0x0000);
        re_mdio_write(sc, 0x16, 0x0306);
        re_mdio_write(sc, 0x16, 0x0300);
        re_mdio_write(sc, 0x1f, 0x0002);
        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x17, 0x2160);
        re_mdio_write(sc, 0x1f, 0x0007);
        re_mdio_write(sc, 0x1e, 0x0040);
        re_mdio_write(sc, 0x18, 0x0004);
        re_mdio_write(sc, 0x18, 0x09d4);
        re_mdio_write(sc, 0x19, 0x4000);
        re_mdio_write(sc, 0x18, 0x09e4);
        re_mdio_write(sc, 0x19, 0x0800);
        re_mdio_write(sc, 0x18, 0x09f4);
        re_mdio_write(sc, 0x19, 0xff00);
        re_mdio_write(sc, 0x18, 0x0a04);
        re_mdio_write(sc, 0x19, 0x4000);
        re_mdio_write(sc, 0x18, 0x0a14);
        re_mdio_write(sc, 0x19, 0x0c00);
        re_mdio_write(sc, 0x18, 0x0a24);
        re_mdio_write(sc, 0x19, 0xff00);
        re_mdio_write(sc, 0x18, 0x0a74);
        re_mdio_write(sc, 0x19, 0xf600);
        re_mdio_write(sc, 0x18, 0x1a24);
        re_mdio_write(sc, 0x19, 0x7d00);
        re_mdio_write(sc, 0x18, 0x1a64);
        re_mdio_write(sc, 0x19, 0x0500);
        re_mdio_write(sc, 0x18, 0x1a74);
        re_mdio_write(sc, 0x19, 0x9500);
        re_mdio_write(sc, 0x18, 0x1a84);
        re_mdio_write(sc, 0x19, 0x8000);
        re_mdio_write(sc, 0x18, 0x1a94);
        re_mdio_write(sc, 0x19, 0x7d00);
        re_mdio_write(sc, 0x18, 0x1aa4);
        re_mdio_write(sc, 0x19, 0x9600);
        re_mdio_write(sc, 0x18, 0x1ac4);
        re_mdio_write(sc, 0x19, 0x4000);
        re_mdio_write(sc, 0x18, 0x1ad4);
        re_mdio_write(sc, 0x19, 0x0800);
        re_mdio_write(sc, 0x18, 0x1af4);
        re_mdio_write(sc, 0x19, 0xc400);
        re_mdio_write(sc, 0x18, 0x1b04);
        re_mdio_write(sc, 0x19, 0x4000);
        re_mdio_write(sc, 0x18, 0x1b14);
        re_mdio_write(sc, 0x19, 0x0800);
        re_mdio_write(sc, 0x18, 0x1b24);
        re_mdio_write(sc, 0x19, 0xfd00);
        re_mdio_write(sc, 0x18, 0x1b34);
        re_mdio_write(sc, 0x19, 0x4000);
        re_mdio_write(sc, 0x18, 0x1b44);
        re_mdio_write(sc, 0x19, 0x0400);
        re_mdio_write(sc, 0x18, 0x1b94);
        re_mdio_write(sc, 0x19, 0xf100);
        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x17, 0x2100);
        re_mdio_write(sc, 0x1f, 0x0007);
        re_mdio_write(sc, 0x1e, 0x0040);
        re_mdio_write(sc, 0x18, 0x0000);
        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x1f, 0x0005);
        re_mdio_write(sc, 0x05, 0xfff6);
        re_mdio_write(sc, 0x06, 0x0080);
        re_mdio_write(sc, 0x05, 0x8000);
        re_mdio_write(sc, 0x06, 0x0280);
        re_mdio_write(sc, 0x06, 0x48f7);
        re_mdio_write(sc, 0x06, 0x00e0);
        re_mdio_write(sc, 0x06, 0xfff7);
        re_mdio_write(sc, 0x06, 0xa080);
        re_mdio_write(sc, 0x06, 0x02ae);
        re_mdio_write(sc, 0x06, 0xf602);
        re_mdio_write(sc, 0x06, 0x0115);
        re_mdio_write(sc, 0x06, 0x0201);
        re_mdio_write(sc, 0x06, 0x2202);
        re_mdio_write(sc, 0x06, 0x80a0);
        re_mdio_write(sc, 0x06, 0x0201);
        re_mdio_write(sc, 0x06, 0x3f02);
        re_mdio_write(sc, 0x06, 0x0159);
        re_mdio_write(sc, 0x06, 0x0280);
        re_mdio_write(sc, 0x06, 0xbd02);
        re_mdio_write(sc, 0x06, 0x80da);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x88e1);
        re_mdio_write(sc, 0x06, 0x8b89);
        re_mdio_write(sc, 0x06, 0x1e01);
        re_mdio_write(sc, 0x06, 0xe18b);
        re_mdio_write(sc, 0x06, 0x8a1e);
        re_mdio_write(sc, 0x06, 0x01e1);
        re_mdio_write(sc, 0x06, 0x8b8b);
        re_mdio_write(sc, 0x06, 0x1e01);
        re_mdio_write(sc, 0x06, 0xe18b);
        re_mdio_write(sc, 0x06, 0x8c1e);
        re_mdio_write(sc, 0x06, 0x01e1);
        re_mdio_write(sc, 0x06, 0x8b8d);
        re_mdio_write(sc, 0x06, 0x1e01);
        re_mdio_write(sc, 0x06, 0xe18b);
        re_mdio_write(sc, 0x06, 0x8e1e);
        re_mdio_write(sc, 0x06, 0x01a0);
        re_mdio_write(sc, 0x06, 0x00c7);
        re_mdio_write(sc, 0x06, 0xaebb);
        re_mdio_write(sc, 0x06, 0xd481);
        re_mdio_write(sc, 0x06, 0xd2e4);
        re_mdio_write(sc, 0x06, 0x8b92);
        re_mdio_write(sc, 0x06, 0xe58b);
        re_mdio_write(sc, 0x06, 0x93d1);
        re_mdio_write(sc, 0x06, 0x03bf);
        re_mdio_write(sc, 0x06, 0x859e);
        re_mdio_write(sc, 0x06, 0x0237);
        re_mdio_write(sc, 0x06, 0x23d1);
        re_mdio_write(sc, 0x06, 0x02bf);
        re_mdio_write(sc, 0x06, 0x85a1);
        re_mdio_write(sc, 0x06, 0x0237);
        re_mdio_write(sc, 0x06, 0x23ee);
        re_mdio_write(sc, 0x06, 0x8608);
        re_mdio_write(sc, 0x06, 0x03ee);
        re_mdio_write(sc, 0x06, 0x860a);
        re_mdio_write(sc, 0x06, 0x60ee);
        re_mdio_write(sc, 0x06, 0x8610);
        re_mdio_write(sc, 0x06, 0x00ee);
        re_mdio_write(sc, 0x06, 0x8611);
        re_mdio_write(sc, 0x06, 0x00ee);
        re_mdio_write(sc, 0x06, 0x8abe);
        re_mdio_write(sc, 0x06, 0x07ee);
        re_mdio_write(sc, 0x06, 0x8abf);
        re_mdio_write(sc, 0x06, 0x73ee);
        re_mdio_write(sc, 0x06, 0x8a95);
        re_mdio_write(sc, 0x06, 0x02bf);
        re_mdio_write(sc, 0x06, 0x8b88);
        re_mdio_write(sc, 0x06, 0xec00);
        re_mdio_write(sc, 0x06, 0x19a9);
        re_mdio_write(sc, 0x06, 0x8b90);
        re_mdio_write(sc, 0x06, 0xf9ee);
        re_mdio_write(sc, 0x06, 0xfff6);
        re_mdio_write(sc, 0x06, 0x00ee);
        re_mdio_write(sc, 0x06, 0xfff7);
        re_mdio_write(sc, 0x06, 0xfed1);
        re_mdio_write(sc, 0x06, 0x00bf);
        re_mdio_write(sc, 0x06, 0x8595);
        re_mdio_write(sc, 0x06, 0x0237);
        re_mdio_write(sc, 0x06, 0x23d1);
        re_mdio_write(sc, 0x06, 0x01bf);
        re_mdio_write(sc, 0x06, 0x8598);
        re_mdio_write(sc, 0x06, 0x0237);
        re_mdio_write(sc, 0x06, 0x2304);
        re_mdio_write(sc, 0x06, 0xf8e0);
        re_mdio_write(sc, 0x06, 0x8b8a);
        re_mdio_write(sc, 0x06, 0xad20);
        re_mdio_write(sc, 0x06, 0x14ee);
        re_mdio_write(sc, 0x06, 0x8b8a);
        re_mdio_write(sc, 0x06, 0x0002);
        re_mdio_write(sc, 0x06, 0x1f9a);
        re_mdio_write(sc, 0x06, 0xe0e4);
        re_mdio_write(sc, 0x06, 0x26e1);
        re_mdio_write(sc, 0x06, 0xe427);
        re_mdio_write(sc, 0x06, 0xeee4);
        re_mdio_write(sc, 0x06, 0x2623);
        re_mdio_write(sc, 0x06, 0xe5e4);
        re_mdio_write(sc, 0x06, 0x27fc);
        re_mdio_write(sc, 0x06, 0x04f8);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x8dad);
        re_mdio_write(sc, 0x06, 0x2014);
        re_mdio_write(sc, 0x06, 0xee8b);
        re_mdio_write(sc, 0x06, 0x8d00);
        re_mdio_write(sc, 0x06, 0xe08a);
        re_mdio_write(sc, 0x06, 0x5a78);
        re_mdio_write(sc, 0x06, 0x039e);
        re_mdio_write(sc, 0x06, 0x0902);
        re_mdio_write(sc, 0x06, 0x05db);
        re_mdio_write(sc, 0x06, 0x0282);
        re_mdio_write(sc, 0x06, 0x7b02);
        re_mdio_write(sc, 0x06, 0x3231);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xf8e0);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0xad20);
        re_mdio_write(sc, 0x06, 0x1df6);
        re_mdio_write(sc, 0x06, 0x20e4);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0x0281);
        re_mdio_write(sc, 0x06, 0x5c02);
        re_mdio_write(sc, 0x06, 0x2bcb);
        re_mdio_write(sc, 0x06, 0x022d);
        re_mdio_write(sc, 0x06, 0x2902);
        re_mdio_write(sc, 0x06, 0x03b4);
        re_mdio_write(sc, 0x06, 0x0285);
        re_mdio_write(sc, 0x06, 0x6402);
        re_mdio_write(sc, 0x06, 0x2eca);
        re_mdio_write(sc, 0x06, 0x0284);
        re_mdio_write(sc, 0x06, 0xcd02);
        re_mdio_write(sc, 0x06, 0x046f);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x8ead);
        re_mdio_write(sc, 0x06, 0x210b);
        re_mdio_write(sc, 0x06, 0xf621);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x8e02);
        re_mdio_write(sc, 0x06, 0x8520);
        re_mdio_write(sc, 0x06, 0x021b);
        re_mdio_write(sc, 0x06, 0xe8e0);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0xad22);
        re_mdio_write(sc, 0x06, 0x05f6);
        re_mdio_write(sc, 0x06, 0x22e4);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x8ead);
        re_mdio_write(sc, 0x06, 0x2308);
        re_mdio_write(sc, 0x06, 0xf623);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x8e02);
        re_mdio_write(sc, 0x06, 0x311c);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x8ead);
        re_mdio_write(sc, 0x06, 0x2405);
        re_mdio_write(sc, 0x06, 0xf624);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x8ee0);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0xad25);
        re_mdio_write(sc, 0x06, 0x05f6);
        re_mdio_write(sc, 0x06, 0x25e4);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x8ead);
        re_mdio_write(sc, 0x06, 0x2608);
        re_mdio_write(sc, 0x06, 0xf626);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x8e02);
        re_mdio_write(sc, 0x06, 0x2df5);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x8ead);
        re_mdio_write(sc, 0x06, 0x2705);
        re_mdio_write(sc, 0x06, 0xf627);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x8e02);
        re_mdio_write(sc, 0x06, 0x037a);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xf8f9);
        re_mdio_write(sc, 0x06, 0xfaef);
        re_mdio_write(sc, 0x06, 0x69e0);
        re_mdio_write(sc, 0x06, 0x8b87);
        re_mdio_write(sc, 0x06, 0xad20);
        re_mdio_write(sc, 0x06, 0x65d2);
        re_mdio_write(sc, 0x06, 0x00bf);
        re_mdio_write(sc, 0x06, 0x2fe9);
        re_mdio_write(sc, 0x06, 0x0236);
        re_mdio_write(sc, 0x06, 0xf61e);
        re_mdio_write(sc, 0x06, 0x21bf);
        re_mdio_write(sc, 0x06, 0x2ff5);
        re_mdio_write(sc, 0x06, 0x0236);
        re_mdio_write(sc, 0x06, 0xf60c);
        re_mdio_write(sc, 0x06, 0x111e);
        re_mdio_write(sc, 0x06, 0x21bf);
        re_mdio_write(sc, 0x06, 0x2ff8);
        re_mdio_write(sc, 0x06, 0x0236);
        re_mdio_write(sc, 0x06, 0xf60c);
        re_mdio_write(sc, 0x06, 0x121e);
        re_mdio_write(sc, 0x06, 0x21bf);
        re_mdio_write(sc, 0x06, 0x2ffb);
        re_mdio_write(sc, 0x06, 0x0236);
        re_mdio_write(sc, 0x06, 0xf60c);
        re_mdio_write(sc, 0x06, 0x131e);
        re_mdio_write(sc, 0x06, 0x21bf);
        re_mdio_write(sc, 0x06, 0x1f97);
        re_mdio_write(sc, 0x06, 0x0236);
        re_mdio_write(sc, 0x06, 0xf60c);
        re_mdio_write(sc, 0x06, 0x141e);
        re_mdio_write(sc, 0x06, 0x21bf);
        re_mdio_write(sc, 0x06, 0x859b);
        re_mdio_write(sc, 0x06, 0x0236);
        re_mdio_write(sc, 0x06, 0xf60c);
        re_mdio_write(sc, 0x06, 0x161e);
        re_mdio_write(sc, 0x06, 0x21e0);
        re_mdio_write(sc, 0x06, 0x8a8c);
        re_mdio_write(sc, 0x06, 0x1f02);
        re_mdio_write(sc, 0x06, 0x9e22);
        re_mdio_write(sc, 0x06, 0xe68a);
        re_mdio_write(sc, 0x06, 0x8cad);
        re_mdio_write(sc, 0x06, 0x3114);
        re_mdio_write(sc, 0x06, 0xad30);
        re_mdio_write(sc, 0x06, 0x11ef);
        re_mdio_write(sc, 0x06, 0x0258);
        re_mdio_write(sc, 0x06, 0x0c9e);
        re_mdio_write(sc, 0x06, 0x07ad);
        re_mdio_write(sc, 0x06, 0x3608);
        re_mdio_write(sc, 0x06, 0x5a30);
        re_mdio_write(sc, 0x06, 0x9f04);
        re_mdio_write(sc, 0x06, 0xd101);
        re_mdio_write(sc, 0x06, 0xae02);
        re_mdio_write(sc, 0x06, 0xd100);
        re_mdio_write(sc, 0x06, 0xbf2f);
        re_mdio_write(sc, 0x06, 0xf202);
        re_mdio_write(sc, 0x06, 0x3723);
        re_mdio_write(sc, 0x06, 0xef96);
        re_mdio_write(sc, 0x06, 0xfefd);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xf8f9);
        re_mdio_write(sc, 0x06, 0xface);
        re_mdio_write(sc, 0x06, 0xfaef);
        re_mdio_write(sc, 0x06, 0x69fa);
        re_mdio_write(sc, 0x06, 0xd401);
        re_mdio_write(sc, 0x06, 0x55b4);
        re_mdio_write(sc, 0x06, 0xfebf);
        re_mdio_write(sc, 0x06, 0x85a7);
        re_mdio_write(sc, 0x06, 0x0236);
        re_mdio_write(sc, 0x06, 0xf6ac);
        re_mdio_write(sc, 0x06, 0x280b);
        re_mdio_write(sc, 0x06, 0xbf85);
        re_mdio_write(sc, 0x06, 0xa402);
        re_mdio_write(sc, 0x06, 0x36f6);
        re_mdio_write(sc, 0x06, 0xac28);
        re_mdio_write(sc, 0x06, 0x49ae);
        re_mdio_write(sc, 0x06, 0x64bf);
        re_mdio_write(sc, 0x06, 0x85a4);
        re_mdio_write(sc, 0x06, 0x0236);
        re_mdio_write(sc, 0x06, 0xf6ac);
        re_mdio_write(sc, 0x06, 0x285b);
        re_mdio_write(sc, 0x06, 0xd000);
        re_mdio_write(sc, 0x06, 0x0282);
        re_mdio_write(sc, 0x06, 0x60ac);
        re_mdio_write(sc, 0x06, 0x2105);
        re_mdio_write(sc, 0x06, 0xac22);
        re_mdio_write(sc, 0x06, 0x02ae);
        re_mdio_write(sc, 0x06, 0x4ebf);
        re_mdio_write(sc, 0x06, 0xe0c4);
        re_mdio_write(sc, 0x06, 0xbe86);
        re_mdio_write(sc, 0x06, 0x14d2);
        re_mdio_write(sc, 0x06, 0x04d8);
        re_mdio_write(sc, 0x06, 0x19d9);
        re_mdio_write(sc, 0x06, 0x1907);
        re_mdio_write(sc, 0x06, 0xdc19);
        re_mdio_write(sc, 0x06, 0xdd19);
        re_mdio_write(sc, 0x06, 0x0789);
        re_mdio_write(sc, 0x06, 0x89ef);
        re_mdio_write(sc, 0x06, 0x645e);
        re_mdio_write(sc, 0x06, 0x07ff);
        re_mdio_write(sc, 0x06, 0x0d65);
        re_mdio_write(sc, 0x06, 0x5cf8);
        re_mdio_write(sc, 0x06, 0x001e);
        re_mdio_write(sc, 0x06, 0x46dc);
        re_mdio_write(sc, 0x06, 0x19dd);
        re_mdio_write(sc, 0x06, 0x19b2);
        re_mdio_write(sc, 0x06, 0xe2d4);
        re_mdio_write(sc, 0x06, 0x0001);
        re_mdio_write(sc, 0x06, 0xbf85);
        re_mdio_write(sc, 0x06, 0xa402);
        re_mdio_write(sc, 0x06, 0x3723);
        re_mdio_write(sc, 0x06, 0xae1d);
        re_mdio_write(sc, 0x06, 0xbee0);
        re_mdio_write(sc, 0x06, 0xc4bf);
        re_mdio_write(sc, 0x06, 0x8614);
        re_mdio_write(sc, 0x06, 0xd204);
        re_mdio_write(sc, 0x06, 0xd819);
        re_mdio_write(sc, 0x06, 0xd919);
        re_mdio_write(sc, 0x06, 0x07dc);
        re_mdio_write(sc, 0x06, 0x19dd);
        re_mdio_write(sc, 0x06, 0x1907);
        re_mdio_write(sc, 0x06, 0xb2f4);
        re_mdio_write(sc, 0x06, 0xd400);
        re_mdio_write(sc, 0x06, 0x00bf);
        re_mdio_write(sc, 0x06, 0x85a4);
        re_mdio_write(sc, 0x06, 0x0237);
        re_mdio_write(sc, 0x06, 0x23fe);
        re_mdio_write(sc, 0x06, 0xef96);
        re_mdio_write(sc, 0x06, 0xfec6);
        re_mdio_write(sc, 0x06, 0xfefd);
        re_mdio_write(sc, 0x06, 0xfc05);
        re_mdio_write(sc, 0x06, 0xf9e2);
        re_mdio_write(sc, 0x06, 0xe0ea);
        re_mdio_write(sc, 0x06, 0xe3e0);
        re_mdio_write(sc, 0x06, 0xeb5a);
        re_mdio_write(sc, 0x06, 0x070c);
        re_mdio_write(sc, 0x06, 0x031e);
        re_mdio_write(sc, 0x06, 0x20e6);
        re_mdio_write(sc, 0x06, 0xe0ea);
        re_mdio_write(sc, 0x06, 0xe7e0);
        re_mdio_write(sc, 0x06, 0xebe0);
        re_mdio_write(sc, 0x06, 0xe0fc);
        re_mdio_write(sc, 0x06, 0xe1e0);
        re_mdio_write(sc, 0x06, 0xfdfd);
        re_mdio_write(sc, 0x06, 0x04f8);
        re_mdio_write(sc, 0x06, 0xf9e0);
        re_mdio_write(sc, 0x06, 0x8b81);
        re_mdio_write(sc, 0x06, 0xac26);
        re_mdio_write(sc, 0x06, 0x1ae0);
        re_mdio_write(sc, 0x06, 0x8b81);
        re_mdio_write(sc, 0x06, 0xac21);
        re_mdio_write(sc, 0x06, 0x14e0);
        re_mdio_write(sc, 0x06, 0x8b85);
        re_mdio_write(sc, 0x06, 0xac20);
        re_mdio_write(sc, 0x06, 0x0ee0);
        re_mdio_write(sc, 0x06, 0x8b85);
        re_mdio_write(sc, 0x06, 0xac23);
        re_mdio_write(sc, 0x06, 0x08e0);
        re_mdio_write(sc, 0x06, 0x8b87);
        re_mdio_write(sc, 0x06, 0xac24);
        re_mdio_write(sc, 0x06, 0x02ae);
        re_mdio_write(sc, 0x06, 0x3802);
        re_mdio_write(sc, 0x06, 0x1ab5);
        re_mdio_write(sc, 0x06, 0xeee4);
        re_mdio_write(sc, 0x06, 0x1c04);
        re_mdio_write(sc, 0x06, 0xeee4);
        re_mdio_write(sc, 0x06, 0x1d04);
        re_mdio_write(sc, 0x06, 0xe2e0);
        re_mdio_write(sc, 0x06, 0x7ce3);
        re_mdio_write(sc, 0x06, 0xe07d);
        re_mdio_write(sc, 0x06, 0xe0e0);
        re_mdio_write(sc, 0x06, 0x38e1);
        re_mdio_write(sc, 0x06, 0xe039);
        re_mdio_write(sc, 0x06, 0xad2e);
        re_mdio_write(sc, 0x06, 0x1bad);
        re_mdio_write(sc, 0x06, 0x390d);
        re_mdio_write(sc, 0x06, 0xd101);
        re_mdio_write(sc, 0x06, 0xbf21);
        re_mdio_write(sc, 0x06, 0xd502);
        re_mdio_write(sc, 0x06, 0x3723);
        re_mdio_write(sc, 0x06, 0x0282);
        re_mdio_write(sc, 0x06, 0xd8ae);
        re_mdio_write(sc, 0x06, 0x0bac);
        re_mdio_write(sc, 0x06, 0x3802);
        re_mdio_write(sc, 0x06, 0xae06);
        re_mdio_write(sc, 0x06, 0x0283);
        re_mdio_write(sc, 0x06, 0x1802);
        re_mdio_write(sc, 0x06, 0x8360);
        re_mdio_write(sc, 0x06, 0x021a);
        re_mdio_write(sc, 0x06, 0xc6fd);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xf8e1);
        re_mdio_write(sc, 0x06, 0x8af4);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x81ad);
        re_mdio_write(sc, 0x06, 0x2605);
        re_mdio_write(sc, 0x06, 0x0222);
        re_mdio_write(sc, 0x06, 0xa4f7);
        re_mdio_write(sc, 0x06, 0x28e0);
        re_mdio_write(sc, 0x06, 0x8b81);
        re_mdio_write(sc, 0x06, 0xad21);
        re_mdio_write(sc, 0x06, 0x0502);
        re_mdio_write(sc, 0x06, 0x23a9);
        re_mdio_write(sc, 0x06, 0xf729);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x85ad);
        re_mdio_write(sc, 0x06, 0x2005);
        re_mdio_write(sc, 0x06, 0x0214);
        re_mdio_write(sc, 0x06, 0xabf7);
        re_mdio_write(sc, 0x06, 0x2ae0);
        re_mdio_write(sc, 0x06, 0x8b85);
        re_mdio_write(sc, 0x06, 0xad23);
        re_mdio_write(sc, 0x06, 0x0502);
        re_mdio_write(sc, 0x06, 0x12e7);
        re_mdio_write(sc, 0x06, 0xf72b);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x87ad);
        re_mdio_write(sc, 0x06, 0x2405);
        re_mdio_write(sc, 0x06, 0x0283);
        re_mdio_write(sc, 0x06, 0xbcf7);
        re_mdio_write(sc, 0x06, 0x2ce5);
        re_mdio_write(sc, 0x06, 0x8af4);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xf8e0);
        re_mdio_write(sc, 0x06, 0x8b81);
        re_mdio_write(sc, 0x06, 0xad26);
        re_mdio_write(sc, 0x06, 0x0302);
        re_mdio_write(sc, 0x06, 0x21e5);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x81ad);
        re_mdio_write(sc, 0x06, 0x2109);
        re_mdio_write(sc, 0x06, 0xe08a);
        re_mdio_write(sc, 0x06, 0xf4ac);
        re_mdio_write(sc, 0x06, 0x2003);
        re_mdio_write(sc, 0x06, 0x0223);
        re_mdio_write(sc, 0x06, 0x98e0);
        re_mdio_write(sc, 0x06, 0x8b85);
        re_mdio_write(sc, 0x06, 0xad20);
        re_mdio_write(sc, 0x06, 0x09e0);
        re_mdio_write(sc, 0x06, 0x8af4);
        re_mdio_write(sc, 0x06, 0xac21);
        re_mdio_write(sc, 0x06, 0x0302);
        re_mdio_write(sc, 0x06, 0x13fb);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x85ad);
        re_mdio_write(sc, 0x06, 0x2309);
        re_mdio_write(sc, 0x06, 0xe08a);
        re_mdio_write(sc, 0x06, 0xf4ac);
        re_mdio_write(sc, 0x06, 0x2203);
        re_mdio_write(sc, 0x06, 0x0212);
        re_mdio_write(sc, 0x06, 0xfae0);
        re_mdio_write(sc, 0x06, 0x8b87);
        re_mdio_write(sc, 0x06, 0xad24);
        re_mdio_write(sc, 0x06, 0x09e0);
        re_mdio_write(sc, 0x06, 0x8af4);
        re_mdio_write(sc, 0x06, 0xac23);
        re_mdio_write(sc, 0x06, 0x0302);
        re_mdio_write(sc, 0x06, 0x83c1);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xf8e1);
        re_mdio_write(sc, 0x06, 0x8af4);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x81ad);
        re_mdio_write(sc, 0x06, 0x2608);
        re_mdio_write(sc, 0x06, 0xe083);
        re_mdio_write(sc, 0x06, 0xd2ad);
        re_mdio_write(sc, 0x06, 0x2502);
        re_mdio_write(sc, 0x06, 0xf628);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x81ad);
        re_mdio_write(sc, 0x06, 0x210a);
        re_mdio_write(sc, 0x06, 0xe084);
        re_mdio_write(sc, 0x06, 0x0af6);
        re_mdio_write(sc, 0x06, 0x27a0);
        re_mdio_write(sc, 0x06, 0x0502);
        re_mdio_write(sc, 0x06, 0xf629);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x85ad);
        re_mdio_write(sc, 0x06, 0x2008);
        re_mdio_write(sc, 0x06, 0xe08a);
        re_mdio_write(sc, 0x06, 0xe8ad);
        re_mdio_write(sc, 0x06, 0x2102);
        re_mdio_write(sc, 0x06, 0xf62a);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x85ad);
        re_mdio_write(sc, 0x06, 0x2308);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x20a0);
        re_mdio_write(sc, 0x06, 0x0302);
        re_mdio_write(sc, 0x06, 0xf62b);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x87ad);
        re_mdio_write(sc, 0x06, 0x2408);
        re_mdio_write(sc, 0x06, 0xe086);
        re_mdio_write(sc, 0x06, 0x02a0);
        re_mdio_write(sc, 0x06, 0x0302);
        re_mdio_write(sc, 0x06, 0xf62c);
        re_mdio_write(sc, 0x06, 0xe58a);
        re_mdio_write(sc, 0x06, 0xf4a1);
        re_mdio_write(sc, 0x06, 0x0008);
        re_mdio_write(sc, 0x06, 0xd100);
        re_mdio_write(sc, 0x06, 0xbf21);
        re_mdio_write(sc, 0x06, 0xd502);
        re_mdio_write(sc, 0x06, 0x3723);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xee86);
        re_mdio_write(sc, 0x06, 0x0200);
        re_mdio_write(sc, 0x06, 0x04f8);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x87ad);
        re_mdio_write(sc, 0x06, 0x241e);
        re_mdio_write(sc, 0x06, 0xe086);
        re_mdio_write(sc, 0x06, 0x02a0);
        re_mdio_write(sc, 0x06, 0x0005);
        re_mdio_write(sc, 0x06, 0x0283);
        re_mdio_write(sc, 0x06, 0xe8ae);
        re_mdio_write(sc, 0x06, 0xf5a0);
        re_mdio_write(sc, 0x06, 0x0105);
        re_mdio_write(sc, 0x06, 0x0283);
        re_mdio_write(sc, 0x06, 0xf8ae);
        re_mdio_write(sc, 0x06, 0x0ba0);
        re_mdio_write(sc, 0x06, 0x0205);
        re_mdio_write(sc, 0x06, 0x0284);
        re_mdio_write(sc, 0x06, 0x14ae);
        re_mdio_write(sc, 0x06, 0x03a0);
        re_mdio_write(sc, 0x06, 0x0300);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xf8fa);
        re_mdio_write(sc, 0x06, 0xef69);
        re_mdio_write(sc, 0x06, 0x0284);
        re_mdio_write(sc, 0x06, 0x2bee);
        re_mdio_write(sc, 0x06, 0x8602);
        re_mdio_write(sc, 0x06, 0x01ef);
        re_mdio_write(sc, 0x06, 0x96fe);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xf8ee);
        re_mdio_write(sc, 0x06, 0x8609);
        re_mdio_write(sc, 0x06, 0x0002);
        re_mdio_write(sc, 0x06, 0x8461);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xae10);
        re_mdio_write(sc, 0x06, 0x0000);
        re_mdio_write(sc, 0x06, 0x0000);
        re_mdio_write(sc, 0x06, 0x0000);
        re_mdio_write(sc, 0x06, 0x0000);
        re_mdio_write(sc, 0x06, 0x0000);
        re_mdio_write(sc, 0x06, 0x0000);
        re_mdio_write(sc, 0x06, 0x0000);
        re_mdio_write(sc, 0x06, 0x0000);
        re_mdio_write(sc, 0x06, 0xf8e0);
        re_mdio_write(sc, 0x06, 0x8608);
        re_mdio_write(sc, 0x06, 0xe186);
        re_mdio_write(sc, 0x06, 0x091f);
        re_mdio_write(sc, 0x06, 0x019e);
        re_mdio_write(sc, 0x06, 0x0611);
        re_mdio_write(sc, 0x06, 0xe586);
        re_mdio_write(sc, 0x06, 0x09ae);
        re_mdio_write(sc, 0x06, 0x04ee);
        re_mdio_write(sc, 0x06, 0x8602);
        re_mdio_write(sc, 0x06, 0x01fc);
        re_mdio_write(sc, 0x06, 0x04f8);
        re_mdio_write(sc, 0x06, 0xf9fa);
        re_mdio_write(sc, 0x06, 0xef69);
        re_mdio_write(sc, 0x06, 0xfbbf);
        re_mdio_write(sc, 0x06, 0x8604);
        re_mdio_write(sc, 0x06, 0xef79);
        re_mdio_write(sc, 0x06, 0xd200);
        re_mdio_write(sc, 0x06, 0xd400);
        re_mdio_write(sc, 0x06, 0x221e);
        re_mdio_write(sc, 0x06, 0x02bf);
        re_mdio_write(sc, 0x06, 0x2fec);
        re_mdio_write(sc, 0x06, 0x0237);
        re_mdio_write(sc, 0x06, 0x23bf);
        re_mdio_write(sc, 0x06, 0x13f2);
        re_mdio_write(sc, 0x06, 0x0236);
        re_mdio_write(sc, 0x06, 0xf60d);
        re_mdio_write(sc, 0x06, 0x4559);
        re_mdio_write(sc, 0x06, 0x1fef);
        re_mdio_write(sc, 0x06, 0x97dd);
        re_mdio_write(sc, 0x06, 0xd308);
        re_mdio_write(sc, 0x06, 0x1a93);
        re_mdio_write(sc, 0x06, 0xdd12);
        re_mdio_write(sc, 0x06, 0x17a2);
        re_mdio_write(sc, 0x06, 0x04de);
        re_mdio_write(sc, 0x06, 0xffef);
        re_mdio_write(sc, 0x06, 0x96fe);
        re_mdio_write(sc, 0x06, 0xfdfc);
        re_mdio_write(sc, 0x06, 0x04f8);
        re_mdio_write(sc, 0x06, 0xf9fa);
        re_mdio_write(sc, 0x06, 0xef69);
        re_mdio_write(sc, 0x06, 0xfbee);
        re_mdio_write(sc, 0x06, 0x8602);
        re_mdio_write(sc, 0x06, 0x03d5);
        re_mdio_write(sc, 0x06, 0x0080);
        re_mdio_write(sc, 0x06, 0xbf86);
        re_mdio_write(sc, 0x06, 0x04ef);
        re_mdio_write(sc, 0x06, 0x79ef);
        re_mdio_write(sc, 0x06, 0x45bf);
        re_mdio_write(sc, 0x06, 0x2fec);
        re_mdio_write(sc, 0x06, 0x0237);
        re_mdio_write(sc, 0x06, 0x23bf);
        re_mdio_write(sc, 0x06, 0x13f2);
        re_mdio_write(sc, 0x06, 0x0236);
        re_mdio_write(sc, 0x06, 0xf6ad);
        re_mdio_write(sc, 0x06, 0x2702);
        re_mdio_write(sc, 0x06, 0x78ff);
        re_mdio_write(sc, 0x06, 0xe186);
        re_mdio_write(sc, 0x06, 0x0a1b);
        re_mdio_write(sc, 0x06, 0x01aa);
        re_mdio_write(sc, 0x06, 0x2eef);
        re_mdio_write(sc, 0x06, 0x97d9);
        re_mdio_write(sc, 0x06, 0x7900);
        re_mdio_write(sc, 0x06, 0x9e2b);
        re_mdio_write(sc, 0x06, 0x81dd);
        re_mdio_write(sc, 0x06, 0xbf85);
        re_mdio_write(sc, 0x06, 0xad02);
        re_mdio_write(sc, 0x06, 0x3723);
        re_mdio_write(sc, 0x06, 0xd101);
        re_mdio_write(sc, 0x06, 0xef02);
        re_mdio_write(sc, 0x06, 0x100c);
        re_mdio_write(sc, 0x06, 0x11b0);
        re_mdio_write(sc, 0x06, 0xfc0d);
        re_mdio_write(sc, 0x06, 0x11bf);
        re_mdio_write(sc, 0x06, 0x85aa);
        re_mdio_write(sc, 0x06, 0x0237);
        re_mdio_write(sc, 0x06, 0x23d1);
        re_mdio_write(sc, 0x06, 0x00bf);
        re_mdio_write(sc, 0x06, 0x85aa);
        re_mdio_write(sc, 0x06, 0x0237);
        re_mdio_write(sc, 0x06, 0x23ee);
        re_mdio_write(sc, 0x06, 0x8602);
        re_mdio_write(sc, 0x06, 0x02ae);
        re_mdio_write(sc, 0x06, 0x0413);
        re_mdio_write(sc, 0x06, 0xa38b);
        re_mdio_write(sc, 0x06, 0xb4d3);
        re_mdio_write(sc, 0x06, 0x8012);
        re_mdio_write(sc, 0x06, 0x17a2);
        re_mdio_write(sc, 0x06, 0x04ad);
        re_mdio_write(sc, 0x06, 0xffef);
        re_mdio_write(sc, 0x06, 0x96fe);
        re_mdio_write(sc, 0x06, 0xfdfc);
        re_mdio_write(sc, 0x06, 0x04f8);
        re_mdio_write(sc, 0x06, 0xf9e0);
        re_mdio_write(sc, 0x06, 0x8b85);
        re_mdio_write(sc, 0x06, 0xad25);
        re_mdio_write(sc, 0x06, 0x48e0);
        re_mdio_write(sc, 0x06, 0x8a96);
        re_mdio_write(sc, 0x06, 0xe18a);
        re_mdio_write(sc, 0x06, 0x977c);
        re_mdio_write(sc, 0x06, 0x0000);
        re_mdio_write(sc, 0x06, 0x9e35);
        re_mdio_write(sc, 0x06, 0xee8a);
        re_mdio_write(sc, 0x06, 0x9600);
        re_mdio_write(sc, 0x06, 0xee8a);
        re_mdio_write(sc, 0x06, 0x9700);
        re_mdio_write(sc, 0x06, 0xe08a);
        re_mdio_write(sc, 0x06, 0xbee1);
        re_mdio_write(sc, 0x06, 0x8abf);
        re_mdio_write(sc, 0x06, 0xe286);
        re_mdio_write(sc, 0x06, 0x10e3);
        re_mdio_write(sc, 0x06, 0x8611);
        re_mdio_write(sc, 0x06, 0x0236);
        re_mdio_write(sc, 0x06, 0x1aad);
        re_mdio_write(sc, 0x06, 0x2012);
        re_mdio_write(sc, 0x06, 0xee8a);
        re_mdio_write(sc, 0x06, 0x9603);
        re_mdio_write(sc, 0x06, 0xee8a);
        re_mdio_write(sc, 0x06, 0x97b7);
        re_mdio_write(sc, 0x06, 0xee86);
        re_mdio_write(sc, 0x06, 0x1000);
        re_mdio_write(sc, 0x06, 0xee86);
        re_mdio_write(sc, 0x06, 0x1100);
        re_mdio_write(sc, 0x06, 0xae11);
        re_mdio_write(sc, 0x06, 0x15e6);
        re_mdio_write(sc, 0x06, 0x8610);
        re_mdio_write(sc, 0x06, 0xe786);
        re_mdio_write(sc, 0x06, 0x11ae);
        re_mdio_write(sc, 0x06, 0x08ee);
        re_mdio_write(sc, 0x06, 0x8610);
        re_mdio_write(sc, 0x06, 0x00ee);
        re_mdio_write(sc, 0x06, 0x8611);
        re_mdio_write(sc, 0x06, 0x00fd);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xf8fa);
        re_mdio_write(sc, 0x06, 0xef69);
        re_mdio_write(sc, 0x06, 0xe0e0);
        re_mdio_write(sc, 0x06, 0x00e1);
        re_mdio_write(sc, 0x06, 0xe001);
        re_mdio_write(sc, 0x06, 0xad27);
        re_mdio_write(sc, 0x06, 0x32e0);
        re_mdio_write(sc, 0x06, 0x8b40);
        re_mdio_write(sc, 0x06, 0xf720);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x40bf);
        re_mdio_write(sc, 0x06, 0x31f5);
        re_mdio_write(sc, 0x06, 0x0236);
        re_mdio_write(sc, 0x06, 0xf6ad);
        re_mdio_write(sc, 0x06, 0x2821);
        re_mdio_write(sc, 0x06, 0xe0e0);
        re_mdio_write(sc, 0x06, 0x20e1);
        re_mdio_write(sc, 0x06, 0xe021);
        re_mdio_write(sc, 0x06, 0xad20);
        re_mdio_write(sc, 0x06, 0x18e0);
        re_mdio_write(sc, 0x06, 0x8b40);
        re_mdio_write(sc, 0x06, 0xf620);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x40ee);
        re_mdio_write(sc, 0x06, 0x8b3b);
        re_mdio_write(sc, 0x06, 0xffe0);
        re_mdio_write(sc, 0x06, 0x8a8a);
        re_mdio_write(sc, 0x06, 0xe18a);
        re_mdio_write(sc, 0x06, 0x8be4);
        re_mdio_write(sc, 0x06, 0xe000);
        re_mdio_write(sc, 0x06, 0xe5e0);
        re_mdio_write(sc, 0x06, 0x01ef);
        re_mdio_write(sc, 0x06, 0x96fe);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xf8fa);
        re_mdio_write(sc, 0x06, 0xef69);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x80ad);
        re_mdio_write(sc, 0x06, 0x2722);
        re_mdio_write(sc, 0x06, 0xbf44);
        re_mdio_write(sc, 0x06, 0xfc02);
        re_mdio_write(sc, 0x06, 0x36f6);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x441f);
        re_mdio_write(sc, 0x06, 0x019e);
        re_mdio_write(sc, 0x06, 0x15e5);
        re_mdio_write(sc, 0x06, 0x8b44);
        re_mdio_write(sc, 0x06, 0xad29);
        re_mdio_write(sc, 0x06, 0x07ac);
        re_mdio_write(sc, 0x06, 0x2804);
        re_mdio_write(sc, 0x06, 0xd101);
        re_mdio_write(sc, 0x06, 0xae02);
        re_mdio_write(sc, 0x06, 0xd100);
        re_mdio_write(sc, 0x06, 0xbf85);
        re_mdio_write(sc, 0x06, 0xb002);
        re_mdio_write(sc, 0x06, 0x3723);
        re_mdio_write(sc, 0x06, 0xef96);
        re_mdio_write(sc, 0x06, 0xfefc);
        re_mdio_write(sc, 0x06, 0x0400);
        re_mdio_write(sc, 0x06, 0xe140);
        re_mdio_write(sc, 0x06, 0x77e1);
        re_mdio_write(sc, 0x06, 0x40dd);
        re_mdio_write(sc, 0x06, 0xe022);
        re_mdio_write(sc, 0x06, 0x32e1);
        re_mdio_write(sc, 0x06, 0x5074);
        re_mdio_write(sc, 0x06, 0xe144);
        re_mdio_write(sc, 0x06, 0xffe0);
        re_mdio_write(sc, 0x06, 0xdaff);
        re_mdio_write(sc, 0x06, 0xe0c0);
        re_mdio_write(sc, 0x06, 0x52e0);
        re_mdio_write(sc, 0x06, 0xeed9);
        re_mdio_write(sc, 0x06, 0xe04c);
        re_mdio_write(sc, 0x06, 0xbbe0);
        re_mdio_write(sc, 0x06, 0x2a00);
        re_mdio_write(sc, 0x05, 0xe142);
        PhyRegValue = re_mdio_read(sc, 0x06);
        PhyRegValue |= BIT_0;
        re_mdio_write(sc, 0x06, PhyRegValue);
        re_mdio_write(sc, 0x05, 0xe140);
        PhyRegValue = re_mdio_read(sc, 0x06);
        PhyRegValue |= BIT_0;
        re_mdio_write(sc, 0x06, PhyRegValue);
        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x1f, 0x0005);
        for (i = 0; i < 200; i++) {
                DELAY(100);
                PhyRegValue = re_mdio_read(sc, 0x00);
                if (PhyRegValue & BIT_7)
                        break;
        }

        re_mdio_write(sc, 0x1F, 0x0003);
        re_mdio_write(sc, 0x09, 0xA20F);
        re_mdio_write(sc, 0x1F, 0x0000);

        re_mdio_write(sc, 0x1f, 0x0003);
        PhyRegValue = re_mdio_read(sc, 0x19);
        PhyRegValue &= ~BIT_0;
        re_mdio_write(sc, 0x19, PhyRegValue);
        PhyRegValue = re_mdio_read(sc, 0x10);
        PhyRegValue &= ~BIT_10;
        re_mdio_write(sc, 0x10, PhyRegValue);
        re_mdio_write(sc, 0x1f, 0x0000);

        re_mdio_write(sc, 0x1f, 0x0007);
        re_mdio_write(sc, 0x1e, 0x0042);
        re_mdio_write(sc, 0x18, 0x2300);
        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x1f, 0x0007);
        re_mdio_write(sc, 0x1e, 0x0023);
        PhyRegValue = re_mdio_read(sc, 0x17);
        if (sc->RequiredSecLanDonglePatch)
                PhyRegValue &= ~(BIT_2);
        re_mdio_write(sc, 0x17, PhyRegValue);
        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x00, 0x9200);
}

static void re_set_phy_mcu_8168f_1(struct re_softc *sc)
{
        u_int16_t PhyRegValue;
        int i;

        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x00, 0x1800);
        PhyRegValue = re_mdio_read(sc, 0x15);
        PhyRegValue &= ~(BIT_12);
        re_mdio_write(sc, 0x15, PhyRegValue);
        re_mdio_write(sc, 0x00, 0x4800);
        re_mdio_write(sc, 0x1f, 0x0007);
        re_mdio_write(sc, 0x1e, 0x002f);
        for (i = 0; i < 1000; i++) {
                DELAY(100);
                PhyRegValue = re_mdio_read(sc, 0x1c);
                if (PhyRegValue & BIT_7)
                        break;
        }
        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x00, 0x1800);
        re_mdio_write(sc, 0x1f, 0x0007);
        re_mdio_write(sc, 0x1e, 0x0023);
        for (i = 0; i < 200; i++) {
                DELAY(100);
                PhyRegValue = re_mdio_read(sc, 0x18);
                if (!(PhyRegValue & BIT_0))
                        break;
        }
        re_mdio_write(sc, 0x1f, 0x0005);
        re_mdio_write(sc, 0x05, 0xfff6);
        re_mdio_write(sc, 0x06, 0x0080);
        re_mdio_write(sc, 0x1f, 0x0007);
        re_mdio_write(sc, 0x1e, 0x0023);
        re_mdio_write(sc, 0x16, 0x0306);
        re_mdio_write(sc, 0x16, 0x0307);
        re_mdio_write(sc, 0x15, 0x0194);
        re_mdio_write(sc, 0x19, 0x407D);
        re_mdio_write(sc, 0x15, 0x0098);
        re_mdio_write(sc, 0x19, 0x7c0b);
        re_mdio_write(sc, 0x15, 0x0099);
        re_mdio_write(sc, 0x19, 0x6c0b);
        re_mdio_write(sc, 0x15, 0x00eb);
        re_mdio_write(sc, 0x19, 0x6c0b);
        re_mdio_write(sc, 0x15, 0x00f8);
        re_mdio_write(sc, 0x19, 0x6f0b);
        re_mdio_write(sc, 0x15, 0x00fe);
        re_mdio_write(sc, 0x19, 0x6f0f);
        re_mdio_write(sc, 0x15, 0x00db);
        re_mdio_write(sc, 0x19, 0x6f09);
        re_mdio_write(sc, 0x15, 0x00dc);
        re_mdio_write(sc, 0x19, 0xaefd);
        re_mdio_write(sc, 0x15, 0x00dd);
        re_mdio_write(sc, 0x19, 0x6f0b);
        re_mdio_write(sc, 0x15, 0x00de);
        re_mdio_write(sc, 0x19, 0xc60b);
        re_mdio_write(sc, 0x15, 0x00df);
        re_mdio_write(sc, 0x19, 0x00fa);
        re_mdio_write(sc, 0x15, 0x00e0);
        re_mdio_write(sc, 0x19, 0x30e1);
        re_mdio_write(sc, 0x15, 0x020c);
        re_mdio_write(sc, 0x19, 0x3224);
        re_mdio_write(sc, 0x15, 0x020e);
        re_mdio_write(sc, 0x19, 0x9813);
        re_mdio_write(sc, 0x15, 0x020f);
        re_mdio_write(sc, 0x19, 0x7801);
        re_mdio_write(sc, 0x15, 0x0210);
        re_mdio_write(sc, 0x19, 0x930f);
        re_mdio_write(sc, 0x15, 0x0211);
        re_mdio_write(sc, 0x19, 0x9206);
        re_mdio_write(sc, 0x15, 0x0212);
        re_mdio_write(sc, 0x19, 0x4002);
        re_mdio_write(sc, 0x15, 0x0213);
        re_mdio_write(sc, 0x19, 0x7800);
        re_mdio_write(sc, 0x15, 0x0214);
        re_mdio_write(sc, 0x19, 0x588f);
        re_mdio_write(sc, 0x15, 0x0215);
        re_mdio_write(sc, 0x19, 0x5520);
        re_mdio_write(sc, 0x15, 0x0216);
        re_mdio_write(sc, 0x19, 0x3224);
        re_mdio_write(sc, 0x15, 0x0217);
        re_mdio_write(sc, 0x19, 0x4002);
        re_mdio_write(sc, 0x15, 0x0218);
        re_mdio_write(sc, 0x19, 0x7800);
        re_mdio_write(sc, 0x15, 0x0219);
        re_mdio_write(sc, 0x19, 0x588d);
        re_mdio_write(sc, 0x15, 0x021a);
        re_mdio_write(sc, 0x19, 0x5540);
        re_mdio_write(sc, 0x15, 0x021b);
        re_mdio_write(sc, 0x19, 0x9e03);
        re_mdio_write(sc, 0x15, 0x021c);
        re_mdio_write(sc, 0x19, 0x7c40);
        re_mdio_write(sc, 0x15, 0x021d);
        re_mdio_write(sc, 0x19, 0x6840);
        re_mdio_write(sc, 0x15, 0x021e);
        re_mdio_write(sc, 0x19, 0x3224);
        re_mdio_write(sc, 0x15, 0x021f);
        re_mdio_write(sc, 0x19, 0x4002);
        re_mdio_write(sc, 0x15, 0x0220);
        re_mdio_write(sc, 0x19, 0x3224);
        re_mdio_write(sc, 0x15, 0x0221);
        re_mdio_write(sc, 0x19, 0x9e03);
        re_mdio_write(sc, 0x15, 0x0222);
        re_mdio_write(sc, 0x19, 0x7c40);
        re_mdio_write(sc, 0x15, 0x0223);
        re_mdio_write(sc, 0x19, 0x6840);
        re_mdio_write(sc, 0x15, 0x0224);
        re_mdio_write(sc, 0x19, 0x7800);
        re_mdio_write(sc, 0x15, 0x0225);
        re_mdio_write(sc, 0x19, 0x3231);
        re_mdio_write(sc, 0x15, 0x0000);
        re_mdio_write(sc, 0x16, 0x0306);
        re_mdio_write(sc, 0x16, 0x0300);
        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x1f, 0x0005);
        re_mdio_write(sc, 0x05, 0xfff6);
        re_mdio_write(sc, 0x06, 0x0080);
        re_mdio_write(sc, 0x05, 0x8000);
        re_mdio_write(sc, 0x06, 0x0280);
        re_mdio_write(sc, 0x06, 0x48f7);
        re_mdio_write(sc, 0x06, 0x00e0);
        re_mdio_write(sc, 0x06, 0xfff7);
        re_mdio_write(sc, 0x06, 0xa080);
        re_mdio_write(sc, 0x06, 0x02ae);
        re_mdio_write(sc, 0x06, 0xf602);
        re_mdio_write(sc, 0x06, 0x0118);
        re_mdio_write(sc, 0x06, 0x0201);
        re_mdio_write(sc, 0x06, 0x2502);
        re_mdio_write(sc, 0x06, 0x8090);
        re_mdio_write(sc, 0x06, 0x0201);
        re_mdio_write(sc, 0x06, 0x4202);
        re_mdio_write(sc, 0x06, 0x015c);
        re_mdio_write(sc, 0x06, 0x0280);
        re_mdio_write(sc, 0x06, 0xad02);
        re_mdio_write(sc, 0x06, 0x80ca);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x88e1);
        re_mdio_write(sc, 0x06, 0x8b89);
        re_mdio_write(sc, 0x06, 0x1e01);
        re_mdio_write(sc, 0x06, 0xe18b);
        re_mdio_write(sc, 0x06, 0x8a1e);
        re_mdio_write(sc, 0x06, 0x01e1);
        re_mdio_write(sc, 0x06, 0x8b8b);
        re_mdio_write(sc, 0x06, 0x1e01);
        re_mdio_write(sc, 0x06, 0xe18b);
        re_mdio_write(sc, 0x06, 0x8c1e);
        re_mdio_write(sc, 0x06, 0x01e1);
        re_mdio_write(sc, 0x06, 0x8b8d);
        re_mdio_write(sc, 0x06, 0x1e01);
        re_mdio_write(sc, 0x06, 0xe18b);
        re_mdio_write(sc, 0x06, 0x8e1e);
        re_mdio_write(sc, 0x06, 0x01a0);
        re_mdio_write(sc, 0x06, 0x00c7);
        re_mdio_write(sc, 0x06, 0xaebb);
        re_mdio_write(sc, 0x06, 0xd484);
        re_mdio_write(sc, 0x06, 0x3ce4);
        re_mdio_write(sc, 0x06, 0x8b92);
        re_mdio_write(sc, 0x06, 0xe58b);
        re_mdio_write(sc, 0x06, 0x93ee);
        re_mdio_write(sc, 0x06, 0x8ac8);
        re_mdio_write(sc, 0x06, 0x03ee);
        re_mdio_write(sc, 0x06, 0x8aca);
        re_mdio_write(sc, 0x06, 0x60ee);
        re_mdio_write(sc, 0x06, 0x8ac0);
        re_mdio_write(sc, 0x06, 0x00ee);
        re_mdio_write(sc, 0x06, 0x8ac1);
        re_mdio_write(sc, 0x06, 0x00ee);
        re_mdio_write(sc, 0x06, 0x8abe);
        re_mdio_write(sc, 0x06, 0x07ee);
        re_mdio_write(sc, 0x06, 0x8abf);
        re_mdio_write(sc, 0x06, 0x73ee);
        re_mdio_write(sc, 0x06, 0x8a95);
        re_mdio_write(sc, 0x06, 0x02bf);
        re_mdio_write(sc, 0x06, 0x8b88);
        re_mdio_write(sc, 0x06, 0xec00);
        re_mdio_write(sc, 0x06, 0x19a9);
        re_mdio_write(sc, 0x06, 0x8b90);
        re_mdio_write(sc, 0x06, 0xf9ee);
        re_mdio_write(sc, 0x06, 0xfff6);
        re_mdio_write(sc, 0x06, 0x00ee);
        re_mdio_write(sc, 0x06, 0xfff7);
        re_mdio_write(sc, 0x06, 0xfed1);
        re_mdio_write(sc, 0x06, 0x00bf);
        re_mdio_write(sc, 0x06, 0x85a4);
        re_mdio_write(sc, 0x06, 0x0238);
        re_mdio_write(sc, 0x06, 0x7dd1);
        re_mdio_write(sc, 0x06, 0x01bf);
        re_mdio_write(sc, 0x06, 0x85a7);
        re_mdio_write(sc, 0x06, 0x0238);
        re_mdio_write(sc, 0x06, 0x7d04);
        re_mdio_write(sc, 0x06, 0xf8e0);
        re_mdio_write(sc, 0x06, 0x8b8a);
        re_mdio_write(sc, 0x06, 0xad20);
        re_mdio_write(sc, 0x06, 0x14ee);
        re_mdio_write(sc, 0x06, 0x8b8a);
        re_mdio_write(sc, 0x06, 0x0002);
        re_mdio_write(sc, 0x06, 0x204b);
        re_mdio_write(sc, 0x06, 0xe0e4);
        re_mdio_write(sc, 0x06, 0x26e1);
        re_mdio_write(sc, 0x06, 0xe427);
        re_mdio_write(sc, 0x06, 0xeee4);
        re_mdio_write(sc, 0x06, 0x2623);
        re_mdio_write(sc, 0x06, 0xe5e4);
        re_mdio_write(sc, 0x06, 0x27fc);
        re_mdio_write(sc, 0x06, 0x04f8);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x8dad);
        re_mdio_write(sc, 0x06, 0x2014);
        re_mdio_write(sc, 0x06, 0xee8b);
        re_mdio_write(sc, 0x06, 0x8d00);
        re_mdio_write(sc, 0x06, 0xe08a);
        re_mdio_write(sc, 0x06, 0x5a78);
        re_mdio_write(sc, 0x06, 0x039e);
        re_mdio_write(sc, 0x06, 0x0902);
        re_mdio_write(sc, 0x06, 0x05e8);
        re_mdio_write(sc, 0x06, 0x0281);
        re_mdio_write(sc, 0x06, 0x4f02);
        re_mdio_write(sc, 0x06, 0x326c);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xf8e0);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0xad20);
        re_mdio_write(sc, 0x06, 0x1df6);
        re_mdio_write(sc, 0x06, 0x20e4);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0x022f);
        re_mdio_write(sc, 0x06, 0x0902);
        re_mdio_write(sc, 0x06, 0x2ab0);
        re_mdio_write(sc, 0x06, 0x0285);
        re_mdio_write(sc, 0x06, 0x1602);
        re_mdio_write(sc, 0x06, 0x03ba);
        re_mdio_write(sc, 0x06, 0x0284);
        re_mdio_write(sc, 0x06, 0xe502);
        re_mdio_write(sc, 0x06, 0x2df1);
        re_mdio_write(sc, 0x06, 0x0283);
        re_mdio_write(sc, 0x06, 0x8302);
        re_mdio_write(sc, 0x06, 0x0475);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x8ead);
        re_mdio_write(sc, 0x06, 0x210b);
        re_mdio_write(sc, 0x06, 0xf621);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x8e02);
        re_mdio_write(sc, 0x06, 0x83f8);
        re_mdio_write(sc, 0x06, 0x021c);
        re_mdio_write(sc, 0x06, 0x99e0);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0xad22);
        re_mdio_write(sc, 0x06, 0x08f6);
        re_mdio_write(sc, 0x06, 0x22e4);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0x0235);
        re_mdio_write(sc, 0x06, 0x63e0);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0xad23);
        re_mdio_write(sc, 0x06, 0x08f6);
        re_mdio_write(sc, 0x06, 0x23e4);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0x0231);
        re_mdio_write(sc, 0x06, 0x57e0);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0xad24);
        re_mdio_write(sc, 0x06, 0x05f6);
        re_mdio_write(sc, 0x06, 0x24e4);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x8ead);
        re_mdio_write(sc, 0x06, 0x2505);
        re_mdio_write(sc, 0x06, 0xf625);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x8ee0);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0xad26);
        re_mdio_write(sc, 0x06, 0x08f6);
        re_mdio_write(sc, 0x06, 0x26e4);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0x022d);
        re_mdio_write(sc, 0x06, 0x1ce0);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0xad27);
        re_mdio_write(sc, 0x06, 0x05f6);
        re_mdio_write(sc, 0x06, 0x27e4);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0x0203);
        re_mdio_write(sc, 0x06, 0x80fc);
        re_mdio_write(sc, 0x06, 0x04f8);
        re_mdio_write(sc, 0x06, 0xf9e0);
        re_mdio_write(sc, 0x06, 0x8b81);
        re_mdio_write(sc, 0x06, 0xac26);
        re_mdio_write(sc, 0x06, 0x1ae0);
        re_mdio_write(sc, 0x06, 0x8b81);
        re_mdio_write(sc, 0x06, 0xac21);
        re_mdio_write(sc, 0x06, 0x14e0);
        re_mdio_write(sc, 0x06, 0x8b85);
        re_mdio_write(sc, 0x06, 0xac20);
        re_mdio_write(sc, 0x06, 0x0ee0);
        re_mdio_write(sc, 0x06, 0x8b85);
        re_mdio_write(sc, 0x06, 0xac23);
        re_mdio_write(sc, 0x06, 0x08e0);
        re_mdio_write(sc, 0x06, 0x8b87);
        re_mdio_write(sc, 0x06, 0xac24);
        re_mdio_write(sc, 0x06, 0x02ae);
        re_mdio_write(sc, 0x06, 0x3802);
        re_mdio_write(sc, 0x06, 0x1ac2);
        re_mdio_write(sc, 0x06, 0xeee4);
        re_mdio_write(sc, 0x06, 0x1c04);
        re_mdio_write(sc, 0x06, 0xeee4);
        re_mdio_write(sc, 0x06, 0x1d04);
        re_mdio_write(sc, 0x06, 0xe2e0);
        re_mdio_write(sc, 0x06, 0x7ce3);
        re_mdio_write(sc, 0x06, 0xe07d);
        re_mdio_write(sc, 0x06, 0xe0e0);
        re_mdio_write(sc, 0x06, 0x38e1);
        re_mdio_write(sc, 0x06, 0xe039);
        re_mdio_write(sc, 0x06, 0xad2e);
        re_mdio_write(sc, 0x06, 0x1bad);
        re_mdio_write(sc, 0x06, 0x390d);
        re_mdio_write(sc, 0x06, 0xd101);
        re_mdio_write(sc, 0x06, 0xbf22);
        re_mdio_write(sc, 0x06, 0x7a02);
        re_mdio_write(sc, 0x06, 0x387d);
        re_mdio_write(sc, 0x06, 0x0281);
        re_mdio_write(sc, 0x06, 0xacae);
        re_mdio_write(sc, 0x06, 0x0bac);
        re_mdio_write(sc, 0x06, 0x3802);
        re_mdio_write(sc, 0x06, 0xae06);
        re_mdio_write(sc, 0x06, 0x0281);
        re_mdio_write(sc, 0x06, 0xe902);
        re_mdio_write(sc, 0x06, 0x822e);
        re_mdio_write(sc, 0x06, 0x021a);
        re_mdio_write(sc, 0x06, 0xd3fd);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xf8e1);
        re_mdio_write(sc, 0x06, 0x8af4);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x81ad);
        re_mdio_write(sc, 0x06, 0x2602);
        re_mdio_write(sc, 0x06, 0xf728);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x81ad);
        re_mdio_write(sc, 0x06, 0x2105);
        re_mdio_write(sc, 0x06, 0x0222);
        re_mdio_write(sc, 0x06, 0x8ef7);
        re_mdio_write(sc, 0x06, 0x29e0);
        re_mdio_write(sc, 0x06, 0x8b85);
        re_mdio_write(sc, 0x06, 0xad20);
        re_mdio_write(sc, 0x06, 0x0502);
        re_mdio_write(sc, 0x06, 0x14b8);
        re_mdio_write(sc, 0x06, 0xf72a);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x85ad);
        re_mdio_write(sc, 0x06, 0x2305);
        re_mdio_write(sc, 0x06, 0x0212);
        re_mdio_write(sc, 0x06, 0xf4f7);
        re_mdio_write(sc, 0x06, 0x2be0);
        re_mdio_write(sc, 0x06, 0x8b87);
        re_mdio_write(sc, 0x06, 0xad24);
        re_mdio_write(sc, 0x06, 0x0502);
        re_mdio_write(sc, 0x06, 0x8284);
        re_mdio_write(sc, 0x06, 0xf72c);
        re_mdio_write(sc, 0x06, 0xe58a);
        re_mdio_write(sc, 0x06, 0xf4fc);
        re_mdio_write(sc, 0x06, 0x04f8);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x81ad);
        re_mdio_write(sc, 0x06, 0x2600);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x81ad);
        re_mdio_write(sc, 0x06, 0x2109);
        re_mdio_write(sc, 0x06, 0xe08a);
        re_mdio_write(sc, 0x06, 0xf4ac);
        re_mdio_write(sc, 0x06, 0x2003);
        re_mdio_write(sc, 0x06, 0x0222);
        re_mdio_write(sc, 0x06, 0x7de0);
        re_mdio_write(sc, 0x06, 0x8b85);
        re_mdio_write(sc, 0x06, 0xad20);
        re_mdio_write(sc, 0x06, 0x09e0);
        re_mdio_write(sc, 0x06, 0x8af4);
        re_mdio_write(sc, 0x06, 0xac21);
        re_mdio_write(sc, 0x06, 0x0302);
        re_mdio_write(sc, 0x06, 0x1408);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x85ad);
        re_mdio_write(sc, 0x06, 0x2309);
        re_mdio_write(sc, 0x06, 0xe08a);
        re_mdio_write(sc, 0x06, 0xf4ac);
        re_mdio_write(sc, 0x06, 0x2203);
        re_mdio_write(sc, 0x06, 0x0213);
        re_mdio_write(sc, 0x06, 0x07e0);
        re_mdio_write(sc, 0x06, 0x8b87);
        re_mdio_write(sc, 0x06, 0xad24);
        re_mdio_write(sc, 0x06, 0x09e0);
        re_mdio_write(sc, 0x06, 0x8af4);
        re_mdio_write(sc, 0x06, 0xac23);
        re_mdio_write(sc, 0x06, 0x0302);
        re_mdio_write(sc, 0x06, 0x8289);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xf8e1);
        re_mdio_write(sc, 0x06, 0x8af4);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x81ad);
        re_mdio_write(sc, 0x06, 0x2602);
        re_mdio_write(sc, 0x06, 0xf628);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x81ad);
        re_mdio_write(sc, 0x06, 0x210a);
        re_mdio_write(sc, 0x06, 0xe083);
        re_mdio_write(sc, 0x06, 0xecf6);
        re_mdio_write(sc, 0x06, 0x27a0);
        re_mdio_write(sc, 0x06, 0x0502);
        re_mdio_write(sc, 0x06, 0xf629);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x85ad);
        re_mdio_write(sc, 0x06, 0x2008);
        re_mdio_write(sc, 0x06, 0xe08a);
        re_mdio_write(sc, 0x06, 0xe8ad);
        re_mdio_write(sc, 0x06, 0x2102);
        re_mdio_write(sc, 0x06, 0xf62a);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x85ad);
        re_mdio_write(sc, 0x06, 0x2308);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x20a0);
        re_mdio_write(sc, 0x06, 0x0302);
        re_mdio_write(sc, 0x06, 0xf62b);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x87ad);
        re_mdio_write(sc, 0x06, 0x2408);
        re_mdio_write(sc, 0x06, 0xe08a);
        re_mdio_write(sc, 0x06, 0xc2a0);
        re_mdio_write(sc, 0x06, 0x0302);
        re_mdio_write(sc, 0x06, 0xf62c);
        re_mdio_write(sc, 0x06, 0xe58a);
        re_mdio_write(sc, 0x06, 0xf4a1);
        re_mdio_write(sc, 0x06, 0x0008);
        re_mdio_write(sc, 0x06, 0xd100);
        re_mdio_write(sc, 0x06, 0xbf22);
        re_mdio_write(sc, 0x06, 0x7a02);
        re_mdio_write(sc, 0x06, 0x387d);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xee8a);
        re_mdio_write(sc, 0x06, 0xc200);
        re_mdio_write(sc, 0x06, 0x04f8);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x87ad);
        re_mdio_write(sc, 0x06, 0x241e);
        re_mdio_write(sc, 0x06, 0xe08a);
        re_mdio_write(sc, 0x06, 0xc2a0);
        re_mdio_write(sc, 0x06, 0x0005);
        re_mdio_write(sc, 0x06, 0x0282);
        re_mdio_write(sc, 0x06, 0xb0ae);
        re_mdio_write(sc, 0x06, 0xf5a0);
        re_mdio_write(sc, 0x06, 0x0105);
        re_mdio_write(sc, 0x06, 0x0282);
        re_mdio_write(sc, 0x06, 0xc0ae);
        re_mdio_write(sc, 0x06, 0x0ba0);
        re_mdio_write(sc, 0x06, 0x0205);
        re_mdio_write(sc, 0x06, 0x0282);
        re_mdio_write(sc, 0x06, 0xcaae);
        re_mdio_write(sc, 0x06, 0x03a0);
        re_mdio_write(sc, 0x06, 0x0300);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xf8fa);
        re_mdio_write(sc, 0x06, 0xef69);
        re_mdio_write(sc, 0x06, 0x0282);
        re_mdio_write(sc, 0x06, 0xe1ee);
        re_mdio_write(sc, 0x06, 0x8ac2);
        re_mdio_write(sc, 0x06, 0x01ef);
        re_mdio_write(sc, 0x06, 0x96fe);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xf8ee);
        re_mdio_write(sc, 0x06, 0x8ac9);
        re_mdio_write(sc, 0x06, 0x0002);
        re_mdio_write(sc, 0x06, 0x8317);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xf8e0);
        re_mdio_write(sc, 0x06, 0x8ac8);
        re_mdio_write(sc, 0x06, 0xe18a);
        re_mdio_write(sc, 0x06, 0xc91f);
        re_mdio_write(sc, 0x06, 0x019e);
        re_mdio_write(sc, 0x06, 0x0611);
        re_mdio_write(sc, 0x06, 0xe58a);
        re_mdio_write(sc, 0x06, 0xc9ae);
        re_mdio_write(sc, 0x06, 0x04ee);
        re_mdio_write(sc, 0x06, 0x8ac2);
        re_mdio_write(sc, 0x06, 0x01fc);
        re_mdio_write(sc, 0x06, 0x04f8);
        re_mdio_write(sc, 0x06, 0xf9fa);
        re_mdio_write(sc, 0x06, 0xef69);
        re_mdio_write(sc, 0x06, 0xfbbf);
        re_mdio_write(sc, 0x06, 0x8ac4);
        re_mdio_write(sc, 0x06, 0xef79);
        re_mdio_write(sc, 0x06, 0xd200);
        re_mdio_write(sc, 0x06, 0xd400);
        re_mdio_write(sc, 0x06, 0x221e);
        re_mdio_write(sc, 0x06, 0x02bf);
        re_mdio_write(sc, 0x06, 0x3024);
        re_mdio_write(sc, 0x06, 0x0238);
        re_mdio_write(sc, 0x06, 0x7dbf);
        re_mdio_write(sc, 0x06, 0x13ff);
        re_mdio_write(sc, 0x06, 0x0238);
        re_mdio_write(sc, 0x06, 0x500d);
        re_mdio_write(sc, 0x06, 0x4559);
        re_mdio_write(sc, 0x06, 0x1fef);
        re_mdio_write(sc, 0x06, 0x97dd);
        re_mdio_write(sc, 0x06, 0xd308);
        re_mdio_write(sc, 0x06, 0x1a93);
        re_mdio_write(sc, 0x06, 0xdd12);
        re_mdio_write(sc, 0x06, 0x17a2);
        re_mdio_write(sc, 0x06, 0x04de);
        re_mdio_write(sc, 0x06, 0xffef);
        re_mdio_write(sc, 0x06, 0x96fe);
        re_mdio_write(sc, 0x06, 0xfdfc);
        re_mdio_write(sc, 0x06, 0x04f8);
        re_mdio_write(sc, 0x06, 0xf9fa);
        re_mdio_write(sc, 0x06, 0xef69);
        re_mdio_write(sc, 0x06, 0xfbee);
        re_mdio_write(sc, 0x06, 0x8ac2);
        re_mdio_write(sc, 0x06, 0x03d5);
        re_mdio_write(sc, 0x06, 0x0080);
        re_mdio_write(sc, 0x06, 0xbf8a);
        re_mdio_write(sc, 0x06, 0xc4ef);
        re_mdio_write(sc, 0x06, 0x79ef);
        re_mdio_write(sc, 0x06, 0x45bf);
        re_mdio_write(sc, 0x06, 0x3024);
        re_mdio_write(sc, 0x06, 0x0238);
        re_mdio_write(sc, 0x06, 0x7dbf);
        re_mdio_write(sc, 0x06, 0x13ff);
        re_mdio_write(sc, 0x06, 0x0238);
        re_mdio_write(sc, 0x06, 0x50ad);
        re_mdio_write(sc, 0x06, 0x2702);
        re_mdio_write(sc, 0x06, 0x78ff);
        re_mdio_write(sc, 0x06, 0xe18a);
        re_mdio_write(sc, 0x06, 0xca1b);
        re_mdio_write(sc, 0x06, 0x01aa);
        re_mdio_write(sc, 0x06, 0x2eef);
        re_mdio_write(sc, 0x06, 0x97d9);
        re_mdio_write(sc, 0x06, 0x7900);
        re_mdio_write(sc, 0x06, 0x9e2b);
        re_mdio_write(sc, 0x06, 0x81dd);
        re_mdio_write(sc, 0x06, 0xbf85);
        re_mdio_write(sc, 0x06, 0xad02);
        re_mdio_write(sc, 0x06, 0x387d);
        re_mdio_write(sc, 0x06, 0xd101);
        re_mdio_write(sc, 0x06, 0xef02);
        re_mdio_write(sc, 0x06, 0x100c);
        re_mdio_write(sc, 0x06, 0x11b0);
        re_mdio_write(sc, 0x06, 0xfc0d);
        re_mdio_write(sc, 0x06, 0x11bf);
        re_mdio_write(sc, 0x06, 0x85aa);
        re_mdio_write(sc, 0x06, 0x0238);
        re_mdio_write(sc, 0x06, 0x7dd1);
        re_mdio_write(sc, 0x06, 0x00bf);
        re_mdio_write(sc, 0x06, 0x85aa);
        re_mdio_write(sc, 0x06, 0x0238);
        re_mdio_write(sc, 0x06, 0x7dee);
        re_mdio_write(sc, 0x06, 0x8ac2);
        re_mdio_write(sc, 0x06, 0x02ae);
        re_mdio_write(sc, 0x06, 0x0413);
        re_mdio_write(sc, 0x06, 0xa38b);
        re_mdio_write(sc, 0x06, 0xb4d3);
        re_mdio_write(sc, 0x06, 0x8012);
        re_mdio_write(sc, 0x06, 0x17a2);
        re_mdio_write(sc, 0x06, 0x04ad);
        re_mdio_write(sc, 0x06, 0xffef);
        re_mdio_write(sc, 0x06, 0x96fe);
        re_mdio_write(sc, 0x06, 0xfdfc);
        re_mdio_write(sc, 0x06, 0x04f8);
        re_mdio_write(sc, 0x06, 0xf9e0);
        re_mdio_write(sc, 0x06, 0x8b85);
        re_mdio_write(sc, 0x06, 0xad25);
        re_mdio_write(sc, 0x06, 0x48e0);
        re_mdio_write(sc, 0x06, 0x8a96);
        re_mdio_write(sc, 0x06, 0xe18a);
        re_mdio_write(sc, 0x06, 0x977c);
        re_mdio_write(sc, 0x06, 0x0000);
        re_mdio_write(sc, 0x06, 0x9e35);
        re_mdio_write(sc, 0x06, 0xee8a);
        re_mdio_write(sc, 0x06, 0x9600);
        re_mdio_write(sc, 0x06, 0xee8a);
        re_mdio_write(sc, 0x06, 0x9700);
        re_mdio_write(sc, 0x06, 0xe08a);
        re_mdio_write(sc, 0x06, 0xbee1);
        re_mdio_write(sc, 0x06, 0x8abf);
        re_mdio_write(sc, 0x06, 0xe28a);
        re_mdio_write(sc, 0x06, 0xc0e3);
        re_mdio_write(sc, 0x06, 0x8ac1);
        re_mdio_write(sc, 0x06, 0x0237);
        re_mdio_write(sc, 0x06, 0x74ad);
        re_mdio_write(sc, 0x06, 0x2012);
        re_mdio_write(sc, 0x06, 0xee8a);
        re_mdio_write(sc, 0x06, 0x9603);
        re_mdio_write(sc, 0x06, 0xee8a);
        re_mdio_write(sc, 0x06, 0x97b7);
        re_mdio_write(sc, 0x06, 0xee8a);
        re_mdio_write(sc, 0x06, 0xc000);
        re_mdio_write(sc, 0x06, 0xee8a);
        re_mdio_write(sc, 0x06, 0xc100);
        re_mdio_write(sc, 0x06, 0xae11);
        re_mdio_write(sc, 0x06, 0x15e6);
        re_mdio_write(sc, 0x06, 0x8ac0);
        re_mdio_write(sc, 0x06, 0xe78a);
        re_mdio_write(sc, 0x06, 0xc1ae);
        re_mdio_write(sc, 0x06, 0x08ee);
        re_mdio_write(sc, 0x06, 0x8ac0);
        re_mdio_write(sc, 0x06, 0x00ee);
        re_mdio_write(sc, 0x06, 0x8ac1);
        re_mdio_write(sc, 0x06, 0x00fd);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xae20);
        re_mdio_write(sc, 0x06, 0x0000);
        re_mdio_write(sc, 0x06, 0x0000);
        re_mdio_write(sc, 0x06, 0x0000);
        re_mdio_write(sc, 0x06, 0x0000);
        re_mdio_write(sc, 0x06, 0x0000);
        re_mdio_write(sc, 0x06, 0x0000);
        re_mdio_write(sc, 0x06, 0x0000);
        re_mdio_write(sc, 0x06, 0x0000);
        re_mdio_write(sc, 0x06, 0x0000);
        re_mdio_write(sc, 0x06, 0x0000);
        re_mdio_write(sc, 0x06, 0x0000);
        re_mdio_write(sc, 0x06, 0x0000);
        re_mdio_write(sc, 0x06, 0x0000);
        re_mdio_write(sc, 0x06, 0x0000);
        re_mdio_write(sc, 0x06, 0x0000);
        re_mdio_write(sc, 0x06, 0x0000);
        re_mdio_write(sc, 0x06, 0xf8fa);
        re_mdio_write(sc, 0x06, 0xef69);
        re_mdio_write(sc, 0x06, 0xe0e0);
        re_mdio_write(sc, 0x06, 0x00e1);
        re_mdio_write(sc, 0x06, 0xe001);
        re_mdio_write(sc, 0x06, 0xad27);
        re_mdio_write(sc, 0x06, 0x32e0);
        re_mdio_write(sc, 0x06, 0x8b40);
        re_mdio_write(sc, 0x06, 0xf720);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x40bf);
        re_mdio_write(sc, 0x06, 0x3230);
        re_mdio_write(sc, 0x06, 0x0238);
        re_mdio_write(sc, 0x06, 0x50ad);
        re_mdio_write(sc, 0x06, 0x2821);
        re_mdio_write(sc, 0x06, 0xe0e0);
        re_mdio_write(sc, 0x06, 0x20e1);
        re_mdio_write(sc, 0x06, 0xe021);
        re_mdio_write(sc, 0x06, 0xad20);
        re_mdio_write(sc, 0x06, 0x18e0);
        re_mdio_write(sc, 0x06, 0x8b40);
        re_mdio_write(sc, 0x06, 0xf620);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x40ee);
        re_mdio_write(sc, 0x06, 0x8b3b);
        re_mdio_write(sc, 0x06, 0xffe0);
        re_mdio_write(sc, 0x06, 0x8a8a);
        re_mdio_write(sc, 0x06, 0xe18a);
        re_mdio_write(sc, 0x06, 0x8be4);
        re_mdio_write(sc, 0x06, 0xe000);
        re_mdio_write(sc, 0x06, 0xe5e0);
        re_mdio_write(sc, 0x06, 0x01ef);
        re_mdio_write(sc, 0x06, 0x96fe);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xf8f9);
        re_mdio_write(sc, 0x06, 0xface);
        re_mdio_write(sc, 0x06, 0xfaef);
        re_mdio_write(sc, 0x06, 0x69fa);
        re_mdio_write(sc, 0x06, 0xd401);
        re_mdio_write(sc, 0x06, 0x55b4);
        re_mdio_write(sc, 0x06, 0xfebf);
        re_mdio_write(sc, 0x06, 0x1c1e);
        re_mdio_write(sc, 0x06, 0x0238);
        re_mdio_write(sc, 0x06, 0x50ac);
        re_mdio_write(sc, 0x06, 0x280b);
        re_mdio_write(sc, 0x06, 0xbf1c);
        re_mdio_write(sc, 0x06, 0x1b02);
        re_mdio_write(sc, 0x06, 0x3850);
        re_mdio_write(sc, 0x06, 0xac28);
        re_mdio_write(sc, 0x06, 0x49ae);
        re_mdio_write(sc, 0x06, 0x64bf);
        re_mdio_write(sc, 0x06, 0x1c1b);
        re_mdio_write(sc, 0x06, 0x0238);
        re_mdio_write(sc, 0x06, 0x50ac);
        re_mdio_write(sc, 0x06, 0x285b);
        re_mdio_write(sc, 0x06, 0xd000);
        re_mdio_write(sc, 0x06, 0x0284);
        re_mdio_write(sc, 0x06, 0xcaac);
        re_mdio_write(sc, 0x06, 0x2105);
        re_mdio_write(sc, 0x06, 0xac22);
        re_mdio_write(sc, 0x06, 0x02ae);
        re_mdio_write(sc, 0x06, 0x4ebf);
        re_mdio_write(sc, 0x06, 0xe0c4);
        re_mdio_write(sc, 0x06, 0xbe85);
        re_mdio_write(sc, 0x06, 0xf6d2);
        re_mdio_write(sc, 0x06, 0x04d8);
        re_mdio_write(sc, 0x06, 0x19d9);
        re_mdio_write(sc, 0x06, 0x1907);
        re_mdio_write(sc, 0x06, 0xdc19);
        re_mdio_write(sc, 0x06, 0xdd19);
        re_mdio_write(sc, 0x06, 0x0789);
        re_mdio_write(sc, 0x06, 0x89ef);
        re_mdio_write(sc, 0x06, 0x645e);
        re_mdio_write(sc, 0x06, 0x07ff);
        re_mdio_write(sc, 0x06, 0x0d65);
        re_mdio_write(sc, 0x06, 0x5cf8);
        re_mdio_write(sc, 0x06, 0x001e);
        re_mdio_write(sc, 0x06, 0x46dc);
        re_mdio_write(sc, 0x06, 0x19dd);
        re_mdio_write(sc, 0x06, 0x19b2);
        re_mdio_write(sc, 0x06, 0xe2d4);
        re_mdio_write(sc, 0x06, 0x0001);
        re_mdio_write(sc, 0x06, 0xbf1c);
        re_mdio_write(sc, 0x06, 0x1b02);
        re_mdio_write(sc, 0x06, 0x387d);
        re_mdio_write(sc, 0x06, 0xae1d);
        re_mdio_write(sc, 0x06, 0xbee0);
        re_mdio_write(sc, 0x06, 0xc4bf);
        re_mdio_write(sc, 0x06, 0x85f6);
        re_mdio_write(sc, 0x06, 0xd204);
        re_mdio_write(sc, 0x06, 0xd819);
        re_mdio_write(sc, 0x06, 0xd919);
        re_mdio_write(sc, 0x06, 0x07dc);
        re_mdio_write(sc, 0x06, 0x19dd);
        re_mdio_write(sc, 0x06, 0x1907);
        re_mdio_write(sc, 0x06, 0xb2f4);
        re_mdio_write(sc, 0x06, 0xd400);
        re_mdio_write(sc, 0x06, 0x00bf);
        re_mdio_write(sc, 0x06, 0x1c1b);
        re_mdio_write(sc, 0x06, 0x0238);
        re_mdio_write(sc, 0x06, 0x7dfe);
        re_mdio_write(sc, 0x06, 0xef96);
        re_mdio_write(sc, 0x06, 0xfec6);
        re_mdio_write(sc, 0x06, 0xfefd);
        re_mdio_write(sc, 0x06, 0xfc05);
        re_mdio_write(sc, 0x06, 0xf9e2);
        re_mdio_write(sc, 0x06, 0xe0ea);
        re_mdio_write(sc, 0x06, 0xe3e0);
        re_mdio_write(sc, 0x06, 0xeb5a);
        re_mdio_write(sc, 0x06, 0x070c);
        re_mdio_write(sc, 0x06, 0x031e);
        re_mdio_write(sc, 0x06, 0x20e6);
        re_mdio_write(sc, 0x06, 0xe0ea);
        re_mdio_write(sc, 0x06, 0xe7e0);
        re_mdio_write(sc, 0x06, 0xebe0);
        re_mdio_write(sc, 0x06, 0xe0fc);
        re_mdio_write(sc, 0x06, 0xe1e0);
        re_mdio_write(sc, 0x06, 0xfdfd);
        re_mdio_write(sc, 0x06, 0x04f8);
        re_mdio_write(sc, 0x06, 0xfaef);
        re_mdio_write(sc, 0x06, 0x69e0);
        re_mdio_write(sc, 0x06, 0x8b80);
        re_mdio_write(sc, 0x06, 0xad27);
        re_mdio_write(sc, 0x06, 0x22bf);
        re_mdio_write(sc, 0x06, 0x4616);
        re_mdio_write(sc, 0x06, 0x0238);
        re_mdio_write(sc, 0x06, 0x50e0);
        re_mdio_write(sc, 0x06, 0x8b44);
        re_mdio_write(sc, 0x06, 0x1f01);
        re_mdio_write(sc, 0x06, 0x9e15);
        re_mdio_write(sc, 0x06, 0xe58b);
        re_mdio_write(sc, 0x06, 0x44ad);
        re_mdio_write(sc, 0x06, 0x2907);
        re_mdio_write(sc, 0x06, 0xac28);
        re_mdio_write(sc, 0x06, 0x04d1);
        re_mdio_write(sc, 0x06, 0x01ae);
        re_mdio_write(sc, 0x06, 0x02d1);
        re_mdio_write(sc, 0x06, 0x00bf);
        re_mdio_write(sc, 0x06, 0x85b0);
        re_mdio_write(sc, 0x06, 0x0238);
        re_mdio_write(sc, 0x06, 0x7def);
        re_mdio_write(sc, 0x06, 0x96fe);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xf8e0);
        re_mdio_write(sc, 0x06, 0x8b85);
        re_mdio_write(sc, 0x06, 0xad26);
        re_mdio_write(sc, 0x06, 0x30e0);
        re_mdio_write(sc, 0x06, 0xe036);
        re_mdio_write(sc, 0x06, 0xe1e0);
        re_mdio_write(sc, 0x06, 0x37e1);
        re_mdio_write(sc, 0x06, 0x8b3f);
        re_mdio_write(sc, 0x06, 0x1f10);
        re_mdio_write(sc, 0x06, 0x9e23);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x3fac);
        re_mdio_write(sc, 0x06, 0x200b);
        re_mdio_write(sc, 0x06, 0xac21);
        re_mdio_write(sc, 0x06, 0x0dac);
        re_mdio_write(sc, 0x06, 0x250f);
        re_mdio_write(sc, 0x06, 0xac27);
        re_mdio_write(sc, 0x06, 0x11ae);
        re_mdio_write(sc, 0x06, 0x1202);
        re_mdio_write(sc, 0x06, 0x2c47);
        re_mdio_write(sc, 0x06, 0xae0d);
        re_mdio_write(sc, 0x06, 0x0285);
        re_mdio_write(sc, 0x06, 0x4fae);
        re_mdio_write(sc, 0x06, 0x0802);
        re_mdio_write(sc, 0x06, 0x2c69);
        re_mdio_write(sc, 0x06, 0xae03);
        re_mdio_write(sc, 0x06, 0x022c);
        re_mdio_write(sc, 0x06, 0x7cfc);
        re_mdio_write(sc, 0x06, 0x04f8);
        re_mdio_write(sc, 0x06, 0xfaef);
        re_mdio_write(sc, 0x06, 0x6902);
        re_mdio_write(sc, 0x06, 0x856c);
        re_mdio_write(sc, 0x06, 0xe0e0);
        re_mdio_write(sc, 0x06, 0x14e1);
        re_mdio_write(sc, 0x06, 0xe015);
        re_mdio_write(sc, 0x06, 0xad26);
        re_mdio_write(sc, 0x06, 0x08d1);
        re_mdio_write(sc, 0x06, 0x1ebf);
        re_mdio_write(sc, 0x06, 0x2cd9);
        re_mdio_write(sc, 0x06, 0x0238);
        re_mdio_write(sc, 0x06, 0x7def);
        re_mdio_write(sc, 0x06, 0x96fe);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xf8e0);
        re_mdio_write(sc, 0x06, 0x8b85);
        re_mdio_write(sc, 0x06, 0xad27);
        re_mdio_write(sc, 0x06, 0x2fd0);
        re_mdio_write(sc, 0x06, 0x0b02);
        re_mdio_write(sc, 0x06, 0x3682);
        re_mdio_write(sc, 0x06, 0x5882);
        re_mdio_write(sc, 0x06, 0x7882);
        re_mdio_write(sc, 0x06, 0x9f24);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x32e1);
        re_mdio_write(sc, 0x06, 0x8b33);
        re_mdio_write(sc, 0x06, 0x1f10);
        re_mdio_write(sc, 0x06, 0x9e1a);
        re_mdio_write(sc, 0x06, 0x10e4);
        re_mdio_write(sc, 0x06, 0x8b32);
        re_mdio_write(sc, 0x06, 0xe0e0);
        re_mdio_write(sc, 0x06, 0x28e1);
        re_mdio_write(sc, 0x06, 0xe029);
        re_mdio_write(sc, 0x06, 0xf72c);
        re_mdio_write(sc, 0x06, 0xe4e0);
        re_mdio_write(sc, 0x06, 0x28e5);
        re_mdio_write(sc, 0x06, 0xe029);
        re_mdio_write(sc, 0x06, 0xf62c);
        re_mdio_write(sc, 0x06, 0xe4e0);
        re_mdio_write(sc, 0x06, 0x28e5);
        re_mdio_write(sc, 0x06, 0xe029);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0x00e1);
        re_mdio_write(sc, 0x06, 0x4077);
        re_mdio_write(sc, 0x06, 0xe140);
        re_mdio_write(sc, 0x06, 0x52e0);
        re_mdio_write(sc, 0x06, 0xeed9);
        re_mdio_write(sc, 0x06, 0xe04c);
        re_mdio_write(sc, 0x06, 0xbbe0);
        re_mdio_write(sc, 0x06, 0x2a00);
        re_mdio_write(sc, 0x05, 0xe142);
        PhyRegValue = re_mdio_read(sc, 0x06);
        PhyRegValue |= BIT_0;
        re_mdio_write(sc, 0x06, PhyRegValue);
        re_mdio_write(sc, 0x05, 0xe140);
        PhyRegValue = re_mdio_read(sc, 0x06);
        PhyRegValue |= BIT_0;
        re_mdio_write(sc, 0x06, PhyRegValue);
        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x1f, 0x0005);
        for (i = 0; i < 200; i++) {
                DELAY(100);
                PhyRegValue = re_mdio_read(sc, 0x00);
                if (PhyRegValue & BIT_7)
                        break;
        }

        re_mdio_write(sc, 0x1f, 0x0007);
        re_mdio_write(sc, 0x1e, 0x0023);
        PhyRegValue = re_mdio_read(sc, 0x17);
        PhyRegValue |= BIT_1;
        if (sc->RequiredSecLanDonglePatch)
                PhyRegValue &= ~(BIT_2);
        re_mdio_write(sc, 0x17, PhyRegValue);
        re_mdio_write(sc, 0x1f, 0x0000);

        re_mdio_write(sc, 0x1f, 0x0003);
        re_mdio_write(sc, 0x09, 0xA20F);
        re_mdio_write(sc, 0x1f, 0x0000);

        re_mdio_write(sc, 0x1f, 0x0003);
        re_mdio_write(sc, 0x01, 0x328A);
        re_mdio_write(sc, 0x1f, 0x0000);

        re_mdio_write(sc, 0x1f, 0x0003);
        PhyRegValue = re_mdio_read(sc, 0x19);
        PhyRegValue &= ~BIT_0;
        re_mdio_write(sc, 0x19, PhyRegValue);
        PhyRegValue = re_mdio_read(sc, 0x10);
        PhyRegValue &= ~BIT_10;
        re_mdio_write(sc, 0x10, PhyRegValue);
        re_mdio_write(sc, 0x1f, 0x0000);


        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x00, 0x9200);
}

static void re_set_phy_mcu_8168f_2(struct re_softc *sc)
{
        u_int16_t PhyRegValue;
        int i;

        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x00, 0x1800);
        PhyRegValue = re_mdio_read(sc, 0x15);
        PhyRegValue &= ~(BIT_12);
        re_mdio_write(sc, 0x15, PhyRegValue);
        re_mdio_write(sc, 0x00, 0x4800);
        re_mdio_write(sc, 0x1f, 0x0007);
        re_mdio_write(sc, 0x1e, 0x002f);
        for (i = 0; i < 1000; i++) {
                DELAY(100);
                PhyRegValue = re_mdio_read(sc, 0x1c);
                if (PhyRegValue & BIT_7)
                        break;
        }
        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x00, 0x1800);
        re_mdio_write(sc, 0x1f, 0x0007);
        re_mdio_write(sc, 0x1e, 0x0023);
        for (i = 0; i < 200; i++) {
                DELAY(100);
                PhyRegValue = re_mdio_read(sc, 0x18);
                if (!(PhyRegValue & BIT_0))
                        break;
        }
        re_mdio_write(sc, 0x1f, 0x0005);
        re_mdio_write(sc, 0x05, 0xfff6);
        re_mdio_write(sc, 0x06, 0x0080);
        re_mdio_write(sc, 0x1f, 0x0007);
        re_mdio_write(sc, 0x1e, 0x0023);
        re_mdio_write(sc, 0x16, 0x0306);
        re_mdio_write(sc, 0x16, 0x0307);
        re_mdio_write(sc, 0x15, 0x0098);
        re_mdio_write(sc, 0x19, 0x7c0b);
        re_mdio_write(sc, 0x15, 0x0099);
        re_mdio_write(sc, 0x19, 0x6c0b);
        re_mdio_write(sc, 0x15, 0x00eb);
        re_mdio_write(sc, 0x19, 0x6c0b);
        re_mdio_write(sc, 0x15, 0x00f8);
        re_mdio_write(sc, 0x19, 0x6f0b);
        re_mdio_write(sc, 0x15, 0x00fe);
        re_mdio_write(sc, 0x19, 0x6f0f);
        re_mdio_write(sc, 0x15, 0x00db);
        re_mdio_write(sc, 0x19, 0x6f09);
        re_mdio_write(sc, 0x15, 0x00dc);
        re_mdio_write(sc, 0x19, 0xaefd);
        re_mdio_write(sc, 0x15, 0x00dd);
        re_mdio_write(sc, 0x19, 0x6f0b);
        re_mdio_write(sc, 0x15, 0x00de);
        re_mdio_write(sc, 0x19, 0xc60b);
        re_mdio_write(sc, 0x15, 0x00df);
        re_mdio_write(sc, 0x19, 0x00fa);
        re_mdio_write(sc, 0x15, 0x00e0);
        re_mdio_write(sc, 0x19, 0x30e1);
        re_mdio_write(sc, 0x15, 0x020c);
        re_mdio_write(sc, 0x19, 0x3224);
        re_mdio_write(sc, 0x15, 0x020e);
        re_mdio_write(sc, 0x19, 0x9813);
        re_mdio_write(sc, 0x15, 0x020f);
        re_mdio_write(sc, 0x19, 0x7801);
        re_mdio_write(sc, 0x15, 0x0210);
        re_mdio_write(sc, 0x19, 0x930f);
        re_mdio_write(sc, 0x15, 0x0211);
        re_mdio_write(sc, 0x19, 0x9206);
        re_mdio_write(sc, 0x15, 0x0212);
        re_mdio_write(sc, 0x19, 0x4002);
        re_mdio_write(sc, 0x15, 0x0213);
        re_mdio_write(sc, 0x19, 0x7800);
        re_mdio_write(sc, 0x15, 0x0214);
        re_mdio_write(sc, 0x19, 0x588f);
        re_mdio_write(sc, 0x15, 0x0215);
        re_mdio_write(sc, 0x19, 0x5520);
        re_mdio_write(sc, 0x15, 0x0216);
        re_mdio_write(sc, 0x19, 0x3224);
        re_mdio_write(sc, 0x15, 0x0217);
        re_mdio_write(sc, 0x19, 0x4002);
        re_mdio_write(sc, 0x15, 0x0218);
        re_mdio_write(sc, 0x19, 0x7800);
        re_mdio_write(sc, 0x15, 0x0219);
        re_mdio_write(sc, 0x19, 0x588d);
        re_mdio_write(sc, 0x15, 0x021a);
        re_mdio_write(sc, 0x19, 0x5540);
        re_mdio_write(sc, 0x15, 0x021b);
        re_mdio_write(sc, 0x19, 0x9e03);
        re_mdio_write(sc, 0x15, 0x021c);
        re_mdio_write(sc, 0x19, 0x7c40);
        re_mdio_write(sc, 0x15, 0x021d);
        re_mdio_write(sc, 0x19, 0x6840);
        re_mdio_write(sc, 0x15, 0x021e);
        re_mdio_write(sc, 0x19, 0x3224);
        re_mdio_write(sc, 0x15, 0x021f);
        re_mdio_write(sc, 0x19, 0x4002);
        re_mdio_write(sc, 0x15, 0x0220);
        re_mdio_write(sc, 0x19, 0x3224);
        re_mdio_write(sc, 0x15, 0x0221);
        re_mdio_write(sc, 0x19, 0x9e03);
        re_mdio_write(sc, 0x15, 0x0222);
        re_mdio_write(sc, 0x19, 0x7c40);
        re_mdio_write(sc, 0x15, 0x0223);
        re_mdio_write(sc, 0x19, 0x6840);
        re_mdio_write(sc, 0x15, 0x0224);
        re_mdio_write(sc, 0x19, 0x7800);
        re_mdio_write(sc, 0x15, 0x0225);
        re_mdio_write(sc, 0x19, 0x3231);
        re_mdio_write(sc, 0x15, 0x0000);
        re_mdio_write(sc, 0x16, 0x0306);
        re_mdio_write(sc, 0x16, 0x0300);
        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x1f, 0x0005);
        re_mdio_write(sc, 0x05, 0xfff6);
        re_mdio_write(sc, 0x06, 0x0080);
        re_mdio_write(sc, 0x05, 0x8000);
        re_mdio_write(sc, 0x06, 0x0280);
        re_mdio_write(sc, 0x06, 0x48f7);
        re_mdio_write(sc, 0x06, 0x00e0);
        re_mdio_write(sc, 0x06, 0xfff7);
        re_mdio_write(sc, 0x06, 0xa080);
        re_mdio_write(sc, 0x06, 0x02ae);
        re_mdio_write(sc, 0x06, 0xf602);
        re_mdio_write(sc, 0x06, 0x011b);
        re_mdio_write(sc, 0x06, 0x0201);
        re_mdio_write(sc, 0x06, 0x2802);
        re_mdio_write(sc, 0x06, 0x0135);
        re_mdio_write(sc, 0x06, 0x0201);
        re_mdio_write(sc, 0x06, 0x4502);
        re_mdio_write(sc, 0x06, 0x015f);
        re_mdio_write(sc, 0x06, 0x0280);
        re_mdio_write(sc, 0x06, 0x6b02);
        re_mdio_write(sc, 0x06, 0x80e5);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x88e1);
        re_mdio_write(sc, 0x06, 0x8b89);
        re_mdio_write(sc, 0x06, 0x1e01);
        re_mdio_write(sc, 0x06, 0xe18b);
        re_mdio_write(sc, 0x06, 0x8a1e);
        re_mdio_write(sc, 0x06, 0x01e1);
        re_mdio_write(sc, 0x06, 0x8b8b);
        re_mdio_write(sc, 0x06, 0x1e01);
        re_mdio_write(sc, 0x06, 0xe18b);
        re_mdio_write(sc, 0x06, 0x8c1e);
        re_mdio_write(sc, 0x06, 0x01e1);
        re_mdio_write(sc, 0x06, 0x8b8d);
        re_mdio_write(sc, 0x06, 0x1e01);
        re_mdio_write(sc, 0x06, 0xe18b);
        re_mdio_write(sc, 0x06, 0x8e1e);
        re_mdio_write(sc, 0x06, 0x01a0);
        re_mdio_write(sc, 0x06, 0x00c7);
        re_mdio_write(sc, 0x06, 0xaebb);
        re_mdio_write(sc, 0x06, 0xbf8b);
        re_mdio_write(sc, 0x06, 0x88ec);
        re_mdio_write(sc, 0x06, 0x0019);
        re_mdio_write(sc, 0x06, 0xa98b);
        re_mdio_write(sc, 0x06, 0x90f9);
        re_mdio_write(sc, 0x06, 0xeeff);
        re_mdio_write(sc, 0x06, 0xf600);
        re_mdio_write(sc, 0x06, 0xeeff);
        re_mdio_write(sc, 0x06, 0xf7fe);
        re_mdio_write(sc, 0x06, 0xd100);
        re_mdio_write(sc, 0x06, 0xbf81);
        re_mdio_write(sc, 0x06, 0x9802);
        re_mdio_write(sc, 0x06, 0x39f3);
        re_mdio_write(sc, 0x06, 0xd101);
        re_mdio_write(sc, 0x06, 0xbf81);
        re_mdio_write(sc, 0x06, 0x9b02);
        re_mdio_write(sc, 0x06, 0x39f3);
        re_mdio_write(sc, 0x06, 0x04f8);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x8dad);
        re_mdio_write(sc, 0x06, 0x2014);
        re_mdio_write(sc, 0x06, 0xee8b);
        re_mdio_write(sc, 0x06, 0x8d00);
        re_mdio_write(sc, 0x06, 0xe08a);
        re_mdio_write(sc, 0x06, 0x5a78);
        re_mdio_write(sc, 0x06, 0x039e);
        re_mdio_write(sc, 0x06, 0x0902);
        re_mdio_write(sc, 0x06, 0x05fc);
        re_mdio_write(sc, 0x06, 0x0280);
        re_mdio_write(sc, 0x06, 0x8802);
        re_mdio_write(sc, 0x06, 0x32dd);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xf8f9);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x81ac);
        re_mdio_write(sc, 0x06, 0x261a);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x81ac);
        re_mdio_write(sc, 0x06, 0x2114);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x85ac);
        re_mdio_write(sc, 0x06, 0x200e);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x85ac);
        re_mdio_write(sc, 0x06, 0x2308);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x87ac);
        re_mdio_write(sc, 0x06, 0x2402);
        re_mdio_write(sc, 0x06, 0xae38);
        re_mdio_write(sc, 0x06, 0x021a);
        re_mdio_write(sc, 0x06, 0xd6ee);
        re_mdio_write(sc, 0x06, 0xe41c);
        re_mdio_write(sc, 0x06, 0x04ee);
        re_mdio_write(sc, 0x06, 0xe41d);
        re_mdio_write(sc, 0x06, 0x04e2);
        re_mdio_write(sc, 0x06, 0xe07c);
        re_mdio_write(sc, 0x06, 0xe3e0);
        re_mdio_write(sc, 0x06, 0x7de0);
        re_mdio_write(sc, 0x06, 0xe038);
        re_mdio_write(sc, 0x06, 0xe1e0);
        re_mdio_write(sc, 0x06, 0x39ad);
        re_mdio_write(sc, 0x06, 0x2e1b);
        re_mdio_write(sc, 0x06, 0xad39);
        re_mdio_write(sc, 0x06, 0x0dd1);
        re_mdio_write(sc, 0x06, 0x01bf);
        re_mdio_write(sc, 0x06, 0x22c8);
        re_mdio_write(sc, 0x06, 0x0239);
        re_mdio_write(sc, 0x06, 0xf302);
        re_mdio_write(sc, 0x06, 0x21f0);
        re_mdio_write(sc, 0x06, 0xae0b);
        re_mdio_write(sc, 0x06, 0xac38);
        re_mdio_write(sc, 0x06, 0x02ae);
        re_mdio_write(sc, 0x06, 0x0602);
        re_mdio_write(sc, 0x06, 0x222d);
        re_mdio_write(sc, 0x06, 0x0222);
        re_mdio_write(sc, 0x06, 0x7202);
        re_mdio_write(sc, 0x06, 0x1ae7);
        re_mdio_write(sc, 0x06, 0xfdfc);
        re_mdio_write(sc, 0x06, 0x04f8);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x8ead);
        re_mdio_write(sc, 0x06, 0x201a);
        re_mdio_write(sc, 0x06, 0xf620);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x8e02);
        re_mdio_write(sc, 0x06, 0x2afe);
        re_mdio_write(sc, 0x06, 0x022c);
        re_mdio_write(sc, 0x06, 0x5c02);
        re_mdio_write(sc, 0x06, 0x03c5);
        re_mdio_write(sc, 0x06, 0x0281);
        re_mdio_write(sc, 0x06, 0x6702);
        re_mdio_write(sc, 0x06, 0x2e4f);
        re_mdio_write(sc, 0x06, 0x0204);
        re_mdio_write(sc, 0x06, 0x8902);
        re_mdio_write(sc, 0x06, 0x2f7a);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x8ead);
        re_mdio_write(sc, 0x06, 0x210b);
        re_mdio_write(sc, 0x06, 0xf621);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x8e02);
        re_mdio_write(sc, 0x06, 0x0445);
        re_mdio_write(sc, 0x06, 0x021c);
        re_mdio_write(sc, 0x06, 0xb8e0);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0xad22);
        re_mdio_write(sc, 0x06, 0x08f6);
        re_mdio_write(sc, 0x06, 0x22e4);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0x0235);
        re_mdio_write(sc, 0x06, 0xd4e0);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0xad23);
        re_mdio_write(sc, 0x06, 0x08f6);
        re_mdio_write(sc, 0x06, 0x23e4);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0x0231);
        re_mdio_write(sc, 0x06, 0xc8e0);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0xad24);
        re_mdio_write(sc, 0x06, 0x05f6);
        re_mdio_write(sc, 0x06, 0x24e4);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x8ead);
        re_mdio_write(sc, 0x06, 0x2505);
        re_mdio_write(sc, 0x06, 0xf625);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x8ee0);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0xad26);
        re_mdio_write(sc, 0x06, 0x08f6);
        re_mdio_write(sc, 0x06, 0x26e4);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0x022d);
        re_mdio_write(sc, 0x06, 0x6ae0);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0xad27);
        re_mdio_write(sc, 0x06, 0x05f6);
        re_mdio_write(sc, 0x06, 0x27e4);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0x0203);
        re_mdio_write(sc, 0x06, 0x8bfc);
        re_mdio_write(sc, 0x06, 0x04f8);
        re_mdio_write(sc, 0x06, 0xfaef);
        re_mdio_write(sc, 0x06, 0x69e0);
        re_mdio_write(sc, 0x06, 0x8b80);
        re_mdio_write(sc, 0x06, 0xad27);
        re_mdio_write(sc, 0x06, 0x22bf);
        re_mdio_write(sc, 0x06, 0x479a);
        re_mdio_write(sc, 0x06, 0x0239);
        re_mdio_write(sc, 0x06, 0xc6e0);
        re_mdio_write(sc, 0x06, 0x8b44);
        re_mdio_write(sc, 0x06, 0x1f01);
        re_mdio_write(sc, 0x06, 0x9e15);
        re_mdio_write(sc, 0x06, 0xe58b);
        re_mdio_write(sc, 0x06, 0x44ad);
        re_mdio_write(sc, 0x06, 0x2907);
        re_mdio_write(sc, 0x06, 0xac28);
        re_mdio_write(sc, 0x06, 0x04d1);
        re_mdio_write(sc, 0x06, 0x01ae);
        re_mdio_write(sc, 0x06, 0x02d1);
        re_mdio_write(sc, 0x06, 0x00bf);
        re_mdio_write(sc, 0x06, 0x819e);
        re_mdio_write(sc, 0x06, 0x0239);
        re_mdio_write(sc, 0x06, 0xf3ef);
        re_mdio_write(sc, 0x06, 0x96fe);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0x00e1);
        re_mdio_write(sc, 0x06, 0x4077);
        re_mdio_write(sc, 0x06, 0xe140);
        re_mdio_write(sc, 0x06, 0xbbe0);
        re_mdio_write(sc, 0x06, 0x2a00);
        re_mdio_write(sc, 0x05, 0xe142);
        PhyRegValue = re_mdio_read(sc, 0x06);
        PhyRegValue |= BIT_0;
        re_mdio_write(sc, 0x06, PhyRegValue);
        re_mdio_write(sc, 0x05, 0xe140);
        PhyRegValue = re_mdio_read(sc, 0x06);
        PhyRegValue |= BIT_0;
        re_mdio_write(sc, 0x06, PhyRegValue);
        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x1f, 0x0005);
        for (i = 0; i < 200; i++) {
                DELAY(100);
                PhyRegValue = re_mdio_read(sc, 0x00);
                if (PhyRegValue & BIT_7)
                        break;
        }

        re_mdio_write(sc, 0x1f, 0x0007);
        re_mdio_write(sc, 0x1e, 0x0023);
        PhyRegValue = re_mdio_read(sc, 0x17);
        PhyRegValue |= BIT_1;
        if (sc->RequiredSecLanDonglePatch)
                PhyRegValue &= ~(BIT_2);
        re_mdio_write(sc, 0x17, PhyRegValue);
        re_mdio_write(sc, 0x1f, 0x0000);

        re_mdio_write(sc, 0x1f, 0x0003);
        PhyRegValue = re_mdio_read(sc, 0x19);
        PhyRegValue &= ~BIT_0;
        re_mdio_write(sc, 0x19, PhyRegValue);
        PhyRegValue = re_mdio_read(sc, 0x10);
        PhyRegValue &= ~BIT_10;
        re_mdio_write(sc, 0x10, PhyRegValue);

        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x00, 0x9200);
}

static void re_set_phy_mcu_8411_1(struct re_softc *sc)
{
        u_int16_t PhyRegValue;
        int i;

        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x00, 0x1800);
        PhyRegValue = re_mdio_read(sc, 0x15);
        PhyRegValue &= ~(BIT_12);
        re_mdio_write(sc, 0x15, PhyRegValue);
        re_mdio_write(sc, 0x00, 0x4800);
        re_mdio_write(sc, 0x1f, 0x0007);
        re_mdio_write(sc, 0x1e, 0x002f);
        for (i = 0; i < 1000; i++) {
                DELAY(100);
                PhyRegValue = re_mdio_read(sc, 0x1c);
                if (PhyRegValue & BIT_7)
                        break;
        }
        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x00, 0x1800);
        re_mdio_write(sc, 0x1f, 0x0007);
        re_mdio_write(sc, 0x1e, 0x0023);
        for (i = 0; i < 200; i++) {
                DELAY(100);
                PhyRegValue = re_mdio_read(sc, 0x18);
                if (!(PhyRegValue & BIT_0))
                        break;
        }
        re_mdio_write(sc, 0x1f, 0x0005);
        re_mdio_write(sc, 0x05, 0xfff6);
        re_mdio_write(sc, 0x06, 0x0080);
        re_mdio_write(sc, 0x1f, 0x0007);
        re_mdio_write(sc, 0x1e, 0x0023);
        re_mdio_write(sc, 0x16, 0x0306);
        re_mdio_write(sc, 0x16, 0x0307);
        re_mdio_write(sc, 0x15, 0x0098);
        re_mdio_write(sc, 0x19, 0x7c0b);
        re_mdio_write(sc, 0x15, 0x0099);
        re_mdio_write(sc, 0x19, 0x6c0b);
        re_mdio_write(sc, 0x15, 0x00eb);
        re_mdio_write(sc, 0x19, 0x6c0b);
        re_mdio_write(sc, 0x15, 0x00f8);
        re_mdio_write(sc, 0x19, 0x6f0b);
        re_mdio_write(sc, 0x15, 0x00fe);
        re_mdio_write(sc, 0x19, 0x6f0f);
        re_mdio_write(sc, 0x15, 0x00db);
        re_mdio_write(sc, 0x19, 0x6f09);
        re_mdio_write(sc, 0x15, 0x00dc);
        re_mdio_write(sc, 0x19, 0xaefd);
        re_mdio_write(sc, 0x15, 0x00dd);
        re_mdio_write(sc, 0x19, 0x6f0b);
        re_mdio_write(sc, 0x15, 0x00de);
        re_mdio_write(sc, 0x19, 0xc60b);
        re_mdio_write(sc, 0x15, 0x00df);
        re_mdio_write(sc, 0x19, 0x00fa);
        re_mdio_write(sc, 0x15, 0x00e0);
        re_mdio_write(sc, 0x19, 0x30e1);
        re_mdio_write(sc, 0x15, 0x020c);
        re_mdio_write(sc, 0x19, 0x3224);
        re_mdio_write(sc, 0x15, 0x020e);
        re_mdio_write(sc, 0x19, 0x9813);
        re_mdio_write(sc, 0x15, 0x020f);
        re_mdio_write(sc, 0x19, 0x7801);
        re_mdio_write(sc, 0x15, 0x0210);
        re_mdio_write(sc, 0x19, 0x930f);
        re_mdio_write(sc, 0x15, 0x0211);
        re_mdio_write(sc, 0x19, 0x9206);
        re_mdio_write(sc, 0x15, 0x0212);
        re_mdio_write(sc, 0x19, 0x4002);
        re_mdio_write(sc, 0x15, 0x0213);
        re_mdio_write(sc, 0x19, 0x7800);
        re_mdio_write(sc, 0x15, 0x0214);
        re_mdio_write(sc, 0x19, 0x588f);
        re_mdio_write(sc, 0x15, 0x0215);
        re_mdio_write(sc, 0x19, 0x5520);
        re_mdio_write(sc, 0x15, 0x0216);
        re_mdio_write(sc, 0x19, 0x3224);
        re_mdio_write(sc, 0x15, 0x0217);
        re_mdio_write(sc, 0x19, 0x4002);
        re_mdio_write(sc, 0x15, 0x0218);
        re_mdio_write(sc, 0x19, 0x7800);
        re_mdio_write(sc, 0x15, 0x0219);
        re_mdio_write(sc, 0x19, 0x588d);
        re_mdio_write(sc, 0x15, 0x021a);
        re_mdio_write(sc, 0x19, 0x5540);
        re_mdio_write(sc, 0x15, 0x021b);
        re_mdio_write(sc, 0x19, 0x9e03);
        re_mdio_write(sc, 0x15, 0x021c);
        re_mdio_write(sc, 0x19, 0x7c40);
        re_mdio_write(sc, 0x15, 0x021d);
        re_mdio_write(sc, 0x19, 0x6840);
        re_mdio_write(sc, 0x15, 0x021e);
        re_mdio_write(sc, 0x19, 0x3224);
        re_mdio_write(sc, 0x15, 0x021f);
        re_mdio_write(sc, 0x19, 0x4002);
        re_mdio_write(sc, 0x15, 0x0220);
        re_mdio_write(sc, 0x19, 0x3224);
        re_mdio_write(sc, 0x15, 0x0221);
        re_mdio_write(sc, 0x19, 0x9e03);
        re_mdio_write(sc, 0x15, 0x0222);
        re_mdio_write(sc, 0x19, 0x7c40);
        re_mdio_write(sc, 0x15, 0x0223);
        re_mdio_write(sc, 0x19, 0x6840);
        re_mdio_write(sc, 0x15, 0x0224);
        re_mdio_write(sc, 0x19, 0x7800);
        re_mdio_write(sc, 0x15, 0x0225);
        re_mdio_write(sc, 0x19, 0x3231);
        re_mdio_write(sc, 0x15, 0x0000);
        re_mdio_write(sc, 0x16, 0x0306);
        re_mdio_write(sc, 0x16, 0x0300);
        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x1f, 0x0005);
        re_mdio_write(sc, 0x05, 0xfff6);
        re_mdio_write(sc, 0x06, 0x0080);
        re_mdio_write(sc, 0x05, 0x8000);
        re_mdio_write(sc, 0x06, 0x0280);
        re_mdio_write(sc, 0x06, 0x48f7);
        re_mdio_write(sc, 0x06, 0x00e0);
        re_mdio_write(sc, 0x06, 0xfff7);
        re_mdio_write(sc, 0x06, 0xa080);
        re_mdio_write(sc, 0x06, 0x02ae);
        re_mdio_write(sc, 0x06, 0xf602);
        re_mdio_write(sc, 0x06, 0x011e);
        re_mdio_write(sc, 0x06, 0x0201);
        re_mdio_write(sc, 0x06, 0x2b02);
        re_mdio_write(sc, 0x06, 0x8077);
        re_mdio_write(sc, 0x06, 0x0201);
        re_mdio_write(sc, 0x06, 0x4802);
        re_mdio_write(sc, 0x06, 0x0162);
        re_mdio_write(sc, 0x06, 0x0280);
        re_mdio_write(sc, 0x06, 0x9402);
        re_mdio_write(sc, 0x06, 0x810e);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x88e1);
        re_mdio_write(sc, 0x06, 0x8b89);
        re_mdio_write(sc, 0x06, 0x1e01);
        re_mdio_write(sc, 0x06, 0xe18b);
        re_mdio_write(sc, 0x06, 0x8a1e);
        re_mdio_write(sc, 0x06, 0x01e1);
        re_mdio_write(sc, 0x06, 0x8b8b);
        re_mdio_write(sc, 0x06, 0x1e01);
        re_mdio_write(sc, 0x06, 0xe18b);
        re_mdio_write(sc, 0x06, 0x8c1e);
        re_mdio_write(sc, 0x06, 0x01e1);
        re_mdio_write(sc, 0x06, 0x8b8d);
        re_mdio_write(sc, 0x06, 0x1e01);
        re_mdio_write(sc, 0x06, 0xe18b);
        re_mdio_write(sc, 0x06, 0x8e1e);
        re_mdio_write(sc, 0x06, 0x01a0);
        re_mdio_write(sc, 0x06, 0x00c7);
        re_mdio_write(sc, 0x06, 0xaebb);
        re_mdio_write(sc, 0x06, 0xd481);
        re_mdio_write(sc, 0x06, 0xd4e4);
        re_mdio_write(sc, 0x06, 0x8b92);
        re_mdio_write(sc, 0x06, 0xe58b);
        re_mdio_write(sc, 0x06, 0x9302);
        re_mdio_write(sc, 0x06, 0x2e5a);
        re_mdio_write(sc, 0x06, 0xbf8b);
        re_mdio_write(sc, 0x06, 0x88ec);
        re_mdio_write(sc, 0x06, 0x0019);
        re_mdio_write(sc, 0x06, 0xa98b);
        re_mdio_write(sc, 0x06, 0x90f9);
        re_mdio_write(sc, 0x06, 0xeeff);
        re_mdio_write(sc, 0x06, 0xf600);
        re_mdio_write(sc, 0x06, 0xeeff);
        re_mdio_write(sc, 0x06, 0xf7fc);
        re_mdio_write(sc, 0x06, 0xd100);
        re_mdio_write(sc, 0x06, 0xbf83);
        re_mdio_write(sc, 0x06, 0x3c02);
        re_mdio_write(sc, 0x06, 0x3a21);
        re_mdio_write(sc, 0x06, 0xd101);
        re_mdio_write(sc, 0x06, 0xbf83);
        re_mdio_write(sc, 0x06, 0x3f02);
        re_mdio_write(sc, 0x06, 0x3a21);
        re_mdio_write(sc, 0x06, 0x04f8);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x8aad);
        re_mdio_write(sc, 0x06, 0x2014);
        re_mdio_write(sc, 0x06, 0xee8b);
        re_mdio_write(sc, 0x06, 0x8a00);
        re_mdio_write(sc, 0x06, 0x0220);
        re_mdio_write(sc, 0x06, 0x8be0);
        re_mdio_write(sc, 0x06, 0xe426);
        re_mdio_write(sc, 0x06, 0xe1e4);
        re_mdio_write(sc, 0x06, 0x27ee);
        re_mdio_write(sc, 0x06, 0xe426);
        re_mdio_write(sc, 0x06, 0x23e5);
        re_mdio_write(sc, 0x06, 0xe427);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xf8e0);
        re_mdio_write(sc, 0x06, 0x8b8d);
        re_mdio_write(sc, 0x06, 0xad20);
        re_mdio_write(sc, 0x06, 0x14ee);
        re_mdio_write(sc, 0x06, 0x8b8d);
        re_mdio_write(sc, 0x06, 0x00e0);
        re_mdio_write(sc, 0x06, 0x8a5a);
        re_mdio_write(sc, 0x06, 0x7803);
        re_mdio_write(sc, 0x06, 0x9e09);
        re_mdio_write(sc, 0x06, 0x0206);
        re_mdio_write(sc, 0x06, 0x2802);
        re_mdio_write(sc, 0x06, 0x80b1);
        re_mdio_write(sc, 0x06, 0x0232);
        re_mdio_write(sc, 0x06, 0xfdfc);
        re_mdio_write(sc, 0x06, 0x04f8);
        re_mdio_write(sc, 0x06, 0xf9e0);
        re_mdio_write(sc, 0x06, 0x8b81);
        re_mdio_write(sc, 0x06, 0xac26);
        re_mdio_write(sc, 0x06, 0x1ae0);
        re_mdio_write(sc, 0x06, 0x8b81);
        re_mdio_write(sc, 0x06, 0xac21);
        re_mdio_write(sc, 0x06, 0x14e0);
        re_mdio_write(sc, 0x06, 0x8b85);
        re_mdio_write(sc, 0x06, 0xac20);
        re_mdio_write(sc, 0x06, 0x0ee0);
        re_mdio_write(sc, 0x06, 0x8b85);
        re_mdio_write(sc, 0x06, 0xac23);
        re_mdio_write(sc, 0x06, 0x08e0);
        re_mdio_write(sc, 0x06, 0x8b87);
        re_mdio_write(sc, 0x06, 0xac24);
        re_mdio_write(sc, 0x06, 0x02ae);
        re_mdio_write(sc, 0x06, 0x3802);
        re_mdio_write(sc, 0x06, 0x1b02);
        re_mdio_write(sc, 0x06, 0xeee4);
        re_mdio_write(sc, 0x06, 0x1c04);
        re_mdio_write(sc, 0x06, 0xeee4);
        re_mdio_write(sc, 0x06, 0x1d04);
        re_mdio_write(sc, 0x06, 0xe2e0);
        re_mdio_write(sc, 0x06, 0x7ce3);
        re_mdio_write(sc, 0x06, 0xe07d);
        re_mdio_write(sc, 0x06, 0xe0e0);
        re_mdio_write(sc, 0x06, 0x38e1);
        re_mdio_write(sc, 0x06, 0xe039);
        re_mdio_write(sc, 0x06, 0xad2e);
        re_mdio_write(sc, 0x06, 0x1bad);
        re_mdio_write(sc, 0x06, 0x390d);
        re_mdio_write(sc, 0x06, 0xd101);
        re_mdio_write(sc, 0x06, 0xbf22);
        re_mdio_write(sc, 0x06, 0xe802);
        re_mdio_write(sc, 0x06, 0x3a21);
        re_mdio_write(sc, 0x06, 0x0222);
        re_mdio_write(sc, 0x06, 0x10ae);
        re_mdio_write(sc, 0x06, 0x0bac);
        re_mdio_write(sc, 0x06, 0x3802);
        re_mdio_write(sc, 0x06, 0xae06);
        re_mdio_write(sc, 0x06, 0x0222);
        re_mdio_write(sc, 0x06, 0x4d02);
        re_mdio_write(sc, 0x06, 0x2292);
        re_mdio_write(sc, 0x06, 0x021b);
        re_mdio_write(sc, 0x06, 0x13fd);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xf8e0);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0xad20);
        re_mdio_write(sc, 0x06, 0x1af6);
        re_mdio_write(sc, 0x06, 0x20e4);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0x022b);
        re_mdio_write(sc, 0x06, 0x1e02);
        re_mdio_write(sc, 0x06, 0x82ae);
        re_mdio_write(sc, 0x06, 0x0203);
        re_mdio_write(sc, 0x06, 0xc002);
        re_mdio_write(sc, 0x06, 0x827d);
        re_mdio_write(sc, 0x06, 0x022e);
        re_mdio_write(sc, 0x06, 0x6f02);
        re_mdio_write(sc, 0x06, 0x047b);
        re_mdio_write(sc, 0x06, 0x022f);
        re_mdio_write(sc, 0x06, 0x9ae0);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0xad21);
        re_mdio_write(sc, 0x06, 0x0bf6);
        re_mdio_write(sc, 0x06, 0x21e4);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0x0281);
        re_mdio_write(sc, 0x06, 0x9002);
        re_mdio_write(sc, 0x06, 0x1cd9);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x8ead);
        re_mdio_write(sc, 0x06, 0x2208);
        re_mdio_write(sc, 0x06, 0xf622);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x8e02);
        re_mdio_write(sc, 0x06, 0x35f4);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x8ead);
        re_mdio_write(sc, 0x06, 0x2308);
        re_mdio_write(sc, 0x06, 0xf623);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x8e02);
        re_mdio_write(sc, 0x06, 0x31e8);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x8ead);
        re_mdio_write(sc, 0x06, 0x2405);
        re_mdio_write(sc, 0x06, 0xf624);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x8ee0);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0xad25);
        re_mdio_write(sc, 0x06, 0x05f6);
        re_mdio_write(sc, 0x06, 0x25e4);
        re_mdio_write(sc, 0x06, 0x8b8e);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x8ead);
        re_mdio_write(sc, 0x06, 0x2608);
        re_mdio_write(sc, 0x06, 0xf626);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x8e02);
        re_mdio_write(sc, 0x06, 0x2d8a);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x8ead);
        re_mdio_write(sc, 0x06, 0x2705);
        re_mdio_write(sc, 0x06, 0xf627);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x8e02);
        re_mdio_write(sc, 0x06, 0x0386);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xf8fa);
        re_mdio_write(sc, 0x06, 0xef69);
        re_mdio_write(sc, 0x06, 0xe0e0);
        re_mdio_write(sc, 0x06, 0x00e1);
        re_mdio_write(sc, 0x06, 0xe001);
        re_mdio_write(sc, 0x06, 0xad27);
        re_mdio_write(sc, 0x06, 0x32e0);
        re_mdio_write(sc, 0x06, 0x8b40);
        re_mdio_write(sc, 0x06, 0xf720);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x40bf);
        re_mdio_write(sc, 0x06, 0x32c1);
        re_mdio_write(sc, 0x06, 0x0239);
        re_mdio_write(sc, 0x06, 0xf4ad);
        re_mdio_write(sc, 0x06, 0x2821);
        re_mdio_write(sc, 0x06, 0xe0e0);
        re_mdio_write(sc, 0x06, 0x20e1);
        re_mdio_write(sc, 0x06, 0xe021);
        re_mdio_write(sc, 0x06, 0xad20);
        re_mdio_write(sc, 0x06, 0x18e0);
        re_mdio_write(sc, 0x06, 0x8b40);
        re_mdio_write(sc, 0x06, 0xf620);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x40ee);
        re_mdio_write(sc, 0x06, 0x8b3b);
        re_mdio_write(sc, 0x06, 0xffe0);
        re_mdio_write(sc, 0x06, 0x8a8a);
        re_mdio_write(sc, 0x06, 0xe18a);
        re_mdio_write(sc, 0x06, 0x8be4);
        re_mdio_write(sc, 0x06, 0xe000);
        re_mdio_write(sc, 0x06, 0xe5e0);
        re_mdio_write(sc, 0x06, 0x01ef);
        re_mdio_write(sc, 0x06, 0x96fe);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xf8f9);
        re_mdio_write(sc, 0x06, 0xface);
        re_mdio_write(sc, 0x06, 0xfaef);
        re_mdio_write(sc, 0x06, 0x69fa);
        re_mdio_write(sc, 0x06, 0xd401);
        re_mdio_write(sc, 0x06, 0x55b4);
        re_mdio_write(sc, 0x06, 0xfebf);
        re_mdio_write(sc, 0x06, 0x1c5e);
        re_mdio_write(sc, 0x06, 0x0239);
        re_mdio_write(sc, 0x06, 0xf4ac);
        re_mdio_write(sc, 0x06, 0x280b);
        re_mdio_write(sc, 0x06, 0xbf1c);
        re_mdio_write(sc, 0x06, 0x5b02);
        re_mdio_write(sc, 0x06, 0x39f4);
        re_mdio_write(sc, 0x06, 0xac28);
        re_mdio_write(sc, 0x06, 0x49ae);
        re_mdio_write(sc, 0x06, 0x64bf);
        re_mdio_write(sc, 0x06, 0x1c5b);
        re_mdio_write(sc, 0x06, 0x0239);
        re_mdio_write(sc, 0x06, 0xf4ac);
        re_mdio_write(sc, 0x06, 0x285b);
        re_mdio_write(sc, 0x06, 0xd000);
        re_mdio_write(sc, 0x06, 0x0282);
        re_mdio_write(sc, 0x06, 0x62ac);
        re_mdio_write(sc, 0x06, 0x2105);
        re_mdio_write(sc, 0x06, 0xac22);
        re_mdio_write(sc, 0x06, 0x02ae);
        re_mdio_write(sc, 0x06, 0x4ebf);
        re_mdio_write(sc, 0x06, 0xe0c4);
        re_mdio_write(sc, 0x06, 0xbe85);
        re_mdio_write(sc, 0x06, 0xecd2);
        re_mdio_write(sc, 0x06, 0x04d8);
        re_mdio_write(sc, 0x06, 0x19d9);
        re_mdio_write(sc, 0x06, 0x1907);
        re_mdio_write(sc, 0x06, 0xdc19);
        re_mdio_write(sc, 0x06, 0xdd19);
        re_mdio_write(sc, 0x06, 0x0789);
        re_mdio_write(sc, 0x06, 0x89ef);
        re_mdio_write(sc, 0x06, 0x645e);
        re_mdio_write(sc, 0x06, 0x07ff);
        re_mdio_write(sc, 0x06, 0x0d65);
        re_mdio_write(sc, 0x06, 0x5cf8);
        re_mdio_write(sc, 0x06, 0x001e);
        re_mdio_write(sc, 0x06, 0x46dc);
        re_mdio_write(sc, 0x06, 0x19dd);
        re_mdio_write(sc, 0x06, 0x19b2);
        re_mdio_write(sc, 0x06, 0xe2d4);
        re_mdio_write(sc, 0x06, 0x0001);
        re_mdio_write(sc, 0x06, 0xbf1c);
        re_mdio_write(sc, 0x06, 0x5b02);
        re_mdio_write(sc, 0x06, 0x3a21);
        re_mdio_write(sc, 0x06, 0xae1d);
        re_mdio_write(sc, 0x06, 0xbee0);
        re_mdio_write(sc, 0x06, 0xc4bf);
        re_mdio_write(sc, 0x06, 0x85ec);
        re_mdio_write(sc, 0x06, 0xd204);
        re_mdio_write(sc, 0x06, 0xd819);
        re_mdio_write(sc, 0x06, 0xd919);
        re_mdio_write(sc, 0x06, 0x07dc);
        re_mdio_write(sc, 0x06, 0x19dd);
        re_mdio_write(sc, 0x06, 0x1907);
        re_mdio_write(sc, 0x06, 0xb2f4);
        re_mdio_write(sc, 0x06, 0xd400);
        re_mdio_write(sc, 0x06, 0x00bf);
        re_mdio_write(sc, 0x06, 0x1c5b);
        re_mdio_write(sc, 0x06, 0x023a);
        re_mdio_write(sc, 0x06, 0x21fe);
        re_mdio_write(sc, 0x06, 0xef96);
        re_mdio_write(sc, 0x06, 0xfec6);
        re_mdio_write(sc, 0x06, 0xfefd);
        re_mdio_write(sc, 0x06, 0xfc05);
        re_mdio_write(sc, 0x06, 0xf9e2);
        re_mdio_write(sc, 0x06, 0xe0ea);
        re_mdio_write(sc, 0x06, 0xe3e0);
        re_mdio_write(sc, 0x06, 0xeb5a);
        re_mdio_write(sc, 0x06, 0x070c);
        re_mdio_write(sc, 0x06, 0x031e);
        re_mdio_write(sc, 0x06, 0x20e6);
        re_mdio_write(sc, 0x06, 0xe0ea);
        re_mdio_write(sc, 0x06, 0xe7e0);
        re_mdio_write(sc, 0x06, 0xebe0);
        re_mdio_write(sc, 0x06, 0xe0fc);
        re_mdio_write(sc, 0x06, 0xe1e0);
        re_mdio_write(sc, 0x06, 0xfdfd);
        re_mdio_write(sc, 0x06, 0x04f8);
        re_mdio_write(sc, 0x06, 0xfaef);
        re_mdio_write(sc, 0x06, 0x69e0);
        re_mdio_write(sc, 0x06, 0x8b80);
        re_mdio_write(sc, 0x06, 0xad27);
        re_mdio_write(sc, 0x06, 0x22bf);
        re_mdio_write(sc, 0x06, 0x47ba);
        re_mdio_write(sc, 0x06, 0x0239);
        re_mdio_write(sc, 0x06, 0xf4e0);
        re_mdio_write(sc, 0x06, 0x8b44);
        re_mdio_write(sc, 0x06, 0x1f01);
        re_mdio_write(sc, 0x06, 0x9e15);
        re_mdio_write(sc, 0x06, 0xe58b);
        re_mdio_write(sc, 0x06, 0x44ad);
        re_mdio_write(sc, 0x06, 0x2907);
        re_mdio_write(sc, 0x06, 0xac28);
        re_mdio_write(sc, 0x06, 0x04d1);
        re_mdio_write(sc, 0x06, 0x01ae);
        re_mdio_write(sc, 0x06, 0x02d1);
        re_mdio_write(sc, 0x06, 0x00bf);
        re_mdio_write(sc, 0x06, 0x8342);
        re_mdio_write(sc, 0x06, 0x023a);
        re_mdio_write(sc, 0x06, 0x21ef);
        re_mdio_write(sc, 0x06, 0x96fe);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xf8e0);
        re_mdio_write(sc, 0x06, 0x8b85);
        re_mdio_write(sc, 0x06, 0xad26);
        re_mdio_write(sc, 0x06, 0x30e0);
        re_mdio_write(sc, 0x06, 0xe036);
        re_mdio_write(sc, 0x06, 0xe1e0);
        re_mdio_write(sc, 0x06, 0x37e1);
        re_mdio_write(sc, 0x06, 0x8b3f);
        re_mdio_write(sc, 0x06, 0x1f10);
        re_mdio_write(sc, 0x06, 0x9e23);
        re_mdio_write(sc, 0x06, 0xe48b);
        re_mdio_write(sc, 0x06, 0x3fac);
        re_mdio_write(sc, 0x06, 0x200b);
        re_mdio_write(sc, 0x06, 0xac21);
        re_mdio_write(sc, 0x06, 0x0dac);
        re_mdio_write(sc, 0x06, 0x250f);
        re_mdio_write(sc, 0x06, 0xac27);
        re_mdio_write(sc, 0x06, 0x11ae);
        re_mdio_write(sc, 0x06, 0x1202);
        re_mdio_write(sc, 0x06, 0x2cb5);
        re_mdio_write(sc, 0x06, 0xae0d);
        re_mdio_write(sc, 0x06, 0x0282);
        re_mdio_write(sc, 0x06, 0xe7ae);
        re_mdio_write(sc, 0x06, 0x0802);
        re_mdio_write(sc, 0x06, 0x2cd7);
        re_mdio_write(sc, 0x06, 0xae03);
        re_mdio_write(sc, 0x06, 0x022c);
        re_mdio_write(sc, 0x06, 0xeafc);
        re_mdio_write(sc, 0x06, 0x04f8);
        re_mdio_write(sc, 0x06, 0xfaef);
        re_mdio_write(sc, 0x06, 0x6902);
        re_mdio_write(sc, 0x06, 0x8304);
        re_mdio_write(sc, 0x06, 0xe0e0);
        re_mdio_write(sc, 0x06, 0x14e1);
        re_mdio_write(sc, 0x06, 0xe015);
        re_mdio_write(sc, 0x06, 0xad26);
        re_mdio_write(sc, 0x06, 0x08d1);
        re_mdio_write(sc, 0x06, 0x1ebf);
        re_mdio_write(sc, 0x06, 0x2d47);
        re_mdio_write(sc, 0x06, 0x023a);
        re_mdio_write(sc, 0x06, 0x21ef);
        re_mdio_write(sc, 0x06, 0x96fe);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0xf8e0);
        re_mdio_write(sc, 0x06, 0x8b85);
        re_mdio_write(sc, 0x06, 0xad27);
        re_mdio_write(sc, 0x06, 0x2fd0);
        re_mdio_write(sc, 0x06, 0x0b02);
        re_mdio_write(sc, 0x06, 0x3826);
        re_mdio_write(sc, 0x06, 0x5882);
        re_mdio_write(sc, 0x06, 0x7882);
        re_mdio_write(sc, 0x06, 0x9f24);
        re_mdio_write(sc, 0x06, 0xe08b);
        re_mdio_write(sc, 0x06, 0x32e1);
        re_mdio_write(sc, 0x06, 0x8b33);
        re_mdio_write(sc, 0x06, 0x1f10);
        re_mdio_write(sc, 0x06, 0x9e1a);
        re_mdio_write(sc, 0x06, 0x10e4);
        re_mdio_write(sc, 0x06, 0x8b32);
        re_mdio_write(sc, 0x06, 0xe0e0);
        re_mdio_write(sc, 0x06, 0x28e1);
        re_mdio_write(sc, 0x06, 0xe029);
        re_mdio_write(sc, 0x06, 0xf72c);
        re_mdio_write(sc, 0x06, 0xe4e0);
        re_mdio_write(sc, 0x06, 0x28e5);
        re_mdio_write(sc, 0x06, 0xe029);
        re_mdio_write(sc, 0x06, 0xf62c);
        re_mdio_write(sc, 0x06, 0xe4e0);
        re_mdio_write(sc, 0x06, 0x28e5);
        re_mdio_write(sc, 0x06, 0xe029);
        re_mdio_write(sc, 0x06, 0xfc04);
        re_mdio_write(sc, 0x06, 0x00e1);
        re_mdio_write(sc, 0x06, 0x4077);
        re_mdio_write(sc, 0x06, 0xe140);
        re_mdio_write(sc, 0x06, 0xbbe0);
        re_mdio_write(sc, 0x06, 0x2a00);
        re_mdio_write(sc, 0x05, 0xe142);
        PhyRegValue = re_mdio_read(sc, 0x06);
        PhyRegValue |= BIT_0;
        re_mdio_write(sc, 0x06,PhyRegValue);
        re_mdio_write(sc, 0x05, 0xe140);
        PhyRegValue = re_mdio_read(sc, 0x06);
        PhyRegValue |= BIT_0;
        re_mdio_write(sc, 0x06, PhyRegValue);
        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x1f, 0x0005);
        for (i = 0; i < 200; i++) {
                DELAY(100);
                PhyRegValue = re_mdio_read(sc, 0x00);
                if (PhyRegValue & BIT_7)
                        break;
        }
        re_mdio_write(sc, 0x1f, 0x0007);
        re_mdio_write(sc, 0x1e, 0x0023);
        PhyRegValue = re_mdio_read(sc, 0x17);
        PhyRegValue |= BIT_1;
        if (sc->RequiredSecLanDonglePatch)
                PhyRegValue &= ~(BIT_2);
        re_mdio_write(sc, 0x17, PhyRegValue);
        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x1f, 0x0003);
        re_mdio_write(sc, 0x09, 0xA20F);
        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x1f, 0x0003);
        re_mdio_write(sc, 0x01, 0x328A);
        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x1f, 0x0003);
        PhyRegValue = re_mdio_read(sc, 0x19);
        PhyRegValue &= ~BIT_0;
        re_mdio_write(sc, 0x19, PhyRegValue);
        PhyRegValue = re_mdio_read(sc, 0x10);
        PhyRegValue &= ~BIT_10;
        re_mdio_write(sc, 0x10, PhyRegValue);
        re_mdio_write(sc, 0x1f, 0x0000);
        re_mdio_write(sc, 0x00, 0x9200);
}

static void re_set_phy_mcu_8168g_1(struct re_softc *sc)
{
        u_int16_t PhyRegValue;

        re_set_phy_mcu_patch_request(sc);

        re_mdio_write(sc, 0x1f, 0x0A43);
        re_mdio_write(sc, 0x13, 0x8146);
        re_mdio_write(sc, 0x14, 0x2300);
        re_mdio_write(sc, 0x13, 0xB820);
        re_mdio_write(sc, 0x14, 0x0210);

        re_mdio_write(sc, 0x1F, 0x0A43);
        re_mdio_write(sc, 0x13, 0xB820);
        re_mdio_write(sc, 0x14, 0x0290);
        re_mdio_write(sc, 0x13, 0xA012);
        re_mdio_write(sc, 0x14, 0x0000);
        re_mdio_write(sc, 0x13, 0xA014);
        re_mdio_write(sc, 0x14, 0x2c04);
        re_mdio_write(sc, 0x14, 0x2c0c);
        re_mdio_write(sc, 0x14, 0x2c6c);
        re_mdio_write(sc, 0x14, 0x2d0d);
        re_mdio_write(sc, 0x14, 0x31ce);
        re_mdio_write(sc, 0x14, 0x506d);
        re_mdio_write(sc, 0x14, 0xd708);
        re_mdio_write(sc, 0x14, 0x3108);
        re_mdio_write(sc, 0x14, 0x106d);
        re_mdio_write(sc, 0x14, 0x1560);
        re_mdio_write(sc, 0x14, 0x15a9);
        re_mdio_write(sc, 0x14, 0x206e);
        re_mdio_write(sc, 0x14, 0x175b);
        re_mdio_write(sc, 0x14, 0x6062);
        re_mdio_write(sc, 0x14, 0xd700);
        re_mdio_write(sc, 0x14, 0x5fae);
        re_mdio_write(sc, 0x14, 0xd708);
        re_mdio_write(sc, 0x14, 0x3107);
        re_mdio_write(sc, 0x14, 0x4c1e);
        re_mdio_write(sc, 0x14, 0x4169);
        re_mdio_write(sc, 0x14, 0x316a);
        re_mdio_write(sc, 0x14, 0x0c19);
        re_mdio_write(sc, 0x14, 0x31aa);
        re_mdio_write(sc, 0x14, 0x0c19);
        re_mdio_write(sc, 0x14, 0x2c1b);
        re_mdio_write(sc, 0x14, 0x5e62);
        re_mdio_write(sc, 0x14, 0x26b5);
        re_mdio_write(sc, 0x14, 0x31ab);
        re_mdio_write(sc, 0x14, 0x5c1e);
        re_mdio_write(sc, 0x14, 0x2c0c);
        re_mdio_write(sc, 0x14, 0xc040);
        re_mdio_write(sc, 0x14, 0x8808);
        re_mdio_write(sc, 0x14, 0xc520);
        re_mdio_write(sc, 0x14, 0xc421);
        re_mdio_write(sc, 0x14, 0xd05a);
        re_mdio_write(sc, 0x14, 0xd19a);
        re_mdio_write(sc, 0x14, 0xd709);
        re_mdio_write(sc, 0x14, 0x608f);
        re_mdio_write(sc, 0x14, 0xd06b);
        re_mdio_write(sc, 0x14, 0xd18a);
        re_mdio_write(sc, 0x14, 0x2c2c);
        re_mdio_write(sc, 0x14, 0xd0be);
        re_mdio_write(sc, 0x14, 0xd188);
        re_mdio_write(sc, 0x14, 0x2c2c);
        re_mdio_write(sc, 0x14, 0xd708);
        re_mdio_write(sc, 0x14, 0x4072);
        re_mdio_write(sc, 0x14, 0xc104);
        re_mdio_write(sc, 0x14, 0x2c3e);
        re_mdio_write(sc, 0x14, 0x4076);
        re_mdio_write(sc, 0x14, 0xc110);
        re_mdio_write(sc, 0x14, 0x2c3e);
        re_mdio_write(sc, 0x14, 0x4071);
        re_mdio_write(sc, 0x14, 0xc102);
        re_mdio_write(sc, 0x14, 0x2c3e);
        re_mdio_write(sc, 0x14, 0x4070);
        re_mdio_write(sc, 0x14, 0xc101);
        re_mdio_write(sc, 0x14, 0x2c3e);
        re_mdio_write(sc, 0x14, 0x175b);
        re_mdio_write(sc, 0x14, 0xd709);
        re_mdio_write(sc, 0x14, 0x3390);
        re_mdio_write(sc, 0x14, 0x5c39);
        re_mdio_write(sc, 0x14, 0x2c4e);
        re_mdio_write(sc, 0x14, 0x175b);
        re_mdio_write(sc, 0x14, 0xd708);
        re_mdio_write(sc, 0x14, 0x6193);
        re_mdio_write(sc, 0x14, 0xd709);
        re_mdio_write(sc, 0x14, 0x5f9d);
        re_mdio_write(sc, 0x14, 0x408b);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x6042);
        re_mdio_write(sc, 0x14, 0xb401);
        re_mdio_write(sc, 0x14, 0x175b);
        re_mdio_write(sc, 0x14, 0xd708);
        re_mdio_write(sc, 0x14, 0x6073);
        re_mdio_write(sc, 0x14, 0x5fbc);
        re_mdio_write(sc, 0x14, 0x2c4d);
        re_mdio_write(sc, 0x14, 0x26ed);
        re_mdio_write(sc, 0x14, 0xb280);
        re_mdio_write(sc, 0x14, 0xa841);
        re_mdio_write(sc, 0x14, 0x9420);
        re_mdio_write(sc, 0x14, 0x8710);
        re_mdio_write(sc, 0x14, 0xd709);
        re_mdio_write(sc, 0x14, 0x42ec);
        re_mdio_write(sc, 0x14, 0x606d);
        re_mdio_write(sc, 0x14, 0xd207);
        re_mdio_write(sc, 0x14, 0x2c57);
        re_mdio_write(sc, 0x14, 0xd203);
        re_mdio_write(sc, 0x14, 0x33ff);
        re_mdio_write(sc, 0x14, 0x563b);
        re_mdio_write(sc, 0x14, 0x3275);
        re_mdio_write(sc, 0x14, 0x7c5e);
        re_mdio_write(sc, 0x14, 0xb240);
        re_mdio_write(sc, 0x14, 0xb402);
        re_mdio_write(sc, 0x14, 0x263b);
        re_mdio_write(sc, 0x14, 0x6096);
        re_mdio_write(sc, 0x14, 0xb240);
        re_mdio_write(sc, 0x14, 0xb406);
        re_mdio_write(sc, 0x14, 0x263b);
        re_mdio_write(sc, 0x14, 0x31d7);
        re_mdio_write(sc, 0x14, 0x7c67);
        re_mdio_write(sc, 0x14, 0xb240);
        re_mdio_write(sc, 0x14, 0xb40e);
        re_mdio_write(sc, 0x14, 0x263b);
        re_mdio_write(sc, 0x14, 0xb410);
        re_mdio_write(sc, 0x14, 0x8802);
        re_mdio_write(sc, 0x14, 0xb240);
        re_mdio_write(sc, 0x14, 0x940e);
        re_mdio_write(sc, 0x14, 0x263b);
        re_mdio_write(sc, 0x14, 0xba04);
        re_mdio_write(sc, 0x14, 0x1cd6);
        re_mdio_write(sc, 0x14, 0xa902);
        re_mdio_write(sc, 0x14, 0xd711);
        re_mdio_write(sc, 0x14, 0x4045);
        re_mdio_write(sc, 0x14, 0xa980);
        re_mdio_write(sc, 0x14, 0x3003);
        re_mdio_write(sc, 0x14, 0x59b1);
        re_mdio_write(sc, 0x14, 0xa540);
        re_mdio_write(sc, 0x14, 0xa601);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x4043);
        re_mdio_write(sc, 0x14, 0xa910);
        re_mdio_write(sc, 0x14, 0xd711);
        re_mdio_write(sc, 0x14, 0x60a0);
        re_mdio_write(sc, 0x14, 0xca33);
        re_mdio_write(sc, 0x14, 0xcb33);
        re_mdio_write(sc, 0x14, 0xa941);
        re_mdio_write(sc, 0x14, 0x2c82);
        re_mdio_write(sc, 0x14, 0xcaff);
        re_mdio_write(sc, 0x14, 0xcbff);
        re_mdio_write(sc, 0x14, 0xa921);
        re_mdio_write(sc, 0x14, 0xce02);
        re_mdio_write(sc, 0x14, 0xe070);
        re_mdio_write(sc, 0x14, 0x0f10);
        re_mdio_write(sc, 0x14, 0xaf01);
        re_mdio_write(sc, 0x14, 0x8f01);
        re_mdio_write(sc, 0x14, 0x1766);
        re_mdio_write(sc, 0x14, 0x8e02);
        re_mdio_write(sc, 0x14, 0x1787);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x609c);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x7fa4);
        re_mdio_write(sc, 0x14, 0x2cd4);
        re_mdio_write(sc, 0x14, 0x1ce9);
        re_mdio_write(sc, 0x14, 0xce04);
        re_mdio_write(sc, 0x14, 0xe070);
        re_mdio_write(sc, 0x14, 0x0f20);
        re_mdio_write(sc, 0x14, 0xaf01);
        re_mdio_write(sc, 0x14, 0x8f01);
        re_mdio_write(sc, 0x14, 0x1766);
        re_mdio_write(sc, 0x14, 0x8e04);
        re_mdio_write(sc, 0x14, 0x6044);
        re_mdio_write(sc, 0x14, 0x2cd4);
        re_mdio_write(sc, 0x14, 0xa520);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x4043);
        re_mdio_write(sc, 0x14, 0x2cc1);
        re_mdio_write(sc, 0x14, 0xe00f);
        re_mdio_write(sc, 0x14, 0x0501);
        re_mdio_write(sc, 0x14, 0x1cef);
        re_mdio_write(sc, 0x14, 0xb801);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x4060);
        re_mdio_write(sc, 0x14, 0x7fc4);
        re_mdio_write(sc, 0x14, 0x2cd4);
        re_mdio_write(sc, 0x14, 0x1cf5);
        re_mdio_write(sc, 0x14, 0xe00f);
        re_mdio_write(sc, 0x14, 0x0502);
        re_mdio_write(sc, 0x14, 0x1cef);
        re_mdio_write(sc, 0x14, 0xb802);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x4061);
        re_mdio_write(sc, 0x14, 0x7fc4);
        re_mdio_write(sc, 0x14, 0x2cd4);
        re_mdio_write(sc, 0x14, 0x1cf5);
        re_mdio_write(sc, 0x14, 0xe00f);
        re_mdio_write(sc, 0x14, 0x0504);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x6099);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x7fa4);
        re_mdio_write(sc, 0x14, 0x2cd4);
        re_mdio_write(sc, 0x14, 0xc17f);
        re_mdio_write(sc, 0x14, 0xc200);
        re_mdio_write(sc, 0x14, 0xc43f);
        re_mdio_write(sc, 0x14, 0xcc03);
        re_mdio_write(sc, 0x14, 0xa701);
        re_mdio_write(sc, 0x14, 0xa510);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x4018);
        re_mdio_write(sc, 0x14, 0x9910);
        re_mdio_write(sc, 0x14, 0x8510);
        re_mdio_write(sc, 0x14, 0x2860);
        re_mdio_write(sc, 0x14, 0xe00f);
        re_mdio_write(sc, 0x14, 0x0504);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x6099);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x7fa4);
        re_mdio_write(sc, 0x14, 0x2cd4);
        re_mdio_write(sc, 0x14, 0xa608);
        re_mdio_write(sc, 0x14, 0xc17d);
        re_mdio_write(sc, 0x14, 0xc200);
        re_mdio_write(sc, 0x14, 0xc43f);
        re_mdio_write(sc, 0x14, 0xcc03);
        re_mdio_write(sc, 0x14, 0xa701);
        re_mdio_write(sc, 0x14, 0xa510);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x4018);
        re_mdio_write(sc, 0x14, 0x9910);
        re_mdio_write(sc, 0x14, 0x8510);
        re_mdio_write(sc, 0x14, 0x2926);
        re_mdio_write(sc, 0x14, 0x1792);
        re_mdio_write(sc, 0x14, 0x27db);
        re_mdio_write(sc, 0x14, 0xc000);
        re_mdio_write(sc, 0x14, 0xc100);
        re_mdio_write(sc, 0x14, 0xc200);
        re_mdio_write(sc, 0x14, 0xc300);
        re_mdio_write(sc, 0x14, 0xc400);
        re_mdio_write(sc, 0x14, 0xc500);
        re_mdio_write(sc, 0x14, 0xc600);
        re_mdio_write(sc, 0x14, 0xc7c1);
        re_mdio_write(sc, 0x14, 0xc800);
        re_mdio_write(sc, 0x14, 0xcc00);
        re_mdio_write(sc, 0x14, 0x0800);
        re_mdio_write(sc, 0x14, 0xca0f);
        re_mdio_write(sc, 0x14, 0xcbff);
        re_mdio_write(sc, 0x14, 0xa901);
        re_mdio_write(sc, 0x14, 0x8902);
        re_mdio_write(sc, 0x14, 0xc900);
        re_mdio_write(sc, 0x14, 0xca00);
        re_mdio_write(sc, 0x14, 0xcb00);
        re_mdio_write(sc, 0x14, 0x0800);
        re_mdio_write(sc, 0x14, 0xb804);
        re_mdio_write(sc, 0x14, 0x0800);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x6044);
        re_mdio_write(sc, 0x14, 0x9804);
        re_mdio_write(sc, 0x14, 0x0800);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x6099);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x7fa4);
        re_mdio_write(sc, 0x14, 0x2cd4);
        re_mdio_write(sc, 0x14, 0x0800);
        re_mdio_write(sc, 0x14, 0xa510);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x6098);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x7fa4);
        re_mdio_write(sc, 0x14, 0x2cd4);
        re_mdio_write(sc, 0x14, 0x8510);
        re_mdio_write(sc, 0x14, 0x0800);
        re_mdio_write(sc, 0x14, 0xd711);
        re_mdio_write(sc, 0x14, 0x3003);
        re_mdio_write(sc, 0x14, 0x1d01);
        re_mdio_write(sc, 0x14, 0x2d0b);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x60be);
        re_mdio_write(sc, 0x14, 0xe060);
        re_mdio_write(sc, 0x14, 0x0920);
        re_mdio_write(sc, 0x14, 0x1cd6);
        re_mdio_write(sc, 0x14, 0x2c89);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x3063);
        re_mdio_write(sc, 0x14, 0x1948);
        re_mdio_write(sc, 0x14, 0x288a);
        re_mdio_write(sc, 0x14, 0x1cd6);
        re_mdio_write(sc, 0x14, 0x29bd);
        re_mdio_write(sc, 0x14, 0xa802);
        re_mdio_write(sc, 0x14, 0xa303);
        re_mdio_write(sc, 0x14, 0x843f);
        re_mdio_write(sc, 0x14, 0x81ff);
        re_mdio_write(sc, 0x14, 0x8208);
        re_mdio_write(sc, 0x14, 0xa201);
        re_mdio_write(sc, 0x14, 0xc001);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x30a0);
        re_mdio_write(sc, 0x14, 0x0d1c);
        re_mdio_write(sc, 0x14, 0x30a0);
        re_mdio_write(sc, 0x14, 0x3d13);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x7f4c);
        re_mdio_write(sc, 0x14, 0x2ab6);
        re_mdio_write(sc, 0x14, 0xe003);
        re_mdio_write(sc, 0x14, 0x0202);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x6090);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x7fac);
        re_mdio_write(sc, 0x14, 0x2ab6);
        re_mdio_write(sc, 0x14, 0xa20c);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x6091);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x7fac);
        re_mdio_write(sc, 0x14, 0x2ab6);
        re_mdio_write(sc, 0x14, 0x820e);
        re_mdio_write(sc, 0x14, 0xa3e0);
        re_mdio_write(sc, 0x14, 0xa520);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x609d);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x7fac);
        re_mdio_write(sc, 0x14, 0x2ab6);
        re_mdio_write(sc, 0x14, 0x8520);
        re_mdio_write(sc, 0x14, 0x6703);
        re_mdio_write(sc, 0x14, 0x2d34);
        re_mdio_write(sc, 0x14, 0xa13e);
        re_mdio_write(sc, 0x14, 0xc001);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x4000);
        re_mdio_write(sc, 0x14, 0x6046);
        re_mdio_write(sc, 0x14, 0x2d0d);
        re_mdio_write(sc, 0x14, 0xa43f);
        re_mdio_write(sc, 0x14, 0xa101);
        re_mdio_write(sc, 0x14, 0xc020);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x3121);
        re_mdio_write(sc, 0x14, 0x0d45);
        re_mdio_write(sc, 0x14, 0x30c0);
        re_mdio_write(sc, 0x14, 0x3d0d);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x7f4c);
        re_mdio_write(sc, 0x14, 0x2ab6);
        re_mdio_write(sc, 0x14, 0xa540);
        re_mdio_write(sc, 0x14, 0xc001);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x4001);
        re_mdio_write(sc, 0x14, 0xe00f);
        re_mdio_write(sc, 0x14, 0x0501);
        re_mdio_write(sc, 0x14, 0x1dac);
        re_mdio_write(sc, 0x14, 0xc1c4);
        re_mdio_write(sc, 0x14, 0xa268);
        re_mdio_write(sc, 0x14, 0xa303);
        re_mdio_write(sc, 0x14, 0x8420);
        re_mdio_write(sc, 0x14, 0xe00f);
        re_mdio_write(sc, 0x14, 0x0502);
        re_mdio_write(sc, 0x14, 0x1dac);
        re_mdio_write(sc, 0x14, 0xc002);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x4000);
        re_mdio_write(sc, 0x14, 0x8208);
        re_mdio_write(sc, 0x14, 0x8410);
        re_mdio_write(sc, 0x14, 0xa121);
        re_mdio_write(sc, 0x14, 0xc002);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x4000);
        re_mdio_write(sc, 0x14, 0x8120);
        re_mdio_write(sc, 0x14, 0x8180);
        re_mdio_write(sc, 0x14, 0x1d97);
        re_mdio_write(sc, 0x14, 0xa180);
        re_mdio_write(sc, 0x14, 0xa13a);
        re_mdio_write(sc, 0x14, 0x8240);
        re_mdio_write(sc, 0x14, 0xa430);
        re_mdio_write(sc, 0x14, 0xc010);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x30e1);
        re_mdio_write(sc, 0x14, 0x0abc);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x7f8c);
        re_mdio_write(sc, 0x14, 0x2ab6);
        re_mdio_write(sc, 0x14, 0xa480);
        re_mdio_write(sc, 0x14, 0xa230);
        re_mdio_write(sc, 0x14, 0xa303);
        re_mdio_write(sc, 0x14, 0xc001);
        re_mdio_write(sc, 0x14, 0xd70c);
        re_mdio_write(sc, 0x14, 0x4124);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x6120);
        re_mdio_write(sc, 0x14, 0xd711);
        re_mdio_write(sc, 0x14, 0x3128);
        re_mdio_write(sc, 0x14, 0x3d76);
        re_mdio_write(sc, 0x14, 0x2d70);
        re_mdio_write(sc, 0x14, 0xa801);
        re_mdio_write(sc, 0x14, 0x2d6c);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x4000);
        re_mdio_write(sc, 0x14, 0xe018);
        re_mdio_write(sc, 0x14, 0x0208);
        re_mdio_write(sc, 0x14, 0xa1f8);
        re_mdio_write(sc, 0x14, 0x8480);
        re_mdio_write(sc, 0x14, 0xc004);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x4000);
        re_mdio_write(sc, 0x14, 0x6046);
        re_mdio_write(sc, 0x14, 0x2d0d);
        re_mdio_write(sc, 0x14, 0xa43f);
        re_mdio_write(sc, 0x14, 0xa105);
        re_mdio_write(sc, 0x14, 0x8228);
        re_mdio_write(sc, 0x14, 0xc004);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x4000);
        re_mdio_write(sc, 0x14, 0x81bc);
        re_mdio_write(sc, 0x14, 0xa220);
        re_mdio_write(sc, 0x14, 0x1d97);
        re_mdio_write(sc, 0x14, 0x8220);
        re_mdio_write(sc, 0x14, 0xa1bc);
        re_mdio_write(sc, 0x14, 0xc040);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x30e1);
        re_mdio_write(sc, 0x14, 0x0abc);
        re_mdio_write(sc, 0x14, 0x30e1);
        re_mdio_write(sc, 0x14, 0x3d0d);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x7f4c);
        re_mdio_write(sc, 0x14, 0x2ab6);
        re_mdio_write(sc, 0x14, 0xa802);
        re_mdio_write(sc, 0x14, 0xd70c);
        re_mdio_write(sc, 0x14, 0x4244);
        re_mdio_write(sc, 0x14, 0xa301);
        re_mdio_write(sc, 0x14, 0xc004);
        re_mdio_write(sc, 0x14, 0xd711);
        re_mdio_write(sc, 0x14, 0x3128);
        re_mdio_write(sc, 0x14, 0x3da5);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x5f80);
        re_mdio_write(sc, 0x14, 0xd711);
        re_mdio_write(sc, 0x14, 0x3109);
        re_mdio_write(sc, 0x14, 0x3da7);
        re_mdio_write(sc, 0x14, 0x2dab);
        re_mdio_write(sc, 0x14, 0xa801);
        re_mdio_write(sc, 0x14, 0x2d9a);
        re_mdio_write(sc, 0x14, 0xa802);
        re_mdio_write(sc, 0x14, 0xc004);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x4000);
        re_mdio_write(sc, 0x14, 0x0800);
        re_mdio_write(sc, 0x14, 0xa510);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x609a);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x7fac);
        re_mdio_write(sc, 0x14, 0x2ab6);
        re_mdio_write(sc, 0x14, 0x8510);
        re_mdio_write(sc, 0x14, 0x0800);
        re_mdio_write(sc, 0x13, 0xA01A);
        re_mdio_write(sc, 0x14, 0x0000);
        re_mdio_write(sc, 0x13, 0xA006);
        re_mdio_write(sc, 0x14, 0x0ad6);
        re_mdio_write(sc, 0x13, 0xA004);
        re_mdio_write(sc, 0x14, 0x07f5);
        re_mdio_write(sc, 0x13, 0xA002);
        re_mdio_write(sc, 0x14, 0x06a9);
        re_mdio_write(sc, 0x13, 0xA000);
        re_mdio_write(sc, 0x14, 0xf069);
        re_mdio_write(sc, 0x13, 0xB820);
        re_mdio_write(sc, 0x14, 0x0210);

        re_mdio_write(sc, 0x1F, 0x0A43);
        re_mdio_write(sc, 0x13, 0x83a0);
        re_mdio_write(sc, 0x14, 0xaf83);
        re_mdio_write(sc, 0x14, 0xacaf);
        re_mdio_write(sc, 0x14, 0x83b8);
        re_mdio_write(sc, 0x14, 0xaf83);
        re_mdio_write(sc, 0x14, 0xcdaf);
        re_mdio_write(sc, 0x14, 0x83d3);
        re_mdio_write(sc, 0x14, 0x0204);
        re_mdio_write(sc, 0x14, 0x9a02);
        re_mdio_write(sc, 0x14, 0x09a9);
        re_mdio_write(sc, 0x14, 0x0284);
        re_mdio_write(sc, 0x14, 0x61af);
        re_mdio_write(sc, 0x14, 0x02fc);
        re_mdio_write(sc, 0x14, 0xad20);
        re_mdio_write(sc, 0x14, 0x0302);
        re_mdio_write(sc, 0x14, 0x867c);
        re_mdio_write(sc, 0x14, 0xad21);
        re_mdio_write(sc, 0x14, 0x0302);
        re_mdio_write(sc, 0x14, 0x85c9);
        re_mdio_write(sc, 0x14, 0xad22);
        re_mdio_write(sc, 0x14, 0x0302);
        re_mdio_write(sc, 0x14, 0x1bc0);
        re_mdio_write(sc, 0x14, 0xaf17);
        re_mdio_write(sc, 0x14, 0xe302);
        re_mdio_write(sc, 0x14, 0x8703);
        re_mdio_write(sc, 0x14, 0xaf18);
        re_mdio_write(sc, 0x14, 0x6201);
        re_mdio_write(sc, 0x14, 0x06e0);
        re_mdio_write(sc, 0x14, 0x8148);
        re_mdio_write(sc, 0x14, 0xaf3c);
        re_mdio_write(sc, 0x14, 0x69f8);
        re_mdio_write(sc, 0x14, 0xf9fa);
        re_mdio_write(sc, 0x14, 0xef69);
        re_mdio_write(sc, 0x14, 0xee80);
        re_mdio_write(sc, 0x14, 0x10f7);
        re_mdio_write(sc, 0x14, 0xee80);
        re_mdio_write(sc, 0x14, 0x131f);
        re_mdio_write(sc, 0x14, 0xd104);
        re_mdio_write(sc, 0x14, 0xbf87);
        re_mdio_write(sc, 0x14, 0xf302);
        re_mdio_write(sc, 0x14, 0x4259);
        re_mdio_write(sc, 0x14, 0x0287);
        re_mdio_write(sc, 0x14, 0x88bf);
        re_mdio_write(sc, 0x14, 0x87cf);
        re_mdio_write(sc, 0x14, 0xd7b8);
        re_mdio_write(sc, 0x14, 0x22d0);
        re_mdio_write(sc, 0x14, 0x0c02);
        re_mdio_write(sc, 0x14, 0x4252);
        re_mdio_write(sc, 0x14, 0xee80);
        re_mdio_write(sc, 0x14, 0xcda0);
        re_mdio_write(sc, 0x14, 0xee80);
        re_mdio_write(sc, 0x14, 0xce8b);
        re_mdio_write(sc, 0x14, 0xee80);
        re_mdio_write(sc, 0x14, 0xd1f5);
        re_mdio_write(sc, 0x14, 0xee80);
        re_mdio_write(sc, 0x14, 0xd2a9);
        re_mdio_write(sc, 0x14, 0xee80);
        re_mdio_write(sc, 0x14, 0xd30a);
        re_mdio_write(sc, 0x14, 0xee80);
        re_mdio_write(sc, 0x14, 0xf010);
        re_mdio_write(sc, 0x14, 0xee80);
        re_mdio_write(sc, 0x14, 0xf38f);
        re_mdio_write(sc, 0x14, 0xee81);
        re_mdio_write(sc, 0x14, 0x011e);
        re_mdio_write(sc, 0x14, 0xee81);
        re_mdio_write(sc, 0x14, 0x0b4a);
        re_mdio_write(sc, 0x14, 0xee81);
        re_mdio_write(sc, 0x14, 0x0c7c);
        re_mdio_write(sc, 0x14, 0xee81);
        re_mdio_write(sc, 0x14, 0x127f);
        re_mdio_write(sc, 0x14, 0xd100);
        re_mdio_write(sc, 0x14, 0x0210);
        re_mdio_write(sc, 0x14, 0xb5ee);
        re_mdio_write(sc, 0x14, 0x8088);
        re_mdio_write(sc, 0x14, 0xa4ee);
        re_mdio_write(sc, 0x14, 0x8089);
        re_mdio_write(sc, 0x14, 0x44ee);
        re_mdio_write(sc, 0x14, 0x809a);
        re_mdio_write(sc, 0x14, 0xa4ee);
        re_mdio_write(sc, 0x14, 0x809b);
        re_mdio_write(sc, 0x14, 0x44ee);
        re_mdio_write(sc, 0x14, 0x809c);
        re_mdio_write(sc, 0x14, 0xa7ee);
        re_mdio_write(sc, 0x14, 0x80a5);
        re_mdio_write(sc, 0x14, 0xa7d2);
        re_mdio_write(sc, 0x14, 0x0002);
        re_mdio_write(sc, 0x14, 0x0e66);
        re_mdio_write(sc, 0x14, 0x0285);
        re_mdio_write(sc, 0x14, 0xc0ee);
        re_mdio_write(sc, 0x14, 0x87fc);
        re_mdio_write(sc, 0x14, 0x00e0);
        re_mdio_write(sc, 0x14, 0x8245);
        re_mdio_write(sc, 0x14, 0xf622);
        re_mdio_write(sc, 0x14, 0xe482);
        re_mdio_write(sc, 0x14, 0x45ef);
        re_mdio_write(sc, 0x14, 0x96fe);
        re_mdio_write(sc, 0x14, 0xfdfc);
        re_mdio_write(sc, 0x14, 0x0402);
        re_mdio_write(sc, 0x14, 0x847a);
        re_mdio_write(sc, 0x14, 0x0284);
        re_mdio_write(sc, 0x14, 0xb302);
        re_mdio_write(sc, 0x14, 0x0cab);
        re_mdio_write(sc, 0x14, 0x020c);
        re_mdio_write(sc, 0x14, 0xc402);
        re_mdio_write(sc, 0x14, 0x0cef);
        re_mdio_write(sc, 0x14, 0x020d);
        re_mdio_write(sc, 0x14, 0x0802);
        re_mdio_write(sc, 0x14, 0x0d33);
        re_mdio_write(sc, 0x14, 0x020c);
        re_mdio_write(sc, 0x14, 0x3d04);
        re_mdio_write(sc, 0x14, 0xf8fa);
        re_mdio_write(sc, 0x14, 0xef69);
        re_mdio_write(sc, 0x14, 0xe182);
        re_mdio_write(sc, 0x14, 0x2fac);
        re_mdio_write(sc, 0x14, 0x291a);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x24ac);
        re_mdio_write(sc, 0x14, 0x2102);
        re_mdio_write(sc, 0x14, 0xae22);
        re_mdio_write(sc, 0x14, 0x0210);
        re_mdio_write(sc, 0x14, 0x57f6);
        re_mdio_write(sc, 0x14, 0x21e4);
        re_mdio_write(sc, 0x14, 0x8224);
        re_mdio_write(sc, 0x14, 0xd101);
        re_mdio_write(sc, 0x14, 0xbf44);
        re_mdio_write(sc, 0x14, 0xd202);
        re_mdio_write(sc, 0x14, 0x4259);
        re_mdio_write(sc, 0x14, 0xae10);
        re_mdio_write(sc, 0x14, 0x0212);
        re_mdio_write(sc, 0x14, 0x4cf6);
        re_mdio_write(sc, 0x14, 0x29e5);
        re_mdio_write(sc, 0x14, 0x822f);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x24f6);
        re_mdio_write(sc, 0x14, 0x21e4);
        re_mdio_write(sc, 0x14, 0x8224);
        re_mdio_write(sc, 0x14, 0xef96);
        re_mdio_write(sc, 0x14, 0xfefc);
        re_mdio_write(sc, 0x14, 0x04f8);
        re_mdio_write(sc, 0x14, 0xe182);
        re_mdio_write(sc, 0x14, 0x2fac);
        re_mdio_write(sc, 0x14, 0x2a18);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x24ac);
        re_mdio_write(sc, 0x14, 0x2202);
        re_mdio_write(sc, 0x14, 0xae26);
        re_mdio_write(sc, 0x14, 0x0284);
        re_mdio_write(sc, 0x14, 0xf802);
        re_mdio_write(sc, 0x14, 0x8565);
        re_mdio_write(sc, 0x14, 0xd101);
        re_mdio_write(sc, 0x14, 0xbf44);
        re_mdio_write(sc, 0x14, 0xd502);
        re_mdio_write(sc, 0x14, 0x4259);
        re_mdio_write(sc, 0x14, 0xae0e);
        re_mdio_write(sc, 0x14, 0x0284);
        re_mdio_write(sc, 0x14, 0xea02);
        re_mdio_write(sc, 0x14, 0x85a9);
        re_mdio_write(sc, 0x14, 0xe182);
        re_mdio_write(sc, 0x14, 0x2ff6);
        re_mdio_write(sc, 0x14, 0x2ae5);
        re_mdio_write(sc, 0x14, 0x822f);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x24f6);
        re_mdio_write(sc, 0x14, 0x22e4);
        re_mdio_write(sc, 0x14, 0x8224);
        re_mdio_write(sc, 0x14, 0xfc04);
        re_mdio_write(sc, 0x14, 0xf9e2);
        re_mdio_write(sc, 0x14, 0x8011);
        re_mdio_write(sc, 0x14, 0xad31);
        re_mdio_write(sc, 0x14, 0x05d2);
        re_mdio_write(sc, 0x14, 0x0002);
        re_mdio_write(sc, 0x14, 0x0e66);
        re_mdio_write(sc, 0x14, 0xfd04);
        re_mdio_write(sc, 0x14, 0xf8f9);
        re_mdio_write(sc, 0x14, 0xfaef);
        re_mdio_write(sc, 0x14, 0x69e0);
        re_mdio_write(sc, 0x14, 0x8011);
        re_mdio_write(sc, 0x14, 0xad21);
        re_mdio_write(sc, 0x14, 0x5cbf);
        re_mdio_write(sc, 0x14, 0x43be);
        re_mdio_write(sc, 0x14, 0x0242);
        re_mdio_write(sc, 0x14, 0x97ac);
        re_mdio_write(sc, 0x14, 0x281b);
        re_mdio_write(sc, 0x14, 0xbf43);
        re_mdio_write(sc, 0x14, 0xc102);
        re_mdio_write(sc, 0x14, 0x4297);
        re_mdio_write(sc, 0x14, 0xac28);
        re_mdio_write(sc, 0x14, 0x12bf);
        re_mdio_write(sc, 0x14, 0x43c7);
        re_mdio_write(sc, 0x14, 0x0242);
        re_mdio_write(sc, 0x14, 0x97ac);
        re_mdio_write(sc, 0x14, 0x2804);
        re_mdio_write(sc, 0x14, 0xd300);
        re_mdio_write(sc, 0x14, 0xae07);
        re_mdio_write(sc, 0x14, 0xd306);
        re_mdio_write(sc, 0x14, 0xaf85);
        re_mdio_write(sc, 0x14, 0x56d3);
        re_mdio_write(sc, 0x14, 0x03e0);
        re_mdio_write(sc, 0x14, 0x8011);
        re_mdio_write(sc, 0x14, 0xad26);
        re_mdio_write(sc, 0x14, 0x25bf);
        re_mdio_write(sc, 0x14, 0x4559);
        re_mdio_write(sc, 0x14, 0x0242);
        re_mdio_write(sc, 0x14, 0x97e2);
        re_mdio_write(sc, 0x14, 0x8073);
        re_mdio_write(sc, 0x14, 0x0d21);
        re_mdio_write(sc, 0x14, 0xf637);
        re_mdio_write(sc, 0x14, 0x0d11);
        re_mdio_write(sc, 0x14, 0xf62f);
        re_mdio_write(sc, 0x14, 0x1b21);
        re_mdio_write(sc, 0x14, 0xaa02);
        re_mdio_write(sc, 0x14, 0xae10);
        re_mdio_write(sc, 0x14, 0xe280);
        re_mdio_write(sc, 0x14, 0x740d);
        re_mdio_write(sc, 0x14, 0x21f6);
        re_mdio_write(sc, 0x14, 0x371b);
        re_mdio_write(sc, 0x14, 0x21aa);
        re_mdio_write(sc, 0x14, 0x0313);
        re_mdio_write(sc, 0x14, 0xae02);
        re_mdio_write(sc, 0x14, 0x2b02);
        re_mdio_write(sc, 0x14, 0x020e);
        re_mdio_write(sc, 0x14, 0x5102);
        re_mdio_write(sc, 0x14, 0x0e66);
        re_mdio_write(sc, 0x14, 0x020f);
        re_mdio_write(sc, 0x14, 0xa3ef);
        re_mdio_write(sc, 0x14, 0x96fe);
        re_mdio_write(sc, 0x14, 0xfdfc);
        re_mdio_write(sc, 0x14, 0x04f8);
        re_mdio_write(sc, 0x14, 0xf9fa);
        re_mdio_write(sc, 0x14, 0xef69);
        re_mdio_write(sc, 0x14, 0xe080);
        re_mdio_write(sc, 0x14, 0x12ad);
        re_mdio_write(sc, 0x14, 0x2733);
        re_mdio_write(sc, 0x14, 0xbf43);
        re_mdio_write(sc, 0x14, 0xbe02);
        re_mdio_write(sc, 0x14, 0x4297);
        re_mdio_write(sc, 0x14, 0xac28);
        re_mdio_write(sc, 0x14, 0x09bf);
        re_mdio_write(sc, 0x14, 0x43c1);
        re_mdio_write(sc, 0x14, 0x0242);
        re_mdio_write(sc, 0x14, 0x97ad);
        re_mdio_write(sc, 0x14, 0x2821);
        re_mdio_write(sc, 0x14, 0xbf45);
        re_mdio_write(sc, 0x14, 0x5902);
        re_mdio_write(sc, 0x14, 0x4297);
        re_mdio_write(sc, 0x14, 0xe387);
        re_mdio_write(sc, 0x14, 0xffd2);
        re_mdio_write(sc, 0x14, 0x001b);
        re_mdio_write(sc, 0x14, 0x45ac);
        re_mdio_write(sc, 0x14, 0x2711);
        re_mdio_write(sc, 0x14, 0xe187);
        re_mdio_write(sc, 0x14, 0xfebf);
        re_mdio_write(sc, 0x14, 0x87e4);
        re_mdio_write(sc, 0x14, 0x0242);
        re_mdio_write(sc, 0x14, 0x590d);
        re_mdio_write(sc, 0x14, 0x11bf);
        re_mdio_write(sc, 0x14, 0x87e7);
        re_mdio_write(sc, 0x14, 0x0242);
        re_mdio_write(sc, 0x14, 0x59ef);
        re_mdio_write(sc, 0x14, 0x96fe);
        re_mdio_write(sc, 0x14, 0xfdfc);
        re_mdio_write(sc, 0x14, 0x04f8);
        re_mdio_write(sc, 0x14, 0xfaef);
        re_mdio_write(sc, 0x14, 0x69d1);
        re_mdio_write(sc, 0x14, 0x00bf);
        re_mdio_write(sc, 0x14, 0x87e4);
        re_mdio_write(sc, 0x14, 0x0242);
        re_mdio_write(sc, 0x14, 0x59bf);
        re_mdio_write(sc, 0x14, 0x87e7);
        re_mdio_write(sc, 0x14, 0x0242);
        re_mdio_write(sc, 0x14, 0x59ef);
        re_mdio_write(sc, 0x14, 0x96fe);
        re_mdio_write(sc, 0x14, 0xfc04);
        re_mdio_write(sc, 0x14, 0xee87);
        re_mdio_write(sc, 0x14, 0xff46);
        re_mdio_write(sc, 0x14, 0xee87);
        re_mdio_write(sc, 0x14, 0xfe01);
        re_mdio_write(sc, 0x14, 0x04f8);
        re_mdio_write(sc, 0x14, 0xfaef);
        re_mdio_write(sc, 0x14, 0x69e0);
        re_mdio_write(sc, 0x14, 0x8241);
        re_mdio_write(sc, 0x14, 0xa000);
        re_mdio_write(sc, 0x14, 0x0502);
        re_mdio_write(sc, 0x14, 0x85eb);
        re_mdio_write(sc, 0x14, 0xae0e);
        re_mdio_write(sc, 0x14, 0xa001);
        re_mdio_write(sc, 0x14, 0x0502);
        re_mdio_write(sc, 0x14, 0x1a5a);
        re_mdio_write(sc, 0x14, 0xae06);
        re_mdio_write(sc, 0x14, 0xa002);
        re_mdio_write(sc, 0x14, 0x0302);
        re_mdio_write(sc, 0x14, 0x1ae6);
        re_mdio_write(sc, 0x14, 0xef96);
        re_mdio_write(sc, 0x14, 0xfefc);
        re_mdio_write(sc, 0x14, 0x04f8);
        re_mdio_write(sc, 0x14, 0xf9fa);
        re_mdio_write(sc, 0x14, 0xef69);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x29f6);
        re_mdio_write(sc, 0x14, 0x21e4);
        re_mdio_write(sc, 0x14, 0x8229);
        re_mdio_write(sc, 0x14, 0xe080);
        re_mdio_write(sc, 0x14, 0x10ac);
        re_mdio_write(sc, 0x14, 0x2202);
        re_mdio_write(sc, 0x14, 0xae76);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x27f7);
        re_mdio_write(sc, 0x14, 0x21e4);
        re_mdio_write(sc, 0x14, 0x8227);
        re_mdio_write(sc, 0x14, 0xbf43);
        re_mdio_write(sc, 0x14, 0x1302);
        re_mdio_write(sc, 0x14, 0x4297);
        re_mdio_write(sc, 0x14, 0xef21);
        re_mdio_write(sc, 0x14, 0xbf43);
        re_mdio_write(sc, 0x14, 0x1602);
        re_mdio_write(sc, 0x14, 0x4297);
        re_mdio_write(sc, 0x14, 0x0c11);
        re_mdio_write(sc, 0x14, 0x1e21);
        re_mdio_write(sc, 0x14, 0xbf43);
        re_mdio_write(sc, 0x14, 0x1902);
        re_mdio_write(sc, 0x14, 0x4297);
        re_mdio_write(sc, 0x14, 0x0c12);
        re_mdio_write(sc, 0x14, 0x1e21);
        re_mdio_write(sc, 0x14, 0xe682);
        re_mdio_write(sc, 0x14, 0x43a2);
        re_mdio_write(sc, 0x14, 0x000a);
        re_mdio_write(sc, 0x14, 0xe182);
        re_mdio_write(sc, 0x14, 0x27f6);
        re_mdio_write(sc, 0x14, 0x29e5);
        re_mdio_write(sc, 0x14, 0x8227);
        re_mdio_write(sc, 0x14, 0xae42);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x44f7);
        re_mdio_write(sc, 0x14, 0x21e4);
        re_mdio_write(sc, 0x14, 0x8244);
        re_mdio_write(sc, 0x14, 0x0246);
        re_mdio_write(sc, 0x14, 0xaebf);
        re_mdio_write(sc, 0x14, 0x4325);
        re_mdio_write(sc, 0x14, 0x0242);
        re_mdio_write(sc, 0x14, 0x97ef);
        re_mdio_write(sc, 0x14, 0x21bf);
        re_mdio_write(sc, 0x14, 0x431c);
        re_mdio_write(sc, 0x14, 0x0242);
        re_mdio_write(sc, 0x14, 0x970c);
        re_mdio_write(sc, 0x14, 0x121e);
        re_mdio_write(sc, 0x14, 0x21bf);
        re_mdio_write(sc, 0x14, 0x431f);
        re_mdio_write(sc, 0x14, 0x0242);
        re_mdio_write(sc, 0x14, 0x970c);
        re_mdio_write(sc, 0x14, 0x131e);
        re_mdio_write(sc, 0x14, 0x21bf);
        re_mdio_write(sc, 0x14, 0x4328);
        re_mdio_write(sc, 0x14, 0x0242);
        re_mdio_write(sc, 0x14, 0x970c);
        re_mdio_write(sc, 0x14, 0x141e);
        re_mdio_write(sc, 0x14, 0x21bf);
        re_mdio_write(sc, 0x14, 0x44b1);
        re_mdio_write(sc, 0x14, 0x0242);
        re_mdio_write(sc, 0x14, 0x970c);
        re_mdio_write(sc, 0x14, 0x161e);
        re_mdio_write(sc, 0x14, 0x21e6);
        re_mdio_write(sc, 0x14, 0x8242);
        re_mdio_write(sc, 0x14, 0xee82);
        re_mdio_write(sc, 0x14, 0x4101);
        re_mdio_write(sc, 0x14, 0xef96);
        re_mdio_write(sc, 0x14, 0xfefd);
        re_mdio_write(sc, 0x14, 0xfc04);
        re_mdio_write(sc, 0x14, 0xf8fa);
        re_mdio_write(sc, 0x14, 0xef69);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x46a0);
        re_mdio_write(sc, 0x14, 0x0005);
        re_mdio_write(sc, 0x14, 0x0286);
        re_mdio_write(sc, 0x14, 0x96ae);
        re_mdio_write(sc, 0x14, 0x06a0);
        re_mdio_write(sc, 0x14, 0x0103);
        re_mdio_write(sc, 0x14, 0x0219);
        re_mdio_write(sc, 0x14, 0x19ef);
        re_mdio_write(sc, 0x14, 0x96fe);
        re_mdio_write(sc, 0x14, 0xfc04);
        re_mdio_write(sc, 0x14, 0xf8fa);
        re_mdio_write(sc, 0x14, 0xef69);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x29f6);
        re_mdio_write(sc, 0x14, 0x20e4);
        re_mdio_write(sc, 0x14, 0x8229);
        re_mdio_write(sc, 0x14, 0xe080);
        re_mdio_write(sc, 0x14, 0x10ac);
        re_mdio_write(sc, 0x14, 0x2102);
        re_mdio_write(sc, 0x14, 0xae54);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x27f7);
        re_mdio_write(sc, 0x14, 0x20e4);
        re_mdio_write(sc, 0x14, 0x8227);
        re_mdio_write(sc, 0x14, 0xbf42);
        re_mdio_write(sc, 0x14, 0xe602);
        re_mdio_write(sc, 0x14, 0x4297);
        re_mdio_write(sc, 0x14, 0xac28);
        re_mdio_write(sc, 0x14, 0x22bf);
        re_mdio_write(sc, 0x14, 0x430d);
        re_mdio_write(sc, 0x14, 0x0242);
        re_mdio_write(sc, 0x14, 0x97e5);
        re_mdio_write(sc, 0x14, 0x8247);
        re_mdio_write(sc, 0x14, 0xac28);
        re_mdio_write(sc, 0x14, 0x20d1);
        re_mdio_write(sc, 0x14, 0x03bf);
        re_mdio_write(sc, 0x14, 0x4307);
        re_mdio_write(sc, 0x14, 0x0242);
        re_mdio_write(sc, 0x14, 0x59ee);
        re_mdio_write(sc, 0x14, 0x8246);
        re_mdio_write(sc, 0x14, 0x00e1);
        re_mdio_write(sc, 0x14, 0x8227);
        re_mdio_write(sc, 0x14, 0xf628);
        re_mdio_write(sc, 0x14, 0xe582);
        re_mdio_write(sc, 0x14, 0x27ae);
        re_mdio_write(sc, 0x14, 0x21d1);
        re_mdio_write(sc, 0x14, 0x04bf);
        re_mdio_write(sc, 0x14, 0x4307);
        re_mdio_write(sc, 0x14, 0x0242);
        re_mdio_write(sc, 0x14, 0x59ae);
        re_mdio_write(sc, 0x14, 0x08d1);
        re_mdio_write(sc, 0x14, 0x05bf);
        re_mdio_write(sc, 0x14, 0x4307);
        re_mdio_write(sc, 0x14, 0x0242);
        re_mdio_write(sc, 0x14, 0x59e0);
        re_mdio_write(sc, 0x14, 0x8244);
        re_mdio_write(sc, 0x14, 0xf720);
        re_mdio_write(sc, 0x14, 0xe482);
        re_mdio_write(sc, 0x14, 0x4402);
        re_mdio_write(sc, 0x14, 0x46ae);
        re_mdio_write(sc, 0x14, 0xee82);
        re_mdio_write(sc, 0x14, 0x4601);
        re_mdio_write(sc, 0x14, 0xef96);
        re_mdio_write(sc, 0x14, 0xfefc);
        re_mdio_write(sc, 0x14, 0x04f8);
        re_mdio_write(sc, 0x14, 0xfaef);
        re_mdio_write(sc, 0x14, 0x69e0);
        re_mdio_write(sc, 0x14, 0x8013);
        re_mdio_write(sc, 0x14, 0xad24);
        re_mdio_write(sc, 0x14, 0x1cbf);
        re_mdio_write(sc, 0x14, 0x87f0);
        re_mdio_write(sc, 0x14, 0x0242);
        re_mdio_write(sc, 0x14, 0x97ad);
        re_mdio_write(sc, 0x14, 0x2813);
        re_mdio_write(sc, 0x14, 0xe087);
        re_mdio_write(sc, 0x14, 0xfca0);
        re_mdio_write(sc, 0x14, 0x0005);
        re_mdio_write(sc, 0x14, 0x0287);
        re_mdio_write(sc, 0x14, 0x36ae);
        re_mdio_write(sc, 0x14, 0x10a0);
        re_mdio_write(sc, 0x14, 0x0105);
        re_mdio_write(sc, 0x14, 0x0287);
        re_mdio_write(sc, 0x14, 0x48ae);
        re_mdio_write(sc, 0x14, 0x08e0);
        re_mdio_write(sc, 0x14, 0x8230);
        re_mdio_write(sc, 0x14, 0xf626);
        re_mdio_write(sc, 0x14, 0xe482);
        re_mdio_write(sc, 0x14, 0x30ef);
        re_mdio_write(sc, 0x14, 0x96fe);
        re_mdio_write(sc, 0x14, 0xfc04);
        re_mdio_write(sc, 0x14, 0xf8e0);
        re_mdio_write(sc, 0x14, 0x8245);
        re_mdio_write(sc, 0x14, 0xf722);
        re_mdio_write(sc, 0x14, 0xe482);
        re_mdio_write(sc, 0x14, 0x4502);
        re_mdio_write(sc, 0x14, 0x46ae);
        re_mdio_write(sc, 0x14, 0xee87);
        re_mdio_write(sc, 0x14, 0xfc01);
        re_mdio_write(sc, 0x14, 0xfc04);
        re_mdio_write(sc, 0x14, 0xf8fa);
        re_mdio_write(sc, 0x14, 0xef69);
        re_mdio_write(sc, 0x14, 0xfb02);
        re_mdio_write(sc, 0x14, 0x46d3);
        re_mdio_write(sc, 0x14, 0xad50);
        re_mdio_write(sc, 0x14, 0x2fbf);
        re_mdio_write(sc, 0x14, 0x87ed);
        re_mdio_write(sc, 0x14, 0xd101);
        re_mdio_write(sc, 0x14, 0x0242);
        re_mdio_write(sc, 0x14, 0x59bf);
        re_mdio_write(sc, 0x14, 0x87ed);
        re_mdio_write(sc, 0x14, 0xd100);
        re_mdio_write(sc, 0x14, 0x0242);
        re_mdio_write(sc, 0x14, 0x59e0);
        re_mdio_write(sc, 0x14, 0x8245);
        re_mdio_write(sc, 0x14, 0xf622);
        re_mdio_write(sc, 0x14, 0xe482);
        re_mdio_write(sc, 0x14, 0x4502);
        re_mdio_write(sc, 0x14, 0x46ae);
        re_mdio_write(sc, 0x14, 0xd100);
        re_mdio_write(sc, 0x14, 0xbf87);
        re_mdio_write(sc, 0x14, 0xf002);
        re_mdio_write(sc, 0x14, 0x4259);
        re_mdio_write(sc, 0x14, 0xee87);
        re_mdio_write(sc, 0x14, 0xfc00);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x30f6);
        re_mdio_write(sc, 0x14, 0x26e4);
        re_mdio_write(sc, 0x14, 0x8230);
        re_mdio_write(sc, 0x14, 0xffef);
        re_mdio_write(sc, 0x14, 0x96fe);
        re_mdio_write(sc, 0x14, 0xfc04);
        re_mdio_write(sc, 0x14, 0xf8f9);
        re_mdio_write(sc, 0x14, 0xface);
        re_mdio_write(sc, 0x14, 0xfaef);
        re_mdio_write(sc, 0x14, 0x69fb);
        re_mdio_write(sc, 0x14, 0xbf87);
        re_mdio_write(sc, 0x14, 0xb3d7);
        re_mdio_write(sc, 0x14, 0x001c);
        re_mdio_write(sc, 0x14, 0xd819);
        re_mdio_write(sc, 0x14, 0xd919);
        re_mdio_write(sc, 0x14, 0xda19);
        re_mdio_write(sc, 0x14, 0xdb19);
        re_mdio_write(sc, 0x14, 0x07ef);
        re_mdio_write(sc, 0x14, 0x9502);
        re_mdio_write(sc, 0x14, 0x4259);
        re_mdio_write(sc, 0x14, 0x073f);
        re_mdio_write(sc, 0x14, 0x0004);
        re_mdio_write(sc, 0x14, 0x9fec);
        re_mdio_write(sc, 0x14, 0xffef);
        re_mdio_write(sc, 0x14, 0x96fe);
        re_mdio_write(sc, 0x14, 0xc6fe);
        re_mdio_write(sc, 0x14, 0xfdfc);
        re_mdio_write(sc, 0x14, 0x0400);
        re_mdio_write(sc, 0x14, 0x0145);
        re_mdio_write(sc, 0x14, 0x7d00);
        re_mdio_write(sc, 0x14, 0x0345);
        re_mdio_write(sc, 0x14, 0x5c00);
        re_mdio_write(sc, 0x14, 0x0143);
        re_mdio_write(sc, 0x14, 0x4f00);
        re_mdio_write(sc, 0x14, 0x0387);
        re_mdio_write(sc, 0x14, 0xdb00);
        re_mdio_write(sc, 0x14, 0x0987);
        re_mdio_write(sc, 0x14, 0xde00);
        re_mdio_write(sc, 0x14, 0x0987);
        re_mdio_write(sc, 0x14, 0xe100);
        re_mdio_write(sc, 0x14, 0x0087);
        re_mdio_write(sc, 0x14, 0xeaa4);
        re_mdio_write(sc, 0x14, 0x00b8);
        re_mdio_write(sc, 0x14, 0x20c4);
        re_mdio_write(sc, 0x14, 0x1600);
        re_mdio_write(sc, 0x14, 0x000f);
        re_mdio_write(sc, 0x14, 0xf800);
        re_mdio_write(sc, 0x14, 0x7098);
        re_mdio_write(sc, 0x14, 0xa58a);
        re_mdio_write(sc, 0x14, 0xb6a8);
        re_mdio_write(sc, 0x14, 0x3e50);
        re_mdio_write(sc, 0x14, 0xa83e);
        re_mdio_write(sc, 0x14, 0x33bc);
        re_mdio_write(sc, 0x14, 0xc622);
        re_mdio_write(sc, 0x14, 0xbcc6);
        re_mdio_write(sc, 0x14, 0xaaa4);
        re_mdio_write(sc, 0x14, 0x42ff);
        re_mdio_write(sc, 0x14, 0xc408);
        re_mdio_write(sc, 0x14, 0x00c4);
        re_mdio_write(sc, 0x14, 0x16a8);
        re_mdio_write(sc, 0x14, 0xbcc0);
        re_mdio_write(sc, 0x13, 0xb818);
        re_mdio_write(sc, 0x14, 0x02f3);
        re_mdio_write(sc, 0x13, 0xb81a);
        re_mdio_write(sc, 0x14, 0x17d1);
        re_mdio_write(sc, 0x13, 0xb81c);
        re_mdio_write(sc, 0x14, 0x185a);
        re_mdio_write(sc, 0x13, 0xb81e);
        re_mdio_write(sc, 0x14, 0x3c66);
        re_mdio_write(sc, 0x13, 0xb820);
        re_mdio_write(sc, 0x14, 0x021f);
        re_mdio_write(sc, 0x13, 0xc416);
        re_mdio_write(sc, 0x14, 0x0500);
        re_mdio_write(sc, 0x13, 0xb82e);
        re_mdio_write(sc, 0x14, 0xfffc);

        re_mdio_write(sc, 0x1F, 0x0A43);
        re_mdio_write(sc, 0x13, 0x0000);
        re_mdio_write(sc, 0x14, 0x0000);
        re_mdio_write(sc, 0x1f, 0x0B82);
        PhyRegValue = re_mdio_read(sc, 0x10);
        PhyRegValue &= ~(BIT_9);
        re_mdio_write(sc, 0x10, PhyRegValue);
        re_mdio_write(sc, 0x1f, 0x0A43);
        re_mdio_write(sc, 0x13, 0x8146);
        re_mdio_write(sc, 0x14, 0x0000);

        re_clear_phy_mcu_patch_request(sc);
        if (sc->RequiredSecLanDonglePatch) {
                re_mdio_write(sc, 0x1F, 0x0A43);
                PhyRegValue = re_mdio_read(sc, 0x11);
                PhyRegValue &= ~(BIT_6);
                re_mdio_write(sc, 0x11, PhyRegValue);
        }
}

static void re_set_phy_mcu_8168gu_2(struct re_softc *sc)
{
        u_int16_t PhyRegValue;

        re_set_phy_mcu_patch_request(sc);

        re_mdio_write(sc, 0x1f, 0x0A43);
        re_mdio_write(sc, 0x13, 0x8146);
        re_mdio_write(sc, 0x14, 0x0300);
        re_mdio_write(sc, 0x13, 0xB82E);
        re_mdio_write(sc, 0x14, 0x0001);

        re_mdio_write(sc, 0x1F, 0x0A43);
        re_mdio_write(sc, 0x13, 0xb820);
        re_mdio_write(sc, 0x14, 0x0290);
        re_mdio_write(sc, 0x13, 0xa012);
        re_mdio_write(sc, 0x14, 0x0000);
        re_mdio_write(sc, 0x13, 0xa014);
        re_mdio_write(sc, 0x14, 0x2c04);
        re_mdio_write(sc, 0x14, 0x2c07);
        re_mdio_write(sc, 0x14, 0x2c07);
        re_mdio_write(sc, 0x14, 0x2c07);
        re_mdio_write(sc, 0x14, 0xa304);
        re_mdio_write(sc, 0x14, 0xa301);
        re_mdio_write(sc, 0x14, 0x207e);
        re_mdio_write(sc, 0x13, 0xa01a);
        re_mdio_write(sc, 0x14, 0x0000);
        re_mdio_write(sc, 0x13, 0xa006);
        re_mdio_write(sc, 0x14, 0x0fff);
        re_mdio_write(sc, 0x13, 0xa004);
        re_mdio_write(sc, 0x14, 0x0fff);
        re_mdio_write(sc, 0x13, 0xa002);
        re_mdio_write(sc, 0x14, 0x0fff);
        re_mdio_write(sc, 0x13, 0xa000);
        re_mdio_write(sc, 0x14, 0x107c);
        re_mdio_write(sc, 0x13, 0xb820);
        re_mdio_write(sc, 0x14, 0x0210);

        re_mdio_write(sc, 0x1F, 0x0A43);
        re_mdio_write(sc, 0x13, 0x0000);
        re_mdio_write(sc, 0x14, 0x0000);
        re_mdio_write(sc, 0x1f, 0x0B82);
        PhyRegValue = re_mdio_read(sc, 0x17);
        PhyRegValue &= ~(BIT_0);
        re_mdio_write(sc, 0x17, PhyRegValue);
        re_mdio_write(sc, 0x1f, 0x0A43);
        re_mdio_write(sc, 0x13, 0x8146);
        re_mdio_write(sc, 0x14, 0x0000);

        re_clear_phy_mcu_patch_request(sc);
        if (sc->RequiredSecLanDonglePatch) {
                re_mdio_write(sc, 0x1F, 0x0A43);
                PhyRegValue = re_mdio_read(sc, 0x11);
                PhyRegValue &= ~(BIT_6);
                re_mdio_write(sc, 0x11, PhyRegValue);
        }
}

static void re_set_phy_mcu_8411b_1(struct re_softc *sc)
{
        u_int16_t PhyRegValue;

        re_set_phy_mcu_patch_request(sc);

        re_mdio_write(sc, 0x1f, 0x0A43);
        re_mdio_write(sc, 0x13, 0x8146);
        re_mdio_write(sc, 0x14, 0x0100);
        re_mdio_write(sc, 0x13, 0xB82E);
        re_mdio_write(sc, 0x14, 0x0001);

        re_mdio_write(sc, 0x1F, 0x0A43);
        re_mdio_write(sc, 0x13, 0xb820);
        re_mdio_write(sc, 0x14, 0x0290);
        re_mdio_write(sc, 0x13, 0xa012);
        re_mdio_write(sc, 0x14, 0x0000);
        re_mdio_write(sc, 0x13, 0xa014);
        re_mdio_write(sc, 0x14, 0x2c04);
        re_mdio_write(sc, 0x14, 0x2c07);
        re_mdio_write(sc, 0x14, 0x2c07);
        re_mdio_write(sc, 0x14, 0x2c07);
        re_mdio_write(sc, 0x14, 0xa304);
        re_mdio_write(sc, 0x14, 0xa301);
        re_mdio_write(sc, 0x14, 0x207e);
        re_mdio_write(sc, 0x13, 0xa01a);
        re_mdio_write(sc, 0x14, 0x0000);
        re_mdio_write(sc, 0x13, 0xa006);
        re_mdio_write(sc, 0x14, 0x0fff);
        re_mdio_write(sc, 0x13, 0xa004);
        re_mdio_write(sc, 0x14, 0x0fff);
        re_mdio_write(sc, 0x13, 0xa002);
        re_mdio_write(sc, 0x14, 0x0fff);
        re_mdio_write(sc, 0x13, 0xa000);
        re_mdio_write(sc, 0x14, 0x107c);
        re_mdio_write(sc, 0x13, 0xb820);
        re_mdio_write(sc, 0x14, 0x0210);

        re_mdio_write(sc, 0x1F, 0x0A43);
        re_mdio_write(sc, 0x13, 0x0000);
        re_mdio_write(sc, 0x14, 0x0000);
        re_mdio_write(sc, 0x1f, 0x0B82);
        PhyRegValue = re_mdio_read(sc, 0x17);
        PhyRegValue &= ~(BIT_0);
        re_mdio_write(sc, 0x17, PhyRegValue);
        re_mdio_write(sc, 0x1f, 0x0A43);
        re_mdio_write(sc, 0x13, 0x8146);
        re_mdio_write(sc, 0x14, 0x0000);

        re_clear_phy_mcu_patch_request(sc);
        if (sc->RequiredSecLanDonglePatch) {
                re_mdio_write(sc, 0x1F, 0x0A43);
                PhyRegValue = re_mdio_read(sc, 0x11);
                PhyRegValue &= ~(BIT_6);
                re_mdio_write(sc, 0x11, PhyRegValue);
        }
}

static void re_set_phy_mcu_8168h_1(struct re_softc *sc)
{
        u_int16_t PhyRegValue;

        re_set_phy_mcu_patch_request(sc);

        re_mdio_write(sc, 0x1f, 0x0A43);
        re_mdio_write(sc, 0x13, 0x8028);
        re_mdio_write(sc, 0x14, 0x6200);
        re_mdio_write(sc, 0x13, 0xB82E);
        re_mdio_write(sc, 0x14, 0x0001);

        re_mdio_write(sc, 0x1F, 0x0A43);
        re_mdio_write(sc, 0x13, 0xB820);
        re_mdio_write(sc, 0x14, 0x0290);
        re_mdio_write(sc, 0x13, 0xA012);
        re_mdio_write(sc, 0x14, 0x0000);
        re_mdio_write(sc, 0x13, 0xA014);
        re_mdio_write(sc, 0x14, 0x2c04);
        re_mdio_write(sc, 0x14, 0x2c10);
        re_mdio_write(sc, 0x14, 0x2c10);
        re_mdio_write(sc, 0x14, 0x2c10);
        re_mdio_write(sc, 0x14, 0xa210);
        re_mdio_write(sc, 0x14, 0xa101);
        re_mdio_write(sc, 0x14, 0xce10);
        re_mdio_write(sc, 0x14, 0xe070);
        re_mdio_write(sc, 0x14, 0x0f40);
        re_mdio_write(sc, 0x14, 0xaf01);
        re_mdio_write(sc, 0x14, 0x8f01);
        re_mdio_write(sc, 0x14, 0x183e);
        re_mdio_write(sc, 0x14, 0x8e10);
        re_mdio_write(sc, 0x14, 0x8101);
        re_mdio_write(sc, 0x14, 0x8210);
        re_mdio_write(sc, 0x14, 0x28da);
        re_mdio_write(sc, 0x13, 0xA01A);
        re_mdio_write(sc, 0x14, 0x0000);
        re_mdio_write(sc, 0x13, 0xA006);
        re_mdio_write(sc, 0x14, 0x0017);
        re_mdio_write(sc, 0x13, 0xA004);
        re_mdio_write(sc, 0x14, 0x0015);
        re_mdio_write(sc, 0x13, 0xA002);
        re_mdio_write(sc, 0x14, 0x0013);
        re_mdio_write(sc, 0x13, 0xA000);
        re_mdio_write(sc, 0x14, 0x18d1);
        re_mdio_write(sc, 0x13, 0xB820);
        re_mdio_write(sc, 0x14, 0x0210);

        re_mdio_write(sc, 0x1F, 0x0A43);
        re_mdio_write(sc, 0x13, 0x0000);
        re_mdio_write(sc, 0x14, 0x0000);
        re_mdio_write(sc, 0x1f, 0x0B82);
        PhyRegValue = re_mdio_read(sc, 0x17);
        PhyRegValue &= ~(BIT_0);
        re_mdio_write(sc, 0x17, PhyRegValue);
        re_mdio_write(sc, 0x1f, 0x0A43);
        re_mdio_write(sc, 0x13, 0x8028);
        re_mdio_write(sc, 0x14, 0x0000);

        re_clear_phy_mcu_patch_request(sc);
        if (sc->RequiredSecLanDonglePatch) {
                re_mdio_write(sc, 0x1F, 0x0A43);
                PhyRegValue = re_mdio_read(sc, 0x11);
                PhyRegValue &= ~(BIT_6);
                re_mdio_write(sc, 0x11, PhyRegValue);
        }
}

static void re_set_phy_mcu_8168h_2(struct re_softc *sc)
{
        u_int16_t PhyRegValue;

        re_set_phy_mcu_patch_request(sc);

        re_mdio_write(sc, 0x1f, 0x0A43);
        re_mdio_write(sc, 0x13, 0x8028);
        re_mdio_write(sc, 0x14, 0x6201);
        re_mdio_write(sc, 0x13, 0xB82E);
        re_mdio_write(sc, 0x14, 0x0001);

        re_mdio_write(sc, 0x1F, 0x0A43);
        re_mdio_write(sc, 0x13, 0xB820);
        re_mdio_write(sc, 0x14, 0x0290);
        re_mdio_write(sc, 0x13, 0xA012);
        re_mdio_write(sc, 0x14, 0x0000);
        re_mdio_write(sc, 0x13, 0xA014);
        re_mdio_write(sc, 0x14, 0x2c04);
        re_mdio_write(sc, 0x14, 0x2c09);
        re_mdio_write(sc, 0x14, 0x2c0d);
        re_mdio_write(sc, 0x14, 0x2c12);
        re_mdio_write(sc, 0x14, 0xad01);
        re_mdio_write(sc, 0x14, 0xad01);
        re_mdio_write(sc, 0x14, 0xad01);
        re_mdio_write(sc, 0x14, 0xad01);
        re_mdio_write(sc, 0x14, 0x236c);
        re_mdio_write(sc, 0x14, 0xd03c);
        re_mdio_write(sc, 0x14, 0xd1aa);
        re_mdio_write(sc, 0x14, 0xc010);
        re_mdio_write(sc, 0x14, 0x2745);
        re_mdio_write(sc, 0x14, 0x33de);
        re_mdio_write(sc, 0x14, 0x16ba);
        re_mdio_write(sc, 0x14, 0x31ee);
        re_mdio_write(sc, 0x14, 0x2712);
        re_mdio_write(sc, 0x14, 0x274e);
        re_mdio_write(sc, 0x14, 0xc2bb);
        re_mdio_write(sc, 0x14, 0xd500);
        re_mdio_write(sc, 0x14, 0xc426);
        re_mdio_write(sc, 0x14, 0xd01d);
        re_mdio_write(sc, 0x14, 0xd1c3);
        re_mdio_write(sc, 0x14, 0x401c);
        re_mdio_write(sc, 0x14, 0xd501);
        re_mdio_write(sc, 0x14, 0xc2b3);
        re_mdio_write(sc, 0x14, 0xd500);
        re_mdio_write(sc, 0x14, 0xd00b);
        re_mdio_write(sc, 0x14, 0xd1c3);
        re_mdio_write(sc, 0x14, 0x401c);
        re_mdio_write(sc, 0x14, 0x241a);
        re_mdio_write(sc, 0x13, 0xA01A);
        re_mdio_write(sc, 0x14, 0x0000);
        re_mdio_write(sc, 0x13, 0xA006);
        re_mdio_write(sc, 0x14, 0x0414);
        re_mdio_write(sc, 0x13, 0xA004);
        re_mdio_write(sc, 0x14, 0x074c);
        re_mdio_write(sc, 0x13, 0xA002);
        re_mdio_write(sc, 0x14, 0x0744);
        re_mdio_write(sc, 0x13, 0xA000);
        re_mdio_write(sc, 0x14, 0xf36b);
        re_mdio_write(sc, 0x13, 0xB820);
        re_mdio_write(sc, 0x14, 0x0210);

        re_mdio_write(sc, 0x1F, 0x0A43);
        re_mdio_write(sc, 0x13, 0x8323);
        re_mdio_write(sc, 0x14, 0xaf83);
        re_mdio_write(sc, 0x14, 0x2faf);
        re_mdio_write(sc, 0x14, 0x853d);
        re_mdio_write(sc, 0x14, 0xaf85);
        re_mdio_write(sc, 0x14, 0x3daf);
        re_mdio_write(sc, 0x14, 0x853d);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x45ad);
        re_mdio_write(sc, 0x14, 0x2052);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x7ae3);
        re_mdio_write(sc, 0x14, 0x85fe);
        re_mdio_write(sc, 0x14, 0x1a03);
        re_mdio_write(sc, 0x14, 0x10e4);
        re_mdio_write(sc, 0x14, 0x85f6);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x7a1b);
        re_mdio_write(sc, 0x14, 0x03e4);
        re_mdio_write(sc, 0x14, 0x85fa);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x7be3);
        re_mdio_write(sc, 0x14, 0x85fe);
        re_mdio_write(sc, 0x14, 0x1a03);
        re_mdio_write(sc, 0x14, 0x10e4);
        re_mdio_write(sc, 0x14, 0x85f7);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x7b1b);
        re_mdio_write(sc, 0x14, 0x03e4);
        re_mdio_write(sc, 0x14, 0x85fb);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x7ce3);
        re_mdio_write(sc, 0x14, 0x85fe);
        re_mdio_write(sc, 0x14, 0x1a03);
        re_mdio_write(sc, 0x14, 0x10e4);
        re_mdio_write(sc, 0x14, 0x85f8);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x7c1b);
        re_mdio_write(sc, 0x14, 0x03e4);
        re_mdio_write(sc, 0x14, 0x85fc);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x7de3);
        re_mdio_write(sc, 0x14, 0x85fe);
        re_mdio_write(sc, 0x14, 0x1a03);
        re_mdio_write(sc, 0x14, 0x10e4);
        re_mdio_write(sc, 0x14, 0x85f9);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x7d1b);
        re_mdio_write(sc, 0x14, 0x03e4);
        re_mdio_write(sc, 0x14, 0x85fd);
        re_mdio_write(sc, 0x14, 0xae50);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x7ee3);
        re_mdio_write(sc, 0x14, 0x85ff);
        re_mdio_write(sc, 0x14, 0x1a03);
        re_mdio_write(sc, 0x14, 0x10e4);
        re_mdio_write(sc, 0x14, 0x85f6);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x7e1b);
        re_mdio_write(sc, 0x14, 0x03e4);
        re_mdio_write(sc, 0x14, 0x85fa);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x7fe3);
        re_mdio_write(sc, 0x14, 0x85ff);
        re_mdio_write(sc, 0x14, 0x1a03);
        re_mdio_write(sc, 0x14, 0x10e4);
        re_mdio_write(sc, 0x14, 0x85f7);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x7f1b);
        re_mdio_write(sc, 0x14, 0x03e4);
        re_mdio_write(sc, 0x14, 0x85fb);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x80e3);
        re_mdio_write(sc, 0x14, 0x85ff);
        re_mdio_write(sc, 0x14, 0x1a03);
        re_mdio_write(sc, 0x14, 0x10e4);
        re_mdio_write(sc, 0x14, 0x85f8);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x801b);
        re_mdio_write(sc, 0x14, 0x03e4);
        re_mdio_write(sc, 0x14, 0x85fc);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x81e3);
        re_mdio_write(sc, 0x14, 0x85ff);
        re_mdio_write(sc, 0x14, 0x1a03);
        re_mdio_write(sc, 0x14, 0x10e4);
        re_mdio_write(sc, 0x14, 0x85f9);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x811b);
        re_mdio_write(sc, 0x14, 0x03e4);
        re_mdio_write(sc, 0x14, 0x85fd);
        re_mdio_write(sc, 0x14, 0xe085);
        re_mdio_write(sc, 0x14, 0xf6ad);
        re_mdio_write(sc, 0x14, 0x2404);
        re_mdio_write(sc, 0x14, 0xee85);
        re_mdio_write(sc, 0x14, 0xf610);
        re_mdio_write(sc, 0x14, 0xe085);
        re_mdio_write(sc, 0x14, 0xf7ad);
        re_mdio_write(sc, 0x14, 0x2404);
        re_mdio_write(sc, 0x14, 0xee85);
        re_mdio_write(sc, 0x14, 0xf710);
        re_mdio_write(sc, 0x14, 0xe085);
        re_mdio_write(sc, 0x14, 0xf8ad);
        re_mdio_write(sc, 0x14, 0x2404);
        re_mdio_write(sc, 0x14, 0xee85);
        re_mdio_write(sc, 0x14, 0xf810);
        re_mdio_write(sc, 0x14, 0xe085);
        re_mdio_write(sc, 0x14, 0xf9ad);
        re_mdio_write(sc, 0x14, 0x2404);
        re_mdio_write(sc, 0x14, 0xee85);
        re_mdio_write(sc, 0x14, 0xf910);
        re_mdio_write(sc, 0x14, 0xe085);
        re_mdio_write(sc, 0x14, 0xfaad);
        re_mdio_write(sc, 0x14, 0x2704);
        re_mdio_write(sc, 0x14, 0xee85);
        re_mdio_write(sc, 0x14, 0xfa00);
        re_mdio_write(sc, 0x14, 0xe085);
        re_mdio_write(sc, 0x14, 0xfbad);
        re_mdio_write(sc, 0x14, 0x2704);
        re_mdio_write(sc, 0x14, 0xee85);
        re_mdio_write(sc, 0x14, 0xfb00);
        re_mdio_write(sc, 0x14, 0xe085);
        re_mdio_write(sc, 0x14, 0xfcad);
        re_mdio_write(sc, 0x14, 0x2704);
        re_mdio_write(sc, 0x14, 0xee85);
        re_mdio_write(sc, 0x14, 0xfc00);
        re_mdio_write(sc, 0x14, 0xe085);
        re_mdio_write(sc, 0x14, 0xfdad);
        re_mdio_write(sc, 0x14, 0x2704);
        re_mdio_write(sc, 0x14, 0xee85);
        re_mdio_write(sc, 0x14, 0xfd00);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x44ad);
        re_mdio_write(sc, 0x14, 0x203f);
        re_mdio_write(sc, 0x14, 0xe085);
        re_mdio_write(sc, 0x14, 0xf6e4);
        re_mdio_write(sc, 0x14, 0x8288);
        re_mdio_write(sc, 0x14, 0xe085);
        re_mdio_write(sc, 0x14, 0xfae4);
        re_mdio_write(sc, 0x14, 0x8289);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x440d);
        re_mdio_write(sc, 0x14, 0x0458);
        re_mdio_write(sc, 0x14, 0x01bf);
        re_mdio_write(sc, 0x14, 0x8264);
        re_mdio_write(sc, 0x14, 0x0215);
        re_mdio_write(sc, 0x14, 0x38bf);
        re_mdio_write(sc, 0x14, 0x824e);
        re_mdio_write(sc, 0x14, 0x0213);
        re_mdio_write(sc, 0x14, 0x06a0);
        re_mdio_write(sc, 0x14, 0x010f);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x44f6);
        re_mdio_write(sc, 0x14, 0x20e4);
        re_mdio_write(sc, 0x14, 0x8244);
        re_mdio_write(sc, 0x14, 0x580f);
        re_mdio_write(sc, 0x14, 0xe582);
        re_mdio_write(sc, 0x14, 0x5aae);
        re_mdio_write(sc, 0x14, 0x0ebf);
        re_mdio_write(sc, 0x14, 0x825e);
        re_mdio_write(sc, 0x14, 0xe382);
        re_mdio_write(sc, 0x14, 0x44f7);
        re_mdio_write(sc, 0x14, 0x3ce7);
        re_mdio_write(sc, 0x14, 0x8244);
        re_mdio_write(sc, 0x14, 0x0212);
        re_mdio_write(sc, 0x14, 0xf0ad);
        re_mdio_write(sc, 0x14, 0x213f);
        re_mdio_write(sc, 0x14, 0xe085);
        re_mdio_write(sc, 0x14, 0xf7e4);
        re_mdio_write(sc, 0x14, 0x8288);
        re_mdio_write(sc, 0x14, 0xe085);
        re_mdio_write(sc, 0x14, 0xfbe4);
        re_mdio_write(sc, 0x14, 0x8289);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x440d);
        re_mdio_write(sc, 0x14, 0x0558);
        re_mdio_write(sc, 0x14, 0x01bf);
        re_mdio_write(sc, 0x14, 0x826b);
        re_mdio_write(sc, 0x14, 0x0215);
        re_mdio_write(sc, 0x14, 0x38bf);
        re_mdio_write(sc, 0x14, 0x824f);
        re_mdio_write(sc, 0x14, 0x0213);
        re_mdio_write(sc, 0x14, 0x06a0);
        re_mdio_write(sc, 0x14, 0x010f);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x44f6);
        re_mdio_write(sc, 0x14, 0x21e4);
        re_mdio_write(sc, 0x14, 0x8244);
        re_mdio_write(sc, 0x14, 0x580f);
        re_mdio_write(sc, 0x14, 0xe582);
        re_mdio_write(sc, 0x14, 0x5bae);
        re_mdio_write(sc, 0x14, 0x0ebf);
        re_mdio_write(sc, 0x14, 0x8265);
        re_mdio_write(sc, 0x14, 0xe382);
        re_mdio_write(sc, 0x14, 0x44f7);
        re_mdio_write(sc, 0x14, 0x3de7);
        re_mdio_write(sc, 0x14, 0x8244);
        re_mdio_write(sc, 0x14, 0x0212);
        re_mdio_write(sc, 0x14, 0xf0ad);
        re_mdio_write(sc, 0x14, 0x223f);
        re_mdio_write(sc, 0x14, 0xe085);
        re_mdio_write(sc, 0x14, 0xf8e4);
        re_mdio_write(sc, 0x14, 0x8288);
        re_mdio_write(sc, 0x14, 0xe085);
        re_mdio_write(sc, 0x14, 0xfce4);
        re_mdio_write(sc, 0x14, 0x8289);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x440d);
        re_mdio_write(sc, 0x14, 0x0658);
        re_mdio_write(sc, 0x14, 0x01bf);
        re_mdio_write(sc, 0x14, 0x8272);
        re_mdio_write(sc, 0x14, 0x0215);
        re_mdio_write(sc, 0x14, 0x38bf);
        re_mdio_write(sc, 0x14, 0x8250);
        re_mdio_write(sc, 0x14, 0x0213);
        re_mdio_write(sc, 0x14, 0x06a0);
        re_mdio_write(sc, 0x14, 0x010f);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x44f6);
        re_mdio_write(sc, 0x14, 0x22e4);
        re_mdio_write(sc, 0x14, 0x8244);
        re_mdio_write(sc, 0x14, 0x580f);
        re_mdio_write(sc, 0x14, 0xe582);
        re_mdio_write(sc, 0x14, 0x5cae);
        re_mdio_write(sc, 0x14, 0x0ebf);
        re_mdio_write(sc, 0x14, 0x826c);
        re_mdio_write(sc, 0x14, 0xe382);
        re_mdio_write(sc, 0x14, 0x44f7);
        re_mdio_write(sc, 0x14, 0x3ee7);
        re_mdio_write(sc, 0x14, 0x8244);
        re_mdio_write(sc, 0x14, 0x0212);
        re_mdio_write(sc, 0x14, 0xf0ad);
        re_mdio_write(sc, 0x14, 0x233f);
        re_mdio_write(sc, 0x14, 0xe085);
        re_mdio_write(sc, 0x14, 0xf9e4);
        re_mdio_write(sc, 0x14, 0x8288);
        re_mdio_write(sc, 0x14, 0xe085);
        re_mdio_write(sc, 0x14, 0xfde4);
        re_mdio_write(sc, 0x14, 0x8289);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x440d);
        re_mdio_write(sc, 0x14, 0x0758);
        re_mdio_write(sc, 0x14, 0x01bf);
        re_mdio_write(sc, 0x14, 0x8279);
        re_mdio_write(sc, 0x14, 0x0215);
        re_mdio_write(sc, 0x14, 0x38bf);
        re_mdio_write(sc, 0x14, 0x8251);
        re_mdio_write(sc, 0x14, 0x0213);
        re_mdio_write(sc, 0x14, 0x06a0);
        re_mdio_write(sc, 0x14, 0x010f);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x44f6);
        re_mdio_write(sc, 0x14, 0x23e4);
        re_mdio_write(sc, 0x14, 0x8244);
        re_mdio_write(sc, 0x14, 0x580f);
        re_mdio_write(sc, 0x14, 0xe582);
        re_mdio_write(sc, 0x14, 0x5dae);
        re_mdio_write(sc, 0x14, 0x0ebf);
        re_mdio_write(sc, 0x14, 0x8273);
        re_mdio_write(sc, 0x14, 0xe382);
        re_mdio_write(sc, 0x14, 0x44f7);
        re_mdio_write(sc, 0x14, 0x3fe7);
        re_mdio_write(sc, 0x14, 0x8244);
        re_mdio_write(sc, 0x14, 0x0212);
        re_mdio_write(sc, 0x14, 0xf0ee);
        re_mdio_write(sc, 0x14, 0x8288);
        re_mdio_write(sc, 0x14, 0x10ee);
        re_mdio_write(sc, 0x14, 0x8289);
        re_mdio_write(sc, 0x14, 0x00af);
        re_mdio_write(sc, 0x14, 0x14aa);
        re_mdio_write(sc, 0x13, 0xb818);
        re_mdio_write(sc, 0x14, 0x13cf);
        re_mdio_write(sc, 0x13, 0xb81a);
        re_mdio_write(sc, 0x14, 0xfffd);
        re_mdio_write(sc, 0x13, 0xb81c);
        re_mdio_write(sc, 0x14, 0xfffd);
        re_mdio_write(sc, 0x13, 0xb81e);
        re_mdio_write(sc, 0x14, 0xfffd);
        re_mdio_write(sc, 0x13, 0xb832);
        re_mdio_write(sc, 0x14, 0x0001);

        re_mdio_write(sc, 0x1F, 0x0A43);
        re_mdio_write(sc, 0x13, 0x0000);
        re_mdio_write(sc, 0x14, 0x0000);
        re_mdio_write(sc, 0x1f, 0x0B82);
        PhyRegValue = re_mdio_read(sc, 0x17);
        PhyRegValue &= ~(BIT_0);
        re_mdio_write(sc, 0x17, PhyRegValue);
        re_mdio_write(sc, 0x1f, 0x0A43);
        re_mdio_write(sc, 0x13, 0x8028);
        re_mdio_write(sc, 0x14, 0x0000);

        re_clear_phy_mcu_patch_request(sc);
        if (sc->RequiredSecLanDonglePatch) {
                re_mdio_write(sc, 0x1F, 0x0A43);
                PhyRegValue = re_mdio_read(sc, 0x11);
                PhyRegValue &= ~(BIT_6);
                re_mdio_write(sc, 0x11, PhyRegValue);
        }
}

static void re_set_phy_mcu_8168h_3(struct re_softc *sc)
{
        u_int16_t PhyRegValue;

        re_set_phy_mcu_patch_request(sc);


        re_mdio_write(sc, 0x1f, 0x0A43);
        re_mdio_write(sc, 0x13, 0x8042);
        re_mdio_write(sc, 0x14, 0x3800);
        re_mdio_write(sc, 0x13, 0xB82E);
        re_mdio_write(sc, 0x14, 0x0001);


        re_mdio_write(sc, 0x1F, 0x0A43);
        re_mdio_write(sc, 0x13, 0xB820);
        re_mdio_write(sc, 0x14, 0x0090);
        re_mdio_write(sc, 0x13, 0xA016);
        re_mdio_write(sc, 0x14, 0x0000);
        re_mdio_write(sc, 0x13, 0xA012);
        re_mdio_write(sc, 0x14, 0x0000);
        re_mdio_write(sc, 0x13, 0xA014);
        re_mdio_write(sc, 0x14, 0x1800);
        re_mdio_write(sc, 0x14, 0x8010);
        re_mdio_write(sc, 0x14, 0x1800);
        re_mdio_write(sc, 0x14, 0x8014);
        re_mdio_write(sc, 0x14, 0x1800);
        re_mdio_write(sc, 0x14, 0x8022);
        re_mdio_write(sc, 0x14, 0x1800);
        re_mdio_write(sc, 0x14, 0x8022);
        re_mdio_write(sc, 0x14, 0x1800);
        re_mdio_write(sc, 0x14, 0x8022);
        re_mdio_write(sc, 0x14, 0x1800);
        re_mdio_write(sc, 0x14, 0x8022);
        re_mdio_write(sc, 0x14, 0x1800);
        re_mdio_write(sc, 0x14, 0x8022);
        re_mdio_write(sc, 0x14, 0x1800);
        re_mdio_write(sc, 0x14, 0x8022);
        re_mdio_write(sc, 0x14, 0x2b5d);
        re_mdio_write(sc, 0x14, 0x0c68);
        re_mdio_write(sc, 0x14, 0x1800);
        re_mdio_write(sc, 0x14, 0x0b3c);
        re_mdio_write(sc, 0x14, 0xc2bb);
        re_mdio_write(sc, 0x14, 0xd500);
        re_mdio_write(sc, 0x14, 0xc426);
        re_mdio_write(sc, 0x14, 0xd01d);
        re_mdio_write(sc, 0x14, 0xd1c3);
        re_mdio_write(sc, 0x14, 0x401c);
        re_mdio_write(sc, 0x14, 0xd501);
        re_mdio_write(sc, 0x14, 0xc2b3);
        re_mdio_write(sc, 0x14, 0xd500);
        re_mdio_write(sc, 0x14, 0xd00b);
        re_mdio_write(sc, 0x14, 0xd1c3);
        re_mdio_write(sc, 0x14, 0x401c);
        re_mdio_write(sc, 0x14, 0x1800);
        re_mdio_write(sc, 0x14, 0x0478);
        re_mdio_write(sc, 0x13, 0xA026);
        re_mdio_write(sc, 0x14, 0x0fff);
        re_mdio_write(sc, 0x13, 0xA024);
        re_mdio_write(sc, 0x14, 0x0fff);
        re_mdio_write(sc, 0x13, 0xA022);
        re_mdio_write(sc, 0x14, 0x0fff);
        re_mdio_write(sc, 0x13, 0xA020);
        re_mdio_write(sc, 0x14, 0x0fff);
        re_mdio_write(sc, 0x13, 0xA006);
        re_mdio_write(sc, 0x14, 0x0fff);
        re_mdio_write(sc, 0x13, 0xA004);
        re_mdio_write(sc, 0x14, 0x0fff);
        re_mdio_write(sc, 0x13, 0xA002);
        re_mdio_write(sc, 0x14, 0x0472);
        re_mdio_write(sc, 0x13, 0xA000);
        re_mdio_write(sc, 0x14, 0x0b3a);
        re_mdio_write(sc, 0x13, 0xA008);
        re_mdio_write(sc, 0x14, 0x0300);
        re_mdio_write(sc, 0x13, 0xB820);
        re_mdio_write(sc, 0x14, 0x0010);


        re_mdio_write(sc, 0x13, 0x83f3);
        re_mdio_write(sc, 0x14, 0xaf84);
        re_mdio_write(sc, 0x14, 0x0baf);
        re_mdio_write(sc, 0x14, 0x8466);
        re_mdio_write(sc, 0x14, 0xaf84);
        re_mdio_write(sc, 0x14, 0xcdaf);
        re_mdio_write(sc, 0x14, 0x873c);
        re_mdio_write(sc, 0x14, 0xaf87);
        re_mdio_write(sc, 0x14, 0x3faf);
        re_mdio_write(sc, 0x14, 0x8760);
        re_mdio_write(sc, 0x14, 0xaf87);
        re_mdio_write(sc, 0x14, 0x60af);
        re_mdio_write(sc, 0x14, 0x8760);
        re_mdio_write(sc, 0x14, 0xef79);
        re_mdio_write(sc, 0x14, 0xfb89);
        re_mdio_write(sc, 0x14, 0xe987);
        re_mdio_write(sc, 0x14, 0xffd7);
        re_mdio_write(sc, 0x14, 0x0017);
        re_mdio_write(sc, 0x14, 0xd400);
        re_mdio_write(sc, 0x14, 0x051c);
        re_mdio_write(sc, 0x14, 0x421a);
        re_mdio_write(sc, 0x14, 0x741b);
        re_mdio_write(sc, 0x14, 0x97e9);
        re_mdio_write(sc, 0x14, 0x87fe);
        re_mdio_write(sc, 0x14, 0xffef);
        re_mdio_write(sc, 0x14, 0x97e0);
        re_mdio_write(sc, 0x14, 0x82aa);
        re_mdio_write(sc, 0x14, 0xa000);
        re_mdio_write(sc, 0x14, 0x08ef);
        re_mdio_write(sc, 0x14, 0x46dc);
        re_mdio_write(sc, 0x14, 0x19dd);
        re_mdio_write(sc, 0x14, 0xaf1a);
        re_mdio_write(sc, 0x14, 0x37a0);
        re_mdio_write(sc, 0x14, 0x012d);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0xa7ac);
        re_mdio_write(sc, 0x14, 0x2013);
        re_mdio_write(sc, 0x14, 0xe087);
        re_mdio_write(sc, 0x14, 0xffe1);
        re_mdio_write(sc, 0x14, 0x87fe);
        re_mdio_write(sc, 0x14, 0xac27);
        re_mdio_write(sc, 0x14, 0x05a1);
        re_mdio_write(sc, 0x14, 0x0807);
        re_mdio_write(sc, 0x14, 0xae0f);
        re_mdio_write(sc, 0x14, 0xa107);
        re_mdio_write(sc, 0x14, 0x02ae);
        re_mdio_write(sc, 0x14, 0x0aef);
        re_mdio_write(sc, 0x14, 0x4619);
        re_mdio_write(sc, 0x14, 0x19dc);
        re_mdio_write(sc, 0x14, 0x19dd);
        re_mdio_write(sc, 0x14, 0xaf1a);
        re_mdio_write(sc, 0x14, 0x37d8);
        re_mdio_write(sc, 0x14, 0x19d9);
        re_mdio_write(sc, 0x14, 0x19dc);
        re_mdio_write(sc, 0x14, 0x19dd);
        re_mdio_write(sc, 0x14, 0xaf1a);
        re_mdio_write(sc, 0x14, 0x3719);
        re_mdio_write(sc, 0x14, 0x19ae);
        re_mdio_write(sc, 0x14, 0xcfbf);
        re_mdio_write(sc, 0x14, 0x878a);
        re_mdio_write(sc, 0x14, 0x0244);
        re_mdio_write(sc, 0x14, 0xdc3c);
        re_mdio_write(sc, 0x14, 0x0005);
        re_mdio_write(sc, 0x14, 0xaaf5);
        re_mdio_write(sc, 0x14, 0x0249);
        re_mdio_write(sc, 0x14, 0xcaef);
        re_mdio_write(sc, 0x14, 0x67d7);
        re_mdio_write(sc, 0x14, 0x0014);
        re_mdio_write(sc, 0x14, 0x0249);
        re_mdio_write(sc, 0x14, 0xe5ad);
        re_mdio_write(sc, 0x14, 0x50f7);
        re_mdio_write(sc, 0x14, 0xd400);
        re_mdio_write(sc, 0x14, 0x01bf);
        re_mdio_write(sc, 0x14, 0x46a7);
        re_mdio_write(sc, 0x14, 0x0244);
        re_mdio_write(sc, 0x14, 0x98bf);
        re_mdio_write(sc, 0x14, 0x465c);
        re_mdio_write(sc, 0x14, 0x024a);
        re_mdio_write(sc, 0x14, 0x5fd4);
        re_mdio_write(sc, 0x14, 0x0003);
        re_mdio_write(sc, 0x14, 0xbf87);
        re_mdio_write(sc, 0x14, 0x9c02);
        re_mdio_write(sc, 0x14, 0x4498);
        re_mdio_write(sc, 0x14, 0xbf87);
        re_mdio_write(sc, 0x14, 0x9902);
        re_mdio_write(sc, 0x14, 0x4a5f);
        re_mdio_write(sc, 0x14, 0xbf87);
        re_mdio_write(sc, 0x14, 0x8d02);
        re_mdio_write(sc, 0x14, 0x4a5f);
        re_mdio_write(sc, 0x14, 0xbf87);
        re_mdio_write(sc, 0x14, 0x9002);
        re_mdio_write(sc, 0x14, 0x44dc);
        re_mdio_write(sc, 0x14, 0xad28);
        re_mdio_write(sc, 0x14, 0xf7bf);
        re_mdio_write(sc, 0x14, 0x8796);
        re_mdio_write(sc, 0x14, 0x0244);
        re_mdio_write(sc, 0x14, 0xdcad);
        re_mdio_write(sc, 0x14, 0x28f7);
        re_mdio_write(sc, 0x14, 0xbf87);
        re_mdio_write(sc, 0x14, 0x9302);
        re_mdio_write(sc, 0x14, 0x4a5f);
        re_mdio_write(sc, 0x14, 0xbf87);
        re_mdio_write(sc, 0x14, 0x9302);
        re_mdio_write(sc, 0x14, 0x4a56);
        re_mdio_write(sc, 0x14, 0xbf46);
        re_mdio_write(sc, 0x14, 0x5c02);
        re_mdio_write(sc, 0x14, 0x4a56);
        re_mdio_write(sc, 0x14, 0xbf45);
        re_mdio_write(sc, 0x14, 0x21af);
        re_mdio_write(sc, 0x14, 0x020e);
        re_mdio_write(sc, 0x14, 0xee82);
        re_mdio_write(sc, 0x14, 0x5000);
        re_mdio_write(sc, 0x14, 0x0284);
        re_mdio_write(sc, 0x14, 0xdd02);
        re_mdio_write(sc, 0x14, 0x8521);
        re_mdio_write(sc, 0x14, 0x0285);
        re_mdio_write(sc, 0x14, 0x36af);
        re_mdio_write(sc, 0x14, 0x03d2);
        re_mdio_write(sc, 0x14, 0xf8f9);
        re_mdio_write(sc, 0x14, 0xfafb);
        re_mdio_write(sc, 0x14, 0xef59);
        re_mdio_write(sc, 0x14, 0xbf45);
        re_mdio_write(sc, 0x14, 0x3002);
        re_mdio_write(sc, 0x14, 0x44dc);
        re_mdio_write(sc, 0x14, 0x3c00);
        re_mdio_write(sc, 0x14, 0x03aa);
        re_mdio_write(sc, 0x14, 0x2cbf);
        re_mdio_write(sc, 0x14, 0x8790);
        re_mdio_write(sc, 0x14, 0x0244);
        re_mdio_write(sc, 0x14, 0xdcad);
        re_mdio_write(sc, 0x14, 0x2823);
        re_mdio_write(sc, 0x14, 0xbf87);
        re_mdio_write(sc, 0x14, 0x9602);
        re_mdio_write(sc, 0x14, 0x44dc);
        re_mdio_write(sc, 0x14, 0xad28);
        re_mdio_write(sc, 0x14, 0x1a02);
        re_mdio_write(sc, 0x14, 0x49ca);
        re_mdio_write(sc, 0x14, 0xef67);
        re_mdio_write(sc, 0x14, 0xd700);
        re_mdio_write(sc, 0x14, 0x0202);
        re_mdio_write(sc, 0x14, 0x49e5);
        re_mdio_write(sc, 0x14, 0xad50);
        re_mdio_write(sc, 0x14, 0xf7bf);
        re_mdio_write(sc, 0x14, 0x8793);
        re_mdio_write(sc, 0x14, 0x024a);
        re_mdio_write(sc, 0x14, 0x5fbf);
        re_mdio_write(sc, 0x14, 0x8793);
        re_mdio_write(sc, 0x14, 0x024a);
        re_mdio_write(sc, 0x14, 0x56ef);
        re_mdio_write(sc, 0x14, 0x95ff);
        re_mdio_write(sc, 0x14, 0xfefd);
        re_mdio_write(sc, 0x14, 0xfc04);
        re_mdio_write(sc, 0x14, 0xf8fa);
        re_mdio_write(sc, 0x14, 0xef69);
        re_mdio_write(sc, 0x14, 0xe080);
        re_mdio_write(sc, 0x14, 0x15ad);
        re_mdio_write(sc, 0x14, 0x2406);
        re_mdio_write(sc, 0x14, 0xbf87);
        re_mdio_write(sc, 0x14, 0x8702);
        re_mdio_write(sc, 0x14, 0x4a56);
        re_mdio_write(sc, 0x14, 0xef96);
        re_mdio_write(sc, 0x14, 0xfefc);
        re_mdio_write(sc, 0x14, 0x04f8);
        re_mdio_write(sc, 0x14, 0xe087);
        re_mdio_write(sc, 0x14, 0xf9e1);
        re_mdio_write(sc, 0x14, 0x87fa);
        re_mdio_write(sc, 0x14, 0x1b10);
        re_mdio_write(sc, 0x14, 0x9f1e);
        re_mdio_write(sc, 0x14, 0xee87);
        re_mdio_write(sc, 0x14, 0xf900);
        re_mdio_write(sc, 0x14, 0xe080);
        re_mdio_write(sc, 0x14, 0x15ac);
        re_mdio_write(sc, 0x14, 0x2606);
        re_mdio_write(sc, 0x14, 0xee87);
        re_mdio_write(sc, 0x14, 0xf700);
        re_mdio_write(sc, 0x14, 0xae12);
        re_mdio_write(sc, 0x14, 0x0286);
        re_mdio_write(sc, 0x14, 0x9d02);
        re_mdio_write(sc, 0x14, 0x8565);
        re_mdio_write(sc, 0x14, 0x0285);
        re_mdio_write(sc, 0x14, 0x9d02);
        re_mdio_write(sc, 0x14, 0x8660);
        re_mdio_write(sc, 0x14, 0xae04);
        re_mdio_write(sc, 0x14, 0x10e4);
        re_mdio_write(sc, 0x14, 0x87f9);
        re_mdio_write(sc, 0x14, 0xfc04);
        re_mdio_write(sc, 0x14, 0xf8f9);
        re_mdio_write(sc, 0x14, 0xfaef);
        re_mdio_write(sc, 0x14, 0x69fa);
        re_mdio_write(sc, 0x14, 0xbf45);
        re_mdio_write(sc, 0x14, 0x3002);
        re_mdio_write(sc, 0x14, 0x44dc);
        re_mdio_write(sc, 0x14, 0xa103);
        re_mdio_write(sc, 0x14, 0x22e0);
        re_mdio_write(sc, 0x14, 0x87eb);
        re_mdio_write(sc, 0x14, 0xe187);
        re_mdio_write(sc, 0x14, 0xecef);
        re_mdio_write(sc, 0x14, 0x64bf);
        re_mdio_write(sc, 0x14, 0x876f);
        re_mdio_write(sc, 0x14, 0x0244);
        re_mdio_write(sc, 0x14, 0xdc1b);
        re_mdio_write(sc, 0x14, 0x46aa);
        re_mdio_write(sc, 0x14, 0x0abf);
        re_mdio_write(sc, 0x14, 0x8772);
        re_mdio_write(sc, 0x14, 0x0244);
        re_mdio_write(sc, 0x14, 0xdc1b);
        re_mdio_write(sc, 0x14, 0x46ab);
        re_mdio_write(sc, 0x14, 0x06bf);
        re_mdio_write(sc, 0x14, 0x876c);
        re_mdio_write(sc, 0x14, 0x024a);
        re_mdio_write(sc, 0x14, 0x5ffe);
        re_mdio_write(sc, 0x14, 0xef96);
        re_mdio_write(sc, 0x14, 0xfefd);
        re_mdio_write(sc, 0x14, 0xfc04);
        re_mdio_write(sc, 0x14, 0xf8f9);
        re_mdio_write(sc, 0x14, 0xef59);
        re_mdio_write(sc, 0x14, 0xf9bf);
        re_mdio_write(sc, 0x14, 0x4530);
        re_mdio_write(sc, 0x14, 0x0244);
        re_mdio_write(sc, 0x14, 0xdca1);
        re_mdio_write(sc, 0x14, 0x0310);
        re_mdio_write(sc, 0x14, 0xe087);
        re_mdio_write(sc, 0x14, 0xf7ac);
        re_mdio_write(sc, 0x14, 0x2605);
        re_mdio_write(sc, 0x14, 0x0285);
        re_mdio_write(sc, 0x14, 0xc9ae);
        re_mdio_write(sc, 0x14, 0x0d02);
        re_mdio_write(sc, 0x14, 0x8613);
        re_mdio_write(sc, 0x14, 0xae08);
        re_mdio_write(sc, 0x14, 0xe287);
        re_mdio_write(sc, 0x14, 0xf7f6);
        re_mdio_write(sc, 0x14, 0x36e6);
        re_mdio_write(sc, 0x14, 0x87f7);
        re_mdio_write(sc, 0x14, 0xfdef);
        re_mdio_write(sc, 0x14, 0x95fd);
        re_mdio_write(sc, 0x14, 0xfc04);
        re_mdio_write(sc, 0x14, 0xf8f9);
        re_mdio_write(sc, 0x14, 0xfafb);
        re_mdio_write(sc, 0x14, 0xef79);
        re_mdio_write(sc, 0x14, 0xfbbf);
        re_mdio_write(sc, 0x14, 0x876f);
        re_mdio_write(sc, 0x14, 0x0244);
        re_mdio_write(sc, 0x14, 0xdcef);
        re_mdio_write(sc, 0x14, 0x64e2);
        re_mdio_write(sc, 0x14, 0x87e9);
        re_mdio_write(sc, 0x14, 0xe387);
        re_mdio_write(sc, 0x14, 0xea1b);
        re_mdio_write(sc, 0x14, 0x659e);
        re_mdio_write(sc, 0x14, 0x10e4);
        re_mdio_write(sc, 0x14, 0x87e9);
        re_mdio_write(sc, 0x14, 0xe587);
        re_mdio_write(sc, 0x14, 0xeae2);
        re_mdio_write(sc, 0x14, 0x87f7);
        re_mdio_write(sc, 0x14, 0xf636);
        re_mdio_write(sc, 0x14, 0xe687);
        re_mdio_write(sc, 0x14, 0xf7ae);
        re_mdio_write(sc, 0x14, 0x19e2);
        re_mdio_write(sc, 0x14, 0x87f7);
        re_mdio_write(sc, 0x14, 0xf736);
        re_mdio_write(sc, 0x14, 0xe687);
        re_mdio_write(sc, 0x14, 0xf700);
        re_mdio_write(sc, 0x14, 0x00ae);
        re_mdio_write(sc, 0x14, 0x0200);
        re_mdio_write(sc, 0x14, 0x0002);
        re_mdio_write(sc, 0x14, 0x49ca);
        re_mdio_write(sc, 0x14, 0xef57);
        re_mdio_write(sc, 0x14, 0xe687);
        re_mdio_write(sc, 0x14, 0xe7e7);
        re_mdio_write(sc, 0x14, 0x87e8);
        re_mdio_write(sc, 0x14, 0xffef);
        re_mdio_write(sc, 0x14, 0x97ff);
        re_mdio_write(sc, 0x14, 0xfefd);
        re_mdio_write(sc, 0x14, 0xfc04);
        re_mdio_write(sc, 0x14, 0xf8f9);
        re_mdio_write(sc, 0x14, 0xfafb);
        re_mdio_write(sc, 0x14, 0xef79);
        re_mdio_write(sc, 0x14, 0xfbe2);
        re_mdio_write(sc, 0x14, 0x87e7);
        re_mdio_write(sc, 0x14, 0xe387);
        re_mdio_write(sc, 0x14, 0xe8ef);
        re_mdio_write(sc, 0x14, 0x65e2);
        re_mdio_write(sc, 0x14, 0x87fb);
        re_mdio_write(sc, 0x14, 0xe387);
        re_mdio_write(sc, 0x14, 0xfcef);
        re_mdio_write(sc, 0x14, 0x7502);
        re_mdio_write(sc, 0x14, 0x49e5);
        re_mdio_write(sc, 0x14, 0xac50);
        re_mdio_write(sc, 0x14, 0x1abf);
        re_mdio_write(sc, 0x14, 0x876f);
        re_mdio_write(sc, 0x14, 0x0244);
        re_mdio_write(sc, 0x14, 0xdcef);
        re_mdio_write(sc, 0x14, 0x64e2);
        re_mdio_write(sc, 0x14, 0x87e9);
        re_mdio_write(sc, 0x14, 0xe387);
        re_mdio_write(sc, 0x14, 0xea1b);
        re_mdio_write(sc, 0x14, 0x659e);
        re_mdio_write(sc, 0x14, 0x16e4);
        re_mdio_write(sc, 0x14, 0x87e9);
        re_mdio_write(sc, 0x14, 0xe587);
        re_mdio_write(sc, 0x14, 0xeaae);
        re_mdio_write(sc, 0x14, 0x06bf);
        re_mdio_write(sc, 0x14, 0x876c);
        re_mdio_write(sc, 0x14, 0x024a);
        re_mdio_write(sc, 0x14, 0x5fe2);
        re_mdio_write(sc, 0x14, 0x87f7);
        re_mdio_write(sc, 0x14, 0xf636);
        re_mdio_write(sc, 0x14, 0xe687);
        re_mdio_write(sc, 0x14, 0xf7ff);
        re_mdio_write(sc, 0x14, 0xef97);
        re_mdio_write(sc, 0x14, 0xfffe);
        re_mdio_write(sc, 0x14, 0xfdfc);
        re_mdio_write(sc, 0x14, 0x04f8);
        re_mdio_write(sc, 0x14, 0xf9fa);
        re_mdio_write(sc, 0x14, 0xef69);
        re_mdio_write(sc, 0x14, 0xbf87);
        re_mdio_write(sc, 0x14, 0x6602);
        re_mdio_write(sc, 0x14, 0x44dc);
        re_mdio_write(sc, 0x14, 0xad28);
        re_mdio_write(sc, 0x14, 0x29bf);
        re_mdio_write(sc, 0x14, 0x8763);
        re_mdio_write(sc, 0x14, 0x0244);
        re_mdio_write(sc, 0x14, 0xdcef);
        re_mdio_write(sc, 0x14, 0x54bf);
        re_mdio_write(sc, 0x14, 0x8760);
        re_mdio_write(sc, 0x14, 0x0244);
        re_mdio_write(sc, 0x14, 0xdcac);
        re_mdio_write(sc, 0x14, 0x290d);
        re_mdio_write(sc, 0x14, 0xac28);
        re_mdio_write(sc, 0x14, 0x05a3);
        re_mdio_write(sc, 0x14, 0x020c);
        re_mdio_write(sc, 0x14, 0xae10);
        re_mdio_write(sc, 0x14, 0xa303);
        re_mdio_write(sc, 0x14, 0x07ae);
        re_mdio_write(sc, 0x14, 0x0ba3);
        re_mdio_write(sc, 0x14, 0x0402);
        re_mdio_write(sc, 0x14, 0xae06);
        re_mdio_write(sc, 0x14, 0xbf87);
        re_mdio_write(sc, 0x14, 0x6c02);
        re_mdio_write(sc, 0x14, 0x4a5f);
        re_mdio_write(sc, 0x14, 0xef96);
        re_mdio_write(sc, 0x14, 0xfefd);
        re_mdio_write(sc, 0x14, 0xfc04);
        re_mdio_write(sc, 0x14, 0xf8f9);
        re_mdio_write(sc, 0x14, 0xfafb);
        re_mdio_write(sc, 0x14, 0xef69);
        re_mdio_write(sc, 0x14, 0xfae0);
        re_mdio_write(sc, 0x14, 0x8015);
        re_mdio_write(sc, 0x14, 0xad25);
        re_mdio_write(sc, 0x14, 0x41d2);
        re_mdio_write(sc, 0x14, 0x0002);
        re_mdio_write(sc, 0x14, 0x86f3);
        re_mdio_write(sc, 0x14, 0xe087);
        re_mdio_write(sc, 0x14, 0xebe1);
        re_mdio_write(sc, 0x14, 0x87ec);
        re_mdio_write(sc, 0x14, 0x1b46);
        re_mdio_write(sc, 0x14, 0xab26);
        re_mdio_write(sc, 0x14, 0xd40b);
        re_mdio_write(sc, 0x14, 0xff1b);
        re_mdio_write(sc, 0x14, 0x46aa);
        re_mdio_write(sc, 0x14, 0x1fac);
        re_mdio_write(sc, 0x14, 0x3204);
        re_mdio_write(sc, 0x14, 0xef32);
        re_mdio_write(sc, 0x14, 0xae02);
        re_mdio_write(sc, 0x14, 0xd304);
        re_mdio_write(sc, 0x14, 0x0c31);
        re_mdio_write(sc, 0x14, 0xbf87);
        re_mdio_write(sc, 0x14, 0xeb1a);
        re_mdio_write(sc, 0x14, 0x93d8);
        re_mdio_write(sc, 0x14, 0x19d9);
        re_mdio_write(sc, 0x14, 0x1b46);
        re_mdio_write(sc, 0x14, 0xab0e);
        re_mdio_write(sc, 0x14, 0x19d8);
        re_mdio_write(sc, 0x14, 0x19d9);
        re_mdio_write(sc, 0x14, 0x1b46);
        re_mdio_write(sc, 0x14, 0xaa06);
        re_mdio_write(sc, 0x14, 0x12a2);
        re_mdio_write(sc, 0x14, 0x08c9);
        re_mdio_write(sc, 0x14, 0xae06);
        re_mdio_write(sc, 0x14, 0xbf87);
        re_mdio_write(sc, 0x14, 0x6902);
        re_mdio_write(sc, 0x14, 0x4a5f);
        re_mdio_write(sc, 0x14, 0xfeef);
        re_mdio_write(sc, 0x14, 0x96ff);
        re_mdio_write(sc, 0x14, 0xfefd);
        re_mdio_write(sc, 0x14, 0xfc04);
        re_mdio_write(sc, 0x14, 0xf8fb);
        re_mdio_write(sc, 0x14, 0xef79);
        re_mdio_write(sc, 0x14, 0xa200);
        re_mdio_write(sc, 0x14, 0x05bf);
        re_mdio_write(sc, 0x14, 0x876f);
        re_mdio_write(sc, 0x14, 0xae33);
        re_mdio_write(sc, 0x14, 0xa201);
        re_mdio_write(sc, 0x14, 0x05bf);
        re_mdio_write(sc, 0x14, 0x8772);
        re_mdio_write(sc, 0x14, 0xae2b);
        re_mdio_write(sc, 0x14, 0xa202);
        re_mdio_write(sc, 0x14, 0x05bf);
        re_mdio_write(sc, 0x14, 0x8775);
        re_mdio_write(sc, 0x14, 0xae23);
        re_mdio_write(sc, 0x14, 0xa203);
        re_mdio_write(sc, 0x14, 0x05bf);
        re_mdio_write(sc, 0x14, 0x8778);
        re_mdio_write(sc, 0x14, 0xae1b);
        re_mdio_write(sc, 0x14, 0xa204);
        re_mdio_write(sc, 0x14, 0x05bf);
        re_mdio_write(sc, 0x14, 0x877b);
        re_mdio_write(sc, 0x14, 0xae13);
        re_mdio_write(sc, 0x14, 0xa205);
        re_mdio_write(sc, 0x14, 0x05bf);
        re_mdio_write(sc, 0x14, 0x877e);
        re_mdio_write(sc, 0x14, 0xae0b);
        re_mdio_write(sc, 0x14, 0xa206);
        re_mdio_write(sc, 0x14, 0x05bf);
        re_mdio_write(sc, 0x14, 0x8781);
        re_mdio_write(sc, 0x14, 0xae03);
        re_mdio_write(sc, 0x14, 0xbf87);
        re_mdio_write(sc, 0x14, 0x8402);
        re_mdio_write(sc, 0x14, 0x44dc);
        re_mdio_write(sc, 0x14, 0xef64);
        re_mdio_write(sc, 0x14, 0xef97);
        re_mdio_write(sc, 0x14, 0xfffc);
        re_mdio_write(sc, 0x14, 0x04af);
        re_mdio_write(sc, 0x14, 0x00ed);
        re_mdio_write(sc, 0x14, 0x0220);
        re_mdio_write(sc, 0x14, 0xa5f8);
        re_mdio_write(sc, 0x14, 0xfaef);
        re_mdio_write(sc, 0x14, 0x69bf);
        re_mdio_write(sc, 0x14, 0x4554);
        re_mdio_write(sc, 0x14, 0x0244);
        re_mdio_write(sc, 0x14, 0xdce0);
        re_mdio_write(sc, 0x14, 0x87ff);
        re_mdio_write(sc, 0x14, 0x1f01);
        re_mdio_write(sc, 0x14, 0x9e06);
        re_mdio_write(sc, 0x14, 0xe587);
        re_mdio_write(sc, 0x14, 0xff02);
        re_mdio_write(sc, 0x14, 0x4b05);
        re_mdio_write(sc, 0x14, 0xef96);
        re_mdio_write(sc, 0x14, 0xfefc);
        re_mdio_write(sc, 0x14, 0xaf03);
        re_mdio_write(sc, 0x14, 0x8c54);
        re_mdio_write(sc, 0x14, 0xa434);
        re_mdio_write(sc, 0x14, 0x74a6);
        re_mdio_write(sc, 0x14, 0x0022);
        re_mdio_write(sc, 0x14, 0xa434);
        re_mdio_write(sc, 0x14, 0x11b8);
        re_mdio_write(sc, 0x14, 0x4222);
        re_mdio_write(sc, 0x14, 0xb842);
        re_mdio_write(sc, 0x14, 0xf0a2);
        re_mdio_write(sc, 0x14, 0x00f0);
        re_mdio_write(sc, 0x14, 0xa202);
        re_mdio_write(sc, 0x14, 0xf0a2);
        re_mdio_write(sc, 0x14, 0x04f0);
        re_mdio_write(sc, 0x14, 0xa206);
        re_mdio_write(sc, 0x14, 0xf0a2);
        re_mdio_write(sc, 0x14, 0x08f0);
        re_mdio_write(sc, 0x14, 0xa20a);
        re_mdio_write(sc, 0x14, 0xf0a2);
        re_mdio_write(sc, 0x14, 0x0cf0);
        re_mdio_write(sc, 0x14, 0xa20e);
        re_mdio_write(sc, 0x14, 0x55b8);
        re_mdio_write(sc, 0x14, 0x20d9);
        re_mdio_write(sc, 0x14, 0xc608);
        re_mdio_write(sc, 0x14, 0xaac4);
        re_mdio_write(sc, 0x14, 0x3000);
        re_mdio_write(sc, 0x14, 0xc614);
        re_mdio_write(sc, 0x14, 0x33c4);
        re_mdio_write(sc, 0x14, 0x1a88);
        re_mdio_write(sc, 0x14, 0xc42e);
        re_mdio_write(sc, 0x14, 0x22c4);
        re_mdio_write(sc, 0x14, 0x2e54);
        re_mdio_write(sc, 0x14, 0xc41a);
        re_mdio_write(sc, 0x13, 0xb818);
        re_mdio_write(sc, 0x14, 0x1a01);
        re_mdio_write(sc, 0x13, 0xb81a);
        re_mdio_write(sc, 0x14, 0x020b);
        re_mdio_write(sc, 0x13, 0xb81c);
        re_mdio_write(sc, 0x14, 0x03ce);
        re_mdio_write(sc, 0x13, 0xb81e);
        re_mdio_write(sc, 0x14, 0x00e7);
        re_mdio_write(sc, 0x13, 0xb846);
        re_mdio_write(sc, 0x14, 0x0389);
        re_mdio_write(sc, 0x13, 0xb848);
        re_mdio_write(sc, 0x14, 0xffff);
        re_mdio_write(sc, 0x13, 0xb84a);
        re_mdio_write(sc, 0x14, 0xffff);
        re_mdio_write(sc, 0x13, 0xb84c);
        re_mdio_write(sc, 0x14, 0xffff);
        re_mdio_write(sc, 0x13, 0xb832);
        re_mdio_write(sc, 0x14, 0x001f);


        re_mdio_write(sc, 0x1F, 0x0A43);
        re_mdio_write(sc, 0x13, 0x0000);
        re_mdio_write(sc, 0x14, 0x0000);
        re_mdio_write(sc, 0x1f, 0x0B82);
        PhyRegValue = re_mdio_read(sc, 0x17);
        PhyRegValue &= ~(BIT_0);
        re_mdio_write(sc, 0x17, PhyRegValue);
        re_mdio_write(sc, 0x1f, 0x0A43);
        re_mdio_write(sc, 0x13, 0x8042);
        re_mdio_write(sc, 0x14, 0x0000);

        re_clear_phy_mcu_patch_request(sc);

        if (sc->RequiredSecLanDonglePatch) {
                re_mdio_write(sc, 0x1F, 0x0A43);
                PhyRegValue = re_mdio_read(sc, 0x11);
                PhyRegValue &= ~(BIT_6);
                re_mdio_write(sc, 0x11, PhyRegValue);
        }
}

static void re_set_phy_mcu_8168ep_1(struct re_softc *sc)
{
        u_int16_t PhyRegValue;

        re_set_phy_mcu_patch_request(sc);

        re_mdio_write(sc,0x1f, 0x0A43);
        re_mdio_write(sc,0x13, 0x8146);
        re_mdio_write(sc,0x14, 0x2700);
        re_mdio_write(sc,0x13, 0xB82E);
        re_mdio_write(sc,0x14, 0x0001);


        re_mdio_write(sc, 0x1F, 0x0A43);
        re_mdio_write(sc, 0x13, 0xb820);
        re_mdio_write(sc, 0x14, 0x0090);
        re_mdio_write(sc, 0x13, 0xa012);
        re_mdio_write(sc, 0x14, 0x0000);
        re_mdio_write(sc, 0x13, 0xa014);
        re_mdio_write(sc, 0x14, 0x2c04);
        re_mdio_write(sc, 0x14, 0x2c1b);
        re_mdio_write(sc, 0x14, 0x2c65);
        re_mdio_write(sc, 0x14, 0x2d14);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x4092);
        re_mdio_write(sc, 0x14, 0xba04);
        re_mdio_write(sc, 0x14, 0x3084);
        re_mdio_write(sc, 0x14, 0x1d04);
        re_mdio_write(sc, 0x14, 0x1cdd);
        re_mdio_write(sc, 0x14, 0x1ce8);
        re_mdio_write(sc, 0x14, 0xaeff);
        re_mdio_write(sc, 0x14, 0xaf02);
        re_mdio_write(sc, 0x14, 0x8f02);
        re_mdio_write(sc, 0x14, 0x8eff);
        re_mdio_write(sc, 0x14, 0xce01);
        re_mdio_write(sc, 0x14, 0xe070);
        re_mdio_write(sc, 0x14, 0x0f00);
        re_mdio_write(sc, 0x14, 0xaf01);
        re_mdio_write(sc, 0x14, 0x8f01);
        re_mdio_write(sc, 0x14, 0xd712);
        re_mdio_write(sc, 0x14, 0x5fe8);
        re_mdio_write(sc, 0x14, 0xaf02);
        re_mdio_write(sc, 0x14, 0x8f02);
        re_mdio_write(sc, 0x14, 0x8e01);
        re_mdio_write(sc, 0x14, 0x1cf2);
        re_mdio_write(sc, 0x14, 0x2825);
        re_mdio_write(sc, 0x14, 0xd05a);
        re_mdio_write(sc, 0x14, 0xd19a);
        re_mdio_write(sc, 0x14, 0xd709);
        re_mdio_write(sc, 0x14, 0x608f);
        re_mdio_write(sc, 0x14, 0xd06b);
        re_mdio_write(sc, 0x14, 0xd18a);
        re_mdio_write(sc, 0x14, 0x2c25);
        re_mdio_write(sc, 0x14, 0xd0be);
        re_mdio_write(sc, 0x14, 0xd188);
        re_mdio_write(sc, 0x14, 0x2c25);
        re_mdio_write(sc, 0x14, 0xd708);
        re_mdio_write(sc, 0x14, 0x4072);
        re_mdio_write(sc, 0x14, 0xc104);
        re_mdio_write(sc, 0x14, 0x2c37);
        re_mdio_write(sc, 0x14, 0x4076);
        re_mdio_write(sc, 0x14, 0xc110);
        re_mdio_write(sc, 0x14, 0x2c37);
        re_mdio_write(sc, 0x14, 0x4071);
        re_mdio_write(sc, 0x14, 0xc102);
        re_mdio_write(sc, 0x14, 0x2c37);
        re_mdio_write(sc, 0x14, 0x4070);
        re_mdio_write(sc, 0x14, 0xc101);
        re_mdio_write(sc, 0x14, 0x2c37);
        re_mdio_write(sc, 0x14, 0x1786);
        re_mdio_write(sc, 0x14, 0xd709);
        re_mdio_write(sc, 0x14, 0x3390);
        re_mdio_write(sc, 0x14, 0x5c32);
        re_mdio_write(sc, 0x14, 0x2c47);
        re_mdio_write(sc, 0x14, 0x1786);
        re_mdio_write(sc, 0x14, 0xd708);
        re_mdio_write(sc, 0x14, 0x6193);
        re_mdio_write(sc, 0x14, 0xd709);
        re_mdio_write(sc, 0x14, 0x5f9d);
        re_mdio_write(sc, 0x14, 0x408b);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x6042);
        re_mdio_write(sc, 0x14, 0xb401);
        re_mdio_write(sc, 0x14, 0x1786);
        re_mdio_write(sc, 0x14, 0xd708);
        re_mdio_write(sc, 0x14, 0x6073);
        re_mdio_write(sc, 0x14, 0x5fbc);
        re_mdio_write(sc, 0x14, 0x2c46);
        re_mdio_write(sc, 0x14, 0x26fe);
        re_mdio_write(sc, 0x14, 0xb280);
        re_mdio_write(sc, 0x14, 0xa841);
        re_mdio_write(sc, 0x14, 0x94e0);
        re_mdio_write(sc, 0x14, 0x8710);
        re_mdio_write(sc, 0x14, 0xd709);
        re_mdio_write(sc, 0x14, 0x42ec);
        re_mdio_write(sc, 0x14, 0x606d);
        re_mdio_write(sc, 0x14, 0xd207);
        re_mdio_write(sc, 0x14, 0x2c50);
        re_mdio_write(sc, 0x14, 0xd203);
        re_mdio_write(sc, 0x14, 0x33ff);
        re_mdio_write(sc, 0x14, 0x5647);
        re_mdio_write(sc, 0x14, 0x3275);
        re_mdio_write(sc, 0x14, 0x7c57);
        re_mdio_write(sc, 0x14, 0xb240);
        re_mdio_write(sc, 0x14, 0xb402);
        re_mdio_write(sc, 0x14, 0x2647);
        re_mdio_write(sc, 0x14, 0x6096);
        re_mdio_write(sc, 0x14, 0xb240);
        re_mdio_write(sc, 0x14, 0xb406);
        re_mdio_write(sc, 0x14, 0x2647);
        re_mdio_write(sc, 0x14, 0x31d7);
        re_mdio_write(sc, 0x14, 0x7c60);
        re_mdio_write(sc, 0x14, 0xb240);
        re_mdio_write(sc, 0x14, 0xb40e);
        re_mdio_write(sc, 0x14, 0x2647);
        re_mdio_write(sc, 0x14, 0xb410);
        re_mdio_write(sc, 0x14, 0x8802);
        re_mdio_write(sc, 0x14, 0xb240);
        re_mdio_write(sc, 0x14, 0x940e);
        re_mdio_write(sc, 0x14, 0x2647);
        re_mdio_write(sc, 0x14, 0xba04);
        re_mdio_write(sc, 0x14, 0x1cdd);
        re_mdio_write(sc, 0x14, 0xa902);
        re_mdio_write(sc, 0x14, 0xd711);
        re_mdio_write(sc, 0x14, 0x4045);
        re_mdio_write(sc, 0x14, 0xa980);
        re_mdio_write(sc, 0x14, 0x3003);
        re_mdio_write(sc, 0x14, 0x5a19);
        re_mdio_write(sc, 0x14, 0xa540);
        re_mdio_write(sc, 0x14, 0xa601);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x4043);
        re_mdio_write(sc, 0x14, 0xa910);
        re_mdio_write(sc, 0x14, 0xd711);
        re_mdio_write(sc, 0x14, 0x60a0);
        re_mdio_write(sc, 0x14, 0xca33);
        re_mdio_write(sc, 0x14, 0xcb33);
        re_mdio_write(sc, 0x14, 0xa941);
        re_mdio_write(sc, 0x14, 0x2c7b);
        re_mdio_write(sc, 0x14, 0xcaff);
        re_mdio_write(sc, 0x14, 0xcbff);
        re_mdio_write(sc, 0x14, 0xa921);
        re_mdio_write(sc, 0x14, 0xce02);
        re_mdio_write(sc, 0x14, 0xe070);
        re_mdio_write(sc, 0x14, 0x0f10);
        re_mdio_write(sc, 0x14, 0xaf01);
        re_mdio_write(sc, 0x14, 0x8f01);
        re_mdio_write(sc, 0x14, 0x1791);
        re_mdio_write(sc, 0x14, 0x8e02);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x41a3);
        re_mdio_write(sc, 0x14, 0xa140);
        re_mdio_write(sc, 0x14, 0xa220);
        re_mdio_write(sc, 0x14, 0xce10);
        re_mdio_write(sc, 0x14, 0xe070);
        re_mdio_write(sc, 0x14, 0x0f40);
        re_mdio_write(sc, 0x14, 0xaf01);
        re_mdio_write(sc, 0x14, 0x8f01);
        re_mdio_write(sc, 0x14, 0x1791);
        re_mdio_write(sc, 0x14, 0x8e10);
        re_mdio_write(sc, 0x14, 0x8140);
        re_mdio_write(sc, 0x14, 0x8220);
        re_mdio_write(sc, 0x14, 0xa301);
        re_mdio_write(sc, 0x14, 0x17b2);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x609c);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x7fa4);
        re_mdio_write(sc, 0x14, 0x2cdb);
        re_mdio_write(sc, 0x14, 0x1cf0);
        re_mdio_write(sc, 0x14, 0xce04);
        re_mdio_write(sc, 0x14, 0xe070);
        re_mdio_write(sc, 0x14, 0x0f20);
        re_mdio_write(sc, 0x14, 0xaf01);
        re_mdio_write(sc, 0x14, 0x8f01);
        re_mdio_write(sc, 0x14, 0x1791);
        re_mdio_write(sc, 0x14, 0x8e04);
        re_mdio_write(sc, 0x14, 0x6044);
        re_mdio_write(sc, 0x14, 0x2cdb);
        re_mdio_write(sc, 0x14, 0xa520);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x4043);
        re_mdio_write(sc, 0x14, 0x2cc8);
        re_mdio_write(sc, 0x14, 0xe00f);
        re_mdio_write(sc, 0x14, 0x0501);
        re_mdio_write(sc, 0x14, 0x1cf6);
        re_mdio_write(sc, 0x14, 0xb801);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x4060);
        re_mdio_write(sc, 0x14, 0x7fc4);
        re_mdio_write(sc, 0x14, 0x2cdb);
        re_mdio_write(sc, 0x14, 0x1cfc);
        re_mdio_write(sc, 0x14, 0xe00f);
        re_mdio_write(sc, 0x14, 0x0502);
        re_mdio_write(sc, 0x14, 0x1cf6);
        re_mdio_write(sc, 0x14, 0xb802);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x4061);
        re_mdio_write(sc, 0x14, 0x7fc4);
        re_mdio_write(sc, 0x14, 0x2cdb);
        re_mdio_write(sc, 0x14, 0x1cfc);
        re_mdio_write(sc, 0x14, 0xe00f);
        re_mdio_write(sc, 0x14, 0x0504);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x6099);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x7fa4);
        re_mdio_write(sc, 0x14, 0x2cdb);
        re_mdio_write(sc, 0x14, 0xc17f);
        re_mdio_write(sc, 0x14, 0xc200);
        re_mdio_write(sc, 0x14, 0xc43f);
        re_mdio_write(sc, 0x14, 0xcc03);
        re_mdio_write(sc, 0x14, 0xa701);
        re_mdio_write(sc, 0x14, 0xa510);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x4018);
        re_mdio_write(sc, 0x14, 0x9910);
        re_mdio_write(sc, 0x14, 0x8510);
        re_mdio_write(sc, 0x14, 0x28a1);
        re_mdio_write(sc, 0x14, 0xe00f);
        re_mdio_write(sc, 0x14, 0x0504);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x6099);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x7fa4);
        re_mdio_write(sc, 0x14, 0x2cdb);
        re_mdio_write(sc, 0x14, 0xa608);
        re_mdio_write(sc, 0x14, 0xc17d);
        re_mdio_write(sc, 0x14, 0xc200);
        re_mdio_write(sc, 0x14, 0xc43f);
        re_mdio_write(sc, 0x14, 0xcc03);
        re_mdio_write(sc, 0x14, 0xa701);
        re_mdio_write(sc, 0x14, 0xa510);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x4018);
        re_mdio_write(sc, 0x14, 0x9910);
        re_mdio_write(sc, 0x14, 0x8510);
        re_mdio_write(sc, 0x14, 0x298e);
        re_mdio_write(sc, 0x14, 0x17bd);
        re_mdio_write(sc, 0x14, 0x2815);
        re_mdio_write(sc, 0x14, 0xc000);
        re_mdio_write(sc, 0x14, 0xc100);
        re_mdio_write(sc, 0x14, 0xc200);
        re_mdio_write(sc, 0x14, 0xc300);
        re_mdio_write(sc, 0x14, 0xc400);
        re_mdio_write(sc, 0x14, 0xc500);
        re_mdio_write(sc, 0x14, 0xc600);
        re_mdio_write(sc, 0x14, 0xc7c1);
        re_mdio_write(sc, 0x14, 0xc800);
        re_mdio_write(sc, 0x14, 0xcc00);
        re_mdio_write(sc, 0x14, 0x0800);
        re_mdio_write(sc, 0x14, 0xca0f);
        re_mdio_write(sc, 0x14, 0xcbff);
        re_mdio_write(sc, 0x14, 0xa901);
        re_mdio_write(sc, 0x14, 0x8902);
        re_mdio_write(sc, 0x14, 0xc900);
        re_mdio_write(sc, 0x14, 0xca00);
        re_mdio_write(sc, 0x14, 0xcb00);
        re_mdio_write(sc, 0x14, 0x0800);
        re_mdio_write(sc, 0x14, 0xb804);
        re_mdio_write(sc, 0x14, 0x0800);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x6044);
        re_mdio_write(sc, 0x14, 0x9804);
        re_mdio_write(sc, 0x14, 0x0800);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x6099);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x7fa4);
        re_mdio_write(sc, 0x14, 0x2cdb);
        re_mdio_write(sc, 0x14, 0x0800);
        re_mdio_write(sc, 0x14, 0xa510);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x6098);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x7fa4);
        re_mdio_write(sc, 0x14, 0x2cdb);
        re_mdio_write(sc, 0x14, 0x8510);
        re_mdio_write(sc, 0x14, 0x0800);
        re_mdio_write(sc, 0x14, 0xd711);
        re_mdio_write(sc, 0x14, 0x3003);
        re_mdio_write(sc, 0x14, 0x1d08);
        re_mdio_write(sc, 0x14, 0x2d12);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x60be);
        re_mdio_write(sc, 0x14, 0xe060);
        re_mdio_write(sc, 0x14, 0x0920);
        re_mdio_write(sc, 0x14, 0x1cdd);
        re_mdio_write(sc, 0x14, 0x2c90);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x3063);
        re_mdio_write(sc, 0x14, 0x19b0);
        re_mdio_write(sc, 0x14, 0x28d5);
        re_mdio_write(sc, 0x14, 0x1cdd);
        re_mdio_write(sc, 0x14, 0x2a25);
        re_mdio_write(sc, 0x14, 0xa802);
        re_mdio_write(sc, 0x14, 0xa303);
        re_mdio_write(sc, 0x14, 0x843f);
        re_mdio_write(sc, 0x14, 0x81ff);
        re_mdio_write(sc, 0x14, 0x8208);
        re_mdio_write(sc, 0x14, 0xa201);
        re_mdio_write(sc, 0x14, 0xc001);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x30a0);
        re_mdio_write(sc, 0x14, 0x0d23);
        re_mdio_write(sc, 0x14, 0x30a0);
        re_mdio_write(sc, 0x14, 0x3d1a);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x7f4c);
        re_mdio_write(sc, 0x14, 0x2b1e);
        re_mdio_write(sc, 0x14, 0xe003);
        re_mdio_write(sc, 0x14, 0x0202);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x6090);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x7fac);
        re_mdio_write(sc, 0x14, 0x2b1e);
        re_mdio_write(sc, 0x14, 0xa20c);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x6091);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x7fac);
        re_mdio_write(sc, 0x14, 0x2b1e);
        re_mdio_write(sc, 0x14, 0x820e);
        re_mdio_write(sc, 0x14, 0xa3e0);
        re_mdio_write(sc, 0x14, 0xa520);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x609d);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x7fac);
        re_mdio_write(sc, 0x14, 0x2b1e);
        re_mdio_write(sc, 0x14, 0x8520);
        re_mdio_write(sc, 0x14, 0x6703);
        re_mdio_write(sc, 0x14, 0x2d3b);
        re_mdio_write(sc, 0x14, 0xa13e);
        re_mdio_write(sc, 0x14, 0xc001);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x4000);
        re_mdio_write(sc, 0x14, 0x6046);
        re_mdio_write(sc, 0x14, 0x2d14);
        re_mdio_write(sc, 0x14, 0xa43f);
        re_mdio_write(sc, 0x14, 0xa101);
        re_mdio_write(sc, 0x14, 0xc020);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x3121);
        re_mdio_write(sc, 0x14, 0x0d4c);
        re_mdio_write(sc, 0x14, 0x30c0);
        re_mdio_write(sc, 0x14, 0x3d14);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x7f4c);
        re_mdio_write(sc, 0x14, 0x2b1e);
        re_mdio_write(sc, 0x14, 0xa540);
        re_mdio_write(sc, 0x14, 0xc001);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x4001);
        re_mdio_write(sc, 0x14, 0xe00f);
        re_mdio_write(sc, 0x14, 0x0501);
        re_mdio_write(sc, 0x14, 0x1db3);
        re_mdio_write(sc, 0x14, 0xc1c4);
        re_mdio_write(sc, 0x14, 0xa268);
        re_mdio_write(sc, 0x14, 0xa303);
        re_mdio_write(sc, 0x14, 0x8420);
        re_mdio_write(sc, 0x14, 0xe00f);
        re_mdio_write(sc, 0x14, 0x0502);
        re_mdio_write(sc, 0x14, 0x1db3);
        re_mdio_write(sc, 0x14, 0xc002);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x4000);
        re_mdio_write(sc, 0x14, 0x8208);
        re_mdio_write(sc, 0x14, 0x8410);
        re_mdio_write(sc, 0x14, 0xa121);
        re_mdio_write(sc, 0x14, 0xc002);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x4000);
        re_mdio_write(sc, 0x14, 0x8120);
        re_mdio_write(sc, 0x14, 0x8180);
        re_mdio_write(sc, 0x14, 0x1d9e);
        re_mdio_write(sc, 0x14, 0xa180);
        re_mdio_write(sc, 0x14, 0xa13a);
        re_mdio_write(sc, 0x14, 0x8240);
        re_mdio_write(sc, 0x14, 0xa430);
        re_mdio_write(sc, 0x14, 0xc010);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x30e1);
        re_mdio_write(sc, 0x14, 0x0b24);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x7f8c);
        re_mdio_write(sc, 0x14, 0x2b1e);
        re_mdio_write(sc, 0x14, 0xa480);
        re_mdio_write(sc, 0x14, 0xa230);
        re_mdio_write(sc, 0x14, 0xa303);
        re_mdio_write(sc, 0x14, 0xc001);
        re_mdio_write(sc, 0x14, 0xd70c);
        re_mdio_write(sc, 0x14, 0x4124);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x6120);
        re_mdio_write(sc, 0x14, 0xd711);
        re_mdio_write(sc, 0x14, 0x3128);
        re_mdio_write(sc, 0x14, 0x3d7d);
        re_mdio_write(sc, 0x14, 0x2d77);
        re_mdio_write(sc, 0x14, 0xa801);
        re_mdio_write(sc, 0x14, 0x2d73);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x4000);
        re_mdio_write(sc, 0x14, 0xe018);
        re_mdio_write(sc, 0x14, 0x0208);
        re_mdio_write(sc, 0x14, 0xa1f8);
        re_mdio_write(sc, 0x14, 0x8480);
        re_mdio_write(sc, 0x14, 0xc004);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x4000);
        re_mdio_write(sc, 0x14, 0x6046);
        re_mdio_write(sc, 0x14, 0x2d14);
        re_mdio_write(sc, 0x14, 0xa43f);
        re_mdio_write(sc, 0x14, 0xa105);
        re_mdio_write(sc, 0x14, 0x8228);
        re_mdio_write(sc, 0x14, 0xc004);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x4000);
        re_mdio_write(sc, 0x14, 0x81bc);
        re_mdio_write(sc, 0x14, 0xa220);
        re_mdio_write(sc, 0x14, 0x1d9e);
        re_mdio_write(sc, 0x14, 0x8220);
        re_mdio_write(sc, 0x14, 0xa1bc);
        re_mdio_write(sc, 0x14, 0xc040);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x30e1);
        re_mdio_write(sc, 0x14, 0x0b24);
        re_mdio_write(sc, 0x14, 0x30e1);
        re_mdio_write(sc, 0x14, 0x3d14);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x7f4c);
        re_mdio_write(sc, 0x14, 0x2b1e);
        re_mdio_write(sc, 0x14, 0xa802);
        re_mdio_write(sc, 0x14, 0xd70c);
        re_mdio_write(sc, 0x14, 0x4244);
        re_mdio_write(sc, 0x14, 0xa301);
        re_mdio_write(sc, 0x14, 0xc004);
        re_mdio_write(sc, 0x14, 0xd711);
        re_mdio_write(sc, 0x14, 0x3128);
        re_mdio_write(sc, 0x14, 0x3dac);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x5f80);
        re_mdio_write(sc, 0x14, 0xd711);
        re_mdio_write(sc, 0x14, 0x3109);
        re_mdio_write(sc, 0x14, 0x3dae);
        re_mdio_write(sc, 0x14, 0x2db2);
        re_mdio_write(sc, 0x14, 0xa801);
        re_mdio_write(sc, 0x14, 0x2da1);
        re_mdio_write(sc, 0x14, 0xa802);
        re_mdio_write(sc, 0x14, 0xc004);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x4000);
        re_mdio_write(sc, 0x14, 0x0800);
        re_mdio_write(sc, 0x14, 0xa510);
        re_mdio_write(sc, 0x14, 0xd710);
        re_mdio_write(sc, 0x14, 0x609a);
        re_mdio_write(sc, 0x14, 0xd71e);
        re_mdio_write(sc, 0x14, 0x7fac);
        re_mdio_write(sc, 0x14, 0x2b1e);
        re_mdio_write(sc, 0x14, 0x8510);
        re_mdio_write(sc, 0x14, 0x0800);
        re_mdio_write(sc, 0x13, 0xa01a);
        re_mdio_write(sc, 0x14, 0x0000);
        re_mdio_write(sc, 0x13, 0xa006);
        re_mdio_write(sc, 0x14, 0x0b3e);
        re_mdio_write(sc, 0x13, 0xa004);
        re_mdio_write(sc, 0x14, 0x0828);
        re_mdio_write(sc, 0x13, 0xa002);
        re_mdio_write(sc, 0x14, 0x06dd);
        re_mdio_write(sc, 0x13, 0xa000);
        re_mdio_write(sc, 0x14, 0xf815);
        re_mdio_write(sc, 0x13, 0xb820);
        re_mdio_write(sc, 0x14, 0x0010);


        re_mdio_write(sc,0x1F, 0x0A43);
        re_mdio_write(sc, 0x13, 0x83b0);
        re_mdio_write(sc, 0x14, 0xaf83);
        re_mdio_write(sc, 0x14, 0xbcaf);
        re_mdio_write(sc, 0x14, 0x83c8);
        re_mdio_write(sc, 0x14, 0xaf83);
        re_mdio_write(sc, 0x14, 0xddaf);
        re_mdio_write(sc, 0x14, 0x83e0);
        re_mdio_write(sc, 0x14, 0x0204);
        re_mdio_write(sc, 0x14, 0xa102);
        re_mdio_write(sc, 0x14, 0x09b4);
        re_mdio_write(sc, 0x14, 0x0284);
        re_mdio_write(sc, 0x14, 0x62af);
        re_mdio_write(sc, 0x14, 0x02ec);
        re_mdio_write(sc, 0x14, 0xad20);
        re_mdio_write(sc, 0x14, 0x0302);
        re_mdio_write(sc, 0x14, 0x867d);
        re_mdio_write(sc, 0x14, 0xad21);
        re_mdio_write(sc, 0x14, 0x0302);
        re_mdio_write(sc, 0x14, 0x85ca);
        re_mdio_write(sc, 0x14, 0xad22);
        re_mdio_write(sc, 0x14, 0x0302);
        re_mdio_write(sc, 0x14, 0x1bce);
        re_mdio_write(sc, 0x14, 0xaf18);
        re_mdio_write(sc, 0x14, 0x11af);
        re_mdio_write(sc, 0x14, 0x1811);
        re_mdio_write(sc, 0x14, 0x0106);
        re_mdio_write(sc, 0x14, 0xe081);
        re_mdio_write(sc, 0x14, 0x48af);
        re_mdio_write(sc, 0x14, 0x3b1f);
        re_mdio_write(sc, 0x14, 0xf8f9);
        re_mdio_write(sc, 0x14, 0xfaef);
        re_mdio_write(sc, 0x14, 0x69ee);
        re_mdio_write(sc, 0x14, 0x8010);
        re_mdio_write(sc, 0x14, 0xf7d1);
        re_mdio_write(sc, 0x14, 0x04bf);
        re_mdio_write(sc, 0x14, 0x8776);
        re_mdio_write(sc, 0x14, 0x0241);
        re_mdio_write(sc, 0x14, 0x0a02);
        re_mdio_write(sc, 0x14, 0x8704);
        re_mdio_write(sc, 0x14, 0xbf87);
        re_mdio_write(sc, 0x14, 0x4fd7);
        re_mdio_write(sc, 0x14, 0xb822);
        re_mdio_write(sc, 0x14, 0xd00c);
        re_mdio_write(sc, 0x14, 0x0241);
        re_mdio_write(sc, 0x14, 0x03ee);
        re_mdio_write(sc, 0x14, 0x80cd);
        re_mdio_write(sc, 0x14, 0xa0ee);
        re_mdio_write(sc, 0x14, 0x80ce);
        re_mdio_write(sc, 0x14, 0x8bee);
        re_mdio_write(sc, 0x14, 0x80d1);
        re_mdio_write(sc, 0x14, 0xf5ee);
        re_mdio_write(sc, 0x14, 0x80d2);
        re_mdio_write(sc, 0x14, 0xa9ee);
        re_mdio_write(sc, 0x14, 0x80d3);
        re_mdio_write(sc, 0x14, 0x0aee);
        re_mdio_write(sc, 0x14, 0x80f0);
        re_mdio_write(sc, 0x14, 0x10ee);
        re_mdio_write(sc, 0x14, 0x80f3);
        re_mdio_write(sc, 0x14, 0x8fee);
        re_mdio_write(sc, 0x14, 0x8101);
        re_mdio_write(sc, 0x14, 0x1eee);
        re_mdio_write(sc, 0x14, 0x810b);
        re_mdio_write(sc, 0x14, 0x4aee);
        re_mdio_write(sc, 0x14, 0x810c);
        re_mdio_write(sc, 0x14, 0x7cee);
        re_mdio_write(sc, 0x14, 0x8112);
        re_mdio_write(sc, 0x14, 0x7fd1);
        re_mdio_write(sc, 0x14, 0x0002);
        re_mdio_write(sc, 0x14, 0x10e3);
        re_mdio_write(sc, 0x14, 0xee80);
        re_mdio_write(sc, 0x14, 0x8892);
        re_mdio_write(sc, 0x14, 0xee80);
        re_mdio_write(sc, 0x14, 0x8922);
        re_mdio_write(sc, 0x14, 0xee80);
        re_mdio_write(sc, 0x14, 0x9a80);
        re_mdio_write(sc, 0x14, 0xee80);
        re_mdio_write(sc, 0x14, 0x9b22);
        re_mdio_write(sc, 0x14, 0xee80);
        re_mdio_write(sc, 0x14, 0x9ca7);
        re_mdio_write(sc, 0x14, 0xee80);
        re_mdio_write(sc, 0x14, 0xa010);
        re_mdio_write(sc, 0x14, 0xee80);
        re_mdio_write(sc, 0x14, 0xa5a7);
        re_mdio_write(sc, 0x14, 0xd200);
        re_mdio_write(sc, 0x14, 0x020e);
        re_mdio_write(sc, 0x14, 0x4b02);
        re_mdio_write(sc, 0x14, 0x85c1);
        re_mdio_write(sc, 0x14, 0xef96);
        re_mdio_write(sc, 0x14, 0xfefd);
        re_mdio_write(sc, 0x14, 0xfc04);
        re_mdio_write(sc, 0x14, 0x0284);
        re_mdio_write(sc, 0x14, 0x7b02);
        re_mdio_write(sc, 0x14, 0x84b4);
        re_mdio_write(sc, 0x14, 0x020c);
        re_mdio_write(sc, 0x14, 0x9202);
        re_mdio_write(sc, 0x14, 0x0cab);
        re_mdio_write(sc, 0x14, 0x020c);
        re_mdio_write(sc, 0x14, 0xd602);
        re_mdio_write(sc, 0x14, 0x0cef);
        re_mdio_write(sc, 0x14, 0x020d);
        re_mdio_write(sc, 0x14, 0x1a02);
        re_mdio_write(sc, 0x14, 0x0c24);
        re_mdio_write(sc, 0x14, 0x04f8);
        re_mdio_write(sc, 0x14, 0xfaef);
        re_mdio_write(sc, 0x14, 0x69e1);
        re_mdio_write(sc, 0x14, 0x8234);
        re_mdio_write(sc, 0x14, 0xac29);
        re_mdio_write(sc, 0x14, 0x1ae0);
        re_mdio_write(sc, 0x14, 0x8229);
        re_mdio_write(sc, 0x14, 0xac21);
        re_mdio_write(sc, 0x14, 0x02ae);
        re_mdio_write(sc, 0x14, 0x2202);
        re_mdio_write(sc, 0x14, 0x1085);
        re_mdio_write(sc, 0x14, 0xf621);
        re_mdio_write(sc, 0x14, 0xe482);
        re_mdio_write(sc, 0x14, 0x29d1);
        re_mdio_write(sc, 0x14, 0x01bf);
        re_mdio_write(sc, 0x14, 0x4364);
        re_mdio_write(sc, 0x14, 0x0241);
        re_mdio_write(sc, 0x14, 0x0aae);
        re_mdio_write(sc, 0x14, 0x1002);
        re_mdio_write(sc, 0x14, 0x127a);
        re_mdio_write(sc, 0x14, 0xf629);
        re_mdio_write(sc, 0x14, 0xe582);
        re_mdio_write(sc, 0x14, 0x34e0);
        re_mdio_write(sc, 0x14, 0x8229);
        re_mdio_write(sc, 0x14, 0xf621);
        re_mdio_write(sc, 0x14, 0xe482);
        re_mdio_write(sc, 0x14, 0x29ef);
        re_mdio_write(sc, 0x14, 0x96fe);
        re_mdio_write(sc, 0x14, 0xfc04);
        re_mdio_write(sc, 0x14, 0xf8e1);
        re_mdio_write(sc, 0x14, 0x8234);
        re_mdio_write(sc, 0x14, 0xac2a);
        re_mdio_write(sc, 0x14, 0x18e0);
        re_mdio_write(sc, 0x14, 0x8229);
        re_mdio_write(sc, 0x14, 0xac22);
        re_mdio_write(sc, 0x14, 0x02ae);
        re_mdio_write(sc, 0x14, 0x2602);
        re_mdio_write(sc, 0x14, 0x84f9);
        re_mdio_write(sc, 0x14, 0x0285);
        re_mdio_write(sc, 0x14, 0x66d1);
        re_mdio_write(sc, 0x14, 0x01bf);
        re_mdio_write(sc, 0x14, 0x4367);
        re_mdio_write(sc, 0x14, 0x0241);
        re_mdio_write(sc, 0x14, 0x0aae);
        re_mdio_write(sc, 0x14, 0x0e02);
        re_mdio_write(sc, 0x14, 0x84eb);
        re_mdio_write(sc, 0x14, 0x0285);
        re_mdio_write(sc, 0x14, 0xaae1);
        re_mdio_write(sc, 0x14, 0x8234);
        re_mdio_write(sc, 0x14, 0xf62a);
        re_mdio_write(sc, 0x14, 0xe582);
        re_mdio_write(sc, 0x14, 0x34e0);
        re_mdio_write(sc, 0x14, 0x8229);
        re_mdio_write(sc, 0x14, 0xf622);
        re_mdio_write(sc, 0x14, 0xe482);
        re_mdio_write(sc, 0x14, 0x29fc);
        re_mdio_write(sc, 0x14, 0x04f9);
        re_mdio_write(sc, 0x14, 0xe280);
        re_mdio_write(sc, 0x14, 0x11ad);
        re_mdio_write(sc, 0x14, 0x3105);
        re_mdio_write(sc, 0x14, 0xd200);
        re_mdio_write(sc, 0x14, 0x020e);
        re_mdio_write(sc, 0x14, 0x4bfd);
        re_mdio_write(sc, 0x14, 0x04f8);
        re_mdio_write(sc, 0x14, 0xf9fa);
        re_mdio_write(sc, 0x14, 0xef69);
        re_mdio_write(sc, 0x14, 0xe080);
        re_mdio_write(sc, 0x14, 0x11ad);
        re_mdio_write(sc, 0x14, 0x215c);
        re_mdio_write(sc, 0x14, 0xbf42);
        re_mdio_write(sc, 0x14, 0x5002);
        re_mdio_write(sc, 0x14, 0x4148);
        re_mdio_write(sc, 0x14, 0xac28);
        re_mdio_write(sc, 0x14, 0x1bbf);
        re_mdio_write(sc, 0x14, 0x4253);
        re_mdio_write(sc, 0x14, 0x0241);
        re_mdio_write(sc, 0x14, 0x48ac);
        re_mdio_write(sc, 0x14, 0x2812);
        re_mdio_write(sc, 0x14, 0xbf42);
        re_mdio_write(sc, 0x14, 0x5902);
        re_mdio_write(sc, 0x14, 0x4148);
        re_mdio_write(sc, 0x14, 0xac28);
        re_mdio_write(sc, 0x14, 0x04d3);
        re_mdio_write(sc, 0x14, 0x00ae);
        re_mdio_write(sc, 0x14, 0x07d3);
        re_mdio_write(sc, 0x14, 0x06af);
        re_mdio_write(sc, 0x14, 0x8557);
        re_mdio_write(sc, 0x14, 0xd303);
        re_mdio_write(sc, 0x14, 0xe080);
        re_mdio_write(sc, 0x14, 0x11ad);
        re_mdio_write(sc, 0x14, 0x2625);
        re_mdio_write(sc, 0x14, 0xbf43);
        re_mdio_write(sc, 0x14, 0xeb02);
        re_mdio_write(sc, 0x14, 0x4148);
        re_mdio_write(sc, 0x14, 0xe280);
        re_mdio_write(sc, 0x14, 0x730d);
        re_mdio_write(sc, 0x14, 0x21f6);
        re_mdio_write(sc, 0x14, 0x370d);
        re_mdio_write(sc, 0x14, 0x11f6);
        re_mdio_write(sc, 0x14, 0x2f1b);
        re_mdio_write(sc, 0x14, 0x21aa);
        re_mdio_write(sc, 0x14, 0x02ae);
        re_mdio_write(sc, 0x14, 0x10e2);
        re_mdio_write(sc, 0x14, 0x8074);
        re_mdio_write(sc, 0x14, 0x0d21);
        re_mdio_write(sc, 0x14, 0xf637);
        re_mdio_write(sc, 0x14, 0x1b21);
        re_mdio_write(sc, 0x14, 0xaa03);
        re_mdio_write(sc, 0x14, 0x13ae);
        re_mdio_write(sc, 0x14, 0x022b);
        re_mdio_write(sc, 0x14, 0x0202);
        re_mdio_write(sc, 0x14, 0x0e36);
        re_mdio_write(sc, 0x14, 0x020e);
        re_mdio_write(sc, 0x14, 0x4b02);
        re_mdio_write(sc, 0x14, 0x0f91);
        re_mdio_write(sc, 0x14, 0xef96);
        re_mdio_write(sc, 0x14, 0xfefd);
        re_mdio_write(sc, 0x14, 0xfc04);
        re_mdio_write(sc, 0x14, 0xf8f9);
        re_mdio_write(sc, 0x14, 0xfaef);
        re_mdio_write(sc, 0x14, 0x69e0);
        re_mdio_write(sc, 0x14, 0x8012);
        re_mdio_write(sc, 0x14, 0xad27);
        re_mdio_write(sc, 0x14, 0x33bf);
        re_mdio_write(sc, 0x14, 0x4250);
        re_mdio_write(sc, 0x14, 0x0241);
        re_mdio_write(sc, 0x14, 0x48ac);
        re_mdio_write(sc, 0x14, 0x2809);
        re_mdio_write(sc, 0x14, 0xbf42);
        re_mdio_write(sc, 0x14, 0x5302);
        re_mdio_write(sc, 0x14, 0x4148);
        re_mdio_write(sc, 0x14, 0xad28);
        re_mdio_write(sc, 0x14, 0x21bf);
        re_mdio_write(sc, 0x14, 0x43eb);
        re_mdio_write(sc, 0x14, 0x0241);
        re_mdio_write(sc, 0x14, 0x48e3);
        re_mdio_write(sc, 0x14, 0x87ff);
        re_mdio_write(sc, 0x14, 0xd200);
        re_mdio_write(sc, 0x14, 0x1b45);
        re_mdio_write(sc, 0x14, 0xac27);
        re_mdio_write(sc, 0x14, 0x11e1);
        re_mdio_write(sc, 0x14, 0x87fe);
        re_mdio_write(sc, 0x14, 0xbf87);
        re_mdio_write(sc, 0x14, 0x6702);
        re_mdio_write(sc, 0x14, 0x410a);
        re_mdio_write(sc, 0x14, 0x0d11);
        re_mdio_write(sc, 0x14, 0xbf87);
        re_mdio_write(sc, 0x14, 0x6a02);
        re_mdio_write(sc, 0x14, 0x410a);
        re_mdio_write(sc, 0x14, 0xef96);
        re_mdio_write(sc, 0x14, 0xfefd);
        re_mdio_write(sc, 0x14, 0xfc04);
        re_mdio_write(sc, 0x14, 0xf8fa);
        re_mdio_write(sc, 0x14, 0xef69);
        re_mdio_write(sc, 0x14, 0xd100);
        re_mdio_write(sc, 0x14, 0xbf87);
        re_mdio_write(sc, 0x14, 0x6702);
        re_mdio_write(sc, 0x14, 0x410a);
        re_mdio_write(sc, 0x14, 0xbf87);
        re_mdio_write(sc, 0x14, 0x6a02);
        re_mdio_write(sc, 0x14, 0x410a);
        re_mdio_write(sc, 0x14, 0xef96);
        re_mdio_write(sc, 0x14, 0xfefc);
        re_mdio_write(sc, 0x14, 0x04ee);
        re_mdio_write(sc, 0x14, 0x87ff);
        re_mdio_write(sc, 0x14, 0x46ee);
        re_mdio_write(sc, 0x14, 0x87fe);
        re_mdio_write(sc, 0x14, 0x0104);
        re_mdio_write(sc, 0x14, 0xf8fa);
        re_mdio_write(sc, 0x14, 0xef69);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x46a0);
        re_mdio_write(sc, 0x14, 0x0005);
        re_mdio_write(sc, 0x14, 0x0285);
        re_mdio_write(sc, 0x14, 0xecae);
        re_mdio_write(sc, 0x14, 0x0ea0);
        re_mdio_write(sc, 0x14, 0x0105);
        re_mdio_write(sc, 0x14, 0x021a);
        re_mdio_write(sc, 0x14, 0x68ae);
        re_mdio_write(sc, 0x14, 0x06a0);
        re_mdio_write(sc, 0x14, 0x0203);
        re_mdio_write(sc, 0x14, 0x021a);
        re_mdio_write(sc, 0x14, 0xf4ef);
        re_mdio_write(sc, 0x14, 0x96fe);
        re_mdio_write(sc, 0x14, 0xfc04);
        re_mdio_write(sc, 0x14, 0xf8f9);
        re_mdio_write(sc, 0x14, 0xfaef);
        re_mdio_write(sc, 0x14, 0x69e0);
        re_mdio_write(sc, 0x14, 0x822e);
        re_mdio_write(sc, 0x14, 0xf621);
        re_mdio_write(sc, 0x14, 0xe482);
        re_mdio_write(sc, 0x14, 0x2ee0);
        re_mdio_write(sc, 0x14, 0x8010);
        re_mdio_write(sc, 0x14, 0xac22);
        re_mdio_write(sc, 0x14, 0x02ae);
        re_mdio_write(sc, 0x14, 0x76e0);
        re_mdio_write(sc, 0x14, 0x822c);
        re_mdio_write(sc, 0x14, 0xf721);
        re_mdio_write(sc, 0x14, 0xe482);
        re_mdio_write(sc, 0x14, 0x2cbf);
        re_mdio_write(sc, 0x14, 0x41a5);
        re_mdio_write(sc, 0x14, 0x0241);
        re_mdio_write(sc, 0x14, 0x48ef);
        re_mdio_write(sc, 0x14, 0x21bf);
        re_mdio_write(sc, 0x14, 0x41a8);
        re_mdio_write(sc, 0x14, 0x0241);
        re_mdio_write(sc, 0x14, 0x480c);
        re_mdio_write(sc, 0x14, 0x111e);
        re_mdio_write(sc, 0x14, 0x21bf);
        re_mdio_write(sc, 0x14, 0x41ab);
        re_mdio_write(sc, 0x14, 0x0241);
        re_mdio_write(sc, 0x14, 0x480c);
        re_mdio_write(sc, 0x14, 0x121e);
        re_mdio_write(sc, 0x14, 0x21e6);
        re_mdio_write(sc, 0x14, 0x8248);
        re_mdio_write(sc, 0x14, 0xa200);
        re_mdio_write(sc, 0x14, 0x0ae1);
        re_mdio_write(sc, 0x14, 0x822c);
        re_mdio_write(sc, 0x14, 0xf629);
        re_mdio_write(sc, 0x14, 0xe582);
        re_mdio_write(sc, 0x14, 0x2cae);
        re_mdio_write(sc, 0x14, 0x42e0);
        re_mdio_write(sc, 0x14, 0x8249);
        re_mdio_write(sc, 0x14, 0xf721);
        re_mdio_write(sc, 0x14, 0xe482);
        re_mdio_write(sc, 0x14, 0x4902);
        re_mdio_write(sc, 0x14, 0x4520);
        re_mdio_write(sc, 0x14, 0xbf41);
        re_mdio_write(sc, 0x14, 0xb702);
        re_mdio_write(sc, 0x14, 0x4148);
        re_mdio_write(sc, 0x14, 0xef21);
        re_mdio_write(sc, 0x14, 0xbf41);
        re_mdio_write(sc, 0x14, 0xae02);
        re_mdio_write(sc, 0x14, 0x4148);
        re_mdio_write(sc, 0x14, 0x0c12);
        re_mdio_write(sc, 0x14, 0x1e21);
        re_mdio_write(sc, 0x14, 0xbf41);
        re_mdio_write(sc, 0x14, 0xb102);
        re_mdio_write(sc, 0x14, 0x4148);
        re_mdio_write(sc, 0x14, 0x0c13);
        re_mdio_write(sc, 0x14, 0x1e21);
        re_mdio_write(sc, 0x14, 0xbf41);
        re_mdio_write(sc, 0x14, 0xba02);
        re_mdio_write(sc, 0x14, 0x4148);
        re_mdio_write(sc, 0x14, 0x0c14);
        re_mdio_write(sc, 0x14, 0x1e21);
        re_mdio_write(sc, 0x14, 0xbf43);
        re_mdio_write(sc, 0x14, 0x4602);
        re_mdio_write(sc, 0x14, 0x4148);
        re_mdio_write(sc, 0x14, 0x0c16);
        re_mdio_write(sc, 0x14, 0x1e21);
        re_mdio_write(sc, 0x14, 0xe682);
        re_mdio_write(sc, 0x14, 0x47ee);
        re_mdio_write(sc, 0x14, 0x8246);
        re_mdio_write(sc, 0x14, 0x01ef);
        re_mdio_write(sc, 0x14, 0x96fe);
        re_mdio_write(sc, 0x14, 0xfdfc);
        re_mdio_write(sc, 0x14, 0x04f8);
        re_mdio_write(sc, 0x14, 0xfaef);
        re_mdio_write(sc, 0x14, 0x69e0);
        re_mdio_write(sc, 0x14, 0x824b);
        re_mdio_write(sc, 0x14, 0xa000);
        re_mdio_write(sc, 0x14, 0x0502);
        re_mdio_write(sc, 0x14, 0x8697);
        re_mdio_write(sc, 0x14, 0xae06);
        re_mdio_write(sc, 0x14, 0xa001);
        re_mdio_write(sc, 0x14, 0x0302);
        re_mdio_write(sc, 0x14, 0x1937);
        re_mdio_write(sc, 0x14, 0xef96);
        re_mdio_write(sc, 0x14, 0xfefc);
        re_mdio_write(sc, 0x14, 0x04f8);
        re_mdio_write(sc, 0x14, 0xfaef);
        re_mdio_write(sc, 0x14, 0x69e0);
        re_mdio_write(sc, 0x14, 0x822e);
        re_mdio_write(sc, 0x14, 0xf620);
        re_mdio_write(sc, 0x14, 0xe482);
        re_mdio_write(sc, 0x14, 0x2ee0);
        re_mdio_write(sc, 0x14, 0x8010);
        re_mdio_write(sc, 0x14, 0xac21);
        re_mdio_write(sc, 0x14, 0x02ae);
        re_mdio_write(sc, 0x14, 0x54e0);
        re_mdio_write(sc, 0x14, 0x822c);
        re_mdio_write(sc, 0x14, 0xf720);
        re_mdio_write(sc, 0x14, 0xe482);
        re_mdio_write(sc, 0x14, 0x2cbf);
        re_mdio_write(sc, 0x14, 0x4175);
        re_mdio_write(sc, 0x14, 0x0241);
        re_mdio_write(sc, 0x14, 0x48ac);
        re_mdio_write(sc, 0x14, 0x2822);
        re_mdio_write(sc, 0x14, 0xbf41);
        re_mdio_write(sc, 0x14, 0x9f02);
        re_mdio_write(sc, 0x14, 0x4148);
        re_mdio_write(sc, 0x14, 0xe582);
        re_mdio_write(sc, 0x14, 0x4cac);
        re_mdio_write(sc, 0x14, 0x2820);
        re_mdio_write(sc, 0x14, 0xd103);
        re_mdio_write(sc, 0x14, 0xbf41);
        re_mdio_write(sc, 0x14, 0x9902);
        re_mdio_write(sc, 0x14, 0x410a);
        re_mdio_write(sc, 0x14, 0xee82);
        re_mdio_write(sc, 0x14, 0x4b00);
        re_mdio_write(sc, 0x14, 0xe182);
        re_mdio_write(sc, 0x14, 0x2cf6);
        re_mdio_write(sc, 0x14, 0x28e5);
        re_mdio_write(sc, 0x14, 0x822c);
        re_mdio_write(sc, 0x14, 0xae21);
        re_mdio_write(sc, 0x14, 0xd104);
        re_mdio_write(sc, 0x14, 0xbf41);
        re_mdio_write(sc, 0x14, 0x9902);
        re_mdio_write(sc, 0x14, 0x410a);
        re_mdio_write(sc, 0x14, 0xae08);
        re_mdio_write(sc, 0x14, 0xd105);
        re_mdio_write(sc, 0x14, 0xbf41);
        re_mdio_write(sc, 0x14, 0x9902);
        re_mdio_write(sc, 0x14, 0x410a);
        re_mdio_write(sc, 0x14, 0xe082);
        re_mdio_write(sc, 0x14, 0x49f7);
        re_mdio_write(sc, 0x14, 0x20e4);
        re_mdio_write(sc, 0x14, 0x8249);
        re_mdio_write(sc, 0x14, 0x0245);
        re_mdio_write(sc, 0x14, 0x20ee);
        re_mdio_write(sc, 0x14, 0x824b);
        re_mdio_write(sc, 0x14, 0x01ef);
        re_mdio_write(sc, 0x14, 0x96fe);
        re_mdio_write(sc, 0x14, 0xfc04);
        re_mdio_write(sc, 0x14, 0xf8f9);
        re_mdio_write(sc, 0x14, 0xface);
        re_mdio_write(sc, 0x14, 0xfaef);
        re_mdio_write(sc, 0x14, 0x69fb);
        re_mdio_write(sc, 0x14, 0xbf87);
        re_mdio_write(sc, 0x14, 0x2fd7);
        re_mdio_write(sc, 0x14, 0x0020);
        re_mdio_write(sc, 0x14, 0xd819);
        re_mdio_write(sc, 0x14, 0xd919);
        re_mdio_write(sc, 0x14, 0xda19);
        re_mdio_write(sc, 0x14, 0xdb19);
        re_mdio_write(sc, 0x14, 0x07ef);
        re_mdio_write(sc, 0x14, 0x9502);
        re_mdio_write(sc, 0x14, 0x410a);
        re_mdio_write(sc, 0x14, 0x073f);
        re_mdio_write(sc, 0x14, 0x0004);
        re_mdio_write(sc, 0x14, 0x9fec);
        re_mdio_write(sc, 0x14, 0xffef);
        re_mdio_write(sc, 0x14, 0x96fe);
        re_mdio_write(sc, 0x14, 0xc6fe);
        re_mdio_write(sc, 0x14, 0xfdfc);
        re_mdio_write(sc, 0x14, 0x0400);
        re_mdio_write(sc, 0x14, 0x0144);
        re_mdio_write(sc, 0x14, 0x0000);
        re_mdio_write(sc, 0x14, 0x0343);
        re_mdio_write(sc, 0x14, 0xee00);
        re_mdio_write(sc, 0x14, 0x0087);
        re_mdio_write(sc, 0x14, 0x5b00);
        re_mdio_write(sc, 0x14, 0x0141);
        re_mdio_write(sc, 0x14, 0xe100);
        re_mdio_write(sc, 0x14, 0x0387);
        re_mdio_write(sc, 0x14, 0x5e00);
        re_mdio_write(sc, 0x14, 0x0987);
        re_mdio_write(sc, 0x14, 0x6100);
        re_mdio_write(sc, 0x14, 0x0987);
        re_mdio_write(sc, 0x14, 0x6400);
        re_mdio_write(sc, 0x14, 0x0087);
        re_mdio_write(sc, 0x14, 0x6da4);
        re_mdio_write(sc, 0x14, 0x00b8);
        re_mdio_write(sc, 0x14, 0x20c4);
        re_mdio_write(sc, 0x14, 0x1600);
        re_mdio_write(sc, 0x14, 0x000f);
        re_mdio_write(sc, 0x14, 0xf800);
        re_mdio_write(sc, 0x14, 0x7000);
        re_mdio_write(sc, 0x14, 0xb82e);
        re_mdio_write(sc, 0x14, 0x98a5);
        re_mdio_write(sc, 0x14, 0x8ab6);
        re_mdio_write(sc, 0x14, 0xa83e);
        re_mdio_write(sc, 0x14, 0x50a8);
        re_mdio_write(sc, 0x14, 0x3e33);
        re_mdio_write(sc, 0x14, 0xbcc6);
        re_mdio_write(sc, 0x14, 0x22bc);
        re_mdio_write(sc, 0x14, 0xc6aa);
        re_mdio_write(sc, 0x14, 0xa442);
        re_mdio_write(sc, 0x14, 0xffc4);
        re_mdio_write(sc, 0x14, 0x0800);
        re_mdio_write(sc, 0x14, 0xc416);
        re_mdio_write(sc, 0x14, 0xa8bc);
        re_mdio_write(sc, 0x14, 0xc000);
        re_mdio_write(sc, 0x13, 0xb818);
        re_mdio_write(sc, 0x14, 0x02e3);
        re_mdio_write(sc, 0x13, 0xb81a);
        re_mdio_write(sc, 0x14, 0x17ff);
        re_mdio_write(sc, 0x13, 0xb81e);
        re_mdio_write(sc, 0x14, 0x3b1c);
        re_mdio_write(sc, 0x13, 0xb820);
        re_mdio_write(sc, 0x14, 0x021b);
        re_mdio_write(sc, 0x1f, 0x0000);


        re_mdio_write(sc,0x1F, 0x0A43);
        re_mdio_write(sc,0x13, 0x0000);
        re_mdio_write(sc,0x14, 0x0000);
        re_mdio_write(sc,0x1f, 0x0B82);
        PhyRegValue = re_mdio_read(sc, 0x17);
        PhyRegValue &= ~(BIT_0);
        re_mdio_write(sc,0x17, PhyRegValue);
        re_mdio_write(sc,0x1f, 0x0A43);
        re_mdio_write(sc,0x13, 0x8146);
        re_mdio_write(sc,0x14, 0x0000);

        re_clear_phy_mcu_patch_request(sc);
}

static void re_set_phy_mcu_8168ep_2(struct re_softc *sc)
{
        u_int16_t PhyRegValue;

        re_set_phy_mcu_patch_request(sc);

        re_mdio_write(sc,0x1f, 0x0A43);
        re_mdio_write(sc,0x13, 0x8146);
        re_mdio_write(sc,0x14, 0x8700);
        re_mdio_write(sc,0x13, 0xB82E);
        re_mdio_write(sc,0x14, 0x0001);

        re_mdio_write(sc, 0x1F, 0x0A43);

        re_mdio_write(sc, 0x13, 0x83DD);
        re_mdio_write(sc, 0x14, 0xAF83);
        re_mdio_write(sc, 0x14, 0xE9AF);
        re_mdio_write(sc, 0x14, 0x83EE);
        re_mdio_write(sc, 0x14, 0xAF83);
        re_mdio_write(sc, 0x14, 0xF1A1);
        re_mdio_write(sc, 0x14, 0x83F4);
        re_mdio_write(sc, 0x14, 0xD149);
        re_mdio_write(sc, 0x14, 0xAF06);
        re_mdio_write(sc, 0x14, 0x47AF);
        re_mdio_write(sc, 0x14, 0x0000);
        re_mdio_write(sc, 0x14, 0xAF00);
        re_mdio_write(sc, 0x14, 0x00AF);
        re_mdio_write(sc, 0x14, 0x0000);

        re_mdio_write(sc, 0x13, 0xB818);
        re_mdio_write(sc, 0x14, 0x0645);

        re_mdio_write(sc, 0x13, 0xB81A);
        re_mdio_write(sc, 0x14, 0x0000);

        re_mdio_write(sc, 0x13, 0xB81C);
        re_mdio_write(sc, 0x14, 0x0000);

        re_mdio_write(sc, 0x13, 0xB81E);
        re_mdio_write(sc, 0x14, 0x0000);

        re_mdio_write(sc, 0x13, 0xB832);
        re_mdio_write(sc, 0x14, 0x0001);

        re_mdio_write(sc,0x1F, 0x0A43);
        re_mdio_write(sc,0x13, 0x0000);
        re_mdio_write(sc,0x14, 0x0000);
        re_mdio_write(sc,0x1f, 0x0B82);
        PhyRegValue = re_mdio_read(sc, 0x17);
        PhyRegValue &= ~(BIT_0);
        re_mdio_write(sc,0x17, PhyRegValue);
        re_mdio_write(sc,0x1f, 0x0A43);
        re_mdio_write(sc,0x13, 0x8146);
        re_mdio_write(sc,0x14, 0x0000);

        re_clear_phy_mcu_patch_request(sc);
}

static void
re_real_set_phy_mcu_8125a_1(struct re_softc *sc)
{
        re_acquire_phy_mcu_patch_key_lock(sc);


        re_set_eth_ocp_phy_bit(sc, 0xB820, BIT_7);


        re_real_ocp_phy_write(sc, 0xA436, 0xA016);
        re_real_ocp_phy_write(sc, 0xA438, 0x0000);
        re_real_ocp_phy_write(sc, 0xA436, 0xA012);
        re_real_ocp_phy_write(sc, 0xA438, 0x0000);
        re_real_ocp_phy_write(sc, 0xA436, 0xA014);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x8010);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x8013);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x8021);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x802f);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x803d);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x8042);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x8051);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x8051);
        re_real_ocp_phy_write(sc, 0xA438, 0xa088);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x0a50);
        re_real_ocp_phy_write(sc, 0xA438, 0x8008);
        re_real_ocp_phy_write(sc, 0xA438, 0xd014);
        re_real_ocp_phy_write(sc, 0xA438, 0xd1a3);
        re_real_ocp_phy_write(sc, 0xA438, 0xd700);
        re_real_ocp_phy_write(sc, 0xA438, 0x401a);
        re_real_ocp_phy_write(sc, 0xA438, 0xd707);
        re_real_ocp_phy_write(sc, 0xA438, 0x40c2);
        re_real_ocp_phy_write(sc, 0xA438, 0x60a6);
        re_real_ocp_phy_write(sc, 0xA438, 0xd700);
        re_real_ocp_phy_write(sc, 0xA438, 0x5f8b);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x0a86);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x0a6c);
        re_real_ocp_phy_write(sc, 0xA438, 0x8080);
        re_real_ocp_phy_write(sc, 0xA438, 0xd019);
        re_real_ocp_phy_write(sc, 0xA438, 0xd1a2);
        re_real_ocp_phy_write(sc, 0xA438, 0xd700);
        re_real_ocp_phy_write(sc, 0xA438, 0x401a);
        re_real_ocp_phy_write(sc, 0xA438, 0xd707);
        re_real_ocp_phy_write(sc, 0xA438, 0x40c4);
        re_real_ocp_phy_write(sc, 0xA438, 0x60a6);
        re_real_ocp_phy_write(sc, 0xA438, 0xd700);
        re_real_ocp_phy_write(sc, 0xA438, 0x5f8b);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x0a86);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x0a84);
        re_real_ocp_phy_write(sc, 0xA438, 0xd503);
        re_real_ocp_phy_write(sc, 0xA438, 0x8970);
        re_real_ocp_phy_write(sc, 0xA438, 0x0c07);
        re_real_ocp_phy_write(sc, 0xA438, 0x0901);
        re_real_ocp_phy_write(sc, 0xA438, 0xd500);
        re_real_ocp_phy_write(sc, 0xA438, 0xce01);
        re_real_ocp_phy_write(sc, 0xA438, 0xcf09);
        re_real_ocp_phy_write(sc, 0xA438, 0xd705);
        re_real_ocp_phy_write(sc, 0xA438, 0x4000);
        re_real_ocp_phy_write(sc, 0xA438, 0xceff);
        re_real_ocp_phy_write(sc, 0xA438, 0xaf0a);
        re_real_ocp_phy_write(sc, 0xA438, 0xd504);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x1213);
        re_real_ocp_phy_write(sc, 0xA438, 0x8401);
        re_real_ocp_phy_write(sc, 0xA438, 0xd500);
        re_real_ocp_phy_write(sc, 0xA438, 0x8580);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x1253);
        re_real_ocp_phy_write(sc, 0xA438, 0xd064);
        re_real_ocp_phy_write(sc, 0xA438, 0xd181);
        re_real_ocp_phy_write(sc, 0xA438, 0xd704);
        re_real_ocp_phy_write(sc, 0xA438, 0x4018);
        re_real_ocp_phy_write(sc, 0xA438, 0xd504);
        re_real_ocp_phy_write(sc, 0xA438, 0xc50f);
        re_real_ocp_phy_write(sc, 0xA438, 0xd706);
        re_real_ocp_phy_write(sc, 0xA438, 0x2c59);
        re_real_ocp_phy_write(sc, 0xA438, 0x804d);
        re_real_ocp_phy_write(sc, 0xA438, 0xc60f);
        re_real_ocp_phy_write(sc, 0xA438, 0xf002);
        re_real_ocp_phy_write(sc, 0xA438, 0xc605);
        re_real_ocp_phy_write(sc, 0xA438, 0xae02);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x10fd);
        re_real_ocp_phy_write(sc, 0xA436, 0xA026);
        re_real_ocp_phy_write(sc, 0xA438, 0xffff);
        re_real_ocp_phy_write(sc, 0xA436, 0xA024);
        re_real_ocp_phy_write(sc, 0xA438, 0xffff);
        re_real_ocp_phy_write(sc, 0xA436, 0xA022);
        re_real_ocp_phy_write(sc, 0xA438, 0x10f4);
        re_real_ocp_phy_write(sc, 0xA436, 0xA020);
        re_real_ocp_phy_write(sc, 0xA438, 0x1252);
        re_real_ocp_phy_write(sc, 0xA436, 0xA006);
        re_real_ocp_phy_write(sc, 0xA438, 0x1206);
        re_real_ocp_phy_write(sc, 0xA436, 0xA004);
        re_real_ocp_phy_write(sc, 0xA438, 0x0a78);
        re_real_ocp_phy_write(sc, 0xA436, 0xA002);
        re_real_ocp_phy_write(sc, 0xA438, 0x0a60);
        re_real_ocp_phy_write(sc, 0xA436, 0xA000);
        re_real_ocp_phy_write(sc, 0xA438, 0x0a4f);
        re_real_ocp_phy_write(sc, 0xA436, 0xA008);
        re_real_ocp_phy_write(sc, 0xA438, 0x3f00);


        re_real_ocp_phy_write(sc, 0xA436, 0xA016);
        re_real_ocp_phy_write(sc, 0xA438, 0x0010);
        re_real_ocp_phy_write(sc, 0xA436, 0xA012);
        re_real_ocp_phy_write(sc, 0xA438, 0x0000);
        re_real_ocp_phy_write(sc, 0xA436, 0xA014);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x8010);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x8066);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x807c);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x8089);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x808e);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x80a0);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x80b2);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x80c2);
        re_real_ocp_phy_write(sc, 0xA438, 0xd501);
        re_real_ocp_phy_write(sc, 0xA438, 0xce01);
        re_real_ocp_phy_write(sc, 0xA438, 0xd700);
        re_real_ocp_phy_write(sc, 0xA438, 0x62db);
        re_real_ocp_phy_write(sc, 0xA438, 0x655c);
        re_real_ocp_phy_write(sc, 0xA438, 0xd73e);
        re_real_ocp_phy_write(sc, 0xA438, 0x60e9);
        re_real_ocp_phy_write(sc, 0xA438, 0x614a);
        re_real_ocp_phy_write(sc, 0xA438, 0x61ab);
        re_real_ocp_phy_write(sc, 0xA438, 0x0c0f);
        re_real_ocp_phy_write(sc, 0xA438, 0x0501);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x0304);
        re_real_ocp_phy_write(sc, 0xA438, 0x0c0f);
        re_real_ocp_phy_write(sc, 0xA438, 0x0503);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x0304);
        re_real_ocp_phy_write(sc, 0xA438, 0x0c0f);
        re_real_ocp_phy_write(sc, 0xA438, 0x0505);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x0304);
        re_real_ocp_phy_write(sc, 0xA438, 0x0c0f);
        re_real_ocp_phy_write(sc, 0xA438, 0x0509);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x0304);
        re_real_ocp_phy_write(sc, 0xA438, 0x653c);
        re_real_ocp_phy_write(sc, 0xA438, 0xd73e);
        re_real_ocp_phy_write(sc, 0xA438, 0x60e9);
        re_real_ocp_phy_write(sc, 0xA438, 0x614a);
        re_real_ocp_phy_write(sc, 0xA438, 0x61ab);
        re_real_ocp_phy_write(sc, 0xA438, 0x0c0f);
        re_real_ocp_phy_write(sc, 0xA438, 0x0503);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x0304);
        re_real_ocp_phy_write(sc, 0xA438, 0x0c0f);
        re_real_ocp_phy_write(sc, 0xA438, 0x0502);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x0304);
        re_real_ocp_phy_write(sc, 0xA438, 0x0c0f);
        re_real_ocp_phy_write(sc, 0xA438, 0x0506);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x0304);
        re_real_ocp_phy_write(sc, 0xA438, 0x0c0f);
        re_real_ocp_phy_write(sc, 0xA438, 0x050a);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x0304);
        re_real_ocp_phy_write(sc, 0xA438, 0xd73e);
        re_real_ocp_phy_write(sc, 0xA438, 0x60e9);
        re_real_ocp_phy_write(sc, 0xA438, 0x614a);
        re_real_ocp_phy_write(sc, 0xA438, 0x61ab);
        re_real_ocp_phy_write(sc, 0xA438, 0x0c0f);
        re_real_ocp_phy_write(sc, 0xA438, 0x0505);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x0304);
        re_real_ocp_phy_write(sc, 0xA438, 0x0c0f);
        re_real_ocp_phy_write(sc, 0xA438, 0x0506);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x0304);
        re_real_ocp_phy_write(sc, 0xA438, 0x0c0f);
        re_real_ocp_phy_write(sc, 0xA438, 0x0504);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x0304);
        re_real_ocp_phy_write(sc, 0xA438, 0x0c0f);
        re_real_ocp_phy_write(sc, 0xA438, 0x050c);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x0304);
        re_real_ocp_phy_write(sc, 0xA438, 0xd73e);
        re_real_ocp_phy_write(sc, 0xA438, 0x60e9);
        re_real_ocp_phy_write(sc, 0xA438, 0x614a);
        re_real_ocp_phy_write(sc, 0xA438, 0x61ab);
        re_real_ocp_phy_write(sc, 0xA438, 0x0c0f);
        re_real_ocp_phy_write(sc, 0xA438, 0x0509);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x0304);
        re_real_ocp_phy_write(sc, 0xA438, 0x0c0f);
        re_real_ocp_phy_write(sc, 0xA438, 0x050a);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x0304);
        re_real_ocp_phy_write(sc, 0xA438, 0x0c0f);
        re_real_ocp_phy_write(sc, 0xA438, 0x050c);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x0304);
        re_real_ocp_phy_write(sc, 0xA438, 0x0c0f);
        re_real_ocp_phy_write(sc, 0xA438, 0x0508);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x0304);
        re_real_ocp_phy_write(sc, 0xA438, 0xd501);
        re_real_ocp_phy_write(sc, 0xA438, 0xce01);
        re_real_ocp_phy_write(sc, 0xA438, 0xd73e);
        re_real_ocp_phy_write(sc, 0xA438, 0x60e9);
        re_real_ocp_phy_write(sc, 0xA438, 0x614a);
        re_real_ocp_phy_write(sc, 0xA438, 0x61ab);
        re_real_ocp_phy_write(sc, 0xA438, 0x0c0f);
        re_real_ocp_phy_write(sc, 0xA438, 0x0501);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x0321);
        re_real_ocp_phy_write(sc, 0xA438, 0x0c0f);
        re_real_ocp_phy_write(sc, 0xA438, 0x0502);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x0321);
        re_real_ocp_phy_write(sc, 0xA438, 0x0c0f);
        re_real_ocp_phy_write(sc, 0xA438, 0x0504);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x0321);
        re_real_ocp_phy_write(sc, 0xA438, 0x0c0f);
        re_real_ocp_phy_write(sc, 0xA438, 0x0508);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x0321);
        re_real_ocp_phy_write(sc, 0xA438, 0x1000);
        re_real_ocp_phy_write(sc, 0xA438, 0x0346);
        re_real_ocp_phy_write(sc, 0xA438, 0xd501);
        re_real_ocp_phy_write(sc, 0xA438, 0xce01);
        re_real_ocp_phy_write(sc, 0xA438, 0x8208);
        re_real_ocp_phy_write(sc, 0xA438, 0x609d);
        re_real_ocp_phy_write(sc, 0xA438, 0xa50f);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x001a);
        re_real_ocp_phy_write(sc, 0xA438, 0x0c0f);
        re_real_ocp_phy_write(sc, 0xA438, 0x0503);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x001a);
        re_real_ocp_phy_write(sc, 0xA438, 0x607d);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x00ab);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x00ab);
        re_real_ocp_phy_write(sc, 0xA438, 0xd501);
        re_real_ocp_phy_write(sc, 0xA438, 0xce01);
        re_real_ocp_phy_write(sc, 0xA438, 0xd700);
        re_real_ocp_phy_write(sc, 0xA438, 0x60fd);
        re_real_ocp_phy_write(sc, 0xA438, 0xa50f);
        re_real_ocp_phy_write(sc, 0xA438, 0xce00);
        re_real_ocp_phy_write(sc, 0xA438, 0xd500);
        re_real_ocp_phy_write(sc, 0xA438, 0xaa0f);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x017b);
        re_real_ocp_phy_write(sc, 0xA438, 0x0c0f);
        re_real_ocp_phy_write(sc, 0xA438, 0x0503);
        re_real_ocp_phy_write(sc, 0xA438, 0xce00);
        re_real_ocp_phy_write(sc, 0xA438, 0xd500);
        re_real_ocp_phy_write(sc, 0xA438, 0x0c0f);
        re_real_ocp_phy_write(sc, 0xA438, 0x0a05);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x017b);
        re_real_ocp_phy_write(sc, 0xA438, 0xd501);
        re_real_ocp_phy_write(sc, 0xA438, 0xce01);
        re_real_ocp_phy_write(sc, 0xA438, 0xd700);
        re_real_ocp_phy_write(sc, 0xA438, 0x60fd);
        re_real_ocp_phy_write(sc, 0xA438, 0xa50f);
        re_real_ocp_phy_write(sc, 0xA438, 0xce00);
        re_real_ocp_phy_write(sc, 0xA438, 0xd500);
        re_real_ocp_phy_write(sc, 0xA438, 0xaa0f);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x01e0);
        re_real_ocp_phy_write(sc, 0xA438, 0x0c0f);
        re_real_ocp_phy_write(sc, 0xA438, 0x0503);
        re_real_ocp_phy_write(sc, 0xA438, 0xce00);
        re_real_ocp_phy_write(sc, 0xA438, 0xd500);
        re_real_ocp_phy_write(sc, 0xA438, 0x0c0f);
        re_real_ocp_phy_write(sc, 0xA438, 0x0a05);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x01e0);
        re_real_ocp_phy_write(sc, 0xA438, 0xd700);
        re_real_ocp_phy_write(sc, 0xA438, 0x60fd);
        re_real_ocp_phy_write(sc, 0xA438, 0xa50f);
        re_real_ocp_phy_write(sc, 0xA438, 0xce00);
        re_real_ocp_phy_write(sc, 0xA438, 0xd500);
        re_real_ocp_phy_write(sc, 0xA438, 0xaa0f);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x0231);
        re_real_ocp_phy_write(sc, 0xA438, 0x0c0f);
        re_real_ocp_phy_write(sc, 0xA438, 0x0503);
        re_real_ocp_phy_write(sc, 0xA438, 0xce00);
        re_real_ocp_phy_write(sc, 0xA438, 0xd500);
        re_real_ocp_phy_write(sc, 0xA438, 0x0c0f);
        re_real_ocp_phy_write(sc, 0xA438, 0x0a05);
        re_real_ocp_phy_write(sc, 0xA438, 0x1800);
        re_real_ocp_phy_write(sc, 0xA438, 0x0231);
        re_real_ocp_phy_write(sc, 0xA436, 0xA08E);
        re_real_ocp_phy_write(sc, 0xA438, 0xffff);
        re_real_ocp_phy_write(sc, 0xA436, 0xA08C);
        re_real_ocp_phy_write(sc, 0xA438, 0x0221);
        re_real_ocp_phy_write(sc, 0xA436, 0xA08A);
        re_real_ocp_phy_write(sc, 0xA438, 0x01ce);
        re_real_ocp_phy_write(sc, 0xA436, 0xA088);
        re_real_ocp_phy_write(sc, 0xA438, 0x0169);
        re_real_ocp_phy_write(sc, 0xA436, 0xA086);
        re_real_ocp_phy_write(sc, 0xA438, 0x00a6);
        re_real_ocp_phy_write(sc, 0xA436, 0xA084);
        re_real_ocp_phy_write(sc, 0xA438, 0x000d);
        re_real_ocp_phy_write(sc, 0xA436, 0xA082);
        re_real_ocp_phy_write(sc, 0xA